/*
 * Copyright (C) 2008 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef USERDATA_H_
#define USERDATA_H_
#include <cstdint>
#include <string>
#include <map>
#include <memory>
#include <vector>

#include "SmsHeader.h"

class UserData {
public:
  UserData();
  virtual ~UserData();
  std::string toString();
  /**
   * User data encoding types.
   * (See 3GPP2 C.R1001-F, v1.0, table 9.1-1)
   */
  static constexpr int ENCODING_OCTET = 0x00;
  static constexpr int ENCODING_IS91_EXTENDED_PROTOCOL = 0x01;
  static constexpr int ENCODING_7BIT_ASCII = 0x02;
  static constexpr int ENCODING_IA5 = 0x03;
  static constexpr int ENCODING_UNICODE_16 = 0x04;
  static constexpr int ENCODING_SHIFT_JIS = 0x05;
  static constexpr int ENCODING_KOREAN = 0x06;
  static constexpr int ENCODING_LATIN_HEBREW = 0x07;
  static constexpr int ENCODING_LATIN = 0x08;
  static constexpr int ENCODING_GSM_7BIT_ALPHABET = 0x09;
  static constexpr int ENCODING_GSM_DCS = 0x0A;

  /**
   * User data message type encoding types.
   * (See 3GPP2 C.S0015-B, 4.5.2 and 3GPP 23.038, Section 4)
   */
  static constexpr int ENCODING_GSM_DCS_7BIT = 0x00;
  static constexpr int ENCODING_GSM_DCS_8BIT = 0x01;
  static constexpr int ENCODING_GSM_DCS_16BIT = 0x02;

  /**
   * IS-91 message types.
   * (See TIA/EIS/IS-91-A-ENGL 1999, table 3.7.1.1-3)
   */
  static constexpr int IS91_MSG_TYPE_VOICEMAIL_STATUS = 0x82;
  static constexpr int IS91_MSG_TYPE_SHORT_MESSAGE_FULL = 0x83;
  static constexpr int IS91_MSG_TYPE_CLI = 0x84;
  static constexpr int IS91_MSG_TYPE_SHORT_MESSAGE = 0x85;

  /**
   * US ASCII character mapping table.
   *
   * This table contains only the printable ASCII characters, with a
   * 0x20 offset, meaning that the ASCII SPACE character is at index
   * 0, with the resulting code of 0x20.
   *
   * Note this mapping is also equivalent to that used by both the
   * IA5 and the IS-91 encodings.  For the former this is defined
   * using CCITT Rec. T.50 Tables 1 and 3.  For the latter IS 637 B,
   * Table 4.3.1.4.1-1 -- and note the encoding uses only 6 bits,
   * and hence only maps entries up to the '_' character.
   *
   */
  static const char ASCII_MAP[];

  /**
   * Character to use when forced to encode otherwise unencodable
   * characters, meaning those not in the respective ASCII or GSM
   * 7-bit encoding tables.  Current choice is SPACE, which is 0x20
   * in both the GSM-7bit and ASCII-7bit encodings.
   */
  static constexpr uint8_t UNENCODABLE_7_BIT_CHAR = 0x20;

  /**
   * Only elements between these indices in the ASCII table are printable.
   */
  static constexpr int PRINTABLE_ASCII_MIN_INDEX = 0x20;
  static constexpr int ASCII_NL_INDEX = 0x0A;
  ;
  static constexpr int ASCII_CR_INDEX = 0x0D;
  static std::map<char, uint8_t> charToAscii;

  /*
   * TODO(cleanup): Move this very generic functionality somewhere
   * more general.
   */
  /**
   * Given a string generate a corresponding ASCII-encoded byte
   * array, but limited to printable characters.  If the input
   * contains unprintable characters, return null.
   */
  //static uint8_t* stringToAscii(std::string msg, int* length);
  static std::vector<uint8_t> stringToAscii(std::string msg);
  /**
   * Mapping for ASCII values less than 32 are flow control signals
   * and not used here.
   */
  static constexpr int ASCII_MAP_BASE_INDEX = 0x20;
  static const int ASCII_MAP_MAX_INDEX;

  /**
   * Contains the data header of the user data
   */
  std::shared_ptr<SmsHeader> userDataHeader = nullptr;
  /**
   * Contains the data encoding type for the SMS message
   */
  int msgEncoding = -1;
  bool msgEncodingSet = false;

  int msgType = 0;

  /**
   * Number of invalid bits in the last byte of data.
   */
  int paddingBits = 0;

  int numFields = 0;

  /**
   * Contains the user data of a SMS message
   * (See 3GPP2 C.S0015-B, v2, 4.5.2)
   */
  //uint8_t* payload = nullptr;
  //int payload_length = 0;
  std::vector<uint8_t> payload;
  std::string payloadStr;
  //char* payloadStr = nullptr;
private:
  static std::map<char, uint8_t> init();
};

#endif /* USERDATA_H_ */
