/*
 * Copyright (C) 2008 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef BEARERDATA_H_
#define BEARERDATA_H_
#include <cstdint>
#include <ctime>
#include <memory>
#include <string>
#include <vector>
#include "UserData.h"
#include "BitwiseOutputStream.h"
#include "CdmaSmsAddress.h"
#include "BitwiseInputStream.h"
#include "CdmaSmsCbProgramData.h"
#include "SmsCbCmasInfo.h"

class BearerData {
public:
  BearerData();
  virtual ~BearerData();
  /**
   * Supported message types for CDMA SMS messages
   * (See 3GPP2 C.S0015-B, v2.0, table 4.5.1-1)
   */
  static constexpr int MESSAGE_TYPE_DELIVER = 0x01;
  static constexpr int MESSAGE_TYPE_SUBMIT = 0x02;
  static constexpr int MESSAGE_TYPE_CANCELLATION = 0x03;
  static constexpr int MESSAGE_TYPE_DELIVERY_ACK = 0x04;
  static constexpr int MESSAGE_TYPE_USER_ACK = 0x05;
  static constexpr int MESSAGE_TYPE_READ_ACK = 0x06;
  static constexpr int MESSAGE_TYPE_DELIVER_REPORT = 0x07;
  static constexpr int MESSAGE_TYPE_SUBMIT_REPORT = 0x08;
  uint32_t messageType = 0;
  /**
   * 16-bit value indicating the message ID, which increments modulo 65536.
   * (Special rules apply for WAP-messages.)
   * (See 3GPP2 C.S0015-B, v2, 4.5.1)
   */
  uint32_t messageId = 0;
  /**
   * Supported priority modes for CDMA SMS messages
   * (See 3GPP2 C.S0015-B, v2.0, table 4.5.9-1)
   */
  static constexpr int PRIORITY_NORMAL = 0x0;
  static constexpr int PRIORITY_INTERACTIVE = 0x1;
  static constexpr int PRIORITY_URGENT = 0x2;
  static constexpr int PRIORITY_EMERGENCY = 0x3;

  bool priorityIndicatorSet = false;
  int priority = PRIORITY_NORMAL;

  /**
   * Supported privacy modes for CDMA SMS messages
   * (See 3GPP2 C.S0015-B, v2.0, table 4.5.10-1)
   */
  static constexpr int PRIVACY_NOT_RESTRICTED = 0x0;
  static constexpr int PRIVACY_RESTRICTED = 0x1;
  static constexpr int PRIVACY_CONFIDENTIAL = 0x2;
  static constexpr int PRIVACY_SECRET = 0x3;

  bool privacyIndicatorSet = false;
  int privacy = PRIVACY_NOT_RESTRICTED;

  /**
   * Supported alert priority modes for CDMA SMS messages
   * (See 3GPP2 C.S0015-B, v2.0, table 4.5.13-1)
   */
  static constexpr int ALERT_DEFAULT = 0x0;
  static constexpr int ALERT_LOW_PRIO = 0x1;
  static constexpr int ALERT_MEDIUM_PRIO = 0x2;
  static constexpr int ALERT_HIGH_PRIO = 0x3;

  bool alertIndicatorSet = false;
  int alert = ALERT_DEFAULT;

  /**
   * Supported display modes for CDMA SMS messages.  Display mode is
   * a 2-bit value used to indicate to the mobile station when to
   * display the received message.  (See 3GPP2 C.S0015-B, v2,
   * 4.5.16)
   */
  static constexpr int DISPLAY_MODE_IMMEDIATE = 0x0;
  static constexpr int DISPLAY_MODE_DEFAULT = 0x1;
  static constexpr int DISPLAY_MODE_USER = 0x2;

  bool displayModeSet = false;
  int displayMode = DISPLAY_MODE_DEFAULT;

  /**
   * Language Indicator values.  NOTE: the spec (3GPP2 C.S0015-B,
   * v2, 4.5.14) is ambiguous as to the meaning of this field, as it
   * refers to C.R1001-D but that reference has been crossed out.
   * It would seem reasonable to assume the values from C.R1001-F
   * (table 9.2-1) are to be used instead.
   */
  static constexpr int LANGUAGE_UNKNOWN = 0x00;
  static constexpr int LANGUAGE_ENGLISH = 0x01;
  static constexpr int LANGUAGE_FRENCH = 0x02;
  static constexpr int LANGUAGE_SPANISH = 0x03;
  static constexpr int LANGUAGE_JAPANESE = 0x04;
  static constexpr int LANGUAGE_KOREAN = 0x05;
  static constexpr int LANGUAGE_CHINESE = 0x06;
  static constexpr int LANGUAGE_HEBREW = 0x07;

  bool languageIndicatorSet = false;
  int language = LANGUAGE_UNKNOWN;

  /**
   * SMS Message Status Codes.  The first component of the Message
   * status indicates if an error has occurred and whether the error
   * is considered permanent or temporary.  The second component of
   * the Message status indicates the cause of the error (if any).
   * (See 3GPP2 C.S0015-B, v2.0, 4.5.21)
   */
  /* no-error codes */
  static constexpr int ERROR_NONE = 0x00;
  static constexpr int STATUS_ACCEPTED = 0x00;
  static constexpr int STATUS_DEPOSITED_TO_INTERNET = 0x01;
  static constexpr int STATUS_DELIVERED = 0x02;
  static constexpr int STATUS_CANCELLED = 0x03;
  /* temporary-error and permanent-error codes */
  static constexpr int ERROR_TEMPORARY = 0x02;
  static constexpr int STATUS_NETWORK_CONGESTION = 0x04;
  static constexpr int STATUS_NETWORK_ERROR = 0x05;
  static constexpr int STATUS_UNKNOWN_ERROR = 0x1F;
  /* permanent-error codes */
  static constexpr int ERROR_PERMANENT = 0x03;
  static constexpr int STATUS_CANCEL_FAILED = 0x06;
  static constexpr int STATUS_BLOCKED_DESTINATION = 0x07;
  static constexpr int STATUS_TEXT_TOO_LONG = 0x08;
  static constexpr int STATUS_DUPLICATE_MESSAGE = 0x09;
  static constexpr int STATUS_INVALID_DESTINATION = 0x0A;
  static constexpr int STATUS_MESSAGE_EXPIRED = 0x0D;
  /* undefined-status codes */
  static constexpr int ERROR_UNDEFINED = 0xFF;
  static constexpr int STATUS_UNDEFINED = 0xFF;

  bool messageStatusSet = false;
  int errorClass = ERROR_UNDEFINED;
  int messageStatus = STATUS_UNDEFINED;

  /**
   * 1-bit value that indicates whether a User Data Header (UDH) is present.
   * (See 3GPP2 C.S0015-B, v2, 4.5.1)
   *
   * NOTE: during encoding, this value will be set based on the
   * presence of a UDH in the structured data, any existing setting
   * will be overwritten.
   */
  bool hasUserDataHeader = false;

  /**
   * provides the information for the user data
   * (e.g. padding bits, user data, user data header, etc)
   * (See 3GPP2 C.S.0015-B, v2, 4.5.2)
   */
  std::shared_ptr<UserData> userData = nullptr;

  /**
   * The User Response Code subparameter is used in the SMS User
   * Acknowledgment Message to respond to previously received short
   * messages. This message center-specific element carries the
   * identifier of a predefined response. (See 3GPP2 C.S.0015-B, v2,
   * 4.5.3)
   */
  bool userResponseCodeSet = false;
  uint32_t userResponseCode = 0;

  struct tm msgCenterTimeStamp;
  struct tm validityPeriodAbsolute;
  struct tm deferredDeliveryTimeAbsolute;

  static int cdmaBcdByteToInt(uint8_t b);
  static void fromByteArray(std::vector<uint8_t> data, struct tm& ts);
  /**
   * Relative time is specified as one byte, the value of which
   * falls into a series of ranges, as specified below.  The idea is
   * that shorter time intervals allow greater precision -- the
   * value means minutes from zero until the MINS_LIMIT (inclusive),
   * upon which it means hours until the HOURS_LIMIT, and so
   * forth. (See 3GPP2 C.S0015-B, v2, 4.5.6-1)
   */
  static constexpr int RELATIVE_TIME_MINS_LIMIT = 143;
  static constexpr int RELATIVE_TIME_HOURS_LIMIT = 167;
  static constexpr int RELATIVE_TIME_DAYS_LIMIT = 196;
  static constexpr int RELATIVE_TIME_WEEKS_LIMIT = 244;
  static constexpr int RELATIVE_TIME_INDEFINITE = 245;
  static constexpr int RELATIVE_TIME_NOW = 246;
  static constexpr int RELATIVE_TIME_MOBILE_INACTIVE = 247;
  static constexpr int RELATIVE_TIME_RESERVED = 248;

  bool validityPeriodRelativeSet = false;
  uint32_t validityPeriodRelative = 0;
  bool deferredDeliveryTimeRelativeSet = false;
  uint32_t deferredDeliveryTimeRelative = 0;

  /**
   * The Reply Option subparameter contains 1-bit values which
   * indicate whether SMS acknowledgment is requested or not.  (See
   * 3GPP2 C.S0015-B, v2, 4.5.11)
   */
  bool userAckReq = false;
  bool deliveryAckReq = false;
  bool readAckReq = false;
  bool reportReq = false;

  /**
   * The Number of Messages subparameter (8-bit value) is a decimal
   * number in the 0 to 99 range representing the number of messages
   * stored at the Voice Mail System. This element is used by the
   * Voice Mail Notification service.  (See 3GPP2 C.S0015-B, v2,
   * 4.5.12)
   */
  uint32_t numberOfMessages = 0;

  /**
   * The Message Deposit Index subparameter is assigned by the
   * message center as a unique index to the contents of the User
   * Data subparameter in each message sent to a particular mobile
   * station. The mobile station, when replying to a previously
   * received short message which included a Message Deposit Index
   * subparameter, may include the Message Deposit Index of the
   * received message to indicate to the message center that the
   * original contents of the message are to be included in the
   * reply.  (See 3GPP2 C.S0015-B, v2, 4.5.18)
   */
  uint32_t depositIndex = 0;
  /**
   * 4-bit or 8-bit value that indicates the number to be dialed in reply to a
   * received SMS message.
   * (See 3GPP2 C.S0015-B, v2, 4.5.15)
   */
  //CdmaSmsAddress* callbackNumber = nullptr;
  std::shared_ptr<CdmaSmsAddress> callbackNumber = nullptr;
  /**
   * CMAS warning notification information.
   * @see #decodeCmasUserData(BearerData, int)
   */
  std::shared_ptr<SmsCbCmasInfo> cmasWarningInfo = nullptr;
  std::shared_ptr<std::list<std::shared_ptr<CdmaSmsCbProgramData>>> serviceCategoryProgramData;
  static void encodeMessageId(BearerData* bData,
      std::shared_ptr<BitwiseOutputStream> outStream);
  static uint32_t countAsciiSeptets(char* msg, bool force);

  /**
   * Create serialized representation for BearerData object.
   * (See 3GPP2 C.R1001-F, v1.0, section 4.5 for layout details)
   *
   * @param bData an instance of BearerData.
   *
   * @return uint8_t array of raw encoded SMS bearer data.
   */
  static std::vector<uint8_t> encode(BearerData* bData);
  static std::shared_ptr<BearerData> decode(std::vector<uint8_t> smsData);
  static std::shared_ptr<BearerData> decode(std::vector<uint8_t> smsData,
      int serviceCategory);
  std::string toString();
private:
  /**
   * Bearer Data Subparameter Identifiers
   * (See 3GPP2 C.S0015-B, v2.0, table 4.5-1)
   * NOTE: Commented subparameter types are not implemented.
   */
  constexpr static uint8_t SUBPARAM_MESSAGE_IDENTIFIER = 0x00;
  constexpr static uint8_t SUBPARAM_USER_DATA = 0x01;
  constexpr static uint8_t SUBPARAM_USER_RESPONSE_CODE = 0x02;
  constexpr static uint8_t SUBPARAM_MESSAGE_CENTER_TIME_STAMP = 0x03;
  constexpr static uint8_t SUBPARAM_VALIDITY_PERIOD_ABSOLUTE = 0x04;
  constexpr static uint8_t SUBPARAM_VALIDITY_PERIOD_RELATIVE = 0x05;
  constexpr static uint8_t SUBPARAM_DEFERRED_DELIVERY_TIME_ABSOLUTE = 0x06;
  constexpr static uint8_t SUBPARAM_DEFERRED_DELIVERY_TIME_RELATIVE = 0x07;
  constexpr static uint8_t SUBPARAM_PRIORITY_INDICATOR = 0x08;
  constexpr static uint8_t SUBPARAM_PRIVACY_INDICATOR = 0x09;
  constexpr static uint8_t SUBPARAM_REPLY_OPTION = 0x0A;
  constexpr static uint8_t SUBPARAM_NUMBER_OF_MESSAGES = 0x0B;
  constexpr static uint8_t SUBPARAM_ALERT_ON_MESSAGE_DELIVERY = 0x0C;
  constexpr static uint8_t SUBPARAM_LANGUAGE_INDICATOR = 0x0D;
  constexpr static uint8_t SUBPARAM_CALLBACK_NUMBER = 0x0E;
  constexpr static uint8_t SUBPARAM_MESSAGE_DISPLAY_MODE = 0x0F;
  //constexpr static uint8_t SUBPARAM_MULTIPLE_ENCODING_USER_DATA      = 0x10;
  constexpr static uint8_t SUBPARAM_MESSAGE_DEPOSIT_INDEX = 0x11;
  constexpr static uint8_t SUBPARAM_SERVICE_CATEGORY_PROGRAM_DATA = 0x12;
  constexpr static uint8_t SUBPARAM_SERVICE_CATEGORY_PROGRAM_RESULTS = 0x13;
  constexpr static uint8_t SUBPARAM_MESSAGE_STATUS = 0x14;
  //constexpr static uint8_t SUBPARAM_TP_FAILURE_CAUSE                 = 0x15;
  //constexpr static uint8_t SUBPARAM_ENHANCED_VMN                     = 0x16;
  //constexpr static uint8_t SUBPARAM_ENHANCED_VMN_ACK                 = 0x17;

  // All other values after this are reserved.
  constexpr static uint8_t SUBPARAM_ID_LAST_DEFINED = 0x17;
  static void encodeUserData(BearerData* bData,
      std::shared_ptr<BitwiseOutputStream> outStream);
  static void encodeUserDataPayload(std::shared_ptr<UserData> uData);
  static void encodeReplyOption(BearerData* bData,
      std::shared_ptr<BitwiseOutputStream> outStream);
  static std::vector<uint8_t> encodeDtmfSmsAddress(std::string address);
  static void encodeCdmaSmsAddress(std::shared_ptr<CdmaSmsAddress> addr);
  static void encodeCallbackNumber(BearerData* bData,
      std::shared_ptr<BitwiseOutputStream> outStream);
  static void encodeMsgStatus(BearerData* bData,
      std::shared_ptr<BitwiseOutputStream> outStream);
  static void encodeMsgCount(BearerData* bData,
      std::shared_ptr<BitwiseOutputStream> outStream);
  static void encodeValidityPeriodRel(BearerData* bData,
      std::shared_ptr<BitwiseOutputStream> outStream);
  static void encodePrivacyIndicator(BearerData* bData,
      std::shared_ptr<BitwiseOutputStream> outStream);
  static void encodeLanguageIndicator(BearerData* bData,
      std::shared_ptr<BitwiseOutputStream> outStream);
  static void encodeDisplayMode(BearerData* bData,
      std::shared_ptr<BitwiseOutputStream> outStream);
  static void encodePriorityIndicator(BearerData* bData,
      std::shared_ptr<BitwiseOutputStream> outStream);
  static void encodeMsgDeliveryAlert(BearerData* bData,
      std::shared_ptr<BitwiseOutputStream> outStream);
  static bool decodeMessageId(std::shared_ptr<BearerData> bData,
      std::shared_ptr<BitwiseInputStream> inStream);
  static bool decodeUserData(std::shared_ptr<BearerData> bData,
      std::shared_ptr<BitwiseInputStream> inStream);
  static bool decodeUserResponseCode(std::shared_ptr<BearerData> bData,
      std::shared_ptr<BitwiseInputStream> inStream);
  static bool decodeReplyOption(std::shared_ptr<BearerData> bData,
      std::shared_ptr<BitwiseInputStream> inStream);
  static bool decodeMsgCount(std::shared_ptr<BearerData> bData,
      std::shared_ptr<BitwiseInputStream> inStream);
  static bool decodeCallbackNumber(std::shared_ptr<BearerData> bData,
      std::shared_ptr<BitwiseInputStream> inStream);
  static void decodeSmsAddress(std::shared_ptr<CdmaSmsAddress> addr);
  static std::string decodeDtmfSmsAddress(std::vector<uint8_t> rawData,
      int numFields);
  static bool decodeMsgStatus(std::shared_ptr<BearerData> bData,
      std::shared_ptr<BitwiseInputStream> inStream);
  static bool decodeMsgCenterTimeStamp(std::shared_ptr<BearerData> bData,
      std::shared_ptr<BitwiseInputStream> inStream);
  static bool decodeValidityAbs(std::shared_ptr<BearerData> bData,
      std::shared_ptr<BitwiseInputStream> inStream);
  static bool decodeValidityRel(std::shared_ptr<BearerData> bData,
      std::shared_ptr<BitwiseInputStream> inStream);
  static bool decodeDeferredDeliveryAbs(std::shared_ptr<BearerData> bData,
      std::shared_ptr<BitwiseInputStream> inStream);
  static bool decodeDeferredDeliveryRel(std::shared_ptr<BearerData> bData,
      std::shared_ptr<BitwiseInputStream> inStream);
  static bool decodePrivacyIndicator(std::shared_ptr<BearerData> bData,
      std::shared_ptr<BitwiseInputStream> inStream);
  static bool decodeLanguageIndicator(std::shared_ptr<BearerData> bData,
      std::shared_ptr<BitwiseInputStream> inStream);
  static bool decodeDisplayMode(std::shared_ptr<BearerData> bData,
      std::shared_ptr<BitwiseInputStream> inStream);
  static bool decodePriorityIndicator(std::shared_ptr<BearerData> bData,
      std::shared_ptr<BitwiseInputStream> inStream);
  static bool decodeMsgDeliveryAlert(std::shared_ptr<BearerData> bData,
      std::shared_ptr<BitwiseInputStream> inStream);
  static bool decodeDepositIndex(std::shared_ptr<BearerData> bData,
      std::shared_ptr<BitwiseInputStream> inStream);
  static bool decodeServiceCategoryProgramData(
      std::shared_ptr<BearerData> bData,
      std::shared_ptr<BitwiseInputStream> inStream);
  static void decodeUserDataPayload(std::shared_ptr<UserData> userData,
      bool hasUserDataHeader);
  static int getBitsForNumFields(int msgEncoding, int numFields);
  static std::string decode7bitAscii(std::vector<uint8_t> data, int offset,
      int numFields);
  static std::string decode7bitGsm(std::vector<uint8_t> data, int offset,
      int numFields);
  static std::string decodeGsmDcs(std::vector<uint8_t> data, int offset,
      int numFields, int msgType);
  static bool decodeReserved(std::shared_ptr<BearerData> bData,
      std::shared_ptr<BitwiseInputStream> inStream, int subparamId);
  static bool isCmasAlertCategory(int category);
  static void decodeCmasUserData(std::shared_ptr<BearerData> bData,
      int serviceCategory);
  static int serviceCategoryToCmasMessageClass(int serviceCategory);
  static void decodeIs91(std::shared_ptr<BearerData> bData);
  static void decodeIs91VoicemailStatus(std::shared_ptr<BearerData> bData);
  static void decodeIs91Cli(std::shared_ptr<BearerData> bData);
  static void decodeIs91ShortMessage(std::shared_ptr<BearerData> bData);
  static std::string decodeLatin(std::vector<uint8_t> data, int offset, int numFields);
  static std::string decodeUtf16(std::vector<uint8_t> data, int offset, int numFields);
  static std::string decodeUtf8(std::vector<uint8_t> data, int offset, int numFields);
  static std::vector<uint8_t> encode7bitAscii(std::string msgs, bool force);
  static std::vector<uint8_t> encodeUtf16(std::string msg);
  static void encodeEmsUserDataPayload(std::shared_ptr<UserData> uData);
  static void encode7bitEms(std::shared_ptr<UserData> uData, std::vector<uint8_t> udhData, bool force);
  static void encode16bitEms(std::shared_ptr<UserData> uData, std::vector<uint8_t> udhData);
  static void encodeOctetEms(std::shared_ptr<UserData> uData, std::vector<uint8_t> udhData);
  class Gsm7bitCodingResult {
  public:
     int septets;
     std::vector<uint8_t> data;
  };
  static BearerData::Gsm7bitCodingResult encode7bitGsm(std::string msg, int septetOffset, bool force);
};

#endif /* BEARERDATA_H_ */
