#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <binder/Parcel.h>
#include <log/log.h>
#include "lynq_data.h"
#include <cutils/jstring.h>
#include <pthread.h>
#include "liblog/lynq_deflog.h"
#include <sys/time.h>
#include <include/lynq_uci.h>
#include <errno.h>
#include <vector>
#include "lynq_data_urc.h"

#define LYNQ_SERVICE_PORT 8088
#define LYNQ_RIL_FWK_IP "127.0.0.1"

#define LYNQ_REC_BUF 8192
#define LYNQ_REQUEST_PARAM_BUF 8192
#define LYQN_SEDN_BUF 1024*8+sizeof(int)*3
#define USER_LOG_TAG "LYNQ_DATA"

#define LYNQ_DATA_UCI_BUF 258
#define LYNQ_DATA_TIME_OUT 1000*120
#define TELEPHONY_RESTART 10


using ::android::Parcel;
typedef struct{
    int uToken;
    int request;
    int paramLen;
    char param[LYNQ_REQUEST_PARAM_BUF];
}lynq_client_t;
//xy.he add for bug-view-73 on 2024-08-26 start
#ifdef MODE_DSDS
typedef struct{
    int uToken;
    int request;
    int paramLen;
    char param[LYNQ_REQUEST_PARAM_BUF];
    char slot_status;
    char slot;
}lynq_client_dual_t;

#endif
//xy.he add for bug-view-73 on 2024-08-26 end
typedef enum{
    LYNQ_E_CARDSTATE_ERROR=8000,
    /* The voice service state is out of service*/
    LYNQ_E_STATE_OUT_OF_SERVICE=8001,
    /* The voice service state is EMERGENCY_ONLY*/
    LYNQ_E_STATE_EMERGENCY_ONLY=8002,
    /* The radio power is power off*/
    LYNQ_E_STATE_POWER_OFF=8003,
    LYNQ_E_TIME_OUT=8004,
    /*create or open sms DB fail */
    LYNQ_E_SMS_DB_FAIL=8005,
    /*Failed to execute sql statement*/
    LYNQ_E_SMS_SQL_FAIL = 8006,
    LYNQ_E_SMS_NOT_FIND = 8007,
    LYNQ_E_GET_RESP_FAIL = 8008,
    LYNQ_E_NOT_THIS_APN = 8087,
    LYNQ_E_NOT_ANY_APN = 8088,
    LYNQ_E_MD_NOT_READY = 8089,
    /* The logic conflict*/
    LYNQ_E_CONFLICT=9000,
    /*Null anomaly*/
    LYNQ_E_NULL_ANONALY=9001
}LYNQ_E;

typedef enum {
    PDN_IDLE,
    PDN_CONNECTING,
    PDN_CONNECTED,
    PDN_DISCONNECTING,
    PDN_DISCONNECTED,
    PDN_RETRYING,
    PDN_FAILED,
    PDN_SCANNING,
    PDN_TIMEOUT_CANCEL,
} RIL_Data_Call_PdnState;

int lynq_client_sockfd = 0;
int Global_uToken = 0;
struct sockaddr_in lynq_data_socket_server_addr;
int lynq_data_socket_server_addr_len;

int lynq_data_call_change_id = -1;
int lynq_telephony_restart_g = 0;
pthread_t lynq_data_tid =-1;
static pthread_mutex_t s_data_call_state_change_mutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t s_data_call_state_change_cond = PTHREAD_COND_INITIALIZER;

static pthread_mutex_t s_data_call_deactived_mutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t s_data_call_deactived_cond = PTHREAD_COND_INITIALIZER;


static pthread_mutex_t s_lynq_apn_change_mutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t s_lynq_apn_change_cond = PTHREAD_COND_INITIALIZER;
static pthread_mutex_t s_lynq_urc_vector_mutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t s_lynq_urc_vector_cond = PTHREAD_COND_INITIALIZER;
/**g_lynq_data_sendto_mutex
* @brief mark data send request mutex
*/
static pthread_mutex_t g_lynq_data_sendto_mutex;
/*This value data the state of the wait*/
static int data_waiting_status = 0;
/*The value indicates that 8085 error occurs in data*/
static int data_invaild_error = 0;
/*This value ensure the data call timing is correct*/
static int data_timelimit = 0;

/**g_lynq_data_init_flag
* @brief mark data initialization state
* 0:deinit status
* 1:init state
*/
static int g_lynq_data_init_flag = 0;
/**g_lynq_apn_result
* @brief temp of apn result info
*/
char g_lynq_apn_result[1024] = {};
int g_data_call_timeout_value = LYNQ_DATA_TIME_OUT;

static std::vector<int> s_data_urc_wait_list;

typedef struct
{
    char apn[LYNQ_APN_MAX_LEN];
    char apnType[LYNQ_APN_TYPE_MAX_LEN];
    char ifaceName[LYNQ_IFACE_NAME_MAX_LEN];
    int hasUsed;
    int hasTimeout;
    int status;
    char statusApnType[LYNQ_APN_TYPE_MAX_LEN];
}lynq_apn_t;
lynq_apn_t lynq_apn_table[LYNQ_APN_CHANNEL_MAX] = {};
lynq_data_call_response_v11_t lynq_data_call_lists[LYNQ_APN_CHANNEL_MAX] = {};
int lynq_data_call = 0;

int radio_switch(int status);

int getLynqApnID(char apnType[])
{
    int ret = 0;
    int len = 0;
    for(ret;ret<LYNQ_APN_CHANNEL_MAX;ret++)
    {
        len = strlen(apnType);
        LYINFLOG("apn_table[%d].apnType:%s,input apntype:%s,len:%d",ret,lynq_apn_table[ret].apnType,apnType,len);
        if(strncmp(lynq_apn_table[ret].apnType,apnType,len)==0)
        {
            return ret;
        }
    }
    return -1;
}

int  getDeactApnID(char apnType[])
{
    int ret = 0;
    int len = 0;
    for(ret;ret<LYNQ_APN_CHANNEL_MAX;ret++)
    {
        len = strlen(apnType);
        LYINFLOG("apn_table[%d].apnType:%s,input apntype:%s,len:%d",ret,lynq_apn_table[ret].apnType,apnType,len);
        if(strncmp(lynq_apn_table[ret].statusApnType,apnType,len)==0)
        {
            return ret;
        }
    }
    return -1;
}

void updateApnTable(lynq_apn_t *apn_table,char apn[],char apntype[],char ifaceName[])
{
    LYDBGLOG("[updateApnTable] apn:%s,apntype:%s,ifaceName:%s",apn,apntype,ifaceName);
    if(apn_table==NULL)
    {
        LYERRLOG("apn_table is null");
        return;
    }
    memcpy(apn_table->apn,apn,strlen(apn)+1);
    memcpy(apn_table->apnType,apntype,strlen(apntype)+1);
    memcpy(apn_table->ifaceName,ifaceName,strlen(ifaceName)+1);
    apn_table->hasTimeout = 0;
    apn_table->hasUsed = 1;
    return;
}

void cleanOnceApnTable(int apnId)
{
    LYDBGLOG("%s:apn id:%d",__FUNCTION__,apnId);
    if((apnId < 0) || (apnId > LYNQ_APN_CHANNEL_MAX-1))
    {
        LYERRLOG("apn id is invalid!!!");
        return;
    }
    lynq_apn_table[apnId].hasTimeout = 0;
    lynq_apn_table[apnId].hasUsed = 0;
    memcpy(lynq_apn_table[apnId].statusApnType,lynq_apn_table[apnId].apnType,strlen(lynq_apn_table[apnId].apnType));
    bzero(lynq_apn_table[apnId].apn,LYNQ_APN_MAX_LEN);
    bzero(lynq_apn_table[apnId].apnType,LYNQ_APN_TYPE_MAX_LEN);
    bzero(lynq_apn_table[apnId].ifaceName,LYNQ_IFACE_NAME_MAX_LEN);
    lynq_apn_table[apnId].status = 32;
    return;
}
void cleanDeactApn(int apnId)
{
    LYDBGLOG("%s:apn id:%d",__FUNCTION__,apnId);
    if((apnId < 0) || (apnId > LYNQ_APN_CHANNEL_MAX-1))
    {
        LYERRLOG("apn id is invalid!!!");
        return;
    }
    lynq_apn_table[apnId].status = 0;
    bzero(lynq_apn_table[apnId].statusApnType,LYNQ_APN_TYPE_MAX_LEN);
}

void updateDeactApn(int apnId,int pdnState)
{
    LYDBGLOG("%s:apn id:%d",__FUNCTION__,apnId);
    if((apnId < 0) || (apnId > LYNQ_APN_CHANNEL_MAX-1))
    {
        LYERRLOG("apn id is invalid!!!");
        return;
    }
    lynq_apn_table[apnId].status = pdnState;
}


int getUnusedElement()
{
    if (lynq_apn_table == NULL)
    {
        LYERRLOG("get UnusedElemnt apn_table is null");
        return -1;
    }
    for(int i=0;i < LYNQ_APN_CHANNEL_MAX; i++)
    {
        if(lynq_apn_table[i].hasUsed!=1)
        {
            return i;
        }
    }
	LYERRLOG("None of get unused Element");
    return -1;
}
int updateApn(char apnType[])
{
    int ret = 0;
    ret = getUnusedElement();
    if(ret >= 0)
    {
        memcpy(lynq_apn_table[ret].apnType,apnType,strlen(apnType)+1);
        lynq_apn_table[ret].hasUsed = 1;
    }
    return ret;
}
//@return 
//other:this apn has been used in apn table
//-1:this apn not has been used in apn table
int check_used_apn(char apnType[])
{
    LYINFLOG("check_used_apn.apnType:%s",apnType);
    int ret = 0;
    int len = 0;
    for(ret;ret<LYNQ_APN_CHANNEL_MAX;ret++)
    {
        len = strlen(apnType);
        if(strncmp(lynq_apn_table[ret].apnType,apnType,len)==0)
        {
            LYINFLOG("check_used_apn.apnType:%s in apn table",apnType);
            return ret;
        }
    }
    return -1;
}
int get_handle(char apnType[])
{
    int call_id = 0;
    call_id = check_used_apn(apnType);
    if(call_id == -1)
    {
        call_id = updateApn(apnType);
    }
    LYINFLOG("handle:%d",call_id);
    return call_id;
}

int handleCheck(int handle)
{
    if ((handle >= 0) && (handle  <= 6))
    {
        if (lynq_apn_table[handle].hasUsed == 1)
        {
            return 0;
        }
        return -1;
    }
    return -1;
}
int waitApnResult()
{
    int ret = 0;
    LYINFLOG("start wait apn result!!!");
    int sec = 0;
    int usec = 0;
    struct timeval now;
    struct timespec timeout;
    gettimeofday(&now, NULL);
    sec = 20000 / 1000;
    usec = 20000 % 1000;
    timeout.tv_sec = now.tv_sec + sec;
    timeout.tv_nsec = now.tv_usec * 1000 + usec * 1000000;
    pthread_mutex_lock(&s_lynq_apn_change_mutex);
    ret = pthread_cond_timedwait(&s_lynq_apn_change_cond, &s_lynq_apn_change_mutex, &timeout);
    pthread_mutex_unlock(&s_lynq_apn_change_mutex);
    return ret;
}

void sendSignalApnChange()
{
    LYINFLOG("start send Signal Apn Change");
    pthread_mutex_lock(&s_lynq_apn_change_mutex);
    pthread_cond_signal(&s_lynq_apn_change_cond);
    pthread_mutex_unlock(&s_lynq_apn_change_mutex);
    return;
}

int waitPdnChange()
{
    int ret = 0;
    pthread_mutex_lock(&s_lynq_urc_vector_mutex);
    ret = pthread_cond_wait(&s_lynq_urc_vector_cond,&s_lynq_urc_vector_mutex);
    pthread_mutex_unlock(&s_lynq_urc_vector_mutex);
    return ret;
}
int waitDataCallstateChange(int mtime)
{
    int ret = 0;
    int sec = 0;
    int usec = 0;
    struct timeval now;
    struct timespec timeout;
    gettimeofday(&now,NULL);
    sec = mtime/1000;
    usec = mtime%1000;
    timeout.tv_sec = now.tv_sec+sec;
    timeout.tv_nsec = now.tv_usec*1000+usec*1000000;
    pthread_mutex_lock(&s_data_call_state_change_mutex);
    ret = pthread_cond_timedwait(&s_data_call_state_change_cond,&s_data_call_state_change_mutex,&timeout);
    pthread_mutex_unlock(&s_data_call_state_change_mutex);
    return ret;
}
void sendSignalDataCallStateChange()
{
    pthread_mutex_lock(&s_data_call_state_change_mutex);
    pthread_cond_signal(&s_data_call_state_change_cond);
    pthread_mutex_unlock(&s_data_call_state_change_mutex);
    return;
}
int waitDeactived(int mtime)
{
    int ret = 0;
    int sec = 0;
    int usec = 0;
    struct timeval now;
    struct timespec timeout;
    gettimeofday(&now,NULL);
    sec = mtime/1000;
    usec = mtime%1000;
    timeout.tv_sec = now.tv_sec+sec;
    timeout.tv_nsec = now.tv_usec*1000+usec*1000000;
    pthread_mutex_lock(&s_data_call_deactived_mutex);
    ret = pthread_cond_timedwait(&s_data_call_deactived_cond,&s_data_call_deactived_mutex,&timeout);
    pthread_mutex_unlock(&s_data_call_deactived_mutex);
    return ret;
}
void sendSignalDeactvied()
{
    pthread_mutex_lock(&s_data_call_deactived_mutex);
    pthread_cond_signal(&s_data_call_deactived_cond);
    pthread_mutex_unlock(&s_data_call_deactived_mutex);
    return;
}

void sendSignalPdnChange()
{
    pthread_mutex_lock(&s_lynq_urc_vector_mutex);
    pthread_cond_signal(&s_lynq_urc_vector_cond);
    pthread_mutex_unlock(&s_lynq_urc_vector_mutex);
    return;
}

int get_response(int sockfd,Parcel &p)
{
    int len = 0;
    char recvline[LYNQ_REC_BUF];
    bzero(recvline,LYNQ_REC_BUF);
    /* receive data from server */
    len = recvfrom(sockfd,recvline,LYNQ_REC_BUF,0,(struct sockaddr *)&lynq_data_socket_server_addr,(socklen_t *)&lynq_data_socket_server_addr_len);
    //len = read(sockfd, recvline, LYNQ_REC_BUF);
    if(len == -1)
    {
        LYERRLOG("get_response fail,errno:%d",errno);
        return -1;
    }
    if (recvline != NULL) {
        p.setData((uint8_t *)recvline,len); // p.setData((uint8_t *) buffer, buflen);
        p.setDataPosition(0);
    }
    return 0;
}
/**@brief get utoken in range 0 to 10000
* @return utoken
*/
int get_utoken()
{
    return (Global_uToken++)%10000;/*0-10000*/
}
/**@brief wait response with expected token and write msg to parcel in some time
* @param fd [IN]: socket fd
* @param p [OUT]: quote the parcel,if return success need delete p.
* @param token [IN]: the expected token for the response msg
* @return
*  0:success
*  other:failure
*/
int wait_response(int sockfd,Parcel *& p,int utoken)
{
    int len = 0;
    int flag = 1;
    int count = 0;
    int in_utoken = -1;
    int resp_type = -1;
    Parcel *temp = NULL;
    char recvline[LYNQ_REC_BUF];
    //Sometimes the socket is abnormal, causing the socket buffer to store the response of the last request.
    //Here it does not return until the response corresponding to the request is read.
    while(flag && (count < 20))//why?
    {
        bzero(recvline,LYNQ_REC_BUF);
        count++;
        LYINFLOG("wait_response,count:%d",count);
        len = recvfrom(sockfd,recvline,LYNQ_REC_BUF,0,(struct sockaddr *)&lynq_data_socket_server_addr,(socklen_t *)&lynq_data_socket_server_addr_len);
        if(len == -1)
        {
            LYERRLOG("wait_response fail,errno:%d",errno);
            return LYNQ_E_GET_RESP_FAIL;
        }
        if (len != 0)
        {
            temp = new Parcel;
            int i = 0;
            while((NULL == temp) && (i < 100))
            {
                usleep(1000);
                temp = new Parcel;
                i++;
            }
            if((i >= 100) || (NULL == temp))
            {
                LYERRLOG("wait_response i:%d",i);
                return LYNQ_E_GET_RESP_FAIL;
            }
            temp->setData((uint8_t *)recvline,len); // p.setData((uint8_t *) buffer, buflen);
            temp->setDataPosition(0);
            temp->readInt32(&resp_type);
            temp->readInt32(&in_utoken);
            LYINFLOG("[%s]in_utoken:%d,utoken:%d",__FUNCTION__,in_utoken,utoken);
            if (in_utoken != utoken)
            {
                delete temp;
                temp = NULL;
                in_utoken = 0;
                continue;
            }
            temp->setDataPosition(0);
            p = temp;
            flag = 0;
            return 0;
        }
        else
        {
            LYERRLOG("recvline is null,errno:%d",errno);
            return LYNQ_E_GET_RESP_FAIL;
        }
    }
    return LYNQ_E_GET_RESP_FAIL;
}

int JumpHeader(Parcel *p,int *resp_type,int *utoken,int *request,int *slot_id,int *error)
{
    if(NULL == p)
    {
        LYERRLOG("JumpHeader is null");
        return -1;
    }
    if(p->dataAvail() > 0)
    {
        p->readInt32(resp_type);
        p->readInt32(utoken);
        p->readInt32(request);
        p->readInt32(slot_id);
        p->readInt32(error);
        return 0;
    }
    else
    {
        return -1;
    }
}
void free_parcel(Parcel *p)
{
    if(p)
    {
        delete p;
        p = NULL;
    }
}
//xy.he add for bug-view-73 on 2024-08-26 start
int send_request_dual(int sockfd,lynq_client_dual_t *client_tmp,int size)
{
    int ret=0;
    LYINFLOG("client size = %d",size);
    ret = sendto(sockfd,client_tmp,size,0,(struct sockaddr *)&lynq_data_socket_server_addr,lynq_data_socket_server_addr_len);
    LYINFLOG("sendto size = %d",ret);
    if(ret==-1)
    {
        LYERRLOG("%s:errno code:%d",__FUNCTION__,errno);
        return -1;
    }
    return 0;
}
//xy.he add for bug-view-73 on 2024-08-26 end
int send_request(int sockfd,lynq_client_t *client_tmp)
{
    int ret=0;
    ret = sendto(sockfd,client_tmp,LYQN_SEDN_BUF,0,(struct sockaddr *)&lynq_data_socket_server_addr,lynq_data_socket_server_addr_len);
    if(ret==-1)
    {
        LYERRLOG("%s:errno code:%d",__FUNCTION__,errno);
        return -1;
    }
    return 0;
}
static char *strdupReadString(Parcel &p) {
    size_t stringlen;
    const char16_t *s16;
    s16 = p.readString16Inplace(&stringlen);
    return strndup16to8(s16, stringlen);
}
static char *strdupReadString_p(Parcel *p) {
    size_t stringlen;
    const char16_t *s16;
    s16 = p->readString16Inplace(&stringlen);
    return strndup16to8(s16, stringlen);
}

/*Warren add for T800 platform 2021/11/19 start*/
int lynq_socket_client_start()
{
    int ret;
    struct timeval timeOut;
    struct sockaddr_in liblynq_data_socket;

    ret = 0;
    timeOut.tv_sec = 30;
    timeOut.tv_usec = 0; 
    bzero(&liblynq_data_socket, sizeof(liblynq_data_socket));
    bzero(&lynq_data_socket_server_addr, sizeof(lynq_data_socket_server_addr));
    
    //set this lib socket config 
    liblynq_data_socket.sin_family = AF_INET;
    liblynq_data_socket.sin_addr.s_addr = inet_addr(LYNQ_RIL_FWK_IP);

    //set ril service socket config
    lynq_data_socket_server_addr.sin_family = AF_INET;
    lynq_data_socket_server_addr.sin_port = htons(LYNQ_SERVICE_PORT);
    lynq_data_socket_server_addr.sin_addr.s_addr = inet_addr(LYNQ_RIL_FWK_IP);
    lynq_data_socket_server_addr_len = sizeof(lynq_data_socket_server_addr);

    lynq_client_sockfd = socket(AF_INET, SOCK_DGRAM, 0);
    ret = bind(lynq_client_sockfd, (struct sockaddr *)&liblynq_data_socket, sizeof(liblynq_data_socket));
    if (-1 == ret)
    {
        LYERRLOG("liblynq_data_socket bind fail,errno:%d",errno);
        return -1;
    }
    if (setsockopt(lynq_client_sockfd, SOL_SOCKET, SO_RCVTIMEO, &timeOut, sizeof(timeOut)) < 0) 
    { 
        LYERRLOG("time out setting failed,errno:%d",errno);
        return -1;
    }
    return 0;
}

bool is_support_urc(int urc_id)
{
    switch(urc_id)
    {
        case LYNQ_URC_DATA_CALL_STATUS_IND:

        case LYNQ_URC_MODIFY_APNDB:
        case LYNQ_URC_RESET_APNDB:
        case LYNQ_TELEPHONY_RESTART:
            return true;                
        default:
            return false;
    }    
}

int printf_apn_table()
{
    int ret = 0;
    if (lynq_apn_table == NULL)
    {
        LYERRLOG("apn table is null");
        return -1;
    }
    for(ret;ret<LYNQ_APN_CHANNEL_MAX;ret++)
    {
        LYINFLOG("[%s][%d] apn=%s ;apntype=%s ;ifname=%s ;hasTimeout = %d ; hasUsed = %d,status = %d,statusDeactApn = %s",__FUNCTION__,ret, \
        lynq_apn_table[ret].apn,lynq_apn_table[ret].apnType,lynq_apn_table[ret].ifaceName, \
        lynq_apn_table[ret].hasTimeout,lynq_apn_table[ret].hasUsed,lynq_apn_table[ret].status,lynq_apn_table[ret].statusApnType);
    }
    return 0;
}

void printf_apn_table_debug(const char *fun,int line)
{
    LYINFLOG("[%s][%d]apn_table msg",fun,line);
    printf_apn_table();
}

void urc_msg_process(Parcel *p)
{
    int len;
    int resp_type;
    int urcid;
    int slot_id;
    int check_count = 0;
    static int apnId=-1;

    int pdnState = 0;
    char apn[LYNQ_APN_MAX_LEN];
    char apnType[LYNQ_APN_TYPE_MAX_LEN];
    char ifaceName[LYNQ_IFACE_NAME_MAX_LEN];
    char *urc_msg = NULL;

    int size = p->dataSize();
    p->readInt32(&resp_type);
    p->readInt32(&urcid);
    p->readInt32(&slot_id);
    LYINFLOG("data lib recv urc:resp_type=%d,urcid=%d,slot_id=%d,size=%d\n",resp_type,urcid,slot_id,size);
    switch(urcid)
    {
        case LYNQ_URC_DATA_CALL_STATUS_IND:
            p->readInt32(&pdnState);
            bzero(apn,LYNQ_APN_MAX_LEN);
            bzero(apnType,LYNQ_APN_TYPE_MAX_LEN);
            bzero(ifaceName,LYNQ_IFACE_NAME_MAX_LEN);
            if(pdnState != RIL_Data_Call_PdnState::PDN_DISCONNECTED)//PDN_DISCONNECTED
            {
                urc_msg = strdupReadString_p(p);
                int len = strlen(urc_msg);
                if((len < LYNQ_APN_MAX_LEN-1) && (len > 0))
                {
                    memcpy(apn,urc_msg,len+1);
                }
                free(urc_msg);
                urc_msg = strdupReadString_p(p);
                len = strlen(urc_msg);
                if((len < LYNQ_APN_TYPE_MAX_LEN-1) && (len > 0))
                {
                    memcpy(apnType,urc_msg,len+1);
                }
                free(urc_msg);
                urc_msg = strdupReadString_p(p);
                len = strlen(urc_msg);
                if((len < LYNQ_IFACE_NAME_MAX_LEN-1) && (len > 0))
                {
                    memcpy(ifaceName,urc_msg,strlen(urc_msg)+1);
                }
                free(urc_msg);
                //sendSignalDataCallStateChange();
                apnId = getLynqApnID(apnType);
                if(apnId >= 0)
                {
                    LYINFLOG("URC apnType:%s,ifaceName:%s,apn:%s pdnState:%d,handle:%d",apnType,ifaceName,apn,pdnState,apnId);
                    if(lynq_apn_table[apnId].hasTimeout==1)
                    {
                        /*whether timeout?,real or not,*/
                        printf_apn_table();
                        LYERRLOG("apn:%s has time out,deacive this apn",lynq_apn_table[apnId].apn);
                        if (NULL != lynq_apn_table[apnId].apn && NULL != lynq_apn_table[apnId].apnType && strlen(lynq_apn_table[apnId].apn)>0)
                        {
                            LYERRLOG("deactive this time out APN");
                            lynq_deactive_data_call(&apnId);
                        }
                        else
                        {
                            /*if apn lose,update apn and deactive all apn*/
                            LYERRLOG("this table is invalid update APN table");
                        }
                        break;
                    }
                    updateApnTable(&lynq_apn_table[apnId],apn,apnType,ifaceName);
                    printf_apn_table_debug(__FUNCTION__,__LINE__);
                }
                /*To be completed*/
                else
                {
                    data_invaild_error = 1;
                    printf_apn_table_debug(__FUNCTION__,__LINE__);
                    apnId = getDeactApnID(apnType);
                    if(apnId < 0)
                    {
                        LYERRLOG("[%d]invalid apnId:%d",__LINE__,apnId);
                        break;
                    }
                    LYINFLOG("URC apnType:%s,ifaceName:%s,apn:%s pdnState:%d,handle:%d",apnType,ifaceName,apn,pdnState,apnId);
                    updateDeactApn(apnId,pdnState);
                    printf_apn_table_debug(__FUNCTION__,__LINE__);
                }
                pthread_mutex_lock(&s_lynq_urc_vector_mutex);
                s_data_urc_wait_list.push_back(apnId);
                pthread_mutex_unlock(&s_lynq_urc_vector_mutex);
                lynq_data_call_change_id = apnId;
                sendSignalPdnChange();
                LYDBGLOG("data call state:%d",lynq_data_call);
                if(lynq_data_call==1)
                {
                    while (data_timelimit == 0)
                    {
                        LYINFLOG("client not ready to wait");
                        for (check_count = 0;check_count < 500;check_count++)
                        {
                            /*wait 10ms*/
                            usleep(10*1000);
                        }
                        LYERRLOG("client still without res");
                        break;
                    }
                    if(pdnState == RIL_Data_Call_PdnState::PDN_TIMEOUT_CANCEL)
                    {
                        lynq_data_call = LYNQ_E_TIME_OUT;
                    }
                    else
                    {
                        lynq_data_call = 0;
                    }
                    sendSignalDataCallStateChange();
                    data_timelimit = 0;
                }
                printf_apn_table_debug(__FUNCTION__,__LINE__);
            }
            else
            {
                urc_msg = strdupReadString_p(p);
                free(urc_msg);
                urc_msg = strdupReadString_p(p);
                len = strlen(urc_msg);
                if(len < LYNQ_APN_TYPE_MAX_LEN-1)
                {
                    memcpy(apnType,urc_msg,len+1);
                }
                free(urc_msg);
                LYDBGLOG("[data thread_urc_recv] apntype:%s",apnType);
                apnId = getLynqApnID(apnType);
                if(apnId >= 0)
                {
                    LYINFLOG("[%d]URC apnType:%s,ifaceName:%s,apn:%s pdnState:%d,handle:%d",__LINE__,apnType,ifaceName,apn,pdnState,apnId);
                    lynq_data_call_change_id = apnId;
                    //bzero(lynq_apn_table[apnId].apnType,LYNQ_APN_TYPE_MAX_LEN);//async clean
                    pthread_mutex_lock(&s_lynq_urc_vector_mutex);
                    s_data_urc_wait_list.push_back(apnId);
                    pthread_mutex_unlock(&s_lynq_urc_vector_mutex);
                    sendSignalPdnChange();
                }
                else
                {
                    apnId = getDeactApnID(apnType);
                    if(apnId < 0)
                    {
                        LYERRLOG("[%d]invalid apnId:%d",__LINE__,apnId);
                        break;
                    }
                    LYINFLOG("[%d]URC apnType:%s,ifaceName:%s,apn:%s pdnState:%d,handle:%d",__LINE__,apnType,ifaceName,apn,pdnState,apnId);
                    cleanDeactApn(apnId);
                    lynq_data_call_change_id = apnId;
                    bzero(lynq_apn_table[apnId].apnType,LYNQ_APN_TYPE_MAX_LEN);//async clean
                    pthread_mutex_lock(&s_lynq_urc_vector_mutex);
                    s_data_urc_wait_list.push_back(apnId);
                    pthread_mutex_unlock(&s_lynq_urc_vector_mutex);
                    sendSignalDeactvied();
                    sendSignalPdnChange();
                    printf_apn_table_debug(__FUNCTION__,__LINE__);
                }
                LYDBGLOG("data call state:%d",lynq_data_call);
                if(lynq_data_call==1)
                {
                    while (data_timelimit == 0)
                    {
                        LYINFLOG("client not ready to wait");
                        for (check_count = 0;check_count < 500;check_count++)
                        {
                            /*wait 10ms*/
                            usleep(10*1000);
                        }
                        LYERRLOG("client still without res");
                        break;
                    }
                    sendSignalDataCallStateChange();
                    lynq_data_call = 0;
                    data_timelimit = 0;
                }
                printf_apn_table_debug(__FUNCTION__,__LINE__);
            }
            break;
        case LYNQ_URC_MODIFY_APNDB:
            urc_msg = strdupReadString_p(p);
            if (NULL == urc_msg)
            {
                LYERRLOG("error apn msg");
            }
            else
            {
                bzero(g_lynq_apn_result, 1024);
                strcpy(g_lynq_apn_result, urc_msg);
                sendSignalApnChange();
            }
            free(urc_msg);
            break;
        case LYNQ_URC_RESET_APNDB:
            {
                urc_msg = strdupReadString_p(p);
                if (NULL == urc_msg)
                {
                    LYERRLOG("error apn msg");
                }
                else
                {
                    bzero(g_lynq_apn_result, 1024);
                    strcpy(g_lynq_apn_result, urc_msg);
                    sendSignalApnChange();
                }
                free(urc_msg);
                break;
            }
        case LYNQ_TELEPHONY_RESTART:
            {
                if(slot_id == 0)
                {
                    RLOGI("data has received telephony has restart");
                    RLOGI("handle set to 10");
                    lynq_telephony_restart_g = 1;
                    pthread_mutex_lock(&s_lynq_urc_vector_mutex);
                    s_data_urc_wait_list.push_back(TELEPHONY_RESTART);
                    pthread_mutex_unlock(&s_lynq_urc_vector_mutex);
                    sendSignalPdnChange();
                }
                break;
            }
        default:
            break;
    }
    
}

int create_urc_vector_signal_thread()
{
    int ret;
    pthread_mutex_init(&s_lynq_urc_vector_mutex,NULL);
    pthread_mutex_lock(&s_lynq_urc_vector_mutex);
    s_data_urc_wait_list.clear();
    pthread_mutex_unlock(&s_lynq_urc_vector_mutex);
    return 0;
}
int get_imsi();
int check_apn_status();
//xy.he add for bug-view-73 on 2024-08-26 start
int getDataSim();
//xy.he add for bug-view-73 on 2024-08-26 end
int get_timeout_value();
int lynq_init_data(int uToken)
{
    LYLOGSET(LOG_INFO);
    LYLOGEINIT(USER_LOG_TAG);
    LYERRLOG("%s start, parameter is %d", __func__,uToken);

    if (g_lynq_data_init_flag == 1)
    {
        LYERRLOG("init twice is not allowed");
        return -1;
    }
    g_lynq_data_init_flag = 1;
    int result = 0;
    Global_uToken = uToken;

    int ret = get_timeout_value();
    if(ret >= 30000)
    {
        g_data_call_timeout_value = ret;
    }
    else
    {
        LYERRLOG("timeout must greater or equal to 30s!!!");
    }
    result = lynq_socket_client_start();
    pthread_mutex_init(&g_lynq_data_sendto_mutex, NULL);
    if(result!=0)
    {
        LYERRLOG("init socket client fail!!!");
        return -1;
    }
    result = lynq_init_data_urc_thread();
    if(result!=0)
    {
        LYERRLOG("init socket urc fail!!!");
        return -1;
    }

    result = create_urc_vector_signal_thread();
    if(result!=0)
    {
        LYERRLOG("init socket urc fail!!!");
        return -1;
    }
    memset(lynq_apn_table,0,sizeof(lynq_apn_table));
    LYINFLOG("[%s] radio on/off solution 20230724",__FUNCTION__);
    int count = 0;
    while(count < 2)//try recover the network within 10s.
    {
        result =  check_apn_status();
        if(result==0)
        {
            break;
        }
        radio_switch(0);
        sleep(1);
        radio_switch(1);
        sleep(3);
        count++;
    }
    LYINFLOG("[%s] count is %d",__FUNCTION__,count);
    if(result!=0)
    {
        LYDBGLOG("lynq init call fail!!!");
        return LYNQ_E_MD_NOT_READY;//
    }
    /* old
    while(count < 10)
    {
        result =  get_imsi();
        if(result==0)
        {
            break;
        }
        sleep(1);
        count++;
    }
    LYINFLOG("[%s] count is %d",__FUNCTION__,count);
    if(result!=0)
    {
        LYDBGLOG("lynq init call fail!!!");
        return LYNQ_E_MD_NOT_READY;//
    }
    */
    LYERRLOG("%s end suc", __func__);
    return 0;

}
int lynq_deinit_data()
{
    LYERRLOG("%s start", __func__);

    int ret = -1;    
    if (g_lynq_data_init_flag == 0)
    {
        LYERRLOG("deinit twice is not allowed");
        return ret;
    }
    g_lynq_data_init_flag = 0;
    for(int i =0;i<LYNQ_APN_CHANNEL_MAX;i++)
    {
        if(strlen(lynq_apn_table[i].apnType)!=0)
        {
            lynq_deactive_data_call(&i);
        }
    }
    if(lynq_client_sockfd>0)
    {
        close(lynq_client_sockfd);
    }
    ret = lynq_deinit_data_urc_thread();
    if (ret != 0)
    {
        LYERRLOG("lynq_deinit_data_urc_thread fail");
        return ret;
    }
    pthread_mutex_lock(&s_lynq_urc_vector_mutex);
    s_data_urc_wait_list.clear();
    pthread_mutex_unlock(&s_lynq_urc_vector_mutex);
    LYERRLOG("%s end suc", __func__);
    return 0;
}
int lynq_setup_data_call(int *handle)
{
    int error = -1;
    if (g_lynq_data_init_flag == 0)
    {
        LYERRLOG("[%s][%d]Invalid operation",__FUNCTION__,__LINE__);
        return error;
    }
    #ifdef GSW_RIL_CFG  //becuase gsw not have connman,data can not be triggered by connman.
    LYINFLOG("[%s][%d]",__FUNCTION__,__LINE__);
    error = lynq_setup_data_call_sp(handle,NULL,"default",NULL,NULL,NULL,NULL,NULL);
    #else
    Parcel *p = NULL;
    lynq_client_t client;
    int resp_type = -1;
    int request = -1;
    int slot_id = -1;
    int lynq_data_call_id = 0;
    if(handle==NULL)
    {
        LYERRLOG("handle is null!!!");
        return LYNQ_E_NULL_ANONALY;
    }
    client.uToken = get_utoken();
    client.request = 27;//RIL_REQUEST_SETUP_DATA_CALL
    client.paramLen = 0;
    bzero(client.param,LYNQ_REQUEST_PARAM_BUF);
    LYINFLOG("uToken=%d,request=%d,paralen=%d,param=%s",client.uToken,client.request,client.paramLen,client.param);
    lynq_data_call_id = get_handle("default");
    if (lynq_data_call_id < 0)
    {
        LYERRLOG("update apn table fail error id = %d",lynq_data_call_id);
        return LYNQ_E_NULL_ANONALY+1;
    }
    *handle = lynq_data_call_id;//T8TSK-211 (2023/5/16) why? If it times out, the client application will also try to reconnect.
    //Reconnection needs to be deactivated first, and deactivation needs to know which apn to activate.
    lynq_data_call = 1;
    pthread_mutex_lock(&g_lynq_data_sendto_mutex);
    if(send_request(lynq_client_sockfd,&client)==-1)
    {
        LYERRLOG("send request fail");
        perror("[LYNQ_DATA] send request fail:");
        pthread_mutex_unlock(&g_lynq_data_sendto_mutex);
        return LYNQ_E_NULL_ANONALY+2;
    }
    //get_response(lynq_client_sockfd,p);
    error = wait_response(lynq_client_sockfd,p,client.uToken);
    if(error!=0)
    {
        LYERRLOG("wait_response fail,ret:%d",error);
        printf_apn_table_debug(__FUNCTION__,__LINE__);
        pthread_mutex_unlock(&g_lynq_data_sendto_mutex);
        return error;
    }
    pthread_mutex_unlock(&g_lynq_data_sendto_mutex);
    JumpHeader(p,&resp_type,&client.uToken,&request,&slot_id,&error);
    free_parcel(p);
    LYINFLOG("resp_type=%d,request=%d,slot_id=%d,error_code=%d",resp_type,request,slot_id,error);
    if(error==0)
    {
        data_timelimit = 1;
        if (waitDataCallstateChange(g_data_call_timeout_value) == ETIMEDOUT)
        {
            error = LYNQ_E_TIME_OUT;
            LYERRLOG("timeout:wait data Call state fail!!!");
            lynq_apn_table[lynq_data_call_id].hasTimeout = 1;
            if (data_invaild_error == 1)
            {
                data_invaild_error = 0;
                LYERRLOG("urc apn info error!!!");
                return 8085;
            }
            printf_apn_table_debug(__FUNCTION__,__LINE__);
            return error;
        }
        if(lynq_data_call == LYNQ_E_TIME_OUT)
        {
            error = LYNQ_E_TIME_OUT;
            LYERRLOG("PDN_TIMEOUT_CANCLE:wait data Call state fail!!!");
            printf_apn_table_debug(__FUNCTION__,__LINE__);
            return error;
        }
    }
    printf_apn_table_debug(__FUNCTION__,__LINE__);
    #endif //GSW_RIL_CFG
    return error;
}

void deactiveAfterTimeout(int apnId)
{
    LYDBGLOG("%s:apn id:%d",__FUNCTION__,apnId);
    cleanDeactApn(apnId);
    pthread_mutex_lock(&s_lynq_urc_vector_mutex);
    s_data_urc_wait_list.push_back(apnId);
    pthread_mutex_unlock(&s_lynq_urc_vector_mutex);
    sendSignalPdnChange();
}
int lynq_deactive_data_call(int *handle)
{
    Parcel *p = NULL;
    lynq_client_t client;
    int resp_type = -1;
    int request = -1;
    int slot_id = -1;
    int error = -1;
    int lynq_data_call_id = -1;
    int ret = 0;
    if (g_lynq_data_init_flag == 0)
    {
        LYERRLOG("[%s][%d]Invalid operation",__FUNCTION__,__LINE__);
        return error;
    }
    if(handle==NULL)
    {
        LYERRLOG("handle is null!!!");
        return -1;
    }
    ret = handleCheck(*handle);
    if (ret != 0)
    {
        LYERRLOG("incomming handle is invalid");
        return -1;
    }
    lynq_data_call_id = *handle;
    client.uToken = get_utoken();
    client.request = 41;//RIL_REQUEST_DEACTIVATE_DATA_CALL
    client.paramLen = 0;
    bzero(client.param,LYNQ_REQUEST_PARAM_BUF);
    #ifdef GSW_RIL_CFG
    client.paramLen = 1;
    sprintf(client.param,"%s",lynq_apn_table[lynq_data_call_id].apnType);
    #else
    if(strcmp(lynq_apn_table[lynq_data_call_id].apnType,"default")!=0)
    {
        client.paramLen = 1;
        sprintf(client.param,"%s",lynq_apn_table[lynq_data_call_id].apnType);
    }
    #endif //GSW_RIL_CFG
    LYINFLOG("uToken=%d,request=%d,paralen=%d,param=%s",client.uToken,client.request,client.paramLen,client.param);
    pthread_mutex_lock(&g_lynq_data_sendto_mutex);
    if(send_request(lynq_client_sockfd,&client)==-1)
    {
        LYERRLOG("send request fail");
        perror("[LYNQ_DATA] send request fail:");
        pthread_mutex_unlock(&g_lynq_data_sendto_mutex);
        return -1;
    }
    //get_response(lynq_client_sockfd,p);
    error = wait_response(lynq_client_sockfd,p,client.uToken);
    if(error!=0)
    {
        LYERRLOG("wait_response fail,ret:%d",error);
        printf_apn_table_debug(__FUNCTION__,__LINE__);
        pthread_mutex_unlock(&g_lynq_data_sendto_mutex);
        return error;
    }
    pthread_mutex_unlock(&g_lynq_data_sendto_mutex);
    JumpHeader(p,&resp_type,&client.uToken,&request,&slot_id,&error);
    LYINFLOG("resp_type=%d,uToken=%d,request=%d,slot_id=%d,error_code=%d",resp_type,client.uToken,request,slot_id,error);
    cleanOnceApnTable(lynq_data_call_id);
    printf_apn_table_debug(__FUNCTION__,__LINE__);
    if(error==0)
    {
        if(waitDeactived(20000)==ETIMEDOUT)
        {
            error = LYNQ_E_TIME_OUT;
            LYERRLOG("[lynq_deactive_data_call] timeout:wait data Call state fail!!!");
            deactiveAfterTimeout(lynq_data_call_id);
            printf_apn_table_debug(__FUNCTION__,__LINE__);
        }
    }
    free_parcel(p);
    return error;
}
int lynq_setup_data_call_sp(int *handle,char *apn,char *apnType,char *user,char *password,char *authType,char *normalProtocol,char *roamingProtocol)
{
    Parcel *p = NULL;
    lynq_client_t client;
    int resp_type = -1;
    int request = -1;
    int slot_id = -1;
    int error = -1;
    int lynq_data_call_id = -1;
    char *argv[10] = {};
    if (g_lynq_data_init_flag == 0)
    {
        LYERRLOG("[%s][%d]Invalid operation",__FUNCTION__,__LINE__);
        return error;
    }
    #ifdef GSW_RIL_CFG
    LYINFLOG("[%s][%d]",__FUNCTION__,__LINE__);
    if(handle==NULL||apnType==NULL)
    {
        LYERRLOG("handle or apntype is null!!!");
        return -1;
    }
    #else
    if(handle==NULL||apn==NULL||apnType==NULL)
    {
        LYERRLOG("handle ,apn or apntype is null!!!");
        return LYNQ_E_NULL_ANONALY;
    }
    #endif //GSW_RIL_CFG
    if(user==NULL)
    {
        argv[1] = "null";
    }
    else
    {
        argv[1] = user;
    }
    if(password==NULL)
    {
        argv[2] = "null";
    }
    else
    {
        argv[2] = password;
    }
    if(authType==NULL)
    {
        argv[3] = "null";
    }
    else
    {
        argv[3] = authType;
    }
    if(normalProtocol==NULL)
    {
        argv[4] = "null";
    }
    else
    {
        argv[4] = normalProtocol;
    }
    if(roamingProtocol==NULL)
    {
        argv[5] = "null";
    }
    else
    {
        argv[5] = roamingProtocol;
    }
    client.uToken = get_utoken();
    client.request = 27;//RIL_REQUEST_SETUP_DATA_CALL
    client.paramLen = 7;
    bzero(client.param,LYNQ_REQUEST_PARAM_BUF);
    #ifdef GSW_RIL_CFG
    if(NULL == apn)
    {
        sprintf(client.param,"null %s %s %s %s %s %s",apnType,argv[1],argv[2],argv[3],argv[4],argv[5]);
    }
    else
    {
        sprintf(client.param,"%s %s %s %s %s %s %s",apn,apnType,argv[1],argv[2],argv[3],argv[4],argv[5]);
    }
    #else
    sprintf(client.param,"%s %s %s %s %s %s %s",apn,apnType,argv[1],argv[2],argv[3],argv[4],argv[5]);
    #endif  //GSW_RIL_CFG
    LYINFLOG("uToken=%d,request=%d,paralen=%d,param=%s",client.uToken,client.request,client.paramLen,client.param);
    lynq_data_call_id = get_handle(apnType);
    if (lynq_data_call_id < 0)
    {
        LYERRLOG("update apn table fail error id = %d",lynq_data_call_id);
        return LYNQ_E_NULL_ANONALY+1;
    }
    *handle = lynq_data_call_id;//T8TSK-211 (2023/5/16)
    lynq_data_call = 1;
    pthread_mutex_lock(&g_lynq_data_sendto_mutex);
    if(send_request(lynq_client_sockfd,&client)==-1)
    {
        LYERRLOG("send request fail");
        perror("[LYNQ_DATA] send request fail:");
        pthread_mutex_unlock(&g_lynq_data_sendto_mutex);
        return LYNQ_E_NULL_ANONALY+2;
    }
    //get_response(lynq_client_sockfd,p);
    error = wait_response(lynq_client_sockfd,p,client.uToken);
    if(error!=0)
    {
        LYERRLOG("wait_response fail,ret:%d",error);
        printf_apn_table_debug(__FUNCTION__,__LINE__);
        pthread_mutex_unlock(&g_lynq_data_sendto_mutex);
        return error;
    }
    pthread_mutex_unlock(&g_lynq_data_sendto_mutex);
    JumpHeader(p,&resp_type,&client.uToken,&request,&slot_id,&error);
    LYINFLOG("resp_type=%d,uToken=%d,request=%d,slot_id=%d,error_code=%d",resp_type,client.uToken,request,slot_id,error);
    free_parcel(p);
    if(error==0)
    {
        data_timelimit = 1;
        if(waitDataCallstateChange(g_data_call_timeout_value)==ETIMEDOUT)
        {
            error = LYNQ_E_TIME_OUT;
            LYERRLOG("timeout:wait data Call state fail!!!");
            lynq_apn_table[lynq_data_call_id].hasTimeout = 1;
            if (data_invaild_error == 1)
            {
                data_invaild_error = 0;
                LYERRLOG("urc apn info error!!!");
                return 8085;
            }
            printf_apn_table_debug(__FUNCTION__,__LINE__);
            return error;
        }
        if(lynq_data_call == LYNQ_E_TIME_OUT)
        {
            error = LYNQ_E_TIME_OUT;
            LYERRLOG("PDN_TIMEOUT_CANCLE:wait data Call state fail!!!");
            printf_apn_table_debug(__FUNCTION__,__LINE__);
            return error;
        }
    }
    printf_apn_table_debug(__FUNCTION__,__LINE__);
    return error;
}
/*
int lynq_deactive_data_call_sp(int *handle,char *apnType)
{
    Parcel p;
    lynq_client_t client;
    int resp_type = -1;
    int request = -1;
    int slot_id = -1;
    int error = -1;
    if(handle==NULL||apnType==NULL)
    {
        LYERRLOG("handle is null!!!");
        return -1;
    }
    client.uToken = Global_uToken;
    client.request = 41;//RIL_REQUEST_DEACTIVATE_DATA_CALL
    client.paramLen = 1;
    bzero(client.param,LYNQ_REQUEST_PARAM_BUF);
    sprintf(client.param,"%s",apnType);
    LYERRLOG("uToken=%d,request=%d,paralen=%d,param=%s",client.uToken,client.request,client.paramLen,client.param);
    if(send_request(lynq_client_sockfd,&client)==-1)
    {
        LYERRLOG("send request fail");
        perror("[LYNQ_DATA] send request fail:");
        return -1;
    }
    get_response(lynq_client_sockfd,p);
    JumpHeader(p,&resp_type,&request,&slot_id,&error);
    LYINFLOG("resp_type=%d,request=%d,slot_id=%d,error_code=%d",resp_type,request,slot_id,error);
    return error;
}
*/
int getDataCallLists(lynq_data_call_response_v11_t dataCallList[LYNQ_APN_CHANNEL_MAX],int *realNum)
{
    Parcel *p = NULL;
    //xy.he add for bug-view-73 on 2024-08-26 start
    #ifdef MODE_DSDS
    lynq_client_dual_t client;
    #else
    lynq_client_t client;
    #endif
    int resp_type = -1;
    int token;
    int request = -1;
    int slot_id = -1;
    int error = -1;
    int version =0;
    int num = 0;
    char *temp_char = NULL;
    if(dataCallList==NULL)
    {
        LYERRLOG("dataCallList is null!!!");
        return -1;
    }
    client.uToken = get_utoken();;
    client.request = 57;//RIL_REQUEST_DATA_CALL_LIST
    client.paramLen = 0;
    #ifdef MODE_DSDS
    bzero(client.param,LYNQ_REQUEST_PARAM_BUF);
    char slot_status = '0';
    int data_sim = -1;
    data_sim = getDataSim();
    if(data_sim >= 0)
    {
        client.slot_status = '1';
        client.slot = (char)('0'+data_sim);
    }
    LYINFLOG("uToken=%d,request=%d,slot_status=%c,slot=%c,paralen=%d,param=%s",client.uToken,client.request,client.slot_status,client.slot,client.paramLen,client.param);
    pthread_mutex_lock(&g_lynq_data_sendto_mutex);
    if(send_request_dual(lynq_client_sockfd,&client,sizeof(client))==-1)
    {
        LYERRLOG("send request fail");
        perror("[LYNQ_DATA] send request fail:");
        pthread_mutex_unlock(&g_lynq_data_sendto_mutex);
        return -1;
    }
    #else
    bzero(client.param,LYNQ_REQUEST_PARAM_BUF);
    LYINFLOG("uToken=%d,request=%d,paralen=%d,param=%s",client.uToken,client.request,client.paramLen,client.param);
    pthread_mutex_lock(&g_lynq_data_sendto_mutex);
    if(send_request(lynq_client_sockfd,&client)==-1)
    {
        LYERRLOG("send request fail");
        perror("[LYNQ_DATA] send request fail:");
        pthread_mutex_unlock(&g_lynq_data_sendto_mutex);
        return -1;
    }
    #endif
    //xy.he add for bug-view-73 on 2024-08-26 start
    //get_response(lynq_client_sockfd,p);
    error = wait_response(lynq_client_sockfd,p,client.uToken);
    if(error!=0)
    {
        LYERRLOG("wait_response fail,ret:%d",error);
        pthread_mutex_unlock(&g_lynq_data_sendto_mutex);
        return error;
    }
    pthread_mutex_unlock(&g_lynq_data_sendto_mutex);
    if(JumpHeader(p,&resp_type,&client.uToken,&request,&slot_id,&error)!=0)
    {
        LYERRLOG("JumpHeader fail");
        return -1;
    }
    LYINFLOG("resp_type=%d,uToken=%d,request=%d,slot_id=%d,error_code=%d",resp_type,client.uToken,request,slot_id,error);
    p->readInt32(&version);
    if(version==11)
    {
        p->readInt32(&num);
        *realNum = num;
        for (int i = 0; i < num; i++)
        {
            p->readInt32(&dataCallList[i].status);
            p->readInt32(&dataCallList[i].suggestedRetryTime);
            p->readInt32(&dataCallList[i].cid);
            p->readInt32(&dataCallList[i].active);
            temp_char = strdupReadString_p(p);
            LYINFLOG("[%s][%d]%s",__FUNCTION__,__LINE__,temp_char);
            if(temp_char)
            {
                memcpy(dataCallList[i].type,temp_char,strlen(temp_char)+1);
                free(temp_char);
            }
            temp_char = strdupReadString_p(p);
            LYINFLOG("[%s][%d]%s",__FUNCTION__,__LINE__,temp_char);
            if(temp_char)
            {
                memcpy(dataCallList[i].ifname,temp_char,strlen(temp_char)+1);
                free(temp_char);
            }
            temp_char = strdupReadString_p(p);
            LYINFLOG("[%s][%d]%s",__FUNCTION__,__LINE__,temp_char);
            if(temp_char)
            {
                memcpy(dataCallList[i].addresses,temp_char,strlen(temp_char)+1);
                free(temp_char);
            }
            temp_char = strdupReadString_p(p);
            LYINFLOG("[%s][%d]%s",__FUNCTION__,__LINE__,temp_char);
            if(temp_char)
            {
                memcpy(dataCallList[i].dnses,temp_char,strlen(temp_char)+1);
                free(temp_char);
            }
            temp_char = strdupReadString_p(p);
            LYINFLOG("[%s][%d]%s",__FUNCTION__,__LINE__,temp_char);
            if(temp_char)
            {
                memcpy(dataCallList[i].gateways,temp_char,strlen(temp_char)+1);
                free(temp_char);
            }
            temp_char = strdupReadString_p(p);
            LYINFLOG("[%s][%d]%s",__FUNCTION__,__LINE__,temp_char);
            if(temp_char)
            {
                memcpy(dataCallList[i].pcscf,temp_char,strlen(temp_char)+1);
                free(temp_char);
            }
            p->readInt32(&dataCallList[i].mtu);
        }
    }
    free_parcel(p);
    return error;
}

int lynq_get_apn_msg(int handle,lynq_apn_msg_t *apn_msg)
{
    LYINFLOG("[lynq_get_apn_msg] enter,handle:%d",handle);
    if((handle < 0) || (handle > 6) || (NULL == apn_msg))
    {
        LYERRLOG("handle value error,or apn_msg is null");
        return LYNQ_E_NULL_ANONALY;
    }
    apn_msg->handle = handle;
    if(apn_msg->apn!=NULL)
    {
        memcpy(apn_msg->apn,lynq_apn_table[handle].apn,LYNQ_APN_MAX_LEN);
    }
    if(apn_msg->apnType!=NULL)
    {
        memcpy(apn_msg->apnType,lynq_apn_table[handle].apnType,LYNQ_APN_TYPE_MAX_LEN);
    }
    return 0;
}

int lynq_get_data_call_list(int *handle,lynq_data_call_response_v11_t *dataCallList)
{
    lynq_data_call_response_v11_t interDataCallList[LYNQ_APN_CHANNEL_MAX]={};
    int number = 0;
    int lynq_data_call_id = 0;
    int error = 0;
    if (g_lynq_data_init_flag == 0)
    {
        LYERRLOG("[%s][%d]Invalid operation",__FUNCTION__,__LINE__);
        return -1;
    }
    lynq_data_call_id = *handle;
    if((handle==NULL) || (dataCallList==NULL))
    {
        LYERRLOG("[lynq_get_data_call_list] handle or datacalllist is NULL");
        return LYNQ_E_NULL_ANONALY;
    }
    if ((*handle < 0) || (*handle > 6))
    {
        LYERRLOG("[lynq_get_data_call_list] handle value error");
        return LYNQ_E_NULL_ANONALY;
    }
    LYINFLOG("[lynq_get_data_call_list] incoming handle value: %d",*handle);
    memset(interDataCallList,0,sizeof(interDataCallList));
    error = getDataCallLists(interDataCallList,&number);
    if(error == 0)
    {
        for(int i = 0;i < number;i++)
        {
            if(strcmp(interDataCallList[i].ifname,lynq_apn_table[lynq_data_call_id].ifaceName)==0)
            {
                dataCallList->active = interDataCallList[i].active;
                dataCallList->suggestedRetryTime = interDataCallList[i].suggestedRetryTime;
                dataCallList->cid = interDataCallList[i].cid;
                dataCallList->status = interDataCallList[i].status;
                dataCallList->mtu = interDataCallList[i].mtu;
                memcpy(dataCallList->addresses,interDataCallList[i].addresses,sizeof(interDataCallList[i].addresses));
                memcpy(dataCallList->ifname,interDataCallList[i].ifname,sizeof(interDataCallList[i].ifname));
                memcpy(dataCallList->dnses,interDataCallList[i].dnses,sizeof(interDataCallList[i].dnses));
                memcpy(dataCallList->type,interDataCallList[i].type,sizeof(interDataCallList[i].type));
                memcpy(dataCallList->gateways,interDataCallList[i].gateways,sizeof(interDataCallList[i].gateways));
                memcpy(dataCallList->pcscf,interDataCallList[i].pcscf,sizeof(interDataCallList[i].pcscf));
                LYDBGLOG("ifname:%s,addr:%s",dataCallList->ifname,dataCallList->addresses);
            }
        }
    }
    return error;
}
int lynq_wait_data_call_state_change(int *handle)
{
    if (data_waiting_status == 1)
    {
        LYDBGLOG("some thread is waiting");
        return -3;
    }
    LYDBGLOG("is empty :%d",s_data_urc_wait_list.empty());
    if (s_data_urc_wait_list.empty())
    {
        LYDBGLOG("start wait");
        data_waiting_status = 1;
        waitPdnChange();
    }
    data_waiting_status = 0;
    std::vector<int>::iterator iter;

    pthread_mutex_lock(&s_lynq_urc_vector_mutex);
    iter = s_data_urc_wait_list.begin();
    if (iter != s_data_urc_wait_list.end())
    {
        *handle = *iter;
    }
    s_data_urc_wait_list.erase(iter);
    pthread_mutex_unlock(&s_lynq_urc_vector_mutex);
    
    LYINFLOG("lynq_wait_data_call_state_change handle:%d",*handle);
    return 0;
}
/*Warren add for T800 platform 2021/11/19 end*/

/*Typethree add for T800 platform 2022/04/21 start*/

int insert_apn_char(char *agc, char *id,char *mcc, char *mnc, char *apn, char *apntype, char *user, char *password, char *normalprotocol, char *roamingprotocol, char *carrier)
{
    char strtmp[10][32];
    if (id == NULL)
    {
        sprintf(strtmp[0], "id=;");
    }
    else
    {
        sprintf(strtmp[0], "id=%s;", id);
    }
    if (mcc == NULL)
    {
        sprintf(strtmp[1], "mcc=;");
    }
    else
    {
        sprintf(strtmp[1], "mcc=%s;", mcc);
    }
    if (mnc == NULL)
    {
        sprintf(strtmp[2], "mnc=;");
    }
    else
    {
        sprintf(strtmp[2], "mnc=%s;", mnc);
    }
    if (apn == NULL)
    {
        sprintf(strtmp[3], "apn=;");
    }
    else
    {
        sprintf(strtmp[3], "apn=%s;", apn);
    }
    if (apntype == NULL)
    {
        sprintf(strtmp[4], "apntype=;");
    }
    else
    {
        sprintf(strtmp[4], "apntype=%s;", apntype);
    }
    if (user == NULL)
    {
        sprintf(strtmp[5], "user=;");
    }
    else
    {
        sprintf(strtmp[5], "user=%s;", user);
    }
    if (password == NULL)
    {
        sprintf(strtmp[6], "password=;");
    }
    else
    {
        sprintf(strtmp[6], "password=%s;", password);
    }
    if (normalprotocol == NULL)
    {
        sprintf(strtmp[7], "protocol=;");
    }
    else
    {
        sprintf(strtmp[7], "protocol=%s;", normalprotocol);
    }
    if (roamingprotocol == NULL)
    {
        sprintf(strtmp[8], "roaming_protocol=;");
    }
    else
    {
        sprintf(strtmp[8], "roaming_protocol=%s;", roamingprotocol);
    }
    if (carrier == NULL)
    {
        sprintf(strtmp[9], "carrier=;");
    }
    else
    {
        sprintf(strtmp[9], "carrier=%s;", carrier);
    }
    sprintf(agc, "%s%s%s%s%s%s%s%s%s%s",strtmp[0], strtmp[1], strtmp[2], strtmp[3], strtmp[4], strtmp[5], strtmp[6], strtmp[7], strtmp[8], strtmp[9]);

    return 0;
}

int modify_apn_char(char *agc, char *id, char *mcc, char *mnc, char *apn, char *apntype, char *user, char *password, char *normalprotocol, char *roamingprotocol, char *carrier)
{
    char strtmp[10][32];
    if (id == NULL)
    {
        sprintf(strtmp[0], "id=;");
    }
    else
    {
        sprintf(strtmp[0], "id=%s;", id);
    }
    if (mcc == NULL)
    {
        sprintf(strtmp[1], "mcc=;");
    }
    else
    {
        sprintf(strtmp[1], "mcc=%s;", mcc);
    }
    if (mnc == NULL)
    {
        sprintf(strtmp[2], "mnc=;");
    }
    else
    {
        sprintf(strtmp[2], "mnc=%s;", mnc);
    }
    if (apn == NULL)
    {
        sprintf(strtmp[3], "apn=;");
    }
    else
    {
        sprintf(strtmp[3], "apn=%s;", apn);
    }
    if (apntype == NULL)
    {
        sprintf(strtmp[4], "apntype=;");
    }
    else
    {
        sprintf(strtmp[4], "apntype=%s;", apntype);
    }
    if (user == NULL)
    {
        sprintf(strtmp[5], "user=;");
    }
    else
    {
        sprintf(strtmp[5], "user=%s;", user);
    }
    if (password == NULL)
    {
        sprintf(strtmp[6], "password=;");
    }
    else
    {
        sprintf(strtmp[6], "password=%s;", password);
    }
    if (normalprotocol == NULL)
    {
        sprintf(strtmp[7], "protocol=;");
    }
    else
    {
        sprintf(strtmp[7], "protocol=%s;", normalprotocol);
    }
    if (roamingprotocol == NULL)
    {
        sprintf(strtmp[8], "roaming_protocol=;");
    }
    else
    {
        sprintf(strtmp[8], "roaming_protocol=%s;", roamingprotocol);
    }
    if (carrier == NULL)
    {
        sprintf(strtmp[9], "carrier=;");
    }
    else
    {
        sprintf(strtmp[9], "carrier=%s;", carrier);
    }
    sprintf(agc, "%s%s%s%s%s%s%s%s%s%s", strtmp[0], strtmp[1], strtmp[2], strtmp[3], strtmp[4], strtmp[5], strtmp[6], strtmp[7], strtmp[8], strtmp[9]);
    return 0;
}


int query_apn_char(char *agc, char *id, char *mcc, char *mnc, char *apn, char *apntype, char *user, char *password, char *normalprotocol, char *roamingprotocol, char *carrier)
{
    char strtmp[10][32];
    if (id == NULL)
    {
        sprintf(strtmp[0], "");
    }
    else
    {
        sprintf(strtmp[0], "id=%s;", id);
    }
    if (mcc == NULL)
    {
        sprintf(strtmp[1], "");
    }
    else
    {
        sprintf(strtmp[1], "mcc=%s;", mcc);
    }
    if (mnc == NULL)
    {
        sprintf(strtmp[2], "");
    }
    else
    {
        sprintf(strtmp[2], "mnc=%s;", mnc);
    }
    if (apn == NULL)
    {
        sprintf(strtmp[3], "");
    }
    else
    {
        sprintf(strtmp[3], "apn=%s;", apn);
    }
    if (apntype == NULL)
    {
        sprintf(strtmp[4], "");
    }
    else
    {
        sprintf(strtmp[4], "apntype=%s;", apntype);
    }
    if (user == NULL)
    {
        sprintf(strtmp[5], "");
    }
    else
    {
        sprintf(strtmp[5], "user=%s;", user);
    }
    if (password == NULL)
    {
        sprintf(strtmp[6], "");
    }
    else
    {
        sprintf(strtmp[6], "password=%s;", password);
    }
    if (normalprotocol == NULL)
    {
        sprintf(strtmp[7], "");
    }
    else
    {
        sprintf(strtmp[7], "normalprotocol=%s;", normalprotocol);
    }
    if (roamingprotocol == NULL)
    {
        sprintf(strtmp[8], "");
    }
    else
    {
        sprintf(strtmp[8], "roamingprotocol=%s;", roamingprotocol);
    }
    if (carrier == NULL)
    {
        sprintf(strtmp[9], "");
    }
    else
    {
        sprintf(strtmp[9], "carrier=%s;", carrier);
    }
    sprintf(agc, "%s%s%s%s%s%s%s%s%s%s", strtmp[0], strtmp[1], strtmp[2], strtmp[3], strtmp[4], strtmp[5], strtmp[6], strtmp[7], strtmp[8], strtmp[9]);
    return 0;
}

static char *lynqStrdupReadString(Parcel &p)
{
    size_t stringlen;
    const char16_t *s16;

    s16 = p.readString16Inplace(&stringlen);
    return strndup16to8(s16, stringlen);
}
int get_imsi()
{
    Parcel *p =NULL;
    lynq_client_t client;
    int resp_type = -1;
    int token;
    int request = -1;
    int slot_id = -1;
    int error = -1;
    int version =0;
    int num = 0;
    char *temp_char = NULL;
    char mccmnc[32] = {0};
    char mccmnckey[64] = {0};
    client.uToken = get_utoken();
    client.request = 11;//RIL_REQUEST_GET_IMSI 11
    client.paramLen = 0;
    bzero(client.param,LYNQ_REQUEST_PARAM_BUF);
    LYINFLOG("uToken=%d,request=%d,paralen=%d,param=%s",client.uToken,client.request,client.paramLen,client.param);
    pthread_mutex_lock(&g_lynq_data_sendto_mutex);
    if(send_request(lynq_client_sockfd,&client)==-1)
    {
        LYERRLOG("send request fail");
        perror("[LYNQ_DATA] send request fail:");
        pthread_mutex_unlock(&g_lynq_data_sendto_mutex);
        return -1;
    }
    //get_response(lynq_client_sockfd,p);
    error = wait_response(lynq_client_sockfd,p,client.uToken);
    if(error!=0)
    {
        LYERRLOG("wait_response fail,ret:%d",error);
        pthread_mutex_unlock(&g_lynq_data_sendto_mutex);
        return error;
    }
    pthread_mutex_unlock(&g_lynq_data_sendto_mutex);
    if(JumpHeader(p,&resp_type,&client.uToken,&request,&slot_id,&error)!=0)
    {
        LYERRLOG("JumpHeader fail");
        return -1;
    }
    LYINFLOG("resp_type=%d,uToken=%d,request=%d,slot_id=%d,error_code=%d",resp_type,client.uToken,request,slot_id,error);
    if(error == 0)
    {
        char * test = strdupReadString_p(p);
        memcpy(mccmnc, test,5);
        mccmnc[5]='\0';
        free(test);
        sprintf(mccmnckey,"uci set radio_property.property.vendor_ril_data_gsm_mcc_mnc0=%s",mccmnc);
        system(mccmnckey);
        system("uci commit");
    }
    free_parcel(p);
    return error;
}
int lynq_modify_apn_db(const int cmd, char *id, char *mcc, char *mnc, char *apn, char *apntype, char *user, char *password, char *normalprotocol, char *roamingprotocol, char *carrier, char *out)
{
    if (NULL == id && NULL == mcc && NULL == mnc && NULL == apn && NULL == apntype && NULL == user && NULL == password && NULL == normalprotocol && NULL == roamingprotocol && NULL == carrier)
    {
        LYERRLOG("There are no valid parameters");
        return -1;
    }
    lynq_client_t client;
    char argc[512];
    int res = -1;
    if (cmd == 0) // insert apn db
    {
        res = insert_apn_char(argc, id, mcc, mnc, apn, apntype, user, password, normalprotocol, roamingprotocol, carrier);
        client.uToken = get_utoken();
        client.request = 2000 + 193; // RIL_REQUEST_MODIFY_APN
        client.paramLen = 2;
        bzero(client.param, LYNQ_REQUEST_PARAM_BUF);
        sprintf(client.param, "%d %s", cmd, argc);
    }
    else if (cmd == 1) //delete apn db
    {
        if (NULL == id)
        {
            LYERRLOG("id is NULL!!!please input id: ");
        }
        sprintf(argc, "id=%s", id);
        client.uToken = get_utoken();
        client.request = 2000 + 193; // RIL_REQUEST_MODIFY_APN
        client.paramLen = 2;
        bzero(client.param, LYNQ_REQUEST_PARAM_BUF);
        sprintf(client.param, "%d %s", cmd, argc);
    }
    else if (cmd == 2) //query apn db
    {
        query_apn_char(argc, id, mcc, mnc, apn, apntype, user, password, normalprotocol, roamingprotocol, carrier);
        client.uToken = get_utoken();
        client.request = 2000 + 193; // RIL_REQUEST_MODIFY_APN
        client.paramLen = 2;
        bzero(client.param, LYNQ_REQUEST_PARAM_BUF);
        sprintf(client.param, "%d %s", cmd, argc);
    }
    else if (cmd == 3)  //modify apn db
    {
        modify_apn_char(argc, id, mcc, mnc, apn, apntype, user, password, normalprotocol, roamingprotocol, carrier);
        client.uToken = get_utoken();
        client.request = 2000 + 193; // RIL_REQUEST_MODIFY_APN
        client.paramLen = 2;
        bzero(client.param, LYNQ_REQUEST_PARAM_BUF);
        sprintf(client.param, "%d %s", cmd, argc);
    }
    else
    {
        LYERRLOG("incoming command is invalid");
        return -1;
    }
    LYINFLOG("uToken=%d,request=%d,paralen=%d,param=%s",client.uToken,client.request,client.paramLen,client.param);
    pthread_mutex_lock(&g_lynq_data_sendto_mutex);
    if(send_request(lynq_client_sockfd,&client)==-1)
    {
        LYERRLOG("send request fail");
        pthread_mutex_unlock(&g_lynq_data_sendto_mutex);
        return -1;
    }
    pthread_mutex_unlock(&g_lynq_data_sendto_mutex);
    waitApnResult();
    strcpy(out, g_lynq_apn_result);
    LYINFLOG(">>>>>output info:%s",out);
    return 0;
}

int lynq_reset_apn(char *result)
{
    lynq_client_t client;
    if (NULL == result)
    {
        LYERRLOG("incoming paramters error");
    }
    client.uToken = get_utoken();
    client.request = 2000 + 194;
    client.paramLen = 0;
    bzero(client.param, LYNQ_REQUEST_PARAM_BUF);
    LYINFLOG("uToken=%d,request=%d,paralen=%d,param=%s", client.uToken, client.request, client.paramLen, client.param);
    pthread_mutex_lock(&g_lynq_data_sendto_mutex);
    if (send_request(lynq_client_sockfd, &client) == -1)
    {
        LYERRLOG("send request fail");
        pthread_mutex_unlock(&g_lynq_data_sendto_mutex);
        return -1;
    }
    pthread_mutex_unlock(&g_lynq_data_sendto_mutex);
    waitApnResult();
    strcpy(result, g_lynq_apn_result);
    LYINFLOG(">>>>>result:%s",result);
    return 0;
}

/*Typethree add for T800 platform 2022/04/21 end*/
static int exec_cmd(const char *str_cmd, char * str_cmd_ret, size_t max_len) {
    FILE *fp;
    //printf("to exec cmd:%s\n", str_cmd);
    if((fp=popen(str_cmd,"r"))==NULL)
    {
        perror("popen error!");
        return -1;
    }
    if((fread(str_cmd_ret,max_len,1,fp))<0)
    {
        perror("fread fail!");
        fclose(fp);
        return -1;
    }
    fclose(fp);
    return 0;
}
//xy.he add for bug-view-73 on 2024-08-26 start
int getDataSim()
{
    LYINFLOG("getDataSim enter");
    const char *data_sim_cmd = "uci get radio_property.property.persist_vendor_radio_data_sim";
    char cmd_ret[8] ={0};
    int slot = 0;
    int sim_ret = -1;
    int ret = -1;
    const char *empty_str = ":empty";

    sim_ret = exec_cmd(data_sim_cmd,cmd_ret,8);
    if(sim_ret==0)
    {
        if(strlen(cmd_ret)==0)
        {
            LYERRLOG("not have data_sim config in uci ram");
            return -1;
        }
        if(strncmp(cmd_ret,empty_str,strlen(empty_str))==0)
        {
            LYERRLOG("sim is empty");
            return -2;
        }
        LYINFLOG("sim is %s",cmd_ret);
        slot = atoi(cmd_ret);
        return (slot-1);
    }
    else
    {
        LYERRLOG("exec cmd fail");
        return -1;
    }
}
//xy.he add for bug-view-73 on 2024-08-26 end

int check_apn_status()
{
    LYINFLOG("check_apn_status enter");
    int ret = -1;
    int sim_ret = -1;
    int slot = -1;
    char cmd_ret[8] ={0};//get mccmnc
    const char *sim_0_cmd = "uci get radio_property.property.vendor_ril_data_gsm_mcc_mnc0";
    const char *sim_1_cmd = "uci get radio_property.property.vendor_ril_data_gsm_mcc_mnc1";
    const char *sim_cmd = "uci get radio_property.property.persist_vendor_radio_data_sim";
    const char *empty_str = ":empty";
    //xy.he add for get sim slot at 20240507 start
    sim_ret = exec_cmd(sim_cmd,cmd_ret,8);
    if(sim_ret==0)
    {
        if(strlen(cmd_ret)==0)
        {
            LYERRLOG("not have sim config in uci ram");
            return 1;
        }
        if(strncmp(cmd_ret,empty_str,strlen(empty_str))==0)
        {
            LYERRLOG("sim is empty");
            return 2;
        }
        LYINFLOG("sim is %s",cmd_ret);
        slot = atoi(cmd_ret);
    }
    else
    {
        LYERRLOG("exec cmd fail");
        return -1;
    }

    if(slot == 1)
    {
        ret = exec_cmd(sim_0_cmd,cmd_ret,8);
    }

    else if(slot == 2)
    {
        ret = exec_cmd(sim_1_cmd,cmd_ret,8);
    }
    //xy.he add for get sim slot at 20240507 end
	
    if(ret==0)
    {
        if(strlen(cmd_ret)==0)
        {
            LYERRLOG("not have mccmnc config in uci ram");
            return 1;
        }
        if(strncmp(cmd_ret,empty_str,strlen(empty_str))==0)
        {
            LYERRLOG("mccmnc is empty");
            return 2;
        }
        LYINFLOG("mccmnc is %s",cmd_ret);
        return 0;
    }
    else
    {
        LYERRLOG("exec cmd fail");
        return -1;
    }
    return 0;
}
int get_timeout_value()
{
    LYINFLOG("get_timeout_value enter");
    int ret = -1;
    char timeout_value[8] ={0};//get get_timeout_value
    ret = lynq_get_value("lynq_uci","data_call","timeout",timeout_value);
    if(ret == 0)
    {
        LYINFLOG("data_call.timeout is %s",timeout_value);
        return atoi(timeout_value);
    }
    else
    {
        LYERRLOG("get_timeout_value");
        return -1;
    }
}

int radio_switch(int status)
{
    Parcel *p =NULL;
    lynq_client_t client;
    int resp_type = -1;
    int token;
    int request = -1;
    int slot_id = -1;
    int error = -1;
    int version =0;
    int num = 0;
    char *temp_char = NULL;
    char mccmnc[32] = {0};
    char mccmnckey[64] = {0};
    client.uToken = get_utoken();
    client.request = 59;//RIL_REQUEST_OEM_HOOK_RAW 59
    client.paramLen = 1;
    bzero(client.param,LYNQ_REQUEST_PARAM_BUF);
    sprintf(client.param,"AT+CFUN=%d",status);
    LYINFLOG("uToken=%d,request=%d,paralen=%d,param=%s",client.uToken,client.request,client.paramLen,client.param);
    pthread_mutex_lock(&g_lynq_data_sendto_mutex);
    if(send_request(lynq_client_sockfd,&client)==-1)
    {
        LYERRLOG("send request fail");
        perror("[LYNQ_DATA] send request fail:");
        pthread_mutex_unlock(&g_lynq_data_sendto_mutex);
        return -1;
    }
    //get_response(lynq_client_sockfd,p);
    error = wait_response(lynq_client_sockfd,p,client.uToken);
    if(error!=0)
    {
        LYERRLOG("wait_response fail,ret:%d",error);
        pthread_mutex_unlock(&g_lynq_data_sendto_mutex);
        return error;
    }
    pthread_mutex_unlock(&g_lynq_data_sendto_mutex);
    if(JumpHeader(p,&resp_type,&client.uToken,&request,&slot_id,&error)!=0)
    {
        LYERRLOG("JumpHeader fail");
        return -1;
    }
    LYINFLOG("resp_type=%d,uToken=%d,request=%d,slot_id=%d,error_code=%d",resp_type,client.uToken,request,slot_id,error);
    free_parcel(p);
    return error;
}
