/*
 * Copyright (C) 2008 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef SMSCBCMASINFO_H_
#define SMSCBCMASINFO_H_

#include <string>
class SmsCbCmasInfo {
public:
  SmsCbCmasInfo(int messageClass, int category, int responseType, int severity,
      int urgency, int certainty);
  virtual ~SmsCbCmasInfo();

  // CMAS message class (in GSM/UMTS message identifier or CDMA service category).

  /** Presidential-level alert (Korean Public Alert System Class 0 message). */
  static constexpr int CMAS_CLASS_PRESIDENTIAL_LEVEL_ALERT = 0x00;

  /** Extreme threat to life and property (Korean Public Alert System Class 1 message). */
  static constexpr int CMAS_CLASS_EXTREME_THREAT = 0x01;

  /** Severe threat to life and property (Korean Public Alert System Class 1 message). */
  static constexpr int CMAS_CLASS_SEVERE_THREAT = 0x02;

  /** Child abduction emergency (AMBER Alert). */
  static constexpr int CMAS_CLASS_CHILD_ABDUCTION_EMERGENCY = 0x03;

  /** CMAS test message. */
  static constexpr int CMAS_CLASS_REQUIRED_MONTHLY_TEST = 0x04;

  /** CMAS exercise. */
  static constexpr int CMAS_CLASS_CMAS_EXERCISE = 0x05;

  /** CMAS category for operator defined use. */
  static constexpr int CMAS_CLASS_OPERATOR_DEFINED_USE = 0x06;

  /** CMAS category for warning types that are reserved for future extension. */
  static constexpr int CMAS_CLASS_UNKNOWN = -1;

  // CMAS alert category (in CDMA type 1 elements record).

  /** CMAS alert category: Geophysical including landslide. */
  static constexpr int CMAS_CATEGORY_GEO = 0x00;

  /** CMAS alert category: Meteorological including flood. */
  static constexpr int CMAS_CATEGORY_MET = 0x01;

  /** CMAS alert category: General emergency and public safety. */
  static constexpr int CMAS_CATEGORY_SAFETY = 0x02;

  /** CMAS alert category: Law enforcement, military, homeland/local/private security. */
  static constexpr int CMAS_CATEGORY_SECURITY = 0x03;

  /** CMAS alert category: Rescue and recovery. */
  static constexpr int CMAS_CATEGORY_RESCUE = 0x04;

  /** CMAS alert category: Fire suppression and rescue. */
  static constexpr int CMAS_CATEGORY_FIRE = 0x05;

  /** CMAS alert category: Medical and public health. */
  static constexpr int CMAS_CATEGORY_HEALTH = 0x06;

  /** CMAS alert category: Pollution and other environmental. */
  static constexpr int CMAS_CATEGORY_ENV = 0x07;

  /** CMAS alert category: Public and private transportation. */
  static constexpr int CMAS_CATEGORY_TRANSPORT = 0x08;

  /** CMAS alert category: Utility, telecom, other non-transport infrastructure. */
  static constexpr int CMAS_CATEGORY_INFRA = 0x09;

  /** CMAS alert category: Chem, bio, radiological, nuclear, high explosive threat or attack. */
  static constexpr int CMAS_CATEGORY_CBRNE = 0x0a;

  /** CMAS alert category: Other events. */
  static constexpr int CMAS_CATEGORY_OTHER = 0x0b;

  /**
   * CMAS alert category is unknown. The category is only available for CDMA broadcasts
   * containing a type 1 elements record, so GSM and UMTS broadcasts always return unknown.
   */
  static constexpr int CMAS_CATEGORY_UNKNOWN = -1;

  // CMAS response type (in CDMA type 1 elements record).

  /** CMAS response type: Take shelter in place. */
  static constexpr int CMAS_RESPONSE_TYPE_SHELTER = 0x00;

  /** CMAS response type: Evacuate (Relocate). */
  static constexpr int CMAS_RESPONSE_TYPE_EVACUATE = 0x01;

  /** CMAS response type: Make preparations. */
  static constexpr int CMAS_RESPONSE_TYPE_PREPARE = 0x02;

  /** CMAS response type: Execute a pre-planned activity. */
  static constexpr int CMAS_RESPONSE_TYPE_EXECUTE = 0x03;

  /** CMAS response type: Attend to information sources. */
  static constexpr int CMAS_RESPONSE_TYPE_MONITOR = 0x04;

  /** CMAS response type: Avoid hazard. */
  static constexpr int CMAS_RESPONSE_TYPE_AVOID = 0x05;

  /** CMAS response type: Evaluate the information in this message (not for public warnings). */
  static constexpr int CMAS_RESPONSE_TYPE_ASSESS = 0x06;

  /** CMAS response type: No action recommended. */
  static constexpr int CMAS_RESPONSE_TYPE_NONE = 0x07;

  /**
   * CMAS response type is unknown. The response type is only available for CDMA broadcasts
   * containing a type 1 elements record, so GSM and UMTS broadcasts always return unknown.
   */
  static constexpr int CMAS_RESPONSE_TYPE_UNKNOWN = -1;

  // 4-bit CMAS severity (in GSM/UMTS message identifier or CDMA type 1 elements record).

  /** CMAS severity type: Extraordinary threat to life or property. */
  static constexpr int CMAS_SEVERITY_EXTREME = 0x0;

  /** CMAS severity type: Significant threat to life or property. */
  static constexpr int CMAS_SEVERITY_SEVERE = 0x1;

  /**
   * CMAS alert severity is unknown. The severity is available for CDMA warning alerts
   * containing a type 1 elements record and for all GSM and UMTS alerts except for the
   * Presidential-level alert class (Korean Public Alert System Class 0).
   */
  static constexpr int CMAS_SEVERITY_UNKNOWN = -1;

  // CMAS urgency (in GSM/UMTS message identifier or CDMA type 1 elements record).

  /** CMAS urgency type: Responsive action should be taken immediately. */
  static constexpr int CMAS_URGENCY_IMMEDIATE = 0x0;

  /** CMAS urgency type: Responsive action should be taken within the next hour. */
  static constexpr int CMAS_URGENCY_EXPECTED = 0x1;

  /**
   * CMAS alert urgency is unknown. The urgency is available for CDMA warning alerts
   * containing a type 1 elements record and for all GSM and UMTS alerts except for the
   * Presidential-level alert class (Korean Public Alert System Class 0).
   */
  static constexpr int CMAS_URGENCY_UNKNOWN = -1;

  // CMAS certainty (in GSM/UMTS message identifier or CDMA type 1 elements record).

  /** CMAS certainty type: Determined to have occurred or to be ongoing. */
  static constexpr int CMAS_CERTAINTY_OBSERVED = 0x0;

  /** CMAS certainty type: Likely (probability > ~50%). */
  static constexpr int CMAS_CERTAINTY_LIKELY = 0x1;

  /**
   * CMAS alert certainty is unknown. The certainty is available for CDMA warning alerts
   * containing a type 1 elements record and for all GSM and UMTS alerts except for the
   * Presidential-level alert class (Korean Public Alert System Class 0).
   */
  static constexpr int CMAS_CERTAINTY_UNKNOWN = -1;

  /**
   * Returns the CMAS message class, e.g. {@link #CMAS_CLASS_PRESIDENTIAL_LEVEL_ALERT}.
   * @return one of the {@code CMAS_CLASS} values
   */
  int getMessageClass() {
    return mMessageClass;
  }

  /**
   * Returns the CMAS category, e.g. {@link #CMAS_CATEGORY_GEO}.
   * @return one of the {@code CMAS_CATEGORY} values
   */
  int getCategory() {
    return mCategory;
  }

  /**
   * Returns the CMAS response type, e.g. {@link #CMAS_RESPONSE_TYPE_SHELTER}.
   * @return one of the {@code CMAS_RESPONSE_TYPE} values
   */
  int getResponseType() {
    return mResponseType;
  }

  /**
   * Returns the CMAS severity, e.g. {@link #CMAS_SEVERITY_EXTREME}.
   * @return one of the {@code CMAS_SEVERITY} values
   */
  int getSeverity() {
    return mSeverity;
  }

  /**
   * Returns the CMAS urgency, e.g. {@link #CMAS_URGENCY_IMMEDIATE}.
   * @return one of the {@code CMAS_URGENCY} values
   */
  int getUrgency() {
    return mUrgency;
  }

  /**
   * Returns the CMAS certainty, e.g. {@link #CMAS_CERTAINTY_OBSERVED}.
   * @return one of the {@code CMAS_CERTAINTY} values
   */
  int getCertainty() {
    return mCertainty;
  }

  std::string toString() {
    return "SmsCbCmasInfo{messageClass=" + std::to_string(mMessageClass)
        + ", category=" + std::to_string(mCategory) + ", responseType="
        + std::to_string(mResponseType) + ", severity="
        + std::to_string(mSeverity) + ", urgency=" + std::to_string(mUrgency)
        + ", certainty=" + std::to_string(mCertainty) + '}';
  }

  /**
   * Describe the kinds of special objects contained in the marshalled representation.
   * @return a bitmask indicating this Parcelable contains no special objects
   */
  int describeContents() {
    return 0;
  }

private:
  /** CMAS message class. */
  int mMessageClass;

  /** CMAS category. */
  int mCategory;

  /** CMAS response type. */
  int mResponseType;

  /** CMAS severity. */
  int mSeverity;

  /** CMAS urgency. */
  int mUrgency;

  /** CMAS certainty. */
  int mCertainty;
};

#endif /* SMSCBCMASINFO_H_ */
