/*
 * Copyright (C) 2012 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#define LOG_TAG "AUTOSUSPEND"
//#define LOG_NDEBUG 0

#include <errno.h>
#include <fcntl.h>
#include <pthread.h>
#include <semaphore.h>
#include <stddef.h>
#include <stdbool.h>
#include <string.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/time.h>

#include <unistd.h>
#include <dlfcn.h>
//#include <cutils/properties.h>
#define USER_LOG_TAG "PMS"
#include <liblog/lynq_deflog.h>
#include <log/log.h>
#include <stdlib.h>
#include "autosuspend_ops.h"

#define SYS_POWER_STATE "/sys/power/state"
#define SYS_POWER_WAKEUP_COUNT "/sys/power/wakeup_count"
#define SYS_POWER_SPM_SUSPEND_CRTL "/sys/power/spm/suspend_ctrl"

#define BASE_SLEEP_TIME 100000
#define POSSIBLE_MAX_SLEEP_TIME 60000000

#define LOG_UCI_MODULE "lynq_autosuspend"
#define LOG_UCI_FILE "lynq_uci"


static int state_fd;
static int wakeup_count_fd;
static int suspend_ctrl_fd;
static pthread_t suspend_thread;
static sem_t suspend_lockout;
extern pthread_cond_t feedback_cond;
extern pthread_mutex_t feedback_mutex;
extern pthread_mutex_t time_info_mutex;
static const char *sleep_state = "mem";
static const char *reg_netsys[5] = {"reg_netsys_srcclkena_mask_b 0",
                                    "reg_netsys_infra_req_mask_b 0",
                                    "reg_netsys_apsrc_req_mask_b 0",
                                    "reg_netsys_vrf18_req_mask_b 0",
                                    "reg_netsys_ddr_en_mask_b 0"   };
static void (*wakeup_func)(bool success) = NULL;
static int sleep_time = BASE_SLEEP_TIME;
static int possible_max_sleep_time = POSSIBLE_MAX_SLEEP_TIME;

static void* dlHandle_sim;
extern int adb_debug_mode;

int (*lynq_screen)(int num);
int (*lynq_sim_init)(int utoken);

static long start_time;  // 出错点：time_info_t 结构体两个成员都是long，因此这两个变量必须是long型，不能定义成int
static long end_time; 


# define TEMP_FAILURE_RETRY(expression) \
  (__extension__							      \
    ({ long int __result;						      \
       do __result = (long int) (expression);				      \
       while (__result == -1L && errno == EINTR);			      \
       __result; }))




void *dlHandle_wakelock;
void *dlHandle_log;
void *dlHandle_network;

int (*acquire_wake_lock)(int lock, const char* id);
int (*release_wake_lock)(const char* id);
int (*lynq_query_registration_state)(const char *type,int* regState,int* imsRegState,char * LAC,char * CID,int *netType,int *radioTechFam,int *errorCode);
int (*lynq_network_init)(int utoken);

pid_t pid = 0;

enum {
    PARTIAL_WAKE_LOCK = 1,  // the cpu stays on, but the screen is off
    FULL_WAKE_LOCK = 2      // the screen is also on
};

// while you have a lock held, the device will stay on at least at the
// level you request.

struct time_info_t
{
    long sleep_start_time;
    long wakeup_time;
};

extern struct time_info_t time_info;

struct timeval tv;

void init_wakelock_func(void)
{
    const char *lynqLibPath_WakeLock = "/usr/lib64/libpower.so";
    const char *lynqLibPath_Log = "/lib64/liblynq-log.so";

    dlHandle_wakelock = dlopen(lynqLibPath_WakeLock, RTLD_NOW);
    if (dlHandle_wakelock == NULL) 
    {
        ALOGI("dlopen lynqLibPath_WakeLock failed: %s", dlerror());
        exit(EXIT_FAILURE);
    }
    dlHandle_log = dlopen(lynqLibPath_Log, RTLD_NOW);
    if (dlHandle_log == NULL) 
    {
        ALOGI("dlopen dlHandle_log failed: %s", dlerror());
        exit(EXIT_FAILURE);
    }
    acquire_wake_lock = (int(*)(int,const char*))dlsym(dlHandle_wakelock, "acquire_wake_lock");
    if (acquire_wake_lock == NULL) {
        ALOGI("acquire_wake_lock not defined or exported in %s", lynqLibPath_WakeLock);
        exit(EXIT_FAILURE);
    }
    release_wake_lock = (int(*)( const char*))dlsym(dlHandle_wakelock, "release_wake_lock");
    if (release_wake_lock == NULL) {
        ALOGI("release_wake_lock not defined or exported in %s", lynqLibPath_WakeLock);
        exit(EXIT_FAILURE);
    }
    dlerror(); // Clear any previous dlerror

  return;
}



 void init_sim_func()
{
    int res;
    const char *lynqLibPath_Sim = "/lib64/liblynq-sim.so";

    pid = getpid();
    dlHandle_sim = dlopen(lynqLibPath_Sim, RTLD_NOW);
    if (dlHandle_sim == NULL) 
    {
        ALOGI("dlopen lynqLibPath_Sim failed: %s", dlerror());
        exit(EXIT_FAILURE);
    }

    lynq_screen = (int(*)(int))dlsym(dlHandle_sim, "lynq_screen");
    if (lynq_screen == NULL) {
        ALOGI("lynq_screen not defined or exported in %s", lynqLibPath_Sim);
        exit(EXIT_FAILURE);
    }

    lynq_sim_init = (int(*)(int utoken))dlsym(dlHandle_sim,"lynq_sim_init");
    if (lynq_sim_init == NULL) {
        ALOGI("lynq_sim_init not defined or exported in %s", lynqLibPath_Sim);
        exit(EXIT_FAILURE);
    }
    dlerror(); // Clear any previous dlerror

    res = lynq_sim_init((int)pid);
    if(res == 0)
    {
        ALOGI("Run lynq_sim_init\n");
    }else{
        ALOGI("lynq sim init error\n");
    }
    sleep(1);    

    return;            
}


void init_network_func()
{
    int res;
    const char *lynqLibPath_Network = "/lib64/liblynq-network.so";
    dlHandle_network = dlopen(lynqLibPath_Network, RTLD_NOW);
    if (dlHandle_network == NULL) 
    {
        ALOGI("dlopen lynqLibPath_Network failed: %s", dlerror());
        exit(EXIT_FAILURE);
    }

    lynq_query_registration_state = (int(*)(const char*,int*,int*,char *,char *,int *,int *,int*))dlsym(dlHandle_network, "lynq_query_registration_state");
    if (lynq_query_registration_state == NULL) {
        ALOGI("lynq_query_registration_state not defined or exported in %s", lynqLibPath_Network);
        exit(EXIT_FAILURE);
    }

    lynq_network_init = (int(*)(int))dlsym(dlHandle_network, "lynq_network_init");
    if (lynq_network_init == NULL) {
        ALOGI("lynq_network_init not defined or exported in %s", lynqLibPath_Network);
        exit(EXIT_FAILURE);
    }

    ALOGI("start lynq_network_init\n");
    printf("start lynq_network_init\n");
    res = lynq_network_init(2);
    sleep(10);

    if(res == 0)
    {
        ALOGI("Run lynq_network_init\n");
        printf("Run lynq_network_init\n");
    }else{
        ALOGI("lynq_network_init error\n");
        printf("lynq_network_init error\n");
    }       

    dlerror(); // Clear any previous dlerror
    return;            
}

static int suspend_ctrl(char *wakeup_count,int wakeup_count_len)
{
    char buf[80];
    int ret = 0;
    int i;
    int flag = -1;
    char tmp[20];
    system("echo \"Sys standby mode\" >/dev/console");

    lynq_get_value(LOG_UCI_FILE, LOG_UCI_MODULE, "debug", tmp);
    adb_debug_mode=atoi(tmp);

    if(adb_debug_mode == 2)
    {
        system("echo 11 | emdlogger_ctrl socket");
    }

    if (lynq_screen(0) != 0)  //notify ril for screen off
    {
        ALOGI("lynq_screen off fail\n");
        return -1;
    }

    system("hwclock -w");
    RLOGD("TIME: sys to rtc\n");
    system("echo \"autosuspend:Sys seek\" >/dev/console");

    RLOGD("suspend_ctrl: start write reg_netsys\n");
    for(int i = 0;i < 5;i++)  //notify spm (other core) to handle pre-sleep configuration
    {
        if(TEMP_FAILURE_RETRY(write(suspend_ctrl_fd,reg_netsys[i],strlen(reg_netsys[i]))) < 0)
        {
            strerror_r(errno, buf, sizeof(buf));
            ALOGI("Error writing to %s: %s\n", SYS_POWER_SPM_SUSPEND_CRTL, buf);
            return -1;
        }    
    }
    for(i=0;i<5;i++)
    {
        RLOGD("suspend_ctrl: start read wakeup_count\n");
        lseek(wakeup_count_fd, 0, SEEK_SET);
        wakeup_count_len = TEMP_FAILURE_RETRY(read(wakeup_count_fd, wakeup_count,
               sizeof(wakeup_count)));
        if (wakeup_count_len < 0)
        {
            strerror_r(errno, buf, sizeof(buf));
            ALOGE("Error reading from %s: %s\n", SYS_POWER_WAKEUP_COUNT, buf);
            wakeup_count_len = 0;
            continue;
        }
    
        RLOGD("suspend_ctrl: start write wakkeup_count\n");
        ret = write(wakeup_count_fd, wakeup_count, wakeup_count_len);
        if (ret < 0)
        {
            strerror_r(errno, buf, sizeof(buf));
            RLOGD("Error writing to %s: %s\n", SYS_POWER_WAKEUP_COUNT,buf);
            continue;
        }
        system("echo \"autosuspend:Sys suspend\" >/dev/console");
        RLOGD("suspend_ctrl: start write power_state\n");
        if(TEMP_FAILURE_RETRY(write(state_fd, sleep_state, strlen(sleep_state))) < 0) //enter suspend procedures in kernel
        {
            strerror_r(errno, buf, sizeof(buf));
            ALOGI("Error writing to %s: %s\n", SYS_POWER_STATE, buf);
            continue;
        }
        else
        {
            flag = 0;
            break;
        }

    }
    if(flag == 0)
    {
        return 0;
    }
    else
    {
        return -1;
    }
}


void wakeup_feedback(bool success)
{
    char buf[80];
    system("hwclock -s");
    RLOGD("TIME: rtc to sys\n");
    if (!success)
    {
       if(adb_debug_mode == 2)
        {
            system("echo 12 | emdlogger_ctrl socket");
        }

       usleep(200000); 
       ALOGI("Log on with failure\n");
       return ;
    }

    if (lynq_screen(1) != 0)  // notify ril for screen on
    {
        ALOGI("lynq_screen on fail\n");
    }

     if(adb_debug_mode == 2)
    {
        system("echo 12 | emdlogger_ctrl socket");
    }


    usleep(300000); //delay 2s for ril handling screen on,at least 70ms

    pthread_mutex_lock(&time_info_mutex);

    memset(&tv,0,sizeof(struct timeval));
    gettimeofday(&tv,NULL);
    // time_info.wakeup_time = tv.tv_sec * 1000 + tv.tv_usec / 1000;
    end_time = tv.tv_sec * 1000 + tv.tv_usec / 1000;
    ALOGI("%s: wake up time: %ld ms\n", __func__,end_time);

    memset(&time_info,0,sizeof(struct time_info_t));

    time_info.sleep_start_time = start_time;
    time_info.wakeup_time = end_time;
   
    pthread_mutex_unlock(&time_info_mutex);

    if (pthread_cond_broadcast(&feedback_cond) != 0) {
        strerror_r(errno, buf, sizeof(buf));
        ALOGI("Error broadcast cond: %s\n", buf);
    }   

    return ;

}

static void update_sleep_time(bool success) {
    if (success) {
        sleep_time = BASE_SLEEP_TIME;
        return;
    }
    // double sleep time after each failure up to one minute
    sleep_time = MIN(sleep_time * 2, possible_max_sleep_time);
}

static void *suspend_thread_func(void *arg __attribute__((unused)))
{
    char buf[80];
    char wakeup_count[20];
    int wakeup_count_len;
    int ret;
    bool success = true;

    while (1) {
        update_sleep_time(success);
        usleep(sleep_time);
        success = false;

        RLOGD("%s: read wakeup_count\n", __func__);
        lseek(wakeup_count_fd, 0, SEEK_SET);
        wakeup_count_len = read(wakeup_count_fd, wakeup_count,
                sizeof(wakeup_count));
        if (wakeup_count_len <= 0) {
            strerror_r(errno, buf, sizeof(buf));
            RLOGD("Error reading from %s: %s\n", SYS_POWER_WAKEUP_COUNT, buf);
            wakeup_count_len = 0;
            continue;
        }

        RLOGD("%s: wait\n", __func__);
        ret = sem_wait(&suspend_lockout);
        if (ret < 0) {
            strerror_r(errno, buf, sizeof(buf));
            RLOGD("Error waiting on semaphore: %s\n", buf);
            continue;
        }

        ret = write(wakeup_count_fd, wakeup_count, wakeup_count_len);
        if (ret < 0) {
            strerror_r(errno, buf, sizeof(buf));
            RLOGD("Error writing to %s: %s\n", SYS_POWER_WAKEUP_COUNT,buf);
            RLOGD("%s: sem_post\n", __func__);
            ret = sem_post(&suspend_lockout);
            if (ret < 0)
            {
                strerror_r(errno, buf, sizeof(buf));
                ALOGI("Error releasing semaphore: %s\n", buf);
            }
            continue;
        }

        RLOGD("%s: start suspend_ctrl\n", __func__);
        memset(&tv,0,sizeof(struct timeval));
        // memset(&time_info,0 ,sizeof(struct time_info_t));
        gettimeofday(&tv,NULL);

        start_time = tv.tv_sec * 1000 + tv.tv_usec / 1000;
        ALOGI("%s: suspend start time: %ld ms\n", __func__,start_time);
        // time_info.sleep_start_time = tv.tv_sec * 1000 + tv.tv_usec / 1000;         

        ret = suspend_ctrl(wakeup_count,wakeup_count_len);

        if (ret >= 0) {
                ALOGI("suspend_ctrl success.\n");
                success = true;
         }
         else
         {
                ALOGI("suspend_ctrl false.\n");
                success = false;
         }

         void (*func)(bool success) = wakeup_func;
         if (func != NULL) {
                (*func)(success); //handling resume event for other libs /apps
            }

         ALOGI("%s: release sem\n", __func__);
         ret = sem_post(&suspend_lockout);
         if (ret < 0) {
            strerror_r(errno, buf, sizeof(buf));
            ALOGI("Error releasing semaphore: %s\n", buf);
         } 

        //  ALOGI("%s: START SLEEP\n", __func__);
        //  if (lynq_screen(1) < 0)  // notify ril for screen on
        //  {
        //     ALOGI("lynq_screen on fail\n");
       
        //  }

        //  system("mdlogctl start");

        // //  sleep(1);

        //  system("echo 8 | emdlogger_ctrl");
         
        //  sleep(2); //delay 2s for ril handling screen on

         if(adb_debug_mode == 2) // it's neccessary to wait for autosuspend_disable function calling in debug mode when finish resume procedure.
         {
   
 
           sleep(40);
         }
         else  
         {
            ALOGI("%s: adb_debug unsupported\n", __func__);
         }
        // LYDBGLOG("%s: [936]START wake_lock\n", __func__);
        // if (acquire_wake_lock(PARTIAL_WAKE_LOCK,"aaa") < 0)
        // {
        //     LYDBGLOG("acquire_wake_lock : aaa fail.\n");
        // }
        // else  
        // {
        //     LYDBGLOG("acquire_wake_lock : aaa success.\n");
        // }

        ALOGI("%s: END SLEEP\n", __func__);
    }
    return NULL;
}

static int autosuspend_wakeup_count_enable(void)
{
    char buf[80];
    int ret;

    ALOGI("autosuspend_wakeup_count_enable\n");

    ret = sem_post(&suspend_lockout);

    if (ret < 0) {
        strerror_r(errno, buf, sizeof(buf));
        ALOGI("Error changing semaphore: %s\n", buf);
    }

    ALOGI("autosuspend_wakeup_count_enable done\n");

    return ret;
}

static int autosuspend_wakeup_count_disable(void)
{
    char buf[80];
    int ret;

    ALOGI("autosuspend_wakeup_count_disable\n");

    ret = sem_wait(&suspend_lockout);

    if (ret < 0) {
        strerror_r(errno, buf, sizeof(buf));
        ALOGI("Error changing semaphore: %s\n", buf);
    }

    ALOGI("autosuspend_wakeup_count_disable done\n");

    return ret;
}

void set_wakeup_callback(void (*func)(bool success))
{
    if (wakeup_func != NULL) {
        ALOGI("Duplicate wakeup callback applied, keeping original");
        return;
    }
    wakeup_func = func;
}

struct autosuspend_ops autosuspend_wakeup_count_ops = {
        .enable = autosuspend_wakeup_count_enable,
        .disable = autosuspend_wakeup_count_disable,
};

struct autosuspend_ops *autosuspend_wakeup_count_init(void)
{
    int ret;
    char buf[80];
    char timeout_str[100]="100000";

    //if (property_get("sys.autosuspend.timeout", timeout_str, NULL))
    {
        possible_max_sleep_time = atoi(timeout_str);
        ALOGI("autosuspend timeout is %d\n", possible_max_sleep_time);
    }

    state_fd = TEMP_FAILURE_RETRY(open(SYS_POWER_STATE, O_RDWR));
    if (state_fd < 0) {
        strerror_r(errno, buf, sizeof(buf));
        ALOGI("Error opening %s: %s\n", SYS_POWER_STATE, buf);
        goto err_open_state;
    }

    wakeup_count_fd = TEMP_FAILURE_RETRY(open(SYS_POWER_WAKEUP_COUNT, O_RDWR));
    if (wakeup_count_fd < 0) {
        strerror_r(errno, buf, sizeof(buf));
        ALOGI("Error opening %s: %s\n", SYS_POWER_WAKEUP_COUNT, buf);
        goto err_open_wakeup_count;
    }

    suspend_ctrl_fd = TEMP_FAILURE_RETRY(open(SYS_POWER_SPM_SUSPEND_CRTL, O_RDWR));
    if (suspend_ctrl_fd < 0) {
        strerror_r(errno, buf, sizeof(buf));
        ALOGI("Error opening %s: %s\n", SYS_POWER_SPM_SUSPEND_CRTL, buf);
        goto err_open_suspend_ctrl;
    }    

    ret = sem_init(&suspend_lockout, 0, 0);
    if (ret < 0) {
        strerror_r(errno, buf, sizeof(buf));
        ALOGI("Error creating semaphore: %s\n", buf);
        goto err_sem_init;
    }
    ret = pthread_create(&suspend_thread, NULL, suspend_thread_func, NULL);
    if (ret) {
        strerror_r(ret, buf, sizeof(buf));
        ALOGI("Error creating thread: %s\n", buf);
        goto err_pthread_create;
    }

    ALOGI("Selected wakeup count\n");
    return &autosuspend_wakeup_count_ops;

err_pthread_create:
    sem_destroy(&suspend_lockout);
err_sem_init:
    close(wakeup_count_fd);
err_open_wakeup_count:
    close(state_fd);
err_open_suspend_ctrl:
    close(suspend_ctrl_fd);
err_open_state:
    return NULL;
}
