#include <stdio.h>
#include <sys/socket.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <netinet/in.h>
#include <sys/un.h>
#include <log/log.h>


#include "lynq_gnss.h"
#include "lynq_agps.h"

#define LOG_TAG "LYNQ_AGPS"


// -1 means failure
int do_socket_connect(const char* path)
{
    struct sockaddr_un addr;
    int fd = socket(PF_LOCAL, SOCK_STREAM, 0);
    if(fd < 0) {
        RLOGD("socket() failed fd=%d\n", fd);
        return -1;
    }

    memset(&addr, 0, sizeof(addr));
    addr.sun_path[0] = 0;
    memcpy(addr.sun_path + 1, path, strlen(path));
    addr.sun_family = AF_UNIX;

    if (connect(fd, (struct sockaddr *)&addr, sizeof(addr)) < 0) {
        RLOGD("connect failed reason=[%s] path=[%s]\n", strerror(errno), path);
        close(fd);
        return -1;
    }

    return fd;
}

//get
char get_byte(char* buff, int* offset)
{
    char ret = buff[*offset];
    *offset += 1;
    return ret;
}

short get_short(char* buff, int* offset)
{
    short ret = 0;
    ret |= get_byte(buff, offset) & 0xff;
    ret |= (get_byte(buff, offset) << 8);
    return ret;
}

int get_int(char* buff, int* offset)
{
    int ret = 0;
    ret |= get_short(buff, offset) & 0xffff;
    ret |= (get_short(buff, offset) << 16);
    return ret;
}

int socket_connect() {
    return do_socket_connect("agpsd2");
}

//-1 means failure
static int safe_write(int fd, void* buf, int len)
{
    int n, retry = 10;

    if(fd < 0 || buf == NULL || len < 0) {
        RLOGD("safe_write fd=%d buf=%p len=%d\n", fd, buf, len);
        return -1;
    }

    while((n = write(fd, buf, len)) != len) {
        if(errno == EINTR) continue;
        if(errno == EAGAIN) {
            if(retry-- > 0) {
                usleep(100 * 1000);
                continue;
            }
            goto exit;
        }
        goto exit;
    }
    return n;
exit:
    RLOGD("safe_write reason=[%s]%d\n", strerror(errno), errno);
    return -1;
}

//-1 means failure
static int safe_read(int fd, void* buf, int len)
{
    int n, retry = 10;

    if(fd < 0 || buf == NULL || len < 0)
    {
        RLOGD("safe_read fd=%d buf=%p len=%d\n", fd, buf, len);
        return -1;
    }

    if(len == 0)
    {
        return 0;
    }

    while((n = read(fd, buf, len)) < 0)
    {
        if(errno == EINTR) 
        {
            RLOGD("safe read interrupt");
            continue;
        }
        if(errno == EAGAIN)
        {
            if(retry-- > 0)
            {
                usleep(100 * 1000);
                continue;
            }
            goto exit;
        }
        goto exit;
    }
    return n;

exit:
    if(errno != EAGAIN)
    {
        RLOGD("safe_read reason=[%s] fd=%d len=%d buf=%p\n",
            strerror(errno), fd, len, buf);
    }
    return -1;
}

static char socket_get_byte(int fd)
{
    int read_len = 0;
    char buff[1] = {0};
    int offset = 0;

    read_len = safe_read(fd, buff, sizeof(buff));
    if(read_len != sizeof(buff))
    {
        RLOGD("socket_get_byte read_len=%d\n", read_len);
    }
    return get_byte(buff, &offset);
}

static int socket_get_int(int fd)
{
    int read_len = 0;
    char buff[4] = {0};
    int offset = 0;

    read_len = safe_read(fd, buff, sizeof(buff));
    if(read_len != sizeof(buff))
    {
        RLOGD("socket_get_int read_len=%d\n", read_len);
    }
    return get_int(buff, &offset);
}

static char* socket_get_string(int fd, char* buff, int buff_len)
{
    int read_len = 0;

    char ret = socket_get_byte(fd);
    if(ret == 0)
    {
        return NULL;
    }
    else
    {
        int len = socket_get_int(fd);
        if(len > buff_len)
        {
            RLOGD("socket_get_string your buff len=%d is too small, need len=%d\n",
                buff_len, len);
            return NULL;
        }

        read_len = safe_read(fd, buff, len);
        if(read_len != len)
        {
            RLOGD("socket_get_string read_len=%d len=%d\n", read_len, len);
            return NULL;
        }

        return buff;
    }
}


static int socket_get_binary(int fd, char* buff)
{
    int read_len = 0;

    int len = socket_get_int(fd);
    if(len > 0)
    {
        read_len = safe_read(fd, buff, len);
        if(read_len != len)
        {
            RLOGD("socket_get_binary read_len=%d len=%d\n", read_len, len);
            return 0;
        }
    }
    return len;
}

void put_byte(char* buff, int* offset, const char input)
{
    *((char*)&buff[*offset]) = input;
    *offset += 1;
}

void put_short(char* buff, int* offset, const short input)
{
    put_byte(buff, offset, input & 0xff);
    put_byte(buff, offset, (input >> 8) & 0xff);
}

void put_int(char* buff, int* offset, const int input)
{
    put_short(buff, offset, input & 0xffff);
    put_short(buff, offset, (input >> 16) & 0xffff);
}

int set_template_1(app_mgr_cmd_enum cmd,char data)
{
    char buff[MAX_BUFF_SIZE] = {0};
    int offset = 0;
    int fd = 0;

    fd = socket_connect();
    if(fd < 0)
    {
        RLOGD("Socket connect fail");
        return -1;
    }
    RLOGD("Set template start fd:%d cmd:%d data:%d",fd,cmd,data);
    // write
    put_int(buff, &offset, cmd);
    put_byte(buff, &offset, data);
    if(safe_write(fd, buff, offset) == -1)
    {
        RLOGD("set_template_1 Safe write fail");
        close(fd);
        return -1;
    }

    // read ACK
    socket_get_byte(fd);

    close(fd);
    return 0;
}

int get_template_agps_config(app_mgr_cmd_enum cmd,agps_intf_agps_config *config)
{
    char buff[MAX_BUFF_SIZE] = {0};
    int offset = 0;
    int fd = 0;

    fd = socket_connect();
    if(fd < 0)
    {
        RLOGD("Socket connect fail");
        return -1;
    }
    RLOGD("get_template_agps_config start fd:%d cmd:%d",fd,cmd);

    // write
    put_int(buff, &offset, cmd);
    if(safe_write(fd, buff, offset) == -1)
    {
        RLOGD("get_template_agps_config Safe write fail");
        close(fd);
        return -1;
    }

    // read
    config->agps_setting.agps_enable = socket_get_byte(fd);
    config->agps_setting.agps_protocol = socket_get_int(fd);
    config->agps_setting.gpevt = socket_get_byte(fd);

    config->cp_setting.molr_pos_method = socket_get_int(fd);
    config->cp_setting.external_addr_enable = socket_get_byte(fd);
    socket_get_string(fd, config->cp_setting.external_addr, sizeof(config->cp_setting.external_addr));
    config->cp_setting.mlc_number_enable = socket_get_byte(fd);
    socket_get_string(fd, config->cp_setting.mlc_number, sizeof(config->cp_setting.mlc_number));
    config->cp_setting.cp_auto_reset = socket_get_byte(fd);
    config->cp_setting.epc_molr_lpp_payload_enable = socket_get_byte(fd);
    config->cp_setting.epc_molr_lpp_payload_len =
        socket_get_binary(fd, config->cp_setting.epc_molr_lpp_payload);

    config->up_setting.ca_enable = socket_get_byte(fd);
    config->up_setting.ni_request = socket_get_byte(fd);
    config->up_setting.roaming = socket_get_byte(fd);
    config->up_setting.cdma_preferred = socket_get_int(fd);
    config->up_setting.pref_method = socket_get_int(fd);
    config->up_setting.supl_version = socket_get_int(fd);
    config->up_setting.tls_version = socket_get_int(fd);
    config->up_setting.supl_log = socket_get_byte(fd);
    config->up_setting.msa_enable = socket_get_byte(fd);
    config->up_setting.msb_enable = socket_get_byte(fd);
    config->up_setting.ecid_enable = socket_get_byte(fd);
    config->up_setting.otdoa_enable = socket_get_byte(fd);
    config->up_setting.qop_hacc = socket_get_int(fd);
    config->up_setting.qop_vacc = socket_get_int(fd);
    config->up_setting.qop_loc_age = socket_get_int(fd);
    config->up_setting.qop_delay = socket_get_int(fd);
    config->up_setting.lpp_enable = socket_get_byte(fd);
    config->up_setting.cert_from_sdcard = socket_get_byte(fd);
    if(cmd >= APP_MGR_CMD_GET_CONFIG_V14)
    {
        config->up_setting.auto_profile_enable = socket_get_byte(fd);
        config->up_setting.ut2 = socket_get_byte(fd);
        config->up_setting.ut3 = socket_get_byte(fd);
        config->up_setting.apn_enable = socket_get_byte(fd);
        config->up_setting.sync_to_slp = socket_get_byte(fd);
        config->up_setting.udp_enable = socket_get_byte(fd);
        config->up_setting.autonomous_enable = socket_get_byte(fd);
        config->up_setting.aflt_enable = socket_get_byte(fd);
        config->up_setting.imsi_enable = socket_get_byte(fd);

        config->gnss_setting.sib8_sib16_enable = socket_get_byte(fd);
        config->gnss_setting.gps_satellite_enable = socket_get_byte(fd);
        config->gnss_setting.glonass_satellite_enable = socket_get_byte(fd);
        config->gnss_setting.beidou_satellite_enable = socket_get_byte(fd);
        config->gnss_setting.galileo_satellite_enable = socket_get_byte(fd);
        config->gnss_setting.gps_satellite_support = socket_get_byte(fd);
        config->gnss_setting.glonass_satellite_support = socket_get_byte(fd);
        config->gnss_setting.beidou_satellite_support = socket_get_byte(fd);
        config->gnss_setting.galileo_satellite_support = socket_get_byte(fd);

        config->up_setting.supl_ver_minor = socket_get_byte(fd);
        config->up_setting.supl_ver_ser_ind = socket_get_byte(fd);

        config->gnss_setting.a_glonass_satellite_enable = socket_get_byte(fd);
    }

    socket_get_string(fd, config->cur_supl_profile.name, sizeof(config->cur_supl_profile.name));
    socket_get_string(fd, config->cur_supl_profile.addr, sizeof(config->cur_supl_profile.addr));
    config->cur_supl_profile.port = socket_get_int(fd);
    config->cur_supl_profile.tls = socket_get_byte(fd);
    socket_get_string(fd, config->cur_supl_profile.mcc_mnc, sizeof(config->cur_supl_profile.mcc_mnc));
    socket_get_string(fd, config->cur_supl_profile.app_id, sizeof(config->cur_supl_profile.app_id));
    socket_get_string(fd, config->cur_supl_profile.provider_id, sizeof(config->cur_supl_profile.provider_id));
    socket_get_string(fd, config->cur_supl_profile.default_apn, sizeof(config->cur_supl_profile.default_apn));
    socket_get_string(fd, config->cur_supl_profile.optional_apn, sizeof(config->cur_supl_profile.optional_apn));
    socket_get_string(fd, config->cur_supl_profile.optional_apn_2, sizeof(config->cur_supl_profile.optional_apn_2));
    socket_get_string(fd, config->cur_supl_profile.address_type, sizeof(config->cur_supl_profile.address_type));

    if(cmd >= APP_MGR_CMD_GET_CONFIG_V14)
    {
        socket_get_string(fd, config->cdma_profile.name, sizeof(config->cdma_profile.name));
        config->cdma_profile.mcp_enable = socket_get_byte(fd);;
        socket_get_string(fd, config->cdma_profile.mcp_addr, sizeof(config->cdma_profile.mcp_addr));
        config->cdma_profile.mcp_port = socket_get_int(fd);;
        config->cdma_profile.pde_addr_valid = socket_get_byte(fd);;
        config->cdma_profile.pde_ip_type = socket_get_int(fd);;
        socket_get_string(fd, config->cdma_profile.pde_addr, sizeof(config->cdma_profile.pde_addr));
        config->cdma_profile.pde_port = socket_get_int(fd);;
        config->cdma_profile.pde_url_valid = socket_get_byte(fd);;
        socket_get_string(fd, config->cdma_profile.pde_url_addr, sizeof(config->cdma_profile.pde_url_addr));
    }

    if(cmd >= APP_MGR_CMD_GET_CONFIG_V20)
    {
        //V15
        config->agps_setting.e911_gps_icon_enable = socket_get_byte(fd);
        //V16
        config->agps_setting.e911_open_gps = socket_get_byte(fd);
        //V17
        config->gnss_setting.a_gps_satellite_enable = socket_get_byte(fd);
        config->gnss_setting.a_beidou_satellite_enable = socket_get_byte(fd);
        config->gnss_setting.a_galileo_satellite_enable = socket_get_byte(fd);
        //V18
        config->up_setting.sha_version = socket_get_int(fd);
        config->up_setting.preferred_2g3g_cell_age = socket_get_int(fd);
        config->up_setting.ut1 = socket_get_byte(fd);
        config->up_setting.no_sensitive_log = socket_get_byte(fd);
        config->up_setting.tls_reuse_enable = socket_get_byte(fd);
        config->up_setting.imsi_cache_enable = socket_get_byte(fd);
        config->up_setting.supl_raw_data_enable = socket_get_byte(fd);
        config->up_setting.tc10_enable = socket_get_byte(fd);
        config->up_setting.tc10_use_apn = socket_get_byte(fd);
        config->up_setting.tc10_use_fw_dns = socket_get_byte(fd);
        config->up_setting.allow_ni_for_gps_off = socket_get_byte(fd);
        config->up_setting.force_otdoa_assist_req = socket_get_byte(fd);
        config->cp_setting.reject_non911_nilr_enable = socket_get_byte(fd);
        config->cp_setting.cp_2g_disable = socket_get_byte(fd);
        config->cp_setting.cp_3g_disable = socket_get_byte(fd);
        config->cp_setting.cp_4g_disable = socket_get_byte(fd);
        config->agps_setting.tc10_ignore_fw_config = socket_get_byte(fd);
        config->agps_setting.lppe_hide_wifi_bt_status = socket_get_byte(fd);
        //V19
        config->agps_setting.lppe_network_location_disable = socket_get_byte(fd);
        config->cp_setting.cp_lppe_enable = socket_get_byte(fd);
        config->up_setting.up_lppe_enable = socket_get_byte(fd);
        //V20
        config->cp_setting.support_cp_lppe = socket_get_byte(fd);
        config->gnss_setting.mnl_support_lppe = socket_get_byte(fd);
    }

    if(cmd >= APP_MGR_CMD_GET_CONFIG_V21)
    {
        config->agps_setting.agps_nvram_enable = socket_get_byte(fd);
        config->agps_setting.lbs_log_enable = socket_get_byte(fd);
        config->agps_setting.lppe_crowd_source_confident = socket_get_int(fd);

        config->up_setting.esupl_apn_mode = socket_get_int(fd);
        config->up_setting.tcp_keepalive = socket_get_int(fd);
        config->up_setting.aosp_profile_enable = socket_get_byte(fd);
        config->up_setting.bind_nlp_setting_to_supl = socket_get_byte(fd);
    }

    if(cmd >= APP_MGR_CMD_GET_CONFIG_V22)
    {
        config->agps_setting.ignore_si_for_e911 = socket_get_byte(fd);
        config->cp_setting.cp_lppe_wlan_enable = socket_get_byte(fd);
        config->cp_setting.cp_lppe_srn_enable = socket_get_byte(fd);
        config->cp_setting.cp_lppe_sensor_enable = socket_get_byte(fd);
        config->cp_setting.cp_lppe_dbh_enable = socket_get_byte(fd);

        config->up_setting.up_lppe_wlan_enable = socket_get_byte(fd);
        config->up_setting.up_lppe_srn_enable = socket_get_byte(fd);
        config->up_setting.up_lppe_sensor_enable = socket_get_byte(fd);
        config->up_setting.up_lppe_dbh_enable = socket_get_byte(fd);
        config->up_setting.ip_version_prefer = socket_get_int(fd);
        config->up_setting.up_lppe_in_2g3g_disable = socket_get_byte(fd);
        config->up_setting.up_rrlp_in_4g_disable = socket_get_byte(fd);
        config->up_setting.up_si_disable = socket_get_byte(fd);
    }

    if(cmd >= APP_MGR_CMD_GET_CONFIG_V23)
    {
        config->up_setting.use_ni_slp = socket_get_byte(fd);
        config->agps_setting.use_tc10_config = socket_get_byte(fd);
        config->agps_setting.lppe_def_nlp_enable = socket_get_byte(fd);
    }

    if(cmd >= APP_MGR_CMD_GET_CONFIG_V24)
    {
        config->agps_setting.emergency_ext_secs = socket_get_int(fd);
        config->up_setting.aosp_pos_mode_enable = socket_get_byte(fd);
        config->up_setting.privacy_override_mode = socket_get_int(fd);
    }

    config->valid = 1;
    // read ACK
    socket_get_byte(fd);
    RLOGD("Socket read ACK sucess, close fd");
    close(fd);
    return 0;
}

int agps_get_total_status(agps_intf_agps_config *config)
{
    if (NULL == config)
    {
        RLOGD("agps_get_total_status incoming paramter error");
    }
    int res = 0;
    res = get_template_agps_config(APP_MGR_CMD_GET_CONFIG_V24,config);
    return res;
}


int lynq_agps_set_enabled(LYNQ_CONF_SWITCH agps_status)
{
    int ret = -1;
    RLOGD("[LYNQ_GNSS]set agps:%d",agps_status);
    ret = set_template_1(APP_MGR_CMD_SET_AGPS_ENABLE,agps_status);
    if (ret != 0)
    {   
        RLOGD("set AGPS error ret = %d",ret);
        return ret;
    }
    return ret;
}

int lynq_agps_get_enabled_status(int *status)
{
    int ret = 0;
    if (NULL == status)
    {
        RLOGD("incoming paramter error");
        return -1;
    }

    agps_intf_agps_config config;
    memset(&config, 0, sizeof(config));
    ret = agps_get_total_status(&config);
    if (ret != 0)
    {
        RLOGD("agps get status fail");
        return ret;
    }
    *status = config.agps_setting.agps_enable;
    RLOGD("[LYNQ_GNSS]agps status:%d",*status);
    return ret;
}