/*
 * Driver interaction with extended Linux CFG8021
 * Copyright (c) 2012-2013, Broadcom Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * Alternatively, this software may be distributed under the terms of BSD
 * license.
 *
 */
#include "includes.h"
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <net/if.h>
#include <netpacket/packet.h>
#include <linux/filter.h>
#include <netlink/genl/genl.h>
#include <netlink/genl/family.h>
#include <netlink/genl/ctrl.h>
#include <linux/rtnetlink.h>

#include "common.h"
#include "eloop.h"
#include "common/ieee802_11_defs.h"
#include "netlink.h"
#include "linux_ioctl.h"
#include "radiotap.h"
#include "radiotap_iter.h"
#include "rfkill.h"
#include "drivers/driver_nl80211.h"

#include "wpa_supplicant_i.h"
#include "config.h"
#include "wpabuf.h"
#include "wpa_ctrl.h"
#include "p2p_supplicant.h"
#include "brcm_vendor.h"
#ifdef ANDROID
#include "android_drv.h"
#endif


#ifdef ANDROID_P2P
/* If any feature requires enabling of BCM_GENL code, put it here */
#endif /* ANDROID_P2P */

#ifndef IFF_LOWER_UP
#define IFF_LOWER_UP   0x10000         /* driver signals L1 up         */
#endif
#ifndef IFF_DORMANT
#define IFF_DORMANT    0x20000         /* driver signals dormant       */
#endif

#ifndef IF_OPER_DORMANT
#define IF_OPER_DORMANT 5
#endif
#ifndef IF_OPER_UP
#define IF_OPER_UP 6
#endif

#define WPA_PS_ENABLED		0
#define WPA_PS_DISABLED		1

#define MAX_WPSP2PIE_CMD_SIZE		1024


typedef struct android_wifi_priv_cmd {
	char *buf;
	int used_len;
	int total_len;
} android_wifi_priv_cmd;

#define SDO_FRAGMENT_NUM_MASK	 0x7f
#define SDO_MORE_FRAGMNT_MASK	 0x80
typedef struct sdo_hdr {
    u8	addr[ETH_ALEN];
    u16	freq;        /* channel */
    u8	count;        /* GAS fragment id */
	u16	update_ind;
} sdo_hdr_t;

/* service discovery TLV */
typedef struct sd_tlv {
     u16  length;         /* length of response_data */
     u8   protocol;       /* service protocol type */
     u8   transaction_id;     /* service transaction id */
     u8   status_code;        /* status code */
     u8   data[1];        /* response data */
} sd_tlv_t;

/* Service Protocol Type */
typedef enum svc_protype {
    SVC_RPOTYPE_ALL = 0,
    SVC_RPOTYPE_BONJOUR = 1,
    SVC_RPOTYPE_UPNP = 2,
    SVC_RPOTYPE_WSD = 3,
    SVC_RPOTYPE_VENDOR = 255
} svc_protype_t;

#ifdef CONFIG_BRCM_BT_WIFI_HO
typedef struct setup_netinfo {
        u8 opmode;
        u8 pad;
        u8 macaddr[ETH_ALEN];
        u32 ssid_len;
        u8 ssid[MAX_SSID_LEN];
        u8 passphrase_len;
        u8 passphrase[MAX_PASSPHRASE_LEN];
        u16 channel;
        u8 version;
} setup_netinfo_t;
#endif /* CONFIG_BRCM_BT_WIFI_HO */

#ifndef HOSTAPD
extern int wpas_enable_dd_offload(struct wpa_supplicant *wpa_s, u8 enable);
extern int wpas_get_listen_channel(void *priv);
extern void wpas_p2p_scan_res_handler(void *priv, struct wpa_scan_results *scan_res);
#endif
int send_and_recv_msgs(struct wpa_driver_nl80211_data *drv, struct nl_msg *msg,
		       int (*valid_handler)(struct nl_msg *, void *),
		       void *valid_data,
		       int (*ack_handler_custom)(struct nl_msg *, void *),
		       void *ack_data);
extern int nl80211_send_vendor_command(struct wpa_driver_nl80211_data *drv ,
					struct i802_bss *bss, int,  char *,
					size_t len, int vendor_id);



static int drv_errors = 0;

static void wpa_driver_send_hang_msg(struct wpa_driver_nl80211_data *drv)
{
	drv_errors++;
	if (drv_errors > DRV_NUMBER_SEQUENTIAL_ERRORS) {
		drv_errors = 0;
		wpa_msg(drv->ctx, MSG_INFO, WPA_EVENT_DRIVER_STATE "HANGED");
	}
}

static int wpa_driver_set_power_save(void *priv, int state)
{
	struct i802_bss *bss = priv;
	struct wpa_driver_nl80211_data *drv = bss->drv;
	struct nl_msg *msg;
	int ret = -1;
	enum nl80211_ps_state ps_state;

	msg = nlmsg_alloc();
	if (!msg)
		return -1;

	genlmsg_put(msg, 0, 0, drv->global->nl80211_id, 0, 0,
		    NL80211_CMD_SET_POWER_SAVE, 0);

	if (state == WPA_PS_ENABLED)
		ps_state = NL80211_PS_ENABLED;
	else
		ps_state = NL80211_PS_DISABLED;

	NLA_PUT_U32(msg, NL80211_ATTR_IFINDEX, drv->ifindex);
	NLA_PUT_U32(msg, NL80211_ATTR_PS_STATE, ps_state);

	ret = send_and_recv_msgs(drv, msg, NULL, NULL, NULL, NULL);
	msg = NULL;
	if (ret < 0)
		wpa_printf(MSG_ERROR, "nl80211: Set power mode fail: %d", ret);
nla_put_failure:
	nlmsg_free(msg);
	return ret;
}

static int get_power_mode_handler(struct nl_msg *msg, void *arg)
{
	struct nlattr *tb[NL80211_ATTR_MAX + 1];
	struct genlmsghdr *gnlh = nlmsg_data(nlmsg_hdr(msg));
	int *state = (int *)arg;

	nla_parse(tb, NL80211_ATTR_MAX, genlmsg_attrdata(gnlh, 0),
		  genlmsg_attrlen(gnlh, 0), NULL);

	if (!tb[NL80211_ATTR_PS_STATE])
		return NL_SKIP;

	if (state) {
		*state = (int)nla_get_u32(tb[NL80211_ATTR_PS_STATE]);
		wpa_printf(MSG_DEBUG, "nl80211: Get power mode = %d", *state);
		*state = (*state == NL80211_PS_ENABLED) ?
				WPA_PS_ENABLED : WPA_PS_DISABLED;
	}

	return NL_SKIP;
}

static int wpa_driver_get_power_save(void *priv, int *state)
{
	struct i802_bss *bss = priv;
	struct wpa_driver_nl80211_data *drv = bss->drv;
	struct nl_msg *msg;
	int ret = -1;
	enum nl80211_ps_state ps_state;

	msg = nlmsg_alloc();
	if (!msg)
		return -1;

	genlmsg_put(msg, 0, 0, drv->global->nl80211_id, 0, 0,
		    NL80211_CMD_GET_POWER_SAVE, 0);

	NLA_PUT_U32(msg, NL80211_ATTR_IFINDEX, drv->ifindex);

	ret = send_and_recv_msgs(drv, msg, get_power_mode_handler, state, NULL, NULL);
	msg = NULL;
	if (ret < 0)
		wpa_printf(MSG_ERROR, "nl80211: Get power mode fail: %d", ret);
nla_put_failure:
	nlmsg_free(msg);
	return ret;
}

static int wpa_driver_set_backgroundscan_params(void *priv)
{
	struct i802_bss *bss = priv;
	struct wpa_driver_nl80211_data *drv = bss->drv;
	struct wpa_supplicant *wpa_s;
	struct ifreq ifr;
	android_wifi_priv_cmd priv_cmd;
	int ret = 0, i = 0, bp;
	char buf[WEXT_PNO_MAX_COMMAND_SIZE];
	struct wpa_ssid *ssid_conf;

	if (drv == NULL) {
		wpa_printf(MSG_ERROR, "%s: drv is NULL. Exiting", __func__);
		return -1;
	}
	if (drv->ctx == NULL) {
		wpa_printf(MSG_ERROR, "%s: drv->ctx is NULL. Exiting", __func__);
		return -1;
	}
	wpa_s = (struct wpa_supplicant *)(drv->ctx);
	if (wpa_s->conf == NULL) {
		wpa_printf(MSG_ERROR, "%s: wpa_s->conf is NULL. Exiting", __func__);
		return -1;
	}
	ssid_conf = wpa_s->conf->ssid;

	bp = WEXT_PNOSETUP_HEADER_SIZE;
	os_memcpy(buf, WEXT_PNOSETUP_HEADER, bp);
	buf[bp++] = WEXT_PNO_TLV_PREFIX;
	buf[bp++] = WEXT_PNO_TLV_VERSION;
	buf[bp++] = WEXT_PNO_TLV_SUBVERSION;
	buf[bp++] = WEXT_PNO_TLV_RESERVED;

	while ((i < WEXT_PNO_AMOUNT) && (ssid_conf != NULL)) {
		/* Check that there is enough space needed for 1 more SSID, the other sections and null termination */
		if ((bp + WEXT_PNO_SSID_HEADER_SIZE + MAX_SSID_LEN + WEXT_PNO_NONSSID_SECTIONS_SIZE + 1) >= (int)sizeof(buf))
			break;
		if ((!ssid_conf->disabled) && (ssid_conf->ssid_len <= MAX_SSID_LEN)){
			wpa_printf(MSG_DEBUG, "For PNO Scan: %s", ssid_conf->ssid);
			buf[bp++] = WEXT_PNO_SSID_SECTION;
			buf[bp++] = ssid_conf->ssid_len;
			os_memcpy(&buf[bp], ssid_conf->ssid, ssid_conf->ssid_len);
			bp += ssid_conf->ssid_len;
			i++;
		}
		ssid_conf = ssid_conf->next;
	}

	buf[bp++] = WEXT_PNO_SCAN_INTERVAL_SECTION;
	os_snprintf(&buf[bp], WEXT_PNO_SCAN_INTERVAL_LENGTH + 1, "%x", WEXT_PNO_SCAN_INTERVAL);
	bp += WEXT_PNO_SCAN_INTERVAL_LENGTH;

	buf[bp++] = WEXT_PNO_REPEAT_SECTION;
	os_snprintf(&buf[bp], WEXT_PNO_REPEAT_LENGTH + 1, "%x", WEXT_PNO_REPEAT);
	bp += WEXT_PNO_REPEAT_LENGTH;

	buf[bp++] = WEXT_PNO_MAX_REPEAT_SECTION;
	os_snprintf(&buf[bp], WEXT_PNO_MAX_REPEAT_LENGTH + 1, "%x", WEXT_PNO_MAX_REPEAT);
	bp += WEXT_PNO_MAX_REPEAT_LENGTH + 1;

	memset(&ifr, 0, sizeof(ifr));
	memset(&priv_cmd, 0, sizeof(priv_cmd));
	os_memcpy(ifr.ifr_name, bss->ifname, IFNAMSIZ);

	priv_cmd.buf = buf;
	priv_cmd.used_len = bp;
	priv_cmd.total_len = bp;
	ifr.ifr_data = (void *)&priv_cmd;

	ret = ioctl(drv->global->ioctl_sock, SIOCDEVPRIVATE + 1, &ifr);

	if (ret < 0) {
		wpa_printf(MSG_ERROR, "ioctl[SIOCSIWPRIV] (pnosetup): %d", ret);
		wpa_driver_send_hang_msg(drv);
	} else {
		drv_errors = 0;
	}
	return ret;
}



static int wpa_get_best_channels(void *drv_ctx, char *buf)
{
	union wpa_event_data event;

	if (NULL == drv_ctx) {
		wpa_printf(MSG_ERROR, "%s: drv_ctx is NULL. Exiting", __func__);
		return -1;
	}

	memset(&event, 0, sizeof(event));

	sscanf(buf, "%04d %04d %04d", &(event.best_chan.freq_24),
		&(event.best_chan.freq_5), &(event.best_chan.freq_overall));

	wpa_supplicant_event(drv_ctx, EVENT_BEST_CHANNEL, &event);

	return 0;
}

int wpa_driver_nl80211_driver_cmd(void *priv, char *cmd, char *buf,
				  size_t buf_len )
{
	struct i802_bss *bss = priv;
	struct wpa_driver_nl80211_data *drv = bss->drv;
	struct ifreq ifr;
	android_wifi_priv_cmd priv_cmd;
	int ret = 0;
	struct wpa_supplicant *wpa_s;
	char *token = NULL, *cmd_ptr;
	enum brcm_nl80211_vendor_subcmds type;

	if (bss->ifindex <= 0 && bss->wdev_id > 0) {
		/*
		 * DRIVER CMD received on the DEDICATED P2P Interface which doesn't
		 * have an NETDEVICE associated with it. So we have to re-route the
		 * command to the parent NETDEVICE
		 */
		wpa_printf(MSG_INFO, "%s: DRIVER cmd received on P2P Dedicated"
				"IFACE. Routing the command to parent iface", __func__);
		wpa_s = (struct wpa_supplicant *)(drv->ctx);
		if (wpa_s && wpa_s->parent) {
			/* Update the nl80211 pointers corresponding to parent iface */
			bss = priv = wpa_s->parent->drv_priv;
			if(bss == NULL) {
				wpa_printf(MSG_ERROR, "%s: bss is NULL", __func__);
				return -1;
			}
			drv = bss->drv;
			wpa_printf(MSG_DEBUG, "Re-routing command to iface: %s "
				"cmd (%s)", bss->ifname, cmd);
		}
	}

	if (os_strcasecmp(cmd, "STOP") == 0) {
		linux_set_iface_flags(drv->global->ioctl_sock, bss->ifname, 0);
		wpa_msg(drv->ctx, MSG_INFO, WPA_EVENT_DRIVER_STATE "STOPPED");
#ifdef CONFIG_BRCM_DEBUG
	} else if (os_strncasecmp(cmd, "VENDOR ", 7) == 0) {
		wpa_printf(MSG_INFO, "%s: Vendor cmd(0x%x) received on wlan %s %s",
				__func__, NL80211_CMD_VENDOR, buf, cmd);
		cmd_ptr = cmd + strlen("VENDOR ");
		return nl80211_send_vendor_command(drv,bss, BRCM_VENDOR_SUBCMD_PRIV_STR,
				cmd_ptr, strlen(cmd_ptr), OUI_BRCM);
#endif /* CONFIG_BRCM_DEBUG */
	} else if (os_strcasecmp(cmd, "START") == 0) {
		linux_set_iface_flags(drv->global->ioctl_sock, bss->ifname, 1);
		wpa_msg(drv->ctx, MSG_INFO, WPA_EVENT_DRIVER_STATE "STARTED");
	} else if (os_strcasecmp(cmd, "MACADDR") == 0) {
		u8 macaddr[ETH_ALEN] = {};

		ret = linux_get_ifhwaddr(drv->global->ioctl_sock, bss->ifname, macaddr);
		if (!ret) {
			ret = os_snprintf(buf, buf_len,
					"Macaddr = " MACSTR "\n", MAC2STR(macaddr));
			drv_errors = 0;
		} else {
			wpa_driver_send_hang_msg(drv);
		}
		return ret;
	} else if (os_strcasecmp(cmd, "RELOAD") == 0) {
		wpa_msg(drv->ctx, MSG_INFO, WPA_EVENT_DRIVER_STATE "HANGED");
		return 0;
	} else if (os_strncasecmp(cmd, "POWERMODE ", 10) == 0) {
		int state;

		state = atoi(cmd + 10);
		ret = wpa_driver_set_power_save(priv, state);
		if (ret < 0) {
			wpa_driver_send_hang_msg(drv);
		} else {
			drv_errors = 0;
			ret = 0;
		}
		return ret;
#ifdef CONFIG_BRCM_NAN
	} else if (os_strncasecmp(cmd, "NAN_ADD_CONF", 12) == 0) {
	wpa_s = (struct wpa_supplicant *)(drv->ctx);
	ret = nan_ctrl_create_p2p_ie(wpa_s, cmd, strlen(cmd));
#endif	/* CONFIG_BRCM_NAN */
	} else if (os_strncasecmp(cmd, "GETPOWER", 8) == 0) {
		int state = -1;

		ret = wpa_driver_get_power_save(priv, &state);
		if (!ret && (state != -1)) {
			ret = os_snprintf(buf, buf_len, "POWERMODE = %d\n", state);
			drv_errors = 0;
		} else {
			wpa_driver_send_hang_msg(drv);
		}
		return ret;
	}

		if (os_strcasecmp(cmd, "BGSCAN-START") == 0) {
			ret = wpa_driver_set_backgroundscan_params(priv);
			if (ret < 0) {
				return ret;
			}
			os_memcpy(buf, "PNOFORCE 1", 11);
		} else if (os_strcasecmp(cmd, "BGSCAN-STOP") == 0) {
			os_memcpy(buf, "PNOFORCE 0", 11);
		}
		else {
			os_memcpy(buf, cmd, strlen(cmd) + 1);
		}
		memset(&ifr, 0, sizeof(ifr));
		memset(&priv_cmd, 0, sizeof(priv_cmd));
		os_memcpy(ifr.ifr_name, bss->ifname, IFNAMSIZ);

		priv_cmd.buf = buf;
		priv_cmd.used_len = buf_len;
		priv_cmd.total_len = buf_len;
		ifr.ifr_data = (void *)&priv_cmd;

		if ((ret = ioctl(drv->global->ioctl_sock, SIOCDEVPRIVATE + 1, &ifr)) < 0) {
			wpa_printf(MSG_ERROR, "%s: failed to issue private commands ret = %d\n", __func__,ret);
			wpa_driver_send_hang_msg(drv);
		} else {
			drv_errors = 0;
			ret = strlen(buf);
		}
		if (os_strncasecmp(cmd, "COUNTRY", 7) == 0) {
			wpa_supplicant_event(drv->ctx, EVENT_CHANNEL_LIST_CHANGED,
			     NULL);
		} else if (os_strcasecmp(cmd, "GET_BEST_CHANNELS") == 0) {
			wpa_get_best_channels(drv->ctx, buf);
		} else if (os_strncasecmp(cmd, "SETBAND", 7) == 0) {
			/* private command band needs to update Channel list */
			wpa_supplicant_event(drv->ctx, EVENT_CHANNEL_LIST_CHANGED,
			     NULL);
		}
		if (os_strncasecmp(cmd, "WLS_BATCHING GET", 16) == 0) {
			token = strtok(buf, "\n");
			while (token != NULL) {
				wpa_printf(MSG_DEBUG, "%s", token);
				token = strtok(NULL, "\n");
			}
		} else
			wpa_printf(MSG_DEBUG, "%s %s len = %d, %d", __func__, buf, ret, strlen(buf));

	return ret;
}

int wpa_driver_set_p2p_noa(void *priv, u8 count, int start, int duration)
{
	char buf[MAX_DRV_CMD_SIZE];

	memset(buf, 0, sizeof(buf));
	wpa_printf(MSG_DEBUG, "%s: Entry", __func__);
	snprintf(buf, sizeof(buf), "P2P_SET_NOA %d %d %d",
		count, start, duration);
	return wpa_driver_nl80211_driver_cmd(priv, buf, buf, strlen(buf)+1);
}

int wpa_driver_get_p2p_noa(void *priv, u8 *buf, size_t len)
{
	/* Return 0 till we handle p2p_presence request completely in the driver */
	return 0;
}

int wpa_driver_set_p2p_ps(void *priv, int legacy_ps, int opp_ps, int ctwindow)
{
	char buf[MAX_DRV_CMD_SIZE];

	memset(buf, 0, sizeof(buf));
	wpa_printf(MSG_DEBUG, "%s: Entry", __func__);
	snprintf(buf, sizeof(buf), "P2P_SET_PS %d %d %d", legacy_ps, opp_ps, ctwindow);
	return wpa_driver_nl80211_driver_cmd(priv, buf, buf, strlen(buf) + 1);
}

int wpa_driver_set_ap_wps_p2p_ie(void *priv, const struct wpabuf *beacon,
				 const struct wpabuf *proberesp,
				 const struct wpabuf *assocresp)
{
	char buf[MAX_WPSP2PIE_CMD_SIZE];
	struct wpabuf *ap_wps_p2p_ie = NULL;
	char *_cmd = "SET_AP_WPS_P2P_IE";
	char *pbuf;
	int ret = 0;
	int i;
	struct cmd_desc {
		int cmd;
		const struct wpabuf *src;
	} cmd_arr[] = {
		{0x1, beacon},
		{0x2, proberesp},
		{0x4, assocresp},
		{-1, NULL}
	};

	wpa_printf(MSG_DEBUG, "%s: Entry", __func__);

	for (i = 0; cmd_arr[i].cmd != -1; i++) {
		os_memset(buf, 0, sizeof(buf));
		pbuf = buf;
		if (cmd_arr[i].src == NULL) {
			continue;
		}
		if ((strlen(_cmd) + 3 + wpabuf_len(cmd_arr[i].src)) > MAX_WPSP2PIE_CMD_SIZE ) {
			wpa_printf(MSG_ERROR, "%s: P2P_IE too long", __func__);
			return -1;
		}
		pbuf += sprintf(pbuf, "%s %d", _cmd, cmd_arr[i].cmd);
		*pbuf++ = '\0';
		ap_wps_p2p_ie = cmd_arr[i].src ?
			wpabuf_dup(cmd_arr[i].src) : NULL;
		if (ap_wps_p2p_ie) {
			os_memcpy(pbuf, wpabuf_head(ap_wps_p2p_ie), wpabuf_len(ap_wps_p2p_ie));
			ret = wpa_driver_nl80211_driver_cmd(priv, buf, buf,
				strlen(_cmd) + 3 + wpabuf_len(ap_wps_p2p_ie));
			wpabuf_free(ap_wps_p2p_ie);
			if (ret < 0)
				break;
		}
	}

	return ret;
}

/*
 * This will get call per interface. So any interface
 * specific functionality can be handled here. Any generic
 * or single time intialization has to be appropriately handled
 * so that it doesn't happen mulitple times.
 */
int wpa_driver_priv_lib_init(struct nl80211_global *global)
{
	int ret;

	wpa_printf(MSG_INFO, "libbcmdhd: priv_lib_init");


    return 0;
}

int wpa_driver_priv_lib_deinit(struct nl80211_global *global)
{
	wpa_printf(MSG_INFO, "libbcmdhd: priv_lib_deinit");


    return 0;
}
