
#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <binder/Parcel.h>
#include <log/log.h>
#include <cutils/jstring.h>
#include <pthread.h>
#include "liblog/lynq_deflog.h"
#include <sys/time.h>
#include <string.h>
#include "lynq_call.h"
#include "lynq_module_common.h"
#include "lynq_call_common.h"
#include "lynq_module_socket.h"

#define USER_LOG_TAG "LYNQ_CALL"

#define MAX_IP_LENGTH 128
#define MAX_VLAN_INFO_LENGTH 128


static char * lynqStrdupReadString(Parcel* &p) {
    size_t stringlen;
    const char16_t *s16;

    s16 = p->readString16Inplace(&stringlen);
    return strndup16to8(s16, stringlen);
}

/*set*/
int lynq_set_voice_audio_mode(const LYNQ_Audio_Mode audio_mode)
{
    if(g_module_init_flag != MODULE_RUNNING)
    {
        LYERRLOG("%s module state %d error",__func__,g_module_init_flag);
        return LYNQ_E_CONFLICT;
    }
    if(audio_mode > AUDIO_MODE_RTP || audio_mode < AUDIO_MODE_CODEC)
    {
        return LYNQ_E_PARAMETER_ANONALY;
    }

    Parcel* p=NULL;
    int ret=lynq_send_common_request(p,g_wait_time,LYNQ_REQUEST_SET_VOICE_AUDIO_MODE,1,"%d",audio_mode);
    if(ret==RESULT_OK)
    {
        delete p;        
    }
    return ret;
}

int lynq_set_remote_rtp_ip(const char* ip, const int ip_length)
{
    if(g_module_init_flag != MODULE_RUNNING)
    {
        LYERRLOG("%s module state %d error",__func__,g_module_init_flag);
        return LYNQ_E_CONFLICT;
    }
    if (NULL == ip)
    {
        LYERRLOG("ip is NULL!!!");
        return LYNQ_E_PARAMETER_ANONALY;
    }
    if ((ip_length < strlen(ip)+1) || (ip_length > MAX_IP_LENGTH))
    {
        LYERRLOG("incoming ip length error %d", ip_length);
        return LYNQ_E_PARAMETER_ANONALY;
    }    

    Parcel* p=NULL;
    int ret=lynq_send_common_request(p,g_wait_time,LYNQ_REQUEST_SET_REMOTE_RTP_IP,1,"%s",ip);
    if(ret==RESULT_OK)
    {
        delete p;        
    }
    return ret;
}
int lynq_set_vlan_info(const char* vlan_info, const int vlan_info_length)
{
    if(g_module_init_flag != MODULE_RUNNING)
    {
        LYERRLOG("%s module state %d error",__func__,g_module_init_flag);
        return LYNQ_E_CONFLICT;
    }

    if (NULL == vlan_info)
    {
        LYERRLOG("vlan_info is NULL!!!");
        return LYNQ_E_PARAMETER_ANONALY;
    }
    
    if ((vlan_info_length < strlen(vlan_info)+1) || (vlan_info_length > MAX_VLAN_INFO_LENGTH))
    {
        LYERRLOG("incoming vlan_info error, vlan info length %d", vlan_info_length);
        return LYNQ_E_PARAMETER_ANONALY;
    }    

    Parcel* p=NULL;
    int ret=lynq_send_common_request(p,g_wait_time,LYNQ_REQUEST_SET_VLAN_INFO,1,"%s",vlan_info);
    if(ret==RESULT_OK)
    {
        delete p;        
    }
    return ret;
}
int lynq_set_rtp_port(const LYNQ_Rtp_Mode rtp_mode, const int port)
{  
    if(g_module_init_flag != MODULE_RUNNING)
    {
        LYERRLOG("%s module state %d error",__func__,g_module_init_flag);
        return LYNQ_E_CONFLICT;
    }

    if (port < 0)
    {
        LYERRLOG("invalid port number %d", port);
        return LYNQ_E_PARAMETER_ANONALY;
    }

    Parcel* p=NULL;
    int ret=lynq_send_common_request(p,g_wait_time,LYNQ_REQUEST_SET_RTP_PORT,2,"%d %d",rtp_mode,port);
    if(ret==RESULT_OK)
    {
        delete p;        
    }
    return ret;
}
int lynq_set_rtp_param(const int clock_rate,const int channels,const int latency) //only for client mode
{
    if(g_module_init_flag != MODULE_RUNNING)
    {
        LYERRLOG("%s module state %d error",__func__,g_module_init_flag);
        return LYNQ_E_CONFLICT;
    }

    Parcel* p=NULL;
    int ret=lynq_send_common_request(p,g_wait_time,LYNQ_REQUEST_SET_RTP_PARAM,3,"%d %d %d",clock_rate,channels,latency);
    if(ret==RESULT_OK)
    {
        delete p;         
    }
    return ret;
}

int lynq_set_call_rtp_ssrc(const LYNQ_Rtp_Ssrc_Mode ssrc_mode,const unsigned int ssrc)
{
    if(g_module_init_flag != MODULE_RUNNING)
    {
        LYERRLOG("%s module state %d error",__func__,g_module_init_flag);
        return LYNQ_E_CONFLICT;
    }

    if(ssrc_mode < Rtp_Ssrc_random || ssrc_mode > Rtp_Ssrc_specific)
    {
        return LYNQ_E_PARAMETER_ANONALY;
    }

    Parcel* p=NULL;
    int ret=lynq_send_common_request(p,g_wait_time,LYNQ_REQUEST_SET_CALL_RTP_SSRC,2,"%d %u",ssrc_mode,ssrc);
    if(ret==RESULT_OK)
    {
        delete p;         
    }
    return ret;      
}
/*get*/
int lynq_get_voice_audio_mode(LYNQ_Audio_Mode* voice_audio_mode)
{
    if(g_module_init_flag != MODULE_RUNNING)
    {
        LYERRLOG("%s module state %d error",__func__,g_module_init_flag);
        return LYNQ_E_CONFLICT;
    }

    if(voice_audio_mode==NULL)
    {
        LYERRLOG("parameter is NULL");
        return LYNQ_E_PARAMETER_ANONALY;
    }    

    int local_mode;

    int ret=lynq_get_common_request(LYNQ_REQUEST_GET_VOICE_AUDIO_MODE,&local_mode);

    *voice_audio_mode = (LYNQ_Audio_Mode) local_mode;
    return ret;
}
int lynq_get_remote_rtp_ip(char* ip, const int ip_length)
{
    if(g_module_init_flag != MODULE_RUNNING)
    {
        LYERRLOG("%s module state %d error",__func__,g_module_init_flag);
        return LYNQ_E_CONFLICT;
    }

    if(ip==NULL)
    {
        LYERRLOG("ip is NULL");
        return LYNQ_E_PARAMETER_ANONALY;
    }    

    if(ip_length > MAX_IP_LENGTH)
    {
        LYERRLOG("ip length %d is greater than max ip length %d",ip_length,MAX_IP_LENGTH);
        return LYNQ_E_PARAMETER_ANONALY;
    }    
   

    Parcel* p=NULL;
    int ret=lynq_send_common_request(p,g_wait_time,LYNQ_REQUEST_GET_REMOTE_RTP_IP,0,"");

    if(ret!=RESULT_OK)
    {
        LYERRLOG("%s call lynq_send_common_request failure, ret is %d",__func__,ret);
        return ret;  
    }       
 
    int num ;
    char *resp;

    p->readInt32(&num);
    if(num != 1)
    {
        LYERRLOG("no paramters or num %d too great",num);        
        delete p;      
        return LYNQ_E_INNER_ERROR;    
    }else{
        resp = lynqStrdupReadString(p);
        strcpy(ip,resp);
        free(resp);                             
    }

    LYINFLOG("%s suc",__func__);   
    delete p;       
    return RESULT_OK;    
}
int lynq_get_vlan_info(char* vlan_info, const int vlan_info_length)
{
    if(g_module_init_flag != MODULE_RUNNING)
    {
        LYERRLOG("%s module state %d error",__func__,g_module_init_flag);
        return LYNQ_E_CONFLICT;
    }

    if(vlan_info==NULL) 
    {
        LYERRLOG("vlan info is NULL");
        return LYNQ_E_PARAMETER_ANONALY;
    }

    Parcel* p=NULL;
    int ret=lynq_send_common_request(p,g_wait_time,LYNQ_REQUEST_GET_VLAN_INFO,0,"");

    if(ret!=RESULT_OK)
    {
        LYERRLOG("%s call lynq_send_common_request failure, ret is %d",__func__,ret);
        return ret;  
    }       
    
    int num ;
    char *resp;

    p->readInt32(&num);
    if(num != 1)
    {
        LYERRLOG("no paramters or num %d too great",num);        
        delete p;      
        return LYNQ_E_INNER_ERROR;    
    }else{
        resp = lynqStrdupReadString(p);
        strcpy(vlan_info,resp);
        free(resp);                             
    }

    LYINFLOG("%s suc",__func__);   
    delete p;       
    return RESULT_OK;    
}

int lynq_get_rtp_port(const LYNQ_Rtp_Mode rtp_mode, int* port)
{
    if(g_module_init_flag != MODULE_RUNNING)
    {
        LYERRLOG("%s module state %d error",__func__,g_module_init_flag);
        return LYNQ_E_CONFLICT;
    }

    if(port==NULL)
    {
        return LYNQ_E_PARAMETER_ANONALY;
    }   

    Parcel* p=NULL;
    int ret=lynq_send_common_request(p,g_wait_time,LYNQ_REQUEST_GET_RTP_PORT,1,"%d",rtp_mode);
    if(ret==RESULT_OK)
    {
        p->readInt32(port);
        delete p;        
    }
    return ret;       
}
int lynq_get_rtp_param(int* clock_rate, int* channels, int* latency)//only for client mode
{
    if(g_module_init_flag != MODULE_RUNNING)
    {
        LYERRLOG("%s module state %d error",__func__,g_module_init_flag);
        return LYNQ_E_CONFLICT;
    }

    if(clock_rate == NULL || channels ==NULL || latency ==NULL)
    {
        LYERRLOG("input parameter is NULL");
        return LYNQ_E_PARAMETER_ANONALY;
    }

    Parcel* p=NULL;
    int ret=lynq_send_common_request(p,g_wait_time,LYNQ_REQUEST_GET_RTP_PARAM,0,"");

    if(ret!=RESULT_OK)
    {
        LYERRLOG("%s call lynq_send_common_request failure, ret is %d",__func__,ret);
        return ret;  
    }           

    p->readInt32(clock_rate);
    p->readInt32(channels);
    p->readInt32(latency);    

    LYINFLOG("%s suc",__func__);   
    delete p;       
    return RESULT_OK;       
}

int lynq_get_call_rtp_ssrc(int* ssrc_mode, unsigned int* ssrc)
{
    if(g_module_init_flag != MODULE_RUNNING)
    {
        LYERRLOG("%s module state %d error",__func__,g_module_init_flag);
        return LYNQ_E_CONFLICT;
    }

    if(ssrc_mode == NULL || ssrc ==NULL)
    {
        LYERRLOG("input parameter is NULL");
        return LYNQ_E_PARAMETER_ANONALY;
    }

    Parcel* p=NULL;
    int ret=lynq_send_common_request(p,g_wait_time,LYNQ_REQUEST_GET_CALL_RTP_SSRC,0,"");

    if(ret!=RESULT_OK)
    {
        LYERRLOG("%s call lynq_send_common_request failure, ret is %d",__func__,ret);
        return ret;  
    }           

    p->readInt32(ssrc_mode);
    p->readUint32(ssrc); 

    LYINFLOG("%s suc",__func__);   
    delete p;       
    return RESULT_OK;                
}
