#include "thread_pool.h"
#include <time.h>
#include "common.h"

EThread_pool_status get_threadpool_status(thread_pool_t *pool);
void set_threadpool_status(thread_pool_t *pool, EThread_pool_status state);
thread_info_t *create_thread_info(EThread_pool_status *pool_status);
thread_handle pool_thread_create(THREAD_FUNC threadFun, int pri, void *para);
BOOL init_threadpool(thread_pool_t *pool);
thread_handle get_self_thread_id(void);

#define THREAD_PRI_NOMAL_LEVEL 60
#define THREAD_PRI_ABOVE_LEVEL 70
#define THREAD_PRI_HIGH_LEVEL 80


#define TASK_QUEUE_COUNT_MAX 100
#define CONDITION_WAIT_TIME 1000
#define INCRE_THREADS_UNIT  5



thread_handle get_self_thread_id() {
	return pthread_self();
}

void tp_sleep(unsigned int ms) {
	struct timeval  delay;
	
	delay.tv_sec = ms/1000;
	delay.tv_usec = (ms%1000) * 1000;
	select(0, NULL, NULL, NULL, &delay);
}


unsigned long get_current_time(void) {
	return time(NULL);
}

void force_exit_thread(thread_info_t *thread_info) {

	int ret;
	pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, NULL);
	pthread_cancel(thread_info->h_thread);
	pthread_setcancelstate(PTHREAD_CANCEL_DISABLE, NULL);

	ret = destroy_condition_handle(&thread_info->thread_cond);
	if (ret) {
		post_condition_signal(&thread_info->thread_cond);
		destroy_condition_handle(&thread_info->thread_cond);
	}
	
	ret = mutex_destroy(&thread_info->thread_lock);
	if (ret) {
		mutex_unlock(&thread_info->thread_lock);
		mutex_destroy(&thread_info->thread_lock);
	}
}


int compare_thread_id(void *usr_data, unsigned int sdata_len, void *list_data, unsigned int ldata_len) {
	if (((thread_info_t *)usr_data)->h_thread == ((thread_info_t *)list_data)->h_thread) {
		return 0;
	} else {
		return -1;
	}
}


static void pool_thread_work(void *para)
{
	int ret = 0;

	thread_info_t *thread_info = (thread_info_t *)para;
	thread_func_t *thread_func;

	if (thread_info == NULL) {
		LYVERBLOG("+[thhandle]: error num = %d\n", ERR_INVOKE);
		return;
	}

	while (*thread_info->pool_status != EThread_pool_exit) {
		mutex_lock(&thread_info->thread_lock);
		while (!thread_info->busy) {
			ret = wait_condition(&thread_info->thread_cond, &thread_info->thread_lock, TIME_WAIT_INFINITE);
		}

		mutex_unlock(&thread_info->thread_lock);

		if (*thread_info->pool_status == EThread_pool_exit || thread_info->release)
			break;

		thread_func = &thread_info->thread_para;
		thread_info->time_out = thread_func->time_out_info.time_out;
		thread_info->launch_time = get_current_time();
		thread_info->thread_para.process_func(thread_info->thread_para.args);
		if (thread_info->thread_para.release_func) {
			thread_info->thread_para.release_func(thread_info->thread_para.args);
		}
		thread_info->launch_time = TIME_WAIT_INFINITE;

		mutex_lock(&thread_info->thread_lock);
		thread_info->busy = FALSE;
		mutex_unlock(&thread_info->thread_lock);
	}

	LYDBGLOG(" thread %lu exit.\n", get_self_thread_id());
	
	pthread_exit(0);
}


BOOL scan_idle_thread_from_busy(thread_pool_t *pool) {

	BOOL ret = FALSE;
	
	d_list_t *idle_list = pool->idle_threads;
	d_list_t *busy_list = pool->busy_threads;
	
	d_list_node_t *temp_node;
	thread_info_t *info;
	if (busy_list == NULL || idle_list == NULL) 
		return FALSE;

	while ((temp_node = get_next_node(busy_list)) != NULL)
	{
		if (temp_node == NULL || temp_node->data == NULL)
			break;
		
		info = (thread_info_t *)temp_node->data;
		if (!info->busy) {

			if (remove_d_list_node(busy_list, temp_node)) {
				insert_d_list_tail_node(idle_list, temp_node);
				ret = TRUE;
				//break;
			}
		} else {
			if (info->time_out == 0 || info->launch_time == TIME_WAIT_INFINITE) {
				continue;
			}

			if (get_current_time() - info->time_out > info->launch_time) {
			//if (get_current_time() - info->launch_time > info->time_out) {
				LYDBGLOG("[%s-%d] ####### time out , force exit thread.  %lu #########\n", __FUNCTION__, __LINE__, info->h_thread);
				LYVERBLOG("+[thhandle]: error num = %d\n", ERR_TIMEOUT);
				if (remove_d_list_node(busy_list, temp_node)) {
					force_exit_thread(info);

					if (info->thread_para.time_out_info.timeout_callback) {
						info->thread_para.time_out_info.timeout_callback(&info->thread_para);
					}

					init_condition_handle(&info->thread_cond);
					init_mutex_handle(&info->thread_lock);
					
					info->busy = FALSE;
					info->release = FALSE;
					info->launch_time = TIME_WAIT_INFINITE;
					info->h_thread = pool_thread_create(pool_thread_work, info->pri, info);
					if (info->h_thread < 0) {
						
						free(info);
						free(temp_node);
						pool->pool_thread_num--;
					} else {
						
						insert_d_list_tail_node(idle_list, temp_node);
						ret = TRUE;
					}
				}				
			}
		
		}
	}
	
	return ret;
}


static void pool_thread_manage(void *para)
{
	thread_func_t thread_para;	
	d_list_node_t *list_node;
	thread_info_t *thread_info;
	thread_pool_t *pool = (thread_pool_t *)para;
	
	if (pool == NULL) {
		return;
	}
	
	while (get_threadpool_status(pool) != EThread_pool_exit) {

		if (get_count_seq_queue(pool->task_queue) > 0) {
	
			pool->release_threads_interval = 0;

			list_node = remove_d_list_head_node(pool->idle_threads);
			if (list_node && list_node->data) {

				memset(&thread_para, 0, sizeof(thread_func_t));
				if (de_seq_queue(pool->task_queue, &thread_para)) {

					thread_info = (thread_info_t *)list_node->data;
					post_sem(&pool->sem_inc);

					mutex_lock(&thread_info->thread_lock);
					memcpy(&thread_info->thread_para, &thread_para, sizeof(thread_func_t));
					thread_info->busy = TRUE;
					post_condition_signal(&thread_info->thread_cond);
					mutex_unlock(&thread_info->thread_lock);
					
					insert_d_list_tail_node(pool->busy_threads, list_node);

				} else {

					insert_d_list_head_node(pool->idle_threads, list_node);
				}
				

			} else {
				if (scan_idle_thread_from_busy(pool))
					continue;

				if (pool->pool_thread_num < pool->max_thread_num) {
					
					int i;
					for (i=0; i<INCRE_THREADS_UNIT; i++) {
						thread_info_t *thread_inf = create_thread_info(&pool->status);
						insert_d_list_tail(pool->idle_threads, thread_inf, sizeof(thread_info_t));
						pool->pool_thread_num++;
					}
					
				} else {
					//continue;
				}
			}

		} else {
			if (get_d_list_node_count(pool->busy_threads) > 0) {
				pool->release_threads_interval = 0;
				scan_idle_thread_from_busy(pool);
			} else {
				BOOL is_release_threads = FALSE;
				if (pool->release_threads_interval == 0) {
					pool->release_threads_interval = get_current_time();
				} else {
					is_release_threads = (get_current_time() - pool->release_threads_interval) > RELEASE_THREAD_INTERVAL;
				}

				if (is_release_threads && get_d_list_node_count(pool->idle_threads) > pool->min_thread_num) {
					//
					list_node = remove_d_list_head_node(pool->idle_threads);
					if (list_node && list_node->data) {							
						thread_info = (thread_info_t *)list_node->data;
						if (!thread_info->busy) {
							mutex_lock(&thread_info->thread_lock);
							thread_info->release = TRUE;
							mutex_unlock(&thread_info->thread_lock);
							post_condition_signal(&thread_info->thread_cond);
							pool->pool_thread_num--;
							free(thread_info);
							free(list_node);
						} else {
							insert_d_list_tail(pool->busy_threads, thread_info, sizeof(thread_info_t));
						}
					}
					
				} else {
					mutex_lock(&pool->mange_lock);
					wait_condition(&pool->manage_cond, &pool->mange_lock, CONDITION_WAIT_TIME);
					mutex_unlock(&pool->mange_lock);
				}
			}

		}
	}
}


thread_handle pool_thread_create(THREAD_FUNC threadFun, int pri, void *para) {
	
		thread_handle h_thread;
		pthread_attr_t attr;
		struct sched_param p;

		pthread_attr_init(&attr);
		p.sched_priority = pri;
		
		pthread_attr_setschedpolicy(&attr,SCHED_OTHER);
		pthread_attr_setschedparam(&attr,&p);

		pthread_create(&h_thread, &attr, (void *)threadFun, para);
		return h_thread;
}

void pool_thread_release(thread_handle h_thread)
{
	pthread_join(h_thread, NULL);
}

EThread_pool_status get_threadpool_status(thread_pool_t *pool) {

	if (pool) {
		return pool->status;
	} else {
		return EThread_pool_unknown;
	}
}

void set_threadpool_status(thread_pool_t *pool, EThread_pool_status state) {
	
	if (pool) {
		pool->status = state;
	}
}

thread_pool_t *threadpool_create(unsigned int min_thread_num, unsigned int max_thread_num)
{
	thread_pool_t *pool = (thread_pool_t *)malloc(sizeof(thread_pool_t));

	if (pool) {
		memset(pool, 0, sizeof(thread_pool_t));
		pool->min_thread_num = min_thread_num;
		pool->max_thread_num = max_thread_num;
		pool->pri = THREAD_PRI_HIGH_LEVEL;
		pool->status = EThread_pool_alloc;

		if (!init_threadpool(pool)) {
			free(pool);
			pool = NULL;
		}
	}

	return pool;
}

thread_info_t *create_thread_info(EThread_pool_status *pool_status) {
	thread_info_t *thread_inf = (thread_info_t *)malloc(sizeof(thread_info_t));
	if (thread_inf == NULL)
		return NULL;
	
	memset(thread_inf, 0, sizeof(thread_info_t));
	
	thread_inf->busy = FALSE;
	thread_inf->release = FALSE;
	thread_inf->pri = THREAD_PRI_ABOVE_LEVEL;
	thread_inf->launch_time = TIME_WAIT_INFINITE;
	thread_inf->pool_status = pool_status;

	init_condition_handle(&thread_inf->thread_cond);
	init_mutex_handle(&thread_inf->thread_lock);
	
	thread_inf->h_thread = pool_thread_create(pool_thread_work, thread_inf->pri, thread_inf);
	if (thread_inf->h_thread < 0) {
		destroy_condition_handle(&thread_inf->thread_cond);
		mutex_destroy(&thread_inf->thread_lock);

		free(thread_inf);
		thread_inf = NULL;
	}
	
	return thread_inf;
}


void destroy_thread_info(thread_info_t *thread_info) {
	if (thread_info == NULL)
		return;

	thread_info->release = TRUE;
	pool_thread_release(thread_info->h_thread);

	destroy_condition_handle(&thread_info->thread_cond);
	mutex_destroy(&thread_info->thread_lock);
}

BOOL init_threadpool(thread_pool_t *pool) {

	unsigned int thread_num = 0;
	if (pool == NULL)
		return FALSE;

	pool->busy_threads = create_d_list();
	pool->idle_threads = create_d_list();
	pool->task_queue = create_seq_queue(sizeof(thread_func_t), TASK_QUEUE_COUNT_MAX, TRUE);
	if (pool->busy_threads == NULL || pool->idle_threads == NULL || pool->task_queue == NULL)
		goto err_flag;

	for (thread_num=0; thread_num<pool->min_thread_num; thread_num++) {
		thread_info_t *thread_inf = create_thread_info(&pool->status);

		if (!insert_d_list_tail(pool->idle_threads, thread_inf, sizeof(thread_info_t)))
			goto err_flag;
		
	}

	create_sem(&pool->sem_inc, TASK_QUEUE_COUNT_MAX);
	init_condition_handle(&pool->manage_cond);
	init_mutex_handle(&pool->mange_lock);
	pool->h_id = pool_thread_create(pool_thread_manage, pool->pri, pool);

	pool->status = EThread_pool_init;
	pool->pool_thread_num = pool->min_thread_num;

	return TRUE;


err_flag:
	LYDBGLOG("[%s-%d]##################### err ################\n", __FUNCTION__, __LINE__);
	LYVERBLOG("+[thhandle]: error num = %d\n", ERR_INVOKE);
	threadpool_destroy(pool);

	return FALSE;
}

void threadpool_destroy(thread_pool_t *pool)
{
	d_list_node_t *temp_node;
	if (pool == NULL)
		return;

	set_threadpool_status(pool, EThread_pool_exit);

	destroy_sem(&pool->sem_inc);
	mutex_lock(&pool->mange_lock);
	post_condition_signal(&pool->manage_cond);
	mutex_unlock(&pool->mange_lock);
	pool_thread_release(pool->h_id);

	mutex_destroy(&pool->mange_lock);
	destroy_condition_handle(&pool->manage_cond);

	while ((temp_node = remove_d_list_head_node(pool->idle_threads))) {
		thread_info_t *info = (thread_info_t *)temp_node->data;
		post_condition_signal(&info->thread_cond);
		destroy_thread_info(info);
		free(info);
		free(temp_node);
	}

	while ((temp_node = remove_d_list_head_node(pool->busy_threads))) {
		thread_info_t *info = (thread_info_t *)temp_node->data;
		post_condition_signal(&info->thread_cond);
		destroy_thread_info(info);
		free(info);
		free(temp_node);
	}
	
	destroy_d_list(pool->busy_threads);
	destroy_d_list(pool->idle_threads);
	destroy_seq_queue(pool->task_queue);

	free(pool);
}

BOOL threadpool_add(thread_pool_t *pool, USER_FUNC process_func, void *args)
{
	BOOL ret = FALSE;
	thread_func_t function;
	
	if (pool == NULL)
		return FALSE;
	
	memset(&function, 0, sizeof(thread_func_t));
	function.process_func = process_func;
	function.args = args;

	wait_sem(&pool->sem_inc);
	ret = en_seq_queue(pool->task_queue, &function);

	mutex_lock(&pool->mange_lock);
	post_condition_signal(&pool->manage_cond);
	mutex_unlock(&pool->mange_lock);
	
	return ret;
}

BOOL threadpool_add_timeout(thread_pool_t *pool, USER_FUNC process_func
		, USER_FUNC release_func, void *args, time_out_t *time_out)
{
	BOOL ret = FALSE;
	thread_func_t function;
	
	if (pool == NULL)
		return FALSE;
	
	memset(&function, 0, sizeof(thread_func_t));
	function.process_func = process_func;
	function.release_func = release_func;
	function.args = args;
	memcpy(&function.time_out_info, time_out, sizeof(time_out_t));
	
	wait_sem(&pool->sem_inc);
	ret = en_seq_queue(pool->task_queue, &function);

	mutex_lock(&pool->mange_lock);
	post_condition_signal(&pool->manage_cond);
	mutex_unlock(&pool->mange_lock);
	
	return ret;
}


