// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (c) 2019 MediaTek Inc.
 */
#include "stmmac.h"
#include "stmmac_platform.h"

#ifdef CONFIG_DEBUG_FS

static int eth_smt_result = 2;

#define MTK_ETH_FRAME_LEN (ETH_FRAME_LEN + ETH_FCS_LEN + VLAN_HLEN)

static int FRAME_PATTERN_CH[8] = {
	0x11111111,
	0x22222222,
	0x33333333,
	0x44444444,
	0x55555555,
	0x66666666,
	0x77777777,
	0x88888888,
};

static int frame_hdrs[8][4] = {
	/* for channel 0 : Non tagged header
	 * Dst addr : 0x00:0x0D:0x56:0x73:0xD0:0xF3
	 * Src addr : 0x00:0x55:0x7B:0xB5:0x7D:0xF7
	 * Type/Length : 0x800
	 */
	{0xFFFFFFFF, 0x5500FFFF, 0xF77DB57B, 0x00000081},

	/* for channel 1 : VLAN tagged header with priority 1
	 * Dst addr : 0x00:0x0D:0x56:0x73:0xD0:0xF3
	 * Src addr : 0x00:0x55:0x7B:0xB5:0x7D:0xF7
	 * Type/Length : 0x8100
	 */
	{0xFFFFFFFF, 0x5500FFFF, 0xF77DB57B, 0x64200081},

	/* for channel 2 : VLAN tagged header with priority 2
	 * Dst addr : 0x00:0x0D:0x56:0x73:0xD0:0xF3
	 * Src addr : 0x00:0x55:0x7B:0xB5:0x7D:0xF7
	 * Type/Length : 0x8100
	 */
	{0xFFFFFFFF, 0x5500FFFF, 0xF77DB57B, 0x64400081},

	/* for channel 3 : VLAN tagged header with priority 3
	 * Dst addr : 0x00:0x0D:0x56:0x73:0xD0:0xF3
	 * Src addr : 0x00:0x55:0x7B:0xB5:0x7D:0xF7
	 * Type/Length : 0x8100
	 */
	{0x73560D00, 0x5500F3D0, 0xF77DB57B, 0x64600081},

	/* for channel 4 : VLAN tagged header with priority 4
	 * Dst addr : 0x00:0x0D:0x56:0x73:0xD0:0xF3
	 * Src addr : 0x00:0x55:0x7B:0xB5:0x7D:0xF7
	 * Type/Length : 0x8100
	 */
	{0x73560D00, 0x5500F3D0, 0xF77DB57B, 0x64800081},

	/* for channel 5 : VLAN tagged header with priority 5
	 * Dst addr : 0x00:0x0D:0x56:0x73:0xD0:0xF3
	 * Src addr : 0x00:0x55:0x7B:0xB5:0x7D:0xF7
	 * Type/Length : 0x8100
	 */
	{0x73560D00, 0x5500F3D0, 0xF77DB57B, 0x64A00081},

	/* for channel 6 : VLAN tagged header with priority 6
	 * Dst addr : 0x00:0x0D:0x56:0x73:0xD0:0xF3
	 * Src addr : 0x00:0x55:0x7B:0xB5:0x7D:0xF7
	 * Type/Length : 0x8100
	 */
	{0x73560D00, 0x5500F3D0, 0xF77DB57B, 0x64C00081},

	/* for channel 7 : VLAN tagged header with priority 7
	 * Dst addr : 0x00:0x0D:0x56:0x73:0xD0:0xF3
	 * Src addr : 0x00:0x55:0x7B:0xB5:0x7D:0xF7
	 * Type/Length : 0x8100
	 */
	{0x73560D00, 0x5500F3D0, 0xF77DB57B, 0x64E00081},
};

static int send_frame(struct stmmac_priv *priv, int num)
{
	struct sk_buff *skb = NULL;
	unsigned int *skb_data = NULL;
	int payload_cnt = 0;
	int i, j, ret, cpu;
	unsigned int frame_size = 1500;
	unsigned int tx_octetcount_gb, tx_framecount_gb;
	unsigned int rx_framecount_gb, rx_octetcount_gb, rx_octetcount_g;
	const struct net_device_ops *netdev_ops = priv->dev->netdev_ops;

	usleep_range(40000, 50000);

	dwmac_mmc_read(priv->mmcaddr, &priv->mmc);
	tx_octetcount_gb = priv->mmc.mmc_tx_octetcount_gb;
	tx_framecount_gb = priv->mmc.mmc_tx_framecount_gb;
	rx_framecount_gb = priv->mmc.mmc_rx_framecount_gb;
	rx_octetcount_gb = priv->mmc.mmc_rx_octetcount_gb;
	rx_octetcount_g = priv->mmc.mmc_rx_octetcount_g;
	usleep_range(1000000, 2000000);
	for (j = 0; j < priv->plat->tx_queues_to_use; j++) {
		for (i = 0; i < num; i++) {
			struct netdev_queue *txq =
					netdev_get_tx_queue(priv->dev, j);

			payload_cnt = 0;
			skb = dev_alloc_skb(MTK_ETH_FRAME_LEN);
			if (!skb) {
				pr_err("Failed to allocate tx skb\n");
				return 1;
			}

			skb_set_queue_mapping(skb, j); /*  map skb to queue0 */
			skb_data = (unsigned int *)skb->data;
			/* Add Ethernet header */
			*skb_data++ = frame_hdrs[j][0];
			*skb_data++ = frame_hdrs[j][1];
			*skb_data++ = frame_hdrs[j][2];
			*skb_data++ = frame_hdrs[j][3];
			/* Add payload */
			for (payload_cnt = 0; payload_cnt < frame_size;) {
				*skb_data++ = FRAME_PATTERN_CH[j];
				/* increment by 4 since we are writing
				 * one dword at a time
				 */
				payload_cnt += 4;
			}
			skb->len = frame_size;
			cpu = smp_processor_id();
			do {
				/* Disable soft irqs for various locks below.
				 * Also stops preemption for RCU.
				 * avoid dql bug-on issue.
				 */
				rcu_read_lock_bh();
				HARD_TX_LOCK(priv->dev, txq, cpu);
				ret = netdev_ops->ndo_start_xmit(skb,
								 priv->dev);
				if (ret == NETDEV_TX_OK)
					txq_trans_update(txq);
				HARD_TX_UNLOCK(priv->dev, txq);
				rcu_read_unlock_bh();
			} while (ret != NETDEV_TX_OK);
		}
	}

	usleep_range(1000000, 2000000);
	dwmac_mmc_read(priv->mmcaddr, &priv->mmc);
	tx_octetcount_gb = priv->mmc.mmc_tx_octetcount_gb - tx_octetcount_gb;
	tx_framecount_gb = priv->mmc.mmc_tx_framecount_gb - tx_framecount_gb;
	rx_framecount_gb = priv->mmc.mmc_rx_framecount_gb - rx_framecount_gb;
	rx_octetcount_gb = priv->mmc.mmc_rx_octetcount_gb - rx_octetcount_gb;
	rx_octetcount_g = priv->mmc.mmc_rx_octetcount_g - rx_octetcount_g;
	usleep_range(40000, 50000);

	if (tx_framecount_gb == rx_framecount_gb &&
	    tx_octetcount_gb == rx_octetcount_gb &&
	    rx_octetcount_gb == rx_octetcount_g &&
	    tx_framecount_gb == (priv->plat->tx_queues_to_use * num)) {
		pr_err("loop back success:\n");
		ret = 0;
	} else {
		pr_err("loop back fail:\n");
		ret = 1;
	}

	pr_err("tx_queues:%d\t pkt_num_per_queue:%d\n",
	       priv->plat->tx_queues_to_use, num);
	pr_err("tx_framecount_gb:%u\t tx_octetcount_gb:%u\n",
	       tx_framecount_gb, tx_octetcount_gb);
	pr_err("rx_framecount_gb:%u\t rx_octetcount_gb:%u, rx_octetcount_g:%u\n",
	       rx_framecount_gb, rx_octetcount_gb, rx_octetcount_g);

	return ret;
}

static ssize_t stmmac_show(struct device *dev,
			   struct device_attribute *attr,
			   char *buf)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct net_device *netdev = platform_get_drvdata(pdev);
	struct stmmac_priv *priv = netdev_priv(netdev);
	int len = 0;
	int buf_len = (int)PAGE_SIZE;
	struct phy_device *phy_dev = of_phy_find_device(priv->plat->phy_node);

	len += snprintf(buf + len, buf_len - len,
			"stmmac debug commands: in hexadecimal notation\n");
	len += snprintf(buf + len, buf_len - len,
			"mac read: echo er reg_start reg_range > stmmac\n");
	len += snprintf(buf + len, buf_len - len,
			"mac write: echo ew reg_addr value > stmmac\n");
	len += snprintf(buf + len, buf_len - len,
			"clause22 read: echo cl22r phy_reg > stmmac\n");
	len += snprintf(buf + len, buf_len - len,
			"clause22 write: echo cl22w phy_reg value > stmmac\n");
	len += snprintf(buf + len, buf_len - len,
			"clause45 read: echo cl45r dev_id reg_addr > stmmac\n");
	len += snprintf(buf + len, buf_len - len,
			"clause45 write: echo cl45w dev_id reg_addr value > stmmac\n");
	len += snprintf(buf + len, buf_len - len,
			"reg read: echo rr reg_addr > stmmac\n");
	len += snprintf(buf + len, buf_len - len,
			"reg write: echo wr reg_addr value > stmmac\n");
	len += snprintf(buf + len, buf_len - len,
			"phy addr:%d\n", phy_dev->mdio.addr);
	len += snprintf(buf + len, buf_len - len,
			"eth smt result:%d\n", eth_smt_result);

	return len;
}

static ssize_t stmmac_store(struct device *dev,
			    struct device_attribute *attr,
			    const char *buf,
			    size_t count)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct net_device *netdev = platform_get_drvdata(pdev);
	struct stmmac_priv *priv = netdev_priv(netdev);
	void __iomem *tmp_addr;
	int reg, data, devid, origin, i, reg_addr;
	struct phy_device *phy_dev = of_phy_find_device(priv->plat->phy_node);

	if (!strncmp(buf, "er", 2) &&
	    (sscanf(buf + 2, "%x %x", &reg, &data) == 2)) {
		for (i = 0; i < data / 0x10 + 1; i++) {
			dev_info(dev,
				 "%08x:\t%08x\t%08x\t%08x\t%08x\t\n",
				 reg + i * 16,
				 readl(priv->ioaddr + reg + i * 0x10),
				 readl(priv->ioaddr + reg + i * 0x10 + 0x4),
				 readl(priv->ioaddr + reg + i * 0x10 + 0x8),
				 readl(priv->ioaddr + reg + i * 0x10 + 0xc));
		}
	} else if (!strncmp(buf, "ew", 2) &&
		   (sscanf(buf + 2, "%x %x", &reg, &data) == 2)) {
		origin = readl(priv->ioaddr + reg);
		writel(data, priv->ioaddr + reg);
		dev_info(dev, "mac reg%#x, value:%#x -> %#x\n",
			 reg, origin, readl(priv->ioaddr + reg));
	} else if (!strncmp(buf, "cl22r", 5) &&
		   (sscanf(buf + 5, "%x", &reg) == 1)) {
		dev_info(dev, "cl22 reg%#x, value:%#x\n",
			 reg, priv->mii->read(priv->mii,
			 phy_dev->mdio.addr, reg));
	} else if (!strncmp(buf, "cl22w", 5) &&
		   (sscanf(buf + 5, "%x %x", &reg, &data) == 2)) {
		origin = priv->mii->read(priv->mii, phy_dev->mdio.addr, reg);
		priv->mii->write(priv->mii, phy_dev->mdio.addr, reg, data);
		dev_info(dev, "cl22 reg%#x, %#x -> %#x\n",
			 reg, origin, priv->mii->read(priv->mii,
			 phy_dev->mdio.addr, reg));
	} else if (!strncmp(buf, "cl45r", 5) &&
		   (sscanf(buf + 5, "%x %x", &devid, &reg) == 2)) {
		reg_addr = MII_ADDR_C45 |
			   ((devid & 0x1f) << 16) |
			   (reg & 0xffff);
		dev_info(dev, "cl45 reg:%#x-%#x, %#x\n", devid, reg,
			 priv->mii->read(priv->mii, phy_dev->mdio.addr,
					 reg_addr));
	} else if (!strncmp(buf, "cl45w", 5) &&
		   (sscanf(buf + 5, "%x %x %x", &devid, &reg, &data) == 3)) {
		reg_addr = MII_ADDR_C45 |
			   ((devid & 0x1f) << 16) |
			   (reg & 0xffff);
		origin = priv->mii->read(priv->mii,
					 phy_dev->mdio.addr,
					 reg_addr);
		priv->mii->write(priv->mii, phy_dev->mdio.addr,
				 reg_addr, data);
		dev_info(dev, "cl45 reg:%#x-%#x, %#x -> %#x\n",
			 devid, reg, origin,
			 priv->mii->read(priv->mii,
					 phy_dev->mdio.addr,
					 reg_addr));
	} else if (!strncmp(buf, "rr", 2) &&
		   (sscanf(buf + 2, "%x", &reg) == 1)) {
		tmp_addr = ioremap_nocache(reg, 32);
		data = readl(tmp_addr);
		dev_info(dev, "rr reg%#x, value:%#x\n", reg, data);
	} else if (!strncmp(buf, "wr", 2) &&
		   (sscanf(buf + 2, "%x %x", &reg, &data) == 2)) {
		tmp_addr = ioremap_nocache(reg, 32);
		origin = readl(tmp_addr);
		writel(data, tmp_addr);
		dev_info(dev, "reg%#x, value:%#x -> %#x\n",
			 reg, origin, readl(tmp_addr));
	} else if (!strncmp(buf, "dump_mac", 8)) {
		for (i = 0; i < 0x1300 / 0x10 + 1; i++) {
			pr_info("%08x:\t%08x\t%08x\t%08x\t%08x\t\n",
				reg + i * 16,
				readl(priv->ioaddr + i * 0x10),
				readl(priv->ioaddr + i * 0x10 + 0x4),
				readl(priv->ioaddr + i * 0x10 + 0x8),
				readl(priv->ioaddr + i * 0x10 + 0xc));
		}
	} else if (!strncmp(buf, "tx", 2) &&
		   (sscanf(buf + 2, "%d", &reg) == 1)) {
		eth_smt_result = 0;
		if (!send_frame(priv, reg))
			eth_smt_result = 1;
	} else if (!strncmp(buf, "carrier_on", 10)) {
		netif_carrier_on(priv->dev);
	} else if (!strncmp(buf, "carrier_off", 11)) {
		netif_carrier_off(priv->dev);
	} else {
		dev_info(dev, "Error: command not support\n");
	}

	return count;
}

static DEVICE_ATTR_RW(stmmac);

int eth_create_attr(struct device *dev)
{
	int err = 0;

	if (!dev)
		return -EINVAL;

	err = device_create_file(dev, &dev_attr_stmmac);
	if (err)
		pr_err("create debug file fail:%s\n", __func__);

	return err;
}
#endif
