/*******************************************************************************
 *                           Include header files                              *
 ******************************************************************************/
#include <linux/module.h>
#include <net/sock.h>
#include <net/af_unix.h>
#include <linux/syscalls.h>
#include <linux/file.h>
#include <linux/debugfs.h>
#include <linux/seq_file.h>
//#include "ram_config.h"
#include <linux/socket_rpmsg.h>

#define SOCK_DATA_ALIGN(X)	ALIGN(X, SMP_CACHE_BYTES)
#define SOCK_DATA_MAX_LEN 8*1024
#define  ICP_CHN_SOCKET 32
#define sock_init_sema sema_init
#define sock_get_sema down_interruptible
#define sock_put_sema up

extern int socket_rpmsg_enable;

struct sock_channel g_sock_chn_info;
struct sock_rpmsg g_sock_rpmsg;
struct ipc_socket *g_socket_ipc;

struct semaphore g_sock_sem;
unsigned int g_sock_timeout = 5*1000;
//·Ϣһsocketرգйsocketɾ·Ϣд5Ԫ飨ͻˡˡʹ
struct hlist_head g_ipc_sockets;
struct hlist_head g_sock_rpmsg_info;
struct hlist_head g_sockets_info;
struct hlist_head g_sunaddr_info;	
char* flag_value[] ={
	"SOCK_DEAD",
	"SOCK_DONE",
	"SOCK_URGINLINE",
	"SOCK_KEEPOPEN",
	"SOCK_LINGER",
	"SOCK_DESTROY",
	"SOCK_BROADCAST",
	"SOCK_TIMESTAMP",
	"SOCK_ZAPPED",
	"SOCK_USE_WRITE_QUEUE", /* whether to call sk->sk_write_space in sock_wfree */
	"SOCK_DBG", /* %SO_DEBUG setting */
	"SOCK_RCVTSTAMP", /* %SO_TIMESTAMP setting */
	"SOCK_RCVTSTAMPNS", /* %SO_TIMESTAMPNS setting */
	"SOCK_LOCALROUTE", /* route locally only, %SO_DONTROUTE setting */
	"SOCK_QUEUE_SHRUNK", /* write queue has been shrunk recently */
	"SOCK_TIMESTAMPING_TX_HARDWARE",  /* %SOF_TIMESTAMPING_TX_HARDWARE */
	"SOCK_TIMESTAMPING_TX_SOFTWARE",  /* %SOF_TIMESTAMPING_TX_SOFTWARE */
	"SOCK_TIMESTAMPING_RX_HARDWARE",  /* %SOF_TIMESTAMPING_RX_HARDWARE */
	"SOCK_TIMESTAMPING_RX_SOFTWARE",  /* %SOF_TIMESTAMPING_RX_SOFTWARE */
	"SOCK_TIMESTAMPING_SOFTWARE",     /* %SOF_TIMESTAMPING_SOFTWARE */
	"SOCK_TIMESTAMPING_RAW_HARDWARE", /* %SOF_TIMESTAMPING_RAW_HARDWARE */
	"SOCK_TIMESTAMPING_SYS_HARDWARE", /* %SOF_TIMESTAMPING_SYS_HARDWARE */
	"SOCK_FASYNC", /* fasync() active */
	"SOCK_RXQ_OVFL",
	"SOCK_ZEROCOPY", /* buffers from userspace */
	"SOCK_WIFI_STATUS", /* push wifi status to userspace */
	"SOCK_NOFCS",
	"SOCK_IPCSOCK"
};

DEFINE_SPINLOCK(sock_table_lock);
EXPORT_SYMBOL_GPL(sock_table_lock);

DEFINE_SPINLOCK(sock_rpmsg_table_lock);
EXPORT_SYMBOL_GPL(sock_rpmsg_table_lock);

DEFINE_SPINLOCK(sock_socket_table_lock);
EXPORT_SYMBOL_GPL(sock_socket_table_lock);

DEFINE_SPINLOCK(sock_release_table_lock);
EXPORT_SYMBOL_GPL(sock_release_table_lock);

DEFINE_SPINLOCK(sock_release_lock);
EXPORT_SYMBOL_GPL(sock_release_lock);

DEFINE_SPINLOCK(sock_release_peer_lock);
EXPORT_SYMBOL_GPL(sock_release_peer_lock);

DEFINE_SPINLOCK(sock_socketaddr_table_lock);
EXPORT_SYMBOL_GPL(sock_socketaddr_table_lock);

static struct dentry *ipcsocket_debugfs_dir_entry_root;
static struct dentry *ipcsocket_debugfs_dir_entry_proc;

#define unix_peer(sk) (unix_sk(sk)->peer)

extern struct sock *unix_find_other_proxy(struct net *net,
				    struct sockaddr_un *sunname, int len,
				    int type, unsigned int hash, int *error);

extern int unix_dgram_peer_wake_connect_proxy(struct sock *sk, struct sock *other);

extern void unix_dgram_peer_wake_disconnect_proxy(struct sock *sk, struct sock *other);

extern void unix_dgram_peer_wake_disconnect_wakeup_proxy(struct sock *sk,
						   struct sock *other);
extern long unix_wait_for_peer_proxy(struct sock *other, long timeo);
extern int unix_recvq_full_proxy(const struct sock *sk);
extern int unix_recvq_full_proxy_lockless_and_deadstate(const struct sock *other);
extern int unix_stream_connect_proxy(struct socket *sock, struct sockaddr *uaddr,
			       int addr_len, int flags);
extern void unix_dgram_disconnected_proxy(struct sock *sk, struct sock *other);
extern int unix_mkname_proxy(struct sockaddr_un *sunaddr, int len, unsigned *hashp);
extern void unix_release_sock_proxy(struct sock *sk);
extern void init_peercred_proxy(struct sock *sk);

//static void delayed_release(struct work_struct *unused);
int sock_soc_socket_is_valid(struct socket* proxysock, struct socket* localsock);
int sock_soc_release_peer(struct release_socket* rsock);

static LIST_HEAD(delayed_release_list);

#define DEFINE_SHOW_ATTRIBUTE(__name)					\
static int __name##_open(struct inode *inode, struct file *file)	\
{									\
	return single_open(file, __name##_show, inode->i_private);	\
}									\
									\
static const struct file_operations __name##_fops = {			\
	.owner		= THIS_MODULE,					\
	.open		= __name##_open,				\
	.read		= seq_read,					\
	.llseek		= seq_lseek,					\
	.release	= single_release,				\
}

static int ipc_socket_info_show(struct seq_file *m, void *unused);
DEFINE_SHOW_ATTRIBUTE(ipc_socket_info);

enum sock_flags_cap {
	SOCK_DEAD_CAP,
	SOCK_DONE_CAP,
	SOCK_URGINLINE_CAP,
	SOCK_KEEPOPEN_CAP,
	SOCK_LINGER_CAP,
	SOCK_DESTROY_CAP,
	SOCK_BROADCAST_CAP,
	SOCK_TIMESTAMP_CAP,
	SOCK_ZAPPED_CAP,
	SOCK_USE_WRITE_QUEUE_CAP, /* whether to call sk->sk_write_space in sock_wfree */
	SOCK_DBG_CAP, /* %SO_DEBUG setting */
	SOCK_RCVTSTAMP_CAP, /* %SO_TIMESTAMP setting */
	SOCK_RCVTSTAMPNS_CAP, /* %SO_TIMESTAMPNS setting */
	SOCK_LOCALROUTE_CAP, /* route locally only, %SO_DONTROUTE setting */
	SOCK_MEMALLOC_CAP, /* VM depends on this socket for swapping */
	SOCK_TIMESTAMPING_RX_SOFTWARE_CAP,  /* %SOF_TIMESTAMPING_RX_SOFTWARE */
	SOCK_FASYNC_CAP, /* fasync() active */
	SOCK_RXQ_OVFL_CAP,
	SOCK_ZEROCOPY_CAP, /* buffers from userspace */
	SOCK_WIFI_STATUS_CAP, /* push wifi status to userspace */
	SOCK_NOFCS_CAP, /* Tell NIC not to do the Ethernet FCS.
		     * Will use last 4 bytes of packet sent from
		     * user-space instead.
		     */
	SOCK_FILTER_LOCKED_CAP, /* Filter cannot be changed anymore */
	SOCK_SELECT_ERR_QUEUE_CAP, /* Wake select on error queue */
	SOCK_RCU_FREE_CAP, /* wait rcu grace period in sk_destruct() */
	SOCK_TXTIME_CAP,
	SOCK_XDP_CAP, /* XDP is attached */
	SOCK_TSTAMP_NEW_CAP, /* Indicates 64 bit timestamps always */

    SOCK_IPCSOCK_CAP,  /*Indicates whether it is a cross core socket */

};
	

enum sock_flags_cap convert_to_cap_sock_flag(enum sock_flags flag)
{
	switch(flag){
		case SOCK_DEAD:
			return SOCK_DEAD_CAP;
		case SOCK_DONE:
			return SOCK_DONE_CAP;
		case SOCK_URGINLINE:
			return SOCK_URGINLINE_CAP;
		case SOCK_KEEPOPEN:
			return SOCK_KEEPOPEN_CAP;
		case SOCK_LINGER:
			return SOCK_LINGER_CAP;
		case SOCK_DESTROY:
			return SOCK_DESTROY_CAP; 
		case SOCK_BROADCAST:
			return SOCK_BROADCAST_CAP;
		case SOCK_TIMESTAMP:
			return SOCK_TIMESTAMP_CAP;
		case SOCK_ZAPPED:
			return SOCK_ZAPPED_CAP;
		case SOCK_USE_WRITE_QUEUE:
			return SOCK_USE_WRITE_QUEUE_CAP;
		case SOCK_DBG:
			return SOCK_DBG_CAP;
		case SOCK_RCVTSTAMP:
			return SOCK_RCVTSTAMP_CAP;
		case SOCK_RCVTSTAMPNS:
			return SOCK_RCVTSTAMPNS_CAP;
		case SOCK_LOCALROUTE:
			return SOCK_LOCALROUTE_CAP;
		case SOCK_QUEUE_SHRUNK:
			return -1;
		case SOCK_TIMESTAMPING_TX_HARDWARE:
			return -1;
		case SOCK_TIMESTAMPING_TX_SOFTWARE:
			return -1;
		case SOCK_TIMESTAMPING_RX_HARDWARE:
			return -1;
		case SOCK_TIMESTAMPING_RX_SOFTWARE:
			return SOCK_TIMESTAMPING_RX_SOFTWARE_CAP;
		case SOCK_TIMESTAMPING_SOFTWARE:
			return -1;
		case SOCK_TIMESTAMPING_RAW_HARDWARE:
			return -1;
		case SOCK_TIMESTAMPING_SYS_HARDWARE:
			return -1;
		case SOCK_FASYNC:
			return SOCK_FASYNC_CAP;
		case SOCK_RXQ_OVFL:
			return SOCK_RXQ_OVFL_CAP;
		case SOCK_ZEROCOPY:
			return SOCK_ZEROCOPY_CAP;
		case SOCK_WIFI_STATUS:
			return SOCK_WIFI_STATUS_CAP;
		case SOCK_NOFCS:
			return SOCK_NOFCS_CAP;		
		default:
			return -1;
	}
}

static int ipc_socket_info_show(struct seq_file *m, void *unused)
{	
	struct task_struct *task = NULL;
	char taskname[TASK_COMM_LEN] = {0};
	struct ipc_socket *p;
	struct hlist_node *pos,*n;
	struct socket* localsocket;	
	
	spin_lock(&sock_table_lock);
	if(hlist_empty(&g_ipc_sockets))
	{
		sk_soc_warn("sock_print_ipc_socket_info  g_ipc_sockets is empty");
		spin_unlock(&sock_table_lock);
		return 0;
	}
	hlist_for_each_entry_safe(p, pos, n, &g_ipc_sockets, hlist_node) {
		memset(taskname, 0, TASK_COMM_LEN);
		localsocket = p->localsocket;
		task = localsocket->task;
		get_task_comm(taskname, task);	
		sk_soc_warn("pid=%d, taskname=%s", task->pid, taskname);		
	}
	spin_unlock(&sock_table_lock);
	return 0;
}

static int sock_soc_is_socket_peer(struct socket* socket, struct socket* peer)
{
	struct socket_info *p;
	struct hlist_node *n, *pos;
	spin_lock(&sock_release_peer_lock);
	hlist_for_each_entry_safe(p, pos, n, &socket->peer, hlist_node) {
		if(p->proxysocket == peer){			
			spin_unlock(&sock_release_peer_lock);			
			return 0;
		}
	}
	spin_unlock(&sock_release_peer_lock);
	return -1;
}

static inline void sock_soc_insert_socket_peer(struct socket* socket, struct socket* peer)
{
	struct socket_info *psock_socket_info = NULL;

	//Ȳ鿴peerǷѾsocketpeer_list
	if(0 == sock_soc_is_socket_peer(socket, peer)){
		sk_soc_info("sock_soc_insert_socket_peer, peer already exist");
		return;
	}
	psock_socket_info = (struct socket_info *)kzalloc(sizeof(struct socket_info), GFP_ATOMIC);
	if(psock_socket_info == NULL){
		return;
	}
	memset(psock_socket_info, 0, sizeof(struct socket_info));
	psock_socket_info->proxysocket = peer;
	psock_socket_info->localsocket = socket;	
	
	sk_soc_info("sock_soc_insert_socket_peer sucess, proxysocket=%x, localsock=%x", psock_socket_info->proxysocket, psock_socket_info->localsocket);
	
	spin_lock(&sock_release_peer_lock);
	INIT_HLIST_NODE(&psock_socket_info->hlist_node);
	hlist_add_head(&psock_socket_info->hlist_node, &socket->peer);
	spin_unlock(&sock_release_peer_lock);
}

static int sock_soc_del_socket_peer(struct socket* socket, struct socket* peer)
{
	struct socket_info *p;
	struct hlist_node *n, *pos;
	spin_lock(&sock_release_peer_lock);
	hlist_for_each_entry_safe(p, pos, n, &socket->peer, hlist_node) {
		if(p->proxysocket == peer){
			hlist_del(&p->hlist_node);
			spin_unlock(&sock_release_peer_lock);
			kfree(p);
			return 0;
		}
	}
	spin_unlock(&sock_release_peer_lock);
	return -1;
}

static inline void sock_soc_insert_socket(struct hlist_node *list_node)
{
	spin_lock(&sock_table_lock);
	hlist_add_head(list_node, &g_ipc_sockets);
	spin_unlock(&sock_table_lock);
}

//жϱsocketǷipcбУܴڶ¼
static struct ipc_socket *sock_soc_get_ipcsocket_by_localsocket(struct socket *localsocket)
{	
	struct ipc_socket *p;
	struct hlist_node *pos,*n;
	spin_lock(&sock_table_lock);
	if(hlist_empty(&g_ipc_sockets))
	{
		sk_soc_info("sock_soc_get_ipcsocket_by_proxysock  g_ipc_sockets is empty");
		spin_unlock(&sock_table_lock);
		return NULL;
	}
	hlist_for_each_entry_safe(p, pos, n, &g_ipc_sockets, hlist_node) {
		if(p->localsocket == localsocket){
			spin_unlock(&sock_table_lock);
			return p;
		}
	}
	spin_unlock(&sock_table_lock);
	return NULL;
}

//˴sockΪsocketӦsockΪÿsocketg_ipc_socketsΨһ
static struct ipc_socket *sock_soc_get_ipcsocket_by_proxysock(struct sock *s)
{	
	struct ipc_socket *p;
	struct hlist_node *pos,*n;
	spin_lock(&sock_table_lock);
	if(hlist_empty(&g_ipc_sockets))
	{
		sk_soc_info("sock_soc_get_ipcsocket_by_proxysock  g_ipc_sockets is empty");
		spin_unlock(&sock_table_lock);
		return NULL;
	}
	hlist_for_each_entry_safe(p, pos, n, &g_ipc_sockets, hlist_node) {
		if(p->sock == s){
			spin_unlock(&sock_table_lock);
			return p;
		}
	}
	spin_unlock(&sock_table_lock);
	return NULL;
}

//ݿͻsocketͷsocketҶӦipc_socket
static struct ipc_socket *sock_get_ipcsocket_by_local_and_remote(void* localsock, void* remotesock)
{	
	struct ipc_socket *p;
	struct hlist_node *pos,*n;
	spin_lock(&sock_table_lock);
	if(hlist_empty(&g_ipc_sockets))
	{
		sk_soc_info("sock_get_ipcsocket_by_local_and_remote  g_ipc_sockets is empty");
		spin_unlock(&sock_table_lock);
		return NULL;
	}
	hlist_for_each_entry_safe(p, pos, n, &g_ipc_sockets, hlist_node) {
		if((p->localsocket == localsock)&&(p->remotesocket == remotesock)){
			spin_unlock(&sock_table_lock);
			return p;
		}
	}
	spin_unlock(&sock_table_lock);
	return NULL;
}

int sock_del_ipc_socket_by_proxysocket(struct socket* socket)
{
	struct ipc_socket *p;
	struct hlist_node *pos, *n;
	spin_lock(&sock_table_lock);
	if(hlist_empty(&g_ipc_sockets))
	{
		sk_soc_info("sock_del_ipc_socket_by_proxysocket  g_ipc_sockets is empty");
		spin_unlock(&sock_table_lock);
		return -1;
	}
	hlist_for_each_entry_safe(p, pos, n, &g_ipc_sockets, hlist_node) {
		if(p->socket == socket){
			hlist_del(&p->hlist_node);
			spin_unlock(&sock_table_lock);
			kfree(p);
			return 0;
		}
	}
	spin_unlock(&sock_table_lock);
	return -1;
}

//Ŀǰȸݿͻsocket˴socketͷsocketƥ
int sock_is_valide_ipc_socket(struct socket* localsocket, struct socket* proxysocket, struct socket* remotesocket/*, struct socket* remoteproxysocket*/)
{
	struct ipc_socket *p;
	struct hlist_node *pos, *n;
	spin_lock(&sock_table_lock);
	if(hlist_empty(&g_ipc_sockets))
	{
		sk_soc_info("sock_is_valide_ipc_socket  g_ipc_sockets is empty");
		spin_unlock(&sock_table_lock);
		return -1;
	}
	hlist_for_each_entry_safe(p, pos, n, &g_ipc_sockets, hlist_node) {
		if((p->localsocket == localsocket)&&(p->proxysocket == proxysocket)&&(p->remotesocket == remotesocket)/*&&(p->remoteproxysocket == remoteproxysocket)*/){
			
			spin_unlock(&sock_table_lock);
			
			return 0;
		}
	}
	spin_unlock(&sock_table_lock);
	sk_soc_info("sock_is_valide_ipc_socket failed, localsocket=%x, proxysocket=%x, remotesocket=%x", localsocket, proxysocket, remotesocket);
	return -1;
}

struct socket *sock_get_local_socket(int fd, int *err)
{
	struct socket* sock; 
	sock = sockfd_lookup(fd, err);
	if(NULL == sock){
		return NULL;
	}
	sockfd_put(sock);
	sk_soc_info("sock_get_local_socket fd=%d, file->f_count=%d ", fd, sock->file->f_count);
	return sock;
}

static inline void sock_soc_insert_rpmsg_info(struct hlist_node *list_node)
{
	spin_lock(&sock_rpmsg_table_lock);
	hlist_add_head(list_node, &g_sock_rpmsg_info);
	spin_unlock(&sock_rpmsg_table_lock);
}

struct sock_rpmsg_info * sock_soc_create_rpmsg_info(void* key)
{
	struct sock_rpmsg_info *psock_sock_rpmsg_info = NULL;
	
	psock_sock_rpmsg_info = (struct sock_rpmsg_info *)kzalloc(sizeof(struct sock_rpmsg_info), GFP_ATOMIC);
	if(psock_sock_rpmsg_info == NULL){
		sk_soc_info("sock_soc_create_rpmsg_info kzalloc failed \n");
		return NULL;
	}
	memset(psock_sock_rpmsg_info, 0, sizeof(struct sock_rpmsg_info));
	psock_sock_rpmsg_info->key = key;
	sema_init(&psock_sock_rpmsg_info->sock_sem, 0);
	sk_soc_info("sock_soc_create_rpmsg_info sucess, key=%x", key);
	sock_soc_insert_rpmsg_info(&psock_sock_rpmsg_info->hlist_node);
	return psock_sock_rpmsg_info;
}

struct sock_rpmsg_info * sock_soc_get_rpmsg_info(void* key)
{
	struct sock_rpmsg_info *p;
	struct hlist_node *pos,*n;
	//sk_soc_info("sock_soc_get_rpmsg_info key=%x \n", key);
	spin_lock(&sock_rpmsg_table_lock);
	hlist_for_each_entry_safe(p, pos, n, &g_sock_rpmsg_info, hlist_node) {		
		if(p->key == key){			
			spin_unlock(&sock_rpmsg_table_lock);
			return p;
		}
	}
	spin_unlock(&sock_rpmsg_table_lock);
	return NULL;
}

struct sock_rpmsg_info * sock_soc_del_rpmsg_info(void* key)
{
	struct sock_rpmsg_info *p;
	struct hlist_node *pos,*n;
	//sk_soc_info("sock_soc_del_rpmsg_info key=%x \n", key);
	spin_lock(&sock_rpmsg_table_lock);
	hlist_for_each_entry_safe(p, pos, n, &g_sock_rpmsg_info, hlist_node) {
		if(p->key == key){
			hlist_del(&p->hlist_node);
			spin_unlock(&sock_rpmsg_table_lock);
			sk_soc_info("sock_soc_del_rpmsg_info sucess, key=%x \n", key);
			kfree(p);
			return 0;
		}
	}
	spin_unlock(&sock_rpmsg_table_lock);
	sk_soc_info("sock_soc_del_rpmsg_info failed, key=%x \n", key);
	return -1;
}

static inline void sock_soc_insert_socket_info(struct hlist_node *list_node)
{
	spin_lock(&sock_socket_table_lock);
	hlist_add_head(list_node, &g_sockets_info);
	spin_unlock(&sock_socket_table_lock);
}

struct socket_info * sock_soc_set_socket_info(struct socket* proxysock, struct socket* localsock)
{
	struct socket_info *psock_socket_info = NULL;

	if(0 == sock_soc_socket_is_valid(proxysock, localsock)){
		sk_soc_info("sock_soc_socket_is_valid is true , do not add again");
		return NULL;
	}
	psock_socket_info = (struct socket_info *)kzalloc(sizeof(struct socket_info), GFP_ATOMIC);
	if(psock_socket_info == NULL){
		return NULL;
	}
	memset(psock_socket_info, 0, sizeof(struct socket_info));
	psock_socket_info->proxysocket = proxysock;
	psock_socket_info->localsocket = localsock;
	//sk_soc_info("sock_soc_set_socket_info sucess, key=%x", sock);
	sock_soc_insert_socket_info(&psock_socket_info->hlist_node);
	sk_soc_info("sock_soc_set_socket_info sucess, proxysocket=%x, localsock=%x", psock_socket_info->proxysocket, psock_socket_info->localsocket);
	return psock_socket_info;
}

int sock_soc_socket_is_valid(struct socket* proxysock, struct socket* localsock)
{
	struct ipc_socket *p;
	struct hlist_node *pos,*n;
	spin_lock(&sock_table_lock);
	if(hlist_empty(&g_ipc_sockets))
	{
		sk_soc_info("sock_soc_socket_is_valid  g_ipc_sockets is empty");
		spin_unlock(&sock_table_lock);
		return -1;
	}
	hlist_for_each_entry_safe(p, pos, n, &g_ipc_sockets, hlist_node) {
		if((p->proxysocket == proxysock)&&(p->localsocket == localsock)){
			spin_unlock(&sock_table_lock);
			return 0;
		}
	}
	spin_unlock(&sock_table_lock);
	return -1;
}

int sock_soc_del_socket_info(struct socket* proxysock, struct socket* localsock)
{
	struct ipc_socket *p;
	struct hlist_node *pos,*n;
	//sk_soc_info("sock_soc_del_socket_info key=%x \n", sock);
	spin_lock(&sock_table_lock);
	if(hlist_empty(&g_ipc_sockets))
	{
		sk_soc_info("sock_soc_del_socket_info  g_ipc_sockets is empty");
		spin_unlock(&sock_table_lock);
		return -1;
	}
	hlist_for_each_entry_safe(p, pos, n, &g_ipc_sockets, hlist_node) {
		if((p->proxysocket == proxysock)&&(p->localsocket == localsock)){
			hlist_del(&p->hlist_node);
			spin_unlock(&sock_table_lock);
			sk_soc_info("sock_soc_del_socket_info sucess, proxysock=%x, localsock=%x \n", proxysock, localsock);
			kfree(p);
			return 0;
		}
	}
	spin_unlock(&sock_table_lock);
	sk_soc_info("sock_soc_del_socket_info failed, proxysock=%x,localsock=%x \n", proxysock, localsock);
	return -1;
}

int sock_soc_socket_is_valid_other(struct socket* other)
{
	return 0;
}

int sock_soc_socket_is_valid_ipc(struct socket* localsocket, struct socket* remotesocket)
{	
	struct ipc_socket *p;
	struct hlist_node *pos,*n;
	spin_lock(&sock_table_lock);
	if(hlist_empty(&g_ipc_sockets))
	{
		sk_soc_info("sock_soc_socket_is_valid_ipc  g_ipc_sockets is empty");
		spin_unlock(&sock_table_lock);
		return -1;
	}
	hlist_for_each_entry_safe(p, pos, n, &g_ipc_sockets, hlist_node) {
		if((p->localsocket == localsocket)&&(p->remotesocket == remotesocket)){
			spin_unlock(&sock_table_lock);
			return 0;
		}
	}
	spin_unlock(&sock_table_lock);
	return -1;
}

struct sock_rpmsg_info * sock_soc_rsp_proc(struct sock_rpmsg* psock_rpmsg)
{
	struct sock_rpmsg_info *psock_rpmsg_info;
	psock_rpmsg_info = sock_soc_get_rpmsg_info(psock_rpmsg->key);	
	if(NULL == psock_rpmsg_info){
		sk_soc_err("sock_soc_get_rpmsg_info failed!");
		return NULL;
	}
	memset(&psock_rpmsg_info->sock_rpmsg, 0, sizeof(struct sock_rpmsg));
	memcpy(&psock_rpmsg_info->sock_rpmsg, psock_rpmsg, sizeof(struct sock_rpmsg));	
	up(&psock_rpmsg_info->sock_sem);
	return psock_rpmsg_info;
}

bool sock_soc_ipc_sunaddr_exist(struct sockaddr_un *psockaddr_un)
{
	struct sock_sunaddr_info *p;
	struct hlist_node *pos,*n;
	spin_lock(&sock_socketaddr_table_lock);
	hlist_for_each_entry_safe(p, pos, n, &g_sunaddr_info, hlist_node) {
		if((p->sockaddr.sun_family == psockaddr_un->sun_family) \
			&&(0 == memcmp(p->sockaddr.sun_path, psockaddr_un->sun_path, UNIX_PATH_MAX))){			
			spin_unlock(&sock_socketaddr_table_lock);			
			return true;
		}
	}
	spin_unlock(&sock_socketaddr_table_lock);
	return false;
}

static void sock_soc_insert_ipc_sunaddr(struct sockaddr_un *psockaddr_un)
{
	struct sock_sunaddr_info *psock_sunaddr_info = NULL;

	//Ȳ鿴sunaddrǷѾg_sunaddr_info
	if(true == sock_soc_ipc_sunaddr_exist(psockaddr_un)){
		sk_soc_info("sock_soc_insert_ipc_sunaddr, sunaddr already exist");
		return;
	}
	
	psock_sunaddr_info = (struct sock_sunaddr_info *)kzalloc(sizeof(struct sock_sunaddr_info), GFP_ATOMIC);
	if(psock_sunaddr_info == NULL){
		sk_soc_info("sock_soc_insert_ipc_sunaddr, alloc failed");
		return;
	}
	memset(psock_sunaddr_info, 0, sizeof(struct sock_sunaddr_info));
	memcpy(&psock_sunaddr_info->sockaddr, psockaddr_un, sizeof(struct sockaddr_un));	
		
	INIT_HLIST_NODE(&psock_sunaddr_info->hlist_node);
	spin_lock(&sock_socketaddr_table_lock);
	hlist_add_head(&psock_sunaddr_info->hlist_node, &g_sunaddr_info);
	spin_unlock(&sock_socketaddr_table_lock);
}

static int sock_soc_del_ipc_sunaddr(struct sockaddr_un *psockaddr_un)
{
	struct sock_sunaddr_info *p;
	struct hlist_node *pos,*n;
	int ret;
	spin_lock(&sock_socketaddr_table_lock);
	hlist_for_each_entry_safe(p, pos, n, &g_sunaddr_info, hlist_node) {		
		if(psockaddr_un->sun_path[0]){
			ret = strncmp(p->sockaddr.sun_path, psockaddr_un->sun_path, strlen(psockaddr_un->sun_path));
		}else{
			ret = strncmp(&p->sockaddr.sun_path[1], &psockaddr_un->sun_path[1], strlen(&psockaddr_un->sun_path[1]));
		}		
		
		if((p->sockaddr.sun_family == psockaddr_un->sun_family) && (0 == ret)){
			hlist_del(&p->hlist_node);			
			kfree(p);
			sk_soc_info("sun_path=%s, del sucess.", psockaddr_un->sun_path);
			spin_unlock(&sock_socketaddr_table_lock);
			return 0;
		}
	}
	sk_soc_info("sun_path=%s, del failed", psockaddr_un->sun_path);
	spin_unlock(&sock_socketaddr_table_lock);
	return -1;
}

int unix_is_ipc_socket(struct sock *sock)
{
	int result;
	result = test_bit(SOCK_IPCSOCK, &sock->sk_flags);
	//sk_soc_info("unix_is_ipc_socket result=%d", result);
	return result;
}
EXPORT_SYMBOL_GPL(unix_is_ipc_socket);

int unix_is_ipc_local_socket(struct sock *sock)
{
	int result;
	result = test_bit(SOCK_IPC_LOCAL, &sock->sk_flags);
	//sk_soc_info("unix_is_ipc_socket result=%d", result);
	return result;
}
EXPORT_SYMBOL_GPL(unix_is_ipc_local_socket);

/*ֵڵ0ʾдͨɹС0ʾдͨʧ*/
static int sock_channel_write(struct sock_channel *chninfo, void *buf, unsigned int len)
{
    T_ZDrvRpMsg_Msg msg;

    if(NULL == buf) {
        return -EINVAL;
    }
    memset(&msg, 0, sizeof(msg));
    msg.actorID = chninfo->core_id;
    msg.chID   = chninfo->channel_id;
    msg.flag |= RPMSG_WRITE_INT; //| RPMSG_WRITE_IRQLOCK;
    msg.buf = buf;
    msg.len = len;
	//sk_soc_info("sock_channel_write process, len=%d", len);
    return sockSocWrite(&msg);
}

/*ֵ0ʾȡͨɹСڵ0ʾͨΪջʧ*/
static int sock_channel_read(struct sock_channel *chninfo, void *buf, unsigned int len)
{
    T_ZDrvRpMsg_Msg msg;
    int ret = 0;

    if(NULL == buf) {
        return -EINVAL;
    }
	sk_soc_info("sock_channel_read enter");
    memset(&msg, 0, sizeof(msg));
    msg.actorID = chninfo->core_id;
    msg.chID   = chninfo->channel_id;
    msg.buf    = buf;
    msg.len    = len;

    ret = sockSocRead(&msg);
    if (ret <= 0) {
        sk_soc_err("rpm read err=%d!",ret);
        return ret;
    }

    return ret;
}

static int sock_thread_enqueue_data(struct sock_rpmsg *sock_rpmsg)
{
	unsigned long flags;
	struct sock_thread_info* p_main_thread_info;
	p_main_thread_info = &g_sock_chn_info.main_thread_info;
	
	spin_lock_irqsave(&p_main_thread_info->p_sock_lock,flags);
	list_add_tail(&sock_rpmsg->list, &p_main_thread_info->p_sock_list);
	spin_unlock_irqrestore(&p_main_thread_info->p_sock_lock,flags);
	sock_put_sema(&p_main_thread_info->p_sock_sem);

	return 0;
}

struct sock_rpmsg* sock_create_rpmsg(unsigned int size)
{
	struct sock_rpmsg* sock_rpmsg;
	
	size = SOCK_DATA_ALIGN(size);
	size += SOCK_DATA_ALIGN(sizeof(struct sock_rpmsg));
	if(size > SOCK_DATA_MAX_LEN){
		sk_soc_err("size of data is too large, max support size is %d", SOCK_DATA_MAX_LEN-sizeof(struct sock_rpmsg));
	}
	sock_rpmsg = kzalloc(size, GFP_KERNEL);
	if (sock_rpmsg == NULL) {
		sk_soc_err("kzalloc fial!!");
		return NULL;
	}
	return sock_rpmsg;
}

static int sock_send_ipc_msg(struct	sock_rpmsg* psock_rpmsg)
{	
	struct	sock_rpmsg* sock_rpmsg;
	sock_rpmsg = sock_create_rpmsg(0);
	if(NULL == sock_rpmsg){
		sk_soc_err("kzalloc fial!!");
		return -1;
	}
	memcpy(sock_rpmsg, psock_rpmsg, sizeof(struct	sock_rpmsg));
	sock_rpmsg->dir = DIR_DOWNLINK;
	sock_thread_enqueue_data(sock_rpmsg);
	//sk_soc_info("sock_send_ipc_msg write len=%d", len);
	return 0;
}

static int sock_send_ipc_msg_data(struct	sock_rpmsg* psock_rpmsg)
{	
	if(NULL == psock_rpmsg){
		sk_soc_err("psock_rpmsg is NULL!!");
		return -1;
	}
	
	psock_rpmsg->dir = DIR_DOWNLINK;
	sock_thread_enqueue_data(psock_rpmsg);
	//sk_soc_info("sock_send_ipc_msg write len=%d", len);
	return 0;
}

static int sock_recv_ipc_msg(struct	sock_rpmsg* psock_rpmsg)
{	
	struct	sock_rpmsg* sock_rpmsg;
	sk_soc_info("sock_receive_thread sock_rpmsg.msg_type=%d,index=%d", psock_rpmsg->msg_type, psock_rpmsg->index);
	sock_rpmsg = sock_create_rpmsg(psock_rpmsg->data_len);
	if(NULL == sock_rpmsg){
		sk_soc_err("kzalloc fial!!");
		return -1;
	}
	memcpy(sock_rpmsg, psock_rpmsg, psock_rpmsg->data_len+sizeof(struct sock_rpmsg));
	sock_rpmsg->dir = DIR_UPLINK;
	sock_thread_enqueue_data(sock_rpmsg);
	//sk_soc_info("sock_send_ipc_msg write len=%d", len);
	return 0;
}

int sock_lookup_fd_by_sock(struct sock *sock)
{
	struct socket *socket;
	socket = sock->sk_socket;
	sk_soc_info("sock_lookup_fd_by_sock socket=%x, socket->fd=%d\n", socket, socket->fd);
	return socket->fd;
}

static int sock_channel_clear(struct sock_channel *chninfo)
{
    char *tbuf = NULL;
    unsigned int tlen = chninfo->channel_size/2;
    int ret = 0;

    tbuf = (char *)kzalloc(tlen,GFP_ATOMIC);
    if(IS_ERR(tbuf)) {
        sk_soc_err("kzalloc fail! %d byte.", tlen);
        return -ENOMEM;
    }
    ret = sock_channel_read(chninfo, tbuf, tlen);
    if(ret < 0) {
        sk_soc_err("zvnet_channel_read fail!");
        ret = 0;
    }
    kfree(tbuf);
    sk_soc_err("Drop channel data. %d byte.",ret);

    return ret;
}

void usock_print(struct unix_sock* usock){	
	sk_soc_info("usock->rsock.localfd=%d", usock->rsock.localfd);
	sk_soc_info("usock->rsock.localsocket=%x", usock->rsock.localsocket);
	sk_soc_info("usock->rsock.proxyfd=%d", usock->rsock.proxyfd);
	sk_soc_info("usock->rsock.proxysocket=%x", usock->rsock.proxysocket);
	sk_soc_info("usock->rsock.remotefd=%d", usock->rsock.remotefd);
	sk_soc_info("usock->rsock.remotesocket=%x", usock->rsock.remotesocket);
}

void usock_init(struct sock* sock, int localfd, struct socket* localsocket, int proxyfd, struct socket* proxysocket, int remotefd, struct socket* remotesocket){
	struct unix_sock* usock;
	if(NULL == sock){
		return;
	}
	usock = unix_sk(sock);
	usock->rsock.localfd = localfd;
	usock->rsock.localsocket = localsocket;
	usock->rsock.proxyfd = proxyfd;
	usock->rsock.proxysocket = proxysocket;
	usock->rsock.remotefd = remotefd;
	usock->rsock.remotesocket = remotesocket;
}

//socketº󣬸´socketrsockϢ
void usock_update(struct sock* sock, int localfd, struct socket* localsocket)
{
	struct unix_sock* usock;
	if(NULL == sock){
		return;
	}
	usock = unix_sk(sock);
	usock->rsock.localfd = localfd;
	usock->rsock.localsocket = localsocket;
}

void usock_update_remote_proxy_socket(struct sock* sock, struct socket* remoteproxysocket)
{
	struct unix_sock* usock;
	usock = unix_sk(sock);
	usock->rsock.remoteproxysocket = remoteproxysocket;
}

static struct ipc_socket * sock_create_proxy_socket_server(struct sock_rpmsg* psock_rpmsg)
{
	struct ipc_socket *psocket_ipc = NULL;	
	struct socket *sock;
	
	int err;	
	
	sk_soc_info("clientsock=%x, serversock=%x!\n", psock_rpmsg->clientsock, psock_rpmsg->serversock);
	psocket_ipc = sock_get_ipcsocket_by_local_and_remote(psock_rpmsg->clientsock, psock_rpmsg->serversock);
	if(NULL != psocket_ipc) {  //Ѿͬһ˵Ĵظ
		sk_soc_info("socket_server already exist, do not create again!\n");
		sock_hold(psocket_ipc->sock); //0705
		return psocket_ipc;
	}
	
	psocket_ipc = (struct ipc_socket *)kzalloc(sizeof(struct ipc_socket), GFP_ATOMIC);
	if(NULL == psocket_ipc){		
		sk_soc_err("kzalloc failed!\n");
		return NULL;
	}
	psocket_ipc->sockfd = sys_socket(AF_UNIX, psock_rpmsg->socktype, 0); //socket
	sk_soc_info("sock_create_proxy_socket_server socktype=%d, socket->fd=%d", psock_rpmsg->socktype, psocket_ipc->sockfd);
	//memcpy(&psocket_ipc->sock, &g_sock_rpmsg.sock, sizeof(struct sock));
	//psocket_ipc->socket.sk = &psocket_ipc->sock;
    psocket_ipc->socket = sock_get_local_socket(psocket_ipc->sockfd, &err);
	if(NULL != psocket_ipc->socket){
		sk_soc_info("sock_create_proxy_socket_server sucess\n");
		psocket_ipc->sock = psocket_ipc->socket->sk;
		psocket_ipc->proxysocket = psocket_ipc->socket;
		psocket_ipc->socktype = psock_rpmsg->socktype;
		set_bit(SOCK_IPCSOCK,&psocket_ipc->sock->sk_flags);
		psocket_ipc->sock->ipc_flag = 1;
		psocket_ipc->sock->sk_fd = psocket_ipc->sockfd;
		//sock_soc_set_socket_info(psocket_ipc->socket, psock_rpmsg->clientsock);
		sock_hold(psocket_ipc->sock); //0704
		init_peercred_proxy(psocket_ipc->sock);
		//ʼsocketusockϢ
		usock_init(psocket_ipc->sock, psock_rpmsg->clientfd, psock_rpmsg->clientsock, psocket_ipc->sockfd, psocket_ipc->socket, psock_rpmsg->serverfd, psock_rpmsg->serversock);
		usock_update_remote_proxy_socket(psocket_ipc->sock, psock_rpmsg->serverproxysock);
		//usock_print(usock);
	}
	else{
		sk_soc_info("sock_create_proxy_socket_server failed, socket = NULL\n");
	}
	//ͬ
	//psocket_ipc->sock->sk_flags = g_sock_rpmsg.sock.sk_flags;
	//psocket_ipc->sock->sk_state = g_sock_rpmsg.sock.sk_state;
	//psocket_ipc->sock->sk_shutdown = g_sock_rpmsg.sock.sk_shutdown;
	//psocket_ipc->sock->sk_receive_queue.qlen = g_sock_rpmsg.sock.sk_receive_queue.qlen;
	//psocket_ipc->sock->sk_max_ack_backlog = g_sock_rpmsg.sock.sk_max_ack_backlog;
	psocket_ipc->localfd = psock_rpmsg->clientfd;
	psocket_ipc->localsocket = psock_rpmsg->clientsock;
	psocket_ipc->remotefd = psock_rpmsg->serverfd;
	psocket_ipc->remotesocket = psock_rpmsg->serversock;
	psocket_ipc->remoteproxyfd = psock_rpmsg->serverproxyfd;
	psocket_ipc->remoteproxysocket = psock_rpmsg->serverproxysock;
	psocket_ipc->key = psock_rpmsg->serversock;
	sk_soc_info("sock_create_proxy_socket_server sockfd=%d, remotefd=%d, key=%x\n", psocket_ipc->sockfd, psocket_ipc->remotefd, psocket_ipc->key);
	INIT_HLIST_NODE(&psocket_ipc->hlist_node);
	sock_soc_insert_socket(&psocket_ipc->hlist_node);
	return psocket_ipc;
}

static struct ipc_socket * sock_create_proxy_socket_client(struct sock_rpmsg* psock_rpmsg)
{
	struct ipc_socket *psocket_ipc = NULL;	
	struct socket *sock;
	struct unix_sock* usock;
	int err;
	//ƾȷǷͬһͻ˵ĴΪԶ˿ܹرղһַͬsocket
	if(0 == psock_rpmsg->isfistfind){
		psocket_ipc = sock_get_ipcsocket_by_local_and_remote(psock_rpmsg->serversock, psock_rpmsg->clientsock);
		if(NULL != psocket_ipc) {  //Ѿͬһͻ˵Ĵظ
			//sock_hold(psocket_ipc->sock); //0703
			return psocket_ipc;
		}
	}
	
	psocket_ipc = (struct ipc_socket *)kzalloc(sizeof(struct ipc_socket), GFP_ATOMIC);
	if(NULL == psocket_ipc){		
		sk_soc_err("kzalloc failed!\n");
		return NULL;
	}
	sk_soc_info("sock_create_proxy_socket_client socktype=%d\n", psock_rpmsg->socktype);
	psocket_ipc->sockfd = sys_socket(AF_UNIX, psock_rpmsg->socktype, 0); //ͻsocket
	sk_soc_info("sock_create_proxy_socket_client socket->fd=%d\n", psocket_ipc->sockfd);
	psocket_ipc->socket = sock_get_local_socket(psocket_ipc->sockfd, &err);
	
	//usock_print(usock);	
	
	if(NULL != psocket_ipc->socket){
		sk_soc_info("sockfd_lookup sucess, socket=%x \n", psocket_ipc->socket);
		//ʼsocketusockϢ
		usock_init(psocket_ipc->socket->sk, psock_rpmsg->serverfd, psock_rpmsg->serversock, psocket_ipc->sockfd, psocket_ipc->socket, psock_rpmsg->clientfd, psock_rpmsg->clientsock);

		psocket_ipc->sock = psocket_ipc->socket->sk;
		set_bit(SOCK_IPCSOCK,&psocket_ipc->sock->sk_flags);
		psocket_ipc->sock->ipc_flag = 1;
		psocket_ipc->sock->sk_fd = psocket_ipc->sockfd;
		//sock_hold(psocket_ipc->sock); //todo: clientǲǲӦõsock_hold?
	}
	else{
		sk_soc_info("server socket = NULL");
	}
	//ΪʱԶ˵Ĵsocketδ޷ȡremoteproxyfdremoteproxysocket
	psocket_ipc->localfd = 	psock_rpmsg->serverfd;
	psocket_ipc->localsocket = psock_rpmsg->serversock;
	psocket_ipc->remotefd = psock_rpmsg->clientfd;
	psocket_ipc->remotesocket = psock_rpmsg->clientsock;	
	psocket_ipc->proxysocket = psocket_ipc->socket;
	
	psocket_ipc->key = psock_rpmsg->clientsock;
	sk_soc_info("sock_create_proxy_socket_client sockfd=%d,remotefd=%d,key=%x", psocket_ipc->sockfd, psocket_ipc->remotefd,psocket_ipc->key);
	INIT_HLIST_NODE(&psocket_ipc->hlist_node);
	sock_soc_insert_socket(&psocket_ipc->hlist_node);	
	return psocket_ipc;
}

int sock_soc_recvq_full(const struct sock *other)
{
	struct sock_rpmsg sock_rpmsg = {0};
	struct ipc_socket * ipc_socket; 
	struct sock_rpmsg_info *psock_rpmsg_info;
	int sockfd;
	int result;
	sk_soc_info("sock_soc_recvq_full enter");
	sock_rpmsg.msg_type = MSG_TYPE_RECVQ_FULL;
	sockfd = sock_lookup_fd_by_sock(other);
	//sock_rpmsg.serverfd = sock_lookup_serverfd_by_clientfd(sockfd);
	//ipc_socket = sock_lookup_server_by_clientfd(sockfd);
	ipc_socket = sock_soc_get_ipcsocket_by_proxysock(other);
	if(NULL == ipc_socket){
		sk_soc_info("sock_soc_recvq_full get ipc_socket faild \n");
		return -1;
	}
	sock_rpmsg.serverfd = ipc_socket->remotefd;
	sock_rpmsg.serversock = ipc_socket->remotesocket;
	sock_rpmsg.clientsock = ipc_socket->localsocket;
	sock_rpmsg.key = other;
	psock_rpmsg_info = sock_soc_create_rpmsg_info(sock_rpmsg.key);
	if(NULL == psock_rpmsg_info){
		sk_soc_info("sock_soc_recvq_full create rpmsg faild \n");
		return -1;
	}
	sock_send_ipc_msg(&sock_rpmsg);
	down_timeout(&psock_rpmsg_info->sock_sem, msecs_to_jiffies(g_sock_timeout));
	result = psock_rpmsg_info->sock_rpmsg.result;
	sock_soc_del_rpmsg_info(sock_rpmsg.key);
	return result;
}
EXPORT_SYMBOL_GPL(sock_soc_recvq_full);

int sock_soc_recvq_full_proc(struct sock_rpmsg* psock_rpmsg)
{
	struct ipc_socket* ipcsocket;	 
	struct socket *other;
	int err;
	int result;
	struct sock_rpmsg sock_rpmsg = {0};
	sk_soc_info("sock_soc_recvq_full_proc serverfd=%d", psock_rpmsg->serverfd);
	sock_rpmsg.msg_type = MSG_TYPE_RECVQ_FULL_RESULT;
	sock_rpmsg.key = psock_rpmsg->key;
	//other = sock_get_local_socket(psock_rpmsg->serverfd, &err);
	other = (struct socket*)psock_rpmsg->serversock;
	if(sock_soc_socket_is_valid_ipc(other, psock_rpmsg->clientsock) == -1){ //serversockѹر
		sk_soc_info("sock_soc_socket_is_valid is false \n");
		sock_send_ipc_msg(&sock_rpmsg);
		return -1;
	}
	sock_rpmsg.result = unix_recvq_full_proxy(other->sk);
	sk_soc_info("sock_soc_recvq_full_proc result=%d\n", sock_rpmsg.result);
	sock_rpmsg.serverfd = other->fd; //sock_lookup_fd_by_sock(other->sk);	
	sock_send_ipc_msg(&sock_rpmsg);
	return 0;
}

int sock_soc_recvq_full_result_proc(struct sock_rpmsg* psock_rpmsg)
{
	sk_soc_info("sock_soc_recvq_full_result_proc\n");
	sock_soc_rsp_proc(psock_rpmsg);
	return 0;
}

int sock_soc_recvq_full_lockless_and_dead(const struct sock *other)
{
	struct sock_rpmsg sock_rpmsg = {0};
	struct ipc_socket *ipc_socket; 
	struct sock_rpmsg_info *psock_rpmsg_info;
	int sockfd;
	int result;
	sk_soc_info("sock_soc_recvq_full_lockless_and_dead\n");
	sock_rpmsg.msg_type = MSG_TYPE_RECVQ_FULL_LOCKLESS_AND_DEAD;
	sockfd = sock_lookup_fd_by_sock(other);
	//sock_rpmsg.serverfd = sock_lookup_serverfd_by_clientfd(sockfd);
	//ipc_socket = sock_lookup_server_by_clientfd(sockfd);
	ipc_socket = sock_soc_get_ipcsocket_by_proxysock(other);
	if(NULL == ipc_socket){
		sk_soc_info("sock_soc_recvq_full_lockless_and_dead get ipc_socket faild \n");
		return -1;
	}
	sock_rpmsg.serverfd = ipc_socket->remotefd;
	sock_rpmsg.serversock = ipc_socket->remotesocket;
	sock_rpmsg.clientsock = ipc_socket->localsocket;
	sock_rpmsg.key = other;
	psock_rpmsg_info = sock_soc_create_rpmsg_info(sock_rpmsg.key);
	if(NULL == psock_rpmsg_info){
		sk_soc_info("sock_soc_recvq_full_lockless_and_dead create rpmsg faild \n");
		return -1;
	}
	sock_send_ipc_msg(&sock_rpmsg);	
	down_timeout(&psock_rpmsg_info->sock_sem, msecs_to_jiffies(g_sock_timeout));
	result = psock_rpmsg_info->sock_rpmsg.result;
	sock_soc_del_rpmsg_info(sock_rpmsg.key);
	return result;
}
EXPORT_SYMBOL_GPL(sock_soc_recvq_full_lockless_and_dead);

int sock_soc_recvq_full_lockless_and_dead_proc(struct sock_rpmsg* psock_rpmsg)
{
	struct ipc_socket* ipcsocket;	 
	struct socket *other;
	int err;
	int result;
	struct sock_rpmsg sock_rpmsg = {0};
	sk_soc_info("sock_soc_recvq_full_lockless_and_dead_proc");
	sock_rpmsg.msg_type = MSG_TYPE_RECVQ_FULL_LOCKLESS_AND_DEAD_RESULT;
	sock_rpmsg.key = psock_rpmsg->key;
	//other = sock_get_local_socket(psock_rpmsg->serverfd, &err);
	other = (struct socket*)psock_rpmsg->serversock;
	if(sock_soc_socket_is_valid_ipc(other, psock_rpmsg->clientsock) == -1){ //serversockѹر
		sk_soc_info("sock_soc_socket_is_valid is false \n");
		sock_send_ipc_msg(&sock_rpmsg);
		return -1;
	}
	sock_rpmsg.result = unix_recvq_full_proxy_lockless_and_deadstate(other->sk);
	sock_rpmsg.serverfd = sock_lookup_fd_by_sock(other->sk);	
	sock_send_ipc_msg(&sock_rpmsg);
	return 0;
}

int sock_soc_recvq_full_lockless_and_dead_result_proc(struct sock_rpmsg* psock_rpmsg)
{
	sk_soc_info("sock_soc_recvq_full_lockless_and_dead_result_proc");
	sock_soc_rsp_proc(psock_rpmsg);
	return 0;
}
/*
** sock: socket
** other:Զsocketıش
*/
int sock_soc_set_peer(struct sock *sock, struct sock *other)
{
	struct sock_rpmsg sock_rpmsg = {0};
	struct ipc_socket *ipc_socket;
	int sockfd; 
	sock_rpmsg.msg_type = MSG_TYPE_SET_PEER;
	sock_rpmsg.clientfd = sock_lookup_fd_by_sock(sock);
	sock_rpmsg.clientsock = sock->sk_socket;
	sockfd = sock_lookup_fd_by_sock(other);
	//sock_rpmsg.serverfd = sock_lookup_serverfd_by_clientfd(sockfd);	
	//ipc_socket = sock_lookup_server_by_clientfd(sockfd);
	ipc_socket = sock_soc_get_ipcsocket_by_proxysock(other);
	if(NULL == ipc_socket){
		sk_soc_info("sock_soc_set_peer get ipc_socket faild \n");
		return -1;
	}
	sock_rpmsg.serverfd = ipc_socket->remotefd;
	sock_rpmsg.serversock = ipc_socket->remotesocket;
	sk_soc_info("sock_soc_set_peer clientfd=%d, serverfd=%d\n", sock_rpmsg.clientfd, sock_rpmsg.serverfd);

	unix_peer(sock) = other;
	sock_send_ipc_msg(&sock_rpmsg);
	return 0;
}
EXPORT_SYMBOL_GPL(sock_soc_set_peer);

int sock_soc_set_peer_proc(struct sock_rpmsg* psock_rpmsg)
{
	struct ipc_socket* ipcsocket;
	struct socket *clientsocket; 
	struct socket *serversocket;
	struct sock *sock; 
	struct sock *other;
	int err;
	sk_soc_info("sock_soc_set_peer_proc clientfd=%d, serverfd=%d", psock_rpmsg->clientfd, psock_rpmsg->serverfd);
	//ڷ˵,serversockڱsocketclientsockԶsocket
	ipcsocket = sock_get_ipcsocket_by_local_and_remote(psock_rpmsg->serversock, psock_rpmsg->clientsock);
	if(ipcsocket == NULL){
		sk_soc_info("sock_soc_set_peer_proc failed, clientfd=%d,clientsock=%x\n ", psock_rpmsg->clientfd, psock_rpmsg->clientsock);
		return -1;
	}	
	//clientsocket = (struct socket*)psock_rpmsg->clientsock;
	sock = ipcsocket->sock;
	//serversocket = sock_get_local_socket(psock_rpmsg->serverfd, &err);
	serversocket = (struct socket*)psock_rpmsg->serversock;
	if(sock_soc_socket_is_valid_ipc(serversocket, psock_rpmsg->clientsock) == -1){ //serversockѹر
		sk_soc_info("sock_soc_socket_is_valid is false \n");		
		return -1;
	}
	other = serversocket->sk; 
	
	unix_peer(sock) = other;
	sock_hold(other);
	//unix_peer(other) = sock; //0703
	return 0;
}

/*socket->peer==socksocket,ȫΪsocket->peer=newsock*/
//ýӿڷ֪ͨͻ
int sock_soc_update_peer(struct socket *sock, struct socket *newsock)
{
	struct sock_rpmsg sock_rpmsg = {0};
	struct sock* proxysock;
	struct ipc_socket* ipcsocket;
	int sockfd;
	struct unix_sock* usock;
	//newsock->sk->peerָǿͻ˴sock
	proxysock = unix_peer(newsock->sk);
	usock = unix_sk(proxysock);
	usock->rsock.localsocket = newsock;
	ipcsocket = sock_soc_get_ipcsocket_by_proxysock(proxysock);
	if(NULL != ipcsocket){
		sock_rpmsg.serversock = ipcsocket->remotesocket;
		ipcsocket->localsocket = newsock;
		sk_soc_info("remotesocket=%x, localsocket=%x", ipcsocket->remotesocket, ipcsocket->localsocket);
	}else{
		sk_soc_info("sock_soc_get_ipcsocket_by_proxysock failed, proxysock=%x, sock=%x", proxysock, sock);
		return -1;
	}
	sk_soc_info("sock=%x, newsock=%x", sock, newsock);
	sock_rpmsg.msg_type = MSG_TYPE_UPDATE_PEER;
	sock_rpmsg.clientfd = sock->fd;
	sock_rpmsg.clientsock = sock;
	sock_rpmsg.newclientfd = newsock->fd;	
	sock_rpmsg.newsock = newsock;
	sock_soc_del_socket_peer(sock, ipcsocket->proxysocket);
	sock_soc_insert_socket_peer(newsock, ipcsocket->proxysocket);
	usock_update(ipcsocket->sock, newsock->fd, newsock);
	//newsock->peer = sock->peer; //0706
	//sock->peer = NULL; //0706
	sock_rpmsg.key = sock_rpmsg.serversock;
	sk_soc_info("clientfd=%d, newclientfd=%d\n", sock_rpmsg.clientfd, sock_rpmsg.newclientfd);
	//sock_soc_del_socket_info(proxysock->sk_socket, sock); //ɾɵϢ
	//sock_soc_set_socket_info(proxysock->sk_socket, newsock); //µϢ
	sock_send_ipc_msg(&sock_rpmsg);
	return 0;
}

EXPORT_SYMBOL_GPL(sock_soc_update_peer);

int sock_soc_update_peer_proc(struct sock_rpmsg* psock_rpmsg)
{
	struct ipc_socket* ipcsocket;
	struct socket *clientsocket; 
	struct socket *serversocket;
	struct sock *sock; 
	struct sock *other;
	struct unix_sock* usock;
	int err;
	sk_soc_info("sock_soc_update_peer_proc clientfd=%d, newclientfd=%d", psock_rpmsg->clientfd, psock_rpmsg->newclientfd);
	ipcsocket = sock_get_ipcsocket_by_local_and_remote(psock_rpmsg->serversock, psock_rpmsg->clientsock);
	if(ipcsocket == NULL){
		sk_soc_info("sock_soc_update_peer_proc failed, clientfd=%d,clientsock=%x\n ", psock_rpmsg->clientfd, psock_rpmsg->clientsock);
		return -1;
	}	
	sk_soc_info("sock_soc_update_peer_proc sock=%x, newsock=%x", ipcsocket->remotesocket, psock_rpmsg->newsock);
	ipcsocket->remotefd = psock_rpmsg->newclientfd;
	ipcsocket->remotesocket = psock_rpmsg->newsock;
	ipcsocket->key = psock_rpmsg->newsock;
	usock = unix_sk(ipcsocket->proxysocket->sk);
	usock->rsock.remotefd = psock_rpmsg->newclientfd;
	usock->rsock.remotesocket = psock_rpmsg->newsock;
	sock_soc_rsp_proc(psock_rpmsg);//ͷconnectȴź
	return 0;
}

void sock_soc_sock_hold(struct sock *other)
{
	struct ipc_socket *psocket_ipc = NULL;
	struct sock_rpmsg sock_rpmsg = {0};
	sock_rpmsg.msg_type = MSG_TYPE_SOCK_HOLD;
	//dump_stack();	
	//ҷ˴socketϢ
	psocket_ipc = sock_soc_get_ipcsocket_by_proxysock(other);
	if(psocket_ipc == NULL){
		sk_soc_info("sock_soc_sock_hold failed, other->fd=%d\n ", other->sk_fd);
		return;
	}
	//socket fd
	sock_rpmsg.serverfd = psocket_ipc->remotefd;
	sock_rpmsg.serversock = psocket_ipc->remotesocket;
	sock_rpmsg.clientsock = psocket_ipc->localsocket;
	sk_soc_info("sock_soc_sock_hold serverfd=%d \n", sock_rpmsg.serverfd);
	sock_send_ipc_msg(&sock_rpmsg); 
}

int sock_soc_sock_hold_proc(struct sock_rpmsg* psock_rpmsg)
{
	struct socket* serversocket;	
	struct sock* other;
	
	serversocket = (struct socket*)psock_rpmsg->serversock;
	if(sock_soc_socket_is_valid_ipc(serversocket, psock_rpmsg->clientsock) == -1){ //serversockѹر
		sk_soc_info("sock_soc_socket_is_valid is false \n");
		return -1;
	}
	other = serversocket->sk;	
	
	sk_soc_info("sock_soc_sock_hold_proc serverfd=%d \n", psock_rpmsg->serverfd);
	sock_hold(other);
	return 0;
}

int sock_soc_sock_put(struct sock *other)
{
	struct sock_rpmsg sock_rpmsg = {0};
	int sockfd;
	struct ipc_socket *ipc_socket;
	sock_rpmsg.msg_type = MSG_TYPE_SOCK_PUT;
	//other->sk_socketѱΪnull
	if(!other->sk_socket){
		sockfd = other->sk_fd;
	}
	else{
		sockfd = sock_lookup_fd_by_sock(other);
	}
	
	//sock_rpmsg.serverfd = sock_lookup_serverfd_by_clientfd(sockfd);
	//ipc_socket = sock_lookup_server_by_clientfd(sockfd);
	ipc_socket = sock_soc_get_ipcsocket_by_proxysock(other);
	if(NULL == ipc_socket){
		sk_soc_info("sock_soc_sock_put get ipc_socket faild \n");
		return -1;
	}
	sock_rpmsg.serverfd = ipc_socket->remotefd;
	sock_rpmsg.serversock = ipc_socket->remotesocket;
	sock_rpmsg.clientsock = ipc_socket->localsocket;
	sk_soc_info("sock_soc_sock_put serverfd=%d\n", sock_rpmsg.serverfd);
	sock_send_ipc_msg(&sock_rpmsg);
	return 0;
}
EXPORT_SYMBOL_GPL(sock_soc_sock_put);

int sock_soc_sock_put_proc(struct sock_rpmsg* psock_rpmsg)
{	
	struct socket *other;
	int err;	
	//other = sock_get_local_socket(psock_rpmsg->serverfd, &err);
	other = (struct socket*)psock_rpmsg->serversock;
	if(sock_soc_socket_is_valid_ipc(other, psock_rpmsg->clientsock) == -1){ //serversockѹر
		sk_soc_info("sock_soc_socket_is_valid is false \n");		
		return -1;
	}
	sk_soc_info("sock_soc_sock_put_proc serversock=%x, serverfd=%d",other, psock_rpmsg->serverfd);
	if(other->sk){
		sock_put(other->sk);
	}
	
	return 0;
}

int sock_soc_msg_send(struct sock *sock, struct sock *other, struct msghdr *msg, size_t len)
{
	struct ipc_socket *psocket_ipc = NULL;
	//struct sock_rpmsg sock_rpmsg = {0};
	//sock_rpmsg.msg_type = MSG_TYPE_SEND_MSG;
	struct sock_rpmsg* psock_rpmsg =NULL;
	
	psock_rpmsg = sock_create_rpmsg(len);
	if(psock_rpmsg == NULL){
		sk_soc_err("sock_soc_msg_send failed, sock_create_rpmsg failed\n ");
		return -1;
	}
	psock_rpmsg->msg_type = MSG_TYPE_SEND_MSG;
	//ͻsocket fd
	psock_rpmsg->clientfd = sock_lookup_fd_by_sock(sock);
	psock_rpmsg->clientsock = sock->sk_socket;
	//ҷ˴socketϢ
	psocket_ipc = sock_soc_get_ipcsocket_by_proxysock(other);
	if(psocket_ipc == NULL){
		sk_soc_info("sock_soc_msg_send failed, other->sk_fd=%d\n ", other->sk_fd);
		kfree(psock_rpmsg);
		return -1;
	}
	//socket fd
	psock_rpmsg->serverfd = psocket_ipc->remotefd;
	psock_rpmsg->serversock = psocket_ipc->remotesocket;
	psock_rpmsg->clientsock = psocket_ipc->localsocket;
	
	psock_rpmsg->msg_flags = msg->msg_flags;
	//͵
	memcpy_fromiovec(psock_rpmsg->data, msg->msg_iov, len);
	psock_rpmsg->data_len = len;	
	//strcpy(sock_rpmsg.data, "test");
	//sock_rpmsg.data_len = 4;
	//socketַ
	if(msg->msg_namelen > 0){
		memcpy(&psock_rpmsg->sockaddr, (struct sockaddr_un*)msg->msg_name, msg->msg_namelen);
	}
	psock_rpmsg->addr_len = msg->msg_namelen;
	sk_soc_info("sock_soc_msg_send clientfd=%d, serverfd=%d, data=%s, len=%d\n", psock_rpmsg->clientfd, psock_rpmsg->serverfd, psock_rpmsg->data, len);
	sock_send_ipc_msg_data(psock_rpmsg);
	return 0;
}

int sock_soc_msg_send_proc(struct sock_rpmsg* psock_rpmsg)
{
	struct ipc_socket* clientsocket;
	struct socket* serversocket;
	struct sock* sock;
	struct sock* other;
	char __user * p,q;
	mm_segment_t old_fs;
	struct msghdr msg;
	unsigned int  msg_flags;
	struct iovec iov;
	int err;
	
	//serversocket = sock_get_local_socket(psock_rpmsg->serverfd, &err);
	serversocket = (struct socket*)psock_rpmsg->serversock;
	if(sock_soc_socket_is_valid_ipc(serversocket, psock_rpmsg->clientsock) == -1){ //serversockѹر
		sk_soc_info("sock_soc_socket_is_valid is false \n");
		return -1;
	}
	other = serversocket->sk;

	clientsocket = sock_get_ipcsocket_by_local_and_remote(psock_rpmsg->serversock, psock_rpmsg->clientsock);
	if(clientsocket == NULL){
		sk_soc_info("sock_soc_msg_send_proc failed, serversock=%x,clientsock=%x\n ", psock_rpmsg->serversock, psock_rpmsg->clientsock);
		return -1;
	}
	sk_soc_info("sock_soc_msg_send_proc, serversock=%x,clientsock=%x\n ", psock_rpmsg->serversock, psock_rpmsg->clientsock);
	sock = clientsocket->sock;
	
	sk_soc_info("sock_soc_msg_send_proc clientfd=%d, serverfd=%d, len=%d", psock_rpmsg->clientfd, psock_rpmsg->serverfd, psock_rpmsg->data_len);
	
	old_fs = get_fs();
	set_fs(KERNEL_DS);	
	if(psock_rpmsg->addr_len > 0){
		p = (__force struct sockaddr_un __user *)(&psock_rpmsg->sockaddr);	
	}
	q = (__force const char __user *)(psock_rpmsg->data);
	//sys_sendto(psock_rpmsg->serverfd, psock_rpmsg->data, psock_rpmsg->data_len, psock_rpmsg->msg_flags, NULL, 0);
	if(psock_rpmsg->addr_len > 0){
		sys_sendto(clientsocket->sockfd, psock_rpmsg->data, psock_rpmsg->data_len, psock_rpmsg->msg_flags, &psock_rpmsg->sockaddr, psock_rpmsg->addr_len);
	}else{
		sys_sendto(clientsocket->sockfd, psock_rpmsg->data, psock_rpmsg->data_len, psock_rpmsg->msg_flags, NULL, 0);		
	}		
	set_fs(old_fs);
	//other->sk_data_ready(other, psock_rpmsg->data_len);
	//sock_put(other);
	return 0;
}

long sock_wait_for_peer(struct sock *sock, struct sock *other, long timeo)
{
	struct ipc_socket * ipcsocket;	
	struct sock_rpmsg sock_rpmsg = {0};
	sock_rpmsg.clientfd = sock_lookup_fd_by_sock(sock);
	sock_rpmsg.clientsock = sock->sk_socket;
	ipcsocket = sock_soc_get_ipcsocket_by_proxysock(other);
	if(ipcsocket == NULL){
		sk_soc_info("sock_wait_for_peer failed, other->fd=%d\n ", other->sk_fd);
		return -1;
	}
	sock_rpmsg.serverfd = ipcsocket->remotefd;
	sock_rpmsg.serversock = ipcsocket->remotesocket;
	sock_rpmsg.msg_type = MSG_TYPE_WAIT_FOR_PEER;
	sock_rpmsg.timeo = timeo;
	sk_soc_info("sock_wait_for_peer clientfd=%d, serverfd=%d", sock_rpmsg.clientfd, sock_rpmsg.serverfd);
	sock_send_ipc_msg(&sock_rpmsg);
	return 0;
}

static int sock_wait_for_peer_thread(void *argv)
{
	struct peer_sock_info* peer_info = (struct peer_sock_info*)argv;
	sk_soc_info("sock_wait_for_peer_thread enter");
	unix_wait_for_peer_proxy(peer_info->peersock, peer_info->timeo);
	return 0;
}

int sock_wait_for_peer_proc(struct sock_rpmsg* psock_rpmsg)
{
	struct ipc_socket *socket_ipc;
	struct task_struct *th = NULL;
	struct socket* clientsocket;
	struct socket* serversocket;
	int err;
	socket_ipc = sock_get_ipcsocket_by_local_and_remote(psock_rpmsg->serversock, psock_rpmsg->clientsock); 
	if(socket_ipc == NULL){
		sk_soc_info("sock_wait_for_peer_proc failed, clientfd=%d,clientsock=%x\n ", psock_rpmsg->clientfd, psock_rpmsg->clientsock);
		return -1;
	}
	struct peer_sock_info* peer_info = (struct peer_sock_info *)kzalloc(sizeof(struct peer_sock_info), GFP_ATOMIC);
	if(NULL == peer_info){
		sk_soc_err("kzalloc failed!");
		return -1;
	}
	peer_info->timeo = psock_rpmsg->timeo;
	clientsocket= sock_get_local_socket(socket_ipc->sockfd, &err);
	if(NULL == clientsocket){
		kfree(peer_info);
		sk_soc_err("clientsocket is NULL!");
		return -1;
	}
	//clientsocket = (struct socket*)psock_rpmsg->clientsock;
	peer_info->sock = clientsocket->sk;
	//serversocket = sock_get_local_socket(psock_rpmsg->serverfd, &err);
	serversocket = (struct socket*)psock_rpmsg->serversock;
	if(sock_soc_socket_is_valid_other(serversocket) == -1){ //serversockѹر
		sk_soc_info("sock_soc_socket_is_valid is false \n");
		return -1;
	}
	peer_info->peersock = serversocket->sk;
	sk_soc_info("sock_wait_for_peer_proc clientfd=%d, serverfd=%d", socket_ipc->sockfd, psock_rpmsg->serverfd);
	th = kthread_run(sock_wait_for_peer_thread, peer_info, "wait_for_peer");
	    if (IS_ERR(th)) {
	        sk_soc_err("Unable to start receive thread.");
	        return PTR_ERR(th);
	    }
	return 0;
}

int sock_soc_unix_peer_clear(struct sock *sk)
{
	struct sock_rpmsg sock_rpmsg = {0};
	struct ipc_socket *socket_ipc;
	sk_soc_info("sock_soc_unix_peer_clear enter\n");
	//socket_ipc = sock_soc_get_ipcsocket_by_fd(psock_rpmsg->clientfd, psock_rpmsg->clientsock);
	sock_rpmsg.clientfd = sock_lookup_fd_by_sock(sk);
	sock_rpmsg.clientsock = sk->sk_socket;
	sock_rpmsg.msg_type = MSG_TYPE_PEER_CLEAR;
	sock_send_ipc_msg(&sock_rpmsg);
	return 0;
}

int sock_soc_unix_peer_clear_proc(struct sock_rpmsg* psock_rpmsg)
{
	struct socket* socket;
	struct sock* sock;
	int err;
	sk_soc_info("sock_soc_unix_peer_clear_proc enter");
	//socket = sock_get_local_socket(psock_rpmsg->serverfd, &err);
	socket = (struct socket*)psock_rpmsg->serversock;
	if(sock_soc_socket_is_valid_other(socket) == -1){ //serversockѹر
		sk_soc_info("sock_soc_socket_is_valid is false \n");
		return -1;
	}
	sock = socket->sk;
	if(sock == NULL){
		sk_soc_info("sock_soc_unix_peer_clear_proc sock is NULL");		
		return -1;
	}
	unix_sk(sock)->peer = NULL;
	return 0;
}

int sock_soc_wake_up_interruptible_poll(struct sock *sk)
{
	struct sock_rpmsg sock_rpmsg = {0};	
	
	sock_rpmsg.clientfd = sock_lookup_fd_by_sock(sk);	
	sock_rpmsg.clientsock = sk->sk_socket;
	sock_rpmsg.msg_type = MSG_TYPE_WAKE_UP_INTERRUPTIBLE_POLL;
	sk_soc_info("sock_soc_wake_up_interruptible_poll clientfd=%d", sock_rpmsg.clientfd);
	sock_send_ipc_msg(&sock_rpmsg);
	return 0;
}

int sock_soc_wake_up_interruptible_poll_proc(struct sock_rpmsg* psock_rpmsg)
{
	struct socket* socket;
	struct sock* sock;
	int err;
	//serverfd = sock_lookup_serverfd_by_clientfd(psock_rpmsg->clientfd);
	//socket = sock_get_local_socket(serverfd, &err);
	socket = (struct socket*)psock_rpmsg->serversock;
	if(sock_soc_socket_is_valid_other(socket) == -1){ //serversockѹر
		sk_soc_info("sock_soc_socket_is_valid is false \n");
		return -1;
	}
	sock = socket->sk;
	if(sock == NULL){
		sk_soc_info("sock_soc_wake_up_interruptible_poll_proc sock is NULL");		
		return -1;
	}
	
	wake_up_interruptible_poll(sk_sleep(sock),
				   POLLOUT |
				   POLLWRNORM |
				   POLLWRBAND);
	return 0;
}

int sock_soc_sk_data_ready(struct sock *sk)
{
	struct sock_rpmsg sock_rpmsg = {0};
	struct ipc_socket * ipcsocket;
	ipcsocket = sock_soc_get_ipcsocket_by_proxysock(sk);
	if(ipcsocket == NULL){
		sk_soc_info("sock_soc_sk_data_ready failed, sk->fd=%d\n ", sk->sk_fd);
		return -1;
	}
	sock_rpmsg.serverfd = ipcsocket->remotefd;
	sock_rpmsg.serversock = ipcsocket->remotesocket;
	sock_rpmsg.clientsock = ipcsocket->localsocket;
	sk_soc_info("sock_soc_sk_data_ready serverfd=%d", sock_rpmsg.serverfd);
	sock_rpmsg.msg_type = MSG_TYPE_DATA_READY;
	sock_send_ipc_msg(&sock_rpmsg);
	return 0;
}

int sock_soc_sk_data_ready_proc(struct sock_rpmsg* psock_rpmsg)
{
	struct socket* socket;
	struct sock* sock;
	int err;
	//socket = sock_get_local_socket(psock_rpmsg->serverfd, &err);
	socket = (struct socket*)psock_rpmsg->serversock;
	if(sock_soc_socket_is_valid_ipc(socket, psock_rpmsg->clientsock) == -1){ //serversockѹر
		sk_soc_info("sock_soc_socket_is_valid is false \n");
		return -1;
	}
	sk_soc_info("sock_soc_sk_data_ready_proc serverfd=%d", psock_rpmsg->serverfd);
	sock = socket->sk;
	if(sock == NULL){
		sk_soc_info("sock_soc_sk_data_ready_proc sock is NULL");		
		return -1;
	}
	sock->sk_data_ready(sock, 1);
	return 0;
}

int sock_sync_ipc_socket(struct sock_elem_flag* elem_flag, struct sock *other)
{
	struct sock_rpmsg sock_rpmsg = {0};
	struct ipc_socket * ipcsocket;
	sock_rpmsg.msg_type = MSG_TYPE_SYNC_IPC_SOCKET;
	memcpy(&sock_rpmsg.elem, elem_flag, sizeof(struct sock_elem_flag));
	//memcpy(&sock_rpmsg.sock, other, sizeof(struct sock));
	ipcsocket = sock_soc_get_ipcsocket_by_proxysock(other);
	if(ipcsocket == NULL){
		sk_soc_info("sock_sync_ipc_socket failed, other->fd=%d\n ", other->sk_fd);
		return -1;
	}
	sock_rpmsg.serverfd = ipcsocket->remotefd;
	sock_rpmsg.serversock = ipcsocket->remotesocket;
	sock_rpmsg.clientsock = ipcsocket->localsocket;
	sk_soc_info("sock_sync_ipc_socket serverfd=%d", sock_rpmsg.serverfd);
	sock_send_ipc_msg(&sock_rpmsg);
	
	return 0;
}

int sock_sync_ipc_socket_proc(struct sock_rpmsg* psock_rpmsg)
{
	struct socket* socket;
	struct sock *other;
	int err;
	//socket = sock_get_local_socket(psock_rpmsg->serverfd, &err);
	socket = (struct socket*)psock_rpmsg->serversock;
	if(NULL == socket){
		return -1;
	}
	if(sock_soc_socket_is_valid_ipc(socket, psock_rpmsg->clientsock) == -1){ //serversockѹر
		sk_soc_info("sock_soc_socket_is_valid is false \n");
		return -1;
	}
	other = socket->sk;
	if(other == NULL){
		sk_soc_info("sock_sync_ipc_socket_proc other is NULL");		
		return -1;
	}
	sk_soc_info("sock_sync_ipc_socket_proc serverfd=%d", psock_rpmsg->serverfd);
	if(psock_rpmsg->elem.sk_err){
		//other->sk_err = psock_rpmsg->sock.sk_err;		
	}

	if(psock_rpmsg->elem.sk_error_report){
		other->sk_error_report(other);		
	}
	
	return 0;
}

int sock_soc_get_state(struct sock_elem_flag* elem_flag, struct sock *other)
{
	struct sock_rpmsg sock_rpmsg = {0};
	struct ipc_socket * ipcsocket;
	struct sock_rpmsg_info *psock_rpmsg_info;
	sock_rpmsg.msg_type = MSG_TYPE_GET_SOCKET_STATES;
	memcpy(&sock_rpmsg.elem, elem_flag, sizeof(struct sock_elem_flag));	
	ipcsocket = sock_soc_get_ipcsocket_by_proxysock(other);
	if(ipcsocket == NULL){
		sk_soc_info("sock_soc_get_state failed, other->fd=%d\n ", other->sk_fd);
		return -1;
	}
	sock_rpmsg.serverfd = ipcsocket->remotefd;
	sock_rpmsg.serversock = ipcsocket->remotesocket;
	sock_rpmsg.clientsock = ipcsocket->localsocket;
	sock_rpmsg.key = other;
	sk_soc_info("sock_soc_get_state serverfd=%d", sock_rpmsg.serverfd);
	
	psock_rpmsg_info = sock_soc_create_rpmsg_info(sock_rpmsg.key);
	if(NULL == psock_rpmsg_info){
		sk_soc_info("sock_soc_get_state create rpmsg faild \n");
		return -1;
	}
	sock_send_ipc_msg(&sock_rpmsg);
	down_timeout(&psock_rpmsg_info->sock_sem, msecs_to_jiffies(g_sock_timeout));
	
	if(psock_rpmsg_info->sock_rpmsg.elem.skstate){
		other->sk_state = psock_rpmsg_info->sock_rpmsg.elem.skstate;
		sk_soc_info("sk_state=%d", other->sk_state);
	}

	if(psock_rpmsg_info->sock_rpmsg.elem.skshutdown){
		other->sk_shutdown = psock_rpmsg_info->sock_rpmsg.elem.skshutdown;
		sk_soc_info("sk_shutdown=%d", other->sk_shutdown);
	}
	sock_soc_del_rpmsg_info(sock_rpmsg.key);
	sk_soc_info("sock_soc_get_state end");
	return 0;
}

int sock_soc_get_state_proc(struct sock_rpmsg* psock_rpmsg)
{
	struct socket* socket;
	struct sock *other;
	struct sock_rpmsg sock_rpmsg = {0};
	int err;	
	sock_rpmsg.msg_type = MSG_TYPE_GET_SOCKET_STATES_RESULT;
	sock_rpmsg.key = psock_rpmsg->key;
	sk_soc_info("sock_soc_get_state_proc serverfd=%d", psock_rpmsg->serverfd);
	//socket = sock_get_local_socket(psock_rpmsg->serverfd, &err);
	socket = (struct socket*)psock_rpmsg->serversock;
	if(sock_soc_socket_is_valid_ipc(socket, psock_rpmsg->clientsock) == -1){ //serversockѹر
		sk_soc_info("sock_soc_socket_is_valid is false \n");
		sock_send_ipc_msg(&sock_rpmsg);
		return -1;
	}
	if(NULL != socket){
		sk_soc_info("sock_soc_get_state_proc sucess");
		other = socket->sk;
		if(other == NULL){
			sk_soc_info("sock_soc_get_state_proc other is NULL");
			sock_send_ipc_msg(&sock_rpmsg);
			return -1;
		}
		if(psock_rpmsg->elem.skstate){
			sock_rpmsg.elem.skstate = other->sk_state;
			sk_soc_info("sk_state=%d", other->sk_state);
		}

		if(psock_rpmsg->elem.skshutdown){
			sock_rpmsg.elem.skshutdown = other->sk_shutdown;
			sk_soc_info("sk_shutdown=%d", other->sk_shutdown);
		}	
	}
	else
	{
		sk_soc_info("sock_soc_get_state_proc failed");
	}	
	sock_send_ipc_msg(&sock_rpmsg);	
	return 0;
}

int sock_soc_get_state_result_proc(struct sock_rpmsg* psock_rpmsg)
{
	sk_soc_info("sock_soc_find_other_result_proc");
	sock_soc_rsp_proc(psock_rpmsg);
	return 0;
}

int sock_soc_test_flags(struct sock *other, enum sock_flags flag)
{
	struct sock_rpmsg sock_rpmsg = {0};
	struct ipc_socket * ipcsocket;
	struct sock_rpmsg_info *psock_rpmsg_info;
	int result;
	sock_rpmsg.msg_type = MSG_TYPE_TEST_FLAG;	
	ipcsocket = sock_soc_get_ipcsocket_by_proxysock(other);
	if(ipcsocket == NULL){
		sk_soc_info("sock_soc_test_flags failed, other->fd=%d\n ", other->sk_fd);
		return -1;
	}
	sock_rpmsg.serverfd = ipcsocket->remotefd;
	sock_rpmsg.serversock = ipcsocket->remotesocket;
	sock_rpmsg.clientsock = ipcsocket->localsocket;
	//sock_rpmsg.flag = flag;
	sock_rpmsg.flag = convert_to_cap_sock_flag(flag);
	if(sock_rpmsg.flag == -1)
	{
		sk_soc_info("cap do not support flag=%s ", flag_value[flag]);
		return 0;
	}
	sock_rpmsg.key = other;
	sk_soc_info("sock_soc_test_flags serverfd=%d", sock_rpmsg.serverfd);
	
	psock_rpmsg_info = sock_soc_create_rpmsg_info(sock_rpmsg.key);
	if(NULL == psock_rpmsg_info){
		sk_soc_info("sock_soc_get_state create rpmsg faild \n");
		return -1;
	}
	sock_send_ipc_msg(&sock_rpmsg);
	down_timeout(&psock_rpmsg_info->sock_sem, msecs_to_jiffies(g_sock_timeout));
	sk_soc_info("sock_soc_test_flags flag=%s, result=%d\n",flag_value[flag], g_sock_rpmsg.result);
	result = psock_rpmsg_info->sock_rpmsg.result;
	sock_soc_del_rpmsg_info(sock_rpmsg.key);
	return result;	
}
EXPORT_SYMBOL_GPL(sock_soc_test_flags);

int sock_soc_test_flags_proc(struct sock_rpmsg* psock_rpmsg)
{
	struct sock_rpmsg sock_rpmsg = {0};
	struct socket* socket;
	struct sock *other;
	int err;
	//socket = sock_get_local_socket(psock_rpmsg->serverfd, &err);
	sock_rpmsg.msg_type = MSG_TYPE_TEST_FLAG_RESULT;
	sock_rpmsg.key = psock_rpmsg->key;
	socket = (struct socket*)psock_rpmsg->serversock;
	if(sock_soc_socket_is_valid_ipc(socket, psock_rpmsg->clientsock) == -1){ //serversockѹر
		sk_soc_info("sock_soc_socket_is_valid is false \n");
		sock_send_ipc_msg(&sock_rpmsg);
		return -1;
	}
	if(NULL == socket){
		sk_soc_info("sock_soc_test_flags_proc failed, flag=%s", flag_value[psock_rpmsg->flag]);
		sock_send_ipc_msg(&sock_rpmsg);
		return -1;
	}
	other = socket->sk;
	if(other == NULL){
		sock_send_ipc_msg(&sock_rpmsg);
		return -1;
	}	
	sock_rpmsg.result = test_bit(psock_rpmsg->flag, &other->sk_flags);	
	sk_soc_info("sock_soc_test_flags_proc flag=%s, result=%d", flag_value[psock_rpmsg->flag], sock_rpmsg.result);
	sock_send_ipc_msg(&sock_rpmsg);
	return 0;
}

int sock_soc_test_flags_result_proc(struct sock_rpmsg* psock_rpmsg)
{
	sk_soc_info("sock_soc_test_flags_result_proc enter");
	sock_soc_rsp_proc(psock_rpmsg);
	
	return 0;
}

void sock_soc_add_skb_queue_tail(struct sock *sock, struct sock *other)
{
	struct ipc_socket * ipcsocket;	
	struct sock_rpmsg sock_rpmsg = {0};
	sock_rpmsg.clientfd = sock_lookup_fd_by_sock(sock);
	sock_rpmsg.clientsock = sock->sk_socket;
	ipcsocket = sock_soc_get_ipcsocket_by_proxysock(other);
	if(ipcsocket == NULL){
		sk_soc_info("sock_soc_add_skb_queue_tail failed, other->fd=%d\n ", other->sk_fd);
		return;
	}
	sock_rpmsg.serverfd = ipcsocket->remotefd;
	sock_rpmsg.serversock = ipcsocket->remotesocket;
	sock_rpmsg.clientsock = ipcsocket->localsocket;
	sock_rpmsg.msg_type = MSG_TYPE_ADD_SKB_QUEUE_TAIL;
	sk_soc_info("sock_soc_add_skb_queue_tail clientfd=%d, serverfd=%d", sock_rpmsg.clientfd, sock_rpmsg.serverfd);
	sock_send_ipc_msg(&sock_rpmsg);
}

void sock_soc_add_skb_queue_tail_proc(struct sock_rpmsg* psock_rpmsg)
{
	struct ipc_socket * ipcsocket;	
	struct socket* other;
	struct sk_buff *skb = NULL;
	int err;
	//clientfdҵشsocket
	ipcsocket = sock_get_ipcsocket_by_local_and_remote(psock_rpmsg->serversock, psock_rpmsg->clientsock);
	if(ipcsocket == NULL){
		sk_soc_info("sock_soc_add_skb_queue_tail_proc failed, clientfd=%d,clientsock=%x\n ", psock_rpmsg->clientfd, psock_rpmsg->clientsock);
		return;
	}
	//other = sockfd_lookup(psock_rpmsg->serverfd, &err);
	other = (struct socket*)psock_rpmsg->serversock;
	if(sock_soc_socket_is_valid_ipc(other, psock_rpmsg->clientsock) == -1){ //serversockѹر
		sk_soc_info("sock_soc_socket_is_valid is false \n");
		return;
	}
	sk_soc_info("sock_soc_add_skb_queue_tail clientfd=%d, serverfd=%d", psock_rpmsg->clientfd, psock_rpmsg->serverfd);
	if((NULL != ipcsocket) && (NULL != other)){
		/* Allocate skb for sending to listening sock */
		skb = sock_wmalloc(ipcsocket->sock, 1, 0, GFP_KERNEL);
		if (skb == NULL)
		{
			return; 
		}

		__skb_queue_tail(&other->sk->sk_receive_queue, skb);
		sk_soc_info("sock_soc_add_skb_queue_tail sucess! \n");
	}	
}

void sock_soc_stream_connect(struct sock *sock, struct sock *other, struct sockaddr_un *sunname, int len,int flags)
{
	struct ipc_socket * ipcsocket;	
	struct sock_rpmsg sock_rpmsg = {0};
	struct sock_rpmsg_info *psock_rpmsg_info;
	if(len > 0){
		memcpy(&sock_rpmsg.sockaddr, sunname, sizeof(struct sockaddr_un));
		sk_soc_info("sock_soc_stream_connect sockaddr=%s", sock_rpmsg.sockaddr.sun_path);
		sock_rpmsg.addr_len = len;
	}
	sock_rpmsg.flags = flags;
	sock_rpmsg.clientfd = sock_lookup_fd_by_sock(sock);
	sock_rpmsg.clientsock = sock->sk_socket;
	ipcsocket = sock_soc_get_ipcsocket_by_proxysock(other);
	if(ipcsocket == NULL){
		sk_soc_info("sock_soc_stream_connect failed, other->fd=%d\n ", other->sk_fd);
		return;
	}
	sock_rpmsg.serverfd = ipcsocket->remotefd;
	sock_rpmsg.serversock = ipcsocket->remotesocket;
	sock_rpmsg.msg_type = MSG_TYPE_STREAM_CONNECT;
	sock_rpmsg.key = sock_rpmsg.clientsock;
	psock_rpmsg_info = sock_soc_create_rpmsg_info(sock_rpmsg.key);
	if(NULL == psock_rpmsg_info){
		sk_soc_info("sock_soc_stream_connect create rpmsg faild \n");
		return;
	}
	sock_send_ipc_msg(&sock_rpmsg);
	//Ҫȵյupdate_peerܷ
	down_timeout(&psock_rpmsg_info->sock_sem, msecs_to_jiffies(g_sock_timeout));
	sk_soc_info("sock_soc_stream_connect clientfd=%d, serverfd=%d\n", sock_rpmsg.clientfd, sock_rpmsg.serverfd);
	sock_soc_del_rpmsg_info(sock_rpmsg.key);
}

void sock_soc_stream_connect_proc(struct sock_rpmsg* psock_rpmsg)
{
	struct ipc_socket * ipcsocket;	
	struct socket* other;
	struct sk_buff *skb = NULL;
	struct sock *newsk = NULL;
	int result = -1;
	int err;
	//clientfdҵشsocket
	ipcsocket = sock_get_ipcsocket_by_local_and_remote(psock_rpmsg->serversock, psock_rpmsg->clientsock);
	if(ipcsocket == NULL){
		sk_soc_info("sock_soc_stream_connect_proc failed, clientfd=%d,clientsock=%x\n ", psock_rpmsg->clientfd, psock_rpmsg->clientsock);
		return;
	}
	//other = sockfd_lookup(psock_rpmsg->serverfd, &err);
	other = (struct socket*)psock_rpmsg->serversock;
	
	sk_soc_info("sock_soc_stream_connect_proc clientfd=%d, serverfd=%d\n", psock_rpmsg->clientfd, psock_rpmsg->serverfd);
	if((NULL != ipcsocket) && (NULL != other)){	
		sk_soc_info("sockaddr=%s", psock_rpmsg->sockaddr.sun_path);
		result = unix_stream_connect_proxy(ipcsocket->socket, (struct sockaddr *)&psock_rpmsg->sockaddr, psock_rpmsg->addr_len, psock_rpmsg->flags);
		if(0 == result){
			sk_soc_info("sock_soc_stream_connect_proc sucess! ");
		}else{
			sk_soc_info("sock_soc_stream_connect_proc failed! ");
		}	    
	}	
}

void sock_soc_add_wait_queue(struct sock *sock, struct sock *other)
{
	struct ipc_socket * ipcsocket;	
	struct sock_rpmsg sock_rpmsg = {0};
	sock_rpmsg.clientfd = sock_lookup_fd_by_sock(sock);
	sock_rpmsg.clientsock = sock->sk_socket;
	ipcsocket = sock_soc_get_ipcsocket_by_proxysock(other);
	if(ipcsocket == NULL){
		sk_soc_info("sock_soc_add_wait_queue failed, other->fd=%d\n ", other->sk_fd);
		return;
	}
	sock_rpmsg.serverfd = ipcsocket->remotefd;
	sock_rpmsg.serversock = ipcsocket->remotesocket;
	sock_rpmsg.msg_type = MSG_TYPE_ADD_WAIT_QUEUE;
	sk_soc_info("sock_soc_add_wait_queue enter");
	sock_send_ipc_msg(&sock_rpmsg);
}

void sock_soc_add_wait_queue_proc(struct sock_rpmsg* psock_rpmsg)
{
	struct ipc_socket * ipcsocket;	
	struct socket* socket;
	int err;
	//clientfdҵشsocket
	ipcsocket = sock_get_ipcsocket_by_local_and_remote(psock_rpmsg->serversock, psock_rpmsg->clientsock);
	if(ipcsocket == NULL){
		sk_soc_info("sock_soc_add_wait_queue_proc failed, clientfd=%d,clientsock=%x\n ", psock_rpmsg->clientfd, psock_rpmsg->clientsock);
		return;
	}
	//socket = sockfd_lookup(psock_rpmsg->serverfd, &err);
	socket = (struct socket*)psock_rpmsg->serversock;
	
	sk_soc_info("sock_soc_add_wait_queue_proc clientfd=%d, serverfd=%d", psock_rpmsg->clientfd, psock_rpmsg->serverfd);
	if((NULL != ipcsocket) && (NULL != socket)){
		unix_dgram_peer_wake_connect_proxy(ipcsocket->sock, socket->sk);
	}	
}

void sock_soc_remove_wait_queue(struct sock *sock, struct sock *other)
{
	struct ipc_socket * ipcsocket;	
	struct sock_rpmsg sock_rpmsg = {0};
	sock_rpmsg.clientfd = sock_lookup_fd_by_sock(sock);
	sock_rpmsg.clientsock = sock->sk_socket;
	ipcsocket = sock_soc_get_ipcsocket_by_proxysock(other);
	if(ipcsocket == NULL){
		sk_soc_info("sock_soc_remove_wait_queue failed, other->fd=%d\n ", other->sk_fd);
		return;
	}
	sock_rpmsg.serverfd = ipcsocket->remotefd;
	sock_rpmsg.serversock = ipcsocket->remotesocket;
	sock_rpmsg.msg_type = MSG_TYPE_REMOVE_WAIT_QUEUE;
	sk_soc_info("sock_soc_remove_wait_queue clientfd=%d, serverfd=%d", sock_rpmsg.clientfd, sock_rpmsg.serverfd);
	sock_send_ipc_msg(&sock_rpmsg);
}

void sock_soc_remove_wait_queue_proc(struct sock_rpmsg* psock_rpmsg)
{
	struct ipc_socket * ipcsocket;	
	struct socket* socket;
	int err;
	//clientfdҵشsocket
	ipcsocket = sock_get_ipcsocket_by_local_and_remote(psock_rpmsg->serversock, psock_rpmsg->clientsock);
	if(ipcsocket == NULL){
		sk_soc_info("sock_soc_remove_wait_queue_proc failed, clientfd=%d,clientsock=%x\n ", psock_rpmsg->clientfd, psock_rpmsg->clientsock);
		return;
	}
	//socket = sockfd_lookup(psock_rpmsg->serverfd, &err);
	socket = (struct socket*)psock_rpmsg->serversock;
	
	sk_soc_info("sock_soc_remove_wait_queue_proc clientfd=%d, serverfd=%d", psock_rpmsg->clientfd, psock_rpmsg->serverfd);
	if((NULL != ipcsocket) && (NULL != socket)){
		unix_dgram_peer_wake_connect_proxy(ipcsocket->sock, socket->sk);
	}	
}

void sock_soc_notify_proxy_change_to_server(struct socket* proxysocket, struct socket* remoteproxysocket)
{
	struct sock_rpmsg sock_rpmsg = {0};
	sock_rpmsg.clientproxyfd = proxysocket->fd;
	sock_rpmsg.clientproxysock = proxysocket;
	sock_rpmsg.serverproxysock = remoteproxysocket;
	sock_rpmsg.msg_type = MSG_TYPE_NOTIFY_PROXY_CHANGE;
	sk_soc_info("sock_soc_notify_proxy_change_to_server clientproxysock=%x, serverproxysock=%x", sock_rpmsg.clientproxysock, sock_rpmsg.serverproxysock);
	sock_send_ipc_msg(&sock_rpmsg);
}

void sock_soc_notify_proxy_change_to_server_proc(struct sock_rpmsg* psock_rpmsg)
{
	struct ipc_socket* ipc_socket;
	struct socket* serverproxysock;
	serverproxysock = (struct socket*)psock_rpmsg->serverproxysock;
	ipc_socket = sock_soc_get_ipcsocket_by_proxysock(serverproxysock->sk);
	sk_soc_info("sock_soc_notify_proxy_change_to_server_proc serverproxysock=%x", psock_rpmsg->serverproxysock);
	if(NULL != ipc_socket){
		sk_soc_info("sock_soc_notify_proxy_change_to_server_proc remoteproxysocket=%x", psock_rpmsg->clientproxysock);
		ipc_socket->remoteproxyfd = psock_rpmsg->clientproxyfd;
		ipc_socket->remoteproxysocket = psock_rpmsg->clientproxysock;
		serverproxysock = (struct socket*)psock_rpmsg->serverproxysock;
		usock_update_remote_proxy_socket(serverproxysock->sk, psock_rpmsg->clientproxysock);
	}else{
		sk_soc_info("sock_soc_notify_proxy_change_to_server_proc ipc_socket is NULL");
	}
	
}

struct sock *sock_soc_find_other(struct socket * socket/*int fd*/,
				    struct sockaddr_un *sunname, int len,
				    int type, unsigned int hash, int *error)
{
	struct sock_rpmsg sock_rpmsg = {0};
	struct ipc_socket *psocket_ipc = NULL;
	struct sock_rpmsg_info *psock_rpmsg_info = NULL;
	int err;
	
	if(0 == socket_rpmsg_enable){
		return NULL;
	}	
		
	if(false == sock_soc_ipc_sunaddr_exist(sunname)){
		sk_soc_info("sock_soc_ipc_sunaddr_exist is false, sunname=%s", sunname->sun_path);
		return NULL; 
	}
	//struct socket *clisocket = sock_get_local_socket(fd, &err);
	memcpy(&sock_rpmsg.sockaddr, sunname, sizeof(struct sockaddr_un));
	sock_rpmsg.addr_len = len;
	sock_rpmsg.socktype = type;
	sock_rpmsg.hash = hash;
	sock_rpmsg.clientfd = socket->fd;
	sock_rpmsg.clientsock = socket;
	sock_rpmsg.key = socket;
	sock_rpmsg.msg_type = MSG_TYPE_FIND; 
	sock_rpmsg.isfistfind = 0;
	
	if(sock_soc_get_ipcsocket_by_localsocket(socket) == NULL){ //˵ǵһη
		sock_rpmsg.isfistfind = 1;
	}
	
	psock_rpmsg_info = sock_soc_create_rpmsg_info(sock_rpmsg.key);
	if(NULL == psock_rpmsg_info){
		sk_soc_info("sock_soc_find_other create rpmsg faild \n");
		return NULL;
	}
	sk_soc_info("sock_soc_find_other clientfd=%d, key=%x", sock_rpmsg.clientfd, sock_rpmsg.key);
	sock_send_ipc_msg(&sock_rpmsg);
	down_timeout(&psock_rpmsg_info->sock_sem, msecs_to_jiffies(g_sock_timeout));

	if(test_bit(SOCK_OP_RESULT, &psock_rpmsg_info->sock_rpmsg.flags)){		
		sk_soc_info("sock_soc_find_other sucess serverfd=%d", psock_rpmsg_info->sock_rpmsg.serverfd);
		psocket_ipc = psock_rpmsg_info->ipc_socket;
		sk_soc_info("sock_soc_find_other sockfd=%d, remotefd=%d", psocket_ipc->sockfd, psocket_ipc->remotefd);
		//socket->peer = psocket_ipc->socket;
		sock_soc_insert_socket_peer(socket, psocket_ipc->proxysocket);
		sock_soc_del_rpmsg_info(sock_rpmsg.key);
		sock_soc_notify_proxy_change_to_server(psocket_ipc->proxysocket, psocket_ipc->remoteproxysocket);
		//sock_soc_set_socket_info(psocket_ipc->socket, socket);
		return psocket_ipc->sock;
	} else {
		sk_soc_info("sock_soc_find_other failed, no matched socket");
	}
	sock_soc_del_rpmsg_info(sock_rpmsg.key);
	
	return NULL;
}

int sock_soc_find_other_proc(struct sock_rpmsg* psock_rpmsg)
{
	struct sock *u;
	struct ipc_socket *psocket_ipc = NULL;
	//struct net tmpnet={0};
	struct sock_rpmsg sock_rpmsg = {0};
	int serverfd;
	int error = -1;
	unsigned int hash;
	sk_soc_info("sock_soc_find_other_proc index=%d", psock_rpmsg->index);
	if((NULL == psock_rpmsg) || (psock_rpmsg->msg_type != MSG_TYPE_FIND)){
		sk_soc_info("sock_soc_find_other_proc failed, NULL == psock_rpmsg");
		return -1;
	}
	sock_rpmsg.msg_type = MSG_TYPE_FIND_RESULT; 
	sk_soc_info("sockaddr=%s,socktype=%d, hash=%d",psock_rpmsg->sockaddr.sun_path,psock_rpmsg->socktype,psock_rpmsg->hash);
	error = unix_mkname_proxy(&psock_rpmsg->sockaddr, psock_rpmsg->addr_len, &hash);
	if (error < 0){
		sk_soc_info("unix_mkname_proxy failed, error=%d", error);
	}
	sk_soc_info("unix_mkname_proxy sucess, hash=%d", hash);
	//init_net
	u = unix_find_other_proxy(&init_net, &psock_rpmsg->sockaddr, psock_rpmsg->addr_len, psock_rpmsg->socktype, hash, &error);
	if(u){		
		sk_soc_info("find_other_proc: u(%x)->sk_socket=%x\n", u, u->sk_socket);
		serverfd = sock_lookup_fd_by_sock(u);
		psock_rpmsg->serverfd = serverfd;
		psock_rpmsg->serversock = u->sk_socket;
		psocket_ipc = sock_create_proxy_socket_client(psock_rpmsg);
		if(NULL != psocket_ipc){
			psocket_ipc->localsocket = u->sk_socket;
			set_bit(SOCK_OP_RESULT, &sock_rpmsg.flags);
			//memcpy(&sock_rpmsg.sock, u, sizeof(struct sock));
			sock_rpmsg.serverfd = serverfd;
			sock_rpmsg.serversock = u->sk_socket;
			sock_rpmsg.socktype = u->sk_socket->type;		
			sock_rpmsg.serverproxyfd = psocket_ipc->sockfd;
			sock_rpmsg.serverproxysock = psocket_ipc->proxysocket;
			//u->sk_socket->peer = psocket_ipc->socket; //TODO:uܱͻ˲ң˴peerָsocket⣬Ϊܴڶsocket
			sock_soc_insert_socket_peer(u->sk_socket, psocket_ipc->proxysocket);
		}		
		
		sock_put(u); //0707
		sk_soc_info("sock_soc_find_other_proc success, socktype=%d, serverfd=%d, key=%x, index=%d\n", sock_rpmsg.socktype, sock_rpmsg.serverfd, psock_rpmsg->key, psock_rpmsg->index);
	}
	else
	{
		sk_soc_info("sock_soc_find_other_proc failed, error=%d", error);
	}
	sock_rpmsg.key = psock_rpmsg->key;
	sock_rpmsg.clientfd = psock_rpmsg->clientfd;
	sock_rpmsg.clientsock = psock_rpmsg->clientsock;
	sock_rpmsg.index = psock_rpmsg->index;
	sock_send_ipc_msg(&sock_rpmsg);

	return 0;
}

int sock_soc_find_other_result_proc(struct sock_rpmsg* psock_rpmsg)
{
	struct sock_rpmsg_info *psock_rpmsg_info = NULL;	
	
	psock_rpmsg_info = sock_soc_get_rpmsg_info(psock_rpmsg->key);
	if(NULL == psock_rpmsg_info){
		sk_soc_info("sock_soc_find_other_result_proc failed\n");
		return -1;
	}
	memset(&psock_rpmsg_info->sock_rpmsg, 0, sizeof(struct sock_rpmsg));
	memcpy(&psock_rpmsg_info->sock_rpmsg, psock_rpmsg, sizeof(struct sock_rpmsg));
	if(test_bit(SOCK_OP_RESULT, &psock_rpmsg->flags)){
		psock_rpmsg_info->ipc_socket = sock_create_proxy_socket_server(psock_rpmsg);
	}
	up(&psock_rpmsg_info->sock_sem);
	return 0;
}

static void delayed_release(struct work_struct *unused)
{
	//struct llist_node *node = llist_del_all(&delayed_release_list);
	struct release_socket *rsock, *t;
	sk_soc_info("delayed_release enter\n");
	spin_lock(&sock_release_table_lock);
	list_for_each_entry_safe(rsock, t, &delayed_release_list, list_node){
		//sock_soc_release_sock(rsock);
		sock_soc_release_peer(rsock);
		list_del(&rsock->list_node);
		kfree(rsock);
	}
	spin_unlock(&sock_release_table_lock);
}

static DECLARE_DELAYED_WORK(delayed_release_work, delayed_release);

void sock_soc_delay_release(struct socket *sock)
{
	struct release_socket* prelease_socket = NULL;
	struct unix_sock* usock;
	struct socket_info *p;
	struct hlist_node *pos,*n;
	if(!hlist_empty(&sock->peer)){
		spin_lock(&sock_release_peer_lock);
		hlist_for_each_entry_safe(p, pos, n, &sock->peer, hlist_node) {
			usock = unix_sk(p->proxysocket->sk);		    
			//жusockϢǷڴ
			if(0 != sock_is_valide_ipc_socket(usock->rsock.localsocket, usock->rsock.proxysocket, usock->rsock.remotesocket)){
				sk_soc_info("sock_soc_delay_release failed \n");
				continue;
			}
			prelease_socket = (struct release_socket *)kzalloc(sizeof(struct release_socket), GFP_ATOMIC);
			if(NULL == prelease_socket){
				sk_soc_info("kzalloc failed \n");
				continue;
			}
			memcpy(prelease_socket, &usock->rsock, sizeof(struct release_socket));			
			list_add(&prelease_socket->list_node, &delayed_release_list);			
			sock_soc_del_socket_info(p->proxysocket, sock);
		}
		
		//sock_put(sock->sk); /* It may now die */
		
		INIT_HLIST_HEAD(&sock->peer);
		spin_unlock(&sock_release_peer_lock);
		schedule_delayed_work(&delayed_release_work, 1);						
		//}			
	}
	
}

int sock_soc_release_peer(struct release_socket* rsock)
{	
	struct sock_rpmsg sock_rpmsg = {0};
	struct socket* peersocket;
	struct ipc_socket * ipcsocket;
	struct sock* proxy_sock = NULL;
	//ɾضӦĴsocket
	if(rsock->proxysocket){		
		sk_soc_info("sock_soc_release_peer close fd=%d \n", rsock->proxysocket->fd);
		sock_rpmsg.socktype = rsock->proxysocket->type;
		//ipcsocket = sock_soc_get_ipcsocket(rsock->sock->sk); //0706ݱشsocket,ҵӦsocket 
		if(NULL != rsock->remotesocket){
			sock_rpmsg.serverfd = rsock->remotefd;
			sock_rpmsg.serversock = rsock->remotesocket;
		}else{
			sk_soc_info("sock_soc_get_ipcsocket_by_proxysock failed \n");
		}
		
	    spin_lock(&sock_release_lock);
		
		rsock->proxysocket->sk->closed = 1;
		proxy_sock = rsock->proxysocket->sk->proxy_sock;
		sys_close(rsock->proxysocket->fd);
		if(proxy_sock != NULL){ //0706
			unix_release_sock_proxy(proxy_sock);
		}
		//ǰѾɾ
		//sock_del_ipc_socket_by_proxysocket(rsock->proxysocket);
		spin_unlock(&sock_release_lock);
	}
	sock_rpmsg.clientfd = rsock->localfd;
	sock_rpmsg.clientsock = rsock->localsocket;
	sock_rpmsg.serversock = rsock->remotesocket;
	sock_rpmsg.serverproxysock = rsock->remoteproxysocket;
	sock_rpmsg.msg_type = MSG_TYPE_RELEASE_PEER;
	sk_soc_info("sock_soc_release_peer clientfd=%d \n", sock_rpmsg.clientfd);
	sock_send_ipc_msg(&sock_rpmsg);
	return 0;
}

int sock_soc_release_peer_proc(struct sock_rpmsg* psock_rpmsg)
{
	struct ipc_socket * ipcsocket;	
	struct socket* proxysocket;
	struct socket* serversocket;
	struct sock* skpair;
	int			type;
	int closefd;
	int err;
	sk_soc_info("sock_soc_release_peer_proc clientfd=%d \n", psock_rpmsg->clientfd);
	//clientfdҵشsocket
	spin_lock(&sock_release_lock);
	//жsocketϢǷڴ
	if(0 != sock_is_valide_ipc_socket(psock_rpmsg->serversock, psock_rpmsg->serverproxysock, psock_rpmsg->clientsock)){
		sk_soc_info("sock_soc_release_peer_proc failed, sock_is_valide_ipc_socket=false \n");
		spin_unlock(&sock_release_lock);
		return -1;
	}
	
	ipcsocket = sock_get_ipcsocket_by_local_and_remote(psock_rpmsg->serversock, psock_rpmsg->clientsock);
	if(ipcsocket != NULL){
		type = psock_rpmsg->socktype;
		sk_soc_info("sock_soc_release_peer_proc close sockfd=%d, socktype=%d\n", ipcsocket->sockfd, type);
		//sock_soc_del_socket_info(ipcsocket->socket, ipcsocket->localsocket); //0706 socketsocketsбɾ
		
		serversocket = (struct socket*)psock_rpmsg->serversock;
		proxysocket = (struct socket*)ipcsocket->proxysocket;
		//ĿǰSOCK_STREAMSOCK_DGRAMͷ
		if(type == SOCK_STREAM){
			//sock_del_ipc_socket_by_socket(ipcsocket->proxysocket);
			skpair = serversocket->sk;
			unix_state_lock(skpair);
			/* No more writes */
			skpair->sk_shutdown = SHUTDOWN_MASK;
			//if (!skb_queue_empty(&sk->sk_receive_queue) || embrion)
			//	serversocket->sk_err = ECONNRESET;
			unix_state_unlock(skpair);
			skpair->sk_state_change(skpair);
			sk_wake_async(skpair, SOCK_WAKE_WAITD, POLL_HUP);
		}else if(type == SOCK_DGRAM){
			
			sock_soc_del_socket_peer(serversocket, proxysocket);	
			ipcsocket->sock->closed = 1;
			closefd = ipcsocket->sockfd;
			sock_soc_del_socket_info(ipcsocket->socket, ipcsocket->localsocket); //0706 socketsocketsбɾ
			
			sys_close(closefd);			
		}		
	}else{
		sk_soc_info("sock_soc_release_peer_proc failed, localsock=%x, remotesock=%x", psock_rpmsg->serversock, psock_rpmsg->clientsock);
		
	}
	spin_unlock(&sock_release_lock);
	return 0;
}

int sock_soc_dgram_disconnected(struct sock *sock, struct sock *other)
{
	struct ipc_socket *psocket_ipc = NULL;
	struct sock_rpmsg sock_rpmsg = {0};
	sock_rpmsg.msg_type = MSG_TYPE_DGRAM_DISCONNECTED;
	
	//ͻsocket fd
	sock_rpmsg.clientfd = sock_lookup_fd_by_sock(sock);
	sock_rpmsg.clientsock = sock->sk_socket;
	//ҷ˴socketϢ
	psocket_ipc = sock_soc_get_ipcsocket_by_proxysock(other);
	if(psocket_ipc == NULL){
		sk_soc_info("sock_soc_remove_wait_queue failed, other->fd=%d\n ", other->sk_fd);
		return -1;
	}
	//socket fd
	sock_rpmsg.serverfd = psocket_ipc->remotefd;
	sock_rpmsg.serversock = psocket_ipc->remotesocket;
	sk_soc_info("sock_soc_dgram_disconnected clientfd=%d, serverfd=%d \n", sock_rpmsg.clientfd, sock_rpmsg.serverfd);
	sock_send_ipc_msg(&sock_rpmsg);
	return 0;
}

int sock_soc_dgram_disconnected_proc(struct sock_rpmsg* psock_rpmsg)
{
	struct ipc_socket* clientsocket;
	struct socket* serversocket;
	struct sock* sock;
	struct sock* other;
	
	serversocket = (struct socket*)psock_rpmsg->serversock;
	if(sock_soc_socket_is_valid_ipc(serversocket, psock_rpmsg->clientsock) == -1){ //serversockѹر
		sk_soc_info("sock_soc_socket_is_valid is false \n");
		return -1;
	}
	other = serversocket->sk;

	clientsocket = sock_get_ipcsocket_by_local_and_remote(psock_rpmsg->serversock, psock_rpmsg->clientsock);
	if(clientsocket == NULL){
		sk_soc_info("sock_soc_dgram_disconnected_proc failed, clientfd=%d,clientsock=%x\n ", psock_rpmsg->clientfd, psock_rpmsg->clientsock);
		return -1;
	}
	sock = clientsocket->sock;
	
	sk_soc_info("sock_soc_dgram_disconnected_proc clientfd=%d, serverfd=%d \n", psock_rpmsg->clientfd, psock_rpmsg->serverfd);
	unix_dgram_disconnected_proxy(sock, other);
	return 0;
}

void sock_soc_state_lock(struct sock *other)
{
	struct ipc_socket *psocket_ipc = NULL;
	struct sock_rpmsg sock_rpmsg = {0};
	sock_rpmsg.msg_type = MSG_TYPE_STATE_LOCK;
		
	//ҷ˴socketϢ
	psocket_ipc = sock_soc_get_ipcsocket_by_proxysock(other);
	if(psocket_ipc == NULL){
		sk_soc_info("sock_soc_state_lock failed, other->fd=%d\n ", other->sk_fd);
		return;
	}
	//socket fd
	sock_rpmsg.serverfd = psocket_ipc->remotefd;
	sock_rpmsg.serversock = psocket_ipc->remotesocket;
	sock_rpmsg.clientsock = psocket_ipc->localsocket;
	sk_soc_info("sock_soc_state_lock serverfd=%d \n", sock_rpmsg.serverfd);
	sock_send_ipc_msg(&sock_rpmsg); 
}

int sock_soc_state_lock_proc(struct sock_rpmsg* psock_rpmsg)
{	
	struct socket* serversocket;	
	struct sock* other;
	
	serversocket = (struct socket*)psock_rpmsg->serversock;
	if(sock_soc_socket_is_valid_ipc(serversocket, psock_rpmsg->clientsock) == -1){ //serversockѹر
		sk_soc_info("sock_soc_socket_is_valid is false \n");
		return -1;
	}
	other = serversocket->sk;	
	
	sk_soc_info("sock_soc_state_lock_proc serverfd=%d \n", psock_rpmsg->serverfd);
	unix_state_lock(other);
	return 0;
}

void sock_soc_state_unlock(struct sock *other)
{
	struct ipc_socket *psocket_ipc = NULL;
	struct sock_rpmsg sock_rpmsg = {0};
	sock_rpmsg.msg_type = MSG_TYPE_STATE_UNLOCK;
		
	//ҷ˴socketϢ
	psocket_ipc = sock_soc_get_ipcsocket_by_proxysock(other);
	if(psocket_ipc == NULL){
		sk_soc_info("sock_soc_state_unlock failed, other->fd=%d\n ", other->sk_fd);
		return;
	}
	//socket fd
	sock_rpmsg.serverfd = psocket_ipc->remotefd;
	sock_rpmsg.serversock = psocket_ipc->remotesocket;
	sock_rpmsg.clientsock = psocket_ipc->localsocket;
	sk_soc_info("sock_soc_state_unlock serverfd=%d \n", sock_rpmsg.serverfd);
	sock_send_ipc_msg(&sock_rpmsg); 
}

int sock_soc_state_unlock_proc(struct sock_rpmsg* psock_rpmsg)
{
	struct socket* serversocket;	
	struct sock* other;
	
	serversocket = (struct socket*)psock_rpmsg->serversock;
	if(sock_soc_socket_is_valid_ipc(serversocket, psock_rpmsg->clientsock) == -1){ //serversockѹر
		sk_soc_info("sock_soc_socket_is_valid is false \n");
		return -1;
	}
	other = serversocket->sk;	
	
	sk_soc_info("sock_soc_state_unlock_proc serverfd=%d \n", psock_rpmsg->serverfd);
	unix_state_lock(other);
	return 0;
}

void sock_soc_unix_bind(struct sockaddr_un *sunaddr)
{
	struct ipc_socket *psocket_ipc = NULL;
	struct sock_rpmsg sock_rpmsg = {0};
	sock_rpmsg.msg_type = MSG_TYPE_UNIX_BIND;		
	
	memcpy(&sock_rpmsg.sockaddr, sunaddr, sizeof(struct sockaddr_un));
	sk_soc_info("sock_soc_unix_bind  sunaddr=%s\n", sunaddr->sun_path);
	sock_send_ipc_msg(&sock_rpmsg); 
}

int sock_soc_unix_bind_proc(struct sock_rpmsg* psock_rpmsg)
{
	struct sockaddr_un sunaddr = {0};
	memcpy(&sunaddr, &psock_rpmsg->sockaddr, sizeof(struct sockaddr_un));
	sk_soc_info("sock_soc_unix_bind_proc  sunaddr=%s\n", sunaddr.sun_path);
	sock_soc_insert_ipc_sunaddr(&sunaddr);
	return 0;
}

void sock_soc_unix_unbind(struct sockaddr_un *sunaddr)
{
	struct ipc_socket *psocket_ipc = NULL;
	struct sock_rpmsg sock_rpmsg = {0};
	sock_rpmsg.msg_type = MSG_TYPE_UNIX_UNBIND;		
	
	memcpy(&sock_rpmsg.sockaddr, sunaddr, sizeof(struct sockaddr_un));
	sk_soc_info("sock_soc_unix_unbind  sunaddr=%s\n", sunaddr->sun_path);
	sock_send_ipc_msg(&sock_rpmsg); 
}

int sock_soc_unix_unbind_proc(struct sock_rpmsg* psock_rpmsg)
{
	struct sockaddr_un sunaddr = {0};
	memcpy(&sunaddr, &psock_rpmsg->sockaddr, sizeof(struct sockaddr_un));
	sk_soc_info("sock_soc_unix_unbind_proc  sunaddr=%s\n", sunaddr.sun_path);
	sock_soc_del_ipc_sunaddr(&sunaddr);
	return 0;
}

static int sock_create_icp_channel(T_ZDrvRpMsg_ActorID core_id, T_ZDrvRpMsg_ChID channel_id, unsigned int channel_size)
{
    int retval;

    retval = sockSocCreateChannel (core_id, channel_id, channel_size);
    if(retval != RPMSG_SUCCESS && retval != RPMSG_CHANNEL_ALREADY_EXIST)
        goto out;

	sk_soc_info("sock_create_icp_channel success");
    return retval;

out:
    sk_soc_err("could not create channel.");
    return retval;
}

static void sock_dld_handle(struct	sock_rpmsg* sock_rpmsg)
{
	int len;	
	len = sock_channel_write(&g_sock_chn_info, sock_rpmsg, sock_rpmsg->data_len + sizeof(struct sock_rpmsg));
	sk_soc_info("sock_dld_handle len=%d, index=%d", len, sock_rpmsg->index);
	return;
}
static void sock_uld_handle(struct	sock_rpmsg* sock_rpmsg)
{
	if(NULL == sock_rpmsg){
		return;
	}
	
	switch(sock_rpmsg->msg_type) {
		case MSG_TYPE_FIND:
			sock_soc_find_other_proc(sock_rpmsg);
			break;
		case MSG_TYPE_FIND_RESULT:
			sock_soc_find_other_result_proc(sock_rpmsg);
			break;
		case MSG_TYPE_ADD_WAIT_QUEUE:
			sock_soc_add_wait_queue_proc(sock_rpmsg);
			break;
		case MSG_TYPE_REMOVE_WAIT_QUEUE:
			sock_soc_remove_wait_queue_proc(sock_rpmsg);
			break;
		case MSG_TYPE_SYNC_IPC_SOCKET:
			sock_sync_ipc_socket_proc(sock_rpmsg);
			break;
		case MSG_TYPE_WAIT_FOR_PEER:
			sock_wait_for_peer_proc(sock_rpmsg);
			break;
		case MSG_TYPE_DATA_READY:
			sock_soc_sk_data_ready_proc(sock_rpmsg);
			break;
		case MSG_TYPE_SEND_MSG:
			sock_soc_msg_send_proc(sock_rpmsg);
			break;
		case MSG_TYPE_ADD_SKB_QUEUE_TAIL:
			sock_soc_add_skb_queue_tail_proc(sock_rpmsg);
			break;	
		case MSG_TYPE_SET_PEER:
			sock_soc_set_peer_proc(sock_rpmsg);
			break;
		case MSG_TYPE_RECVQ_FULL:
			sock_soc_recvq_full_proc(sock_rpmsg);
			break;
		case MSG_TYPE_RECVQ_FULL_RESULT:
			sock_soc_recvq_full_result_proc(sock_rpmsg);
			break;
		case MSG_TYPE_PEER_CLEAR:
			sock_soc_unix_peer_clear_proc(sock_rpmsg);
			break;
		case MSG_TYPE_WAKE_UP_INTERRUPTIBLE_POLL:
			sock_soc_wake_up_interruptible_poll_proc(sock_rpmsg);
			break;
		case MSG_TYPE_RECVQ_FULL_LOCKLESS_AND_DEAD:
			sock_soc_recvq_full_lockless_and_dead_proc(sock_rpmsg);
			break;
		case MSG_TYPE_RECVQ_FULL_LOCKLESS_AND_DEAD_RESULT:
			sock_soc_recvq_full_lockless_and_dead_result_proc(sock_rpmsg);
			break;
		case MSG_TYPE_TEST_FLAG:
			sock_soc_test_flags_proc(sock_rpmsg);					
			break;
		case MSG_TYPE_TEST_FLAG_RESULT:
			sock_soc_test_flags_result_proc(sock_rpmsg);
			break;
		case MSG_TYPE_SOCK_PUT:
			sock_soc_sock_put_proc(sock_rpmsg);
			break;					
		case MSG_TYPE_GET_SOCKET_STATES:
			sock_soc_get_state_proc(sock_rpmsg);
			break;
		case MSG_TYPE_GET_SOCKET_STATES_RESULT:
			sock_soc_get_state_result_proc(sock_rpmsg);
			break;
		case MSG_TYPE_STREAM_CONNECT:
			sock_soc_stream_connect_proc(sock_rpmsg);
			break;
		case MSG_TYPE_UPDATE_PEER:
			sock_soc_update_peer_proc(sock_rpmsg);
			break;		
		case MSG_TYPE_DGRAM_DISCONNECTED:
			sock_soc_dgram_disconnected_proc(sock_rpmsg);
			break;
		case MSG_TYPE_STATE_LOCK:
			sock_soc_state_lock_proc(sock_rpmsg);
			break;
		case MSG_TYPE_STATE_UNLOCK:
			sock_soc_state_unlock_proc(sock_rpmsg);
			break;
		case MSG_TYPE_SOCK_HOLD:
			sock_soc_sock_hold_proc(sock_rpmsg);
			break;
		case MSG_TYPE_RELEASE_PEER:
			sock_soc_release_peer_proc(sock_rpmsg);
			break;
		case MSG_TYPE_NOTIFY_PROXY_CHANGE:
			sock_soc_notify_proxy_change_to_server_proc(sock_rpmsg);
			break;
		case MSG_TYPE_UNIX_BIND:
			sock_soc_unix_bind_proc(sock_rpmsg);
			break;
		case MSG_TYPE_UNIX_UNBIND:
			sock_soc_unix_unbind_proc(sock_rpmsg);
			break;
		default:
		    break;
	}
}

static int sock_receive_thread(void *argv)
{   
	int index,ret_len,i,num;
    unsigned long flags;
	//static struct sock_rpmsg sock_rpmsg = {0};
	static char recvbuf[SOCK_DATA_MAX_LEN] = {0};
	sk_soc_info("sock_receive_thread process");
	while(1){
		if(0 < sock_channel_read(&g_sock_chn_info, recvbuf, sizeof(recvbuf))) {
			//sk_soc_info("sock_receive_thread sock_rpmsg.msg_type=%d,index=%d", sock_rpmsg.msg_type, sock_rpmsg.index);	
#if 1
			sock_recv_ipc_msg((struct sock_rpmsg*)recvbuf);
#else
	        switch(sock_rpmsg.msg_type) {
				case MSG_TYPE_FIND:
					sock_soc_find_other_proc(&sock_rpmsg);
					break;
				case MSG_TYPE_FIND_RESULT:
					sock_soc_find_other_result_proc(&sock_rpmsg);
					break;
				case MSG_TYPE_ADD_WAIT_QUEUE:
					sock_soc_add_wait_queue_proc(&sock_rpmsg);
					break;
				case MSG_TYPE_REMOVE_WAIT_QUEUE:
					sock_soc_remove_wait_queue_proc(&sock_rpmsg);
					break;
				case MSG_TYPE_SYNC_IPC_SOCKET:
					sock_sync_ipc_socket_proc(&sock_rpmsg);
					break;
				case MSG_TYPE_WAIT_FOR_PEER:
					sock_wait_for_peer_proc(&sock_rpmsg);
					break;
				case MSG_TYPE_DATA_READY:
					sock_soc_sk_data_ready_proc(&sock_rpmsg);
					break;
				case MSG_TYPE_SEND_MSG:
					sock_soc_msg_send_proc(&sock_rpmsg);
					break;
				case MSG_TYPE_ADD_SKB_QUEUE_TAIL:
					sock_soc_add_skb_queue_tail_proc(&sock_rpmsg);
					break;
				case MSG_TYPE_SET_PEER:
					sock_soc_set_peer_proc(&sock_rpmsg);
					break;
				case MSG_TYPE_RECVQ_FULL:
					sock_soc_recvq_full_proc(&sock_rpmsg);
					break;
				case MSG_TYPE_RECVQ_FULL_RESULT:
					sock_soc_recvq_full_result_proc(&sock_rpmsg);
					break;
				case MSG_TYPE_PEER_CLEAR:
					sock_soc_unix_peer_clear_proc(&sock_rpmsg);
					break;
				case MSG_TYPE_WAKE_UP_INTERRUPTIBLE_POLL:
					sock_soc_wake_up_interruptible_poll_proc(&sock_rpmsg);
					break;
				case MSG_TYPE_RECVQ_FULL_LOCKLESS_AND_DEAD:
					sock_soc_recvq_full_lockless_and_dead_proc(&sock_rpmsg);
					break;
				case MSG_TYPE_RECVQ_FULL_LOCKLESS_AND_DEAD_RESULT:
					sock_soc_recvq_full_lockless_and_dead_result_proc(&sock_rpmsg);
					break;
				case MSG_TYPE_TEST_FLAG:
					sock_soc_test_flags_proc(&sock_rpmsg);					
					break;
				case MSG_TYPE_TEST_FLAG_RESULT:
					sock_soc_test_flags_result_proc(&sock_rpmsg);
					break;
				case MSG_TYPE_SOCK_PUT:
					sock_soc_sock_put_proc(&sock_rpmsg);
					break;					
				case MSG_TYPE_GET_SOCKET_STATES:
					sock_soc_get_state_proc(&sock_rpmsg);
					break;
				case MSG_TYPE_GET_SOCKET_STATES_RESULT:
					sock_soc_get_state_result_proc(&sock_rpmsg);
					break;
				case MSG_TYPE_STREAM_CONNECT:
					sock_soc_stream_connect_proc(&sock_rpmsg);
					break;
				case MSG_TYPE_UPDATE_PEER:
					sock_soc_update_peer_proc(&sock_rpmsg);
					break;
				case MSG_TYPE_SOCKET_RELEASE:
					sock_soc_release_sock_proc(&sock_rpmsg);
					break;
				case MSG_TYPE_DGRAM_DISCONNECTED:
					sock_soc_dgram_disconnected_proc(&sock_rpmsg);
					break;
				case MSG_TYPE_STATE_LOCK:
					sock_soc_state_lock_proc(&sock_rpmsg);
					break;
				case MSG_TYPE_STATE_UNLOCK:
					sock_soc_state_unlock_proc(&sock_rpmsg);
					break;
				case MSG_TYPE_SOCK_HOLD:
					sock_soc_sock_hold_proc(&sock_rpmsg);
					break;
				case MSG_TYPE_RELEASE_PEER:
					sock_soc_release_peer_proc(&sock_rpmsg);
					break;
				case MSG_TYPE_NOTIFY_PROXY_CHANGE:
					sock_soc_notify_proxy_change_to_server_proc(&sock_rpmsg);
					break;
				default:
				    break;
			}
#endif			
	    }
	    else {
	        sk_soc_err("sock_channel_read fail.");
	        msleep(1000);
			continue;
	    } 
		
	}      

    sk_soc_err("The receive thread exit!");
    return 0;
}

static int sock_main_thread(void *argv)
{ 	
	struct list_head	tmp_list;
	struct sock_rpmsg *entry,*tmp_entry;
	unsigned long flags;
	struct sock_thread_info* p_main_thread_info;
	p_main_thread_info = &g_sock_chn_info.main_thread_info;
	
	while(!p_main_thread_info->bstop) {
		sock_get_sema(&p_main_thread_info->p_sock_sem);

		spin_lock_irqsave(&p_main_thread_info->p_sock_lock,flags);	
		if (list_empty(&p_main_thread_info->p_sock_list)) {
			spin_unlock_irqrestore(&p_main_thread_info->p_sock_lock,flags);
			continue;
		}
		list_replace_init(&p_main_thread_info->p_sock_list,&tmp_list);
		list_del_init(&p_main_thread_info->p_sock_list);		
		spin_unlock_irqrestore(&p_main_thread_info->p_sock_lock,flags);

		list_for_each_entry_safe(entry,tmp_entry,&tmp_list,list) {
			if(DIR_DOWNLINK == entry->dir) {				
				sock_dld_handle(entry);
			}
			else if(DIR_UPLINK == entry->dir) {
				sock_uld_handle(entry);				
			}
			else {
				sk_soc_warn("sock data unknow dir(%d).",entry->dir);
			}
			list_del(&entry->list);
			kfree(entry);
		}
	}

	sk_soc_err("the sock_main_thread stop!\n");    
    return 0;
}

static int __init sock_soc_init(void)
{
	struct task_struct *th = NULL;
	int retval = 0;
	struct sock_thread_info* p_main_thread_info;
	
	sema_init(&g_sock_sem, 0);
	g_sock_chn_info.core_id = CAP_ID;
	g_sock_chn_info.channel_id = ICP_CHN_SOCKET;
	g_sock_chn_info.channel_size = ICP_CHANNEL_SIZE;
	
	sk_soc_info("sock_soc_init process");
	INIT_HLIST_HEAD(&g_ipc_sockets);
	INIT_HLIST_HEAD(&g_sock_rpmsg_info);
	INIT_HLIST_HEAD(&g_sockets_info);
	INIT_HLIST_HEAD(&g_sunaddr_info);
	retval = sock_create_icp_channel(CAP_ID, ICP_CHN_SOCKET, ICP_CHANNEL_SIZE);
	if(retval < 0) {
		sk_soc_err("Create IcpChannel channel_32 fail.");
		return retval;
	}
	
	p_main_thread_info = &g_sock_chn_info.main_thread_info;
	
	INIT_LIST_HEAD(&p_main_thread_info->p_sock_list);
	spin_lock_init(&p_main_thread_info->p_sock_lock);
	sema_init(&p_main_thread_info->p_sock_sem, 0);
	
	th = kthread_run(sock_main_thread, 0, "sock-soc-main%d", ICP_CHN_SOCKET);
	if (IS_ERR(th)) {
		sk_soc_err("Unable to start receive thread.");
		return PTR_ERR(th);
	}
	g_sock_chn_info.main_thread_info.p_thread = th;

	th = kthread_run(sock_receive_thread, 0, "sock-soc-recv%d", ICP_CHN_SOCKET);
	if (IS_ERR(th)) {
		sk_soc_err("Unable to start receive thread.");
		return PTR_ERR(th);
	}
	g_sock_chn_info.recv_thread_info.p_thread = th;

	ipcsocket_debugfs_dir_entry_root = debugfs_create_dir("ipc_socket", NULL);
	if (ipcsocket_debugfs_dir_entry_root)
		ipcsocket_debugfs_dir_entry_proc = debugfs_create_dir("proc",
						 ipcsocket_debugfs_dir_entry_root);

	if (ipcsocket_debugfs_dir_entry_root) {
		debugfs_create_file("ipc_socket_info",
				    0444,
				    ipcsocket_debugfs_dir_entry_root,
				    NULL,
				    &ipc_socket_info_fops);
	}
	return 0;
}

static void __exit sock_soc_exit(void)
{
    memset(&g_sock_chn_info, 0, sizeof(struct sock_channel));
    sk_soc_warn("success.\n");
}

late_initcall(sock_soc_init);
module_exit(sock_soc_exit);

MODULE_AUTHOR("ZXIC");
MODULE_DESCRIPTION("ZXIC CAP LAN NET DEVICE");
MODULE_LICENSE("GPL");

