/*
 * LEDs driver for GPIOs
 *
 * Copyright (C) 2007 8D Technologies inc.
 * Raphael Assenat <raph@8d.com>
 * Copyright (C) 2008 Freescale Semiconductor, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * Modified by YinWenguan: for
 * (1) All LEDs controlled by GPIOs.
 * (2) Adding the differences of the GPIO functions.
 * (3)
 */
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/gpio.h>
#include <linux/leds.h>
#include <linux/of_platform.h>
#include <linux/of_gpio.h>
#include <linux/slab.h>
#include <linux/workqueue.h>
#include <linux/module.h>

#include <linux/errno.h>
#include <linux/clk.h>
#include <mach/irqs.h>
#include <mach/timex.h>
#include <linux/irqreturn.h>
#include <linux/wakelock.h>
#include <linux/interrupt.h>
#include <mach/pcu.h>
#include <linux/mfd/zx234290.h>
#include <linux/semaphore.h>
#include <linux/kthread.h>

struct gpio_led_data {
	struct led_classdev cdev;
	unsigned gpio;
	unsigned pin_select;
	struct work_struct work;
	u8 new_level;
	u8 can_sleep;
	u8 active_low;
	u8 blinking;
	int (*platform_gpio_blink_set)(unsigned pin_sel,unsigned gpio, int state,
			unsigned long *delay_on, unsigned long *delay_off);
};

struct semaphore	s_sink1_sem;	
struct semaphore	s_sink2_sem;
int s_sink1_level;
int s_sink2_level;

static void gpio_led_work(struct work_struct *work)
{
	struct gpio_led_data	*led_dat =
		container_of(work, struct gpio_led_data, work);

	if (led_dat->blinking) {
		led_dat->platform_gpio_blink_set(led_dat->pin_select,led_dat->gpio,
						 led_dat->new_level,
						 NULL, NULL);
		led_dat->blinking = 0;
	} else
	if(led_dat->pin_select==0)
		gpio_set_value_cansleep(led_dat->gpio, led_dat->new_level);
}

static void gpio_led_set(struct led_classdev *led_cdev,
	enum led_brightness value)
{
	struct gpio_led_data *led_dat =
		container_of(led_cdev, struct gpio_led_data, cdev);
	int level;

	if (value == LED_OFF)
		level = 0;
	else
		level = 1;

	if (led_dat->active_low)
		level = !level;

	/* Setting GPIOs with I2C/etc requires a task context, and we don't
	 * seem to have a reliable way to know if we're already in one; so
	 * let's just assume the worst.
	 */
	if (led_dat->can_sleep) {
		led_dat->new_level = level;
		schedule_work(&led_dat->work);
	} else {
#if 0	
		if (led_dat->blinking) {/*
			led_dat->platform_gpio_blink_set(led_dat->pin_select,led_dat->gpio, level,
							 NULL, NULL);
			led_dat->blinking = 0;*/
		gpio_set_value(led_dat->gpio, level);

		} else
			gpio_set_value(led_dat->gpio, level);
	
#endif
		if(led_dat->pin_select ==0)
			gpio_set_value(led_dat->gpio, level);
		else if(led_dat->pin_select ==1){
			s_sink1_level = level;
			up(&s_sink1_sem);
		}
		else if(led_dat->pin_select ==2){
			//zx234297_set_sink(ZX234297_SINK2,level,SINK_CURRENT_5MA);
			s_sink2_level = level;		
			up(&s_sink2_sem);
		}
		else
			;
	}	
}

static int gpio_blink_set(struct led_classdev *led_cdev,
	unsigned long *delay_on, unsigned long *delay_off)
{
	struct gpio_led_data *led_dat =
		container_of(led_cdev, struct gpio_led_data, cdev);

	if((delay_on ==NULL)||(delay_off ==NULL)){
		led_dat->blinking = 0;
	}else{
		led_dat->blinking = 1;
	}

	return led_dat->platform_gpio_blink_set(led_dat->pin_select,led_dat->gpio, GPIO_LED_BLINK,
						delay_on, delay_off);
}

static int __devinit create_gpio_led(const struct gpio_led *template,
	struct gpio_led_data *led_dat, struct device *parent,
	int (*blink_set)(unsigned, unsigned, int, unsigned long *, unsigned long *))
{
	int ret=0, state;

	led_dat->gpio = -1;

	/* skip leds that aren't available */
	if ((!gpio_is_valid(template->gpio))&&(0==template->pin_select)) {
		printk(KERN_INFO "Skipping unavailable LED gpio %d (%s)\n",
				template->gpio, template->name);
		return 0;
	}

	if(0==template->pin_select){
		ret = gpio_request(template->gpio, template->name);
		if (ret < 0)
			return ret;
		/* Added by YinWenguan: gpio operation */
		zx29_gpio_config(template->gpio, template->func);
		zx29_gpio_set_direction(template->gpio, GPIO_OUT);
		//zx29_gpio_output_data(template->gpio, GPIO_LOW);
	}

	led_dat->cdev.name = template->name;
	led_dat->cdev.default_trigger = template->default_trigger;
	
	if(0==template->pin_select){
		led_dat->gpio = template->gpio;
		led_dat->can_sleep = gpio_cansleep(template->gpio);
	}
	led_dat->active_low = template->active_low;
	led_dat->blinking = 0;
	if (blink_set) {
		led_dat->platform_gpio_blink_set = blink_set;
		led_dat->cdev.blink_set = gpio_blink_set;
	}
	led_dat->cdev.brightness_set = gpio_led_set;
	if ((0==template->pin_select)&&(template->default_state == LEDS_GPIO_DEFSTATE_KEEP))
		state = !!gpio_get_value_cansleep(led_dat->gpio) ^ led_dat->active_low;
	else
		state = (template->default_state == LEDS_GPIO_DEFSTATE_ON);
	led_dat->cdev.brightness = state ? LED_FULL : LED_OFF;
	if (!template->retain_state_suspended)
		led_dat->cdev.flags |= LED_CORE_SUSPENDRESUME;
	if(0==template->pin_select){
		ret = gpio_direction_output(led_dat->gpio, led_dat->active_low ^ state);
	}else if(1==template->pin_select){
		zx234297_set_sink(ZX234297_SINK1,led_dat->active_low ^ state,SINK_CURRENT_5MA);
	}else if(2==template->pin_select){
		zx234297_set_sink(ZX234297_SINK2,led_dat->active_low ^ state,SINK_CURRENT_5MA);
	}else{
		printk("led ctrl by gpio or sink set error\n");
	}
	
	if (ret < 0)
		goto err;
		
	INIT_WORK(&led_dat->work, gpio_led_work);

	ret = led_classdev_register(parent, &led_dat->cdev);
	if (ret < 0)
		goto err;

	return 0;
err:
	gpio_free(led_dat->gpio);
	return ret;
}

static void delete_gpio_led(struct gpio_led_data *led)
{
	if ((0==led->pin_select)&&(!gpio_is_valid(led->gpio)))
		return;
	led_classdev_unregister(&led->cdev);
	cancel_work_sync(&led->work);
	if(0==led->pin_select)/*the pin is gpio ,not sink*/
		gpio_free(led->gpio);
}

struct gpio_leds_priv {
	int num_leds;
	struct gpio_led_data leds[];
};

static inline int sizeof_gpio_leds_priv(int num_leds)
{
	return sizeof(struct gpio_leds_priv) +
		(sizeof(struct gpio_led_data) * num_leds);
}

/* Code to create from OpenFirmware platform devices */
#ifdef CONFIG_OF_GPIO
static struct gpio_leds_priv * __devinit gpio_leds_create_of(struct platform_device *pdev)
{
	struct device_node *np = pdev->dev.of_node, *child;
	struct gpio_leds_priv *priv;
	int count = 0, ret;

	/* count LEDs in this device, so we know how much to allocate */
	for_each_child_of_node(np, child)
		count++;
	if (!count)
		return NULL;

	priv = kzalloc(sizeof_gpio_leds_priv(count), GFP_KERNEL);
	if (!priv)
		return NULL;

	for_each_child_of_node(np, child) {
		struct gpio_led led = {};
		enum of_gpio_flags flags;
		const char *state;

		led.gpio = of_get_gpio_flags(child, 0, &flags);
		led.active_low = flags & OF_GPIO_ACTIVE_LOW;
		led.name = of_get_property(child, "label", NULL) ? : child->name;
		led.default_trigger =
			of_get_property(child, "linux,default-trigger", NULL);
		state = of_get_property(child, "default-state", NULL);
		if (state) {
			if (!strcmp(state, "keep"))
				led.default_state = LEDS_GPIO_DEFSTATE_KEEP;
			else if (!strcmp(state, "on"))
				led.default_state = LEDS_GPIO_DEFSTATE_ON;
			else
				led.default_state = LEDS_GPIO_DEFSTATE_OFF;
		}

		ret = create_gpio_led(&led, &priv->leds[priv->num_leds++],
				      &pdev->dev, NULL);
		if (ret < 0) {
			of_node_put(child);
			goto err;
		}
	}

	return priv;

err:
	for (count = priv->num_leds - 2; count >= 0; count--)
		delete_gpio_led(&priv->leds[count]);
	kfree(priv);
	return NULL;
}

static const struct of_device_id of_gpio_leds_match[] = {
	{ .compatible = "gpio-leds", },
	{},
};
#else /* CONFIG_OF_GPIO */
static struct gpio_leds_priv * __devinit gpio_leds_create_of(struct platform_device *pdev)
{
	return NULL;
}
#define of_gpio_leds_match NULL
#endif /* CONFIG_OF_GPIO */

#if 1 /*qihongfang*/

struct led_hw_timer
{
	struct clk *led_timer_wclk;
	struct clk *led_timer_pclk;
	struct wake_lock led_timer_wake_lock;
	int led_timer_state;
	int led_delay_on;
	int led_delay_off;
	int flag_ledtimer;
};
static struct led_hw_timer s_led_hw_timer;

int  platform_gpio_blink_set(unsigned pin_sel,unsigned gpio, int state,
			unsigned long *delay_on, unsigned long *delay_off)
{
	if((delay_on ==NULL)||(delay_off==NULL)){/*stop timer*/	
		timer_stop(LED_TIMER_BASE);
		return 0;
	}
	
	if(*delay_on ==0 ||*delay_off ==0 ){
		timer_stop(LED_TIMER_BASE);
		return 0;
	}
	
	timer_stop(LED_TIMER_BASE);
	if(pin_sel==0)
		gpio_set_value(gpio, LEDS_GPIO_DEFSTATE_ON);
	else if(pin_sel==1){
		s_sink1_level = 1;		
		up(&s_sink1_sem);	
	}
	else if(pin_sel==2){
		//zx234297_set_sink(ZX234297_SINK2,1,SINK_CURRENT_5MA);		
		s_sink2_level = 1;		
		up(&s_sink2_sem);	
	}	
	s_led_hw_timer.led_timer_state = LEDS_GPIO_DEFSTATE_ON;
	s_led_hw_timer.led_delay_on = *delay_on;
	s_led_hw_timer.led_delay_off = *delay_off;
	timer_set_load(LED_TIMER_BASE, (s_led_hw_timer.led_delay_on*LED_CLOCK_RATE)/1000);
	timer_set_mode(LED_TIMER_BASE, false);
	timer_start(LED_TIMER_BASE);

	return 0;

}


/*******************************************************************************
* Function: zx29_kpd_irq_handler
* Description: clear irq , wake thread irq
* Parameters:
*          Input:
*       Output:
********************************************************************************/
static irqreturn_t led_timer_irq_handler(int irq, void *dev_id)
{
	int gpio = 0;
	const struct gpio_led *pdata;

	//wake_lock(&(s_led_hw_timer.led_timer_wake_lock));
	pcu_clr_irq_pending(irq);
	
	//wake_lock(&(s_led_hw_timer.led_timer_wake_lock));
	pdata = (struct gpio_led_platform_data *)dev_id;
	gpio = pdata->gpio;
		
	if(LEDS_GPIO_DEFSTATE_ON==s_led_hw_timer.led_timer_state){
		gpio_set_value(gpio, LEDS_GPIO_DEFSTATE_OFF);
		s_led_hw_timer.led_timer_state = LEDS_GPIO_DEFSTATE_OFF;

		timer_set_load(LED_TIMER_BASE, (s_led_hw_timer.led_delay_off*LED_CLOCK_RATE)/1000);
		timer_set_mode(LED_TIMER_BASE, false);
		timer_start(LED_TIMER_BASE);
	}
	else{
		gpio_set_value(gpio, LEDS_GPIO_DEFSTATE_ON);
		s_led_hw_timer.led_timer_state = LEDS_GPIO_DEFSTATE_ON;

		timer_set_load(LED_TIMER_BASE,  (s_led_hw_timer.led_delay_on*LED_CLOCK_RATE)/1000);
		timer_set_mode(LED_TIMER_BASE, false);
		timer_start(LED_TIMER_BASE);
	}
	
	//wake_unlock(&(s_led_hw_timer.led_timer_wake_lock));

	return IRQ_HANDLED;
}
static int led_timer_init(const struct gpio_led *template)
{
    int ret = -1;
		
	if(1==s_led_hw_timer.flag_ledtimer){
		printk(KERN_INFO "warning! led_timer_init can  init only one time\n");	
		return 0;
	}
	s_led_hw_timer.flag_ledtimer = 1;
	s_led_hw_timer.led_timer_wclk = timer_get_clk(LED_TIMER_NAME, "work_clk");
	if (s_led_hw_timer.led_timer_wclk == 0)
		return ret;
	
	s_led_hw_timer.led_timer_pclk = timer_get_clk(LED_TIMER_NAME, "apb_clk");
	if (s_led_hw_timer.led_timer_pclk == 0)
		return ret;
	
	clk_prepare_enable(s_led_hw_timer.led_timer_pclk); 
	
	clk_set_rate(s_led_hw_timer.led_timer_wclk, LED_CLOCK_RATE);
	clk_prepare_enable(s_led_hw_timer.led_timer_wclk);	
	//printk("*led_timer=%lu, parent=%s \n", clk_get_rate(s_led_hw_timer.led_timer_wclk), __clk_get_name(clk_get_parent(s_led_hw_timer.led_timer_wclk)));

	irq_set_irq_wake(PS_TIMER2_INT,1);
	
	//wake_lock_init(&(s_led_hw_timer.led_timer_wake_lock), WAKE_LOCK_SUSPEND, "led_timer");

	//ret = request_threaded_irq(AP_TIMER2_INT, led_timer_irq_handler, led_timer_irq_thread,
	//	/*host->irq_flags |*/IRQF_ONESHOT, "led_timer", template);
	ret = request_irq(PS_TIMER2_INT, led_timer_irq_handler, IRQF_ONESHOT, "led_timer", template);
	if(ret < 0)
		printk(KERN_INFO "warning! led_timer_irq_handler request irq error\n");	
	
	return ret;

}
 static void led_timer_unint()
{
   // wake_lock_destroy(&(s_led_hw_timer.led_timer_wake_lock));
}
#endif
#if 1
static irqreturn_t led_sink1_thread(void *data)
{
	struct gpio_led_data *led_data = data;
	struct sched_param param = { .sched_priority = 2 };
	param.sched_priority= 31;
	sched_setscheduler(current, SCHED_FIFO, &param);

	while(1)
	{
		down(&s_sink1_sem);		
		zx234297_set_sink(ZX234297_SINK1,s_sink1_level,SINK_CURRENT_5MA);
	}
}

static irqreturn_t led_sink2_thread(void *data)
{
	struct gpio_led_data *led_data = data;
	struct sched_param param = { .sched_priority = 2 };
	param.sched_priority= 31;
	sched_setscheduler(current, SCHED_FIFO, &param);

	while(1)
	{
		down(&s_sink2_sem);		
		zx234297_set_sink(ZX234297_SINK2,s_sink2_level,SINK_CURRENT_5MA);
	}
}

#endif
static int __devinit gpio_led_probe(struct platform_device *pdev)
{
	struct gpio_led_platform_data *pdata = pdev->dev.platform_data;
	struct gpio_leds_priv *priv;
	int i, ret = 0;	
	struct task_struct	*sink1_thread;
	struct task_struct	*sink2_thread;
	
	if (pdata && pdata->num_leds) {
		priv = kzalloc(sizeof_gpio_leds_priv(pdata->num_leds),
				GFP_KERNEL);
		if (!priv)
			return -ENOMEM;

		priv->num_leds = pdata->num_leds;
		for (i = 0; i < priv->num_leds; i++) {
			priv->leds[i].pin_select = pdata->leds[i].pin_select;
			if(priv->leds[i].pin_select ==1){
				sema_init(&s_sink1_sem, 0);
				sink1_thread = kthread_run(led_sink1_thread, &priv->leds[i], "led_sink1");
				BUG_ON(IS_ERR(sink1_thread));
			}
			if(priv->leds[i].pin_select ==2){			
				sema_init(&s_sink2_sem, 0);
				sink2_thread = kthread_run(led_sink2_thread, &priv->leds[i], "led_sink2");
				BUG_ON(IS_ERR(sink2_thread));
			}
			
			if(1==pdata->leds[i].hw_timer){
				led_timer_init(&pdata->leds[i]);
			ret = create_gpio_led(&pdata->leds[i],
					      &priv->leds[i],
					      &pdev->dev, pdata->gpio_blink_set);

			}
			else{
			ret = create_gpio_led(&pdata->leds[i],
					      &priv->leds[i],
					      &pdev->dev, NULL);
			}
			if (ret < 0) {
				/* On failure: unwind the led creations */
				for (i = i - 1; i >= 0; i--)
					delete_gpio_led(&priv->leds[i]);
				kfree(priv);
				return ret;
			}
		}
	} else {
		priv = gpio_leds_create_of(pdev);
		if (!priv)
			return -ENODEV;
	}

	platform_set_drvdata(pdev, priv);

	return 0;
}

static int __devexit gpio_led_remove(struct platform_device *pdev)
{
	struct gpio_leds_priv *priv = dev_get_drvdata(&pdev->dev);
	int i;
	
	if(!priv)
		return -EINVAL;

	for (i = 0; i < priv->num_leds; i++)
		delete_gpio_led(&priv->leds[i]);

	dev_set_drvdata(&pdev->dev, NULL);
	kfree(priv);
	
	if(s_led_hw_timer.flag_ledtimer)
		led_timer_unint();
	
	return 0;
}

static struct platform_driver gpio_led_driver = {
	.probe		= gpio_led_probe,
	.remove		= __devexit_p(gpio_led_remove),
	.driver		= {
		.name	= "leds-gpio",
		.owner	= THIS_MODULE,
		.of_match_table = of_gpio_leds_match,
	},
};

module_platform_driver(gpio_led_driver);

MODULE_AUTHOR("Raphael Assenat <raph@8d.com>, Trent Piepho <tpiepho@freescale.com>");
MODULE_DESCRIPTION("GPIO LED driver");
MODULE_LICENSE("GPL");
MODULE_ALIAS("platform:leds-gpio");
