/*
 * Synopsys DesignWare Multimedia Card Interface driver
 *  (Based on NXP driver for lpc 31xx)
 *
 * Copyright (C) 2009 NXP Semiconductors
 * Copyright (C) 2009, 2010 Imagination Technologies Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */
//#pragma GCC optimize("O0")

#include <linux/blkdev.h>
#include <linux/clk.h>
#include <linux/debugfs.h>
#include <linux/device.h>
#include <linux/dma-mapping.h>
#include <linux/err.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/seq_file.h>
#include <linux/slab.h>
#include <linux/stat.h>
#include <linux/delay.h>
#include <linux/irq.h>
#include <linux/mmc/host.h>
#include <linux/mmc/mmc.h>
#include <linux/mmc/sdio.h>
#include <linux/mmc/card.h>
//#include <linux/mmc/dw_mmc.h>
#include <linux/bitops.h>
#include <linux/regulator/consumer.h>
#include <linux/workqueue.h>
#include <linux/kernel.h>

#include <linux/sched.h>
#include <linux/kthread.h>
#include <linux/ramdump/ramdump.h>

#include <mach/zx29_mmc.h>
#include "mmc_xlog.h"
#include <mach/iomap.h>
#include <mach/gpio.h>
#include <mach/pcu.h>
#include <linux/gpio.h>
//#include "dw_mmc.h"
#include "zx29_mmc.h"
#include "pub_debug_info.h"
#include <linux/mmc/sdio_func.h>

#if (defined CONFIG_ARCH_ZX297520V2)||(defined CONFIG_ARCH_ZX297520V3)
#include <mach/gpio_cfg.h>
#include <mach/highspeed_debug.h>

#include <linux/android_notify.h>
#include <linux/wakelock.h>
#include <mach/spinlock.h>
#define INT_POSEDGE     0x2          /* 10: raise edge */
static struct wake_lock sd_detect_wake_lock;
#endif
#ifdef CONFIG_ARCH_ZX297510
#include <mach/zx297510_dma.h>
#undef CONFIG_MMC_DW_IDMAC
#define CONFIG_MMC_ZX29_EDMAC

#define PIN_SD0_CLK SDB_GPIO_62
#define PIN_SD0_CMD SDB_GPIO_63
#define PIN_SD0_DATA0 SDB_GPIO_64
#define PIN_SD0_DATA1 SDB_GPIO_65
#define PIN_SD0_DATA2 SDB_GPIO_66
#define PIN_SD0_DATA3 SDB_GPIO_67

#define PIN_SD1_CLK SDB_GPIO_56
#define PIN_SD1_CMD SDB_GPIO_57
#define PIN_SD1_DATA0 SDB_GPIO_58
#define PIN_SD1_DATA1 SDB_GPIO_59
#define PIN_SD1_DATA2 SDB_GPIO_60
#define PIN_SD1_DATA3 SDB_GPIO_61
#else

#define PIN_SD0_CLK 0
#define PIN_SD0_CMD 0
#define PIN_SD0_DATA0 0
#define PIN_SD0_DATA1 0
#define PIN_SD0_DATA2 0
#define PIN_SD0_DATA3 0

#define PIN_SD1_CLK 0
#define PIN_SD1_CMD 0
#define PIN_SD1_DATA0 0
#define PIN_SD1_DATA1 0
#define PIN_SD1_DATA2 0
#define PIN_SD1_DATA3 0

#endif

static u32 updata_timeout = 0;

static struct dw_mci* dw_mmc_host[] = {NULL, NULL, NULL, NULL};
static unsigned int enable_wifi_irq = 0;

//#pragma GCC optimize("O0")

/* Common flag combinations */
#define DW_MCI_DATA_ERROR_FLAGS	(SDMMC_INT_DRTO | SDMMC_INT_DCRC | \
				 SDMMC_INT_HTO | SDMMC_INT_SBE  | \
				 SDMMC_INT_EBE | SDMMC_INT_FRUN)
#define DW_MCI_CMD_ERROR_FLAGS	(SDMMC_INT_RTO | SDMMC_INT_RCRC | \
				 SDMMC_INT_RESP_ERR)
#define DW_MCI_ERROR_FLAGS	(DW_MCI_DATA_ERROR_FLAGS | \
				 DW_MCI_CMD_ERROR_FLAGS  | SDMMC_INT_HLE)
#define DW_MCI_SEND_STATUS	1
#define DW_MCI_RECV_STATUS	2
#define DW_MCI_DMA_THRESHOLD	16 /* 4 */

#define DW_MCI_FREQ_MAX	200000000	/* unit: HZ */
#define DW_MCI_FREQ_MIN	400000		/* unit: HZ */
#define MMC_TIMEOUT_SEC (5)

static const u8 tuning_blk_pattern_4bit[] = {
	0xff, 0x0f, 0xff, 0x00, 0xff, 0xcc, 0xc3, 0xcc,
	0xc3, 0x3c, 0xcc, 0xff, 0xfe, 0xff, 0xfe, 0xef,
	0xff, 0xdf, 0xff, 0xdd, 0xff, 0xfb, 0xff, 0xfb,
	0xbf, 0xff, 0x7f, 0xff, 0x77, 0xf7, 0xbd, 0xef,
	0xff, 0xf0, 0xff, 0xf0, 0x0f, 0xfc, 0xcc, 0x3c,
	0xcc, 0x33, 0xcc, 0xcf, 0xff, 0xef, 0xff, 0xee,
	0xff, 0xfd, 0xff, 0xfd, 0xdf, 0xff, 0xbf, 0xff,
	0xbb, 0xff, 0xf7, 0xff, 0xf7, 0x7f, 0x7b, 0xde,
};

static const u8 tuning_blk_pattern_8bit[] = {
	0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0x00,
	0xff, 0xff, 0xcc, 0xcc, 0xcc, 0x33, 0xcc, 0xcc,
	0xcc, 0x33, 0x33, 0xcc, 0xcc, 0xcc, 0xff, 0xff,
	0xff, 0xee, 0xff, 0xff, 0xff, 0xee, 0xee, 0xff,
	0xff, 0xff, 0xdd, 0xff, 0xff, 0xff, 0xdd, 0xdd,
	0xff, 0xff, 0xff, 0xbb, 0xff, 0xff, 0xff, 0xbb,
	0xbb, 0xff, 0xff, 0xff, 0x77, 0xff, 0xff, 0xff,
	0x77, 0x77, 0xff, 0x77, 0xbb, 0xdd, 0xee, 0xff,
	0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0x00,
	0x00, 0xff, 0xff, 0xcc, 0xcc, 0xcc, 0x33, 0xcc,
	0xcc, 0xcc, 0x33, 0x33, 0xcc, 0xcc, 0xcc, 0xff,
	0xff, 0xff, 0xee, 0xff, 0xff, 0xff, 0xee, 0xee,
	0xff, 0xff, 0xff, 0xdd, 0xff, 0xff, 0xff, 0xdd,
	0xdd, 0xff, 0xff, 0xff, 0xbb, 0xff, 0xff, 0xff,
	0xbb, 0xbb, 0xff, 0xff, 0xff, 0x77, 0xff, 0xff,
	0xff, 0x77, 0x77, 0xff, 0x77, 0xbb, 0xdd, 0xee,
};

//#ifdef CONFIG_ARCH_ZX297510
#define DW_MCI_THREAD
//#endif
#ifdef DW_MCI_THREAD
static void dw_mci_kick_thread(struct dw_mci* host);
#define DW_MCI_SPIN_UNLOCK_BH spin_unlock
#define DW_MCI_SPIN_LOCK_BH spin_lock
#else
#define DW_MCI_SPIN_UNLOCK_BH spin_unlock_bh
#define DW_MCI_SPIN_LOCK_BH spin_lock_bh
#endif
#define DW_DMA_DESC_TRANS_LEN    (8192 -16)//(4*1024)   /*define dw idma one desc can config trans-len*/

extern void mmc_get_host(struct mmc_host * host);
extern void mmc_put_host(struct mmc_host * host);

static void dw_mci_init_dma(struct dw_mci *host);
static void dw_mci_stop_dma(struct dw_mci *host);
static inline bool dw_mci_fifo_reset(struct dw_mci *host);
static inline bool dw_mci_ctrl_all_reset(struct dw_mci *host);
static void dw_mci_request_end(struct dw_mci *host, struct mmc_request *mrq);
static void dw_mci_setup_bus(struct dw_mci_slot *slot, bool force_clkinit);

static void dw_mci_write_data_pio(struct dw_mci *host);
static void dw_mci_read_data_pio(struct dw_mci *host, bool dto);
static bool dw_mci_ctrl_reset(struct dw_mci *host, u32 reset);

#ifdef XLOG_MMC_ENABLE
static void dw_mci_regs_print(struct dw_mci* host);
static void dw_mci_print_variable(struct dw_mci* host);
#endif
extern void ramdump_ram_conf_table_add( char *ram_name, unsigned long ram_phy, unsigned long ram_size, unsigned long ram_virt, unsigned long ram_flag, unsigned long ram_extra);


#define MCI_REG_BIT_OPS(host, regname, bitindex, isset) do { \
       u32 tmpreg; \
       tmpreg = mci_readl(host, regname); \
      if (isset) \
               tmpreg |= (0x1 << bitindex); \
       else \
               tmpreg &= ~(0x1 << bitindex); \
       mci_writel(host, regname, tmpreg); \
} while(0)

static inline int chk_sg_offset_len(struct dw_mci *host)
{
       /* no need to check offset or len for sd slot */
       if (!(host->quirks & DW_MCI_QUIRK_SDIO))
               return 0;
       return !((host->quirks & DW_MCI_QUIRK_UNALIGN_DMA_SZ) && (host->quirks & DW_MCI_QUIRK_UNALIGN_DMA_START));

}
#ifndef CONFIG_ARCH_ZX297510
#define DW_COMMAND_SOFT_TIMEOUT_SUPPORT
#endif
#ifdef DW_COMMAND_SOFT_TIMEOUT_SUPPORT
#define DW_DTO_TMO_SEC		10
static int dw_dto_timeout_value = DW_DTO_TMO_SEC * MSEC_PER_SEC;
static void dw_start_dto_timer(struct dw_mci *host)
{
	mod_timer(&host->dto_timer, jiffies + msecs_to_jiffies(dw_dto_timeout_value));
	return ;
}

static void dw_stop_dto_timer(struct dw_mci *host)
{
	del_timer(&host->dto_timer);
	return ;
}

static void dw_dto_timer_expired(unsigned long data)
{
	struct dw_mci *host = (struct dw_mci *)data;
	struct dw_mci_slot *slot = host->slot[0];
	struct mmc_request *mrq;
	u8 opcode;

	host->dw_dto_timer_expired_cnt++;
	dw_dto_timeout_value = DW_DTO_TMO_SEC * MSEC_PER_SEC;

	opcode = host->cmd ? host->cmd->opcode : 0xFF;
	xlog_mmc_log_opkey(host->host_id, opcode,
		"ERROR dw_dto_timer_expired", host->dw_dto_timer_expired_cnt, host->completed_events, host->state);
	xlog_mmc_log_opkey(host->host_id, opcode,
		"ERROR dw_dto_timer_expired", mci_readl(host, STATUS), mci_readl(host,IDSTS), mci_readl(host, RINTSTS));
	sc_debug_info_record(MODULE_ID_AP_MMC, "ERROR dw_dto_timer_expired cnt=%d,STATUS=0x%x\n",
		host->dw_dto_timer_expired_cnt,mci_readl(host, STATUS));

/*
	mmc_printk("ERROR %s,STATUS=0x%x,IDSTS=0x%x, RINTSTS=0x%x,MINTSTS=0x%x\n", __func__,
			mci_readl(host, STATUS),mci_readl(host,IDSTS),mci_readl(host, RINTSTS),mci_readl(host, MINTSTS));		//xiu
*/
#ifdef XLOG_MMC_ENABLE
	dw_mci_print_variable(host);
	dw_mci_regs_print(host);
#endif

	DW_MCI_SPIN_LOCK_BH(&host->lock);
	/* Clean up queue if present */
	mrq = slot->mrq;
	if (mrq) {
		if (mrq == host->mrq) {
			/*
			host->data = NULL;
			host->cmd = NULL;
			*/
			int req_end = 1;

			switch (host->state) {
			case STATE_IDLE:
				break;
			case STATE_SENDING_CMD:
			case STATE_SENDING_STOP:
				/*
				 * If CMD_DONE interrupt does NOT come in sending command
				 * state, we should notify the driver to terminate current
				 * transfer and report a command timeout to the core.
				 */
				host->cmd_status = SDMMC_INT_RTO;
				set_bit(EVENT_CMD_COMPLETE, &host->pending_events);
				req_end = 0;	/* tasklet_schedule(&host->tasklet); */
				/* cmd done maybe masked, clear it before sending stop. */
				if (mci_readl(host, RINTSTS) & SDMMC_INT_CMD_DONE)
					mci_writel(host, RINTSTS, SDMMC_INT_CMD_DONE);
				break;
			case STATE_SENDING_DATA:
			case STATE_DATA_BUSY:
			case STATE_DATA_ERROR:
				/*
				 * If DTO interrupt does NOT come in sending data state,
				 * we should notify the driver to terminate current transfer
				 * and report a data timeout to the core.
				 */
				host->data_status = SDMMC_INT_DRTO;
				set_bit(EVENT_DATA_ERROR, &host->pending_events);
				set_bit(EVENT_DATA_COMPLETE, &host->pending_events);
				req_end = 0;	/* tasklet_schedule(&host->tasklet); */
				break;
			#if 0
				case STATE_SENDING_CMD:
					mrq->cmd->error = -ETIMEDOUT;
					if (!mrq->data)
						break;
					/* fall through */
				case STATE_SENDING_DATA:
					mrq->data->error = -ETIMEDOUT;
					dw_mci_stop_dma(host);
					break;
				case STATE_DATA_BUSY:
				case STATE_DATA_ERROR:
					if (mrq->data->error == -EINPROGRESS)
						mrq->data->error = -ETIMEDOUT;
					if (!mrq->stop)
						break;
					/* fall through */
				case STATE_SENDING_STOP:
					if (mrq->stop)
					mrq->stop->error = -ETIMEDOUT;
					break;
			#endif
			}

			if (req_end) {
				host->data = NULL;
				host->cmd = NULL;
				dw_mci_request_end(host, mrq);
			} else {
#ifdef DW_MCI_THREAD
				dw_mci_kick_thread(host);
#else
				tasklet_schedule(&host->tasklet);
#endif
			}
		} else {
			list_del(&slot->queue_node);
			mrq->cmd->error = -ETIMEDOUT;
			if (mrq->data)
				mrq->data->error = -ETIMEDOUT;
			if (mrq->stop)
				mrq->stop->error = -ETIMEDOUT;

			spin_unlock(&host->lock);
			mmc_request_done(slot->mmc, mrq);
			dw_stop_dto_timer(host);
			spin_lock(&host->lock);
		}
	}	else {
		//dw_stop_dto_timer(host);
	}
	DW_MCI_SPIN_UNLOCK_BH(&host->lock);
/*
	dw_mci_ctrl_all_reset(host);
	dw_mci_setup_bus(slot, true);
*/
	return ;
}
#endif

#if defined(CONFIG_DEBUG_FS)
static int dw_mci_req_show(struct seq_file *s, void *v)
{
	struct dw_mci_slot *slot = s->private;
	struct mmc_request *mrq;
	struct mmc_command *cmd;
	struct mmc_command *stop;
	struct mmc_data	*data;

	/* Make sure we get a consistent snapshot */
	DW_MCI_SPIN_LOCK_BH(&slot->host->lock);
	mrq = slot->mrq;

	if (mrq) {
		cmd = mrq->cmd;
		data = mrq->data;
		stop = mrq->stop;

		if (cmd)
			seq_printf(s,
				   "CMD%u(0x%x) flg %x rsp %x %x %x %x err %d\n",
				   cmd->opcode, cmd->arg, cmd->flags,
				   cmd->resp[0], cmd->resp[1], cmd->resp[2],
				   cmd->resp[2], cmd->error);
		if (data)
			seq_printf(s, "DATA %u / %u * %u flg %x err %d\n",
				   data->bytes_xfered, data->blocks,
				   data->blksz, data->flags, data->error);
		if (stop)
			seq_printf(s,
				   "CMD%u(0x%x) flg %x rsp %x %x %x %x err %d\n",
				   stop->opcode, stop->arg, stop->flags,
				   stop->resp[0], stop->resp[1], stop->resp[2],
				   stop->resp[2], stop->error);
	}

	DW_MCI_SPIN_UNLOCK_BH(&slot->host->lock);

	return 0;
}

static int dw_mci_req_open(struct inode *inode, struct file *file)
{
	return single_open(file, dw_mci_req_show, inode->i_private);
}

static const struct file_operations dw_mci_req_fops = {
	.owner		= THIS_MODULE,
	.open		= dw_mci_req_open,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release,
};

static int dw_mci_regs_show(struct seq_file *s, void *v)
{
	struct dw_mci* host;

	host = s->private;

	if(!host)
		return 0;

	seq_printf(s, "STATUS:\t0x%08x\n", mci_readl(host,STATUS));
	seq_printf(s, "SDMMC_BLKSIZ:\t0x%08x\n", mci_readl(host,BLKSIZ));
	seq_printf(s, "SDMMC_BYTCNT:\t0x%08x\n", mci_readl(host,BYTCNT));
	seq_printf(s, "MINTSTS:\t0x%08x\n", mci_readl(host,MINTSTS));
	seq_printf(s, "RINTSTS:\t0x%08x\n", mci_readl(host,RINTSTS));
	seq_printf(s, "CMD:\t0x%08x\n", mci_readl(host,CMD));
	seq_printf(s, "CTRL:\t0x%08x\n", mci_readl(host,CTRL));
	seq_printf(s, "INTMASK:\t0x%08x\n", mci_readl(host,INTMASK));
	seq_printf(s, "CLKENA:\t0x%08x\n", mci_readl(host,CLKENA));
	seq_printf(s, "CLKDIV:\t0x%08x\n", mci_readl(host,CLKDIV));
	seq_printf(s, "TCBCNT:\t0x%08x\n", mci_readl(host,TCBCNT));
	seq_printf(s, "TBBCNT:\t0x%08x\n", mci_readl(host,TBBCNT));
	seq_printf(s, "BMOD:\t0x%08x\n", mci_readl(host,BMOD));
	seq_printf(s, "IDSTS:\t0x%08x\n", mci_readl(host,IDSTS));
	seq_printf(s, "UHS_REG_EXT:\t0x%08x\n", mci_readl(host,UHS_REG_EXT));
	seq_printf(s, "SDR50_104:\t0x%08x\n", mci_readl(host,SDR50_104));

	return 0;
}

static int dw_mci_regs_open(struct inode *inode, struct file *file)
{
	return single_open(file, dw_mci_regs_show, inode->i_private);
}

static const struct file_operations dw_mci_regs_fops = {
	.owner		= THIS_MODULE,
	.open		= dw_mci_regs_open,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release,
};


static int mmc_card_present_show(struct seq_file *s, void *v)
{
	struct dw_mci_slot *slot = s->private;
	int present;

	if(!slot)
		return 0;

	if(!(slot->mmc) || !(slot->mmc->card))
		present = 0;
	else
		present = !!mmc_card_present(slot->mmc->card);

	seq_printf(s, "%d\n", present);

	return 0;
}

static int dw_mci_present_open(struct inode *inode, struct file *file)
{
	return single_open(file, mmc_card_present_show, inode->i_private);
}

static const struct file_operations dw_mci_present_fops = {
	.owner		= THIS_MODULE,
	.open		= dw_mci_present_open,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release,
};

#ifdef XLOG_MMC_ENABLE
// xsh+
#define mci_readl_offset(dev, reg)			\
	__raw_readl((dev)->regs + reg)
static void dw_mci_regs_print(struct dw_mci* host)
{
	int i;

	if(!host)
		return ;

	mmc_printk("[MMC%d]%s\n", host->host_id, __func__);		//xiu
	for (i=0; i<0xa0; i+=16){
		mmc_printk(" %02x:%08x %08x %08x %08x\n", i,
				mci_readl_offset(host,i), mci_readl_offset(host,i+4),
				mci_readl_offset(host,i+8), mci_readl_offset(host,i+12));
	}
#if (defined CONFIG_ARCH_ZX297520V2)||(defined CONFIG_ARCH_ZX297520V3)
	i = 0x100;
	mmc_printk("%03x:%08x %08x %08x\n", i,
			mci_readl_offset(host,i), mci_readl_offset(host,i+4),
			mci_readl_offset(host,i+8));
#endif

	return ;
}
static void dw_mci_print_variable(struct dw_mci* host)
{
	mmc_printk("[MMC%d]V1.03", host->host_id);
	mmc_printk("pending_events:%lX completed_events:%lX\n"
			"cmd_status:%X data_status:%X\n"
			"card_status:%u state:%u",
			host->pending_events, host->completed_events,
			host->cmd_status, host->data_status,
			host->card_status, host->state);

	mmc_printk("card_simulate_plugout_cnt:%u card_is_present:%u\n"
			"mmc_trans_err_count:%u dw_dto_timer_expired:%u\n"
			"cmd_error_cnt:%u data_error_cnt:%u\n",
			host->card_simulate_plugout_cnt, host->card_is_present,
			host->mmc_trans_err_count, host->dw_dto_timer_expired_cnt,
			host->cmd_error_cnt, host->data_error_cnt);

	return ;
}
static int xlog_get(void *data, u64 *val)
{
	struct dw_mci *host = data;

	*val = host->host_id;

	return 0;
}

static int xlog_set(void *data, u64 val)
{
	struct dw_mci *host = data;

	switch (val){
		case 1:
			xlog_print_all(host->host_id);
			break;

		case 2:
			xlog_print_all(host->host_id);
			xlog_reset(host->host_id);
			break;

		case 3:
			dw_mci_regs_print(host);
			break;

		case 4:
			dw_mci_print_variable(host);
			break;

		case 5:
			break;

#ifdef DW_COMMAND_SOFT_TIMEOUT_SUPPORT
		case 6:
			dw_dto_timeout_value = 1;
			break;
#endif
		default:
			break;
	}

	return 0;
}

DEFINE_SIMPLE_ATTRIBUTE(xlog_fops, xlog_get, xlog_set, "%llu\n");

#endif

#ifdef CONFIG_ARCH_ZX297510
extern int wifi_chip_power(int on);
#else
int wifi_chip_power(int on)
{
	return 0;
}
#endif

static int card_power(struct dw_mci *host, int on)
{
	if (host->quirks & DW_MCI_QUIRK_SDIO){
		wifi_chip_power(on);
	}else{
	}

	return 0;
}


#ifdef CONFIG_ARCH_ZX297510
extern unsigned int zx29_gpio1v8_driver_strength_set(unsigned int pin_num, unsigned int value);
#endif

static unsigned int dw_mci_gpio_driver_strength_set(unsigned int pin_num, unsigned int value)
{
#ifdef CONFIG_ARCH_ZX297510
	return zx29_gpio1v8_driver_strength_set(pin_num, value);
#else
	mmc_printk("WARNING the platform have not realized gpio driver strength interface\n");
#endif

	return 0;
}

#ifdef CONFIG_ARCH_ZX297510
extern unsigned int zx29_gpio1v8_config_get(unsigned int pin_num);
#endif

static int dw_mci_gpio_config_get(unsigned int pin_num)
{
#ifdef CONFIG_ARCH_ZX297510
	return zx29_gpio1v8_config_get(pin_num);
#else
	mmc_printk("WARNING the platform have not realized the interface\n");
#endif

	return 0;
}
/* set */
static int dw_mci_clock_driver_strength_set(void *data, u64 val)
{
	struct dw_mci *host = data;
	return dw_mci_gpio_driver_strength_set(host->host_id ? PIN_SD1_CLK : PIN_SD0_CLK, val);
}
static int dw_mci_cmd_driver_strength_set(void *data, u64 val)
{
	struct dw_mci *host = data;
	return dw_mci_gpio_driver_strength_set(host->host_id ? PIN_SD1_CMD : PIN_SD0_CMD, val);
}
static int dw_mci_data0_driver_strength_set(void *data, u64 val)
{
	struct dw_mci *host = data;
	return dw_mci_gpio_driver_strength_set(host->host_id ? PIN_SD1_DATA0 : PIN_SD0_DATA0, val);
}
static int dw_mci_data1_driver_strength_set(void *data, u64 val)
{
	struct dw_mci *host = data;
	return dw_mci_gpio_driver_strength_set(host->host_id ? PIN_SD1_DATA1 : PIN_SD0_DATA1, val);
}
static int dw_mci_data2_driver_strength_set(void *data, u64 val)
{
	struct dw_mci *host = data;
	return dw_mci_gpio_driver_strength_set(host->host_id ? PIN_SD1_DATA2 : PIN_SD0_DATA2, val);
}
static int dw_mci_data3_driver_strength_set(void *data, u64 val)
{
	struct dw_mci *host = data;
	return dw_mci_gpio_driver_strength_set(host->host_id ? PIN_SD1_DATA3 : PIN_SD0_DATA3, val);
}

/* get pin all config , pu/pd, driver strength, etc. */
static int dw_mci_clock_pin_config_get(void *data, u64 *val)
{
	struct dw_mci *host = data;
	*val = dw_mci_gpio_config_get(host->host_id ? PIN_SD1_CLK : PIN_SD0_CLK);
	return 0;
}
static int dw_mci_cmd_pin_config_get(void *data, u64 *val)
{
	struct dw_mci *host = data;
	*val = dw_mci_gpio_config_get(host->host_id ? PIN_SD1_CMD : PIN_SD0_CMD);
	return 0;
}
static int dw_mci_data0_pin_config_get(void *data, u64 *val)
{
	struct dw_mci *host = data;
	*val = dw_mci_gpio_config_get(host->host_id ? PIN_SD1_DATA0 : PIN_SD0_DATA0);
	return 0;
}
static int dw_mci_data1_pin_config_get(void *data, u64 *val)
{
	struct dw_mci *host = data;
	*val = dw_mci_gpio_config_get(host->host_id ? PIN_SD1_DATA1 : PIN_SD0_DATA1);
	return 0;
}
static int dw_mci_data2_pin_config_get(void *data, u64 *val)
{
	struct dw_mci *host = data;
	*val = dw_mci_gpio_config_get(host->host_id ? PIN_SD1_DATA2 : PIN_SD0_DATA2);
	return 0;
}
static int dw_mci_data3_pin_config_get(void *data, u64 *val)
{
	struct dw_mci *host = data;
	*val = dw_mci_gpio_config_get(host->host_id ? PIN_SD1_DATA3 : PIN_SD0_DATA3);
	return 0;
}

/* set only pin driver strength, range : (0~3f)@7510, (0,1,2)@7520,7520v2,7520v5
 * get all pin config */
DEFINE_SIMPLE_ATTRIBUTE(dw_mci_pin_clk_fops, dw_mci_clock_pin_config_get, dw_mci_clock_driver_strength_set, "0x%08llx\n");
DEFINE_SIMPLE_ATTRIBUTE(dw_mci_pin_cmd_fops, dw_mci_cmd_pin_config_get, dw_mci_cmd_driver_strength_set, "0x%08llx\n");
DEFINE_SIMPLE_ATTRIBUTE(dw_mci_pin_data0_fops, dw_mci_data0_pin_config_get, dw_mci_data0_driver_strength_set, "0x%08llx\n");
DEFINE_SIMPLE_ATTRIBUTE(dw_mci_pin_data1_fops, dw_mci_data1_pin_config_get, dw_mci_data1_driver_strength_set, "0x%08llx\n");
DEFINE_SIMPLE_ATTRIBUTE(dw_mci_pin_data2_fops, dw_mci_data2_pin_config_get, dw_mci_data2_driver_strength_set, "0x%08llx\n");
DEFINE_SIMPLE_ATTRIBUTE(dw_mci_pin_data3_fops, dw_mci_data3_pin_config_get, dw_mci_data3_driver_strength_set, "0x%08llx\n");

static void dw_mci_pin_config_debugfs(struct dw_mci_slot *slot)
{
	struct mmc_host	*mmc = slot->mmc;
	struct dw_mci *host = slot->host;
	struct dentry *root;

	root = mmc->debugfs_root;
	if (!root)
		return;

	if (!debugfs_create_file("dw_mci_pin_clk_fops", S_IRUGO | S_IWUSR, root, host, &dw_mci_pin_clk_fops))
		goto err;
	if (!debugfs_create_file("dw_mci_pin_cmd_fops", S_IRUGO | S_IWUSR, root, host, &dw_mci_pin_cmd_fops))
		goto err;
	if (!debugfs_create_file("dw_mci_pin_data0_fops", S_IRUGO | S_IWUSR, root, host, &dw_mci_pin_data0_fops))
		goto err;
	if (!debugfs_create_file("dw_mci_pin_data1_fops", S_IRUGO | S_IWUSR, root, host, &dw_mci_pin_data1_fops))
		goto err;
	if (!debugfs_create_file("dw_mci_pin_data2_fops", S_IRUGO | S_IWUSR, root, host, &dw_mci_pin_data2_fops))
		goto err;
	if (!debugfs_create_file("dw_mci_pin_data3_fops", S_IRUGO | S_IWUSR, root, host, &dw_mci_pin_data3_fops))
		goto err;

	return ;

err:
	dev_err(&mmc->class_dev, "failed to initialize pin config debugfs for slot\n");
}

static void dw_mci_init_debugfs(struct dw_mci_slot *slot)
{
	struct mmc_host	*mmc = slot->mmc;
	struct dw_mci *host = slot->host;
	struct dentry *root;
	struct dentry *node;

	root = mmc->debugfs_root;
	if (!root)
		return;

#ifdef XLOG_MMC_ENABLE
	node = debugfs_create_file("xlog", S_IRUGO | S_IWUSR, root,
				   host, &xlog_fops);
	if (!node)
		goto err;
#endif

	dw_mci_pin_config_debugfs(slot);
	node = debugfs_create_file("regs", S_IRUSR, root, host,
				   &dw_mci_regs_fops);
	if (!node)
		goto err;

	node = debugfs_create_file("req", S_IRUSR, root, slot,
				   &dw_mci_req_fops);
	if (!node)
		goto err;

	node = debugfs_create_file("present", S_IRUSR, root, slot,
				   &dw_mci_present_fops);
	if (!node)
		goto err;

	node = debugfs_create_u32("state", S_IRUSR, root, (u32 *)&host->state);
	if (!node)
		goto err;

	node = debugfs_create_x32("pending_events", S_IRUSR, root,
				  (u32 *)&host->pending_events);
	if (!node)
		goto err;

	node = debugfs_create_x32("completed_events", S_IRUSR, root,
				  (u32 *)&host->completed_events);
	if (!node)
		goto err;
	node = debugfs_create_u32("data_error_cnt", S_IRUSR, root, (u32 *)&host->data_error_cnt);
	if (!node)
		goto err;

	node = debugfs_create_x32("cmd_error_cnt", S_IRUSR, root, (u32 *)&host->cmd_error_cnt);
	if (!node)
		goto err;

	node = debugfs_create_x32("dw_dto_timer_expired_cnt", S_IRUSR | S_IWUSR,
										root, &host->dw_dto_timer_expired_cnt);

	if (!node)
		goto err;

	node = debugfs_create_x32("updata_timeout_cnt", S_IRUSR, root,
				(u32 *)&updata_timeout);
	if (!node)
		goto err;

	return;

err:
	dev_err(&mmc->class_dev, "failed to initialize debugfs for slot\n");
}
#endif /* defined(CONFIG_DEBUG_FS) */

static void dw_mci_set_timeout(struct dw_mci *host)
{
#ifdef DW_COMMAND_SOFT_TIMEOUT_SUPPORT
	unsigned int tmo_data;

	/* half of soft timer timeout */
	tmo_data = (host->current_speed * DW_DTO_TMO_SEC) >> 1;
	if (tmo_data > 0xFFFFFF) {
		/* timeout (maximum) */
		tmo_data = 0xFFFFFF;
	}
	mci_writel(host, TMOUT, (tmo_data << 8) | 0xFF);
#else
	/* timeout (maximum) */
	mci_writel(host, TMOUT, 0xffffffff);
#endif
}

static u32 dw_mci_prepare_command(struct mmc_host *mmc, struct mmc_command *cmd)
{
	struct mmc_data	*data;
	struct dw_mci_slot *slot = mmc_priv(mmc);
	const struct dw_mci_drv_data *drv_data = slot->host->drv_data;
	u32 cmdr;
	cmd->error = -EINPROGRESS;

	cmdr = cmd->opcode;

	if (cmd->opcode == MMC_STOP_TRANSMISSION ||
	    cmd->opcode == MMC_GO_IDLE_STATE ||
	    cmd->opcode == MMC_GO_INACTIVE_STATE ||
	    (cmd->opcode == SD_IO_RW_DIRECT &&
	     ((cmd->arg >> 9) & 0x1FFFF) == SDIO_CCCR_ABORT))
		cmdr |= SDMMC_CMD_STOP;
	else if (cmd->data)		//if (cmd->opcode != MMC_SEND_STATUS && cmd->data)
			cmdr |= SDMMC_CMD_PRV_DAT_WAIT;

	if (cmd->flags & MMC_RSP_PRESENT) {
		/* We expect a response, so set this bit */
		cmdr |= SDMMC_CMD_RESP_EXP;
		if (cmd->flags & MMC_RSP_136)
			cmdr |= SDMMC_CMD_RESP_LONG;
	}
	if(!(slot->host->quirks & DW_MCI_QUIRK_SDIO) || !IS_HIGHSPEED(slot->host->pdata->caps)||(slot->host->quirks & DW_MCI_QUIRK_CLK_PHASE_TURN))
		cmdr |= SDMMC_CMD_USE_HOLD_REG;

	if (cmd->flags & MMC_RSP_CRC)
		cmdr |= SDMMC_CMD_RESP_CRC;

	data = cmd->data;
	if (data) {
		cmdr |= SDMMC_CMD_DAT_EXP;
		if (data->flags & MMC_DATA_STREAM)
			cmdr |= SDMMC_CMD_STRM_MODE;
		if (data->flags & MMC_DATA_WRITE)
			cmdr |= SDMMC_CMD_DAT_WR;
	}

	if (drv_data && drv_data->prepare_command)
		drv_data->prepare_command(slot->host, &cmdr);

	return cmdr;
}

static u32 dw_mci_prep_stop_abort(struct dw_mci *host, struct mmc_command *cmd)
{
	struct mmc_command *stop;
	u32 cmdr;

	if (!cmd->data)
		return 0;

	stop = &host->stop_abort;
	cmdr = cmd->opcode;
	memset(stop, 0, sizeof(struct mmc_command));

	if (cmdr == MMC_READ_SINGLE_BLOCK ||
	    cmdr == MMC_READ_MULTIPLE_BLOCK ||
	    cmdr == MMC_WRITE_BLOCK ||
	    cmdr == MMC_WRITE_MULTIPLE_BLOCK) {
		stop->opcode = MMC_STOP_TRANSMISSION;
		stop->arg = 0;
		stop->flags = MMC_RSP_R1B | MMC_CMD_AC;
	} else if (cmdr == SD_IO_RW_EXTENDED) {
		stop->opcode = SD_IO_RW_DIRECT;
		stop->arg |= (1 << 31) | (0 << 28) | (SDIO_CCCR_ABORT << 9) |
			     ((cmd->arg >> 28) & 0x7);
		stop->flags = MMC_RSP_SPI_R5 | MMC_RSP_R5 | MMC_CMD_AC;
	} else {
		return 0;
	}

	cmdr = stop->opcode | SDMMC_CMD_STOP |
		SDMMC_CMD_RESP_CRC | SDMMC_CMD_RESP_EXP;

	return cmdr;
}

static void dw_mci_start_command(struct dw_mci *host,
				 struct mmc_command *cmd, u32 cmd_flags)
{
	host->cmd = cmd;

	if (cmd->data){
		xlog_mmc_cmd(host->host_id, cmd->opcode, cmd->arg, cmd->data->blksz | (cmd->data->blocks<<16), cmd_flags);
	}else{
		xlog_mmc_cmd(host->host_id, cmd->opcode, cmd->arg, 0, cmd_flags);
	}

	mci_writel(host, CMDARG, cmd->arg);
	wmb();

	mci_writel(host, CMD, cmd_flags | SDMMC_CMD_START);
}

static void dw_mci_mask_cmdint(struct dw_mci*host, u32 value);

static inline void send_stop_abort(struct dw_mci *host, struct mmc_data *data)
{
	//struct mmc_command *stop = data->stop ? data->stop : &host->stop_abort;

	dw_mci_mask_cmdint(host, 0);
	//dw_mci_start_command(host, stop, host->stop_cmdr);

	if (data->stop)
		dw_mci_start_command(host, data->stop, host->stop_cmdr);
	else if (host->stop_abort.opcode)
		dw_mci_start_command(host, &host->stop_abort, host->stop_cmdr);
	else
		set_bit(EVENT_CMD_COMPLETE, &host->pending_events);
}

/* DMA interface functions */
static void dw_mci_stop_dma(struct dw_mci *host)
{
	if (host->using_dma) {
		host->dma_ops->stop(host);
		host->dma_ops->cleanup(host);
	}

	/* Data transfer was stopped by the interrupt handler */
	set_bit(EVENT_XFER_COMPLETE, &host->pending_events);
}

static int dw_mci_get_dma_dir(struct mmc_data *data)
{
	if (data->flags & MMC_DATA_WRITE)
		return DMA_TO_DEVICE;
	else
		return DMA_FROM_DEVICE;
}

#ifdef CONFIG_MMC_DW_IDMAC
static void dw_mci_dma_cleanup(struct dw_mci *host)
{
	struct mmc_data *data = host->data;

	if (data)
		if (!data->host_cookie)
			dma_unmap_sg(host->dev,
				     data->sg,
				     data->sg_len,
				     dw_mci_get_dma_dir(data));
}

static void dw_mci_idmac_reset(struct dw_mci *host)
{
	u32 bmod = mci_readl(host, BMOD);
	/* Software reset of DMA */
	bmod |= SDMMC_IDMAC_SWRESET;
	mci_writel(host, BMOD, bmod);
}

static void dw_mci_idmac_stop_dma(struct dw_mci *host)
{
	u32 temp;

	/* Disable and reset the IDMAC interface */
	temp = mci_readl(host, CTRL);
	temp &= ~SDMMC_CTRL_USE_IDMAC;
	temp |= SDMMC_CTRL_DMA_RESET;
	mci_writel(host, CTRL, temp);

	/* Stop the IDMAC running */
	temp = mci_readl(host, BMOD);
	temp &= ~(SDMMC_IDMAC_ENABLE | SDMMC_IDMAC_FB);
	temp |= SDMMC_IDMAC_SWRESET; //FIXIT
	mci_writel(host, BMOD, temp);
}

static void dw_mci_idmac_complete_dma(struct dw_mci *host)
{
	struct mmc_data *data = host->data;

	dev_vdbg(host->dev, "DMA complete\n");

	//host->dma_ops->stop(host);
	host->dma_ops->cleanup(host);

	/*
	 * If the card was removed, data will be NULL. No point in trying to
	 * send the stop command or waiting for NBUSY in this case.
	 */
	if (data) {
		set_bit(EVENT_XFER_COMPLETE, &host->pending_events);
#ifdef DW_MCI_THREAD
		dw_mci_kick_thread(host);
#else
		tasklet_schedule(&host->tasklet);
#endif
	}
}

static void dw_mci_translate_sglist(struct dw_mci *host, struct mmc_data *data,
				    unsigned int sg_len)
{
	int i,j;
	struct idmac_desc *desc = host->idma_des;
	unsigned int length = 0;
	u32 mem_addr = 0;
	unsigned int length_transed = 0;

	dw_mci_idmac_reset(host);

	for (i = 0, j = 0; i < sg_len; i++) {
		length = sg_dma_len(&data->sg[i]);
		mem_addr = sg_dma_address(&data->sg[i]);
		//xlog_mmc_log(host->host_id, "debug log, remove later {length, i/nr}", length, i, sg_len, 0);

		if (host->quirks & DW_MCI_QUIRK_UNALIGN_DMA_SZ) {
			length = (length+3) &(~0x3U);
		}
		do {
#ifdef CONFIG_RTL8192CD
            if(host->host_id == 0){
                if(length%4){
                    length = ((length >> 2)  + 1 ) << 2;
                }
            }
#endif
			length_transed = min(length, (unsigned int)DW_DMA_DESC_TRANS_LEN);

			/* Set the OWN bit and disable interrupts for this descriptor */
			desc->des0 = IDMAC_DES0_OWN | IDMAC_DES0_DIC | IDMAC_DES0_CH;

			/* Buffer length */
			IDMAC_SET_BUFFER1_SIZE(desc, length_transed);

			/* Physical address to DMA to/from */
			desc->des2 = mem_addr;
			mem_addr = mem_addr+length_transed;
			length -=length_transed;
			desc++;
			j++;
		} while (length);
	}

	if (j == 1) {
		/* Set first & last descriptor */
		desc = host->idma_des;
		desc->des0 = IDMAC_DES0_OWN | IDMAC_DES0_FD | IDMAC_DES0_LD;
	} else {
		/* Set first descriptor */
		desc = host->idma_des;
		desc->des0 = IDMAC_DES0_OWN | IDMAC_DES0_DIC | IDMAC_DES0_CH | IDMAC_DES0_FD;

		/* Set last descriptor */
	    desc = host->idma_des + (j - 1);
		desc->des0 = IDMAC_DES0_OWN | IDMAC_DES0_LD;
		//desc->des0 &= ~(IDMAC_DES0_CH | IDMAC_DES0_DIC);
		//desc->des0 |= IDMAC_DES0_LD;
	}

	wmb();
}

static void dw_mci_idmac_start_dma(struct dw_mci *host, unsigned int sg_len)
{
	u32 temp;

	dw_mci_translate_sglist(host, host->data, sg_len);

	/* Select IDMAC interface */
	temp = mci_readl(host, CTRL);
	temp |= SDMMC_CTRL_USE_IDMAC;
	mci_writel(host, CTRL, temp);

	wmb();

	/* Enable the IDMAC */
	temp = mci_readl(host, BMOD);
	temp |= SDMMC_IDMAC_ENABLE | SDMMC_IDMAC_FB;
	mci_writel(host, BMOD, temp);

	/* Start it running */
	mci_writel(host, PLDMND, 1);
}

static int dw_mci_idmac_init(struct dw_mci *host)
{
	struct idmac_desc *p;
	int i;

	/* Number of descriptors in the ring buffer */
	host->ring_size = PAGE_SIZE / sizeof(struct idmac_desc);

	/* Forward link the descriptor list */
	for (i = 0, p = host->sg_cpu; i < host->ring_size - 1; i++, p++)
		p->des3 = host->sg_dma +  (i + 1)*(sizeof(struct idmac_desc));

	/* Set the last descriptor as the end-of-ring descriptor */
	p->des3 = (u32)host->sg_dma;
	p->des0 = IDMAC_DES0_ER;

	dw_mci_idmac_reset(host);

	/* Mask out interrupts - get Tx & Rx complete only */
//	mci_writel(host, IDSTS, IDMAC_INT_CLR); //why
	mci_writel(host, IDINTEN, SDMMC_IDMAC_INT_NI | SDMMC_IDMAC_INT_RI |
		   SDMMC_IDMAC_INT_TI);

	/* Set the descriptor base address */
	//mci_writel(host, DBADDR, (u32)host->idma_des);
	mci_writel(host, DBADDR, (u32)host->sg_dma);
	return 0;
}

static const struct dw_mci_dma_ops dw_mci_idmac_ops = {
	.init = dw_mci_idmac_init,
	.start = dw_mci_idmac_start_dma,
	.stop = dw_mci_idmac_stop_dma,
	.complete = dw_mci_idmac_complete_dma,
	.cleanup = dw_mci_dma_cleanup,
};
#endif /* CONFIG_MMC_DW_IDMAC */
#ifdef CONFIG_MMC_ZX29_EDMAC
static void dw_mci_edma_cleanup(struct dw_mci *host)
{
	struct mmc_data *data = host->data;

	if(data)
	{
		if (data->phy_flag)
			;//   dma_unmap_single(&host->dev,data->sg->dma_address,data->blocks * data->blksz,dw_mci_get_dma_dir(data));
		else
			dma_unmap_sg(host->dev,
					data->sg,
					data->sg_len,
					dw_mci_get_dma_dir(data));
	}


}

static void dw_mci_edmac_stop_dma(struct dw_mci *host)
{
	if(!host->data)
		return;
	if(host->data->flags == MMC_DATA_WRITE)
		zx29_dma_stop(host->channel_id_tx);
	else if(host->data->flags == MMC_DATA_READ)
		zx29_dma_stop(host->channel_id_rx);

}

static void dw_mci_edmac_loop_dma(dma_peripheral_id channel, dma_int_status dmaStatus, void* hostin)
{
	struct dw_mci* host = (struct dw_mci* )hostin;
	struct mmc_data *data = host->data;
	dma_chan_def  *tChanCfg= NULL;
	dma_peripheral_id channel_id=0;
	u32 tempval = 0;
	int ret=0;


	if(data == NULL)
	{
		return ;
	}
	if(data->sg_len > host->tran_dma_cnt)
	{
		//Config DMA
		if (data->flags == MMC_DATA_WRITE){
			tChanCfg=&(host->tChanPar_TX);
			tChanCfg->SrcAddr=sg_dma_address(&data->sg[host->tran_dma_cnt]);
			tChanCfg->DestAddr =( host->sdmmc_addr + SDMMC_FIFO);
			tChanCfg->CONTROL.SrcMod = DMA_ADDRMOD_RAM;
			tChanCfg->CONTROL.DestMod = DMA_ADDRMOD_FIFO;
			tChanCfg->CONTROL.SrcBurstLen = DMA_BURST_LEN_8;
			tChanCfg->CONTROL.DestBurstLen = DMA_BURST_LEN_8;

			channel_id= host->channel_id_tx;

			tempval = (sg_dma_len(&data->sg[host->tran_dma_cnt])+31)>>5;
			tempval = tempval<<5;
			tChanCfg->Count = tempval;

			//printk("write dma_addr is %08x\n",tChanCfg->SrcAddr);
		}
		else if (data->flags == MMC_DATA_READ){
			tChanCfg=&(host->tChanPar_RX);
			tChanCfg->SrcAddr = (host->sdmmc_addr + SDMMC_FIFO);
			tChanCfg->DestAddr=sg_dma_address(&data->sg[host->tran_dma_cnt]);
			tChanCfg->CONTROL.SrcMod = DMA_ADDRMOD_FIFO;
			tChanCfg->CONTROL.DestMod = DMA_ADDRMOD_RAM;
			channel_id = host->channel_id_rx;

			tChanCfg->Count = sg_dma_len(&data->sg[host->tran_dma_cnt]);
			//printk("read dma_addr is %08x\n",tChanCfg->DestAddr);
		}
		tChanCfg->CallBack = dw_mci_edmac_loop_dma;
		tChanCfg->data=host;

		/*tempval = (sg_dma_len(&data->sg[host->tran_dma_cnt])+31)>>5;
		  tempval = tempval<<5;
		  tChanCfg->Count = tempval;*/
		//tChanCfg->Count = sg_dma_len(&data->sg[host->tran_dma_cnt]);
		ret=zx29_dma_config(channel_id, tChanCfg);
		zx29_dma_start(channel_id);
		host->tran_dma_cnt ++;
		return ;
	}

	host->tran_dma_cnt = 0;
	if (data) {
		dw_mci_edmac_stop_dma(host);
		// xsh??? dw_mci_edma_cleanup(host);
		set_bit(EVENT_XFER_COMPLETE, &host->pending_events);
#ifdef DW_MCI_THREAD
		dw_mci_kick_thread(host);
#else
		tasklet_schedule(&host->tasklet);
#endif
	}
}


static void dw_mci_edmac_start_dma(struct dw_mci *host, unsigned int sg_len)
{
	u32 nTempCtrlReg;
	dma_peripheral_id channel_id=0;

	//Reset DMA in Sd Controller
	nTempCtrlReg = mci_readl(host, CTRL);
	nTempCtrlReg |= (SDMMC_CTRL_DMA_RESET|SDMMC_CTRL_FIFO_RESET);
	mci_writel(host, CTRL, nTempCtrlReg);
	do
	{
		nTempCtrlReg = mci_readl(host, CTRL);
	}
	while (nTempCtrlReg & (SDMMC_CTRL_DMA_RESET|SDMMC_CTRL_FIFO_RESET));

	//Enable DMA in Sd Controller
	nTempCtrlReg = mci_readl(host, CTRL);
	nTempCtrlReg |= SDMMC_CTRL_DMA_ENABLE;
	mci_writel(host, CTRL, nTempCtrlReg);
	dw_mci_edmac_loop_dma(channel_id, 0 ,host);
}

static int dw_mci_edmac_init(struct dw_mci *host)
{
	u32 nTempCtrlReg;
	unsigned int *channel_id = &(host->channel_id_tx);
	dma_chan_def *tChanPar =&(host->tChanPar_TX);
	dma_peripheral_id peripheralID = DMAC0_CH_SD0_TX;
	u32 i=0;

	peripheralID = host->channel_id_tx;
	//Disable DMA in Sd Controller
	nTempCtrlReg = mci_readl(host, CTRL);
	nTempCtrlReg &= ~(SDMMC_CTRL_DMA_ENABLE);
	mci_writel(host, CTRL, nTempCtrlReg);
	for (i=0; i<2; i++)
	{
		*channel_id=zx29_dma_request(peripheralID);
		if (*channel_id<0)
			dev_err(host->dev, "%s: can not alloc the dma chan\n",
					__func__);

		tChanPar->SrcAddr = 0;
		tChanPar->DestAddr = 0;
		tChanPar->Count = 0;
		tChanPar->LLI=0;

		tChanPar->CONTROL.BurstReqMod=DMA_PERIPHERAL_REQ;
		tChanPar->CONTROL.SrcMod=DMA_ADDRMOD_RAM;
		tChanPar->CONTROL.DestMod=DMA_ADDRMOD_RAM;
		tChanPar->CONTROL.SrcBurstSize=DMA_BURST_SIZE_32BIT;
		tChanPar->CONTROL.SrcBurstLen=DMA_BURST_LEN_8;
		tChanPar->CONTROL.DestBurstSize=DMA_BURST_SIZE_32BIT;
		tChanPar->CONTROL.DestBurstLen=DMA_BURST_LEN_8;

		tChanPar->CONTROL.IrqMod=DMA_ALL_IRQ_ENABLE;
		tChanPar->CONTROL.IntSel=DMA_INT_TO_A9;

		channel_id=&(host->channel_id_rx);
		tChanPar=&(host->tChanPar_RX);
		peripheralID=host->channel_id_rx;
	}
	host->tran_dma_cnt = 0;
	return 0;

}

static struct dw_mci_dma_ops dw_mci_edmac_ops = {
	.init = dw_mci_edmac_init,
	.start = dw_mci_edmac_start_dma,
	.stop = dw_mci_edmac_stop_dma,
	.complete = NULL,
	.cleanup = dw_mci_edma_cleanup,
};

#endif /* CONFIG_MMC_ZX29_EDMAC */

static int dw_mci_pre_dma_transfer(struct dw_mci *host,
				   struct mmc_data *data,
				   bool next)
{
	struct scatterlist *sg;
	unsigned int i, sg_len;

	if (!next && data->host_cookie)
		return data->host_cookie;

	/*
	 * We don't do DMA on "complex" transfers, i.e. with
	 * non-word-aligned buffers or lengths. Also, we don't bother
	 * with all the DMA setup overhead for short transfers.
	 */
	if (data->blocks * data->blksz < DW_MCI_DMA_THRESHOLD){
		//xlog_mmc_log(host->host_id, "ERROR ERROR ERROR sending data length is less then 4", 0, 0, 0);
		return -EINVAL;
	}

	if (chk_sg_offset_len(host))
		if (!(host->quirks & DW_MCI_QUIRK_UNALIGN_DMA_SZ) && (data->blksz & 3))  {
			xlog_mmc_log(host->host_id, "ERROR ERROR ERROR unalign data len", 0, 0, 0);
				return -EINVAL;
		}

	if (chk_sg_offset_len(host)) {
		for_each_sg(data->sg, sg, data->sg_len, i) {
			if (!(host->quirks & DW_MCI_QUIRK_UNALIGN_DMA_START) && (sg->offset & 3)) {
				dev_err(host->dev, "%s: Invalid sg offset %x length%x\n", __func__, sg->offset, sg->length);
				return -EINVAL;
			}
			if (!(host->quirks & DW_MCI_QUIRK_UNALIGN_DMA_SZ) && (sg->length& 3))
				return -EINVAL;
		}
	}

	sg_len = dma_map_sg(host->dev,
		data->sg,
		data->sg_len,
		dw_mci_get_dma_dir(data));

	if (sg_len == 0)
		return -EINVAL;

	if (next)
		data->host_cookie = sg_len;

	return sg_len;
}

static void dw_mci_pre_req(struct mmc_host *mmc,
			   struct mmc_request *mrq,
			   bool is_first_req)
{
	struct dw_mci_slot *slot = mmc_priv(mmc);
	struct mmc_data *data = mrq->data;

	if (!slot->host->use_dma || !data)
		return;

	if (data->host_cookie) {
		data->host_cookie = 0;
		return;
	}

	if (dw_mci_pre_dma_transfer(slot->host, mrq->data, 1) < 0)
		data->host_cookie = 0;
}

static void dw_mci_post_req(struct mmc_host *mmc,
			    struct mmc_request *mrq,
			    int err)
{
	struct dw_mci_slot *slot = mmc_priv(mmc);
	struct mmc_data *data = mrq->data;

	if (!slot->host->use_dma || !data)
		return;

	if (data->host_cookie)
		dma_unmap_sg(slot->host->dev,
			     data->sg,
			     data->sg_len,
			     dw_mci_get_dma_dir(data));
	data->host_cookie = 0;
}

static void dw_mci_adjust_fifoth(struct dw_mci *host, struct mmc_data *data)
{
#ifdef CONFIG_MMC_DW_IDMAC
	unsigned int blksz = data->blksz;
	const u32 mszs[] = {1, 4, 8, 16, 32, 64, 128, 256};
	u32 fifo_width = 1 << host->data_shift;
	u32 blksz_depth = blksz / fifo_width, fifoth_val;
	u32 msize = 0, rx_wmark = 0, tx_wmark, tx_wmark_invers;
	int idx = (sizeof(mszs) / sizeof(mszs[0])) - 1;

	tx_wmark = (host->fifo_depth) / 2;
	tx_wmark_invers = host->fifo_depth - tx_wmark;

	/*
	 * MSIZE is '1',
	 * if blksz is not a multiple of the FIFO width
	 */
	if (blksz % fifo_width) {
		goto done;
	}

	do {
		if (!((blksz_depth % mszs[idx]) ||
		     (tx_wmark_invers % mszs[idx]))) {
			msize = idx;
			rx_wmark = mszs[idx] - 1;
			break;
		}
	} while (--idx > 0);
	/*
	 * If idx is '0', it won't be tried
	 * Thus, initial values are uesed
	 */
done:
	fifoth_val = SDMMC_SET_FIFOTH(msize, rx_wmark, tx_wmark);
	mci_writel(host, FIFOTH, fifoth_val);
#endif
}

static void dw_mci_ctrl_rd_thld(struct dw_mci *host, struct mmc_data *data)
{
	unsigned int blksz = data->blksz;
	u32 blksz_depth, fifo_depth;
	u16 thld_size;

	WARN_ON(!(data->flags & MMC_DATA_READ));

	if (host->verid < DW_MMC_240A)
		return;
	if (host->timing != MMC_TIMING_MMC_HS200 &&
	    host->timing != MMC_TIMING_UHS_SDR104)
		goto disable;

	blksz_depth = blksz / (1 << host->data_shift);
	fifo_depth = host->fifo_depth;

	if (blksz_depth > fifo_depth)
		goto disable;

	/*
	 * If (blksz_depth) >= (fifo_depth >> 1), should be 'thld_size <= blksz'
	 * If (blksz_depth) <  (fifo_depth >> 1), should be thld_size = blksz
	 * Currently just choose blksz.
	 */
	thld_size = blksz;
	mci_writel(host, CDTHRCTL, SDMMC_SET_RD_THLD(thld_size, 1));
	return;

disable:
	mci_writel(host, CDTHRCTL, SDMMC_SET_RD_THLD(0, 0));
}

static int dw_mci_submit_data_dma(struct dw_mci *host, struct mmc_data *data)
{
	int sg_len;
	u32 temp;

	host->using_dma = 0;

	/* If we don't have a channel, we can't do DMA */
	if (!host->use_dma)
		return -ENODEV;

	sg_len = dw_mci_pre_dma_transfer(host, data, 0);
	if (sg_len < 0) {
		host->dma_ops->stop(host);
		return sg_len;
	}

	host->using_dma = 1;

	dev_vdbg(host->dev,
		 "sd sg_cpu: %#lx sg_dma: %#lx sg_len: %d\n",
		 (unsigned long)host->sg_cpu, (unsigned long)host->sg_dma,
		 sg_len);

	/*
	 * Decide the MSIZE and RX/TX Watermark.
	 * If current block size is same with previous size,
	 * no need to update fifoth.
	 */
	if (host->prev_blksz != data->blksz)
		dw_mci_adjust_fifoth(host, data);

	/* Make sure to reset DMA in case we did PIO before this. Refer to Linux 4.19 */
	if (!host->prev_blksz)
		dw_mci_ctrl_reset(host, SDMMC_CTRL_DMA_RESET);
#ifdef CONFIG_MMC_ZX29_EDMAC
	/* Enable the DMA interface */
	temp = mci_readl(host, CTRL);
	temp |= SDMMC_CTRL_DMA_ENABLE;
	mci_writel(host, CTRL, temp);
#endif
	/* Disable RX/TX IRQs, let DMA handle it */
	temp = mci_readl(host, INTMASK);
	temp  &= ~(SDMMC_INT_RXDR | SDMMC_INT_TXDR);
	mci_writel(host, INTMASK, temp);

	host->dma_ops->start(host, sg_len);

	return 0;
}

static void dw_mci_submit_data(struct dw_mci *host, struct mmc_data *data)
{
	u32 temp;

	data->error = -EINPROGRESS;

	WARN_ON(host->data);
	host->sg = NULL;
	host->data = data;

	if (data->flags & MMC_DATA_READ) {
		host->dir_status = DW_MCI_RECV_STATUS;
		dw_mci_ctrl_rd_thld(host, data);
	} else {
		host->dir_status = DW_MCI_SEND_STATUS;
	}

	if (dw_mci_submit_data_dma(host, data)) {
		int flags = SG_MITER_ATOMIC;
		if (host->data->flags & MMC_DATA_READ)
			flags |= SG_MITER_TO_SG;
		else
			flags |= SG_MITER_FROM_SG;

		sg_miter_start(&host->sg_miter, data->sg, data->sg_len, flags);
		host->sg = data->sg;
		host->part_buf_start = 0;
		host->part_buf_count = 0;

		mci_writel(host, RINTSTS, SDMMC_INT_TXDR | SDMMC_INT_RXDR);
		temp = mci_readl(host, INTMASK);
		temp |= SDMMC_INT_TXDR | SDMMC_INT_RXDR;
		mci_writel(host, INTMASK, temp);
#ifdef CONFIG_MMC_ZX29_EDMAC
		temp = mci_readl(host, CTRL);
		temp &= ~SDMMC_CTRL_DMA_ENABLE;
		mci_writel(host, CTRL, temp);
#endif
		/*
		 * Use the initial fifoth_val for PIO mode.
		 * If next issued data may be transfered by DMA mode,
		 * prev_blksz should be invalidated.
		 */
#if 1
		dw_mci_adjust_fifoth(host, data);
#else
		mci_writel(host, FIFOTH, host->fifoth_val);
#endif
		host->prev_blksz = 0;

		if (host->dir_status == DW_MCI_SEND_STATUS) {
			unsigned long flags;

			local_irq_save(flags);
			dw_mci_write_data_pio(host);
			local_irq_restore(flags);
		}
	} else {
		/*
		 * Keep the current block size.
		 * It will be used to decide whether to update
		 * fifoth register next time.
		 */
		host->prev_blksz = data->blksz;
	}
}
static s32 mci_send_cmd(struct dw_mci_slot *slot, u32 cmd, u32 arg)
{
	struct dw_mci *host = slot->host;
	unsigned long timeout = jiffies + msecs_to_jiffies(500);
	unsigned int cmd_status = 0;

	xlog_mmc_cmd(host->host_id, cmd, arg, 0, cmd);
	mci_writel(host, CMDARG, arg);
	wmb();
	mci_writel(host, CMD, SDMMC_CMD_START | cmd);

	while (time_before(jiffies, timeout)) {
		cmd_status = mci_readl(host, CMD);
		if (!(cmd_status & SDMMC_CMD_START))
			return 0;
	}
	xlog_mmc_log(host->host_id, "INFO mci_send_cmd TIMEOUT", cmd, arg, cmd_status);	
	sc_debug_info_record(MODULE_ID_AP_MMC, "Timeout sending command (cmd %#x arg %#x status %#x)\n",
		cmd, arg, cmd_status);
	dev_err(&slot->mmc->class_dev,
		"Timeout sending command (cmd %#x arg %#x status %#x)\n",
		cmd, arg, cmd_status);
	updata_timeout++;
	if(updata_timeout>0xffff)
		updata_timeout =0;
	return -1;
}

static void dw_mci_setup_bus(struct dw_mci_slot *slot, bool force_clkinit)
{
	struct dw_mci *host = slot->host;
	unsigned int clock = slot->clock;
	u32 div;
	u32 clk_en_a;

	if (!clock) {
		xlog_mmc_log_key(host->host_id, "INFO dw_mci_setup_bus CLKENA 0", 0, 0, 0);
		mci_writel(host, CLKENA, 0);
		mci_send_cmd(slot,
			     SDMMC_CMD_UPD_CLK | SDMMC_CMD_PRV_DAT_WAIT, 0);
	} else if (clock != host->current_speed || force_clkinit) {
		xlog_mmc_log_key(host->host_id, "INFO dw_mci_setup_bus CLKENA 1", 0, 0, 0);
		div = host->bus_hz / clock;
		if (host->bus_hz % clock && host->bus_hz > clock)
			/*
			 * move the + 1 after the divide to prevent
			 * over-clocking the card.
			 */
			div += 1;

		div = (host->bus_hz != clock) ? DIV_ROUND_UP(div, 2) : 0;

		if (clock != slot->__clk_old || force_clkinit)
			dev_info(&slot->mmc->class_dev,
				 "Bus speed (slot %d) = %dHz (slot req %dHz, actual %dHZ div = %d)\n",
				 slot->id, host->bus_hz, clock,
				 div ? ((host->bus_hz / div) >> 1) :
				 host->bus_hz, div);

		/* disable clock */
		mci_writel(host, CLKENA, 0);
		mci_writel(host, CLKSRC, 0);

		/* inform CIU */
		mci_send_cmd(slot,
			     SDMMC_CMD_UPD_CLK | SDMMC_CMD_PRV_DAT_WAIT, 0);

		/* set clock to desired speed */
		mci_writel(host, CLKDIV, div);

		/* inform CIU */
		mci_send_cmd(slot,
			     SDMMC_CMD_UPD_CLK | SDMMC_CMD_PRV_DAT_WAIT, 0);

		/* enable clock; only low power if SDIO*/
		clk_en_a = SDMMC_CLKEN_ENABLE << slot->id;

		if ((host->quirks & DW_MCI_QUIRK_SDIO)) {
			/* Enable low power unless using SDIO data1 IRQ*/
			if (!(mci_readl(host, INTMASK) & SDMMC_INT_SDIO(slot->id))) {
				clk_en_a |= SDMMC_CLKEN_LOW_PWR << slot->id;
//				dev_err(host->dev, "%s: sdlowpower en %d\n",__func__,slot->id);
			}
		}
		mci_writel(host, CLKENA, clk_en_a);

		/* inform CIU */
		mci_send_cmd(slot,
			     SDMMC_CMD_UPD_CLK | SDMMC_CMD_PRV_DAT_WAIT, 0);

		/* keep the clock with reflecting clock dividor */
		slot->__clk_old = clock;
	}else {
		xlog_mmc_log_key(host->host_id, "INFO dw_mci_setup_bus do nothing", 0, 0, 0);
	}

	host->current_speed = clock;

	/* Set the current slot bus width */
	mci_writel(host, CTYPE, (slot->ctype << slot->id));
}

static unsigned long int mmc_busy_check_count, mmc_busy_check_max_count;
static s32 mmc_busy(struct dw_mci *host)
{
	unsigned int status = 0;
	unsigned long timeout = jiffies + msecs_to_jiffies(500);

        mmc_busy_check_count = 1;
        do {
                status = mci_readl(host, STATUS);
                if (!(status & SDMMC_DATA_BUSY)){
                        if (mmc_busy_check_max_count < mmc_busy_check_count){
                                mmc_busy_check_max_count = mmc_busy_check_count;
                        }
                        return 0 ;
                }
                mmc_busy_check_count++;
                if (mmc_busy_check_count < 100){
                        // udelay(10);
                }else {
                        msleep(10);
                }
        } while (time_before(jiffies, timeout));

        dev_err(host->dev, "%s: SDIO device busy timeout,status is 0x%08x\n",__func__,status);
		sc_debug_info_record(MODULE_ID_AP_MMC,"%s(%d): timeout,status is 0x%08x\n",__func__,host->host_id,status);

        return -1;

}
void enable_disableclk(struct mmc_host *mmc, int enable)
{
	u32 clk_en_a;
	struct dw_mci_slot *slot = mmc_priv(mmc);
	s32 ret ;
	u8 i = 3;

	ret = mmc_busy(slot->host);
	if(ret<0)
		return;

	clk_en_a = mci_readl(slot->host,CLKENA);

	if(enable){
		//clk_en_a |= SDMMC_CLKEN_ENABLE << slot->id;
		clk_en_a &= ~(SDMMC_CLKEN_LOW_PWR << slot->id);
	}
	else{
		//clk_en_a &= ~(SDMMC_CLKEN_ENABLE << slot->id);
		clk_en_a |= SDMMC_CLKEN_LOW_PWR << slot->id;
	}
	do{
		mci_writel(slot->host, CLKENA, clk_en_a);
		/* inform CIU */
		ret = mci_send_cmd(slot,SDMMC_CMD_UPD_CLK | SDMMC_CMD_PRV_DAT_WAIT, 0);
	}while(ret&&i--);
}

static void dw_mci_mask_cmdint(struct dw_mci*host, u32 value)
{
	u32 temp;

	temp = mci_readl(host,INTMASK);
	/* 1 to disable command done interrupt, 0 to enable it. */
	if(value){
		temp &= (~SDMMC_INT_CMD_DONE);
	}else{
		temp |= SDMMC_INT_CMD_DONE;
	}

	mci_writel(host,INTMASK,temp);

	return;
}

#if 0
static void dw_mci_mask_dmaint(struct dw_mci*host, u32 value)
{
	/* 1 to disable dma interrupt, 0 to enable it. */
	if (value) {
		mci_writel(host, IDINTEN, 0);
	} else {
		mci_writel(host, IDINTEN, SDMMC_IDMAC_INT_NI | SDMMC_IDMAC_INT_RI |
			   SDMMC_IDMAC_INT_TI);
	}

	return;
}
#endif


static void __dw_mci_start_request(struct dw_mci *host,
				   struct dw_mci_slot *slot,
				   struct mmc_command *cmd)
{
	struct mmc_request *mrq;
	struct mmc_data	*data;
	u32 cmdflags;

	mrq = slot->mrq;
	if (host->pdata->select_slot)
		host->pdata->select_slot(slot->id);

	host->cur_slot = slot;
	host->mrq = mrq;

	host->pending_events = 0;
	host->completed_events = 0;
	host->cmd_status = 0;
	host->data_status = 0;
	host->dir_status = 0;

	data = cmd->data;
	if (data) {
		dw_mci_set_timeout(host);
		mci_writel(host, BYTCNT, data->blksz*data->blocks);
		mci_writel(host, BLKSIZ, data->blksz);
		//xlog_mmc_cmd(host->host_id, cmd->opcode, cmd->arg, data->blksz, data->blocks);
		//xlog_mmc_log(host->host_id, "CMD INFO", data->xferbuf, data->xferlen, data->flags, 0);
		/* Since we cannot mask dma interrupts, disable command done interrupt won't help much. */
		dw_mci_mask_cmdint(host,1);
		/*
		 * If dma interrupts are masked when dma transfer is done,
		 * dma IDSTS won't be set even if we enable them later.
		 */
		#if 0
		if(data->flags&MMC_DATA_READ){
			dw_mci_mask_dmaint(host,0);
		}else{
			dw_mci_mask_dmaint(host,1);
		}
		#endif
	}else{
		//xlog_mmc_cmd(host->host_id, cmd->opcode, cmd->arg, 0, 0);
		dw_mci_mask_cmdint(host,0);
	}

	cmdflags = dw_mci_prepare_command(slot->mmc, cmd);

	/* this is the first command, send the initialization clock */
	if (test_and_clear_bit(DW_MMC_CARD_NEED_INIT, &slot->flags))
		cmdflags |= SDMMC_CMD_INIT;
#ifdef DW_COMMAND_SOFT_TIMEOUT_SUPPORT
	//if (host->dw_dto_timer_expired_cnt > 0){
	//	cmd->error = -EILSEQ;
	//	if (data)
	//	data->error = -EILSEQ;
	//	return dw_mci_request_end(host, mrq);
	//}
#endif

	if (data) {
		if(mmc_busy(host)){ //busy error
			cmd->error = -EBUSY;
			data->error = -EBUSY;
			return dw_mci_request_end(host, mrq);
		}
		dw_mci_submit_data(host, data);
		wmb();
	}

	dw_mci_start_command(host, cmd, cmdflags);
#ifdef DW_COMMAND_SOFT_TIMEOUT_SUPPORT
//	DW_MCI_SPIN_UNLOCK_BH(&host->lock);
	dw_start_dto_timer(host);
//	DW_MCI_SPIN_LOCK_BH(&host->lock);
#endif

	if (mrq->stop)
		host->stop_cmdr = dw_mci_prepare_command(slot->mmc, mrq->stop);
	else
		host->stop_cmdr = dw_mci_prep_stop_abort(host, cmd);
}

static void dw_mci_start_request(struct dw_mci *host,
				 struct dw_mci_slot *slot)
{
	struct mmc_request *mrq = slot->mrq;
	struct mmc_command *cmd;

	cmd = mrq->sbc ? mrq->sbc : mrq->cmd;
	__dw_mci_start_request(host, slot, cmd);
}

/* must be called with host->lock held */
static void dw_mci_queue_request(struct dw_mci *host, struct dw_mci_slot *slot,
				 struct mmc_request *mrq)
{
	dev_vdbg(&slot->mmc->class_dev, "queue request: state=%d\n",
		 host->state);

	slot->mrq = mrq;

	if (host->state == STATE_IDLE) {
		host->state = STATE_SENDING_CMD;
		dw_mci_start_request(host, slot);
	} else {
		list_add_tail(&slot->queue_node, &host->queue);
	}
}
#ifdef _USE_VEHICLE_DC	
volatile int s_request_cnt = 0;
#endif
static void dw_mci_request(struct mmc_host *mmc, struct mmc_request *mrq)
{
	struct dw_mci_slot *slot = mmc_priv(mmc);
	struct dw_mci *host = slot->host;

	WARN_ON(slot->mrq);

	/*
	 * The check for card presence and queueing of the request must be
	 * atomic, otherwise the card could be removed in between and the
	 * request wouldn't fail until another card was inserted.
	 */
#ifdef _USE_VEHICLE_DC	
	if(host->host_id ==1){
	if(s_request_cnt ==0)
		mmc_get_host(slot->mmc);
	s_request_cnt++;
	}
#endif	
	DW_MCI_SPIN_LOCK_BH(&host->lock);

	if (!test_bit(DW_MMC_CARD_PRESENT, &slot->flags)) {
		DW_MCI_SPIN_UNLOCK_BH(&host->lock);
#ifdef _USE_VEHICLE_DC
		if(host->host_id ==1){
			s_request_cnt--;
			if(s_request_cnt ==0)
				mmc_put_host(slot->mmc);
		}
#endif
	
		mrq->cmd->error = -ENOMEDIUM;
		mmc_request_done(mmc, mrq);
		return;
	}

	dw_mci_queue_request(host, slot, mrq);

	DW_MCI_SPIN_UNLOCK_BH(&host->lock);
}

static void dw_mci_set_ios(struct mmc_host *mmc, struct mmc_ios *ios)
{
	struct dw_mci_slot *slot = mmc_priv(mmc);
	const struct dw_mci_drv_data *drv_data = slot->host->drv_data;
	struct dw_mci *host = slot->host;
	u32 regs;

	switch (ios->bus_width) {
	case MMC_BUS_WIDTH_4:
		slot->ctype = SDMMC_CTYPE_4BIT;
		break;
	case MMC_BUS_WIDTH_8:
		slot->ctype = SDMMC_CTYPE_8BIT;
		break;
	default:
		/* set default 1 bit mode */
		slot->ctype = SDMMC_CTYPE_1BIT;
	}

	/* DDR mode set */
	if (ios->timing == MMC_TIMING_UHS_DDR50) {
		regs = mci_readl(host, SDR50_104);
		regs &= ~((1 << slot->id)|(1 << (slot->id + 16)));
		mci_writel(host, SDR50_104, regs);
		MCI_REG_BIT_OPS(host, UHS_REG, (slot->id + 16), 1);
		/* clean in dw_mci_prepare_command() */
		//		MCI_REG_BIT_OPS(host, CMD, SDMMC_CMD_USE_HOLD_REG, 0);
	}
	else if (ios->timing == MMC_TIMING_UHS_SDR50) {
		MCI_REG_BIT_OPS(host, UHS_REG, (slot->id + 16), 0);
		regs = mci_readl(host, SDR50_104);
		regs |= (1 << slot->id);
		regs &= ~(1 << (slot->id + 16));
		mci_writel(host, SDR50_104, regs);
		//		MCI_REG_BIT_OPS(host, CMD, SDMMC_CMD_USE_HOLD_REG, 0);
	}
	else if (ios->timing == MMC_TIMING_UHS_SDR104) {
		MCI_REG_BIT_OPS(host, UHS_REG, (slot->id + 16), 0);
		regs = mci_readl(host, SDR50_104);
		regs |= (1 << (slot->id + 16));
		regs &= ~(1 << slot->id);
		mci_writel(host, SDR50_104, regs);
		//		MCI_REG_BIT_OPS(host, CMD, SDMMC_CMD_USE_HOLD_REG, 0);
	}
	host->timing = ios->timing;

	/*
	 * Use mirror of ios->clock to prevent race with mmc
	 * core ios update when finding the minimum.
	 */
	slot->clock = ios->clock;

	if (drv_data && drv_data->set_ios)
		drv_data->set_ios(host, ios);

	/* Slot specific timing and width adjustment */

	switch (ios->power_mode) {
	case MMC_POWER_UP:
			dw_mci_ctrl_all_reset(host);
			dw_mci_setup_bus(slot, true);

			mci_writel(host, FIFOTH, host->fifoth_val);
			mci_writel(host, RINTSTS, 0xFFFFFFFF);
			mci_writel(host, INTMASK, SDMMC_INT_CMD_DONE | SDMMC_INT_DATA_OVER |
					DW_MCI_ERROR_FLAGS);
			mci_writel(host, CTRL, SDMMC_CTRL_INT_ENABLE); /* Enable mci interrupt */
			mci_writel(host, DEBNCE, 0xfff); /* 0xfff is the default value, spec is wrong @V1.0 */


#ifdef CONFIG_MMC_DW_IDMAC
			dw_mci_init_dma(host);
#endif
			set_bit(DW_MMC_CARD_NEED_INIT, &slot->flags);
			/* Power up slot */
			if (host->pdata->setpower)
				host->pdata->setpower(slot->id, mmc->ocr_avail);
			regs = mci_readl(host, PWREN);
			regs |= (1 << slot->id);
			mci_writel(host, PWREN, regs);
			xlog_mmc_log_key(host->host_id, "INFO dw_mci_set_ios MMC_POWER_UP", ios->clock, ios->timing, ios->vdd);
			break;
		case MMC_POWER_ON:
			xlog_mmc_log_key(host->host_id, "INFO dw_mci_set_ios MMC_POWER_ON", ios->clock, ios->timing, ios->vdd);
			dw_mci_setup_bus(slot, false);
			break;
	case MMC_POWER_OFF:
			dw_mci_setup_bus(slot, false);
			/* Power down slot */
			if (host->pdata->setpower)
				host->pdata->setpower(slot->id, 0);
			regs = mci_readl(host, PWREN);
			regs &= ~(1 << slot->id);
			mci_writel(host, PWREN, regs);
			xlog_mmc_log_key(host->host_id, "INFO dw_mci_set_ios MMC_POWER_OFF", ios->clock, ios->timing, ios->vdd);
			//disable_irq(host->irq);
#ifdef CONFIG_MMC_DW_IDMAC
			dw_mci_stop_dma(host);
#endif
		break;
	default:
			xlog_mmc_log_key(host->host_id, "INFO dw_mci_set_ios default", ios->clock, ios->timing, ios->power_mode);
			//dw_mci_setup_bus(slot, false);
		break;
	}
}

static int dw_mci_get_ro(struct mmc_host *mmc)
{
	int read_only;
	struct dw_mci_slot *slot = mmc_priv(mmc);
	struct dw_mci_board *brd = slot->host->pdata;

	/* Use platform get_ro function, else try on board write protect */
	if (slot->quirks & DW_MCI_SLOT_QUIRK_NO_WRITE_PROTECT)
		read_only = 0;
	else if (brd->get_ro)
		read_only = brd->get_ro(slot->id);
	//else if (gpio_is_valid(slot->wp_gpio))
	//	read_only = gpio_get_value(slot->wp_gpio);
	else
		read_only =
			mci_readl(slot->host, WRTPRT) & (1 << slot->id) ? 1 : 0;

	dev_dbg(&mmc->class_dev, "card is %s\n",
		read_only ? "read-only" : "read-write");

	return read_only;
}

#if (defined CONFIG_ARCH_ZX297520V2)||(defined CONFIG_ARCH_ZX297520V3)
static int dwc_mci_get_detect_val(unsigned char gpio)
{
	int gpio_val = 0;
	/* config as gpio*/
	// xsh- zx29_gpio_config(PIN_MMC_TF_CARD_DET, GPIO54_GPIO54);
	//if (ret)
	//   return ret;
	/* config as input*/
	//ret = gpio_direction_input(PIN_MMC_TF_CARD_DET);
	//if (ret)
	//   return ret;
	/* read gpio level */
	gpio_val = gpio_get_value(PIN_MMC_TF_CARD_DET);
	return gpio_val;
}
#endif
int dw_mci_get_present_from_pin(void)
{
	int gpio_val = 0;

#if (defined CONFIG_ARCH_ZX297520V2)||(defined CONFIG_ARCH_ZX297520V3)
	gpio_val = dwc_mci_get_detect_val(PIN_MMC_TF_CARD_DET);
#endif
	return !gpio_val;
}

static int dw_mci_get_cd(struct mmc_host *mmc)
{
	int present;
	struct dw_mci_slot *slot = mmc_priv(mmc);
	struct dw_mci_board *brd = slot->host->pdata;
	struct dw_mci *host = slot->host;

	/* Use platform get_cd function, else try onboard card detect */
	if (brd->quirks & DW_MCI_QUIRK_BROKEN_CARD_DETECTION){
		present = 1;
	}
	//else if (brd->get_cd)
	//	present = !brd->get_cd(slot->id);
	//else
	//	present = (mci_readl(slot->host, CDETECT) & (1 << slot->id))
	//		== 0 ? 1 : 0;
	else{
		present = dw_mci_get_present_from_pin();
	}


	DW_MCI_SPIN_LOCK_BH(&host->lock);
	if (present) {
		set_bit(DW_MMC_CARD_PRESENT, &slot->flags);
		xlog_mmc_log(slot->host->host_id, "INFO card is present", slot->last_detect_state, 0, 0);
	} else {
		clear_bit(DW_MMC_CARD_PRESENT, &slot->flags);
		xlog_mmc_log(slot->host->host_id, "INFO card is not present", slot->last_detect_state, 0, 0);
	}
	DW_MCI_SPIN_UNLOCK_BH(&host->lock);

	return present;
}

/*
 * Disable lower power mode.
 *
 * Low power mode will stop the card clock when idle.  According to the
 * description of the CLKENA register we should disable low power mode
 * for SDIO cards if we need SDIO interrupts to work.
 *
 * This function is fast if low power mode is already disabled.
 */
static void dw_mci_disable_low_power(struct dw_mci_slot *slot)
{
	struct dw_mci *host = slot->host;
	u32 clk_en_a;
	const u32 clken_low_pwr = SDMMC_CLKEN_LOW_PWR << slot->id;

	clk_en_a = mci_readl(host, CLKENA);

	xlog_mmc_log(slot->host->host_id, "INFO dw_mci_disable_low_power", clken_low_pwr, clk_en_a, 0);
	if (clk_en_a & clken_low_pwr) {
		mci_writel(host, CLKENA, clk_en_a & ~clken_low_pwr);
		mci_send_cmd(slot, SDMMC_CMD_UPD_CLK |
			     SDMMC_CMD_PRV_DAT_WAIT, 0);
	}
//	dev_err(host->dev, "%s: sdlowpower dis %d\n",__func__,slot->id);
}

static void dw_mci_enable_sdio_irq(struct mmc_host *mmc, int enb)
{
	struct dw_mci_slot *slot = mmc_priv(mmc);
	struct dw_mci *host = slot->host;
	u32 int_mask;

	/* Enable/disable Slot Specific SDIO interrupt */
	int_mask = mci_readl(host, INTMASK);
	if (enb) {
		/*
		 * Turn off low power mode if it was enabled.  This is a bit of
		 * a heavy operation and we disable / enable IRQs a lot, so
		 * we'll leave low power mode disabled and it will get
		 * re-enabled again in dw_mci_setup_bus().
		 */
		dw_mci_disable_low_power(slot);

		mci_writel(host, INTMASK,
			   (int_mask | SDMMC_INT_SDIO(slot->id)));
	} else {
		mci_writel(host, INTMASK,
			   (int_mask & ~SDMMC_INT_SDIO(slot->id)));
	}
}

static int dw_mci_execute_tuning(struct mmc_host *mmc, u32 opcode)
{
	struct dw_mci_slot *slot = mmc_priv(mmc);
	struct dw_mci *host = slot->host;
	const struct dw_mci_drv_data *drv_data = host->drv_data;
	struct dw_mci_tuning_data tuning_data;
	int err = -ENOSYS;

	if (opcode == MMC_SEND_TUNING_BLOCK_HS200) {
		if (mmc->ios.bus_width == MMC_BUS_WIDTH_8) {
			tuning_data.blk_pattern = tuning_blk_pattern_8bit;
			tuning_data.blksz = sizeof(tuning_blk_pattern_8bit);
		} else if (mmc->ios.bus_width == MMC_BUS_WIDTH_4) {
			tuning_data.blk_pattern = tuning_blk_pattern_4bit;
			tuning_data.blksz = sizeof(tuning_blk_pattern_4bit);
		} else {
			return -EINVAL;
		}
	} else if (opcode == MMC_SEND_TUNING_BLOCK) {
		tuning_data.blk_pattern = tuning_blk_pattern_4bit;
		tuning_data.blksz = sizeof(tuning_blk_pattern_4bit);
	} else {
		dev_err(host->dev,
			"Undefined command(%d) for tuning\n", opcode);
		return -EINVAL;
	}

	if (drv_data && drv_data->execute_tuning)
		err = drv_data->execute_tuning(slot, opcode, &tuning_data);
	return err;
}

static const struct mmc_host_ops dw_mci_ops = {
	.request		= dw_mci_request,
	.pre_req		= dw_mci_pre_req,
	.post_req		= dw_mci_post_req,
	.set_ios		= dw_mci_set_ios,
	.get_ro			= dw_mci_get_ro,
	.get_cd			= dw_mci_get_cd,
	.enable_sdio_irq	= dw_mci_enable_sdio_irq,
	//.execute_tuning		= dw_mci_execute_tuning,
};

static void dw_mci_request_end(struct dw_mci *host, struct mmc_request *mrq)
	__releases(&host->lock)
	__acquires(&host->lock)
{
	struct dw_mci_slot *slot;
	struct mmc_host	*prev_mmc = host->cur_slot->mmc;

	WARN_ON(host->cmd || host->data);

#ifdef DW_COMMAND_SOFT_TIMEOUT_SUPPORT
	dw_stop_dto_timer(host);
#endif

	host->cur_slot->mrq = NULL;
	host->mrq = NULL;
	if (!list_empty(&host->queue)) {
		slot = list_entry(host->queue.next,
				  struct dw_mci_slot, queue_node);
		list_del(&slot->queue_node);
		xlog_mmc_log_op(host->host_id, mrq->cmd->opcode, "REQ DONE, list not empty",
						mrq->cmd->error, mrq->cmd->resp[0], (mrq->stop ? mrq->stop->resp[0] : 0));
		host->state = STATE_SENDING_CMD;
		dw_mci_start_request(host, slot);
	} else {
		xlog_mmc_log_op(host->host_id, mrq->cmd->opcode, "REQ DONE, list empty",
						mrq->cmd->error, mrq->cmd->resp[0], (mrq->stop ? mrq->stop->resp[0] : 0));
		host->state = STATE_IDLE;
#ifdef _USE_VEHICLE_DC			
		if(host->host_id ==1){
			mmc_put_host(prev_mmc);
			s_request_cnt = 0;
		}
#endif		
	}

	spin_unlock(&host->lock);
	mmc_request_done(prev_mmc, mrq);
	spin_lock(&host->lock);
}

static int dw_mci_command_complete(struct dw_mci *host, struct mmc_command *cmd)
{
	u32 status = host->cmd_status;

	host->cmd_status = 0;

	/* Read the response from the card (up to 16 bytes) */
	if (cmd->flags & MMC_RSP_PRESENT) {
		if (cmd->flags & MMC_RSP_136) {
			cmd->resp[3] = mci_readl(host, RESP0);
			cmd->resp[2] = mci_readl(host, RESP1);
			cmd->resp[1] = mci_readl(host, RESP2);
			cmd->resp[0] = mci_readl(host, RESP3);
		} else {
			cmd->resp[0] = mci_readl(host, RESP0);
			cmd->resp[1] = 0;
			cmd->resp[2] = 0;
			cmd->resp[3] = 0;
		}
	}

	if (status & SDMMC_INT_HLE){
		cmd->error = -EIO;		
		sc_debug_info_record(MODULE_ID_AP_MMC,"%s(%d): cmd->error=0x%x\n",__func__,host->host_id,cmd->error);
	}
	else if (status & SDMMC_INT_RTO)
		cmd->error = -ETIMEDOUT;
	else if ((cmd->flags & MMC_RSP_CRC) && (status & SDMMC_INT_RCRC)){
		cmd->error = -EILSEQ;
		sc_debug_info_record(MODULE_ID_AP_MMC,"%s(%d): cmd->error=0x%x\n",__func__,host->host_id,cmd->error);		
	}
	else if (status & SDMMC_INT_RESP_ERR){
		cmd->error = -EIO;		
		sc_debug_info_record(MODULE_ID_AP_MMC,"%s(%d): cmd->error=0x%x\n",__func__,host->host_id,cmd->error);		
	}
	else
		cmd->error = 0;

	if (cmd->error) {
		xlog_mmc_status(host->host_id, cmd->opcode, XLOG_TF_STATUS_CMD_COMPLETE, cmd->error, status, cmd->resp[0]);
		/* newer ip versions need a delay between retries */
		if (host->quirks & DW_MCI_QUIRK_RETRY_DELAY)
			msleep(20);
		//xlog_mmc_log_op(host->host_id, cmd->opcode, "CMD ERROR status resp0 resp1", status, cmd->resp[0], cmd->resp[1]);
		host->cmd_error_cnt++;
	}

	return cmd->error;
}

static int dw_mci_data_complete(struct dw_mci *host, struct mmc_data *data)
{
	u32 status = host->data_status;
	u32 temp;

	if (status & DW_MCI_DATA_ERROR_FLAGS) {
		if (status & SDMMC_INT_DRTO) {
			data->error = -ETIMEDOUT;
		} else if (status & SDMMC_INT_DCRC) {
			data->error = -EILSEQ;
		} else if (status & SDMMC_INT_EBE) {
			if (host->dir_status ==
				DW_MCI_SEND_STATUS) {
				/*
				 * No data CRC status was returned.
				 * The number of bytes transferred
				 * will be exaggerated in PIO mode.
				 */
				data->bytes_xfered = 0;
				data->error = -ETIMEDOUT;
			} else if (host->dir_status ==
					DW_MCI_RECV_STATUS) {
				data->error = -EIO;
			}
		} else {
			/* SDMMC_INT_SBE is included */
			data->error = -EIO;
		}

		temp = mci_readl(host, STATUS);
		dev_err(host->dev, "DATA ERROR, MINTSTS 0x%X, STATUS 0x%X\n", status, temp);		
		sc_debug_info_record(MODULE_ID_AP_MMC,"DATA ERROR, MINTSTS 0x%X, STATUS 0x%X\n", status, temp);		
		xlog_mmc_status(host->host_id, data->mrq->cmd->opcode, XLOG_TF_STATUS_DATA_COMPLETE, data->error, status, temp);
		if (status & SDMMC_INT_FRUN) {
			xlog_mmc_log_opkey(host->host_id, data->mrq->cmd->opcode,
					"DATA ERROR, MINTSTS 0x%X, STATUS 0x%X\n", status, temp, 0);
		}

		host->data_error_cnt++;

		/*
		 * After an error, there may be data lingering
		 * in the FIFO
		 */
		dw_mci_fifo_reset(host);
		//dw_mci_ctrl_all_reset(host);
	} else {
		data->bytes_xfered = data->blocks * data->blksz;
		data->error = 0;
	}

	return data->error;
}

#ifdef DW_MCI_THREAD


static void dw_mci_kick_thread(struct dw_mci* host)
{

	if(!test_and_set_bit(1, &host->dw_mci_flag)){
		//set_bit(1, &dw_mci_flag);
		//wake_up(&dw_mci_thread_wait);
		up(&host->mmc_sema);
	}
}
static int dw_mci_tasklet_func(void * arg)
{
	struct dw_mci *host = (struct dw_mci *)arg;
	struct mmc_data	*data;
	struct mmc_command *cmd;
	struct mmc_request *mrq;
	enum dw_mci_state state;
	enum dw_mci_state prev_state;
	unsigned int err;
	//int ret;
	//struct scatterlist *sg;
	//unsigned int i, sg_len, count = 0;

	struct sched_param param = { .sched_priority = 1 };


	if (host == NULL)
		return -1;

	mrq = NULL;
	cmd = NULL;
	data = NULL;
#ifdef CONFIG_RTL8192CD
	param.sched_priority = 38;//32
#else
	param.sched_priority = 40;
#endif
	sched_setscheduler(current, SCHED_FIFO, &param);

	while (!kthread_should_stop()){
		down(&host->mmc_sema);
		if(!test_and_clear_bit(1, &host->dw_mci_flag))
			continue;


		spin_lock(&host->lock);

		state = host->state;
		data = host->data;
		mrq = host->mrq;

		if ((state == STATE_SENDING_CMD) || host->cmd)
			xlog_mmc_log_op(host->host_id, host->cmd->opcode,
				"pending,complete,state", host->pending_events, host->completed_events, state);
		else
			xlog_mmc_log(host->host_id,
				"pending,complete,state", host->pending_events, host->completed_events, state);

		do {
			prev_state = state;

			switch (state) {
				case STATE_IDLE:
					break;

				case STATE_SENDING_CMD:
					if (!test_and_clear_bit(EVENT_CMD_COMPLETE,
								&host->pending_events))
						break;

					cmd = host->cmd;
					host->cmd = NULL;
					set_bit(EVENT_CMD_COMPLETE, &host->completed_events);
					err = dw_mci_command_complete(host, cmd);
					if (cmd == mrq->sbc && !err) {
						prev_state = state = STATE_SENDING_CMD;
						__dw_mci_start_request(host, host->cur_slot,
								mrq->cmd);
						goto unlock;
					}

					if (cmd->data && err) {
						dw_mci_stop_dma(host);
						send_stop_abort(host, data);
						state = STATE_SENDING_STOP;
						break;
					}

					if (!cmd->data || err) {
						dw_mci_request_end(host, mrq);
						goto unlock;
					}

					prev_state = state = STATE_SENDING_DATA;
					/* fall through */

				case STATE_SENDING_DATA:
					/*
					 * We could get a data error and never a transfer
					 * complete so we'd better check for it here.
					 *
					 * Note that we don't really care if we also got a
					 * transfer complete; stopping the DMA and sending an
					 * abort won't hurt.
					 */
					if (test_and_clear_bit(EVENT_DATA_ERROR,
								&host->pending_events)) {
						dw_mci_stop_dma(host);
						send_stop_abort(host, data);
						set_bit(EVENT_DATA_ERROR, &host->completed_events);
						state = STATE_DATA_ERROR;
						break;
					}

					if (!test_and_clear_bit(EVENT_XFER_COMPLETE,
								&host->pending_events))
						break;

					set_bit(EVENT_XFER_COMPLETE, &host->completed_events);
					/* DMA transfer is done. */
					prev_state = state = STATE_DATA_BUSY;
					/* fall through */

				case STATE_DATA_BUSY:
					/*
					 * In the following situations the stop command would not be sent:
					 * 1. interrupt may come between the upper 2 checks, EVENT_DATA_ERROR and EVENT_XFER_COMPLETE;
					 * 2. EVENT_XFER_COMPLETE may come much earlier than EVENT_DATA_ERROR;
					 * So it is necessary to check EVENT_DATA_ERROR again.
					 */
					if (test_and_clear_bit(EVENT_DATA_ERROR, &host->pending_events) && \
						!test_bit(EVENT_DATA_ERROR, &host->completed_events)) {
						dw_mci_stop_dma(host);
						send_stop_abort(host, data);
						set_bit(EVENT_DATA_ERROR, &host->completed_events);
					}

					if (!test_and_clear_bit(EVENT_DATA_COMPLETE,
								&host->pending_events))
						break;

					host->data = NULL;
					set_bit(EVENT_DATA_COMPLETE, &host->completed_events);
					err = dw_mci_data_complete(host, data);

					if (!err) {
						if (!data->stop || mrq->sbc) {
							if (mrq->sbc && data->stop)
								data->stop->error = 0;
							dw_mci_request_end(host, mrq);
							goto unlock;
						}

						/* stop command for open-ended transfer*/
						//if (data->stop) /* unnecessary check */
						send_stop_abort(host, data);
					} else if (!test_bit(EVENT_DATA_ERROR, &host->completed_events)) {
						/*
						 * interrupt may come between the upper 2 checks, EVENT_DATA_ERROR and EVENT_DATA_COMPLETE.
						 * So check whether we have send_stop_abort.
						 */
						dw_mci_stop_dma(host);
						send_stop_abort(host, data);
						set_bit(EVENT_DATA_ERROR, &host->completed_events);
						dw_mci_fifo_reset(host);
					}

					/*
					 * If err has non-zero,
					 * stop-abort command has been already issued.
					 */
					prev_state = state = STATE_SENDING_STOP;

					/* fall through */

				case STATE_SENDING_STOP:
					if (!test_and_clear_bit(EVENT_CMD_COMPLETE,
								&host->pending_events))
						break;

					/* CMD error in data command */
					if (mrq->cmd->error && mrq->data)
						dw_mci_fifo_reset(host);

					host->cmd = NULL;
					host->data = NULL;

					if (mrq->stop)
						dw_mci_command_complete(host, mrq->stop);
					else {
						xlog_mmc_log_op(host->host_id, host->stop_abort.opcode,
										"stop cmd st & resp", host->cmd_status, mci_readl(host, RESP0), 0);
						host->cmd_status = 0;
					}

					dw_mci_request_end(host, mrq);
					goto unlock;

				case STATE_DATA_ERROR:
					if (!test_and_clear_bit(EVENT_XFER_COMPLETE,
								&host->pending_events))
						break;

					host->mmc_trans_err_count++;
					xlog_mmc_log(host->host_id, "STATE_DATA_ERROR", host->mmc_trans_err_count,
						host->pending_events, host->completed_events);

					state = STATE_DATA_BUSY;
					break;
			}
		} while (state != prev_state);

		host->state = state;
unlock:
		spin_unlock(&host->lock);
	}

	return 0;
}
#else


static void dw_mci_kick_thread(struct dw_mci* host)
{
	tasklet_schedule(&host->tasklet);
}

void dw_mci_tasklet_func(unsigned long priv)
{
	struct dw_mci *host = (struct dw_mci *)priv;
	struct mmc_data	*data;
	struct mmc_command *cmd;
	struct mmc_request *mrq;
	enum dw_mci_state state;
	enum dw_mci_state prev_state;
	unsigned int err;
	//if(mmc_trans_err_count > 6)
	//	return;

	if (host == NULL)
		return ;

	mrq = NULL;
	cmd = NULL;
	data = NULL;

	spin_lock(&host->lock);

	state = host->state;
	data = host->data;
	mrq = host->mrq;

	if ((state == STATE_SENDING_CMD) || host->cmd)
		xlog_mmc_log_op(host->host_id, host->cmd->opcode,
			"pending,complete,state", host->pending_events, host->completed_events, state);
	else
		xlog_mmc_log(host->host_id,
			"pending,complete,state", host->pending_events, host->completed_events, state);

	do {
		prev_state = state;

		switch (state) {
		case STATE_IDLE:
			break;

		case STATE_SENDING_CMD:
			if (!test_and_clear_bit(EVENT_CMD_COMPLETE,
						&host->pending_events))
				break;

			cmd = host->cmd;
			host->cmd = NULL;
			set_bit(EVENT_CMD_COMPLETE, &host->completed_events);
			err = dw_mci_command_complete(host, cmd);
			if (cmd == mrq->sbc && !err) {
				prev_state = state = STATE_SENDING_CMD;
				__dw_mci_start_request(host, host->cur_slot,
						       mrq->cmd);
				goto unlock;
			}

			if (cmd->data && err) {
				dw_mci_stop_dma(host);
				send_stop_abort(host, data);
				state = STATE_SENDING_STOP;
				break;
			}

			if (!cmd->data || err) {
				dw_mci_request_end(host, mrq);
				goto unlock;
			}

			prev_state = state = STATE_SENDING_DATA;
			/* fall through */

		case STATE_SENDING_DATA:
			/*
			 * We could get a data error and never a transfer
			 * complete so we'd better check for it here.
			 *
			 * Note that we don't really care if we also got a
			 * transfer complete; stopping the DMA and sending an
			 * abort won't hurt.
			 */
			if (test_and_clear_bit(EVENT_DATA_ERROR,
					       &host->pending_events)) {
				dw_mci_stop_dma(host);
				send_stop_abort(host, data);
				set_bit(EVENT_DATA_ERROR, &host->completed_events);
				state = STATE_DATA_ERROR;
				break;
			}

			if (!test_and_clear_bit(EVENT_XFER_COMPLETE,
						&host->pending_events))
				break;

			set_bit(EVENT_XFER_COMPLETE, &host->completed_events);
			/* DMA transfer is done. */
			prev_state = state = STATE_DATA_BUSY;
			/* fall through */

		case STATE_DATA_BUSY:
			/*
			 * In the following situations the stop command would not be sent:
			 * 1. interrupt may come between the upper 2 checks, EVENT_DATA_ERROR and EVENT_XFER_COMPLETE;
			 * 2. EVENT_XFER_COMPLETE may come much earlier than EVENT_DATA_ERROR;
			 * So it is necessary to check EVENT_DATA_ERROR again.
			 */
			if (test_and_clear_bit(EVENT_DATA_ERROR, &host->pending_events) && \
				!test_bit(EVENT_DATA_ERROR, &host->completed_events)) {
				dw_mci_stop_dma(host);
				send_stop_abort(host, data);
				set_bit(EVENT_DATA_ERROR, &host->completed_events);
			}

			if (!test_and_clear_bit(EVENT_DATA_COMPLETE,
						&host->pending_events))
				break;

			host->data = NULL;
			set_bit(EVENT_DATA_COMPLETE, &host->completed_events);
			err = dw_mci_data_complete(host, data);

			if (!err) {
				if (!data->stop || mrq->sbc) {
					if (mrq->sbc && data->stop)
						data->stop->error = 0;
					dw_mci_request_end(host, mrq);
					goto unlock;
				}

				/* stop command for open-ended transfer*/
				//if (data->stop) /* unnecessary check */
				send_stop_abort(host, data);
			} else if (!test_bit(EVENT_DATA_ERROR, &host->completed_events)) {
				/*
				 * interrupt may come between the upper 2 checks, EVENT_DATA_ERROR and EVENT_DATA_COMPLETE.
				 * So check whether we have send_stop_abort.
				 */
				dw_mci_stop_dma(host);
				send_stop_abort(host, data);
				set_bit(EVENT_DATA_ERROR, &host->completed_events);
				dw_mci_fifo_reset(host);
			}

			/*
			 * If err has non-zero,
			 * stop-abort command has been already issued.
			 */
			prev_state = state = STATE_SENDING_STOP;

			/* fall through */

		case STATE_SENDING_STOP:
			if (!test_and_clear_bit(EVENT_CMD_COMPLETE,
						&host->pending_events))
				break;

			/* CMD error in data command */
			if (mrq->cmd->error && mrq->data)
				dw_mci_fifo_reset(host);
				//dw_mci_ctrl_all_reset(host);

			host->cmd = NULL;
			host->data = NULL;

			if (mrq->stop)
				dw_mci_command_complete(host, mrq->stop);
			else {
				xlog_mmc_log_op(host->host_id, host->stop_abort.opcode,
								"stop cmd st & resp", host->cmd_status, mci_readl(host, RESP0), 0);
				host->cmd_status = 0;
			}

			dw_mci_request_end(host, mrq);
			goto unlock;

		case STATE_DATA_ERROR:
			if (!test_and_clear_bit(EVENT_XFER_COMPLETE,
						&host->pending_events))
				break;

				host->mmc_trans_err_count++;
				xlog_mmc_log(host->host_id, "STATE_DATA_ERROR", host->mmc_trans_err_count,
								host->pending_events, host->completed_events);

			state = STATE_DATA_BUSY;
			break;
		}
	} while (state != prev_state);

	host->state = state;
unlock:
	spin_unlock(&host->lock);

}
#endif

/* push final bytes to part_buf, only use during push */
static void dw_mci_set_part_bytes(struct dw_mci *host, void *buf, int cnt)
{
	memcpy((void *)&host->part_buf, buf, cnt);
	host->part_buf_count = cnt;
}

/* append bytes to part_buf, only use during push */
static int dw_mci_push_part_bytes(struct dw_mci *host, void *buf, int cnt)
{
	cnt = min(cnt, (1 << host->data_shift) - host->part_buf_count);
	memcpy((void *)&host->part_buf + host->part_buf_count, buf, cnt);
	host->part_buf_count += cnt;
	return cnt;
}

/* pull first bytes from part_buf, only use during pull */
static int dw_mci_pull_part_bytes(struct dw_mci *host, void *buf, int cnt)
{
	cnt = min(cnt, (int)host->part_buf_count);
	if (cnt) {
		memcpy(buf, (void *)&host->part_buf + host->part_buf_start,
		       cnt);
		host->part_buf_count -= cnt;
		host->part_buf_start += cnt;
	}
	return cnt;
}

/* pull final bytes from the part_buf, assuming it's just been filled */
static void dw_mci_pull_final_bytes(struct dw_mci *host, void *buf, int cnt)
{
	memcpy(buf, &host->part_buf, cnt);
	host->part_buf_start = cnt;
	host->part_buf_count = (1 << host->data_shift) - cnt;
}

static void dw_mci_push_data16(struct dw_mci *host, void *buf, int cnt)
{
	struct mmc_data *data = host->data;
	int init_cnt = cnt;

	/* try and push anything in the part_buf */
	if (unlikely(host->part_buf_count)) {
		int len = dw_mci_push_part_bytes(host, buf, cnt);
		buf += len;
		cnt -= len;
		if (host->part_buf_count == 2) {
			mci_writew(host, DATA(host->data_offset),
					host->part_buf16);
			host->part_buf_count = 0;
		}
	}
#ifndef CONFIG_HAVE_EFFICIENT_UNALIGNED_ACCESS
	if (unlikely((unsigned long)buf & 0x1)) {
		while (cnt >= 2) {
			u16 aligned_buf[64];
			int len = min(cnt & -2, (int)sizeof(aligned_buf));
			int items = len >> 1;
			int i;
			/* memcpy from input buffer into aligned buffer */
			memcpy(aligned_buf, buf, len);
			buf += len;
			cnt -= len;
			/* push data from aligned buffer into fifo */
			for (i = 0; i < items; ++i)
				mci_writew(host, DATA(host->data_offset),
						aligned_buf[i]);
		}
	} else
#endif
	{
		u16 *pdata = buf;
		for (; cnt >= 2; cnt -= 2)
			mci_writew(host, DATA(host->data_offset), *pdata++);
		buf = pdata;
	}
	/* put anything remaining in the part_buf */
	if (cnt) {
		dw_mci_set_part_bytes(host, buf, cnt);
		 /* Push data if we have reached the expected data length */
		if ((data->bytes_xfered + init_cnt) ==
		    (data->blksz * data->blocks))
			mci_writew(host, DATA(host->data_offset),
				   host->part_buf16);
	}
}

static void dw_mci_pull_data16(struct dw_mci *host, void *buf, int cnt)
{
#ifndef CONFIG_HAVE_EFFICIENT_UNALIGNED_ACCESS
	if (unlikely((unsigned long)buf & 0x1)) {
		while (cnt >= 2) {
			/* pull data from fifo into aligned buffer */
			u16 aligned_buf[64];
			int len = min(cnt & -2, (int)sizeof(aligned_buf));
			int items = len >> 1;
			int i;
			for (i = 0; i < items; ++i)
				aligned_buf[i] = mci_readw(host,
						DATA(host->data_offset));
			/* memcpy from aligned buffer into output buffer */
			memcpy(buf, aligned_buf, len);
			buf += len;
			cnt -= len;
		}
	} else
#endif
	{
		u16 *pdata = buf;
		for (; cnt >= 2; cnt -= 2)
			*pdata++ = mci_readw(host, DATA(host->data_offset));
		buf = pdata;
	}
	if (cnt) {
		host->part_buf16 = mci_readw(host, DATA(host->data_offset));
		dw_mci_pull_final_bytes(host, buf, cnt);
	}
}

static void dw_mci_push_data32(struct dw_mci *host, void *buf, int cnt)
{
	struct mmc_data *data = host->data;
	int init_cnt = cnt;

	/* try and push anything in the part_buf */
	if (unlikely(host->part_buf_count)) {
		int len = dw_mci_push_part_bytes(host, buf, cnt);
		buf += len;
		cnt -= len;
		if (host->part_buf_count == 4) {
			mci_writel(host, DATA(host->data_offset),
					host->part_buf32);
			host->part_buf_count = 0;
		}
	}
#ifndef CONFIG_HAVE_EFFICIENT_UNALIGNED_ACCESS
	if (unlikely((unsigned long)buf & 0x3)) {
		while (cnt >= 4) {
			u32 aligned_buf[32];
			int len = min(cnt & -4, (int)sizeof(aligned_buf));
			int items = len >> 2;
			int i;
			/* memcpy from input buffer into aligned buffer */
			memcpy(aligned_buf, buf, len);
			buf += len;
			cnt -= len;
			/* push data from aligned buffer into fifo */
			for (i = 0; i < items; ++i)
				mci_writel(host, DATA(host->data_offset),
						aligned_buf[i]);
		}
	} else
#endif
	{
		u32 *pdata = buf;
		for (; cnt >= 4; cnt -= 4)
			mci_writel(host, DATA(host->data_offset), *pdata++);
		buf = pdata;
	}
	/* put anything remaining in the part_buf */
	if (cnt) {
		dw_mci_set_part_bytes(host, buf, cnt);
		 /* Push data if we have reached the expected data length */
		if ((data->bytes_xfered + init_cnt) ==
		    (data->blksz * data->blocks))
			mci_writel(host, DATA(host->data_offset),
				   host->part_buf32);
	}
}

static void dw_mci_pull_data32(struct dw_mci *host, void *buf, int cnt)
{
#ifndef CONFIG_HAVE_EFFICIENT_UNALIGNED_ACCESS
	if (unlikely((unsigned long)buf & 0x3)) {
		while (cnt >= 4) {
			/* pull data from fifo into aligned buffer */
			u32 aligned_buf[32];
			int len = min(cnt & -4, (int)sizeof(aligned_buf));
			int items = len >> 2;
			int i;
			for (i = 0; i < items; ++i)
				aligned_buf[i] = mci_readl(host,
						DATA(host->data_offset));
			/* memcpy from aligned buffer into output buffer */
			memcpy(buf, aligned_buf, len);
			buf += len;
			cnt -= len;
		}
	} else
#endif
	{
		u32 *pdata = buf;
		for (; cnt >= 4; cnt -= 4)
			*pdata++ = mci_readl(host, DATA(host->data_offset));
		buf = pdata;
	}
	if (cnt) {
		host->part_buf32 = mci_readl(host, DATA(host->data_offset));
		dw_mci_pull_final_bytes(host, buf, cnt);
	}
}

static void dw_mci_push_data64(struct dw_mci *host, void *buf, int cnt)
{
	struct mmc_data *data = host->data;
	int init_cnt = cnt;

	/* try and push anything in the part_buf */
	if (unlikely(host->part_buf_count)) {
		int len = dw_mci_push_part_bytes(host, buf, cnt);
		buf += len;
		cnt -= len;

		if (host->part_buf_count == 8) {
			mci_writeq(host, DATA(host->data_offset),
					host->part_buf);
			host->part_buf_count = 0;
		}
	}
#ifndef CONFIG_HAVE_EFFICIENT_UNALIGNED_ACCESS
	if (unlikely((unsigned long)buf & 0x7)) {
		while (cnt >= 8) {
			u64 aligned_buf[16];
			int len = min(cnt & -8, (int)sizeof(aligned_buf));
			int items = len >> 3;
			int i;
			/* memcpy from input buffer into aligned buffer */
			memcpy(aligned_buf, buf, len);
			buf += len;
			cnt -= len;
			/* push data from aligned buffer into fifo */
			for (i = 0; i < items; ++i)
				mci_writeq(host, DATA(host->data_offset),
						aligned_buf[i]);
		}
	} else
#endif
	{
		u64 *pdata = buf;
		for (; cnt >= 8; cnt -= 8)
			mci_writeq(host, DATA(host->data_offset), *pdata++);
		buf = pdata;
	}
	/* put anything remaining in the part_buf */
	if (cnt) {
		dw_mci_set_part_bytes(host, buf, cnt);
		/* Push data if we have reached the expected data length */
		if ((data->bytes_xfered + init_cnt) ==
		    (data->blksz * data->blocks))
			mci_writeq(host, DATA(host->data_offset),
				   host->part_buf);
	}
}

static void dw_mci_pull_data64(struct dw_mci *host, void *buf, int cnt)
{
#ifndef CONFIG_HAVE_EFFICIENT_UNALIGNED_ACCESS
	if (unlikely((unsigned long)buf & 0x7)) {
		while (cnt >= 8) {
			/* pull data from fifo into aligned buffer */
			u64 aligned_buf[16];
			int len = min(cnt & -8, (int)sizeof(aligned_buf));
			int items = len >> 3;
			int i;
			for (i = 0; i < items; ++i)
				aligned_buf[i] = mci_readq(host,
						DATA(host->data_offset));
			/* memcpy from aligned buffer into output buffer */
			memcpy(buf, aligned_buf, len);
			buf += len;
			cnt -= len;
		}
	} else
#endif
	{
		u64 *pdata = buf;
		for (; cnt >= 8; cnt -= 8)
			*pdata++ = mci_readq(host, DATA(host->data_offset));
		buf = pdata;
	}
	if (cnt) {
		host->part_buf = mci_readq(host, DATA(host->data_offset));
		dw_mci_pull_final_bytes(host, buf, cnt);
	}
}

static void dw_mci_pull_data(struct dw_mci *host, void *buf, int cnt)
{
	int len;

	/* get remaining partial bytes */
	len = dw_mci_pull_part_bytes(host, buf, cnt);
	if (unlikely(len == cnt))
		return;
	buf += len;
	cnt -= len;

	/* get the rest of the data */
	host->pull_data(host, buf, cnt);
}

static void dw_mci_read_data_pio(struct dw_mci *host, bool dto)
{
	struct sg_mapping_iter *sg_miter = &host->sg_miter;
	void *buf;
	unsigned int offset;
	struct mmc_data	*data = host->data;
	int shift = host->data_shift;
	u32 status;
	unsigned int len;
	unsigned int remain, fcnt;

	do {
		if (!sg_miter_next(sg_miter))
			goto done;

		host->sg = sg_miter->__sg;
		buf = sg_miter->addr;
		remain = sg_miter->length;
		offset = 0;

		do {
			fcnt = (SDMMC_GET_FCNT(mci_readl(host, STATUS))
					<< shift) + host->part_buf_count;
			len = min(remain, fcnt);
			if (!len)
				break;
			dw_mci_pull_data(host, (void *)(buf + offset), len);
			data->bytes_xfered += len;
			offset += len;
			remain -= len;
		} while (remain);

		sg_miter->consumed = offset;
		status = mci_readl(host, MINTSTS);
		mci_writel(host, RINTSTS, SDMMC_INT_RXDR);
	/* if the RXDR is ready read again */
	} while ((status & SDMMC_INT_RXDR) ||
		 (dto && SDMMC_GET_FCNT(mci_readl(host, STATUS))));

	if (!remain) {
		if (!sg_miter_next(sg_miter))
			goto done;
		sg_miter->consumed = 0;
	}
	sg_miter_stop(sg_miter);
	//xlog_mmc_log(host->host_id, "dw_mci_read_data_pio", data->bytes_xfered, 0, 0);
	return;

done:
	sg_miter_stop(sg_miter);
	host->sg = NULL;
	smp_wmb();
	set_bit(EVENT_XFER_COMPLETE, &host->pending_events);
	//xlog_mmc_log(host->host_id, "dw_mci_read_data_pio done", data->bytes_xfered, 0, 0);
}

static void dw_mci_write_data_pio(struct dw_mci *host)
{
	struct sg_mapping_iter *sg_miter = &host->sg_miter;
	void *buf;
	unsigned int offset;
	struct mmc_data	*data = host->data;
	int shift = host->data_shift;
	u32 status;
	unsigned int len;
	unsigned int fifo_depth = host->fifo_depth;
	unsigned int remain, fcnt;

	do {
		if (!sg_miter_next(sg_miter))
			goto done;

		host->sg = sg_miter->__sg;
		buf = sg_miter->addr;
		remain = sg_miter->length;
		offset = 0;

		do {
			fcnt = ((fifo_depth -
				 SDMMC_GET_FCNT(mci_readl(host, STATUS)))
					<< shift) - host->part_buf_count;
			len = min(remain, fcnt);
			if (!len)
				break;
			host->push_data(host, (void *)(buf + offset), len);
			data->bytes_xfered += len;
			offset += len;
			remain -= len;
		} while (remain);

		sg_miter->consumed = offset;
		status = mci_readl(host, MINTSTS);
		mci_writel(host, RINTSTS, SDMMC_INT_TXDR);
	} while (status & SDMMC_INT_TXDR); /* if TXDR write again */

	if (!remain) {
		if (!sg_miter_next(sg_miter))
			goto done;
		sg_miter->consumed = 0;
	}
	sg_miter_stop(sg_miter);
	//xlog_mmc_log(host->host_id, "dw_mci_write_data_pio", data->bytes_xfered, 0, 0);
	return;

done:
	sg_miter_stop(sg_miter);
	host->sg = NULL;
	smp_wmb();
	set_bit(EVENT_XFER_COMPLETE, &host->pending_events);
	//xlog_mmc_log(host->host_id, "dw_mci_write_data_pio done", data->bytes_xfered, 0, 0);
}

static void dw_mci_cmd_interrupt(struct dw_mci *host, u32 status)
{
	if (!host->cmd_status)
		host->cmd_status = status;

	smp_wmb();

	set_bit(EVENT_CMD_COMPLETE, &host->pending_events);
	#ifdef DW_MCI_THREAD
	dw_mci_kick_thread(host);
	#else
	tasklet_schedule(&host->tasklet);
	#endif

}

static irqreturn_t dw_mci_interrupt(int irq, void *dev_id)
{
	struct dw_mci *host = dev_id;
	u32 pending;
	u32 rpending;
	u32 dma_pending;
	int i;

	pending = mci_readl(host, MINTSTS); /* read-only mask reg */
	/*
	 * DTO fix - version 2.10a and below, and only if internal DMA
	 * is configured.
	 */
	if (host->quirks & DW_MCI_QUIRK_IDMAC_DTO) {
		if (!pending &&
		    ((mci_readl(host, STATUS) >> 17) & 0x1fff))
			pending |= SDMMC_INT_DATA_OVER;
	}

	if (pending) {
		#if 0
		if (rpending & (DW_MCI_CMD_ERROR_FLAGS |SDMMC_INT_CMD_DONE)) {
			mci_writel(host, RINTSTS, DW_MCI_CMD_ERROR_FLAGS |SDMMC_INT_CMD_DONE );
			host->cmd_status = rpending;
			//dwc_mci_errlog(pending);
			smp_wmb();
			//set_bit(EVENT_CMD_COMPLETE, &host->pending_events);

			dw_mci_cmd_interrupt(host, rpending);
		}
		#endif

		if (pending & DW_MCI_DATA_ERROR_FLAGS) {
			/* if there is an error report DATA_ERROR */
			mci_writel(host, RINTSTS, DW_MCI_DATA_ERROR_FLAGS);
			host->data_status = pending;
			smp_wmb();
			set_bit(EVENT_DATA_ERROR, &host->pending_events);
			set_bit(EVENT_DATA_COMPLETE, &host->pending_events);
			#ifdef DW_MCI_THREAD
			dw_mci_kick_thread(host);
			#else
			tasklet_schedule(&host->tasklet);
			#endif

		}

		if (pending & SDMMC_INT_RXDR) {
			mci_writel(host, RINTSTS, SDMMC_INT_RXDR);
			if (host->dir_status == DW_MCI_RECV_STATUS && host->sg) {
				dw_mci_read_data_pio(host, false);
				if (mci_readl(host, MINTSTS) & SDMMC_INT_DATA_OVER)
					pending |= SDMMC_INT_DATA_OVER;
			}
		}

		if (pending & SDMMC_INT_DATA_OVER) {
			mci_writel(host, RINTSTS, SDMMC_INT_DATA_OVER);
			if (!host->data_status)
				host->data_status = pending;
			smp_wmb();
			if (host->dir_status == DW_MCI_RECV_STATUS) {
				if (host->sg != NULL)
					dw_mci_read_data_pio(host, true);
			}
			set_bit(EVENT_DATA_COMPLETE, &host->pending_events);
			#ifdef DW_MCI_THREAD
			dw_mci_kick_thread(host);
			#else
			tasklet_schedule(&host->tasklet);
			#endif
		}

		if (pending & SDMMC_INT_TXDR) {
			mci_writel(host, RINTSTS, SDMMC_INT_TXDR);
			if (host->dir_status == DW_MCI_SEND_STATUS && host->sg)
				dw_mci_write_data_pio(host);
		}
		if (pending & SDMMC_INT_CD) {
			xlog_mmc_log(host->host_id, "INFO dw_mci_interrupt, INT CD", 0, 0, 0);
			mci_writel(host, RINTSTS, SDMMC_INT_CD);
			//queue_work(host->card_workqueue, &host->card_work);
		}

		/* Handle SDIO Interrupts */
		for (i = 0; i < host->num_slots; i++) {
			struct dw_mci_slot *slot = host->slot[i];
			if (pending & SDMMC_INT_SDIO(i)) {
				mci_writel(host, RINTSTS, SDMMC_INT_SDIO(i));
				mmc_signal_sdio_irq(slot->mmc);
			}
		}

	}

	rpending = mci_readl(host, RINTSTS);
	if (rpending & (DW_MCI_ERROR_FLAGS | SDMMC_INT_CMD_DONE)) {
		mci_writel(host, RINTSTS, DW_MCI_ERROR_FLAGS | SDMMC_INT_CMD_DONE );
		host->cmd_status = rpending;
		smp_wmb();
		dw_mci_cmd_interrupt(host, rpending);
	}

#ifdef CONFIG_MMC_DW_IDMAC
	/* Handle DMA interrupts */
	dma_pending = mci_readl(host, IDSTS);
	if (dma_pending & (SDMMC_IDMAC_INT_TI | SDMMC_IDMAC_INT_RI)) {
		mci_writel(host, IDSTS, SDMMC_IDMAC_INT_TI | SDMMC_IDMAC_INT_RI);
		mci_writel(host, IDSTS, SDMMC_IDMAC_INT_NI);
		if (!test_bit(EVENT_DATA_ERROR, &host->pending_events))
			host->dma_ops->complete(host);
	}
#endif
#if 0
	xlog_mmc_log(host->host_id, "dw_mci_interrupt", pending, rpending, dma_pending);
#endif
	return IRQ_HANDLED;
}
void mmc_enable_irq(struct mmc_host *mmc,int enable)
{
	struct dw_mci_slot *slot = mmc_priv(mmc);
	
#ifdef _USE_VEHICLE_DC
	if(slot->host->host_id != 1)
		return;
	
	mci_writel(slot->host, DBADDR, (u32)slot->host->sg_dma);

	if((slot == NULL)||(slot->host==NULL)||(slot->host->irq==NULL))
		return;
	if(enable){
		enable_irq(slot->host->irq);
	}
	else{
		disable_irq(slot->host->irq);
	}
#endif	
	return;

}

static void dw_mci_work_routine_card(struct work_struct *work)
{
	struct dw_mci *host = container_of(work, struct dw_mci, card_work);
	int i;

	for (i = 0; i < host->num_slots; i++) {
		struct dw_mci_slot *slot = host->slot[i];
		struct mmc_host *mmc = slot->mmc;
		struct mmc_request *mrq;
		int present;

		present = dw_mci_get_cd(mmc);
		xlog_mmc_log(host->host_id, "ERROR dw_mci_work_routine_card", slot->last_detect_state, host->state, 0);
		while (present != slot->last_detect_state) {
			dev_dbg(&slot->mmc->class_dev, "card %s\n",
				present ? "inserted" : "removed");

			DW_MCI_SPIN_LOCK_BH(&host->lock);

			/* Card change detected */
			slot->last_detect_state = present;
			if (present)
				host->dw_dto_timer_expired_cnt = 0;

			/* Clean up queue if present */
			mrq = slot->mrq;
			if (mrq) {
				if (mrq == host->mrq) {
					host->data = NULL;
					host->cmd = NULL;

					switch (host->state) {
					case STATE_IDLE:
						break;
					case STATE_SENDING_CMD:
						mrq->cmd->error = -ENOMEDIUM;
						if (!mrq->data)
							break;
						/* fall through */
					case STATE_SENDING_DATA:
						mrq->data->error = -ENOMEDIUM;
						dw_mci_stop_dma(host);
						break;
					case STATE_DATA_BUSY:
					case STATE_DATA_ERROR:
						if (mrq->data->error == -EINPROGRESS)
							mrq->data->error = -ENOMEDIUM;
						/* fall through */
					case STATE_SENDING_STOP:
						if (mrq->stop)
							mrq->stop->error = -ENOMEDIUM;
						break;
					}

					dw_mci_request_end(host, mrq);
				} else {
					list_del(&slot->queue_node);
					mrq->cmd->error = -ENOMEDIUM;
					if (mrq->data)
						mrq->data->error = -ENOMEDIUM;
					if (mrq->stop)
						mrq->stop->error = -ENOMEDIUM;

					spin_unlock(&host->lock);
					mmc_request_done(slot->mmc, mrq);
					spin_lock(&host->lock);
				}
			}

			/* Power down slot */
			if (present == 0) {
				/* Clear down the FIFO */
				dw_mci_fifo_reset(host);
#ifdef CONFIG_MMC_DW_IDMAC
				dw_mci_idmac_reset(host);
#endif

			}

			DW_MCI_SPIN_UNLOCK_BH(&host->lock);
			present = dw_mci_get_cd(mmc);
		}

		mmc_detect_change(slot->mmc,
			msecs_to_jiffies(host->pdata->detect_delay_ms));
	}
}

static int  dw_mci_init_slot(struct dw_mci *host, unsigned int id)
{
	struct mmc_host *mmc;
	struct dw_mci_slot *slot;

	if(id >= MAX_MCI_SLOTS)
		return -EINVAL;
	
	mmc = mmc_alloc_host(sizeof(struct dw_mci_slot), host->dev);
	if (!mmc)
		return -ENOMEM;

	slot = mmc_priv(mmc);
	slot->id = id;
	slot->mmc = mmc;
	slot->host = host;

	mmc->ops = &dw_mci_ops;
	mmc->f_min = DIV_ROUND_UP(host->bus_hz, 510);
	mmc->f_max = host->bus_hz;

	if (host->pdata->get_ocr)
		mmc->ocr_avail = host->pdata->get_ocr(id);
	else
		mmc->ocr_avail = MMC_VDD_32_33 | MMC_VDD_33_34;

	/*
	 * Start with slot power disabled, it will be enabled when a card
	 * is detected.
	 */
	if (host->pdata->setpower)
		host->pdata->setpower(id, 0);

	if (host->pdata->caps)
		mmc->caps = host->pdata->caps;

	if (host->pdata->caps2)
		mmc->caps2 = host->pdata->caps2;

	if (host->pdata->pm_caps & MMC_PM_KEEP_POWER)
		mmc->pm_caps =  host->pdata->pm_caps;

	mmc->pm_flags = MMC_PM_IGNORE_PM_NOTIFY;
	if (host->pdata->get_bus_wd)
		if (host->pdata->get_bus_wd(slot->id) >= 4)
			mmc->caps |= MMC_CAP_4_BIT_DATA;


	if (host->pdata->blk_settings) {
		mmc->max_segs = host->pdata->blk_settings->max_segs;
		mmc->max_blk_size = host->pdata->blk_settings->max_blk_size;
		mmc->max_blk_count = host->pdata->blk_settings->max_blk_count;
		mmc->max_req_size = host->pdata->blk_settings->max_req_size;
		mmc->max_seg_size = host->pdata->blk_settings->max_seg_size;
	} else {
		/* Useful defaults if platform data is unset. */
#ifdef CONFIG_MMC_DW_IDMAC
		mmc->max_segs = PAGE_SIZE/sizeof(struct idmac_desc);
		mmc->max_blk_size = 65536;
		//mmc->max_blk_count = host->ring_size;
		mmc->max_blk_count = 128;
//		mmc->max_seg_size = 0x1000;
        mmc->max_seg_size = 65536;
		mmc->max_req_size = mmc->max_seg_size * mmc->max_blk_count;
#else
		mmc->max_segs = 64;
		mmc->max_blk_size = 65536; /* BLKSIZ is 16 bits */
		mmc->max_blk_count = 512;
		mmc->max_req_size = mmc->max_blk_size * mmc->max_blk_count;
		mmc->max_seg_size = mmc->max_req_size;
#endif /* CONFIG_MMC_DW_IDMAC */
	}

	host->vmmc = regulator_get(mmc_dev(mmc), "vmmc");
	if (IS_ERR(host->vmmc)) {
		pr_info("%s: no vmmc regulator found\n", mmc_hostname(mmc));
		host->vmmc = NULL;
	} else
		regulator_enable(host->vmmc);

	if (dw_mci_get_cd(mmc))
		set_bit(DW_MMC_CARD_PRESENT, &slot->flags);
	else
		clear_bit(DW_MMC_CARD_PRESENT, &slot->flags);

	host->slot[id] = slot;

	if(host->host_id == 0)
	{
		wake_lock_init(&mmc->work_wake_lock , WAKE_LOCK_SUSPEND,  "mmc_work_wakelock0");
	}
	else if(host->host_id == 1)
	{
		wake_lock_init(&mmc->work_wake_lock , WAKE_LOCK_SUSPEND,  "mmc_work_wakelock1");
	}

	mmc_add_host(mmc);


#if defined(CONFIG_DEBUG_FS)
	dw_mci_init_debugfs(slot);
#endif

	/* Card initially undetected */
	slot->last_detect_state = 0;

	/*
	 * Card may have been plugged in prior to boot so we
	 * need to run the detect tasklet
	 */
	queue_work(host->card_workqueue, &host->card_work);

	return 0;
}

static void dw_mci_cleanup_slot(struct dw_mci_slot *slot, unsigned int id)
{
	/* Shutdown detect IRQ */
	if (slot->host->pdata->exit)
		slot->host->pdata->exit(id);

	/* Debugfs stuff is cleaned up by mmc core */
	mmc_remove_host(slot->mmc);
	slot->host->slot[id] = NULL;
	mmc_free_host(slot->mmc);
}

static void dw_mci_init_dma(struct dw_mci *host)
{
#ifdef CONFIG_MMC_DW_IDMAC
	unsigned char* idma_buffer;
#endif
	if(!host->sg_cpu)
	{
	/* Alloc memory for sg translation */
	host->sg_cpu = dma_alloc_coherent(host->dev, PAGE_SIZE,
					  &host->sg_dma, GFP_KERNEL);
	if (!host->sg_cpu) {
		dev_err(host->dev, "%s: could not alloc DMA memory\n",
			__func__);
		goto no_dma;
	}
	}
	/* Determine which DMA interface to use */
#ifdef CONFIG_MMC_DW_IDMAC
	idma_buffer = (unsigned char*)host->sg_cpu;
	memset(idma_buffer, 0x0, PAGE_SIZE);
	host->idma_des = (struct idmac_desc*)idma_buffer;
	host->ring_size = PAGE_SIZE/sizeof(struct idmac_desc);


	host->dma_ops = &dw_mci_idmac_ops;
	dev_info(host->dev, "Using internal DMA controller.\n");
#endif
#ifdef CONFIG_MMC_ZX29_EDMAC
	host->dma_ops = &dw_mci_edmac_ops;
	dev_info(host->dev, "Using External DMA controller.\n");
#endif

	if (!host->dma_ops)
		goto no_dma;

	if (host->dma_ops->init && host->dma_ops->start &&
	    host->dma_ops->stop && host->dma_ops->cleanup) {
		if (host->dma_ops->init(host)) {
			dev_err(host->dev, "%s: Unable to initialize "
				"DMA Controller.\n", __func__);
			goto no_dma;
		}
	} else {
		dev_err(host->dev, "DMA initialization not found.\n");
		goto no_dma;
	}

	host->use_dma = 1;
	return;

no_dma:
	dev_info(host->dev, "Using PIO mode.\n");
	host->use_dma = 0;
	return;
}

static bool dw_mci_ctrl_reset(struct dw_mci *host, u32 reset)
{
	unsigned long timeout = jiffies + msecs_to_jiffies(500);
	u32 ctrl;

	ctrl = mci_readl(host, CTRL);
	ctrl |= reset;
	mci_writel(host, CTRL, ctrl);

	if (reset == SDMMC_CTRL_DMA_RESET)
		xlog_mmc_log(host->host_id, "dw_mci_ctrl_reset", reset, 0, 0);
	else
		xlog_mmc_log_key(host->host_id, "dw_mci_ctrl_reset", reset, 0, 0);

	/* wait till resets clear */
	do {
		ctrl = mci_readl(host, CTRL);
		if (!(ctrl & reset))
			return true;
	} while (time_before(jiffies, timeout));

	dev_err(host->dev,
		"Timeout resetting block (ctrl reset %#x)\n",
		ctrl & reset);

	xlog_mmc_log_key(host->host_id, "ERROR dw_mci_ctrl_reset", reset, 0, 0);
	return false;
}

static inline bool dw_mci_fifo_reset(struct dw_mci *host)
{
	/*
	 * Reseting generates a block interrupt(DATA Over?), hence setting
	 * the scatter-gather pointer to NULL.
	 */
	#if 1
	if (host->sg) {
		unsigned long flags;

		local_irq_save(flags);
		sg_miter_stop(&host->sg_miter);
		local_irq_restore(flags);

		host->sg = NULL;
	}
	#endif

	return dw_mci_ctrl_reset(host, SDMMC_CTRL_FIFO_RESET);
}

static inline bool dw_mci_ctrl_all_reset(struct dw_mci *host)
{
	return dw_mci_ctrl_reset(host,
				 SDMMC_CTRL_FIFO_RESET |
				 SDMMC_CTRL_RESET |
				 SDMMC_CTRL_DMA_RESET);
}

#if 0
//#ifdef CONFIG_OF
#if 0
static struct dw_mci_of_quirks {
	char *quirk;
	int id;
} of_quirks[] = {
	{
		.quirk	= "broken-cd",
		.id	= DW_MCI_QUIRK_BROKEN_CARD_DETECTION,
	},
};

static struct dw_mci_board *dw_mci_parse_dt(struct dw_mci *host)
{
	struct dw_mci_board *pdata;
	struct device *dev = host->dev;
	struct device_node *np = dev->of_node;
	const struct dw_mci_drv_data *drv_data = host->drv_data;
	int idx, ret;
	u32 clock_frequency;

	pdata = devm_kzalloc(dev, sizeof(*pdata), GFP_KERNEL);
	if (!pdata) {
		dev_err(dev, "could not allocate memory for pdata\n");
		return ERR_PTR(-ENOMEM);
	}

	/* find out number of slots supported */
	if (of_property_read_u32(dev->of_node, "num-slots",
				&pdata->num_slots)) {
		dev_info(dev, "num-slots property not found, "
				"assuming 1 slot is available\n");
		pdata->num_slots = 1;
	}

	/* get quirks */
	for (idx = 0; idx < ARRAY_SIZE(of_quirks); idx++)
		if (of_get_property(np, of_quirks[idx].quirk, NULL))
			pdata->quirks |= of_quirks[idx].id;

	if (of_property_read_u32(np, "fifo-depth", &pdata->fifo_depth))
		dev_info(dev, "fifo-depth property not found, using "
				"value of FIFOTH register as default\n");

	of_property_read_u32(np, "card-detect-delay", &pdata->detect_delay_ms);

	if (!of_property_read_u32(np, "clock-frequency", &clock_frequency))
		pdata->bus_hz = clock_frequency;

	if (drv_data && drv_data->parse_dt) {
		ret = drv_data->parse_dt(host);
		if (ret)
			return ERR_PTR(ret);
	}

	if (of_find_property(np, "keep-power-in-suspend", NULL))
		pdata->pm_caps |= MMC_PM_KEEP_POWER;

	if (of_find_property(np, "enable-sdio-wakeup", NULL))
		pdata->pm_caps |= MMC_PM_WAKE_SDIO_IRQ;

	if (of_find_property(np, "supports-highspeed", NULL))
		pdata->caps |= MMC_CAP_SD_HIGHSPEED | MMC_CAP_MMC_HIGHSPEED;

	if (of_find_property(np, "caps2-mmc-hs200-1_8v", NULL))
		pdata->caps2 |= MMC_CAP2_HS200_1_8V_SDR;

	if (of_find_property(np, "caps2-mmc-hs200-1_2v", NULL))
		pdata->caps2 |= MMC_CAP2_HS200_1_2V_SDR;

	if (of_get_property(np, "cd-inverted", NULL))
		pdata->caps2 |= MMC_CAP2_CD_ACTIVE_HIGH;

	return pdata;
}

#else /* CONFIG_OF */
static struct dw_mci_board *dw_mci_parse_dt(struct dw_mci *host)
{
	return ERR_PTR(-EINVAL);
}
#endif /* CONFIG_OF */
#endif

#if (defined CONFIG_ARCH_ZX297520V2)||(defined CONFIG_ARCH_ZX297520V3)

static irqreturn_t dw_mci_detetect_int(int irq, void *dev_id)
{
	struct dw_mci *host = dev_id;
#if 0
	int gpio_val = 0;
	int gpio = PIN_MMC_TF_CARD_DET;
#endif

	disable_irq_nosync(irq);
	pcu_int_clear(PCU_EX4_INT);

	xlog_mmc_log(host->host_id, "INFO dw_mci_detetect_int", 0, 0, 0);
#if 0
	gpio_val = dwc_mci_get_detect_val(gpio);

	xlog_mmc_log(host->host_id, "INFO dw_mci_detetect_int", gpio_val, 0, 0, 0);

	if (gpio_val == 1){
		//zx29_gpio_set_inttype(gpio, IRQ_TYPE_LEVEL_LOW);
	}
	else{
		//zx29_gpio_set_inttype(gpio, IRQ_TYPE_LEVEL_HIGH);
	}
	pcu_int_clear(PCU_EX4_INT);
#endif

	return IRQ_WAKE_THREAD;
}

static irqreturn_t dw_mci_detetect_thread(int irq, void *dev_id)
{
    int gpio_val = 0;
	struct dw_mci *host = dev_id;
	unsigned char gpio;

	wake_lock(&sd_detect_wake_lock);

	gpio = PIN_MMC_TF_CARD_DET;

	zx29_gpio_config(gpio, PIN_MMC_TF_CARD_DET_GPIO_FUNC);
	msleep(30);
	gpio_val = dwc_mci_get_detect_val(gpio);

	zx29_gpio_config(gpio, PIN_MMC_TF_CARD_DET_FUNC);
	msleep(30);

	xlog_mmc_log(host->host_id, "INFO dw_mci_detetect_thread", gpio_val, 0, 0);

	// card pluged out
	if (gpio_val == 1){
		// msleep(120);		// the delay is important
		irq_set_irq_type(irq, IRQ_TYPE_LEVEL_LOW);
	}
	else{
		irq_set_irq_type(irq, IRQ_TYPE_LEVEL_HIGH);
	}

	queue_work(host->card_workqueue, &host->card_work);
	enable_irq(irq);
	wake_unlock(&sd_detect_wake_lock);

    return IRQ_HANDLED;
}

static int dwc_mci_sdcard_init(struct dw_mci *host)
{
	int ret = 0;
	int gpio_val = 0;
	int ext_int0_no = 0;
	unsigned char gpio = 0;
	unsigned int type;

	gpio = PIN_MMC_TF_CARD_DET;
	ret = gpio_request(gpio, "sd_detect");
	if (ret) {
		mmc_printk("reset gpio request error.\n");
		return ret;
	}

	ret = gpio_direction_input(gpio);
	if (ret)
		return ret;

	zx29_gpio_config(gpio, PIN_MMC_TF_CARD_DET_GPIO_FUNC);
	zx29_gpio_pd_pu_set(gpio, IO_CFG_PULL_DISABLE);
	ext_int0_no = gpio_to_irq(gpio);
	irq_set_irq_wake(ext_int0_no, 1);

	/* wait 30ms for HW voltage rising or down */
	msleep(30);
	gpio_val = dwc_mci_get_detect_val(gpio);
	zx29_gpio_config(gpio, PIN_MMC_TF_CARD_DET_FUNC);
	msleep(30);

	xlog_mmc_log(host->host_id, "INFO dwc_mci_sdcard_init", gpio, gpio_val, ext_int0_no);
	mmc_printk("INFO %s, %d %d %d\n", __func__, gpio, gpio_val, ext_int0_no);
	if (gpio_val < 0){
		mmc_printk("get gpio val error.\n");
	} else {
		pcu_int_clear(PCU_EX4_INT);
		if (gpio_val == 1){
			//zx29_gpio_set_inttype(gpio, IRQ_TYPE_LEVEL_LOW);
			type = IRQF_TRIGGER_LOW;
		} else{
			//zx29_gpio_set_inttype(gpio, IRQ_TYPE_LEVEL_HIGH);
			type = IRQF_TRIGGER_HIGH;
		}
		pcu_int_clear(PCU_EX4_INT);
//		ret = request_irq(ext_int0_no, dw_mci_detetect_int, host->irq_flags | type | IRQF_NO_THREAD | IRQF_ONESHOT, "sd_detect", host);
		ret = request_threaded_irq(ext_int0_no, dw_mci_detetect_int, dw_mci_detetect_thread,
									/*host->irq_flags |*/ type | IRQF_NO_THREAD | IRQF_ONESHOT, "sd_detect", host);
		if(ret)
			return ret;
	}

	//device_init_wakeup(host->dev, true);

	return 0;
}



static void dw_mci_strengthen_driver(int id)
{
	u32 drvcap_clkcmd = 0;
	u32 drvcap_data = 0;

#ifdef CONFIG_ARCH_ZX297520V3
	/* sd0 */
	if (id == 0) {
		mmc_printk("%s, %u:\tclkcmd:0x%08x data:0x%08x\n", __func__, __LINE__, ioread32(ZX_PAD_CTRL_BASE+0x0810), ioread32(ZX_PAD_CTRL_BASE+0x0814));
		drvcap_clkcmd = ioread32(ZX_PAD_CTRL_BASE+0x0810);
		drvcap_clkcmd |= (1<<0) | (1<<1) |(1<<8) | (1<<16) | (1<<24);    //clk 3 data 1 cmd 1
		//drvcap_clkcmd |= (2<<0)|(1<<8) | (1<<16) | (1<<24);    //clk2 data 1 cmd 1
		iowrite32(drvcap_clkcmd,(ZX_PAD_CTRL_BASE+0x0810));

		drvcap_data = ioread32(ZX_PAD_CTRL_BASE+0x0814);
		drvcap_data |= (1<<0) | (1<<8);
		iowrite32(drvcap_data,(ZX_PAD_CTRL_BASE+0x0814));
		mmc_printk("%s, %u:\tclkcmd:0x%08x data:0x%08x\n", __func__, __LINE__, ioread32(ZX_PAD_CTRL_BASE+0x0810), ioread32(ZX_PAD_CTRL_BASE+0x0814));
	}
	/* sd1 */
	else if (id == 1) {
		mmc_printk("%s, %u:\tclkcmd:0x%08x data:0x%08x\n", __func__, __LINE__, ioread32(ZX_PAD_CTRL_BASE+0x0814), ioread32(ZX_PAD_CTRL_BASE+0x0818));
		drvcap_clkcmd = ioread32(ZX_PAD_CTRL_BASE+0x0814);
		drvcap_clkcmd |= (3<<16) | (1<<24);    //clk 3 data 1 cmd 1
		iowrite32(drvcap_clkcmd,(ZX_PAD_CTRL_BASE+0x0814));

		drvcap_data = ioread32(ZX_PAD_CTRL_BASE+0x0818);
		drvcap_data |= (1<<0) | (1<<8) | (1<<16) | (1<<24);
		iowrite32(drvcap_data,(ZX_PAD_CTRL_BASE+0x0818));
		mmc_printk("%s, %u:\tclkcmd:0x%08x data:0x%08x\n", __func__, __LINE__, ioread32(ZX_PAD_CTRL_BASE+0x0814), ioread32(ZX_PAD_CTRL_BASE+0x0818));
	}
#endif
#ifdef CONFIG_ARCH_ZX297520V2
	mmc_printk("%s, %u:\tclkcmd:0x%08x data:0x%08x\n", __func__, __LINE__, ioread32(ZX_PAD_CTRL_BASE+0x0814), ioread32(ZX_PAD_CTRL_BASE+0x0818));
	drvcap_clkcmd = ioread32(ZX_PAD_CTRL_BASE+0x0814);
	drvcap_clkcmd |= (1<<16) | (1<<17)| (1<<24);
	iowrite32(drvcap_clkcmd,(ZX_PAD_CTRL_BASE+0x0814));

	drvcap_data = ioread32(ZX_PAD_CTRL_BASE+0x0818);
	drvcap_data |= (1<<0) | (1<<8) | (1<<16) | (1<<24);
	iowrite32(drvcap_data,(ZX_PAD_CTRL_BASE+0x0818));
	mmc_printk("%s, %u:\tclkcmd:0x%08x data:0x%08x\n", __func__, __LINE__, ioread32(ZX_PAD_CTRL_BASE+0x0814), ioread32(ZX_PAD_CTRL_BASE+0x0818));

#endif
	return ;
}
extern void mmc_stop_host(struct mmc_host *host);
void dw_mci_rescan_card(unsigned id, unsigned insert)
{
	struct dw_mci *dw_host = NULL;
	struct dw_mci_slot *slot = NULL;
	//struct mmc_host *mmc = slot->mmc;

	BUG_ON(id > 3);
	//BUG_ON(dw_mmc_host[id] == NULL);
	if (dw_mmc_host[id] == NULL)
		return;
	dw_host = dw_mmc_host[id];

	#if 1
	slot = dw_host->slot[0];
	slot->last_detect_state = insert ? 1 : 0;

	if(insert == 0)
	{
		mmc_stop_host(slot->mmc);
		return;
	}

	mmc_detect_change(slot->mmc, 0);
	#endif
	//queue_work(dw_host->card_workqueue, &dw_host->card_work);
	//printk("%s,%d\n",__func__,__LINE__);
	return;
}
EXPORT_SYMBOL_GPL(dw_mci_rescan_card);
/* Started by AICoder, pid:x4ec9tda59644d014f920a808038ef3af5296358 */
int zx29_sdio_get_device(u8 id)
{
    struct mmc_host *host;
    struct mmc_card *card;
    struct sdio_func *func;


    if (!dw_mmc_host[id] || !dw_mmc_host[id]->slot[0]) {
        printk(KERN_ERR "Invalid host or slot for id %d\n", id);
        return -4;
    }

    host = dw_mmc_host[id]->slot[0]->mmc;
    if (!host) {
        printk(KERN_ERR "Host is NULL for id %d\n", id);
        return -5;
    }

    card = host->card;
    if (!card) {
        return -6;
    }

    if (card->type != MMC_TYPE_SDIO) {
        return -7;
    }

    func = card->sdio_func[0];
    if (!func) {
        printk(KERN_ERR "No SDIO function found in card of host %s\n", mmc_hostname(host));
        return -8;
    }

    printk("Found SDIO function %d: vendor=0x%x, device=0x%x\n", func->num, func->vendor, func->device);
    return 0;
}
EXPORT_SYMBOL_GPL(zx29_sdio_get_device);
/* Ended by AICoder, pid:x4ec9tda59644d014f920a808038ef3af5296358 */

void zx29_mci_enable_sdio_irq(u32 host_id, int enable)
{
	struct dw_mci *dw_host = dw_mmc_host[host_id];

	enable_wifi_irq = (unsigned int )enable;
	if(dw_host)
		dw_mci_enable_sdio_irq(dw_host->slot[0]->mmc, enable);
}

EXPORT_SYMBOL_GPL(zx29_mci_enable_sdio_irq);

#endif

void mmc_spin_lock(struct dw_mci *host)
{
	
#ifdef _USE_VEHICLE_DC
	if(host->host_id==1)
		soft_spin_lock(SD1_SFLOCK);
#endif	
	return;
}
void mmc_spin_unlock(struct dw_mci *host)
{
	
#ifdef _USE_VEHICLE_DC
	if(host->host_id==1)
		soft_spin_unlock(SD1_SFLOCK);
#endif	
	return;
}

int dw_mci_probe(struct dw_mci *host)
{
	//const struct dw_mci_drv_data *drv_data = host->drv_data;
	struct dw_mci_board *brd = host->pdata;
	int width, i, ret = 0;
	u32 fifo_size;
	int init_slots = 0;
	u32 clk_phase = 0;
	u32 regs = 0;

#if (defined CONFIG_ARCH_ZX297520V2)||(defined CONFIG_ARCH_ZX297520V3)
	if (host->host_id == 0){
		ramdump_ram_conf_table_add("sd0_reg_0x01210000.bin", 0x01210000, 0x110, (unsigned long)host->regs, RAMDUMP_FLAG_LEVEL_MEDIUM, 0);
	}else{
		ramdump_ram_conf_table_add("sd1_reg_0x01211000.bin", 0x01211000, 0x110, (unsigned long)host->regs, RAMDUMP_FLAG_LEVEL_MEDIUM, 0);
	}
#endif

	if (!host->pdata->select_slot && host->pdata->num_slots > 1) {
		dev_err(host->dev,
			"Platform data must supply select_slot function\n");
		return -ENODEV;
	}
	if (!host->pdata->bus_hz) {
		dev_err(host->dev,
			"Platform data must supply bus speed\n");
		return -ENODEV;
	}

	ret = xlog_mmc_probe(host->host_id);
	if (ret)
		return ret;

	/* host->bus_hz = host->pdata->bus_hz; */
	host->quirks = host->pdata->quirks;

	spin_lock_init(&host->lock);
	INIT_LIST_HEAD(&host->queue);
	
	mmc_spin_lock(host);

	/*
	 * Get the host data width - this assumes that HCON has been set with
	 * the correct values.
	 */
	i = (mci_readl(host, HCON) >> 7) & 0x7;
	if (!i) {
		host->push_data = dw_mci_push_data16;
		host->pull_data = dw_mci_pull_data16;
		width = 16;
		host->data_shift = 1;
	} else if (i == 2) {
		host->push_data = dw_mci_push_data64;
		host->pull_data = dw_mci_pull_data64;
		width = 64;
		host->data_shift = 3;
	} else {
		/* Check for a reserved value, and warn if it is */
		WARN((i != 1),
		     "HCON reports a reserved host data width!\n"
		     "Defaulting to 32-bit access.\n");
		host->push_data = dw_mci_push_data32;
		host->pull_data = dw_mci_pull_data32;
		width = 32;
		host->data_shift = 2;
	}

	/* Reset all blocks */
	if (!dw_mci_ctrl_all_reset(host))
		return -ENODEV;

	host->dma_ops = host->pdata->dma_ops;
	host->sg_cpu = NULL;
	dw_mci_init_dma(host);

	/* Clear the interrupts for the host controller */
	mci_writel(host, RINTSTS, 0xFFFFFFFF);
	mci_writel(host, INTMASK, 0); /* disable all mmc interrupt first */

	/* Put in max timeout */
	mci_writel(host, TMOUT, 0xFFFFFFFF);

	/*
	 * FIFO threshold settings  RxMark  = fifo_size / 2 - 1,
	 *                          Tx Mark = fifo_size / 2 DMA Size = 8
	 */
	if (!host->pdata->fifo_depth) {
		/*
		 * Power-on value of RX_WMark is FIFO_DEPTH-1, but this may
		 * have been overwritten by the bootloader, just like we're
		 * about to do, so if you know the value for your hardware, you
		 * should put it in the platform data.
		 */
		fifo_size = mci_readl(host, FIFOTH);
		dev_info(host->dev, "FIFO Threshold 0x%X\n", fifo_size);
		fifo_size = 1 + ((fifo_size >> 16) & 0xfff);
	} else {
		fifo_size = host->pdata->fifo_depth;
	}
	if (fifo_size > 128)
		fifo_size = 128;
	host->fifo_depth = fifo_size;
	host->fifoth_val =
		SDMMC_SET_FIFOTH(0x2, fifo_size / 2 - 1, fifo_size / 2);
	mci_writel(host, FIFOTH, host->fifoth_val);

	/* disable clock to CIU */
	mci_writel(host, CLKENA, 0);
	mci_writel(host, CLKSRC, 0);

	/*
	 * In 2.40a spec, Data offset is changed.
	 * Need to check the version-id and set data-offset for DATA register.
	 */
	host->verid = SDMMC_GET_VERID(mci_readl(host, VERID));
	dev_info(host->dev, "Version ID %04x\n", host->verid);
	xlog_mmc_log(host->host_id, "Version ID", host->verid, 0, 0);
#if 1
	/* zx29 mmc VERID is 210a, and DATA is 0x200. */
	host->data_offset = DATA_240A_OFFSET;
#else
	if (host->verid < DW_MMC_240A)
		host->data_offset = DATA_OFFSET;
	else
		host->data_offset = DATA_240A_OFFSET;
#endif

#ifdef DW_MCI_THREAD
	sema_init(&host->mmc_sema,1);
	host->dw_mci_flag = 0;

	host->sdio_thread  = kthread_run(dw_mci_tasklet_func, host, "zx29_mmc%d", host->host_id);
#else
	tasklet_init(&host->tasklet, dw_mci_tasklet_func, (unsigned long)host);
#endif
	host->card_workqueue = alloc_workqueue("dw-mci-card",
			WQ_MEM_RECLAIM | WQ_NON_REENTRANT, 1);
	if (!host->card_workqueue) {
		ret = -ENOMEM;
		goto err_dmaunmap;
	}
	INIT_WORK(&host->card_work, dw_mci_work_routine_card);
	ret = request_irq(host->irq, dw_mci_interrupt, host->irq_flags | IRQF_NO_THREAD , "dw-mci", host);
	if (ret)
		goto err_workqueue;
#ifdef _USE_VEHICLE_DC
	if(host->host_id ==1)
		disable_irq(host->irq);
#endif

	//ret = irq_set_irq_wake(host->irq, 1);

#if !((defined CONFIG_AIC8800)||(defined CONFIG_AIC8800D80L))
	if (host->quirks & DW_MCI_QUIRK_SDIO) {
			irq_set_irq_wake(brd->data1_irq, 1);
	}
#endif

#if (defined CONFIG_ARCH_ZX297520V2)||(defined CONFIG_ARCH_ZX297520V3)
	if (!(host->quirks & DW_MCI_QUIRK_SDIO) && !(host->quirks & DW_MCI_QUIRK_BROKEN_CARD_DETECTION))
	{
		wake_lock_init(&sd_detect_wake_lock, WAKE_LOCK_SUSPEND, "sd_detect_wakelock");
		dwc_mci_sdcard_init(host);
	}

	dw_mci_strengthen_driver(host->host_id);
#endif

	if (host->pdata->num_slots)
		host->num_slots = host->pdata->num_slots;
	else
		host->num_slots = ((mci_readl(host, HCON) >> 1) & 0x1F) + 1;
	if (host->num_slots > MAX_MCI_SLOTS)
		host->num_slots = MAX_MCI_SLOTS;
	/*
	 * Enable interrupts for command done, data over, data empty, card det,
	 * receive ready and error such as transmit, receive timeout, crc error
	 */
	mci_writel(host, RINTSTS, 0xFFFFFFFF);
	mci_writel(host, INTMASK, SDMMC_INT_CMD_DONE | SDMMC_INT_DATA_OVER |
		   SDMMC_INT_TXDR | SDMMC_INT_RXDR |
		   // xsh20160811 DW_MCI_ERROR_FLAGS | SDMMC_INT_CD );
		   DW_MCI_ERROR_FLAGS);
	mci_writel(host, CTRL, SDMMC_CTRL_INT_ENABLE); /* Enable mci interrupt */

	dev_info(host->dev, "DW MMC controller at irq %d, "
		 "%d bit host data width, "
		 "%u deep fifo\n",
		 host->irq, width, fifo_size);
	xlog_mmc_log(host->host_id, "irq width fifo_size", host->irq, width, fifo_size);
    mci_writel(host, DEBNCE, 0xfff); /* 0xfff is the default value, spec is wrong @V1.0 */

	dw_mmc_host[host->host_id] = host;
	/* We need at least one slot to succeed */
	for (i = 0; i < host->num_slots; i++) {
		ret = dw_mci_init_slot(host, i);
		if (ret)
			dev_dbg(host->dev, "slot %d init failed\n", i);
		else
			init_slots++;
	}

	if (init_slots) {
		dev_info(host->dev, "%d slots initialized\n", init_slots);
	} else {
		dev_dbg(host->dev, "attempted to initialize %d slots, "
					"but failed on all\n", host->num_slots);
		goto err_workqueue;
	}

	if (host->quirks & DW_MCI_QUIRK_IDMAC_DTO)
		dev_info(host->dev, "Internal DMAC interrupt fix enabled.\n");

#ifdef DW_COMMAND_SOFT_TIMEOUT_SUPPORT
	setup_timer(&host->dto_timer, dw_dto_timer_expired, host);
#endif

	if(host->quirks&DW_MCI_QUIRK_CLK_PHASE_TURN){		
		clk_phase = mci_readl(host, UHS_REG_EXT);
		clk_phase &= 0xc000ffff;		
		mci_writel(host, UHS_REG_EXT,clk_phase);
		printk("%s UHS_REG = 0x%x\n",__func__,mci_readl(host, UHS_REG_EXT));	
	}
	
	mmc_spin_unlock(host);
	
	return 0;

err_workqueue:
	destroy_workqueue(host->card_workqueue);

err_dmaunmap:
	if (host->use_dma && host->dma_ops->exit)
		host->dma_ops->exit(host);

err_regulator:
	if (host->vmmc) {
		regulator_disable(host->vmmc);
		regulator_put(host->vmmc);
	}

err_clk_ciu:
	if (!IS_ERR(host->ciu_clk))
		clk_disable_unprepare(host->ciu_clk);

err_clk_biu:
	if (!IS_ERR(host->biu_clk))
		clk_disable_unprepare(host->biu_clk);
	
	mmc_spin_unlock(host);
	return ret;
}
EXPORT_SYMBOL(dw_mci_probe);

void dw_mci_remove(struct dw_mci *host)
{
	int i;

	xlog_mmc_log(host->host_id, "INFO dw_mci_remove +++", 0, 0, 0);
	mmc_printk("INFO %s\n", __func__);
	mci_writel(host, RINTSTS, 0xFFFFFFFF);
	mci_writel(host, INTMASK, 0); /* disable all mmc interrupt first */

	for (i = 0; i < host->num_slots; i++) {
		dev_dbg(host->dev, "remove slot %d\n", i);
		if (host->slot[i])
			dw_mci_cleanup_slot(host->slot[i], i);
	}

	/* disable clock to CIU */
	mci_writel(host, CLKENA, 0);
	mci_writel(host, CLKSRC, 0);

	free_irq(host->irq, host);
	destroy_workqueue(host->card_workqueue);
	dma_free_coherent(host->dev, PAGE_SIZE, host->sg_cpu, host->sg_dma);

	if (host->use_dma && host->dma_ops->exit)
		host->dma_ops->exit(host);

	if (host->vmmc) {
		regulator_disable(host->vmmc);
		regulator_put(host->vmmc);
	}

	if (!IS_ERR(host->ciu_clk))
		clk_disable_unprepare(host->ciu_clk);

	if (!IS_ERR(host->biu_clk))
		clk_disable_unprepare(host->biu_clk);
	xlog_mmc_log(host->host_id, "INFO dw_mci_remove ---", 0, 0, 0);
#if (defined CONFIG_ARCH_ZX297520V2)||(defined CONFIG_ARCH_ZX297520V3)
	if (!(host->quirks & DW_MCI_QUIRK_SDIO) && !(host->quirks & DW_MCI_QUIRK_BROKEN_CARD_DETECTION)) {
		wake_lock_destroy(&sd_detect_wake_lock);
	}
#endif
	dw_mmc_host[host->host_id] = NULL;
}
EXPORT_SYMBOL(dw_mci_remove);



#ifdef CONFIG_PM_SLEEP
/*
 * TODO: we should probably disable the clock to the card in the suspend path.
 */
int dw_mci_suspend(struct dw_mci *host)
{
    int i, ret = 0;

	xlog_mmc_log(host->host_id, "INFO dw_mci_suspend +++", 0, 0, 0);
	mmc_printk("INFO %s\n", __func__);
    for (i = 0; i < host->num_slots; i++) {
        struct dw_mci_slot *slot = host->slot[i];
        if (!slot)
            continue;
        ret = mmc_suspend_host(slot->mmc);
        if (ret < 0) {
            while (--i >= 0) {
                slot = host->slot[i];
                if (slot)
                    mmc_resume_host(host->slot[i]->mmc);
                }
            return ret;
            }
        }

	if (host->vmmc)
		regulator_disable(host->vmmc);
	xlog_mmc_log(host->host_id, "INFO dw_mci_suspend ---", 0, 0, 0);

	return 0;
}
EXPORT_SYMBOL(dw_mci_suspend);

int dw_mci_resume(struct dw_mci *host)
{
	int i, ret;

	xlog_mmc_log(host->host_id, "INFO dw_mci_resume +++", 0, 0, 0);
	mmc_printk("INFO %s\n", __func__);
	if (host->vmmc) {
		ret = regulator_enable(host->vmmc);
		if (ret) {
			dev_err(host->dev,
				"failed to enable regulator: %d\n", ret);
			return ret;
		}
	}

	if (!dw_mci_ctrl_all_reset(host)) {
		ret = -ENODEV;
		return ret;
	}

	if (host->use_dma && host->dma_ops->init)
		host->dma_ops->init(host);

	/*
	 * Restore the initial value at FIFOTH register
	 * And Invalidate the prev_blksz with zero
	 */
	mci_writel(host, FIFOTH, host->fifoth_val);
	host->prev_blksz = 0;

	/* Put in max timeout */
	mci_writel(host, TMOUT, 0xFFFFFFFF);

	mci_writel(host, RINTSTS, 0xFFFFFFFF);
	mci_writel(host, INTMASK, SDMMC_INT_CMD_DONE | SDMMC_INT_DATA_OVER |
		   SDMMC_INT_TXDR | SDMMC_INT_RXDR |
		   //DW_MCI_ERROR_FLAGS | SDMMC_INT_CD);
		   DW_MCI_ERROR_FLAGS );
	mci_writel(host, CTRL, SDMMC_CTRL_INT_ENABLE);

	for (i = 0; i < host->num_slots; i++) {
		struct dw_mci_slot *slot = host->slot[i];
		if (!slot)
			continue;
		if (slot->mmc->pm_flags & MMC_PM_KEEP_POWER) {
			dw_mci_set_ios(slot->mmc, &slot->mmc->ios);
			dw_mci_setup_bus(slot, true);
		}

		ret = mmc_resume_host(host->slot[i]->mmc);
	}
	xlog_mmc_log(host->host_id, "INFO dw_mci_resume ---", 0, 0, 0);
	return 0;
}
EXPORT_SYMBOL(dw_mci_resume);
#endif /* CONFIG_PM_SLEEP */

static int __init dw_mci_init(void)
{
	mmc_printk("ZX29 Multimedia Card Interface Driver");
	return 0;
}

static void __exit dw_mci_exit(void)
{
}

module_init(dw_mci_init);
module_exit(dw_mci_exit);

MODULE_DESCRIPTION("DW Multimedia Card Interface driver");
MODULE_AUTHOR("NXP Semiconductor VietNam");
MODULE_AUTHOR("Imagination Technologies Ltd");
MODULE_LICENSE("GPL v2");
