/*
 * Copyright (C) 2016-2017 ZXIC
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; see the file COPYING. If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Get MTD device info.
 *
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/err.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/partitions.h>
#include <linux/mtd/nand.h>
#include <linux/slab.h>
#include <linux/sched.h>

#define PRINT_PREF KERN_INFO "mtd_bbttest: "

static int dev = -EINVAL;
module_param(dev, int, S_IRUGO);
MODULE_PARM_DESC(dev, "MTD device number to use");

static struct mtd_info *mtd;
static unsigned char *iobuf;
static unsigned char *iobuf1;
static unsigned char *bbt;
static int ebcnt;
static int goodebcnt;

static int is_block_bad(int ebnum)
{
	loff_t addr = ebnum * mtd->erasesize;
	int ret;

	ret = mtd_block_isbad(mtd, addr);
	if (ret)
		printk(PRINT_PREF "block %d is bad\n", ebnum);
	return ret;
}

static int mark_block_bad(int ebnum)
{
	loff_t addr = ebnum * mtd->erasesize;
	int ret;

	ret = mtd_block_markbad(mtd, addr);
	if (ret)
		printk(PRINT_PREF "block %d mark bad failed!\n", ebnum);
	return ret;
}

static int scan_for_bad_eraseblocks(void)
{
	int i, bad = 0;

	bbt = kzalloc(ebcnt, GFP_KERNEL);
	if (!bbt) {
		printk(PRINT_PREF "error: cannot allocate memory\n");
		return -ENOMEM;
	}

	if (!mtd_can_have_bb(mtd))
		goto out;

	printk(PRINT_PREF "scanning for bad eraseblocks\n");
	for (i = 0; i < ebcnt; ++i) {
		bbt[i] = is_block_bad(i) ? 1 : 0;
		if (bbt[i])
			bad += 1;
		cond_resched();
	}
	printk(PRINT_PREF "scanned %d eraseblocks, %d are bad\n", i, bad);
out:
	goodebcnt = ebcnt - bad;
	return 0;
}

/***
 ***	1. read
 ***	2. change and write
 ***	3. read and compare
 ***
 */
static int __init mtd_bbttest_init(void)
{
	int err = 0;
	int i;
	int block_cnt, block;
	int ecc_cap;
	uint64_t tmp;	
	int bad;
	int marked_block = 0;
	
	pr_info("\n");
	pr_info("=================================================\n");

	printk(PRINT_PREF "MTD device: %d\n", dev); 

	mtd = get_mtd_device(NULL, dev);
	if (IS_ERR(mtd)) {
		err = PTR_ERR(mtd);
		printk(PRINT_PREF "error: Cannot get MTD device\n");
		return err;
	}

	if (mtd->writesize == 1) {
		printk(PRINT_PREF "not NAND flash, assume page size is 512 "
			   "bytes.\n");
	} 

	tmp = mtd->size;
	do_div(tmp, mtd->erasesize);
	ebcnt = tmp;

	err = -ENOMEM;
	iobuf = kmalloc(mtd->erasesize, GFP_KERNEL);
	if (!iobuf) {
		printk(PRINT_PREF "error: cannot allocate memory\n");
		goto out;
	}
	iobuf1 = kmalloc(mtd->erasesize, GFP_KERNEL);
	if (!iobuf1) {
		printk(PRINT_PREF "error: cannot allocate memory\n");
		goto out;
	}

	err = scan_for_bad_eraseblocks();
	if (err)
		goto out;

	/* dump bad block */
	bad = 0;	
	for (i = 0; i < ebcnt; i++) 
	{
		if(is_block_bad(i))
		{
			bad += 1;
			printk("block(%d)	 ", i);
			if((bad%4)==0)
				pr_info("\n");
		}
	}	

	/* mark a good block bad */
	pr_info("\n");	
	for (i = 0; i < ebcnt; i++) 
	{
		if(bbt[i] == 0)
		{
			printk(PRINT_PREF "mark block(%d) bad !\n", i);
			if(mark_block_bad(i))
			{
				printk(PRINT_PREF "mark block(%d) bad failed! \n", i);
				BUG();
			}
			marked_block = i;
			break;			
		}
	}	

	/* dump bad block */
	bad = 0;
	pr_info("\n");	
	printk(PRINT_PREF "scan bad block again...\n", i);
	for (i = 0; i < ebcnt; i++) 
	{
		if(is_block_bad(i))
		{
			bad += 1;
			printk("block(%d)	 ", i);
			if((bad%4)==0)
				pr_info("\n");
		}
	}
	printk(PRINT_PREF "Find total %d bad blocks.\n", bad);
		
	if (err)
		printk(PRINT_PREF "finished with errors\n");
	else
		printk(PRINT_PREF "finished\n");

out:
	kfree(iobuf);
	kfree(iobuf1);
	kfree(bbt);
	put_mtd_device(mtd);
	if (err)
		printk(PRINT_PREF "error %d occurred\n", err);
	printk(KERN_INFO "=================================================\n");
	return err;

}


module_init(mtd_bbttest_init);

static void __exit mtd_bbttest_exit(void)
{
	return;
}
module_exit(mtd_bbttest_exit);

MODULE_DESCRIPTION("Nand bbt test module");
MODULE_AUTHOR("ZXIC");
MODULE_LICENSE("GPL");
