/*
 * Copyright (C) 2016-2017 ZXIC
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; see the file COPYING. If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Get MTD device info.
 *
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/err.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/partitions.h>
#include <linux/mtd/nand.h>
#include <linux/slab.h>
#include <linux/sched.h>

#define PRINT_PREF KERN_INFO "mtd_ecctest: "

static int dev = -EINVAL;
module_param(dev, int, S_IRUGO);
MODULE_PARM_DESC(dev, "MTD device number to use");

static struct mtd_info *mtd;
static unsigned char *iobuf;
static unsigned char *iobuf1;
static unsigned char *bbt;
static int ebcnt;

extern int nand_get_ecc_cap(struct mtd_info *mtd);
static unsigned long next = 1;

static inline unsigned int simple_rand(void)
{
	next = next * 1103515245 + 12345;
	return (unsigned int)((next / 65536) % 32768);
}

static inline void simple_srand(unsigned long seed)
{
	next = seed;
}

static void set_random_data(unsigned char *buf, size_t len)
{
	size_t i;

	for (i = 0; i < len; ++i)
		buf[i] = simple_rand();
}

static int is_block_bad(int ebnum)
{
	loff_t addr = ebnum * mtd->erasesize;
	int ret;

	ret = mtd_block_isbad(mtd, addr);
	if (ret)
		printk(PRINT_PREF "block %d is bad\n", ebnum);
	return ret;
}

static int scan_for_bad_eraseblocks(void)
{
	int i, bad = 0;

	bbt = kzalloc(ebcnt, GFP_KERNEL);
	if (!bbt) {
		printk(PRINT_PREF "error: cannot allocate memory\n");
		return -ENOMEM;
	}

	if (!mtd_can_have_bb(mtd))
		return 0;

	printk(PRINT_PREF "scanning for bad eraseblocks\n");
	for (i = 0; i < ebcnt; ++i) {
		bbt[i] = is_block_bad(i) ? 1 : 0;
		if (bbt[i])
			bad += 1;
		cond_resched();
	}
	printk(PRINT_PREF "scanned %d eraseblocks, %d are bad\n", i, bad);
	return 0;
}
int nand_raw = 0;
/* write the first page of the block */
static int write_page(loff_t addr, unsigned char *buf, int enable_ecc)
{
	int err = 0;
	struct mtd_oob_ops ops;

	if(enable_ecc)
		ops.mode	  = MTD_OPS_PLACE_OOB;
	else
		ops.mode	  = MTD_OPS_RAW;
	ops.len 	  = mtd->writesize;
	ops.retlen	  = 0;
	ops.ooblen	  = mtd->oobsize;
	ops.oobretlen = 0;
	ops.ooboffs   = 0;
	ops.datbuf	  = buf;
	ops.oobbuf	  = buf + mtd->writesize;
//	err = mtd_write_oob(mtd, addr, &ops);
	err = mtd->_write_oob(mtd, (loff_t)addr, &ops);	

	return err;
}

/* read the first page of the block */
static int read_page(loff_t addr, unsigned char *buf, int enable_ecc)
{
	int err = 0;
	struct mtd_oob_ops ops;

	if(enable_ecc)
		ops.mode	  = MTD_OPS_PLACE_OOB;
	else
		ops.mode	  = MTD_OPS_RAW;
	ops.len 	  = mtd->writesize;
	ops.retlen	  = 0;
	ops.ooblen	  = mtd->oobsize;
	ops.oobretlen = 0;
	ops.ooboffs   = 0;
	ops.datbuf	  = buf;
	ops.oobbuf	  = buf + mtd->writesize;
	err = mtd_read_oob(mtd, addr, &ops);

	return err;
}



static int erase_eraseblock(int ebnum)
{
	int err;
	struct erase_info ei;
	loff_t addr = ebnum * mtd->erasesize;

	memset(&ei, 0, sizeof(struct erase_info));
	ei.mtd  = mtd;
	ei.addr = addr;
	ei.len  = mtd->erasesize;

	err = mtd_erase(mtd, &ei);
	if (err) {
		printk(PRINT_PREF "error %d while erasing EB %d\n", err, ebnum);
		return err;
	}

	if (ei.state == MTD_ERASE_FAILED) {
		printk(PRINT_PREF "some erase error occurred at EB %d\n",
		       ebnum);
		return -EIO;
	}

	return 0;
}



static int prepare_ecc_block(struct mtd_info *mtd)
{
	int i;
	int err;

	for (i = 0; i < ebcnt; ++i) {
		if (bbt[i])
			continue;
		err = erase_eraseblock(i);
		if (err)
			return err;

//		set_random_data(iobuf, mtd->writesize);
		memset(iobuf, 0x5A, mtd->writesize);
		memset(iobuf+mtd->writesize, 0xff, mtd->oobsize);
		cond_resched();

		err = write_page(i*mtd->erasesize, iobuf, true);
		if (err)
			return err;

		return i;
	}

	return -1;
}

static int make_ecc_err(unsigned char *buf, unsigned int buf_len, int ecc_cnt)
{
	int i, j;
	int count = 0;
	int ret = -1;
	unsigned int *data = (unsigned int *)buf;

	for(i=0; i<buf_len/4; i++)
	{
		if(count == ecc_cnt)
			return 0;	
		
		if(*(data+i))
		{
			*(data+i) &= ~(1<<(ffs(*(data+i))-1));
			count ++;
		}
		else
			continue;
	}

	return ret;
}


/***
 ***	1. dump ecc info:  ecc capacity(1/4/8... bit)  handler(hw/sw)
 ***	2. make ecc and check
 ***
 */
static int __init mtd_ecctest_init(void)
{
	int err = 0;
	int i;
	int block_cnt, block;
	int ecc_cap;
	uint64_t tmp;	
	
	pr_info("\n");
	pr_info("=================================================\n");

	printk(PRINT_PREF "MTD device: %d\n", dev); 

	mtd = get_mtd_device(NULL, dev);
	if (IS_ERR(mtd)) {
		err = PTR_ERR(mtd);
		printk(PRINT_PREF "error: Cannot get MTD device\n");
		return err;
	}

	if (mtd->writesize == 1) {
		printk(PRINT_PREF "not NAND flash, assume page size is 512 "
			   "bytes.\n");
	} 

	tmp = mtd->size;
	do_div(tmp, mtd->erasesize);
	ebcnt = tmp;

	err = -ENOMEM;
	iobuf = kmalloc(mtd->erasesize, GFP_KERNEL);
	if (!iobuf) {
		printk(PRINT_PREF "error: cannot allocate memory\n");
		goto out;
	}
	iobuf1 = kmalloc(mtd->erasesize, GFP_KERNEL);
	if (!iobuf1) {
		printk(PRINT_PREF "error: cannot allocate memory\n");
		goto out;
	}

	err = scan_for_bad_eraseblocks();
	if (err)
		goto out;


	/* dump ecc info */
	ecc_cap = nand_get_ecc_cap(mtd);
	pr_info("ecc_info: capacity(bit)   handler\n");
	pr_info("			   %d           (hw)\n", ecc_cap);	

	/* make 0~~ecc_cap+2 ecc errors and check */
	/** 1. erase 1 block 
	 ** 2. write random to the first page with ecc enabled
	 ** 3. read the first page to iobuf1 with ecc disabled
	 ** 4. make first bit 1-->0, then write the data to the page with ecc disabled
	 ** 5. read the first page with ecc enabled and compare to iobuf, judge the return value
	 **/

	block = prepare_ecc_block(mtd);
	if(block < 0)
	{
		err = block;
		pr_info("prepare_ecc_block error:(%d)\n", err);
		goto out;
	}

	err = read_page(block*mtd->erasesize, iobuf1, false);
	if(err)
	{
		pr_info("read page without ecc error:(%d)\n", err); 	
		goto out;
	}
	memcpy(iobuf1+2*mtd->writesize, iobuf1, mtd->writesize+mtd->oobsize);
	for(i=0; i<ecc_cap+2; i++)
	{
		memcpy(iobuf1, iobuf1+2*mtd->writesize, mtd->writesize+mtd->oobsize);

		/* make i-bit ecc errors */ 
		err = make_ecc_err(iobuf1, mtd->writesize, i);
		if(err)
			goto out;
		pr_info(" Make (%d/%d)bit ecc errors \n", i, ecc_cap);
/*
		err = erase_eraseblock(block);
		if (err)
			return err;
*/			
		err = write_page(block*mtd->erasesize, iobuf1, false);
		if (err)
			goto out;

		/* */
		err = read_page(block*mtd->erasesize, iobuf1, true);
		pr_info(" Read ecc (%d)errors with except!\n", err);
		if(!err)
		{
			if(0 != memcmp(iobuf, iobuf1, mtd->writesize+mtd->oobsize))
			{
				pr_info(" Not correct ecc successfully!\n");
				goto out;
			}
		}
	}
		
	if (err)
		printk(PRINT_PREF "finished with errors\n");
	else
		printk(PRINT_PREF "finished\n");

out:
	kfree(iobuf);
	kfree(iobuf1);
	kfree(bbt);
	put_mtd_device(mtd);
	if (err)
		printk(PRINT_PREF "error %d occurred\n", err);
	printk(KERN_INFO "=================================================\n");
	return err;

}

module_init(mtd_ecctest_init);

static void __exit mtd_ecctest_exit(void)
{
	return;
}
module_exit(mtd_ecctest_exit);

MODULE_DESCRIPTION("Nand ecc test module");
MODULE_AUTHOR("ZXIC");
MODULE_LICENSE("GPL");
