/*
 * Copyright (C) 2016-2017 ZXIC
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; see the file COPYING. If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Get MTD device info.
 *
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/err.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/partitions.h>
#include <linux/mtd/nand.h>
#include <linux/slab.h>
#include <linux/sched.h>

#define PRINT_PREF KERN_INFO "mtd_getinfotest: "

#define IO_BUFF_LEN		0x200

static struct mtd_info *mtd;
static unsigned char *iobuf;

static int dev = 16;
module_param(dev, int, S_IRUGO);
MODULE_PARM_DESC(dev, "MTD device number to use");

static int is_block_bad(int ebnum)
{
	loff_t addr = ebnum * mtd->erasesize;

	return mtd->_block_isbad(mtd, addr);		
}

struct cmdline_mtd_partition {
	struct cmdline_mtd_partition *next;
	char *mtd_id;
	int num_parts;
	struct mtd_partition *parts;
};
struct cmdline_mtd_partition *partitions;

extern int nand_get_info(struct mtd_info *mtd, unsigned char *fac_id, unsigned char *dev_id,
					char *name, struct cmdline_mtd_partition **parts);

/***
 ***	dump nand info: fac_id  dev_id  modal  chip_size(Gb)  block_num  page_size(main+oob)
 ***	dump part info: part_name part_index start_offset part_size
 ***	dump bbt info:  bbt_block_no(main/mirror)  bad_block_no ...
 ***
 */
static int __init mtd_getinfotest_init(void)
{
	int err = 0;
	unsigned char fac_id, dev_id;
	int i, j;
	int block_cnt, bad, total_blks;
	struct nand_chip *	chip;
	unsigned long chip_size;	
	uint64_t tmp;
	
	pr_info("\n");
	pr_info("=================================================\n");

	iobuf = kmalloc(IO_BUFF_LEN, GFP_KERNEL);
	if (!iobuf) {
		pr_info("error: cannot allocate memory\n");
		goto out;
	}

	printk(PRINT_PREF "MTD device: %d\n", dev); 

	mtd = get_mtd_device(NULL, dev);
	if (IS_ERR(mtd)) {
		err = PTR_ERR(mtd);
		pr_info("error: Cannot get MTD device\n");
		return err;
	}

	chip = mtd->priv;

	/* prepare nand info */
	nand_get_info(mtd, &fac_id, &dev_id, iobuf, &partitions);
	chip_size = (1<<chip->chip_shift);
	block_cnt = chip_size/mtd->erasesize; 
	

	pr_info("nand_info: fac_id	dev_id	modal		chip_size(MB)  block_num  page_size\n");
	pr_info("			0x%x	0x%x	%s	 %d 	%d	  (%d+%d)\n", 
				fac_id, dev_id, iobuf, (chip_size>>20), block_cnt, mtd->writesize, mtd->oobsize);	


	/* prepare part info */
	pr_info("part info: part_name	part_index start_offset part_size(MB)\n");
	for(i=0; i<partitions->num_parts;i++)
	{
		pr_info("  %18s 		%d		   0x%8x	   0x%8x\n", 
					partitions->parts[i].name, i, (unsigned long)partitions->parts[i].offset, (unsigned long)partitions->parts[i].size);
	}

	put_mtd_device(mtd);

	/* prepare bbt info */
	bad = 0;
	total_blks = 0; 
	pr_info("\n\nbad block info:\n");
#if 1	
	for (i = 0; i < partitions->num_parts; i++) 
	{
		mtd = get_mtd_device(NULL, i);
		if (IS_ERR(mtd)) {
			err = PTR_ERR(mtd);
			pr_info("error: Cannot get MTD device\n");
			return err;
		}		

		tmp = mtd->size;
		do_div(tmp, mtd->erasesize);
		block_cnt = tmp;

		total_blks += block_cnt;

		for (j = 0; j < block_cnt; j++) 
		{
			if(is_block_bad(j))
			{
				bad += 1;
				printk("block(%d)	 ", j);
				if((bad%4)==0)
					pr_info("\n");
			}
		}
		
		put_mtd_device(mtd);
	}
#else
	for (i = 0; i < block_cnt; i++) 
	{
		if(is_block_bad(i))
		{
			bad += 1;
			printk("block(%d)	 ", i);
			if((bad%4)==0)
				pr_info("\n");
		}
	}	
#endif	
	pr_info("\nTotal %d bad blocks found in %d blocks.\n", bad, total_blks);


out:
	kfree(iobuf);
	if (err)
		printk(PRINT_PREF "error %d occurred\n", err);
	printk(KERN_INFO "=================================================\n");
	return err;
}


module_init(mtd_getinfotest_init);

static void __exit mtd_getinfotest_exit(void)
{
	return;
}
module_exit(mtd_getinfotest_exit);

MODULE_DESCRIPTION("Getinfo test module");
MODULE_AUTHOR("ZXIC");
MODULE_LICENSE("GPL");
