/****************************************************************************/
/*
 *    zx297510_uart.c   sanchips  zx297510
 *
 *	(C) Copyright 2003-2007, gaowei
 *	(C) Copyright 2003-2007, sanchips
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

/****************************************************************************/
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/timer.h>
#include <linux/interrupt.h>
#include <linux/module.h>
#include <linux/console.h>
#include <linux/tty.h>
#include <linux/tty_flip.h>
#include <linux/serial.h>
#include <linux/serial_core.h>
#include <linux/platform_device.h>
#include <linux/of.h>
#include <linux/io.h>
#include <linux/printk.h>
#include <linux/clk.h>
#include <linux/gpio.h>

#include <mach/irqs.h>
#include <mach/clock.h>
#include <mach/gpio.h>
#include <mach/debug.h>
#include <mach/board.h>

#include "zx297510_uart.h"

//#define DEBUG_UART

#ifdef DEBUG_UART
#pragma GCC optimize("O0")
#endif


#define UART_NUM 1

#define UART0 0
#define UART1 1
#define UART2 2

/* 
 *Use device name ttyS, major 4, minor 64-68.  This is the usual serial port
 * name, but it is legally reserved for the 8250 driver. 
 */
#define SERIAL_zx297510_MAJOR	TTY_MAJOR
#define SERIAL_MINOR_START		64


/* UART registers.  hence no GET macro */
#define UART_PUT_IFLS(port, v)  __raw_writel(v, (port)->membase + zx297510_UART_IFLS)

#define UART_PUT_CHAR(port, v)	__raw_writel(v, (port)->membase + zx297510_UART_DR)
#define UART_GET_CHAR(port)	    __raw_readl((port)->membase + zx297510_UART_DR)

#define UART_GET_FR(port)	    __raw_readl((port)->membase + zx297510_UART_FR)
    
#define UART_PUT_IBRD(port, v)	__raw_writel(v, (port)->membase + zx297510_UART_IBRD)
#define UART_GET_IBRD(port)		__raw_readl((port)->membase + zx297510_UART_IBRD)

#define UART_PUT_FBRD(port, v)	__raw_writel(v, (port)->membase + zx297510_UART_FBRD)
#define UART_GET_FBRD(port)		__raw_readl((port)->membase + zx297510_UART_FBRD)

#define UART_PUT_LCRH(port, v)	__raw_writel(v, (port)->membase + zx297510_UART_LCRH)
#define UART_GET_LCRH(port)	    __raw_readl((port)->membase + zx297510_UART_LCRH)
  
#define UART_PUT_CR(port, v)	__raw_writel(v, (port)->membase + zx297510_UART_CR)
#define UART_GET_CR(port)	    __raw_readl((port)->membase + zx297510_UART_CR)

#define UART_PUT_ICR(port, v)	__raw_writel(v, (port)->membase + zx297510_UART_ICR)

#define UART_PUT_IMSC(port, v)  __raw_writel(v, (port)->membase + zx297510_UART_IMSC)

#define UART_PUT_MIS(port, v)   __raw_writel(v, (port)->membase + zx297510_UART_MIS)
#define UART_GET_MIS(port)      __raw_readl((port)->membase + zx297510_UART_MIS)



/*
 *	Local per-uart structure.
 */
struct zx297510_uart_port {
	struct uart_port	port;
	unsigned int		sigs;		/* Local copy of line sigs */
	unsigned int		old_status;
	unsigned char		imr;		/* Local interrupt mask reg mirror */
	bool				rts_state;
	bool				autorts;    /* hardware flow control */
	struct clk          *wclk;      /* uart work clock */
	struct clk          *busclk;      /* uart apb clock */
};


static unsigned int zx297510_uart_tx_empty(struct uart_port *port)
{
	return (UART_GET_FR(port)&(UART_FR_TXBUSY|UART_FR_TXFF)) ? 0 : TIOCSER_TEMT;
}

static void zx297510_uart_set_mctrl(struct uart_port *port, unsigned int mctrl)
{
	unsigned int control = 0;
	struct zx297510_uart_port *zup = container_of(port, struct zx297510_uart_port, port);

	zup->sigs = mctrl;
	control = UART_GET_CR(&zup->port);
	if(mctrl & TIOCM_DTR)
		control |= UART_CR_DTR;
	else
		control &= ~ UART_CR_DTR;
	
	if(mctrl & TIOCM_RTS)
		control |= UART_CR_RTS;
	else
		control &= ~UART_CR_RTS;
	
	if(mctrl & TIOCM_LOOP)
		control |= UART_CR_LBE;
	else
		control &= ~UART_CR_LBE;
	
	/* We need to disable auto-RTS if we want to turn RTS off */	
	if (zup->autorts) {
		if (mctrl & TIOCM_RTS)		
			control |= UART_CR_RTSEN;		
		else				
			control &= ~UART_CR_RTSEN;
	}
	UART_PUT_CR(port, control);
}

static unsigned int zx297510_uart_get_mctrl(struct uart_port *port)
{
	struct zx297510_uart_port *zup = container_of(port, struct zx297510_uart_port, port);
	unsigned int mctrl = 0;
	unsigned int uart_flag = 0;
	
	uart_flag = UART_GET_FR(port);

	mctrl = (uart_flag&UART_FR_CTS) ?TIOCM_CTS : 0;
	mctrl |= (zup->sigs & TIOCM_RTS);
	mctrl |= (uart_flag&UART_FR_DCD) ? TIOCM_CD : 0;
	mctrl |= (uart_flag&UART_FR_DSR) ? TIOCM_DSR : 0;
	mctrl |= (uart_flag&UART_FR_RI) ? TIOCM_RI : 0;

	return mctrl;
}

static void zx297510_uart_start_tx(struct uart_port *port)
{
	struct zx297510_uart_port *zup = container_of(port, struct zx297510_uart_port, port);
	zup->imr |= UART_TXIM;
	UART_PUT_IMSC(port, zup->imr);
}

static void zx297510_uart_stop_tx(struct uart_port *port)
{
	struct zx297510_uart_port *zup = container_of(port, struct zx297510_uart_port, port);
	zup->imr &= ~UART_TXIM;
	UART_PUT_IMSC(port, zup->imr);
}


static void zx297510_uart_stop_rx(struct uart_port *port)
{
	struct zx297510_uart_port *zup = container_of(port, struct zx297510_uart_port, port);

	zup->imr &= ~(UART_RXIM|UART_RTIM|UART_FEIM|UART_PEIM|UART_BEIM|UART_OEIM);
	UART_PUT_IMSC(port, zup->imr);
}

static void zx297510_uart_break_ctl(struct uart_port *port, int break_state)
{
	struct zx297510_uart_port *zup = container_of(port, struct zx297510_uart_port, port);
	unsigned long flags;
	unsigned int lcr_h;
	spin_lock_irqsave(&zup->port.lock, flags);
	lcr_h = UART_GET_LCRH(port);
	if (break_state == -1)
		lcr_h |= UART_LCRH_BRK;
	else
		lcr_h &= ~UART_LCRH_BRK;
	UART_PUT_LCRH(port, lcr_h);
	spin_unlock_irqrestore(&zup->port.lock, flags);
}

static void zx297510_uart_enable_ms(struct uart_port *port)
{
	struct zx297510_uart_port *zup = container_of(port, struct zx297510_uart_port, port);
	zup->imr |= UART_RIMIM|UART_CTSMIM|UART_DCDMIM|UART_DSRMIM;
	UART_PUT_IMSC(port, zup->imr);
}


/*
 * Reads up to 256 characters from the FIFO or until it's empty and
 * inserts them into the TTY layer. Returns the number of characters
 * read from the FIFO.
 */
static int zx297510_uart_fifo_to_tty(struct zx297510_uart_port *zup)
{
	struct uart_port *port = &zup->port;
	u32 status, ch;
	unsigned int flag, max_count = 256;
	int fifotaken = 0;

	while (max_count--) {
		status = UART_GET_FR(port);
		if (status & UART_FR_RXFE)
			break;

		/* Take chars from the FIFO and update status */
		ch = UART_GET_CHAR(port) |	UART_DUMMY_DR_RX;
		flag = TTY_NORMAL;
		zup->port.icount.rx++;
		fifotaken++;

		if (unlikely(ch & UART_DR_ERROR)) {
			if (ch & UART_DR_BE) {
				ch &= ~(UART_DR_FE | UART_DR_PE);
				zup->port.icount.brk++;
				if (uart_handle_break(&zup->port))
					continue;
			} else if (ch & UART_DR_PE)
				zup->port.icount.parity++;
			else if (ch & UART_DR_FE)
				zup->port.icount.frame++;
			if (ch & UART_DR_OE)
				zup->port.icount.overrun++;

			ch &= zup->port.read_status_mask;

			if (ch & UART_DR_BE)
				flag = TTY_BREAK;
			else if (ch & UART_DR_PE)
				flag = TTY_PARITY;
			else if (ch & UART_DR_FE)
				flag = TTY_FRAME;
		}

		if (uart_handle_sysrq_char(&zup->port, ch & 255))
			continue;

		uart_insert_char(&zup->port, ch, UART_DR_OE, ch, flag);
	}

	return fifotaken;
}


static void zx297510_uart_rx_chars(struct zx297510_uart_port *zup)
{
	struct tty_struct *tty = zup->port.state->port.tty;

	zx297510_uart_fifo_to_tty(zup);

	spin_unlock(&zup->port.lock);
	tty_flip_buffer_push(tty);
	spin_lock(&zup->port.lock);

}


static void zx297510_uart_tx_chars(struct zx297510_uart_port *zup)
{
	struct circ_buf *xmit = &zup->port.state->xmit;
	int count;

	if (zup->port.x_char) {
		UART_PUT_CHAR(&zup->port, zup->port.x_char);
		zup->port.icount.tx++;
		zup->port.x_char = 0;
		return;
	}
	if (uart_circ_empty(xmit) || uart_tx_stopped(&zup->port)) {
		zx297510_uart_stop_tx(&zup->port);
		return;
	}
	count = zup->port.fifosize >> 1;
	do {
		UART_PUT_CHAR(&zup->port, xmit->buf[xmit->tail]);
		xmit->tail = (xmit->tail + 1) & (UART_XMIT_SIZE - 1);
		zup->port.icount.tx++;
		if (uart_circ_empty(xmit))
			break;
	} while (--count > 0);

	if (uart_circ_chars_pending(xmit) < WAKEUP_CHARS)
		uart_write_wakeup(&zup->port);

	if (uart_circ_empty(xmit))
		zx297510_uart_stop_tx(&zup->port);
}


static void zx297510_uart_modem_status(struct zx297510_uart_port *zup)
{
	unsigned int status, delta;

	status = UART_GET_FR(&zup->port)& UART_FR_MODEM_ANY;

	delta = status ^ zup->old_status;
	zup->old_status = status;

	if (!delta)
		return; 

	if (delta & UART_FR_DCD)
		uart_handle_dcd_change(&zup->port, status & UART_FR_DCD);

	if (delta & UART_FR_DSR)
		zup->port.icount.dsr++;

	if (delta & UART_FR_CTS)
		uart_handle_cts_change(&zup->port, status & UART_FR_CTS);

	wake_up_interruptible(&zup->port.state->port.delta_msr_wait);
}


static irqreturn_t zx297510_uart_interrupt(int irq, void *dev_id)
{
	struct uart_port *port = dev_id;
	struct zx297510_uart_port *zup = container_of(port, struct zx297510_uart_port, port);
	unsigned long flags;
	unsigned int status, pass_counter = 256;
	int handled = 0;

	spin_lock_irqsave(&zup->port.lock, flags);
	status = UART_GET_MIS(port) & zup->imr;
	if (status) {
		do {
			UART_PUT_ICR(port,(status & ~(UART_TXIS|UART_RTIS|UART_RXIS)));
			if (status & (UART_RTIS|UART_RXIS))
				zx297510_uart_rx_chars(zup);
			
			if (status & (UART_DSRMIS|UART_DCDMIS|UART_CTSMIS|UART_RIMIS))
				zx297510_uart_modem_status(zup);
			
			if (status & UART_TXIS)
				zx297510_uart_tx_chars(zup);

			if (pass_counter-- == 0)
				break;

			status = UART_GET_MIS(port);
		} while (status != 0);
		handled = IRQ_HANDLED;
	}
	spin_unlock_irqrestore(&zup->port.lock, flags);
	
	return IRQ_RETVAL(handled);
}


static int zx297510_uart_startup(struct uart_port *port)
{
	struct zx297510_uart_port *zup = container_of(port, struct zx297510_uart_port, port);
	unsigned long flags = 0;
	unsigned long control = 0;
	int retval = 0;
	struct platform_device *pdev=port->private_data;
	
    /*  config uart apb_clk   */
	clk_enable(zup->busclk);
	/* enable uart work clock */
	clk_enable(zup->wclk);
	
    /*configure gpio pin to UART*/
	retval=gpio_request(UART_PIN_NUM_BASE+(port->line)*2,pdev->dev.kobj.name);
	if(retval)
		BUG();
		
	retval=gpio_request(UART_PIN_NUM_BASE+(port->line)*2+1,pdev->dev.kobj.name);
	if(retval)
		BUG();
	
    zx29_gpio_config(UART_PIN_NUM_BASE+(pdev->id)*2,1);    /*uart rxd*/
	zx29_gpio_config(UART_PIN_NUM_BASE+(pdev->id)*2+1,1);  /*uart txd*/
	
	/* Clear all pending error and receive interrupts */
	UART_PUT_ICR(port, 0xfff);
	
	/*  Allocate the IRQ	 */
	retval = request_irq(port->irq, zx297510_uart_interrupt, 0, "uart-zx297510", zup);
	if (retval){
		pr_err("zx297510 uart: unable to attach zx297510 UART %d "
		       "interrupt vector=%d\n", port->line, port->irq);
		return retval;
	}
	
	/*	set interrupt fifo level RX:1/2 Full, TX:1/2 Full */
	UART_PUT_IFLS(port, UART_IFLS_RX4_8|UART_IFLS_TX4_8);
	
	/* Provoke TX FIFO interrupt into asserting. */
	control = UART_CR_UARTEN | UART_CR_TXE | UART_CR_LBE;
	UART_PUT_CR(port, control);
	UART_PUT_FBRD(port, 0);
	UART_PUT_IBRD(port, 1);	
	UART_PUT_LCRH(port, 0);
	UART_PUT_CHAR(port, 0);
	while (UART_GET_FR(port) & UART_FR_TXBUSY)
		barrier();

	control = UART_CR_UARTEN | UART_CR_RXE | UART_CR_TXE;
	UART_PUT_CR(port, control);

	/*
	 * Finally, enable interrupts, only timeouts when using DMA
	 * if initial RX DMA job failed, start in interrupt mode
	 * as well.
	 */
	spin_lock_irqsave(&zup->port.lock, flags);
	/* Clear out any spuriously appearing RX interrupts */
	UART_PUT_ICR(port, (UART_RTIS | UART_RXIS));
	
	zup->imr = UART_RTIM | UART_RXIM;
	UART_PUT_IMSC(port, zup->imr);
	spin_unlock_irqrestore(&zup->port.lock, flags);

	return 0;
}


static void zx297510_uart_shutdown(struct uart_port *port)
{
	struct zx297510_uart_port *zup = container_of(port, struct zx297510_uart_port, port);
	unsigned long flags;
	uint32_t val;
	
	spin_lock_irqsave(&port->lock, flags);
	/* Disable and clear all interrupts now */
	zup->imr = 0;
	UART_PUT_IMSC(port, zup->imr);
	UART_PUT_ICR(port, 0xFFFF);
	
	spin_unlock_irqrestore(&port->lock, flags);

	/* Free the interrupt */
	free_irq(zup->port.irq, zup);

	/* Disable UART transmitter and receiver */
	zup->autorts = false;
	val = UART_GET_CR(port);
	if (val & UART_CR_RTS) {
		zup->rts_state = true;
		val = UART_CR_RTS;
	} else
		zup->rts_state = false;
	val = UART_CR_UARTEN | UART_CR_TXE;
	UART_PUT_CR(port, val);

	/* disable break condition and fifos */
	val = UART_GET_LCRH(port);
	val &= ~(UART_LCRH_BRK | UART_LCRH_FEN);
	UART_PUT_LCRH(port, val);

	/* Shutdown uart clock */
}


static void zx297510_uart_set_termios(struct uart_port *port, struct ktermios *termios,
	struct ktermios *old)
{
	struct zx297510_uart_port *zup = container_of(port, struct zx297510_uart_port, port);
	unsigned int lcr_h, old_cr;
	unsigned long flags;
	unsigned int baud, ibrd, fbrd;
 
    /* Set baud rate */
	/* Ask the core to calculate the divisor for us. */
	baud = uart_get_baud_rate(port, termios, old, 0, port->uartclk/16);
	ibrd = port->uartclk  / (baud<<4);
	fbrd = ((port->uartclk % (baud<<4) )*8 + baud)/(2*baud);		
	UART_PUT_FBRD(port, fbrd);
	UART_PUT_IBRD(port, ibrd);
	

	switch (termios->c_cflag & CSIZE) {
	case CS5:
		lcr_h = UART_LCRH_WLEN_5;
		break;
	case CS6:
		lcr_h = UART_LCRH_WLEN_6;
		break;
	case CS7:
		lcr_h = UART_LCRH_WLEN_7;
		break;
	default: // CS8
		lcr_h = UART_LCRH_WLEN_8;
		break;
	}
	if (termios->c_cflag & CSTOPB)
		lcr_h |= UART_LCRH_STP2;
	if (termios->c_cflag & PARENB) {
		lcr_h |= UART_LCRH_PEN;
		if (!(termios->c_cflag & PARODD))
			lcr_h |= UART_LCRH_EPS;
	}
	if (port->fifosize > 1)
		lcr_h |= UART_LCRH_FEN;

	spin_lock_irqsave(&port->lock, flags);

	/*
	 * Update the per-port timeout.
	 */
	uart_update_timeout(port, termios->c_cflag, baud);

	port->read_status_mask = UART_DR_OE | 255;
	if (termios->c_iflag & INPCK)
		port->read_status_mask |= UART_DR_FE | UART_DR_PE;
	if (termios->c_iflag & (BRKINT | PARMRK))
		port->read_status_mask |= UART_DR_BE;

	/*
	 * Characters to ignore
	 */
	port->ignore_status_mask = 0;
	if (termios->c_iflag & IGNPAR)
		port->ignore_status_mask |= UART_DR_FE | UART_DR_PE;
	if (termios->c_iflag & IGNBRK) {
		port->ignore_status_mask |= UART_DR_BE;
		/*
		 * If we're ignoring parity and break indicators,
		 * ignore overruns too (for real raw support).
		 */
		if (termios->c_iflag & IGNPAR)
			port->ignore_status_mask |= UART_DR_OE;
	}

	/*
	 * Ignore all characters if CREAD is not set.
	 */
	if ((termios->c_cflag & CREAD) == 0)
		port->ignore_status_mask |= UART_DUMMY_DR_RX;

	if (UART_ENABLE_MS(port, termios->c_cflag))
		zx297510_uart_enable_ms(port);

	/* first, disable everything */
	old_cr = UART_GET_CR(port);
	UART_PUT_CR(port, 0);
	
	if (termios->c_cflag & CRTSCTS) {
		if (old_cr & UART_CR_RTS)
			old_cr |= UART_CR_RTSEN;

		old_cr |= UART_CR_CTSEN;
		zup->autorts = true;
	} else {
		old_cr &= ~(UART_CR_CTSEN | UART_CR_RTSEN);
		zup->autorts = false;
	}

	/*
	 * ----------v----------v----------v----------v-----
	 * NOTE: MUST BE WRITTEN AFTER UARTLCR_M & UARTLCR_L
	 * ----------^----------^----------^----------^-----
	 */
	UART_PUT_LCRH(port, lcr_h);
	UART_PUT_CR(port, old_cr);

	spin_unlock_irqrestore(&port->lock, flags);
}


static const char *zx297510_uart_type(struct uart_port *port)
{
	return (port->type == PORT_ZX297510) ? "zx297510_UART" : NULL;
}


static int zx297510_uart_request_port(struct uart_port *port)
{
	/* UARTs always present */
//	return request_mem_region(port->mapbase, SZ_4K, "uart-zx297510")!= NULL ? 0 : -EBUSY;
	return 0;
}


static void zx297510_uart_config_port(struct uart_port *port, int flags)
{
	if (flags & UART_CONFIG_TYPE) {
		port->type = PORT_ZX297510;
		zx297510_uart_request_port(port);
	}
}


static void zx297510_uart_release_port(struct uart_port *port)
{
//	release_mem_region(port->mapbase, SZ_4K);
}


static int zx297510_uart_verify_port(struct uart_port *port, struct serial_struct *ser)
{
	if ((ser->type != PORT_UNKNOWN) && (ser->type != PORT_ZX297510))
		return -EINVAL;
	return 0;
}

#ifdef CONFIG_CONSOLE_POLL
static int zx297510_get_poll_char(struct uart_port *port)
{
	if (UART_GET_FR(port) & UART_FR_RXFE)
		return NO_POLL_CHAR;

	return UART_PUT_CHAR(port);
}


static void zx297510_put_poll_char(struct uart_port *port, unsigned char ch)
{
	while (UART_GET_FR(port) & UART_FR_TXFF)
		barrier();
	UART_PUT_CHAR(port, ch);
}
#endif /* CONFIG_CONSOLE_POLL */


/*
 *	Define the basic serial functions we support.
 */
static const struct uart_ops zx297510_uart_ops = {
	.tx_empty	= zx297510_uart_tx_empty,
	.set_mctrl	= zx297510_uart_set_mctrl,
	.get_mctrl	= zx297510_uart_get_mctrl,
	.start_tx	= zx297510_uart_start_tx,
	.stop_tx	= zx297510_uart_stop_tx,
	.stop_rx	= zx297510_uart_stop_rx,
	.enable_ms	= zx297510_uart_enable_ms,
	.break_ctl	= zx297510_uart_break_ctl,
	.startup	= zx297510_uart_startup,
	.shutdown	= zx297510_uart_shutdown,
	.set_termios	= zx297510_uart_set_termios,
	.type		= zx297510_uart_type,
	.request_port	= zx297510_uart_request_port,
	.release_port	= zx297510_uart_release_port,
	.config_port	= zx297510_uart_config_port,
	.verify_port	= zx297510_uart_verify_port,
#ifdef CONFIG_CONSOLE_POLL
	.poll_get_char = zx297510_get_poll_char,
	.poll_put_char = zx297510_put_poll_char,
#endif

};


static struct zx297510_uart_port zx297510_uart_ports[UART_NUM];	

#define	zx297510_MAXPORTS	ARRAY_SIZE(zx297510_uart_ports)

static void __devinit zx297510_init_ports(struct zx297510_uart_port *zx297510_port,	
                                           struct platform_device *pdev)
{
	struct uart_port *port=&zx297510_port->port;
	unsigned int offset=(unsigned int)(pdev->id);
	struct clk *pclk=NULL;

    /*  config uart apb_clk   */
	pclk=clk_get(&pdev->dev, "apb_clk");
	if (IS_ERR(pclk))
		ZDRV_ASSERT(0);
	zx297510_port->busclk=pclk;   /*get apb clock*/ 
	

	/*  config uart work_clk   */
	pclk=clk_get(&pdev->dev, "work_clk");
	if (IS_ERR(pclk))
		ZDRV_ASSERT(0);
	zx297510_port->wclk=pclk;   /*get work clock*/ 

	port->line = offset;
	port->type = PORT_ZX297510;
	port->fifosize = UART_TXFIFO_SIZE;
	port->iotype = UPIO_MEM;	
	port->irq = pdev->resource[1].start;
	port->mapbase = pdev->resource[0].start;
	port->membase = ZX29_UART0_VA+0x10000*offset;
	port->flags = UPF_BOOT_AUTOCONF;
	port->ops = &zx297510_uart_ops;
	port->uartclk = clk_get_rate(pclk);
    port->private_data=pdev;

}


#ifdef CONFIG_SERIAL_ZX297510_UART_CONSOLE

static void zx297510_uart_console_putc(struct uart_port *port, const char c)
{
	while (UART_GET_FR(port) & UART_FR_TXFF)
		barrier();
	UART_PUT_CHAR(port, c);
}

static void zx297510_uart_console_write(struct console *co, const char *s, unsigned int count)
{
	struct uart_port *port = &zx297510_uart_ports[co->index].port;

	for (; (count); count--, s++) {
		zx297510_uart_console_putc(port, *s);
		if (*s == '\n')
			zx297510_uart_console_putc(port, '\r');
	}
}

/*
 * If the port was already initialized (eg, by a boot loader),
 * try to determine the current setup.
 */
static void __init zx297510_console_get_options(struct uart_port *port, int *baud,
						 int *parity, int *bits)
{
	if (UART_GET_CR(port) & UART_CR_UARTEN) {
		unsigned int lcr_h, ibrd, fbrd;

		lcr_h = UART_GET_LCRH(port);
		*parity = 'n';
		if (lcr_h & UART_LCRH_PEN) {
			if (lcr_h & UART_LCRH_EPS)
				*parity = 'e';
			else
				*parity = 'o';
		}
		if ((lcr_h & 0x60) == UART_LCRH_WLEN_7)
			*bits = 7;
		else
			*bits = 8;

		ibrd = UART_GET_IBRD(port);
		fbrd = UART_GET_FBRD(port);

		*baud = port->uartclk * 4 / (64 * ibrd + fbrd);
	}
}

static int __init zx297510_uart_console_setup(struct console *co, char *options)
{
	struct uart_port *port;
	int baud = 115200;
	int bits = 8;
	int parity = 'n';
	int flow = 'n';

	if ((co->index < 0) || (co->index >= zx297510_MAXPORTS))
		co->index = 0;
	port = &zx297510_uart_ports[co->index].port;
	if (port->membase == NULL)
		return -ENODEV;

	if (options)
		uart_parse_options(options, &baud, &parity, &bits, &flow);
	else
		zx297510_console_get_options(port, &baud, &parity, &bits);

	return uart_set_options(port, co, baud, parity, bits, flow);
}

static struct uart_driver zx297510_uart_driver;

static struct console zx297510_uart_console = {
	.name		= "ttyS",
	.write		= zx297510_uart_console_write,
	.device		= uart_console_device,
	.setup		= zx297510_uart_console_setup,
	.flags		= CON_PRINTBUFFER,
	.index		= -1,
	.data		= &zx297510_uart_driver,
};

#define	zx297510_UART_CONSOLE	(&zx297510_uart_console)
#else
#define	zx297510_UART_CONSOLE	NULL
#endif /* CONFIG_zx297510_UART_CONSOLE */

/*
 *	Define the zx297510 UART driver structure.
 */
static struct uart_driver zx297510_uart_driver = {
	.owner		    = THIS_MODULE,
	.driver_name	= "zx297510_uart",
	.dev_name	    = "ttyS",
	.major		    = SERIAL_zx297510_MAJOR,
	.minor		    = SERIAL_MINOR_START,
	.nr		        = zx297510_MAXPORTS,
	.cons		    = zx297510_UART_CONSOLE,
};


static int __devinit zx297510_uart_probe(struct platform_device *pdev)
{
	/*struct zx297510_platform_uart *pdata = pdev->dev.platform_data;*/
	struct zx297510_uart_port *port = &zx297510_uart_ports[pdev->id];
	int ret=0;

 	zx297510_init_ports(port,pdev);	
    ret=uart_add_one_port(&zx297510_uart_driver, &port->port);
	if(ret)
		return ret;
	 
	/*platform_set_drvdata(pdev, port);*/
	
    printk(KERN_INFO "TSP zx297510 UART_%d probe OK\n",pdev->id);
	return 0;
}


static int __devexit zx297510_uart_remove(struct platform_device *pdev)
{
	struct uart_port *port;
	int i;

	for (i = 0; (i < zx297510_MAXPORTS); i++) {
		port = &zx297510_uart_ports[i].port;
		if (port)
			uart_remove_one_port(&zx297510_uart_driver, port);
	}

	return 0;
}


static struct platform_driver zx297510_uart_platform_driver = {
	.probe		= zx297510_uart_probe,
	.remove		= zx297510_uart_remove,
	.driver		= {
		.name	= "zx297510_uart",
		.owner	= THIS_MODULE,
	},
};


static int __init zx297510_uart_init(void)
{
	int rc;

	rc = uart_register_driver(&zx297510_uart_driver);
	if (rc)
		return rc;
	rc = platform_driver_register(&zx297510_uart_platform_driver);
	if (rc){
		uart_unregister_driver(&zx297510_uart_driver);
		return rc;
	}
		
	printk(KERN_INFO "TSP zx297510 UART driver registered\n");
	
	return 0;
}


static void __exit zx297510_uart_exit(void)
{
#ifdef CONFIG_SERIAL_ZX297510_UART_CONSOLE
	unregister_console(&zx297510_uart_console);
#endif
	platform_driver_unregister(&zx297510_uart_platform_driver);
	uart_unregister_driver(&zx297510_uart_driver);
}

/****************************************************************************/

module_init(zx297510_uart_init);
module_exit(zx297510_uart_exit);

MODULE_AUTHOR("gaowei");
MODULE_DESCRIPTION("sanchips zx297510 UART driver");
MODULE_LICENSE("GPL");
MODULE_ALIAS("platform:zx297510_uart");

/****************************************************************************/
