/* ==========================================================================
 * $File: //dwh/usb_iip/dev/software/otg/linux/drivers/dwc_otg_driver.c $
 * $Revision: #92 $
 * $Date: 2012/08/10 $
 * $Change: 2047372 $
 *
 * Synopsys HS OTG Linux Software Driver and documentation (hereinafter,
 * "Software") is an Unsupported proprietary work of Synopsys, Inc. unless
 * otherwise expressly agreed to in writing between Synopsys and you.
 *
 * The Software IS NOT an item of Licensed Software or Licensed Product under
 * any End User Software License Agreement or Agreement for Licensed Product
 * with Synopsys or any supplement thereto. You are permitted to use and
 * redistribute this Software in source and binary forms, with or without
 * modification, provided that redistributions of source code must retain this
 * notice. You may not view, use, disclose, copy or distribute this file or
 * any information contained herein except pursuant to this license grant from
 * Synopsys. If you do not agree with this notice, including the disclaimer
 * below, then you are not authorized to use the Software.
 *
 * THIS SOFTWARE IS BEING DISTRIBUTED BY SYNOPSYS SOLELY ON AN "AS IS" BASIS
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE HEREBY DISCLAIMED. IN NO EVENT SHALL SYNOPSYS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 * ========================================================================== */

/** @file
 * The dwc_otg_driver module provides the initialization and cleanup entry
 * points for the DWC_otg driver. This module will be dynamically installed
 * after Linux is booted using the insmod command. When the module is
 * installed, the dwc_otg_driver_init function is called. When the module is
 * removed (using rmmod), the dwc_otg_driver_cleanup function is called.
 *
 * This module also defines a data structure for the dwc_otg_driver, which is
 * used in conjunction with the standard ARM lm_device structure. These
 * structures allow the OTG driver to comply with the standard Linux driver
 * model in which devices and drivers are registered with a bus driver. This
 * has the benefit that Linux can expose attributes of the driver and device
 * in its special sysfs file system. Users can then read or write files in
 * this file system to perform diagnostics on the driver components or the
 * device.
 */

#include "dwc_otg_os_dep.h"
#include "dwc_os.h"
#include "dwc_otg_dbg.h"
#include "dwc_otg_driver.h"
#include "dwc_otg_attr.h"
#include "dwc_otg_cil.h"
#include "dwc_otg_core_if.h"
#include "dwc_otg_pcd_if.h"
#include "dwc_otg_hcd_if.h"
#include "dwc_otg_pcd.h"
#include <mach/zx29_usb.h>
#include <linux/android_notify.h>
#include <mach/highspeed_debug.h>
#include <linux/wakelock.h>
#include <linux/clockchips.h>
#include <linux/clk.h>
#include <mach/gpio.h> 
#include <linux/gpio.h>
#include <mach/pcu.h>

#ifdef USB_CHG_DEV
#define DWC_DRIVER_VERSION	"3.00a 10-AUG-2012"
#define DWC_DRIVER_DESC		"HS OTG USB Controller driver"
static const char dwc_chg_name[] = "zx297520V3_hsotg";
#endif
#define DWC_WRITE_REG32(a,v)    ((*(volatile u32 *)(a)) = v)
#define DWC_READ_REG32(a)     	(*(volatile u32 *)(a))

#define USE_ASIC 1
#define DMA_ENABLE 0

extern void usb_notify_up(usb_notify_event notify_type, void* puf);
//extern void usb_detect_callback(void *buf, unsigned int len);
//extern void zx297510_usbdev_init(void);
extern void dwc_chg_udelay(uint32_t usecs);
extern void dwc_chg_mdelay(uint32_t usecs);
//extern struct wake_lock dwc_otg_wake_lock;

extern void dwc_otg_hal_init(void);
extern void dwc_otg_hal_exit(void);

/*-------------------------------------------------------------------------*/
/* Encapsulate the module parameter settings */

struct dwc_otg_driver_module_params {
	int32_t opt;
	int32_t otg_cap;
	int32_t dma_enable;
	int32_t dma_desc_enable;
	int32_t dma_burst_size;
	int32_t speed;
	int32_t host_support_fs_ls_low_power;
	int32_t host_ls_low_power_phy_clk;
	int32_t enable_dynamic_fifo;
	int32_t data_fifo_size;
	int32_t dev_rx_fifo_size;
	int32_t dev_nperio_tx_fifo_size;
	uint32_t dev_perio_tx_fifo_size[MAX_PERIO_FIFOS];
	int32_t host_rx_fifo_size;
	int32_t host_nperio_tx_fifo_size;
	int32_t host_perio_tx_fifo_size;
	int32_t max_transfer_size;
	int32_t max_packet_count;
	int32_t host_channels;
	int32_t dev_endpoints;
	int32_t phy_type;
	int32_t phy_utmi_width;
	int32_t phy_ulpi_ddr;
	int32_t phy_ulpi_ext_vbus;
	int32_t i2c_enable;
	int32_t ulpi_fs_ls;
	int32_t ts_dline;
	int32_t en_multiple_tx_fifo;
	uint32_t dev_tx_fifo_size[MAX_TX_FIFOS];
	uint32_t thr_ctl;
	uint32_t tx_thr_length;
	uint32_t rx_thr_length;
	int32_t pti_enable;
	int32_t mpi_enable;
	int32_t lpm_enable;
	int32_t ic_usb_cap;
	int32_t ahb_thr_ratio;
	int32_t power_down;
	int32_t reload_ctl;
	int32_t dev_out_nak;
	int32_t cont_on_bna;
	int32_t ahb_single;
	int32_t otg_ver;
	int32_t adp_enable;
};

struct g_data
{
	dwc_otg_device_t 		g_dwc_otg_dev_t;
	u32						dwRxQuit;
	u32						dwTxQuit;
	u32						g_Connet;
	u32						g_USB_MODE;
	u32						g_USB_TIMEOUT;
	u16						g_status_buf;
	int						g_State;
	int 					g_bootfinish;
	int						g_bootaddr;
	int						g_bootsize;

	dwc_otg_core_if_t 		core_if_t;
	dwc_otg_dev_if_t 		dev_if_t;
	dwc_otg_core_params_t 	g_core_params;

	u32						g_in_pPara[3];
	u32						g_out_pPara[3];
//	u_setup_pkt 			g_u_setup_pkt[sizeof(u_setup_pkt)*5];
	dwc_otg_pcd_t 			g_dwc_otg_pcd_tp;
//	T_USB_ENUM                    g_enum;
	//add by 10136329 for charger or PC

};
#ifdef 	CONFIG_DWC_DEVICE_GPIO_CHARGER
	int usb_detect_irq;
	int usb_plugin;
#endif
	T_TYPE_USB_DETECT     g_plug_in = TYPE_UNKNOWN;
	usb_detect_callback	detect_fn = NULL;
struct g_data	  	*g_chg_usb_global = NULL;

void dwc_chg_Regcallback(usb_detect_callback	fn)
{
	USBHAL_DBG("dwc_chg_Regcallback");
	detect_fn = fn;
}
int detected_charger(void)
{
#ifdef CONFIG_DWC_DEVICE_GPIO_CHARGER

//for mdl,enum usb when poweron,maybe no gpio detect	
#ifndef _USE_VEHICLE_DC_REF
	//return 1; jb.qi add for usb wakelock on 20230918
#else
//others ,depends gpio detect
	return usb_plugin;
#endif
#endif
	if(detect_fn == NULL)
		return 1;
	else return 0;
}


static void dwc_chg_modify_reg32(uint32_t volatile *reg, uint32_t clear_mask, uint32_t set_mask)
{
    DWC_WRITE_REG32(reg,(DWC_READ_REG32(reg) & ~clear_mask) | set_mask);
}


static int dwc_otg_chg_setup_params(dwc_otg_core_if_t * core_if)
{
    int i;
    core_if->core_params = &(g_chg_usb_global->g_core_params);
    core_if->core_params->otg_cap = DWC_OTG_CAP_PARAM_NO_HNP_SRP_CAPABLE;
    core_if->core_params->phy_type = DWC_PHY_TYPE_PARAM_UTMI;
    core_if->core_params->speed = DWC_SPEED_PARAM_HIGH;//DWC_SPEED_PARAM_FULL
    core_if->core_params->phy_ulpi_ddr = dwc_param_phy_ulpi_ddr_default;
    core_if->core_params->phy_ulpi_ext_vbus = dwc_param_phy_ulpi_ext_vbus_default;
    core_if->core_params->phy_utmi_width = 8;
    core_if->core_params->ts_dline = dwc_param_ts_dline_default;
    core_if->core_params->ulpi_fs_ls = dwc_param_ulpi_fs_ls_default;
    core_if->core_params->en_multiple_tx_fifo = dwc_param_en_multiple_tx_fifo_default;

    core_if->core_params->dma_enable = dwc_param_dma_enable_default;
    core_if->core_params->dma_desc_enable = dwc_param_dma_desc_enable_default;
    core_if->core_params->max_transfer_size = dwc_param_max_transfer_size_default;
    core_if->core_params->dma_burst_size = dwc_param_dma_burst_size_default;

    for (i = 0; i < 15; i++)
    {
       core_if->core_params->dev_tx_fifo_size[i] = dwc_param_dev_tx_fifo_size_default;
    }
    core_if->core_params->otg_ver = 1;
    return 0;
}

dwc_otg_core_if_t *dwc_otg_chg_cil_init(const uint32_t * reg_base_addr)
{
    dwc_otg_core_if_t *core_if = &(g_chg_usb_global->core_if_t);
    dwc_otg_dev_if_t *dev_if = &(g_chg_usb_global->dev_if_t);
    uint8_t *pt_core_if = ( uint8_t *)&(g_chg_usb_global->core_if_t);
    uint8_t *pt_dev_if = ( uint8_t *)&(g_chg_usb_global->dev_if_t);
    uint8_t *reg_base = (uint8_t *) reg_base_addr;
    int i = 0;
    for(i= 0;i<sizeof(g_chg_usb_global->core_if_t);i++)
    {
        pt_core_if[i] = 0;
    }
    core_if->core_global_regs = (dwc_otg_core_global_regs_t *) reg_base;

    for(i= 0;i<sizeof(g_chg_usb_global->dev_if_t);i++)
    {
        pt_dev_if[i] = 0;
    }

    dev_if->dev_global_regs = (dwc_otg_device_global_regs_t *) (reg_base +DWC_DEV_GLOBAL_REG_OFFSET);

    for (i = 0; i < MAX_EPS_CHANNELS; i++)
    {
        dev_if->in_ep_regs[i] = (dwc_otg_dev_in_ep_regs_t *)
                                (reg_base + DWC_DEV_IN_EP_REG_OFFSET +
                                 (i * DWC_EP_REG_OFFSET));

        dev_if->out_ep_regs[i] = (dwc_otg_dev_out_ep_regs_t *)
                                 (reg_base + DWC_DEV_OUT_EP_REG_OFFSET +
                                  (i * DWC_EP_REG_OFFSET));

        core_if->data_fifo[i] =(uint32_t *) (reg_base + DWC_OTG_DATA_FIFO_OFFSET +(i * DWC_OTG_DATA_FIFO_SIZE));
    }

    dev_if->speed = 0;	

    core_if->dev_if = dev_if;

    core_if->pcgcctl = (uint32_t *) (reg_base + DWC_OTG_PCGCCTL_OFFSET);

    core_if->hwcfg1.d32 =DWC_READ_REG32(&core_if->core_global_regs->ghwcfg1);
    core_if->hwcfg2.d32 =DWC_READ_REG32(&core_if->core_global_regs->ghwcfg2);
    core_if->hwcfg3.d32 =DWC_READ_REG32(&core_if->core_global_regs->ghwcfg3);
    core_if->hwcfg4.d32 =DWC_READ_REG32(&core_if->core_global_regs->ghwcfg4);

    core_if->dcfg.d32 = DWC_READ_REG32(&core_if->dev_if->dev_global_regs->dcfg);

    core_if->snpsid = DWC_READ_REG32(&core_if->core_global_regs->gsnpsid);
	
    dwc_otg_chg_setup_params(core_if);


    return core_if;
}

/**
 * Initializes the DevSpd field of the DCFG register depending on the PHY type
 * and the enumeration speed of the device.
 */
	/**
	 * Initializes the DevSpd field of the DCFG register depending on the PHY type
	 * and the enumeration speed of the device.
	 */
static void init_chg_devspd(dwc_otg_core_if_t * core_if,uint8_t speed)
{
	uint32_t val;
	dcfg_data_t dcfg;

	val = speed;
	dcfg.d32 = DWC_READ_REG32(&core_if->dev_if->dev_global_regs->dcfg);
	dcfg.b.devspd = val;
	DWC_WRITE_REG32(&core_if->dev_if->dev_global_regs->dcfg, dcfg.d32);
}

/**
 * Flush a Tx FIFO.
 *
 * @param core_if Programming view of DWC_otg controller.
 * @param num Tx FIFO to flush.
 */
void dwc_otg_chg_flush_tx_fifo(dwc_otg_core_if_t * core_if, const int num)
{
    dwc_otg_core_global_regs_t *global_regs = core_if->core_global_regs;
    volatile grstctl_t greset;
    int count = 0;
    greset.d32 = 0;

    greset.b.txfflsh = 1;
    greset.b.txfnum = num;

{
    DWC_WRITE_REG32(&global_regs->grstctl, greset.d32);

    do
    {
        greset.d32 = DWC_READ_REG32(&global_regs->grstctl);
        if (++count > 10000)
        {
            break;
        }
        dwc_chg_udelay(10);
    }
    while (greset.b.txfflsh == 1);

    /* Wait for 3 PHY Clocks */
    dwc_chg_udelay(10);
}

}

/**
 * Flush Rx FIFO.
 *
 * @param core_if Programming view of DWC_otg controller.
 */
void dwc_otg_chg_flush_rx_fifo(dwc_otg_core_if_t * core_if)
{
    dwc_otg_core_global_regs_t *global_regs = core_if->core_global_regs;
    volatile grstctl_t greset;
    int count = 0;
    greset.d32 = 0;
    greset.b.rxfflsh = 1;

{
    DWC_WRITE_REG32(&global_regs->grstctl, greset.d32);

    do
    {
        greset.d32 = DWC_READ_REG32(&global_regs->grstctl);
        if (++count > 10000)
        {
            break;
        }
        dwc_chg_udelay(10);   
     }
    while (greset.b.rxfflsh == 1);

    /* Wait for 3 PHY Clocks */

    dwc_chg_udelay(10);
}
}

/**
 * This function initializes the DWC_otg controller registers for
 * device mode.
 *
 * @param core_if Programming view of DWC_otg controller
 *
 */
void dwc_otg_chg_core_dev_init(dwc_otg_core_if_t * core_if)
{
    int i;
//	T_TYPE_USB_DETECT chgtype;
    dwc_otg_dev_if_t *dev_if = core_if->dev_if;
    dcfg_data_t dcfg;
    grstctl_t resetctl;
    dctl_data_t dctl;
    diepmsk_data_t msk;
    dcfg.d32 = 0;
    resetctl.d32 = 0;
    dctl.d32 = 0;
    msk.d32 = 0;
    /* Restart the Phy Clock */
    DWC_WRITE_REG32(core_if->pcgcctl, 0);

    /* Device configuration register */
    init_chg_devspd(core_if,0);//Ĭ\C8\CF\C5\E4\D6óɸ\DF\CB\D9
    dcfg.d32 = DWC_READ_REG32(&dev_if->dev_global_regs->dcfg);
    dcfg.b.descdma = 0;
    dcfg.b.perfrint = DWC_DCFG_FRAME_INTERVAL_80;

    DWC_WRITE_REG32(&dev_if->dev_global_regs->dcfg, dcfg.d32);

    /* Flush the FIFOs */
    dwc_otg_chg_flush_tx_fifo(core_if, 0x10);	/* all Tx FIFOs */
    dwc_otg_chg_flush_rx_fifo(core_if);

    /* Flush the Learning Queue. */
    resetctl.b.intknqflsh = 1;
    DWC_WRITE_REG32(&core_if->core_global_regs->grstctl, resetctl.d32);
    /* Clear all pending Device Interrupts */
    /** @todo - if the condition needed to be checked
     *  or in any case all pending interrutps should be cleared?
     */
    DWC_WRITE_REG32(&dev_if->dev_global_regs->diepmsk, 0);
    DWC_WRITE_REG32(&dev_if->dev_global_regs->doepmsk, 0);
    DWC_WRITE_REG32(&dev_if->dev_global_regs->daint, 0xFFFFFFFF);
    DWC_WRITE_REG32(&dev_if->dev_global_regs->daintmsk, 0);
    
    for (i = 0; i <= dev_if->num_in_eps; i++)
    {
        DWC_WRITE_REG32(&dev_if->in_ep_regs[i]->diepctl,0);

        DWC_WRITE_REG32(&dev_if->in_ep_regs[i]->dieptsiz, 0);
        DWC_WRITE_REG32(&dev_if->in_ep_regs[i]->diepdma, 0);
        DWC_WRITE_REG32(&dev_if->in_ep_regs[i]->diepint, 0xFF);

        DWC_WRITE_REG32(&dev_if->out_ep_regs[i]->doepctl,0);

        DWC_WRITE_REG32(&dev_if->out_ep_regs[i]->doeptsiz, 0);
        DWC_WRITE_REG32(&dev_if->out_ep_regs[i]->doepdma, 0);
		USBHAL_DBG("%s, %u doepdmadbg i:%d", __func__, __LINE__, i);
        DWC_WRITE_REG32(&dev_if->out_ep_regs[i]->doepint, 0xFF);
    }
    msk.b.txfifoundrn = 1;
    dwc_chg_modify_reg32(&dev_if->dev_global_regs->diepmsk,msk.d32, msk.d32);
    dctl.d32 = DWC_READ_REG32(&dev_if->dev_global_regs->dctl);
    dctl.b.sftdiscon = 0;
    DWC_WRITE_REG32(&dev_if->dev_global_regs->dctl, dctl.d32);

    #if 1
    for(i = 0;i<3;i++)
    {
        dwc_chg_mdelay(30);
    	if((((DWC_READ_REG32(&dev_if->dev_global_regs->dsts))>>22)&0x3) ==3)
       {
          	g_plug_in = TYPE_ADAPTER;
       }
		else
       {
         	g_plug_in = TYPE_COMPUTER;
					break;
       }
    }
  //  if(g_plug_in == TYPE_ADAPTER)
	//	WARN_ON(1);
    #else
    dwc_chg_mdelay(100);
    if((((DWC_READ_REG32(&dev_if->dev_global_regs->dsts))>>22)&0x3) ==3)
    {
         g_plug_in = TYPE_ADAPTER;
		USBHAL_DBG("usb-dwc deteck TYPE_ADAPTER");
		 
//		 chgtype = TYPE_ADAPTER;
//		 usb_detect_callback(&chgtype, 0);
    }else
    {
         g_plug_in = TYPE_COMPUTER;
		USBHAL_DBG("usb-dwc deteck TYPE_COMPUTER");
//		 chgtype = TYPE_COMPUTER;
//		 usb_detect_callback(&chgtype, 0);
    }
    #endif

	if(detect_fn)
	{
		USBHAL_DBG("usb-dwc callback");
		detect_fn(g_plug_in);
	}

	dctl.d32 = DWC_READ_REG32(&dev_if->dev_global_regs->dctl);
	dctl.b.sftdiscon = 1;
	DWC_WRITE_REG32(&dev_if->dev_global_regs->dctl, dctl.d32);
}

/**
 * Do core a soft reset of the core.  Be careful with this because it
 * resets all the internal state machines of the core.
 */
void dwc_otg_chg_core_reset(dwc_otg_core_if_t * core_if)
{
    dwc_otg_core_global_regs_t *global_regs = core_if->core_global_regs;
    volatile grstctl_t greset;
    int count = 0;
    greset.d32 =0;
    /* Wait for AHB master IDLE state. */
    do
    {
        dwc_chg_udelay(10);
        greset.d32 = DWC_READ_REG32(&global_regs->grstctl);
        if (++count > 100000)
        {
            return;
        }
    }
    while (greset.b.ahbidle == 0);

    /* Core Soft Reset */
    count = 0;
    greset.b.csftrst = 1;
    DWC_WRITE_REG32(&global_regs->grstctl, greset.d32);
    do
    {
        greset.d32 = DWC_READ_REG32(&global_regs->grstctl);
        if (++count > 10000)
        {
            break;
        }
        dwc_chg_udelay(10);
    }
    while (greset.b.csftrst == 1);
/* Wait for 3 PHY Clocks */
    dwc_chg_udelay(10);
}

/**
 * This function initializes the DWC_otg controller registers and
 * prepares the core for device mode or host mode operation.
 *
 * @param core_if Programming view of the DWC_otg controller
 *
 */
void dwc_otg_chg_core_init(dwc_otg_core_if_t * core_if)
{
    dwc_otg_core_global_regs_t *global_regs = core_if->core_global_regs;
    dwc_otg_dev_if_t *dev_if = core_if->dev_if;
    gahbcfg_data_t ahbcfg;
    gotgctl_data_t gotgctl;

    gusbcfg_data_t usbcfg;
    ahbcfg.d32 = 0;
    usbcfg.d32 = 0;
    gotgctl.d32 = 0;


    usbcfg.d32 = DWC_READ_REG32(&global_regs->gusbcfg);

{

#if !USE_ASIC
    usbcfg.b.ulpi_ext_vbus_drv =(core_if->core_params->phy_ulpi_ext_vbus == DWC_PHY_ULPI_EXTERNAL_VBUS) ? 1 : 0;

    usbcfg.b.term_sel_dl_pulse = (core_if->core_params->ts_dline == 1) ? 1 : 0;


    DWC_WRITE_REG32(&global_regs->gusbcfg, usbcfg.d32);
#endif
    dwc_otg_chg_core_reset(core_if);
	}


    dev_if->num_in_eps =  2;//calc_num_in_eps(core_if);
    dev_if->num_out_eps = 2;//calc_num_out_eps(core_if);

    core_if->total_fifo_size = core_if->hwcfg3.b.dfifo_depth;
    core_if->rx_fifo_size = DWC_READ_REG32(&global_regs->grxfsiz);
    core_if->nperio_tx_fifo_size = DWC_READ_REG32(&global_regs->gnptxfsiz) >> 16;
    {
        /* High speed PHY. */
        if (!core_if->phy_init_done)
        {
            core_if->phy_init_done = 1;
            /* HS PHY parameters.  These parameters are preserved
             * during soft reset so only program the first time.  Do
             * a soft reset immediately after setting phyif.  */
         #if !USE_ASIC
            if (core_if->core_params->phy_type == 2)
            {
                /* ULPI interface */
                usbcfg.b.ulpi_utmi_sel = 1;
                usbcfg.b.phyif = 0;
                usbcfg.b.ddrsel = core_if->core_params->phy_ulpi_ddr;
            }
          #else
            if (core_if->core_params->phy_type == 1)
            {
                /* UTMI+ interface */
                usbcfg.b.ulpi_utmi_sel = 0;
                if (core_if->core_params->phy_utmi_width == 16)
                {
                    usbcfg.b.phyif = 1;

                }
                else
                {
                    usbcfg.b.phyif = 0;
                }
            }
            #endif

	{
            DWC_WRITE_REG32(&global_regs->gusbcfg, usbcfg.d32);
            /* Reset after setting the PHY parameters */
            dwc_otg_chg_core_reset(core_if);
	}
        }
    }

		{
#if !USE_ASIC
    if ((core_if->hwcfg2.b.hs_phy_type == 2) &&(core_if->hwcfg2.b.fs_phy_type == 1) &&(core_if->core_params->ulpi_fs_ls))
    {
        usbcfg.d32 = DWC_READ_REG32(&global_regs->gusbcfg);
        usbcfg.b.ulpi_fsls = 1;
        usbcfg.b.ulpi_clk_sus_m = 1;
        DWC_WRITE_REG32(&global_regs->gusbcfg, usbcfg.d32);
    }
    else
    {
        usbcfg.d32 = DWC_READ_REG32(&global_regs->gusbcfg);
        usbcfg.b.ulpi_fsls = 0;
        usbcfg.b.ulpi_clk_sus_m = 0;
        DWC_WRITE_REG32(&global_regs->gusbcfg, usbcfg.d32);
    }
#endif
		}
#if DMA_ENABLE
        /* Program the GAHBCFG Register. */
	switch (core_if->hwcfg2.b.architecture) 
    {
        case DWC_SLAVE_ONLY_ARCH:
            
    		//ahbcfg.b.nptxfemplvl_txfemplvl = DWC_GAHBCFG_TXFEMPTYLVL_HALFEMPTY;
            //  \C5\E4\D6\C3ȫ\BF\D5FIFO\B2\FA\C9\FA\D6ж\CF
            ahbcfg.b.nptxfemplvl_txfemplvl = DWC_GAHBCFG_TXFEMPTYLVL_EMPTY ;
    		ahbcfg.b.ptxfemplvl = DWC_GAHBCFG_TXFEMPTYLVL_HALFEMPTY;
    		core_if->dma_enable = 0;
    		core_if->dma_desc_enable = 0;
    		break;

        case DWC_EXT_DMA_ARCH:
    		{
    			uint8_t brst_sz = core_if->core_params->dma_burst_size;
    			ahbcfg.b.hburstlen = 0;
    			while (brst_sz > 1) 
                {
    				ahbcfg.b.hburstlen++;
    				brst_sz >>= 1;
    			}
    		}
    		core_if->dma_enable = (core_if->core_params->dma_enable != 0);
    		core_if->dma_desc_enable = (core_if->core_params->dma_desc_enable != 0);
    		break;

        case DWC_INT_DMA_ARCH:
    		/* Old value was DWC_GAHBCFG_INT_DMA_BURST_INCR - done for 
    		   Host mode ISOC in issue fix - vahrama */
    		ahbcfg.b.hburstlen = DWC_GAHBCFG_INT_DMA_BURST_INCR4;
    		core_if->dma_enable = (core_if->core_params->dma_enable != 0);
    		core_if->dma_desc_enable = (core_if->core_params->dma_desc_enable != 0);
    		break;

    }
	 if (core_if->dma_enable)
    {
        if (core_if->dma_desc_enable) 
        {
			printf("Using Descriptor DMA mode\n");
		} 
        else 
       {
			printf("using buffer dma mode\n");

        }
    }
	 else 
	{
		printf("Using Slave mode\n");

		core_if->dma_desc_enable = 0;
	}
	 ahbcfg.b.dmaenable = core_if->dma_enable;
 #else
    ahbcfg.b.nptxfemplvl_txfemplvl = DWC_GAHBCFG_TXFEMPTYLVL_HALFEMPTY;
    ahbcfg.b.ptxfemplvl = DWC_GAHBCFG_TXFEMPTYLVL_HALFEMPTY;


    ahbcfg.b.dmaenable = 0;
//  \C5\E4\D6\C3ȫ\BF\D5FIFO\B2\FA\C9\FA\D6ж\CF
    ahbcfg.b.nptxfemplvl_txfemplvl = DWC_GAHBCFG_TXFEMPTYLVL_EMPTY ;
#endif

    DWC_WRITE_REG32(&global_regs->gahbcfg, ahbcfg.d32);

    core_if->en_multiple_tx_fifo = core_if->hwcfg4.b.ded_fifo_en;

    core_if->pti_enh_enable = core_if->core_params->pti_enable != 0;
    core_if->multiproc_int_enable = dwc_param_mpi_enable_default;//core_if->core_params->mpi_enable;
    /*
     * Program the GUSBCFG register.
     */
    usbcfg.d32 = DWC_READ_REG32(&global_regs->gusbcfg);
    usbcfg.b.hnpcap = 0;
    usbcfg.b.srpcap = 0;

    	DWC_WRITE_REG32(&global_regs->gusbcfg, usbcfg.d32);

        gotgctl.b.otgver = core_if->core_params->otg_ver;

        	dwc_chg_modify_reg32(&core_if->core_global_regs->gotgctl, 0, gotgctl.d32);

        /* Set OTG version supported */
    core_if->otg_ver = core_if->core_params->otg_ver;

   dwc_otg_chg_core_dev_init(core_if);
}

struct platform_device *zx297510_platform_chg_usb_dev;

#ifdef USB_CHG_DEV
static int dwc_otg_chg_driver_probe(struct platform_device *_dev)
{
    zx297510_platform_chg_usb_dev = _dev;
	return 0;
}

static void dwc_otg_chg_driver_remove( struct platform_device *_dev)
{
}

static struct platform_driver dwc_otg_chg_driver = {
	.driver = {
	.name    = (char *)dwc_chg_name,
       .bus       = &platform_bus_type,
       .owner   = THIS_MODULE,
       //.pm       =  DWC_OTG_PM_OPS,
	},
	.probe    = dwc_otg_chg_driver_probe,
	.remove  = __exit_p(dwc_otg_chg_driver_remove),
};
#endif
uint32_t dwc_otg_chg_get_gsnpsid(dwc_otg_core_if_t * core_if)
{
	return core_if->snpsid;
}

/**
 * This function is called when an lm_device is bound to a
 * dwc_otg_driver. It creates the driver components required to
 * control the device (CIL, HCD, and PCD) and it initializes the
 * device. The driver components are stored in a dwc_otg_device
 * structure. A reference to the dwc_otg_device is saved in the
 * lm_device. This allows the driver to access the dwc_otg_device
 * structure on subsequent calls to driver methods for this device.
 *
 * @param _dev Bus device
 */
//extern int dwc_otg_open_power(void);
//extern int dwc_otg_close_power(void);
#ifdef USB_CHG_DEV
int dwc_otg_usb_chg_detect(struct platform_device *_dev)
{
	int retval = 0;
    struct clk* wclk;
	struct clk* pclk;
    struct resource *iomem;

	dwc_otg_device_t *dwc_otg_device = (g_chg_usb_global->g_dwc_otg_dev_t);
	
	iomem = platform_get_resource(_dev, IORESOURCE_MEM,0);
  
    dev_dbg(&_dev->dev, "start=0x%08x\n", (unsigned)iomem->start);

	memset(dwc_otg_device, 0, sizeof(dwc_otg_device_t));
	dwc_otg_device->os_dep.reg_offset = 0xFFFFFFFF;

	dwc_otg_device->os_dep.regs_res = request_mem_region(iomem->start, resource_size(iomem),
										dev_name(_dev));
	dwc_otg_device->os_dep.base= ioremap(iomem->start, resource_size(iomem));
 
	if (!dwc_otg_device->os_dep.base) {
		dev_err(&_dev->dev, "ioremap() failed\n");
		return -ENOMEM;
	}

	/*
	 * Initialize driver data to point to the global DWC_otg
	 * Device structure.
	 */
	platform_set_drvdata(_dev, dwc_otg_device);

	dev_dbg(&_dev->dev, "dwc_otg_device=0x%p\n", dwc_otg_device);

/*
       Please ensure power on the USB controller and Phy
*/
#if 0
    zx29_usbdev_init(); //
    retval = dwc_otg_open_power();
#endif

	pclk = clk_get(&_dev->dev, "ahb_clk");
    if(IS_ERR(pclk)	){
		return -EIO;
    }

	wclk = clk_get(&_dev->dev, "work_clk");
	
    if(IS_ERR(pclk)	){
		return -EIO;
    }

	clk_enable(pclk);
	clk_enable(wclk);

#if 1
    zx29_usbdev_init();
    retval = dwc_otg_open_power();
#endif

	dwc_otg_device->core_if = dwc_otg_chg_cil_init(dwc_otg_device->os_dep.base);
	if (!dwc_otg_device->core_if) {
		dev_err(&_dev->dev, "CIL initialization failed!\n");
		retval = -ENOMEM;
		goto fail;
	}
	if (((dwc_otg_chg_get_gsnpsid(dwc_otg_device->core_if) & 0xFFFFF000) !=	0x4F542000) &&
		((dwc_otg_chg_get_gsnpsid(dwc_otg_device->core_if) & 0xFFFFF000) != 0x4F543000)) {
		dev_err(&_dev->dev, "%s, Bad value for SNPSID: 0x%08x\n",__func__,
			dwc_otg_chg_get_gsnpsid(dwc_otg_device->core_if));
		retval = -EINVAL;
		goto fail;
	}

    dwc_otg_chg_core_init(dwc_otg_device->core_if);

   clk_disable(wclk);
   clk_disable(pclk);
   clk_put(wclk);
   clk_put(pclk);

    retval = dwc_otg_close_power();
    
     /*
      * Return the memory.
      */
     
     if(dwc_otg_device->os_dep.regs_res){
     	release_resource(dwc_otg_device->os_dep.regs_res);
     	kfree(dwc_otg_device->os_dep.regs_res);
     }
     if (dwc_otg_device->os_dep.base) {
     	iounmap(dwc_otg_device->os_dep.base);
     }
    
     /*
      * Clear the drvdata pointer.
      */
     platform_set_drvdata(_dev, 0);

	 platform_driver_unregister(&dwc_otg_chg_driver);

	 if (g_plug_in == TYPE_ADAPTER)
	 {
	     usb_notify_up(USB_CHARGER_PLUGIN, NULL);
	 }
	 else
	 {
	     usb_notify_up(USB_DEVICE_PLUGIN, NULL);
	 }
	 usb_printk("%s, detect done plunin:%d\n", __func__, g_plug_in);
fail:
	return 0;

}
EXPORT_SYMBOL_GPL(dwc_otg_usb_chg_detect);
#else



int dwc_otg_usb_chg_detect(void)
{
	int retval = 0;
    struct clk* wclk;
	struct clk* pclk;
    struct resource iomem;
    unsigned int value;
	g_chg_usb_global = (struct g_data *)kmalloc(sizeof(struct g_data), GFP_KERNEL);
	if(g_chg_usb_global == NULL){	
		g_chg_usb_global = (struct g_data *)kmalloc(sizeof(struct g_data), GFP_KERNEL);
		if(g_chg_usb_global == NULL){
			printk("fetal error alloc g_chg_usb_global fail\n");
			WARN_ON(1);
			return -1;
		}
	}

	memset(g_chg_usb_global, 0, sizeof(struct g_data));		
	
	dwc_otg_device_t *dwc_otg_device = &(g_chg_usb_global->g_dwc_otg_dev_t);

	memset(dwc_otg_device, 0, sizeof(dwc_otg_device_t));
	dwc_otg_device->os_dep.reg_offset = 0xFFFFFFFF;
    iomem.start = ZX29_USB_PHYS;// change by gsn
	iomem.end = ZX29_USB_PHYS+SZ_256K-1;// change by gsn
#if 1
    dwc_otg_device->os_dep.base= ioremap(iomem.start, resource_size(&iomem));
#else
    dwc_otg_device->os_dep.base = ZX29_USB1_CFG_BASE_VA;
#endif
    if(!dwc_otg_device->os_dep.base)
	{
		retval = -ENOMEM;
		goto fail;
	}
/*
       Please ensure power on the USB controller and Phy
*/
	dwc_otg_hal_init();

    dwc_chg_udelay(10);

	dwc_otg_device->core_if = dwc_otg_chg_cil_init(dwc_otg_device->os_dep.base);
	if (!dwc_otg_device->core_if) {
		printk(KERN_ERR "%s, CIL initialization failed!\n", __func__);
		retval = -ENOMEM;
		goto fail;
	}
	if (((dwc_otg_chg_get_gsnpsid(dwc_otg_device->core_if) & 0xFFFFF000) !=	0x4F542000) &&
		((dwc_otg_chg_get_gsnpsid(dwc_otg_device->core_if) & 0xFFFFF000) != 0x4F543000)) {
		printk(KERN_ERR "%s, Bad value for SNPSID: 0x%08x\n",__func__,
			dwc_otg_chg_get_gsnpsid(dwc_otg_device->core_if));
		retval = -EINVAL;
		goto fail;
	}

    dwc_otg_chg_core_init(dwc_otg_device->core_if);

	dwc_otg_hal_exit();
	printk("dwc_otg_usb_chg_detect, g_plug_in:%d, type:%s\n", g_plug_in, (g_plug_in == 1 ? "pc" : "adapter"));
    if (g_plug_in == TYPE_ADAPTER)
    {
        usb_notify_up(USB_CHARGER_PLUGIN, NULL);
    }
    else
    {
        usb_notify_up(USB_DEVICE_PLUGIN, NULL);
//		wake_lock(&dwc_otg_wake_lock);
        dwc_otg_wakelock(1,1);
    }

fail:
    if (dwc_otg_device->os_dep.base) 
    {
#if 1
        iounmap(dwc_otg_device->os_dep.base);
#endif
    }
    kfree(g_chg_usb_global);
    g_chg_usb_global = NULL;
    
	return 0;

}
EXPORT_SYMBOL_GPL(dwc_otg_usb_chg_detect);

#endif
int dwc_otg_usb_chg_remove(void)
{
    if (g_plug_in == TYPE_ADAPTER)
    {
    	usb_notify_up(USB_CHARGER_PLUGOUT, NULL);
    }
    else
    {
    	usb_notify_up(USB_DEVICE_PLUGOUT, NULL);
//		wake_unlock(&dwc_otg_wake_lock);
        //dwc_otg_wakelock(0,0);
    }
    g_plug_in = TYPE_UNKNOWN;
    return 0;
}
EXPORT_SYMBOL_GPL(dwc_otg_usb_chg_remove);


void dwc_otg_chg_inform(int inform)
{
	if(inform == 0)
		dwc_otg_usb_chg_detect();
	else if(inform == 1)
		dwc_otg_usb_chg_remove();
	else
		USBHAL_DBG("dwc_otg_chg_inform %d", inform);
}
EXPORT_SYMBOL_GPL(dwc_otg_chg_inform);

int dwc_otg_chg_inform_type(int inform)
{
	int chg_type = 0;
	if(inform == 0){
		dwc_otg_usb_chg_detect();
		if (g_plug_in == TYPE_ADAPTER){
			chg_type = 0;
		}else{
			chg_type = 1;
		}
	}
	else if(inform == 1)
		dwc_otg_usb_chg_remove();
	else
		USBHAL_DBG("dwc_otg_chg_inform %d", inform);
	return chg_type;
}
EXPORT_SYMBOL_GPL(dwc_otg_chg_inform_type);
#ifdef USB_CHG_DEV

struct platform_device *get_chg_usb_platform_device(void)
{
    usb_printk("%s, dev:%p, name:%s\n", __func__, zx297510_platform_chg_usb_dev, zx297510_platform_chg_usb_dev->name);
    return zx297510_platform_chg_usb_dev;
}
EXPORT_SYMBOL_GPL(get_chg_usb_platform_device);

static int __init dwc_otg_chg_init(void)
{
	int retval = 0;
	int error;

	usb_printk("%s, %s: version %s\n", __func__, dwc_chg_name,
	       DWC_DRIVER_VERSION);
	
	retval = platform_driver_register(&dwc_otg_chg_driver);


	if (retval < 0) {
		printk(KERN_ERR "%s retval=%d\n", __func__, retval);
		return retval;
	}
	return retval;
}

module_init(dwc_otg_chg_init);
#endif
#ifdef CONFIG_DWC_DEVICE_GPIO_CHARGER
extern unsigned int get_usb_gpio_detect_flag(void);
//wangzhen
extern int dwc_otg_disconnect(void);
/*GPIO\BA\CD\CDⲿ\D6жϺŸ\F9\BE\DD\CF\EEĿʵ\BC\CA\C7\E9\BF\F6\D0޸\C4
 *\B4˴\A6Ϊ\B2ο\BC\B4\FA\C2\EB
 */
//xf.li@20230909 modify for usb hot-plug start
 #ifdef _USE_VEHICLE_DC_REF
#define USB_GPIO ZX29_GPIO_125
#define USB_GPIO_FUNC_GPIO GPIO125_GPIO125
#define USB_GPIO_FUNC_EXT_INT GPIO125_EXT_INT14
#define USB_DT_INT  PCU_EX14_INT
//xf.li@20230909 modify for usb hot-plug end
 #else
#define USB_GPIO ZX29_GPIO_52
#define USB_GPIO_FUNC_GPIO GPIO52_GPIO52
#define USB_GPIO_FUNC_EXT_INT GPIO52_EXT_INT5
#define USB_DT_INT  PCU_EX5_INT
#endif
int Usb_Detect_Val(void)
{
    int value;
    
    value = gpio_get_value(USB_GPIO);

	return value;
}
EXPORT_SYMBOL_GPL(Usb_Detect_Val);

#ifdef _USE_VEHICLE_DC

 /*
  *disable or enable usb plug interrupt
  *set_cmd:0:disable;1:enable and check plug state
  */
void usb_detect_set(int set_cmd)
 {
	int value;

	if(set_cmd){
		//should check now plugin or not

		zx29_gpio_config(USB_GPIO,USB_GPIO_FUNC_GPIO);
		gpio_direction_input(USB_GPIO);
		msleep(500);
		value = gpio_get_value(USB_GPIO);
		printk("%s,value:%d, usb_plugin:%d\n", __func__,value, usb_plugin);
		zx29_gpio_config(USB_GPIO,USB_GPIO_FUNC_EXT_INT);

#ifdef _USE_VEHICLE_DC_REF
		if(value == 1)
		{		
			zx29_gpio_set_inttype(USB_GPIO,IRQ_TYPE_EDGE_FALLING);
						
			usb_plugin = 1;
		}else{
			zx29_gpio_set_inttype(USB_GPIO,IRQ_TYPE_EDGE_RISING);
							
			usb_plugin = 0;	
		}
#else
		if(value == 0)
		{		
			zx29_gpio_set_inttype(USB_GPIO,IRQ_TYPE_EDGE_FALLING);						
			usb_plugin = 1;
		}else{
			zx29_gpio_set_inttype(USB_GPIO,IRQ_TYPE_EDGE_RISING);							
			usb_plugin = 0;	
		}
#endif
	 	pcu_int_clear(USB_DT_INT);		
		enable_irq(usb_detect_irq);
		return;
	}
	
	usb_plugin = 0;
	disable_irq_nosync(usb_detect_irq);

 	pcu_int_clear(USB_DT_INT);
	irq_set_irq_wake(usb_detect_irq, 0);

 }

EXPORT_SYMBOL_GPL(usb_detect_set);
#endif
irqreturn_t Usb_Detect_Irq_Handler(int irq, struct platform_device *_dev)
{
	printk(KERN_INFO"%s,end\n", __func__);
	pcu_int_clear(USB_DT_INT);
	//dwc_otg_wakelock(1,0);
	return IRQ_WAKE_THREAD;
}

irqreturn_t  Usb_Detect_Irq_Thread(int irq, struct platform_device *_dev)
{
    int ret = -1;
    int irq_num;
    int retval;
	int value;
	unsigned int gpio_enable = 0;
	struct sched_param param = { .sched_priority = 2 };
	param.sched_priority= 33;
	sched_setscheduler(current, SCHED_FIFO, &param);
#ifndef _USE_VEHICLE_DC	
	gpio_enable = get_usb_gpio_detect_flag();
	printk("Usb_Detect_Irq_Thread, gpio_enable:%d\n", gpio_enable);
	if(gpio_enable == 0){	
		pcu_int_clear(USB_DT_INT); 
		return IRQ_HANDLED;
	}
#endif

	  dwc_otg_wakelock(1,0);
	//5.23
	zx29_gpio_config(USB_GPIO,USB_GPIO_FUNC_GPIO);
	gpio_direction_input(USB_GPIO);
	msleep(500);
	value = gpio_get_value(USB_GPIO);
	printk("%s,value:%d, usb_plugin:%d\n", __func__,value, usb_plugin);
	zx29_gpio_config(USB_GPIO,USB_GPIO_FUNC_EXT_INT);

#ifdef _USE_VEHICLE_DC_REF
	if(value == 1)
	{
	
	  zx29_gpio_set_inttype(USB_GPIO,IRQ_TYPE_EDGE_FALLING);
	  pcu_int_clear(USB_DT_INT); 
	  if(usb_plugin == 0){
	  	dwc_otg_usb_chg_detect(); //plug in;
	  	usb_plugin = 1;
	  }	
	}
	else
	{
	  zx29_gpio_set_inttype(USB_GPIO,IRQ_TYPE_EDGE_RISING);
	  pcu_int_clear(USB_DT_INT);
	  if(usb_plugin == 1){
	 	 dwc_otg_disconnect();
	  	dwc_otg_usb_chg_remove(); //not plug in;
	  	usb_plugin = 0;
	  }
	}
#else
	
	if(value == 1)
	{
	  zx29_gpio_set_inttype(USB_GPIO,IRQ_TYPE_EDGE_FALLING);
	  pcu_int_clear(USB_DT_INT);
	  if(usb_plugin == 1){
	 	 dwc_otg_disconnect();
	  	dwc_otg_usb_chg_remove(); //not plug in;
	  	usb_plugin = 0;
	  }
	}
	else
	{
	  zx29_gpio_set_inttype(USB_GPIO,IRQ_TYPE_EDGE_RISING);
	  pcu_int_clear(USB_DT_INT); 
	  if(usb_plugin == 0){
	  	dwc_otg_usb_chg_detect(); //plug in;
	  	usb_plugin = 1;
	  }
	}
#endif	
	printk(KERN_INFO"%s,value:%d,usb_plugin:%d, end\n", __func__,value, usb_plugin);
    USBSTACK_DBG("%s,value:%d", __func__,value);
	return IRQ_HANDLED;
}

static void usb_detect_typedet(T_TYPE_USB_DETECT chg_type)
{
	printk("register usb_detect_typedet for charger\n");	
}

int Usb_Detect_Irq_probe(struct platform_device *_dev)
{
    USBSTACK_DBG("%s", __func__);
    int ret = -1;
	int value;
	int value2 = 0;
	int i;
	printk("-----------Usb_Detect_Irq_probe\n");

	dwc_chg_Regcallback(usb_detect_typedet);

	ret = gpio_request(USB_GPIO, "usb");
	zx29_gpio_config(USB_GPIO,USB_GPIO_FUNC_GPIO);
	gpio_direction_input(USB_GPIO);
	zx29_gpio_pd_pu_set(USB_GPIO,IO_CFG_PULL_DISABLE);
	gpio_export(USB_GPIO, 1);
	mdelay(5);
	value = gpio_get_value(USB_GPIO);
	if(value == 0){

		for(i = 0; i < 5; i++){
			value = gpio_get_value(USB_GPIO);
			if(value == 0){
				mdelay(50);
			}
		}
	}
	USBSTACK_DBG("%s,value:%d", __func__,value);
    usb_detect_irq= gpio_to_irq(USB_GPIO);
	printk(KERN_INFO "%s,value:%d, irq_num:%d\n",__func__,value, usb_detect_irq);
	
    zx29_gpio_config(USB_GPIO,USB_GPIO_FUNC_EXT_INT);
 #ifdef _USE_VEHICLE_DC_REF
 	if(value == 1)
	{
	  dwc_otg_usb_chg_detect(); //plug in;
	  usb_plugin = 1;
	  zx29_gpio_set_inttype(USB_GPIO,IRQ_TYPE_EDGE_FALLING);
	}
	else
	{
	  usb_plugin = 0;
	  //zx29_gpio_set_inttype(USB_GPIO,IRQ_TYPE_EDGE_RISING);
	  zx29_gpio_set_inttype(USB_GPIO,IRQ_TYPE_LEVEL_HIGH);
	}
 #else
 
	if(value == 1)
	{
	  usb_plugin = 0;
	  zx29_gpio_set_inttype(USB_GPIO,IRQ_TYPE_EDGE_FALLING);
	}
	else
	{
	  dwc_otg_usb_chg_detect(); //plug in;
	  usb_plugin = 1;
	  zx29_gpio_set_inttype(USB_GPIO,IRQ_TYPE_EDGE_RISING);
	}
#endif	
    pcu_int_clear(USB_DT_INT);
	ret = request_threaded_irq(usb_detect_irq, Usb_Detect_Irq_Handler,Usb_Detect_Irq_Thread,IRQF_ONESHOT,
                "usb", _dev);

	printk(KERN_INFO "%s,ret:%d, usb_plugin:%d\n",__func__,ret, usb_plugin);
   	if (ret)
    {   
	   printk(KERN_INFO"cannot request Usb_Detect_Irq\n");
	   return ret;
    }
	
	irq_set_irq_wake(usb_detect_irq, 1);
	
	return 0;
	
}

static int usb_detect_suspend(void)
{
	disable_irq(usb_detect_irq);
	return 0;
}

static int usb_detect_resume(void)
{
	enable_irq(usb_detect_irq);
	return 0;
}

static struct platform_driver usb_detect_platform_driver = {
	.probe		= Usb_Detect_Irq_probe,
	.driver		= {
		.name	= "usb_detect",
		.owner	= THIS_MODULE,
	},
	.suspend = usb_detect_suspend,
	.resume = usb_detect_resume,
};

static int __init usb_detect_init(void)
{
	printk("-----------usb_detect_init\n");
	return platform_driver_register(&usb_detect_platform_driver);
}

module_init(usb_detect_init);

#endif
