/*
 * Gadget Driver for Android ADB
 *
 * Copyright (C) 2008 Google, Inc.
 * Author: Mike Lockwood <lockwood@android.com>
 *
 * This software is licensed under the terms of the GNU General Public
 * License version 2, as published by the Free Software Foundation, and
 * may be copied, distributed, and modified under those terms.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/poll.h>
#include <linux/delay.h>
#include <linux/wait.h>
#include <linux/err.h>
#include <linux/interrupt.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/device.h>
#include <linux/miscdevice.h>
#ifdef _USE_VEHICLE_DC
#include <linux/soc/zte/rpm/rpmsg.h>
#include <linux/android_notify.h>
#endif
#define ADB_BULK_BUFFER_SIZE           4096

/* number of tx requests to allocate */
#define TX_REQ_MAX 4
#ifdef _USE_VEHICLE_DC
extern int adb_rpmsg_write2ap(void *buf, unsigned int len);
extern int adb_agent_close(void);
extern void adb_server_plug_notify(e_usb_rpmsg_cmd_type plug_type);
extern int usb_get_adb_agent(void);

#endif

static const char adb_shortname[] = "android_adb";

struct adb_dev {
	struct usb_function function;
	struct usb_composite_dev *cdev;
	spinlock_t lock;

	struct usb_ep *ep_in;
	struct usb_ep *ep_out;

	int online;
	int error;

	atomic_t read_excl;
	atomic_t write_excl;
	atomic_t open_excl;
	atomic_t enable_excl;

	struct list_head tx_idle;

	wait_queue_head_t read_wq;
	wait_queue_head_t write_wq;
	struct usb_request *rx_req;
	int rx_done;

#ifdef _USE_VEHICLE_DC	
	atomic_t write_busy;
	atomic_t agt_read_flag;
	atomic_t adb_read_flag;
	atomic_t agt_write_flag;
	int set_test;
	int agt_error;
	char   *rx2agt_buf;
	struct task_struct	*rpmsg_thread;	
	struct task_struct	*agent_monitor_thread;	
	wait_queue_head_t agent_monitor_wq;	
	wait_queue_head_t agt_start_wait;
	wait_queue_head_t agt_read_wq;
	wait_queue_head_t agt_write_wq;
	atomic_t agent_switch;
	int agent_state;
	
	int agent_start;//using for recv thread
	struct work_struct agent_inform_work;	 /* inform USB adb state */	
	struct delayed_work agent_switch_work;
#endif	
	struct work_struct online_inform;	 /* online inform USB adb  */	
	struct work_struct offline_inform;	 /* offline inform USB adb  */	

	//this flag means adb is ok
	int adb_ready;	
#ifdef CONFIG_PM
	u32 suspend_state;
#endif
};

static struct usb_interface_descriptor adb_interface_desc = {
	.bLength                = USB_DT_INTERFACE_SIZE,
	.bDescriptorType        = USB_DT_INTERFACE,
	.bInterfaceNumber       = 0,
	.bNumEndpoints          = 2,
	.bInterfaceClass        = 0xFF,
	.bInterfaceSubClass     = 0x42,
	.bInterfaceProtocol     = 1,
};

static struct usb_endpoint_descriptor adb_highspeed_in_desc = {
	.bLength                = USB_DT_ENDPOINT_SIZE,
	.bDescriptorType        = USB_DT_ENDPOINT,
	.bEndpointAddress       = USB_DIR_IN,
	.bmAttributes           = USB_ENDPOINT_XFER_BULK,
	.wMaxPacketSize         = __constant_cpu_to_le16(512),
};

static struct usb_endpoint_descriptor adb_highspeed_out_desc = {
	.bLength                = USB_DT_ENDPOINT_SIZE,
	.bDescriptorType        = USB_DT_ENDPOINT,
	.bEndpointAddress       = USB_DIR_OUT,
	.bmAttributes           = USB_ENDPOINT_XFER_BULK,
	.wMaxPacketSize         = __constant_cpu_to_le16(512),
};

static struct usb_endpoint_descriptor adb_fullspeed_in_desc = {
	.bLength                = USB_DT_ENDPOINT_SIZE,
	.bDescriptorType        = USB_DT_ENDPOINT,
	.bEndpointAddress       = USB_DIR_IN,
	.bmAttributes           = USB_ENDPOINT_XFER_BULK,
};

static struct usb_endpoint_descriptor adb_fullspeed_out_desc = {
	.bLength                = USB_DT_ENDPOINT_SIZE,
	.bDescriptorType        = USB_DT_ENDPOINT,
	.bEndpointAddress       = USB_DIR_OUT,
	.bmAttributes           = USB_ENDPOINT_XFER_BULK,
};

static struct usb_descriptor_header *fs_adb_descs[] = {
	(struct usb_descriptor_header *) &adb_interface_desc,
	(struct usb_descriptor_header *) &adb_fullspeed_in_desc,
	(struct usb_descriptor_header *) &adb_fullspeed_out_desc,
	NULL,
};

static struct usb_descriptor_header *hs_adb_descs[] = {
	(struct usb_descriptor_header *) &adb_interface_desc,
	(struct usb_descriptor_header *) &adb_highspeed_in_desc,
	(struct usb_descriptor_header *) &adb_highspeed_out_desc,
	NULL,
};

static void adb_ready_callback(void);
static void adb_closed_callback(void);

#ifdef _USE_VEHICLE_DC
extern int adb_server_init(struct adb_dev *dev);
extern int adb_server_release(void);
#endif


/* temporary variable used between adb_open() and adb_gadget_bind() */
 struct adb_dev *_adb_dev;

static inline struct adb_dev *func_to_adb(struct usb_function *f)
{
	return container_of(f, struct adb_dev, function);
}


static struct usb_request *adb_request_new(struct usb_ep *ep, int buffer_size)
{
	struct usb_request *req = usb_ep_alloc_request(ep, GFP_KERNEL);
	if (!req)
		return NULL;

	/* now allocate buffers for the requests */
	req->buf = kmalloc(buffer_size, GFP_KERNEL);
	if (!req->buf) {
		usb_ep_free_request(ep, req);
		return NULL;
	}

	return req;
}

static void adb_request_free(struct usb_request *req, struct usb_ep *ep)
{
	if (req) {
		kfree(req->buf);
		usb_ep_free_request(ep, req);
	}
}

static inline int adb_lock(atomic_t *excl)
{
	if (atomic_inc_return(excl) == 1) {
		return 0;
	} else {
		atomic_dec(excl);
		return -1;
	}
}

static inline void adb_unlock(atomic_t *excl)
{
	atomic_dec(excl);
}

/* add a request to the tail of a list */
void adb_req_put(struct adb_dev *dev, struct list_head *head,
		struct usb_request *req)
{
	unsigned long flags;

	spin_lock_irqsave(&dev->lock, flags);
	list_add_tail(&req->list, head);
	spin_unlock_irqrestore(&dev->lock, flags);
}

/* remove a request from the head of a list */
struct usb_request *adb_req_get(struct adb_dev *dev, struct list_head *head)
{
	unsigned long flags;
	struct usb_request *req;

	spin_lock_irqsave(&dev->lock, flags);
	if (list_empty(head)) {
		req = 0;
	} else {
		req = list_first_entry(head, struct usb_request, list);
		list_del(&req->list);
	}
	spin_unlock_irqrestore(&dev->lock, flags);
	return req;
}

static void adb_complete_in(struct usb_ep *ep, struct usb_request *req)
{
	struct adb_dev *dev = _adb_dev;

	if (req->status != 0)
		USBSTACK_DBG("adb_complete_in err: %d\n", req->status);
	//	dev->error = 1;
		USBSTACK_DBG("adb_complete_in status: %d\n", req->status);

	adb_req_put(dev, &dev->tx_idle, req);

	wake_up(&dev->write_wq);
}

static void adb_complete_out(struct usb_ep *ep, struct usb_request *req)
{
	struct adb_dev *dev = _adb_dev;

	dev->rx_done = 1;
	//if (req->status != 0 && req->status != -ECONNRESET)
	//	dev->error = 1;

		USBSTACK_DBG("adb_complete_out status: %d, len:%d\n", req->status, req->actual);
	if (req->status != 0)
		USBSTACK_DBG("adb_complete_out err: %d\n", req->status);

	wake_up(&dev->read_wq);
}

static int adb_create_bulk_endpoints(struct adb_dev *dev,
				struct usb_endpoint_descriptor *in_desc,
				struct usb_endpoint_descriptor *out_desc)
{
	struct usb_composite_dev *cdev = dev->cdev;
	struct usb_request *req;
	struct usb_ep *ep;
	int i;

	DBG(cdev, "create_bulk_endpoints dev: %p\n", dev);

	ep = usb_ep_autoconfig(cdev->gadget, in_desc);
	if (!ep) {
		DBG(cdev, "usb_ep_autoconfig for ep_in failed\n");
		return -ENODEV;
	}
	DBG(cdev, "usb_ep_autoconfig for ep_in got %s\n", ep->name);
	ep->driver_data = dev;		/* claim the endpoint */
	dev->ep_in = ep;

	ep = usb_ep_autoconfig(cdev->gadget, out_desc);
	if (!ep) {
		DBG(cdev, "usb_ep_autoconfig for ep_out failed\n");
		return -ENODEV;
	}
	DBG(cdev, "usb_ep_autoconfig for adb ep_out got %s\n", ep->name);
	ep->driver_data = dev;		/* claim the endpoint */
	dev->ep_out = ep;

	/* now allocate requests for our endpoints */
	req = adb_request_new(dev->ep_out, ADB_BULK_BUFFER_SIZE);
	if (!req)
		goto fail;
	req->complete = adb_complete_out;
	dev->rx_req = req;

	for (i = 0; i < TX_REQ_MAX; i++) {
		req = adb_request_new(dev->ep_in, ADB_BULK_BUFFER_SIZE);
		if (!req)
			goto fail;
		req->complete = adb_complete_in;
		adb_req_put(dev, &dev->tx_idle, req);
	}

	return 0;

fail:
	printk(KERN_ERR "adb_bind() could not allocate requests\n");
	return -1;
}

static ssize_t adb_read(struct file *fp, char __user *buf,
				size_t count, loff_t *pos)
{
	struct adb_dev *dev = fp->private_data;
	struct usb_request *req;
	int r = count, xfer;
	int ret;

	USBSTACK_DBG("adb_read   enter(%d)\n", count);
	if (!_adb_dev){
		USBSTACK_DBG("adb_read: _adb_dev is NULL\n");
		return -ENODEV;
	}
	if (count > ADB_BULK_BUFFER_SIZE)
		return -EINVAL;

	if (adb_lock(&dev->read_excl)){
		USBSTACK_DBG("adb_read: read_excl lock fail, \n");
		return -EBUSY;
	}
	/* we will block until we're online */
	while (!(dev->online || dev->error)) {
		USBSTACK_DBG("adb_read: waiting for online state\n");
		ret = wait_event_interruptible(dev->read_wq,
				(dev->online || dev->error));
		if (ret < 0) {
			adb_unlock(&dev->read_excl);
			return ret;
		}
	}
	if (dev->error) {
		r = -EIO;
		USBSTACK_DBG("adb_read block fail ret:%d", r);
		goto done;
	}
#ifdef _USE_VEHICLE_DC	
	if(atomic_read(&dev->agt_read_flag)){
		USBSTACK_DBG("adb_read wait for agt_read_wq\n");
		
		req = dev->rx_req;
		ret = wait_event_interruptible(dev->agt_read_wq, dev->rx_done);
		if (ret < 0) {
			if (ret != -ERESTARTSYS)
				dev->error = 1;
			USBSTACK_DBG("adb_read wait agt_read_wq fail ret: 0x%x", ret);
			goto done;
		}
		
		USBSTACK_DBG("adb_read after agt_read_wq rx %p len:%d\n", req, req->actual);
		xfer = (req->actual < count) ? req->actual : count;
		
		if (copy_to_user(buf, req->buf, xfer))
			r = -EFAULT;
		goto done;			
	}
#endif	

requeue_req:
	if(!dev->online){
		USBSTACK_DBG("adb_read dev is offline\n");
		r = -EIO;
		goto done;
	}
	/* queue a request */
	req = dev->rx_req;
	req->length = count;
	dev->rx_done = 0;

		USBSTACK_DBG("adb_read waiting usb_ep_queue \n");
	ret = usb_ep_queue(dev->ep_out, req, GFP_ATOMIC);
	if (ret < 0) {
		pr_debug("adb_read: failed to queue req %p (%d)\n", req, ret);
		r = -EIO;
		dev->error = 1;
		USBSTACK_DBG("adb_read ep-queue fail ret:%d", r);
		goto done;
	} else {
		pr_debug("rx %p queue\n", req);
	}

#ifdef _USE_VEHICLE_DC
	atomic_set(&dev->adb_read_flag, 1);
#endif
		USBSTACK_DBG("adb_read waiting for read_wq, rx_done:%d\n", dev->rx_done);
	/* wait for a request to complete */
	ret = wait_event_interruptible(dev->read_wq, dev->rx_done);
	if (ret < 0) {
		if (ret != -ERESTARTSYS)
			dev->error = 1;
		r = ret;
		usb_ep_dequeue(dev->ep_out, req);
		USBSTACK_DBG("adb_read stop ret: 0x%x", ret);
		goto done;
	}

#ifdef _USE_VEHICLE_DC	
	atomic_set(&dev->adb_read_flag, 0);	
#endif	
	if (!dev->error) {
		if(!dev->online){
			USBSTACK_DBG("adb_read dev is offline cannot requeue req\n");
			r = -EIO;
			goto done;
		}
		/* If we got a 0-len packet, throw it back and try again. */
		if (req->actual == 0
#ifdef _USE_VEHICLE_DC			
			&& _adb_dev->agent_start == 0
#endif			
			){
			USBSTACK_DBG("adb_read actual == 0, agent_start == 0, requeue\n");
			
			goto requeue_req;
		}
		USBSTACK_DBG("adb_read rx %p len:%d\n", req, req->actual);
		
		xfer = (req->actual < count) ? req->actual : count;
#ifdef _USE_VEHICLE_DC
		r = xfer;
		if(_adb_dev->agent_start && req->actual){
			adb_rpmsg_write2ap(req->buf, xfer);
			goto done;
		}
#endif
		if (copy_to_user(buf, req->buf, xfer))
			r = -EFAULT;

	} else{
		USBSTACK_DBG("adb_read fail %d", r);
		r = -EIO;
	}

done:
	adb_unlock(&dev->read_excl);
	USBSTACK_DBG("adb_read returning %d\n", r);
	return r;
}
#ifdef CONFIG_PM
unsigned int g_dbg_adb_times = 0;
#endif
static ssize_t adb_write(struct file *fp, const char __user *buf,
				 size_t count, loff_t *pos)
{
	struct adb_dev *dev = fp->private_data;
	struct usb_request *req = 0;
	int r = count, xfer;
	int ret;

	if (!_adb_dev)
		return -ENODEV;
	//pr_debug("adb_write(%d)\n", count);
	USBSTACK_DBG("adb write begin");

#ifdef CONFIG_PM
	if (dev->suspend_state == 1){
		g_dbg_adb_times++;
		usb_printk("%s, %u portname:%s, wrtime:%d\n", __func__, __LINE__, dev->function.name, g_dbg_adb_times);
		USBSTACK_DBG("%s, %u portname:%s, wrtime:%d", __func__, __LINE__, dev->function.name, g_dbg_adb_times);
		return -EBUSY;
#if 0
        usb_gadget_wakeup(dev->cdev->gadget);
		do{
			msleep(2);
		}while(dev->suspend_state==1);
#endif
	}
#endif
	if (adb_lock(&dev->write_excl))
		return -EBUSY;
	while (count > 0) {
		if (dev->error) {
			pr_debug("adb_write dev->error\n");
			r = -EIO;
			break;
		}

		/* get an idle tx request to use */
		req = 0;
		ret = wait_event_interruptible(dev->write_wq,
			(req = adb_req_get(dev, &dev->tx_idle)) || dev->error);

		if (ret < 0) {
			r = ret;
			break;
		}
		if(dev->error){
			USBSTACK_DBG("get an err, break write\n");
			r = -EIO;
			break;			
		}

		if (req != 0) {
			if (count > ADB_BULK_BUFFER_SIZE)
				xfer = ADB_BULK_BUFFER_SIZE;
			else
				xfer = count;
			if (copy_from_user(req->buf, buf, xfer)) {
				r = -EFAULT;
				break;
			}

			req->length = xfer;
			ret = usb_ep_queue(dev->ep_in, req, GFP_ATOMIC);
			if (ret < 0) {
				pr_debug("adb_write: xfer error %d\n", ret);
				dev->error = 1;
				r = -EIO;
				break;
			}

			buf += xfer;
			count -= xfer;

			/* zero this so we don't try to free it on error exit */
			req = 0;
		}
	}

	if (req)
		adb_req_put(dev, &dev->tx_idle, req);

	adb_unlock(&dev->write_excl);
	pr_debug("adb_write returning %d\n", r);
	return r;
}

static int adb_open(struct inode *ip, struct file *fp)
{
	//pr_info("adb_open\n");
	if (!_adb_dev)
		return -ENODEV;
	printk("\n adb_open----enter\n");
#ifdef _USE_VEHICLE_DC	
	if(_adb_dev->agent_state){
		printk("\n adb_open,now is agent mode ,return\n");
		return -ENODEV;
	}
	if(usb_get_adb_agent() == 1){
		//delay time is num of jiffies
		printk("---adb_open, switch to cap\n");
		schedule_delayed_work(&_adb_dev->agent_switch_work, 200);
	}
#endif
	USBSTACK_DBG("%s", __func__);
	if (adb_lock(&_adb_dev->open_excl)){
		printk("\n adb_open,open_excl:%d\n", atomic_read(&_adb_dev->open_excl));
		return -EBUSY;
	}
	fp->private_data = _adb_dev;

	/* clear the error latch */
	_adb_dev->error = 0;

	//adb_ready_callback();

	if(atomic_read(&_adb_dev->enable_excl)){
		_adb_dev->online = 1;
		wake_up(&_adb_dev->read_wq);
	}else{
		adb_unlock(&_adb_dev->open_excl);
		return -ENODEV;
	}
	usb_record_dbginfo(USB_DEV_OPEN, 0, 3);

	printk("\n adb_open--ok--exit,online:%d\n", _adb_dev->online);
	return 0;
}

static int adb_release(struct inode *ip, struct file *fp)
{
	int ret;
	//pr_info("adb_release\n");
	USBSTACK_DBG("%s", __func__);
	printk("\n adb_release----enter, read_excl:%d\n", atomic_read(&_adb_dev->read_excl));
	//adb_closed_callback();

	_adb_dev->error = 1;
	if(list_empty(&_adb_dev->tx_idle) && atomic_read(&_adb_dev->write_excl)){
		USBSTACK_DBG("adb_release, disable and reenable endpoint\n");
		ret = usb_ep_disable(_adb_dev->ep_in);
		if(ret){
			printk("adb_release, usb_ep_disable fail,%d\n", ret);
			WARN_ON(1);
		}
		ret = usb_ep_enable(_adb_dev->ep_in);
		if(ret){
			printk("adb_release, usb_ep_enable fail,%d\n", ret);
			WARN_ON(1);
		}		
	}
	if(atomic_read(&_adb_dev->read_excl)){	
	//if(_adb_dev->ep_out){
		printk("adb_release, disable and reenable ep_out\n");
		ret = usb_ep_disable(_adb_dev->ep_out);
		if(ret){
			printk("adb_release, usb_ep_disable fail,%d\n", ret);
			WARN_ON(1);
		}
		ret = usb_ep_enable(_adb_dev->ep_out);
		if(ret){
			printk("adb_release, usb_ep_enable fail,%d\n", ret);
			WARN_ON(1);
		}

	}
	if(atomic_read(&_adb_dev->read_excl) || atomic_read(&_adb_dev->open_excl))
		adb_unlock(&_adb_dev->open_excl);
	
#ifdef _USE_VEHICLE_DC	
	if(_adb_dev->agent_state){
		wake_up(&_adb_dev->agt_start_wait);
		printk("\n now is agent mode ,switch\n");
		
	}
#endif	

	usb_record_dbginfo(USB_DEV_CLOSE, 0, 3);
	return 0;
}

/* file operations for ADB device /dev/android_adb */
static const struct file_operations adb_fops = {
	.owner = THIS_MODULE,
	.read = adb_read,
	.write = adb_write,
	.open = adb_open,
	.release = adb_release,
};

static struct miscdevice adb_device = {
	.minor = MISC_DYNAMIC_MINOR,
	.name = adb_shortname,
	.fops = &adb_fops,
};




static int
adb_function_bind(struct usb_configuration *c, struct usb_function *f)
{
	struct usb_composite_dev *cdev = c->cdev;
	struct adb_dev	*dev = func_to_adb(f);
	int			id;
	int			ret;
	USBSTACK_DBG("%s", __func__);
	dev->cdev = cdev;
	DBG(cdev, "adb_function_bind dev: %p\n", dev);

	/* allocate interface ID(s) */
	id = usb_interface_id(c, f);
	if (id < 0)
		return id;
	adb_interface_desc.bInterfaceNumber = id;

	/* allocate endpoints */
	ret = adb_create_bulk_endpoints(dev, &adb_fullspeed_in_desc,
			&adb_fullspeed_out_desc);
	if (ret)
		return ret;

	/* support high speed hardware */
	if (gadget_is_dualspeed(c->cdev->gadget)) {
		adb_highspeed_in_desc.bEndpointAddress =
			adb_fullspeed_in_desc.bEndpointAddress;
		adb_highspeed_out_desc.bEndpointAddress =
			adb_fullspeed_out_desc.bEndpointAddress;
	}

	DBG(cdev, "%s speed %s: IN/%s, OUT/%s\n",
			gadget_is_dualspeed(c->cdev->gadget) ? "dual" : "full",
			f->name, dev->ep_in->name, dev->ep_out->name);
	return 0;
}

static void
adb_function_unbind(struct usb_configuration *c, struct usb_function *f)
{
	struct adb_dev	*dev = func_to_adb(f);
	struct usb_request *req;

	USBSTACK_DBG("%s", __func__);
	dev->online = 0;
	dev->error = 1;

	wake_up(&dev->read_wq);
#ifdef _USE_VEHICLE_DC
	if(atomic_read(&dev->agt_write_flag)){
		printk("adb_function_unbind, agt_write_flag set, wakeup write_wq\n");
		atomic_set(&dev->write_busy, 0);
		wake_up(&dev->write_wq);
		dev->set_test |= 2;
	}
#endif	
	adb_request_free(dev->rx_req, dev->ep_out);
	dev->rx_req = NULL;
	while ((req = adb_req_get(dev, &dev->tx_idle)))
		adb_request_free(req, dev->ep_in);
}

static int adb_function_set_alt(struct usb_function *f,
		unsigned intf, unsigned alt)
{
	struct adb_dev	*dev = func_to_adb(f);
	struct usb_composite_dev *cdev = f->config->cdev;
	int ret;

	DBG(cdev, "adb_function_set_alt intf: %d alt: %d\n", intf, alt);
	USBSTACK_DBG("%s", __func__);
	#ifdef CONFIG_PM
	if(dev->suspend_state == 1)
		dev->suspend_state = 0;
	#endif
	ret = config_ep_by_speed(cdev->gadget, f, dev->ep_in);
	if (ret)
		return ret;

	ret = usb_ep_enable(dev->ep_in);
	if (ret)
		return ret;

	ret = config_ep_by_speed(cdev->gadget, f, dev->ep_out);
	if (ret)
		return ret;

	ret = usb_ep_enable(dev->ep_out);
	if (ret) {
		usb_ep_disable(dev->ep_in);
		return ret;
	}
	
	dev->adb_ready = 1;
	
#ifdef _USE_VEHICLE_DC	
	//maybe need clear adb server state
	adb_agent_close();	
	if(usb_get_adb_agent() == 1){
		//delay time is num of jiffies
		printk("---adb_function_set_alt, switch to cap\n");
		dev->online = 1;
		schedule_delayed_work(&_adb_dev->agent_switch_work, 0);
	}else
#endif
	schedule_work(&dev->online_inform);

	if(adb_lock(&dev->enable_excl))
		return 0;
		
	if(atomic_read(&dev->open_excl)==1){
		printk("adb_function_set_alt, adb open, and set online \n");
		dev->online = 1;
		dev->error= 0;		
		/* readers may be blocked waiting for us to go online */
		wake_up(&dev->read_wq);
	}
	return 0;
}

static void adb_function_disable(struct usb_function *f)
{
	struct adb_dev	*dev = func_to_adb(f);
	struct usb_composite_dev	*cdev = dev->cdev;

	USBSTACK_DBG("%s", __func__);

	dev->adb_ready = 0;
	if(atomic_read(&dev->enable_excl)==0)
		return;
	else
		adb_unlock(&dev->enable_excl);

#if 0
	if(dev->suspend_state == 1)
		return;
#endif

	DBG(cdev, "adb_function_disable cdev %p\n", cdev);
	dev->online = 0;
	//dev->error = 1;
	usb_ep_disable(dev->ep_in);
	usb_ep_disable(dev->ep_out);


#ifdef _USE_VEHICLE_DC
	printk("adb_function_disable, notify adb offline\n");

	dev->agent_start = 0;
	dev->agent_state = 0;
	adb_server_plug_notify(USB_RPMSG_NOTIFY_ADB_OFFLINE);
	if(atomic_read(&dev->write_busy) == 1){
	printk("adb_function_disable, clear agt_write_wq\n");
		atomic_set(&dev->write_busy, 0);
			printk("adb_function_disable, clear write_wq\n");
			dev->agt_error = 1;
			wake_up(&dev->write_wq);
			
		wake_up(&dev->agt_write_wq);
		dev->set_test |= 1;
	}
	dev->agt_error = 0;		
#endif	
	
	schedule_work(&dev->offline_inform);
	/* readers may be blocked waiting for us to go online */
	wake_up(&dev->read_wq);

	VDBG(cdev, "%s disabled\n", dev->function.name);
}
#ifdef CONFIG_PM
unsigned int g_adb_suspend_cnt = 0;
unsigned int g_adb_resume_cnt = 0;
static void adb_function_suspend(struct usb_function *f)
{
	struct adb_dev	*dev = func_to_adb(f);
	struct usb_composite_dev	*cdev = dev->cdev;
    g_adb_suspend_cnt++;
	usb_printk("%s, %u, suspendcnt:%d\n", __func__, __LINE__, g_adb_suspend_cnt);
    USBSTACK_DBG("%s, %u suspendcnt:%d", __func__, __LINE__, g_adb_suspend_cnt);
	dev->suspend_state = 1;
//	adb_function_disable(f);
	usb_ep_disable(dev->ep_in);
	usb_record_dbginfo(USB_DEV_SUSPEND, 0, 3);
//	usb_ep_disable(dev->ep_out);
}
static void adb_function_resume(struct usb_function *f)
{
	struct adb_dev	*dev = func_to_adb(f);
	struct usb_composite_dev	*cdev = dev->cdev;
    g_adb_resume_cnt++;
	usb_printk("%s, %u, resumecnt:%d\n", __func__, __LINE__, g_adb_resume_cnt);
    USBSTACK_DBG("%s, %u resumecnt:%d", __func__, __LINE__, g_adb_resume_cnt);
//	adb_function_set_alt(f, NULL, NULL);
	//usb_ep_enable(dev->ep_in);
	usb_ep_resume_enable(dev->ep_in);
//	usb_ep_enable(dev->ep_out);
	dev->suspend_state = 0;
	usb_record_dbginfo(USB_DEV_RESUME, 0, 3);
}
#endif
static int adb_bind_config(struct usb_configuration *c)
{
	struct adb_dev *dev = _adb_dev;

	usb_printk("adb_bind_config\n");

	dev->cdev = c->cdev;
	dev->function.name = "adb";
	dev->function.descriptors = fs_adb_descs;
	dev->function.hs_descriptors = hs_adb_descs;
	dev->function.bind = adb_function_bind;
	dev->function.unbind = adb_function_unbind;
	dev->function.set_alt = adb_function_set_alt;
	dev->function.disable = adb_function_disable;
#ifdef CONFIG_PM
    dev->function.suspend = adb_function_suspend;
    dev->function.resume = adb_function_resume;
#endif
	return usb_add_function(c, &dev->function);
}


void adb_online_inform_work(struct work_struct *work)
{
	struct adb_dev	*dev = container_of(work, struct adb_dev, online_inform);
	printk("adb_online_inform_work\n");
	kobject_uevent(&adb_device.this_device->kobj, KOBJ_ONLINE);
	usb_record_dbginfo(USB_DEV_ONLINE, 0, 3);
}

void adb_offline_inform_work(struct work_struct *work)
{
	struct adb_dev	*dev = container_of(work, struct adb_dev, offline_inform);
	printk("adb_offline_inform_work\n");

	kobject_uevent(&adb_device.this_device->kobj,  KOBJ_OFFLINE);
	usb_record_dbginfo(USB_DEV_OFFLINE, 0, 3);
}



#ifdef _USE_VEHICLE_DC
static ssize_t agent_state_show(struct device *_dev,
				struct device_attribute *attr, char *buf)
{
	if(_adb_dev){	
		return sprintf(buf, "\n agent_state = %d\n",_adb_dev->agent_state);
	}
	
	return sprintf(buf, "\n _adb_dev is NULL agent_state = NULL\n");
}

static ssize_t agent_state_store(struct device *_dev,
				 struct device_attribute *attr,
				 const char *buf, size_t count)
{
 	int flag = 0;
	flag = simple_strtoul(buf, NULL, 16);
	if(_adb_dev){
		if(_adb_dev->agent_state == flag){
			printk("agent_state_store, already in this state,don't change again\n");
			return count;
		}
		_adb_dev->agent_state = (flag ? 1 : 0);
		atomic_set(&_adb_dev->agent_switch, 1);
		wake_up(&_adb_dev->agent_monitor_wq);
	}

	return count;
}
DEVICE_ATTR(agent_state, S_IRUGO | S_IWUSR, agent_state_show,
	    agent_state_store);
#endif
static int adb_setup(void)
{
	struct adb_dev *dev;
	int ret;

	dev = kzalloc(sizeof(*dev), GFP_KERNEL);
	if (!dev)
		return -ENOMEM;

	spin_lock_init(&dev->lock);

	init_waitqueue_head(&dev->read_wq);
	init_waitqueue_head(&dev->write_wq);

	atomic_set(&dev->open_excl, 0);
	atomic_set(&dev->read_excl, 0);
	atomic_set(&dev->write_excl, 0);
	atomic_set(&dev->enable_excl, 0);
	
	INIT_WORK(&dev->online_inform, adb_online_inform_work);
	INIT_WORK(&dev->offline_inform, adb_offline_inform_work);
	INIT_LIST_HEAD(&dev->tx_idle);

	_adb_dev = dev;

	ret = misc_register(&adb_device);
	if (ret)
		goto err;

#ifdef _USE_VEHICLE_DC
	ret = device_create_file(adb_device.this_device, &dev_attr_agent_state);
	if(ret){
		printk(KERN_ERR "adb create attr_agent_state fail\n");
	}
	ret = adb_server_init(dev);
#endif	
	return 0;

err:
	kfree(dev);
	_adb_dev = NULL;
	printk(KERN_ERR "adb gadget driver failed to initialize\n");
	return ret;
}

static void adb_cleanup(void)
{

	flush_work_sync(&_adb_dev->online_inform);
	flush_work_sync(&_adb_dev->offline_inform);
#ifdef _USE_VEHICLE_DC
	adb_server_release();
#endif
	misc_deregister(&adb_device);

	kfree(_adb_dev);
	_adb_dev = NULL;
}
