/*
 * u_diag.c -- Debuglog-over-USB link layer utilities for Gadget stack
 *
 * Copyright (C) 2019 ZTE Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/export.h>
	 
#include "u_diag.h"


#define DIAG_MAX_NUM	1
#define DIAG_TX_QUEUE_SIZE		16
#define RX_QUEUE_SIZE		3

#define DIAG_TX_DIRECTION 0
#define DIAG_RX_DIRECTION 1
#define DIAG_RX_MAX_PACKET_LEN 2048

char g_diag_flag = 0;
EXPORT_SYMBOL(g_diag_flag);

//#define DIAG_DRV_TEST

/*
 * The port structure holds info for each port, one for each minor number
 * (and thus for each /dev/ node).
 */
struct diag_port {
	struct mutex	lock;			/* protect open/close */
	spinlock_t		port_lock;	/* guard port_* access */

	struct u_diag		*port_usb;
	u8			port_num;

	//struct list_head	read_pool;
	struct list_head	write_pool;
	struct usb_request *rx_req;

	int write_started;
	int write_allocated;

	int connected;

	usb_diag_rx_complete_callback rx_complete_callback;
	usb_diag_tx_complete_callback tx_complete_callback;
};

struct diag_ports{
	//u8 num;
	struct diag_port * diag_port[DIAG_MAX_NUM];
};

static struct diag_ports n_diag_ports = {0};
static int n_diag_num = 0;
static unsigned	n_ports;

#ifdef DIAG_DRV_TEST
struct semaphore diag_test_sem;
#endif

static unsigned diag_start_rx(struct diag_port *port);

static void diag_read_complete(struct usb_ep *ep, struct usb_request *req)
{
	struct diag_port	*port = ep->driver_data;
	bool			disconnect = false;

	/* Queue all received data until the tty layer is ready for it. */
	spin_lock(&port->port_lock);
	/*callback func*/
	if(req->status == -ESHUTDOWN)
		disconnect = true;
	if(req->actual){
		if(port->rx_complete_callback){ 
		/*if req->length==0, maybe rx stopped by ourself, we should not tell apps*/
			port->rx_complete_callback(req->buf, req->actual);
			//req->buf will be freed in rx_complete_callback
			req->buf = NULL;	
		}
	}
	
	if(!disconnect && port->connected)
		diag_start_rx(port);
	
	spin_unlock(&port->port_lock);
}

volatile unsigned int u_diag_xfer_cnt = 0;
volatile unsigned int u_diag_xfer_success_cnt = 0;
volatile unsigned int u_diag_xfer_complete_cnt = 0;
volatile unsigned int u_diag_xfer_complete_success_cnt = 0;
static void diag_write_complete(struct usb_ep *ep, struct usb_request *req)
{
	struct diag_port	*port = ep->driver_data;

	spin_lock(&port->port_lock);
	//list_add(&req->list, &port->write_pool);
	list_add_tail(&req->list, &port->write_pool);
	port->write_started--;

	u_diag_xfer_complete_cnt++;

	switch (req->status) {
	default:
		/* presumably a transient fault */
		pr_warning("%s: unexpected %s status %d\n",
				__func__, ep->name, req->status);
		/* FALL THROUGH */
	case 0:
		/* normal completion */
		//gs_start_tx(port);
		if(port->tx_complete_callback) {
			u_diag_xfer_complete_success_cnt++;
			port->tx_complete_callback(req->buf);
		}
		break;

	case -ESHUTDOWN:
		/* disconnect */
		pr_vdebug("%s: %s shutdown\n", __func__, ep->name);
		break;
	}

	spin_unlock(&port->port_lock);
}
/*
 * gdiag_free_req
 *
 * Free a usb_request and its buffer.
 */
void gdiag_free_req(struct usb_ep *ep, struct usb_request *req)
{
	if(req->buf) {
		kfree(req->buf);
		req->buf = NULL;
	}
	usb_ep_free_request(ep, req);
}
EXPORT_SYMBOL_GPL(gdiag_free_req);

/* I/O glue between TTY (upper) and USB function (lower) driver layers */

/*
 * gdiag_alloc_req
 *
 * Allocate a usb_request and its buffer.  Returns a pointer to the
 * usb_request or NULL if there is an error.
 */
struct usb_request *
gdiag_alloc_req(struct usb_ep *ep, unsigned len, gfp_t kmalloc_flags)
{
	struct usb_request *req;

	req = usb_ep_alloc_request(ep, kmalloc_flags);

	if (req != NULL) {
		if(len > 0){
			req->length = len;
			req->buf = kmalloc(len, kmalloc_flags);
			if (req->buf == NULL) {
				printk("gdiag_alloc_req,fail, no mem\n");
				
				usb_ep_free_request(ep, req);
				return NULL;
			}
		}
	}

	return req;
}
EXPORT_SYMBOL_GPL(gdiag_alloc_req);

static void diag_free_requests(struct usb_ep *ep, struct list_head *head,
							 int *allocated)
{
	struct usb_request	*req;

	while (!list_empty(head)) {
		req = list_entry(head->next, struct usb_request, list);
		list_del(&req->list);
		usb_ep_free_request(ep, req);
		if (allocated)
			(*allocated)--;
	}
}

static int diag_alloc_requests(struct usb_ep *ep, struct list_head *head,
		void (*fn)(struct usb_ep *, struct usb_request *),
		int *allocated)
{
	int			i;
	struct usb_request	*req;
	int n = 0;
#if 0
	if(direction == DIAG_TX_DIRECTION)
		n = allocated ? TX_QUEUE_SIZE - *allocated : TX_QUEUE_SIZE;
	else
		n = allocated ? RX_QUEUE_SIZE - *allocated : RX_QUEUE_SIZE;
#else
	n = allocated ? DIAG_TX_QUEUE_SIZE - *allocated : DIAG_TX_QUEUE_SIZE;
#endif

	/* Pre-allocate up to QUEUE_SIZE transfers, but if we can't
	 * do quite that many this time, don't fail ... we just won't
	 * be as speedy as we might otherwise be.
	 */
	for (i = 0; i < n; i++) {
		req = usb_ep_alloc_request(ep, GFP_ATOMIC);
#if 0
		if(direction == DIAG_RX_DIRECTION) {
			if (req != NULL) {
				req->length = DIAG_RX_MAX_PACKET_LEN;
				req->buf = kmalloc(DIAG_RX_MAX_PACKET_LEN, GFP_ATOMIC);
				if (req->buf == NULL) {
					usb_ep_free_request(ep, req);
					return NULL;
				}
			}
		}
#endif
		if (!req)
			return list_empty(head) ? -ENOMEM : 0;
		req->complete = fn;
		list_add_tail(&req->list, head);
		if (allocated)
			(*allocated)++;
	}
	return 0;
}

/*
 * Context: caller owns port_lock, and port_usb is set
 */
static unsigned diag_start_rx(struct diag_port *port)
/*
__releases(&port->port_lock)
__acquires(&port->port_lock)
*/
{
#if 0//def RX_QUEUE_USED
	struct list_head	*pool = &port->read_pool;
	struct usb_ep		*out = port->port_usb->out;

	while (!list_empty(pool)) {
		struct usb_request	*req;
		int			status;

		if (port->read_started >= QUEUE_SIZE)
			break;

		req = list_entry(pool->next, struct usb_request, list);
		list_del(&req->list);
		//req->length = out->maxpacket;
		req->length = DIAG_RX_MAX_PACKET_LEN;
		req->buf = kmalloc(DIAG_RX_MAX_PACKET_LEN, GFP_ATOMIC);

		/* drop lock while we call out; the controller driver
		 * may need to call us back (e.g. for disconnect)
		 */
		spin_unlock(&port->port_lock);
		status = usb_ep_queue(out, req, GFP_ATOMIC);
		spin_lock(&port->port_lock);

		if (status) {
			pr_debug("%s: %s %s err %d\n",
					__func__, "queue", out->name, status);
			list_add(&req->list, pool);
			break;
		}
		port->read_started++;

		/* abort immediately after disconnect */
		if (!port->port_usb)
			break;
	}
	return port->read_started;
#else
	struct usb_ep		*out = NULL;
	struct usb_request	*req = port->rx_req;
	int			status;
	if(NULL ==port->port_usb)
        return 0;        
		out = port->port_usb->out;
	if (req == NULL) {
		port->rx_req = usb_ep_alloc_request(out, GFP_ATOMIC);
		if(port->rx_req == NULL)
			BUG_ON(1);
		req = port->rx_req; 
		req->length = DIAG_RX_MAX_PACKET_LEN;
		req->complete = diag_read_complete;
	}

	//GFP_KERNEL can cause mem recovery
	if(req->buf == NULL)
		req->buf = kzalloc(DIAG_RX_MAX_PACKET_LEN, GFP_KERNEL);

	if(req->buf == NULL){
		usb_ep_free_request(out, req);
		BUG_ON(1);
	}

	spin_unlock(&port->port_lock);
	status = usb_ep_queue(out, req, GFP_ATOMIC);
	spin_lock(&port->port_lock);
#endif
	return 1;
}


static int diag_start_io(struct diag_port *port)
{
	//struct list_head	*head = &port->read_pool;
	//struct usb_ep		*ep = port->port_usb->out;
	int			status;
	unsigned		started;

	/* Allocate RX and TX I/O buffers.  We can't easily do this much
	 * earlier (with GFP_KERNEL) because the requests are coupled to
	 * endpoints, as are the packet sizes we'll be using.  Different
	 * configurations may use different endpoints with a given port;
	 * and high speed vs full speed changes packet sizes too.
	 */
#if 0
	status = diag_alloc_requests(port->port_usb->out, &port->read_pool, 
			diag_read_complete, &port->read_allocated);
	if (status)
		return status;
#endif

	status = diag_alloc_requests(port->port_usb->in, &port->write_pool,
			diag_write_complete, &port->write_allocated);

	if (status) {
		diag_free_requests(port->port_usb->in, &port->write_pool,
			&port->write_allocated);
		return status;
	}

	started = diag_start_rx(port);

	/* unblock any pending writes into our circular buffer */
	if (!started) {
#if 0
		diag_free_requests(port->port_usb->out, &port->read_pool, 
			&port->read_allocated);
#endif
		diag_free_requests(port->port_usb->in, &port->write_pool,
			&port->write_allocated);
		status = -EIO;
	}

	return status;
}

static int
diag_port_alloc(unsigned port_num)
{
	struct diag_port	*port;

	port = kzalloc(sizeof(struct diag_port), GFP_KERNEL);
	if (port == NULL)
		return -ENOMEM;

	spin_lock_init(&port->port_lock);

	//INIT_LIST_HEAD(&port->read_pool);
	//INIT_LIST_HEAD(&port->read_queue);
	INIT_LIST_HEAD(&port->write_pool);

	port->port_num = port_num;

	n_diag_ports.diag_port[port_num] = port;

	return 0;
}


int diag_do_xfer(void *buf, unsigned int len)
{
	struct diag_port	*port = n_diag_ports.diag_port[n_diag_num];
	struct list_head	*pool = &port->write_pool;
	struct usb_ep		*in = NULL;
	int			status = 0;

	u_diag_xfer_cnt++;
	
	if(!port->connected)
		return -1;
	if(port->port_usb == NULL)
		return -1;
	#ifdef CONFIG_PM
	if(port->port_usb->suspend_state == 1)
		return -1;
	#endif
	spin_lock(&port->port_lock);
	if(port->port_usb)
		in = port->port_usb->in;

	if (!list_empty(&port->write_pool)) {
		struct usb_request	*req;

		if (port->write_started >= DIAG_TX_QUEUE_SIZE) {
			spin_unlock(&port->port_lock);
			return -1;
		}
		if((pool->next == NULL) || (pool->next == &port->write_pool)){
			spin_unlock(&port->port_lock);
			return -1;
		}
		
		req = container_of(pool->next, struct usb_request, list);

		req->buf = buf;
		req->length = len;
		
		req->zero = ((len % in->maxpacket) == 0) ? 1: 0;

		list_del_init(&req->list);

		spin_unlock(&port->port_lock);
		status = usb_ep_queue(in, req, GFP_ATOMIC);
        spin_lock(&port->port_lock);

		if (status) {
			pr_debug("%s: %s %s err %d\n",
					__func__, "queue", in->name, status);
			if(port->port_usb && port->connected)
				list_add(&req->list, &port->write_pool);
			else
				usb_ep_free_request(in, req);
			spin_unlock(&port->port_lock);
			return -1;
		}

		port->write_started++;

	} else {
		spin_unlock(&port->port_lock);
		return -1;
	}
	
	spin_unlock(&port->port_lock);

	u_diag_xfer_success_cnt++;

	return 0;
}
EXPORT_SYMBOL(diag_do_xfer);

int diag_callback_register(usb_diag_rx_complete_callback rx_callback, 
							usb_diag_tx_complete_callback tx_callback)
{
	struct diag_port	*port = n_diag_ports.diag_port[n_diag_num];
	if(port == NULL)
		return -1;
	port->rx_complete_callback = rx_callback;
	port->tx_complete_callback = tx_callback;

	return 0;
}
EXPORT_SYMBOL(diag_callback_register);

//test
#ifdef DIAG_DRV_TEST
volatile unsigned int g_test_rx_cb_cnt = 0;
volatile char g_test_rx_buffer[50][50] = {0};
volatile char g_test_rx_len[50] = {0};
void test_rx_callback(void *buf, unsigned int len)
{
	//printk("test_rx_callback get data.\n");
	memcpy(g_test_rx_buffer[g_test_rx_cb_cnt%50], buf, 50);
	g_test_rx_len[g_test_rx_cb_cnt%50] = len;
	g_test_rx_cb_cnt++;
	kfree(buf);

	up(&diag_test_sem);
}

volatile unsigned int g_test_tx_cb_cnt = 0;
void test_tx_callback(void *buf)
{
	g_test_tx_cb_cnt++;
	kfree(buf);
}

volatile unsigned int g_test_thread_cnt = 0;
int32_t test_tx_thread(unsigned long data)
{
	struct sched_param param = { .sched_priority = 1 };

	void *buf1 = NULL;

	param.sched_priority = 15;
	
	sched_setscheduler(current, SCHED_FIFO, &param);

	while (!kthread_should_stop()){
		down(&diag_test_sem);
		g_test_thread_cnt ++;
		buf1 = kzalloc(1024, GFP_ATOMIC);
		memset(buf1, g_test_rx_cb_cnt, 1024);
		diag_do_xfer(buf1, 1000);
	}
}
#endif
void diag_connect_ext(struct u_diag *diag)
{
	struct diag_port	*port = diag->ioport;
	unsigned long	flags;

	spin_lock_irqsave(&port->port_lock, flags);

	port->connected = 1;
	g_diag_flag = 1;
	spin_unlock_irqrestore(&port->port_lock, flags);	

}

int diag_connect(struct u_diag *diag, u8 port_num)
{
	struct diag_port	*port = n_diag_ports.diag_port[port_num];
	unsigned long	flags;
	int		status;
	
	n_diag_num = port_num;

	/* activate the endpoints */
	status = usb_ep_enable(diag->in);
	if (status < 0)
		return status;
	diag->in->driver_data = port;

	status = usb_ep_enable(diag->out);
	if (status < 0)
		return status;
	diag->out->driver_data = port;
	
	spin_lock_irqsave(&port->port_lock, flags);
	diag->ioport = port;
	port->port_usb = diag;

	port->connected = 1;
	g_diag_flag = 1;
	diag_start_io(port);

#if 0
	/* if it's already open, start I/O ... and notify the serial
	 * protocol about open/close status (connect/disconnect).
	 */
	if (port->open_count) {
		diag_start_io(port);
		if (diag->connect)
			diag->connect(diag);
	} else {
		if (diag->disconnect)
			diag->disconnect(diag);
	}
#endif

#ifdef DIAG_DRV_TEST
	//test
	diag_callback_register(test_rx_callback, test_tx_callback);
	sema_init(&diag_test_sem, 0);
	kthread_run(test_tx_thread, NULL, "diag_test_thread/%s",
	"0");
#endif

	spin_unlock_irqrestore(&port->port_lock, flags);

	return status;
}

void diag_disconnect_ext(struct u_diag *diag)
{
	struct diag_port	*port = diag->ioport;
	unsigned long	flags;

	spin_lock_irqsave(&port->port_lock, flags);

	g_diag_flag = 0;
	port->connected = 0;

	spin_unlock_irqrestore(&port->port_lock, flags);	

}

int diag_disconnect(struct u_diag *diag)
{
	struct diag_port	*port = diag->ioport;
	unsigned long	flags;

	if (!port)
		return 0;
	
	g_diag_flag = 0;
	port->connected = 0;

	/* disable endpoints, aborting down any active I/O */
	usb_ep_disable(diag->out);
	diag->out->driver_data = NULL;

	usb_ep_disable(diag->in);
	diag->in->driver_data = NULL;

	spin_lock_irqsave(&port->port_lock, flags);

	/* REVISIT as above: how best to track this? */
	port->port_usb = NULL;
	diag->ioport = NULL;
	if(port->rx_req){
		if(port->rx_req->buf){
			kfree(port->rx_req->buf);
			port->rx_req->buf = NULL;
		}
		usb_ep_free_request(diag->out, port->rx_req);
		port->rx_req = NULL;
	}	
	/* finally, free any unused/unusable I/O buffers */
	diag_free_requests(diag->in, &port->write_pool,
			&port->write_allocated);

	port->write_allocated = port->write_started = 0;

	spin_unlock_irqrestore(&port->port_lock, flags);
    return 1;
}

int diag_setup(struct usb_gadget *g, unsigned count)
{
	unsigned			i;
	int				status;

	if (count == 0 || count > DIAG_MAX_NUM)
		return -EINVAL;

	/* make devices be openable */
	for (i = 0; i < count; i++) {
		status = diag_port_alloc(i);
		if (status) {
			count = i;
			goto fail;
		}
		mutex_init(&n_diag_ports.diag_port[i]->lock);
	}

	n_ports = count;

	return status;
fail:
	while (count--)
		kfree(n_diag_ports.diag_port[count]);
	return status;
}

/**
 * diag_cleanup - remove diag driver and devices
 * Context: may sleep
 *
 * This is called to free all resources allocated by @diag_setup().
 * Accordingly, it may need to wait until some open /dev/ files have
 * closed.
 *
 * The caller must have issued @diag_disconnect() for any ports
 * that had previously been connected, so that there is never any
 * I/O pending when it's called.
 */
void diag_cleanup(void)
{
	unsigned	i;
	struct diag_port	*port;

	for (i = 0; i < n_ports; i++) {
		/* prevent new opens */
		mutex_lock(&n_diag_ports.diag_port[i]->lock);
		port = n_diag_ports.diag_port[i];
		n_diag_ports.diag_port[i] = NULL;
		mutex_unlock(&n_diag_ports.diag_port[i]->lock);

		WARN_ON(port->port_usb != NULL);

		kfree(port);
	}
}


