/*
 * u_ether.c -- Ethernet-over-USB link layer utilities for Gadget stack
 *
 * Copyright (C) 2003-2005,2008 David Brownell
 * Copyright (C) 2003-2004 Robert Schwebel, Benedikt Spranger
 * Copyright (C) 2008 Nokia Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

/* #define VERBOSE_DEBUG */

#include <linux/kernel.h>
#include <linux/gfp.h>
#include <linux/device.h>
#include <linux/ctype.h>
#include <linux/etherdevice.h>
#include <linux/ethtool.h>
#include <net/SI/ext_mem.h>


#include "u_ether.h"
#include "multi_packet.h"
#include <mach/highspeed_debug.h>
#include <linux/nvro_comm.h>
//youchen@2024-06-20 add for lynq nv config begin
#include <lynq/lynq_ap_nv_cfg.h>
//youchen@2024-06-20 add for lynq nv config end

/*
 * This component encapsulates the Ethernet link glue needed to provide
 * one (!) network link through the USB gadget stack, normally "usb0".
 *
 * The control and data models are handled by the function driver which
 * connects to this code; such as CDC Ethernet (ECM or EEM),
 * "CDC Subset", or RNDIS.  That includes all descriptor and endpoint
 * management.
 *
 * Link level addressing is handled by this component using module
 * parameters; if no such parameters are provided, random link level
 * addresses are used.  Each end of the link uses one address.  The
 * host end address is exported in various ways, and is often recorded
 * in configuration databases.
 *
 * The driver which assembles each configuration using such a link is
 * responsible for ensuring that each configuration includes at most one
 * instance of is network link.  (The network layer provides ways for
 * this single "physical" link to be used by multiple virtual links.)
 */

#define UETH__VERSION	"29-May-2008"

#define ETH_MAX_NUM	4

extern int (*fast_from_driver)(struct sk_buff *skb, struct net_device* dev);

#if 0  //remove to u_ether.h
struct eth_dev {
	/* lock is held while accessing port_usb
	 * or updating its backlink port_usb->ioport
	 */
	spinlock_t		lock;
	struct gether		*port_usb;

	struct net_device	*net;
	struct usb_gadget	*gadget;

	spinlock_t		req_lock;	/* guard {rx,tx}_reqs */
	struct list_head	tx_reqs, rx_reqs;
	atomic_t		tx_qlen;

	struct sk_buff_head	rx_frames;

	unsigned		header_len;
	struct sk_buff		*(*wrap)(struct gether *, struct sk_buff *skb);
	int			(*unwrap)(struct gether *,
						struct sk_buff *skb,
						struct sk_buff_head *list);

	struct work_struct	work;

	struct work_struct inform_work;	 /* inform USB netCard state( online or offline ) */
	int		eth_state; /* 0: offline,  1:online */
	
	unsigned long		todo;
#define	WORK_RX_MEMORY		0

	bool			zlp;
	u8			host_mac[ETH_ALEN];
};
#endif

struct eth_devs{
	//u8 num;
	struct eth_dev * eth_dev[ETH_MAX_NUM];
};
unsigned long long g_test_xmit_pktnum = 0;
unsigned long long g_test_xmit_pkterr1 = 0;
unsigned long long g_test_xmit_pkterr2 = 0;
unsigned long long g_test_xmit_pkterr3 = 0;
unsigned long long g_test_rx_pkt = 0;
unsigned long long g_test_rx_complt_pkt = 0;

/*-------------------------------------------------------------------------*/

#define RX_EXTRA	20	/* bytes guarding against rx overflows */

#define DEFAULT_QLEN	2	/* double buffering by default */


#ifdef CONFIG_USB_GADGET_DUALSPEED

static unsigned qmult = 5;
module_param(qmult, uint, S_IRUGO|S_IWUSR);
MODULE_PARM_DESC(qmult, "queue length multiplier at high/super speed");

#else	/* full speed (low speed doesn't do bulk) */
//#define qmult		512
static unsigned int qmult = 512;
#define TX_QMULT    128//512
#endif

/* for dual-speed hardware, use deeper queues at high/super speed */
static inline int qlen(struct usb_gadget *gadget)
{
	USBSTACK_DBG("usb ether buf num:%d, skbnum:%d, usbSpeed:%d", qmult * DEFAULT_QLEN, gether_ether_skb_num(), gadget->speed);
#if 0
	if (gadget_is_dualspeed(gadget) && (gadget->speed == USB_SPEED_HIGH ||
					    gadget->speed == USB_SPEED_SUPER)){
		qmult = gether_ether_skb_num();
		return qmult * DEFAULT_QLEN;
	}
	else{
#endif		
		qmult = gether_ether_skb_num();
		return qmult * DEFAULT_QLEN;
	//}
}

/*-------------------------------------------------------------------------*/

/* REVISIT there must be a better way than having two sets
 * of debug calls ...
 */

#undef DBG
#undef VDBG
#undef ERROR
#undef INFO

#define xprintk(d, level, fmt, args...) \
	printk(level "%s: " fmt , (d)->net->name , ## args)

#ifdef DEBUG
#undef DEBUG
#define DBG(dev, fmt, args...) \
	xprintk(dev , KERN_DEBUG , fmt , ## args)
#else
#define DBG(dev, fmt, args...) \
	do { } while (0)
#endif /* DEBUG */

#ifdef VERBOSE_DEBUG
#define VDBG	DBG
#else
#define VDBG(dev, fmt, args...) \
	do { } while (0)
#endif /* DEBUG */

#define ERROR(dev, fmt, args...) \
	xprintk(dev , KERN_ERR , fmt , ## args)
#define INFO(dev, fmt, args...) \
	xprintk(dev , KERN_INFO , fmt , ## args)

/*-------------------------------------------------------------------------*/

/* NETWORK DRIVER HOOKUP (to the layer above this driver) */

static int ueth_change_mtu(struct net_device *net, int new_mtu)
{
	struct eth_dev	*dev = netdev_priv(net);
	unsigned long	flags;
	int		status = 0;

	/* don't change MTU on "live" link (peer won't know) */
	spin_lock_irqsave(&dev->lock, flags);
	if (dev->port_usb)
		status = -EBUSY;
	else if (new_mtu <= ETH_HLEN || new_mtu > ETH_FRAME_LEN)
		status = -ERANGE;
	else
		net->mtu = new_mtu;
	spin_unlock_irqrestore(&dev->lock, flags);

	return status;
}

static void eth_get_drvinfo(struct net_device *net, struct ethtool_drvinfo *p)
{
	struct eth_dev	*dev = netdev_priv(net);

	strlcpy(p->driver, "g_ether", sizeof p->driver);
	strlcpy(p->version, UETH__VERSION, sizeof p->version);
	strlcpy(p->fw_version, dev->gadget->name, sizeof p->fw_version);
	strlcpy(p->bus_info, dev_name(&dev->gadget->dev), sizeof p->bus_info);
}

/* REVISIT can also support:
 *   - WOL (by tracking suspends and issuing remote wakeup)
 *   - msglevel (implies updated messaging)
 *   - ... probably more ethtool ops
 */

static const struct ethtool_ops ops = {
	.get_drvinfo = eth_get_drvinfo,
	.get_link = ethtool_op_get_link,
};

static void defer_kevent(struct eth_dev *dev, int flag)
{
	if (test_and_set_bit(flag, &dev->todo))
		return;
	if (!schedule_work(&dev->work))
		ERROR(dev, "kevent %d may have been dropped\n", flag);
	else
		USBSTACK_DBG("kevent %d scheduled\n", flag);
}

static void rx_complete(struct usb_ep *ep, struct usb_request *req);
void u_ether_rx_vnic_packet_list(void);
void u_ether_tx_vnic_packet_list(void);
static void  mbim_loop_test_rx_callback(struct usb_ep *ep, struct usb_request *req);
static void  mbim_loop_test_tx_callback(struct usb_ep *ep, struct usb_request *req) ;


static void test_rx_complete(struct usb_ep *ep, struct usb_request *req)
{
     mbim_loop_test_rx_callback(ep,req);

}

static void test_tx_complete(struct usb_ep *ep, struct usb_request *req)
{
    mbim_loop_test_tx_callback(ep,req);
}


static int
rx_submit(struct eth_dev *dev, struct usb_request *req, gfp_t gfp_flags)
{
	struct sk_buff	*skb;
	int		retval = -ENOMEM;
	size_t		size = 0;
	struct usb_ep	*out;
	unsigned long	flags;

	spin_lock_irqsave(&dev->lock, flags);
	if (dev->port_usb)
		out = dev->port_usb->out_ep;
	else
		out = NULL;
	spin_unlock_irqrestore(&dev->lock, flags);

	if (!out)
		return -ENOTCONN;


	/* Padding up to RX_EXTRA handles minor disagreements with host.
	 * Normally we use the USB "terminate on short read" convention;
	 * so allow up to (N*maxpacket), since that memory is normally
	 * already allocated.  Some hardware doesn't deal well with short
	 * reads (e.g. DMA must be N*maxpacket), so for now don't trim a
	 * byte off the end (to force hardware errors on overflow).
	 *
	 * RNDIS uses internal framing, and explicitly allows senders to
	 * pad to end-of-packet.  That's potentially nice for speed, but
	 * means receivers can't recover lost synch on their own (because
	 * new packets don't only start after a short RX).
	 */
	size += sizeof(struct ethhdr) + dev->net->mtu + RX_EXTRA;
	size += dev->port_usb->header_len;
#if 0
/***  cancel 512Bytes align, per SKB Bytes form 4K Bytes to 2K Bytes ***/
	size += out->maxpacket - 1;
	size -= size % out->maxpacket;
#endif
	if (dev->port_usb->is_fixed)
		size = max_t(size_t, size, dev->port_usb->fixed_out_len);

	//skb = alloc_skb(size + NET_IP_ALIGN, gfp_flags);

	skb = dev_alloc_skb(size);
	if (skb == NULL) {
		DBG(dev, "no rx skb\n");
		goto enomem;
	}

	/* Some platforms perform better when IP packets are aligned,
	 * but on at least one, checksumming fails otherwise.  Note:
	 * RNDIS headers involve variable numbers of LE32 values.
	 */
	skb_reserve(skb, NET_IP_ALIGN);

	req->buf = skb->data;
	req->length = size;
	//req->complete = rx_complete;
	req->context = skb;
    req->status = 0;
    if(atomic_read(&dev->work_mode))
    {
    	//printk("rx_submit, complete is test\n");
        req->complete = test_rx_complete ;
    }
    else
    {
	//printk("rx_submit, complete is normal\n");
        req->complete = rx_complete;
    }
    

	//if(g_VNIC_MultiPacket_MaxNum ==1 )
	//	retval = usb_ep_queue(out, req, GFP_ATOMIC);
	//else	
#if 1
		retval = multi_packet_rx_queue(out, req, GFP_ATOMIC);
		g_test_rx_pkt++;
#else
	if(strstr(dev->net->name, "rndis"))
	{
		retval = multi_packet_rx_queue(out, req, GFP_ATOMIC);
	}	
	else
	{
		retval = usb_ep_queue(out, req, GFP_ATOMIC);
	}
#endif
	if (retval == -ENOMEM && (atomic_read(&dev->work_mode) == 0)){
enomem:
		defer_kevent(dev, WORK_RX_MEMORY);
	}
	if (retval) {
		USBSTACK_DBG("rx submit --> %d\n", retval);
		if (skb)
			dev_kfree_skb_any(skb);
		spin_lock_irqsave(&dev->req_lock, flags);
		list_add_tail(&req->list, &dev->rx_reqs);
		spin_unlock_irqrestore(&dev->req_lock, flags);
	}
	return retval;
}

int g_net_rx_cnt = 0;
extern int get_vnic_multi_packet_num(void);
static void rx_complete(struct usb_ep *ep, struct usb_request *req)
{
	struct sk_buff	*skb = req->context, *skb2;
	struct eth_dev	*dev = ep->driver_data;
	int		status = req->status;
	unsigned long flags;
	if(atomic_read(&dev->work_mode) == 1){
		printk("rx_complete, loopback mode,\r\n");
		mbim_loop_test_rx_callback(ep, req);
		return;
	}
	int packet_num = get_vnic_multi_packet_num();
		g_test_rx_complt_pkt++;
	//printk("usb ether rx_complete status:%d, len:%d\n", status, req->actual);
	switch (status) {

	/* normal completion */
	case 0:
		skb_put(skb, req->actual);

		if (dev->unwrap) {
			unsigned long	flags;

			spin_lock_irqsave(&dev->lock, flags);
			if (dev->port_usb) {
				status = dev->unwrap(dev->port_usb,
							skb,
							&dev->rx_frames);
			} else {
				dev_kfree_skb_any(skb);
				status = -ENOTCONN;
			}
			spin_unlock_irqrestore(&dev->lock, flags);
		} else {
			skb_queue_tail(&dev->rx_frames, skb);
		}
		
		skb = NULL;

		skb2 = skb_dequeue(&dev->rx_frames);
		while (skb2) {
			if (status < 0
					|| ETH_HLEN > skb2->len
					|| skb2->len > ETH_FRAME_LEN) {
				dev->net->stats.rx_errors++;
				dev->net->stats.rx_length_errors++;
				DBG(dev, "rx length %d\n", skb2->len);
				dev_kfree_skb_any(skb2);
				goto next_frame;
			}
#if 1
		if((strstr(dev->port_usb->func.name, "rndis"))||
			(strstr(dev->port_usb->func.name, "mbim")) ||
			(strstr(dev->port_usb->func.name, "cdc_ethernet")) ){
	            if (fast_from_driver && fast_from_driver(skb2, dev->net))
	            {
	//                print_sun1("<--- usb fastnat okokokokok !!!! \n");
	                goto next_frame;
	            }
		}
#else
		if(strstr(dev->port_usb->func.name, "rndis")){
			if (fast_from_driver && fast_from_driver(skb2, dev->net))
			{
//                print_sun1("<--- usb fastnat okokokokok !!!! \n");

				goto next_frame;
			}
		} else if(strstr(dev->port_usb->func.name, "cdc_ethernet")){
			
			if(((g_net_rx_cnt++)% 41) != 0) {
				if (fast_from_driver && fast_from_driver(skb2, dev->net))
				{
	//                print_sun1("<--- usb fastnat okokokokok !!!! \n");

					goto next_frame;
				}
			}
			
		}
#endif
//			print_sun1("<--- usb fastnat failed !!!! \n");

			skb2->protocol = eth_type_trans(skb2, dev->net);
			dev->net->stats.rx_packets++;
			dev->net->stats.rx_bytes += skb2->len;

			/* no buffer copies needed, unless hardware can't
			 * use skb buffers.
			 */
#if 0
			usb_dbg_printf("usb ether Send the packet\n");
			status = netif_rx(skb2);
#else
			/* Send the packet */
			if (in_interrupt()) {
				netif_rx(skb2);
			} else {
				netif_rx_ni(skb2);
			}
#endif

next_frame:
			skb2 = skb_dequeue(&dev->rx_frames);
		}
		break;

	/* software-driven interface shutdown */
	case -ECONNRESET:		/* unlink */
	case -ESHUTDOWN:		/* disconnect etc */
		VDBG(dev, "rx shutdown, code %d\n", status);
		goto quiesce;

	/* for hardware automagic (such as pxa) */
	case -ECONNABORTED:		/* endpoint reset */
		DBG(dev, "rx %s reset\n", ep->name);
		defer_kevent(dev, WORK_RX_MEMORY);
quiesce:
		dev_kfree_skb_any(skb);
		goto clean;

	/* data overrun */
	case -EOVERFLOW:
		dev->net->stats.rx_over_errors++;
		/* FALLTHROUGH */

	default:
		dev->net->stats.rx_errors++;
		DBG(dev, "rx status %d\n", status);
		break;
	}

	if (skb)
		dev_kfree_skb_any(skb);
	if (!netif_running(dev->net)) {
clean:
		spin_lock_irqsave(&dev->req_lock, flags);
		list_add_tail(&req->list, &dev->rx_reqs);
		spin_unlock_irqrestore(&dev->req_lock, flags);
		req = NULL;
	}
	if (req)
		rx_submit(dev, req, GFP_ATOMIC);
}

static int prealloc(struct list_head *list, struct usb_ep *ep, unsigned n)
{
	unsigned		i;
	struct usb_request	*req;

	if (!n)
		return -ENOMEM;

	/* queue/recycle up to N requests */
	i = n;
	list_for_each_entry(req, list, list) {
		if (i-- == 0)
			goto extra;
	}
	while (i--) {
		req = usb_ep_alloc_request(ep, GFP_ATOMIC);
		if (!req)
			return list_empty(list) ? -ENOMEM : 0;
		list_add_tail(&req->list, list);
	}
	return 0;

extra:
	/* free extras */
	for (;;) {
		struct list_head	*next;

		next = req->list.next;
		list_del(&req->list);
		usb_ep_free_request(ep, req);

		if (next == list)
			break;

		req = container_of(next, struct usb_request, list);
	}
	return 0;
}

static int alloc_requests(struct eth_dev *dev, struct gether *link, unsigned n)
{
	int	status;
	unsigned long flags;

	spin_lock_irqsave(&dev->req_lock, flags);
	status = prealloc(&dev->tx_reqs, link->in_ep, (TX_QMULT*DEFAULT_QLEN));
	if (status < 0)
		goto fail;
	status = prealloc(&dev->rx_reqs, link->out_ep, n);
	if (status < 0)
		goto fail;
	goto done;
fail:
	DBG(dev, "can't alloc requests\n");
done:
	spin_unlock_irqrestore(&dev->req_lock, flags);
	return status;
}

void rx_fill(struct eth_dev *dev, gfp_t gfp_flags)
{
	struct usb_request	*req;
	unsigned long		flags;
    unsigned int        skb_times = 0;
	
	/* fill unused rxq slots with some skb */
	spin_lock_irqsave(&dev->req_lock, flags);
	while (!list_empty(&dev->rx_reqs)) {
		skb_times++;
		req = container_of(dev->rx_reqs.next,
				struct usb_request, list);
		list_del_init(&req->list);
		spin_unlock_irqrestore(&dev->req_lock, flags);

		if (rx_submit(dev, req, gfp_flags) < 0) {
			if(atomic_read(&dev->work_mode) == 0)
				defer_kevent(dev, WORK_RX_MEMORY);
			return;
		}

		spin_lock_irqsave(&dev->req_lock, flags);
	}
	spin_unlock_irqrestore(&dev->req_lock, flags);
	USBSTACK_DBG("%s, %u skbtime:%d, qmult:%d, skbnum:%d", __func__, __LINE__, skb_times, qmult, gether_ether_skb_num());
}

static void eth_work(struct work_struct *work)
{
	struct eth_dev	*dev = container_of(work, struct eth_dev, work);

	if (test_and_clear_bit(WORK_RX_MEMORY, &dev->todo)) {
		if (netif_running(dev->net))
			rx_fill(dev, GFP_KERNEL);
	}

	if (dev->todo)
		USBSTACK_DBG("work done, flags = 0x%lx\n", dev->todo);
}

static void eth_inform_work(struct work_struct *inform_work)
{
	struct eth_dev	*dev = container_of(inform_work, struct eth_dev, inform_work);
	
	USBSTACK_DBG("%s --Inform USB net state: %d, reset:%d\n", __func__, dev->eth_state, dev->net_func_reset);
	
	if(dev->eth_state == dev->work_state){
		printk("%s:%d eth_state already inform:%d\n", __func__, __LINE__,dev->eth_state);
		USBSTACK_DBG("%s:%d already inform", __func__, __LINE__);
		return;
	}
	if(dev->net_func_reset == 1)
	{
        dev->net_func_reset = 0 ;
        kobject_uevent(&dev->net->dev.kobj, KOBJ_CHANGE);
    }
    
	if(dev->eth_state == 1){
		kobject_uevent(&dev->net->dev.kobj, KOBJ_ONLINE);
		dev->work_state = 1;
	usb_record_dbginfo(USB_DEV_ONLINE, 0, 1);
	}
	else{
		dev->work_state = 0;
		kobject_uevent(&dev->net->dev.kobj, KOBJ_OFFLINE);
	usb_record_dbginfo(USB_DEV_OFFLINE, 0, 1);
	}

	usb_printk("%s:%d Ether uevent %d %s\n", __func__, __LINE__, (dev->eth_state), (dev->eth_state) ? "ONLINE":"OFFLINE");
	USBSTACK_DBG("Inform USB net state: %d", dev->eth_state);
}

static void tx_complete(struct usb_ep *ep, struct usb_request *req)
{
	unsigned long			flags;

	struct sk_buff	*skb = req->context;
	struct eth_dev	*dev = ep->driver_data;

	switch (req->status) {
	default:
		dev->net->stats.tx_errors++;
		VDBG(dev, "tx err %d\n", req->status);
		/* FALLTHROUGH */
	case -ECONNRESET:		/* unlink */
	case -ESHUTDOWN:		/* disconnect etc */
		break;
	case 0:
		dev->net->stats.tx_bytes += skb->len;
	}
	dev->net->stats.tx_packets++;

	spin_lock_irqsave(&dev->req_lock, flags);
	list_add_tail(&req->list, &dev->tx_reqs);
	spin_unlock_irqrestore(&dev->req_lock, flags);
	dev_kfree_skb_any(skb);

	atomic_dec(&dev->tx_qlen);
	if (netif_carrier_ok(dev->net))
		netif_wake_queue(dev->net);
}

static inline int is_promisc(u16 cdc_filter)
{
	return cdc_filter & USB_CDC_PACKET_TYPE_PROMISCUOUS;
}

static netdev_tx_t eth_start_xmit(struct sk_buff *skb,
					struct net_device *net)
{
	struct eth_dev		*dev = netdev_priv(net);
	int			length = skb->len;
	int			retval;
	struct usb_request	*req = NULL;
	unsigned long		flags;
	struct usb_ep		*in;
	u16			cdc_filter;
	//printk("eth_start_xmit,eth_state:%d, length:%d\n",dev->eth_state, length);
	spin_lock_irqsave(&dev->lock, flags);
	if (dev->port_usb) {
		in = dev->port_usb->in_ep;
		cdc_filter = dev->port_usb->cdc_filter;
	} else {
		in = NULL;
		cdc_filter = 0;
	}
	spin_unlock_irqrestore(&dev->lock, flags);
    g_test_xmit_pktnum++;
	if (!in) {
		dev_kfree_skb_any(skb);
		g_test_xmit_pkterr1++;
		return NETDEV_TX_OK;
	}

	/* apply outgoing CDC or RNDIS filters */
	if (!is_promisc(cdc_filter)) {
		u8		*dest = skb->data;

		if (is_multicast_ether_addr(dest)) {
			u16	type;

			/* ignores USB_CDC_PACKET_TYPE_MULTICAST and host
			 * SET_ETHERNET_MULTICAST_FILTERS requests
			 */
			if (is_broadcast_ether_addr(dest))
				type = USB_CDC_PACKET_TYPE_BROADCAST;
			else
				type = USB_CDC_PACKET_TYPE_ALL_MULTICAST;
			if (!(cdc_filter & type)) {
				dev_kfree_skb_any(skb);
				return NETDEV_TX_OK;
			}
		}
		/* ignores USB_CDC_PACKET_TYPE_DIRECTED */
	}

	spin_lock_irqsave(&dev->req_lock, flags);
	/*
	 * this freelist can be empty if an interrupt triggered disconnect()
	 * and reconfigured the gadget (shutting down this queue) after the
	 * network stack decided to xmit but before we got the spinlock.
	 */
	 if (dev->eth_state == 0) {
				spin_unlock_irqrestore(&dev->req_lock, flags);
				g_test_xmit_pkterr3++;
				dev_kfree_skb_any(skb);
				return NET_XMIT_DROP;
	}
			
	if (list_empty(&dev->tx_reqs)
#ifdef CONFIG_PM
		||((dev->port_usb->suspend_state == 1) && (!strstr(dev->port_usb->func.name, "mbim")))
#endif
		) {
		spin_unlock_irqrestore(&dev->req_lock, flags);
        g_test_xmit_pkterr2++;
		dev_kfree_skb_any(skb);
		return NET_XMIT_DROP;
	}
#if 1
#ifdef CONFIG_PM
	if(strstr(dev->port_usb->func.name, "mbim")){
		if(dev->port_usb->suspend_state == 1 && (atomic_read(&dev->port_usb->wake_state) == 0)){
			
			struct usb_gadget *gadget = dev->port_usb->func.config->cdev->gadget;
			atomic_set(&dev->port_usb->wake_state, 1);
			spin_unlock_irqrestore(&dev->req_lock, flags);
			//if(dev->port_usb->func.config && dev->port_usb->func.config->cdev && dev->port_usb->func.config->cdev->gadget) {
				printk("\n---eth_start_xmit, call usb_gadget_wakeup\n");
			if(gadget){
			    usb_gadget_wakeup(dev->port_usb->func.config->cdev->gadget);
			} else{
		        g_test_xmit_pkterr2++;
				dev_kfree_skb_any(skb);
				return NET_XMIT_DROP;			
			}
			atomic_set(&dev->port_usb->wake_state, 0);
		spin_lock_irqsave(&dev->req_lock, flags);
		}
	}
#endif	
#endif
	req = container_of(dev->tx_reqs.next, struct usb_request, list);
	list_del(&req->list);

	/* temporarily stop TX queue when the freelist empties */
	if (list_empty(&dev->tx_reqs))
		netif_stop_queue(net);
	spin_unlock_irqrestore(&dev->req_lock, flags);

	/* no buffer copies needed, unless the network stack did it
	 * or the hardware can't use skb buffers.
	 * or there's not enough space for extra headers we need
	 */
	if (dev->wrap) {
		unsigned long	flags;

		spin_lock_irqsave(&dev->lock, flags);
		if (dev->port_usb)
			skb = dev->wrap(dev->port_usb, skb);
		spin_unlock_irqrestore(&dev->lock, flags);
		if (!skb)
			goto drop;

		length = skb->len;
	}
	skb = flush_skbuf(skb);
	req->buf = skb->data;
	req->dma = virtaddr_to_phys(skb->data);
	req->context = skb;
	req->complete = tx_complete;
	if(dev->port_usb == NULL){
		dev->net->stats.tx_dropped++;
		//when port_usb is null,  should  free the req 
		kfree(req);
		dev_kfree_skb_any(skb);
		return NET_XMIT_DROP;		
	}
	/* NCM requires no zlp if transfer is dwNtbInMaxSize */
	if (dev->port_usb->is_fixed &&
	    length == dev->port_usb->fixed_in_len &&
	    (length % in->maxpacket) == 0)
		req->zero = 0;
	else
		req->zero = 1;

	/* use zlp framing on tx for strict CDC-Ether conformance,
	 * though any robust network rx path ignores extra padding.
	 * and some hardware doesn't like to write zlps.
	 */
	if (req->zero && !dev->zlp && (length % in->maxpacket) == 0)
		length++;

	req->length = length;

	/* throttle high/super speed IRQ rate back slightly */
	if (gadget_is_dualspeed(dev->gadget))
		req->no_interrupt = (dev->gadget->speed == USB_SPEED_HIGH ||
				     dev->gadget->speed == USB_SPEED_SUPER)
			? ((atomic_read(&dev->tx_qlen) % qmult) != 0)
			: 0;
#if 0
    if (dev->port_usb->suspend_state == 1){
		usb_gadget_wakeup(dev->gadget);
		do{
			msleep(2);
		}while(dev->port_usb->suspend_state==1);
	}
#endif
	//if(g_VNIC_MultiPacket_MaxNum ==1 )
	//	retval = usb_ep_queue(in, req, GFP_ATOMIC);
	//else
	#if 1
		retval = multi_packet_tx_queue(in, req, GFP_ATOMIC);
	#else
	if(strstr(net->name, "rndis"))
	{
		retval = multi_packet_tx_queue(in, req, GFP_ATOMIC);
	}	
	else
	{
		USBSTACK_DBG("usb_ep_queue");
		retval = usb_ep_queue(in, req, GFP_ATOMIC);
	}
	#endif	
	switch (retval) {
	default:
		DBG(dev, "tx queue err %d\n", retval);
		break;
	case 0:
		net->trans_start = jiffies;
		atomic_inc(&dev->tx_qlen);
	}

	if (retval) {
		dev_kfree_skb_any(skb);
drop:
		dev->net->stats.tx_dropped++;
		spin_lock_irqsave(&dev->req_lock, flags);
		if (list_empty(&dev->tx_reqs))
			netif_start_queue(net);
		list_add_tail(&req->list, &dev->tx_reqs);
		spin_unlock_irqrestore(&dev->req_lock, flags);
	}
	return NETDEV_TX_OK;
}

/*-------------------------------------------------------------------------*/

static void eth_start(struct eth_dev *dev, gfp_t gfp_flags)
{
	USBSTACK_DBG("%s", __func__);
	//DBG(dev, "%s\n", __func__);
	//if(strstr(dev->net->name, "rndis"))
	{
		multi_packet_activate();
	}

	/* fill the rx queue */
	rx_fill(dev, gfp_flags);

	/* and open the tx floodgates */
	atomic_set(&dev->tx_qlen, 0);
	netif_wake_queue(dev->net);
}

static int eth_open(struct net_device *net)
{
	struct eth_dev	*dev = netdev_priv(net);
	struct gether	*link;
	int rtv;
	USBSTACK_DBG("%s", __func__);
	//DBG(dev, "%s\n", __func__);

    if (dev->eth_state == 0){
        USBSTACK_DBG("%s, %u eth_state:%d", __func__, __LINE__, dev->eth_state);
        return -EAGAIN;
    }
	
	if((dev->port_usb == NULL) || (&dev->port_usb->func == NULL))
		return -ENODEV;

#if MULTIPACKET_BUF_ALLOC
    if(strstr(dev->port_usb->func.name, "rndis"))
    {
        struct f_rndis		*rndis = func_to_rndis(&dev->port_usb->func);
    	if(rndis->state != 2)
        {	
    		rtv = multi_packet_buf_alloc();
    		if(rtv < 0)
            {
    			printk("eth_open, net is rndis and req alloc faild with no memory\n");
    			return rtv;
    		}
    		rndis->state = 2;
    	}        
    }
    else if(strstr(dev->port_usb->func.name, "mbim"))
    {
        struct f_mbim		*mbim = func_to_mbim(&dev->port_usb->func);
    	if(mbim->state != 2)
        {	
    		rtv = multi_packet_buf_alloc();
    		if(rtv < 0)
            {
    			printk("eth_open, net is mbim and req alloc faild with no memory\n");
    			return rtv;
    		}
    		mbim->state = 2;
    	}
        
    }

#endif

	if (netif_carrier_ok(dev->net))
		eth_start(dev, GFP_KERNEL);

	spin_lock_irq(&dev->lock);
	link = dev->port_usb;
	if (link && link->open)
		link->open(link);
	spin_unlock_irq(&dev->lock);
	usb_record_dbginfo(USB_DEV_OPEN, 0, 1);

	return 0;
}

static int eth_stop(struct net_device *net)
{
	struct eth_dev	*dev = netdev_priv(net);
	unsigned long	flags;

	USBSTACK_DBG("%s", __func__);
	//VDBG(dev, "%s\n", __func__);
	netif_stop_queue(net);

	USBSTACK_DBG("stop stats: rx/tx %ld/%ld, errs %ld/%ld\n",
		dev->net->stats.rx_packets, dev->net->stats.tx_packets,
		dev->net->stats.rx_errors, dev->net->stats.tx_errors
		);

	/* ensure there are no more active requests */
	spin_lock_irqsave(&dev->lock, flags);
	if (dev->port_usb) {
		struct gether	*link = dev->port_usb;
		const struct usb_endpoint_descriptor *in;
		const struct usb_endpoint_descriptor *out;

		if (link->close)
			link->close(link);

		/* NOTE:  we have no abort-queue primitive we could use
		 * to cancel all pending I/O.  Instead, we disable then
		 * reenable the endpoints ... this idiom may leave toggle
		 * wrong, but that's a self-correcting error.
		 *
		 * REVISIT:  we *COULD* just let the transfers complete at
		 * their own pace; the network stack can handle old packets.
		 * For the moment we leave this here, since it works.
		 */
		 
		//wangchao edit for ep-dma-free warning...
#if 0		
		in = link->in_ep->desc;
		out = link->out_ep->desc;
		usb_ep_disable(link->in_ep);
		usb_ep_disable(link->out_ep);
		if (netif_carrier_ok(net)) {
			DBG(dev, "host still using in/out endpoints\n");
			link->in_ep->desc = in;
			link->out_ep->desc = out;
			usb_ep_enable(link->in_ep);
			usb_ep_enable(link->out_ep);
		}
#endif
//when stop, data in vincPkt_list need release
	u_ether_tx_vnic_packet_list();

	}
	spin_unlock_irqrestore(&dev->lock, flags);
	usb_record_dbginfo(USB_DEV_CLOSE, 0, 1);

	return 0;
}

/*----------mac addr---------------------------------------*/

static u8 eth_addrs[ETH_MAX_NUM][ETH_ALEN] = {
	{0x34,0x4b,0x50,0x00,0x00,0x00},
	{0x34,0x4b,0x50,0x00,0x00,0x0a},
	{0x34,0x4b,0x50,0x00,0x00,0x0b},
	{0x34,0x4b,0x50,0x00,0x00,0x0c}};
//youchen@2024-06-20 add for lynq nv config begin
static u8 eth_addrs2[ETH_MAX_NUM][ETH_ALEN] = {
	{0x3a,0x4b,0x50,0x00,0x00,0x00},
	{0x34,0x4b,0x50,0x00,0x00,0x0a},
	{0x34,0x4b,0x50,0x00,0x00,0x0b},
	{0x34,0x4b,0x50,0x00,0x00,0x0c}};
//youchen@2024-06-20 add for lynq nv config end
#define NVUSB_MAC_LEN 6


/* initial value, changed by "ifconfig usb0 hw ether xx:xx:xx:xx:xx:xx" */
static char *dev_addr;
module_param(dev_addr, charp, S_IRUGO);
MODULE_PARM_DESC(dev_addr, "Device Ethernet Address");

/* this address is invisible to ifconfig */
static char *host_addr;
module_param(host_addr, charp, S_IRUGO);
MODULE_PARM_DESC(host_addr, "Host Ethernet Address");

static int get_ether_addr(const char *str, u8 *dev_addr)
{
	if (str) {
		unsigned	i;

		for (i = 0; i < 6; i++) {
			unsigned char num;

			if ((*str == '.') || (*str == ':'))
				str++;
			num = hex_to_bin(*str++) << 4;
			num |= hex_to_bin(*str++);
			dev_addr [i] = num;
		}
		if (is_valid_ether_addr(dev_addr))
			return 0;
	}
	random_ether_addr(dev_addr);
	return 1;
}

static int usb_set_mac_address(struct net_device *dev, void *p)
{
    int ret = eth_mac_addr(dev, p);
    int retval = 0;
	
    return ret;
}

static struct eth_dev *the_dev;
static struct eth_devs n_eth_dev = {0};

static const struct net_device_ops eth_netdev_ops = {
	.ndo_open		= eth_open,
	.ndo_stop		= eth_stop,
	.ndo_start_xmit		= eth_start_xmit,
	.ndo_change_mtu		= ueth_change_mtu,
	.ndo_set_mac_address 	= eth_mac_addr,
	.ndo_validate_addr	= eth_validate_addr,
};

static ssize_t show_gether_state(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct eth_dev	*pDev = netdev_priv(to_net_dev(dev));
	return sprintf(buf, "%d\n", pDev->eth_state);
}

static DEVICE_ATTR(gether_state, S_IRUGO, show_gether_state, NULL);

static struct attribute *gether_dev_attrs[] = {
	&dev_attr_gether_state.attr,
	NULL,
};

static struct attribute_group gether_dev_attr_grp = {
	.attrs = gether_dev_attrs,
};

static const struct attribute_group *gether_dev_groups[] = {
	&gether_dev_attr_grp,
	NULL
};

static struct device_type gadget_type = {
	.name	= "gadget",
	.groups	= &gether_dev_groups,
};

/**
 * gether_setup - initialize one ethernet-over-usb link
 * @g: gadget to associated with these links
 * @ethaddr: NULL, or a buffer in which the ethernet address of the
 *	host side of the link is recorded
 * Context: may sleep
 *
 * This sets up the single network link that may be exported by a
 * gadget driver using this framework.  The link layer addresses are
 * set up using module parameters.
 *
 * Returns negative errno, or zero on success
 */
int gether_setup(struct usb_gadget *g, u8 ethaddr[ETH_ALEN])
{
	return gether_setup_name(g, ethaddr, "usb");
}

/**
 * gether_setup_name - initialize one ethernet-over-usb link
 * @g: gadget to associated with these links
 * @ethaddr: NULL, or a buffer in which the ethernet address of the
 *	host side of the link is recorded
 * @netname: name for network device (for example, "usb")
 * Context: may sleep
 *
 * This sets up the single network link that may be exported by a
 * gadget driver using this framework.  The link layer addresses are
 * set up using module parameters.
 *
 * Returns negative errno, or zero on success
 */
int gether_setup_name(struct usb_gadget *g, u8 ethaddr[ETH_ALEN],
		const char *netname)
{
	int status;
	status = gether_setup_name_num(g, ethaddr, netname, 0);
	if (!status) {
		the_dev = n_eth_dev.eth_dev[0];
	}

	return status;
}

/**
 * gether_cleanup - remove Ethernet-over-USB device
 * Context: may sleep
 *
 * This is called to free all resources allocated by @gether_setup().
 */
void gether_cleanup(void)
{
	if (!the_dev)
		return;

	gether_cleanup_num(0);
	the_dev = NULL;
}

/**
 * gether_connect - notify network layer that USB link is active
 * @link: the USB link, set up with endpoints, descriptors matching
 *	current device speed, and any framing wrapper(s) set up.
 * Context: irqs blocked
 *
 * This is called to activate endpoints and let the network layer know
 * the connection is active ("carrier detect").  It may cause the I/O
 * queues to open and start letting network packets flow, but will in
 * any case activate the endpoints so that they respond properly to the
 * USB host.
 *
 * Verify net_device pointer returned using IS_ERR().  If it doesn't
 * indicate some error code (negative errno), ep->driver_data values
 * have been overwritten.
 */
struct net_device *gether_connect(struct gether *link)
{
	return gether_connect_num(link, 0);
}

/**
 * gether_disconnect - notify network layer that USB link is inactive
 * @link: the USB link, on which gether_connect() was called
 * Context: irqs blocked
 *
 * This is called to deactivate endpoints and let the network layer know
 * the connection went inactive ("no carrier").
 *
 * On return, the state is as if gether_connect() had never been called.
 * The endpoints are inactive, and accordingly without active USB I/O.
 * Pointers to endpoint descriptors and endpoint private data are nulled.
 */
void gether_disconnect(struct gether *link)
{
	struct eth_dev		*dev = link->ioport;
	struct usb_request	*req;
	unsigned long flags;

	WARN_ON(!dev);
	if (!dev)
		return;
	USBSTACK_DBG("%s", __func__);
	DBG(dev, "%s\n", __func__);

	//wangchao add to inform  USBnetcard down
	//dev->eth_state = 0;
	//schedule_work(&dev->inform_work);
	
	netif_stop_queue(dev->net);
	netif_carrier_off(dev->net);
	//wangchao add to inform  USBnetcard down
	if(strcmp(link->func.name,"mbim"))
	{
		printk("gether_disconnect , eth_state set 0\n");
        dev->eth_state = 0;
        schedule_work(&dev->inform_work);
       // return ;
    }	
	//if(strstr(dev->net->name, "rndis"))
	{
		multi_packet_deactivate();
	}
	/* disable endpoints, forcing (synchronous) completion
	 * of all pending i/o.  then free the request objects
	 * and forget about the endpoints.
	 */
	if(usb_ep_disable(link->in_ep)!=0)
	{
        printk("###################[%s], line:%u \n", __func__, __LINE__);
    }
	u_ether_tx_vnic_packet_list();
	spin_lock_irqsave(&dev->req_lock, flags);
	while (!list_empty(&dev->tx_reqs)) {
		req = container_of(dev->tx_reqs.next,
					struct usb_request, list);
		list_del(&req->list);

		spin_unlock_irqrestore(&dev->req_lock, flags);
		usb_ep_free_request(link->in_ep, req);
		spin_lock_irqsave(&dev->req_lock, flags);
	}
	spin_unlock_irqrestore(&dev->req_lock, flags);
	link->in_ep->driver_data = NULL;
	link->in_ep->desc = NULL;

	if(usb_ep_disable(link->out_ep) != 0)
	{
        printk("\n###################[%s], line:%u \n", __func__, __LINE__);
    }
	u_ether_rx_vnic_packet_list();
	spin_lock_irqsave(&dev->req_lock, flags);
	while (!list_empty(&dev->rx_reqs)) {
		req = container_of(dev->rx_reqs.next,
					struct usb_request, list);
		list_del(&req->list);

		spin_unlock_irqrestore(&dev->req_lock, flags);
		usb_ep_free_request(link->out_ep, req);
		spin_lock_irqsave(&dev->req_lock, flags);
	}
	spin_unlock_irqrestore(&dev->req_lock, flags);
	link->out_ep->driver_data = NULL;
	link->out_ep->desc = NULL;

	/* finish forgetting about this USB link episode */
	dev->header_len = 0;
	dev->unwrap = NULL;
	dev->wrap = NULL;

	spin_lock_irqsave(&dev->lock, flags);
#ifdef CONFIG_PM	
	atomic_set(&dev->port_usb->wake_state, 0);	
#endif
	dev->port_usb = NULL;
	link->ioport = NULL;
	spin_unlock_irqrestore(&dev->lock, flags);
}

/* ------------------------- multi ether ifac----------------------------*/
int gether_setup_name_num(struct usb_gadget *g, u8 ethaddr[ETH_ALEN],
		const char *netname, u8 num)
{
	struct eth_dev		*dev;
	struct net_device	*net;
	int			status;
	int                  ret;
	int           is_valid = 0;
	int           i;

	if(num >= ETH_MAX_NUM)
		return -ENOMEM;
	
	if (n_eth_dev.eth_dev[num])
		return -EBUSY;

	net = alloc_etherdev(sizeof *dev);
	if (!net)
		return -ENOMEM;
    if(!ethaddr)
        return -ENXIO;    

	dev = netdev_priv(net);
	spin_lock_init(&dev->lock);
	spin_lock_init(&dev->req_lock);
	INIT_WORK(&dev->work, eth_work);
	INIT_WORK(&dev->inform_work, eth_inform_work);
	INIT_LIST_HEAD(&dev->tx_reqs);
	INIT_LIST_HEAD(&dev->rx_reqs);
    atomic_set(&dev->work_mode,0);

	skb_queue_head_init(&dev->rx_frames);

	/* network device setup */
	dev->net = net;
	dev->eth_state = 0;
	dev->work_state = 0;
    dev->net_func_reset = 0 ;
	snprintf(net->name, sizeof(net->name), "%s%%d", netname);
  #if 1
	if(is_valid_ether_addr(ethaddr))
	{
		usb_printk("mac address is valid\n");
	       is_valid = 1;
		for(i = 0;i<NVUSB_MAC_LEN;i++)
		{
			net->dev_addr[i]   = ethaddr[i];
		}	
		net->dev_addr[NVUSB_MAC_LEN-1]   = ethaddr[NVUSB_MAC_LEN-1]-16;
		memcpy(dev->host_mac, ethaddr, ETH_ALEN);
	}
   
    if(is_valid == 0)
    {
    	random_ether_addr(net->dev_addr);
	usb_printk("random_ether_addr ok\n");
	//youchen@2024-06-20 add for lynq nv config begin
	if (get_rndis_mac_convert() == 1)
	{
		host_addr = &eth_addrs2[num][0];
		printk("rndis_mac_convert finish\n");
	}
	else
	{
		host_addr = &eth_addrs[num][0];
	}
	//youchen@2024-06-20 add for lynq nv config end
	memcpy(dev->host_mac, host_addr, ETH_ALEN);

	if (ethaddr)
		memcpy(ethaddr, dev->host_mac, ETH_ALEN);
    }

  #else
  if (get_ether_addr(dev_addr, net->dev_addr))
		dev_warn(&g->dev,
			"using random %s ethernet address\n", "self");
  #endif

	//host_addr = &eth_addrs[num][0];
	//memcpy(dev->host_mac, host_addr, ETH_ALEN);
	
#if 0	
	if (get_ether_addr(host_addr, dev->host_mac))
		dev_warn(&g->dev,
			"using random %s ethernet address\n", "host");
#endif

	//if (ethaddr)
		//memcpy(ethaddr, dev->host_mac, ETH_ALEN);

	net->netdev_ops = &eth_netdev_ops;
	net->flags |=IFF_NOARP ;
	SET_ETHTOOL_OPS(net, &ops);

	dev->gadget = g;
	SET_NETDEV_DEV(net, &g->dev);
	SET_NETDEV_DEVTYPE(net, &gadget_type);

	status = register_netdev(net);
	if (status < 0) {
		dev_dbg(&g->dev, "register_netdev failed, %d\n", status);
		free_netdev(net);
	} else {
		INFO(dev, "MAC %pM\n", net->dev_addr);
		INFO(dev, "HOST MAC %pM\n", dev->host_mac);

		n_eth_dev.eth_dev[num] = dev;

		/* two kinds of host-initiated state changes:
		 *  - iff DATA transfer is active, carrier is "on"
		 *  - tx queueing enabled if open *and* carrier is "on"
		 */
		netif_carrier_off(net);
	}

	return status;
}


void gether_cleanup_num(u8 num)
{
	struct eth_dev *eth_dev = n_eth_dev.eth_dev[num];
	
	if (!eth_dev)
		return;
	
	unregister_netdev(eth_dev->net);
	flush_work_sync(&eth_dev->work);
	flush_work_sync(&eth_dev->inform_work);
	free_netdev(eth_dev->net);

	n_eth_dev.eth_dev[num] = NULL;
}

struct net_device *gether_connect_num(struct gether *link, u8 num)
{
	struct eth_dev		*dev = n_eth_dev.eth_dev[num];
	int			result = 0;
	unsigned long flags;
    printk("########%s, %u.......  \n\n",__func__, __LINE__  );

	USBSTACK_DBG("%s: %d", __func__, num);
	if (!dev){
		printk("gether_connect_num dev is null\n");
		return ERR_PTR(-EINVAL);
	}
#ifdef CONFIG_PM
    if (link->suspend_state == 1)
		link->suspend_state = 0;
#endif	
	link->in_ep->driver_data = dev;
	result = usb_ep_enable(link->in_ep);
	if (result != 0) {
		printk("enable %s --> %d\n", link->in_ep->name, result);
		DBG(dev, "enable %s --> %d\n", link->in_ep->name, result);
		goto fail0;
	}

	link->out_ep->driver_data = dev;
	result = usb_ep_enable(link->out_ep);
	if (result != 0) {
		printk("enable %s --> %d\n", link->in_ep->name, result);
		DBG(dev, "enable %s --> %d\n",
			link->out_ep->name, result);
		goto fail1;
	}

	if (result == 0){
		result = alloc_requests(dev, link, qlen(dev->gadget));
		USBSTACK_DBG("alloc_requests fail result: %d", result);
	}
	
	if (result == 0) {
		dev->zlp = link->is_zlp_ok;
		USBSTACK_DBG("qlen %d\n", qlen(dev->gadget));

		dev->header_len = link->header_len;
		dev->unwrap = link->unwrap;
		dev->wrap = link->wrap;

		spin_lock_irqsave(&dev->lock, flags);
		dev->port_usb = link;
		link->ioport = dev;
		if (netif_running(dev->net)) {
			if (link->open)
				link->open(link);
		} else {
			if (link->close)
				link->close(link);
		}
		spin_unlock_irqrestore(&dev->lock, flags);
		
		netif_carrier_on(dev->net);
		if (netif_running(dev->net))
			eth_start(dev, GFP_ATOMIC);
		
		//wangchao add to inform  USBnetcard up
		//dev->eth_state = 1;
		//schedule_work(&dev->inform_work);
	/* on error, disable any endpoints  */
	} else {
		(void) usb_ep_disable(link->out_ep);
fail1:
		(void) usb_ep_disable(link->in_ep);
	}
fail0:
	/* caller is responsible for cleanup on error */
	if (result < 0){
		printk("gether_connect_num fail, ret:%d\n", result);
		return ERR_PTR(result);
	}
	return dev->net;
}

void gether_uevent_eth_rndis(struct gether *link)
{
	struct eth_dev		*dev = link->ioport;
    if (dev->eth_state == 1){
		gether_uevent_eth_switch(dev->net, 0);
    }
}

void gether_uevent_eth_ecm(struct gether *link,int state)
{
	//struct eth_dev		*dev = link->ioport;
   struct eth_dev *dev = n_eth_dev.eth_dev[0];
	if(state == 0){
    if (dev && (dev->eth_state==1)){
		gether_ecm_uevent(0, 0);
   	 }
    }
	else{
		if (dev && (dev->eth_state==0)){
		gether_ecm_uevent(0, 1);
    		}
	}
}

#define RNDIS_NUM 4
extern unsigned int usblan[RNDIS_NUM];
void gether_uevent_eth_switch(struct net_device *net, int eth_switch)
{
	struct eth_dev *dev;
	dev = netdev_priv(net);
	dev->eth_state = eth_switch;
	usblan[0] = eth_switch;
	schedule_work(&dev->inform_work);
    return;
}

void gether_ecm_uevent(u8 num, int ecm_switch)
{
	int i =0;
	int dev_num = num;
	struct eth_dev *dev;

	if(dev_num == 0)
		dev_num = ETH_MAX_NUM;

	for( i = 0; i < dev_num; i++){
		dev = n_eth_dev.eth_dev[i];
		if(dev){
			dev->eth_state = ecm_switch;
			usblan[i] = ecm_switch;
			schedule_work(&dev->inform_work);
		}
	}
	
	return;
}

extern void mbim_clean_ntb_param_flag(void);

void gether_mbim_uevent(int ecm_switch)
{
	struct eth_dev *dev;

	dev = n_eth_dev.eth_dev[0];
	if(dev == NULL || ecm_switch <0 || ecm_switch >2)
    {
        printk("[func]:%s ,[line]:%d \n",__func__,__LINE__) ;
        return ;
	}
	//printk("[func]:%s ,[line]:%d ,ecm_switch:%d\n",__func__,__LINE__,ecm_switch) ;
	
    switch(ecm_switch)
    {
        case 2 :  //λ
        {
            dev->net_func_reset = 1 ;
            schedule_work(&dev->inform_work);
			mbim_clean_ntb_param_flag();
            break ;
        }
        case 1:
        case 0:
        {
    		dev->eth_state = ecm_switch;
    		usblan[0] = ecm_switch;
    		schedule_work(&dev->inform_work);
            break ;
	    }
    }
  
	return;
}


/*=======================MBIM·ÂÕæ================================*/

static void  mbim_loop_test_tx_callback(struct usb_ep *ep, struct usb_request *req)
{

	struct eth_dev	*dev = ep->driver_data;

    if(strstr(dev->port_usb->func.name, "mbim") == 0)
    {
        printk("[%s]:net type invalid \n",__func__) ;
        return  ;
    }
    struct f_mbim   *mbim = func_to_mbim(&dev->port_usb->func);
	unsigned long			flags;
	struct sk_buff	*skb = req->context;


	switch (req->status) {
	default:
		VDBG(dev, "tx err %d\n", req->status);
		/* FALLTHROUGH */
	case -ECONNRESET:		/* unlink */
	case -ESHUTDOWN:		/* disconnect etc */
		break;
	case 0:
		break ;
	}
	
    //atomic_dec(&mbim->data_tx_cnt);
	spin_lock_irqsave(&dev->req_lock, flags);
	list_add_tail(&req->list, &dev->tx_reqs);
	spin_unlock_irqrestore(&dev->req_lock, flags);
	dev_kfree_skb_any(skb);
	atomic_dec(&dev->tx_qlen);
	
    wake_up(&mbim->data_write_wq) ;
    
}

static void  mbim_loop_test_rx_callback(struct usb_ep *ep, struct usb_request *req)
{

    struct eth_dev  *dev = ep->driver_data;
    if(strstr(dev->port_usb->func.name, "mbim") == 0)
    {
        printk("[%s]:net type invalid \n",__func__) ;
        return  ;
    }
   // printk("[%s]: ###function name :%s \n",__func__,dev->port_usb->func.name) ;
    struct f_mbim   *mbim = func_to_mbim(&dev->port_usb->func);
    //´Ódata idle
    //¿½±´Êý¾Ý
    //²åÈëÁ´±í
    //Í¨Öª
    struct sk_buff  *skb = req->context, *skb2 = NULL;

    int     status = req->status;

	//printk("[%s]:dataidlecnt:%d, skb data len:%d, req len:%d, req status:%d\n",__func__, 
	//	atomic_read(&mbim->data_idle_cnt),skb->data_len, req->actual, req->status) ;		
    unsigned long flags;
    //int packet_num = get_vnic_multi_packet_num();
    struct  mbim_pool_data_s *  data_channel_p = NULL ;
	if((atomic_read(&mbim->lb_flag) == 0) || (atomic_read(&dev->work_mode) == 0)){
		//loopback test is end
		printk("[%s]: loopback test end \n",__func__) ;
		goto quiesce;
	}
    switch (status) 
    {
    /* normal completion */
    case 0:
        skb_put(skb, req->actual + sizeof(struct ethhdr));
		//skb_pull(skb,sizeof(struct ethhdr));
        spin_lock_irqsave(&mbim->data_lock,flags) ;
        //if(atomic_read(&mbim->data_idle_cnt)== 0)
        if(list_empty(&mbim->data_ilde_list))
        {
            printk("[%s]:rx queue full..... \n",__func__) ;
			if(list_empty(&mbim->data_rx_list))
        	{
				 spin_unlock_irqrestore(&mbim->data_lock,flags) ; 
				 WARN_ON(1) ;
				 goto quiesce ;
			}
            data_channel_p = list_first_entry(&mbim->data_rx_list, struct mbim_pool_data_s, list) ;
            list_del_init(&data_channel_p->list) ;
            atomic_dec(&mbim->data_rx_cnt) ;
     
            skb2 = ( struct sk_buff  *) (data_channel_p->pdata);
        }
        else
        {
            data_channel_p = list_first_entry(&mbim->data_ilde_list, struct mbim_pool_data_s, list) ;
            atomic_dec(&mbim->data_idle_cnt) ;
            list_del_init(&data_channel_p->list) ;
        }
        //²åÈëÐÂµÄskb
        data_channel_p->pdata = skb ;
        list_add_tail(&data_channel_p->list, &mbim->data_rx_list) ;
		
		atomic_inc(&mbim->data_rx_cnt) ;
        //atomic_inc(mbim->data_rx_cnt) ;
        spin_unlock_irqrestore(&mbim->data_lock,flags) ; 
        //»½ÐÑ
       // wake_up(&mbim->data_read_wq)  ;
       wake_up(&mbim->lp_wait)  ;
		if(skb2 != NULL)
		{
			dev_kfree_skb_any(skb2) ;
		}

        break;
        
    case -ECONNRESET:       /* unlink */
    case -ESHUTDOWN:        /* disconnect etc */
        VDBG(dev, "rx shutdown, code %d\n", status);
        goto quiesce;

quiesce:
        dev_kfree_skb_any(skb);
        goto clean;
        
    case -EOVERFLOW:
    default:
        DBG(dev, "rx status %d\n", status);
        break;
    }

    if ((atomic_read(&dev->work_mode) == 0)) 
    {
clean:
        spin_lock_irqsave(&dev->req_lock, flags);
        list_add_tail(&req->list, &dev->rx_reqs);
        spin_unlock_irqrestore(&dev->req_lock, flags);
        req = NULL;
    }
    if (req)
    {
        rx_submit(dev, req, GFP_ATOMIC);
    }

}

int mbim_loop_test_xmit( struct eth_dev      *dev ,struct sk_buff* skb) 
{
    //»ñÈ¡req
    //ÅäÖÃreq
    //·ÅÈëÁ´±í
    //struct eth_dev      *dev = netdev_priv(net);
    int         length = 0;
    int         retval;
    struct usb_request  *req = NULL;
    unsigned long       flags;
    struct usb_ep       *in;
    u16         cdc_filter;

    if(dev == NULL || skb == NULL)
    {
        printk("%s : param invalid \n",__func__) ;
        return -NET_XMIT_DROP ;
    }
	length = skb->len;
    spin_lock_irqsave(&dev->lock, flags);
    if (dev->port_usb) {
        in = dev->port_usb->in_ep;
        cdc_filter = dev->port_usb->cdc_filter;
    } else {
        in = NULL;
        cdc_filter = 0;
    }
    spin_unlock_irqrestore(&dev->lock, flags);
    if (!in) {
        dev_kfree_skb_any(skb);
        return -NET_XMIT_DROP;
    }
    spin_lock_irqsave(&dev->req_lock, flags);
    /*
     * this freelist can be empty if an interrupt triggered disconnect()
     * and reconfigured the gadget (shutting down this queue) after the
     * network stack decided to xmit but before we got the spinlock.
     */
 #if 0    
     if (dev->eth_state == 0) {
                spin_unlock_irqrestore(&dev->req_lock, flags);
                dev_kfree_skb_any(skb);
                return -NET_XMIT_DROP;
    }
#endif            
    if (list_empty(&dev->tx_reqs) || (dev->port_usb == NULL)
#ifdef CONFIG_PM
        ||(dev->port_usb != NULL && dev->port_usb->suspend_state == 1)
#endif
        ) {
        spin_unlock_irqrestore(&dev->req_lock, flags);
        dev_kfree_skb_any(skb);
        return -NET_XMIT_DROP;
    }

    req = container_of(dev->tx_reqs.next, struct usb_request, list);
    list_del(&req->list);

    /* temporarily stop TX queue when the freelist empties */
    
    spin_unlock_irqrestore(&dev->req_lock, flags);

    /* no buffer copies needed, unless the network stack did it
     * or the hardware can't use skb buffers.
     * or there's not enough space for extra headers we need
     */


    skb = flush_skbuf(skb);
	if(skb == NULL)
	{
		return -NET_XMIT_DROP;
	}
	
    req->buf = skb->data;
    req->dma = virtaddr_to_phys(skb->data);
    req->context = skb;
    req->complete = mbim_loop_test_tx_callback;


    if(dev->port_usb == NULL)
    {
        //when port_usb is null,  should  free the req 
        kfree(req);
        dev_kfree_skb_any(skb);
        return -NET_XMIT_DROP;
    }
    /* NCM requires no zlp if transfer is dwNtbInMaxSize */
    if (dev->port_usb->is_fixed &&
        length == dev->port_usb->fixed_in_len &&
        (length % in->maxpacket) == 0)
        req->zero = 0;
    else
        req->zero = 1;

    /* use zlp framing on tx for strict CDC-Ether conformance,
     * though any robust network rx path ignores extra padding.
     * and some hardware doesn't like to write zlps.
     */
    if (req->zero && !dev->zlp && (length % in->maxpacket) == 0)
        length++;

    req->length = length;

    /* throttle high/super speed IRQ rate back slightly */
    if (gadget_is_dualspeed(dev->gadget))
        req->no_interrupt = (dev->gadget->speed == USB_SPEED_HIGH ||
                     dev->gadget->speed == USB_SPEED_SUPER)
            ? ((atomic_read(&dev->tx_qlen) % qmult) != 0) : 0;

	//printk("-----------mbim_loop_test_xmit, now multi_packet_tx_queue\r\n");
    retval = multi_packet_tx_queue(in, req, GFP_ATOMIC);

    switch (retval) 
    {
    default:
        DBG(dev, "tx queue err %d\n", retval);
        break;
    case 0:
        atomic_inc(&dev->tx_qlen);
    }

    if (retval) 
    {
        dev_kfree_skb_any(skb);
drop:
        spin_lock_irqsave(&dev->req_lock, flags);
        list_add_tail(&req->list, &dev->tx_reqs);
        spin_unlock_irqrestore(&dev->req_lock, flags);
    }
    return 0;

}



int mbim_switch_network_mode(struct gether *link ,int type)
{

    if(link == NULL || link->ioport == NULL || link->ioport->net == NULL)
    {
        printk("[%s],[line]:%d ,param invalid \n",__func__,__LINE__);
        return -ENODEV ;
    }
    struct eth_dev		*dev = link->ioport;
    if(type == 0)
    {
		printk("[%s],[line]:%d ,now switch to normal mode \n",__func__,__LINE__);
        
        atomic_set(&dev->work_mode, 0) ;		
		mbim_change_rx_complete(rx_complete);
     //   netif_start_queue(dev->net);
    }
    else
    {    
	printk("[%s],[line]:%d ,now switch to loop test mode \n",__func__,__LINE__);
	
         atomic_set(&dev->work_mode, 1) ;
		 
        // netif_stop_queue(dev->net);
		 //reset rx req  
		 
	mbim_change_rx_complete(test_rx_complete);
    }
    return 0 ;
}


