/*
 * zx29_i2s.c  --  ZX29_I2S ALSA SoC Audio platform driver
 *
 * Copyright (C) 2017, ZTE Corporation.
 *
 * Based on zx-i2s.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

/****************************************************************************
* 	                                 Include files
****************************************************************************/
#include <linux/clk.h>
#include <linux/device.h>
#include <linux/init.h>
#include <linux/io.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/gpio.h>
#include <linux/clk/zx29-clk.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/soc.h>
#include <sound/soc-dai.h>
#include <linux/math64.h>

#include <sound/core.h>
#include <sound/dmaengine_pcm.h>
#include <sound/initval.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/soc.h>
#include <linux/clk/zx29-clk.h>
#include <mach/iomap.h>
#include <mach/gpio.h>
#include <mach/clk.h>
#include <mach/gpio_def.h>
#include <mach/gpio_cfg.h>
#include <mach/gpio-names.h>
#include <mach/board.h>
#include <mach/dma.h>
#include <mach/gpio_cfg.h>
#include "i2s.h"
#include "dma.h"
#include <linux/wakelock.h>
#include <linux/soc/zte/pm/drv_idle.h>
#include <linux/string.h>
#include <linux/delay.h>
/****************************************************************************
* 	                                Local Macros
****************************************************************************/

#define ZX29_I2S_VERSION	0x00
#define ZX29_I2S_PROCESS_CTRL	0x04
#define ZX29_I2S_TIMING_CTRL	0x08
#define	ZX29_I2S_FIFO_CTRL	0x0C
#define	ZX29_I2S_FIFO_STATUS	0x10
#define ZX29_I2S_INT_EN		0x14
#define ZX29_I2S_INT_STATUS	0x18
#define ZX29_I2S_DATA		0x1C
#define ZX29_I2S_FRAME_CNTR	0x20
#define ZX29_I2S_DATA2		0x24

#define I2S_DEAGULT_FIFO_THRES	(0xf)
#define I2S_MAX_FIFO_THRES	(0x20)

#define ZX29_I2S_PROCESS_TX_EN	(1 << 0)
#define ZX29_I2S_PROCESS_TX_DIS	(0 << 0)
#define ZX29_I2S_PROCESS_RX_EN	(1 << 1)
#define ZX29_I2S_PROCESS_RX_DIS	(0 << 1)
#define ZX29_I2S_PROCESS_I2S_EN	(1 << 2)
#define ZX29_I2S_PROCESS_I2S_DIS	(0 << 2)

#define ZX29_I2S_TIMING_MAST		(1 << 0)
#define ZX29_I2S_TIMING_SLAVE		(0 << 0)
#define ZX29_I2S_TIMING_MS_MASK		(1 << 0)
#define ZX29_I2S_TIMING_LOOP		(1 << 1)
#define ZX29_I2S_TIMING_NOR			(0 << 1)
#define ZX29_I2S_TIMING_LOOP_MASK	(1 << 1)
#define ZX29_I2S_TIMING_PTNR		(1 << 2)
#define ZX29_I2S_TIMING_NTPR		(0 << 2)
#define ZX29_I2S_TIMING_PHASE_MASK	(1 << 2)
#define ZX29_I2S_TIMING_TDM			(1 << 3)
#define ZX29_I2S_TIMING_I2S			(0 << 3)
#define ZX29_I2S_TIMING_TIMING_MASK	(1 << 3)
#define ZX29_I2S_TIMING_LONG_SYNC	(1 << 4)
#define ZX29_I2S_TIMING_SHORT_SYNC	(0 << 4)
#define ZX29_I2S_TIMING_SYNC_MASK	(1 << 4)
#define ZX29_I2S_TIMING_TEAK_EN		(1 << 5)
#define ZX29_I2S_TIMING_TEAK_DIS	(0 << 5)
#define ZX29_I2S_TIMING_TEAK_MASK	(1 << 5)
#define ZX29_I2S_TIMING_STD_I2S		(0 << 6)
#define ZX29_I2S_TIMING_MSB_JUSTIF	(1 << 6)
#define ZX29_I2S_TIMING_LSB_JUSTIF	(2 << 6)
#define ZX29_I2S_TIMING_NORMAL_MODE	(3 << 6)
#define ZX29_I2S_TIMING_ALIGN_MASK	(3 << 6)
#define ZX29_I2S_TIMING_CHN_MASK	(7 << 8)
#define ZX29_I2S_TIMING_CHN(x)		((x - 1) << 8)
#define ZX29_I2S_TIMING_LANE_MASK	(3 << 11)
#define ZX29_I2S_TIMING_LANE(x)		((x - 1) << 11)
#define ZX29_I2S_TIMING_TSCFG_MASK	(7 << 13)
#define ZX29_I2S_TIMING_TSCFG(x)	(x << 13)
#define ZX29_I2S_TIMING_TS_WIDTH_MASK	(0x1f << 16)
#define ZX29_I2S_TIMING_TS_WIDTH(x)	((x - 1) << 16)
#define ZX29_I2S_TIMING_DATA_SIZE_MASK	(0x1f << 21)
#define ZX29_I2S_TIMING_DATA_SIZE(x)	((x - 1) << 21)
#define ZX29_I2S_TIMING_CFG_SMJZ_NORMAL_MODE	(0 << 30)
#define ZX29_I2S_TIMING_CFG_SMJZ_DMIC_MODE	(1 << 30)
#define ZX29_I2S_TIMING_CFG_SMJZ_MODE_MASK	(1 << 30)

#define ZX29_I2S_TIMING_CFG_ERR_MASK	(1 << 31)

#define ZX29_I2S_FIFO_CTRL_TX_RST		(1 << 0)
#define ZX29_I2S_FIFO_CTRL_TX_RST_MASK	(1 << 0)
#define ZX29_I2S_FIFO_CTRL_RX_RST		(1 << 1)
#define ZX29_I2S_FIFO_CTRL_RX_RST_MASK	(1 << 1)
#define ZX29_I2S_FIFO_CTRL_TX_DMA_EN	(1 << 4)
#define ZX29_I2S_FIFO_CTRL_TX_DMA_DIS	(0 << 4)
#define ZX29_I2S_FIFO_CTRL_TX_DMA_MASK	(1 << 4)
#define ZX29_I2S_FIFO_CTRL_RX_DMA_EN0	(1 << 5)
#define ZX29_I2S_FIFO_CTRL_RX_DMA_DIS0	(0 << 5)
#define ZX29_I2S_FIFO_CTRL_RX_DMA_MASK0	(1 << 5)
#define ZX29_I2S_FIFO_CTRL_RX_DMA_EN1	(1 << 6)
#define ZX29_I2S_FIFO_CTRL_RX_DMA_DIS1	(0 << 6)
#define ZX29_I2S_FIFO_CTRL_RX_DMA_MASK1	(1 << 6)
#define ZX29_I2S_FIFO_CTRL_TX_THRES_MASK	(0x1F << 8)
#define ZX29_I2S_FIFO_CTRL_RX_THRES_MASK	(0x1F << 16)

#define CLK_RAT (32 * 4)

#define ZX_I2S0_CRM_CLKSEL     			(ZX_LSP_CRPM_BASE+0x14)
#define ZX_I2S1_CRM_CLKSEL     			(ZX_LSP_CRPM_BASE+0x1C)

#if (defined CONFIG_SND_SOC_ZX297520V2) || (defined CONFIG_SND_SOC_ZX297520V2_MODULE)
#define ZX_I2S0_CLKDIV_1       			(ZX_LSP_CRPM_BASE+0x18)
#define ZX_I2S1_CLKDIV_1       			(ZX_LSP_CRPM_BASE+0x20)
#elif defined(CONFIG_SND_SOC_ZX297520V3) || (defined CONFIG_SND_SOC_ZX297520V3_MODULE)
#define ZX_I2S0_CLKDIV_1       			(ZX_LSP_CRPM_BASE+0x58)
#define ZX_I2S0_CLKDIV_2       			(ZX_LSP_CRPM_BASE+0x5C)
#define ZX_I2S1_CLKDIV_1       			(ZX_LSP_CRPM_BASE+0x60)
#define ZX_I2S1_CLKDIV_2       			(ZX_LSP_CRPM_BASE+0x64)
#endif

#define zx_read_i2s_reg(addr)   zx_read_reg(addr)
#define zx_write_i2s_reg(val, addr)   zx_write_reg(addr, val)

/****************************************************************************
* 	                                Local Types
****************************************************************************/

struct zx29_i2s {
	struct zx29_dma_params	dma_playback;
	struct zx29_dma_params	dma_capture;
	struct clk			*dai_wclk;
	struct clk			*dai_pclk;
	struct mutex         i2s_mutex;
	struct wake_lock     pm_lock;
	volatile int 		i2s_active_count;
	void __iomem		*reg_base;
	void __iomem		*reg_clksel;
	void __iomem		*reg_clkdiv1;
	void __iomem		*reg_clkdiv2 ;
	int					master;
	resource_size_t		mapbase;
	int                 i2s_num;
	unsigned int        refclk;
	unsigned int        suspend_i2stimingctrl;
	unsigned int		suspend_i2sfifoctrl;
	unsigned int		suspend_i2sprocessctrl;
	unsigned int		suspend_i2sclksel;
	unsigned int		suspend_i2sclkdiv1;
	unsigned int		suspend_i2sclkdiv2;
	unsigned int 		suspend_active;
};

//static int zx29_i2s_set_clkdiv(struct snd_soc_dai *dai, 	struct zx_i2s_clkdiv *pdiv);
#ifdef CONFIG_SND_SOC_ES8312
static struct zx29_i2s *zx29_i2s;
#endif

#ifdef CONFIG_SYSFS

static ssize_t zx29_i2s_reg_show(struct device *dev,
                                 struct device_attribute *attr, char *buf)
{
	struct zx29_i2s *i2s = dev_get_drvdata(dev);
	unsigned int val, len1, len2 = 0;
	char tmpbuf[30];
//	memset(tmpbuf, 'X', 30);
//  tmpbuf[29] = '\0';

	val = zx_read_i2s_reg(i2s->reg_base + ZX29_I2S_PROCESS_CTRL);
	snprintf(tmpbuf, sizeof(tmpbuf), "%s: 0x%x\n", "process_ctl", val);
	len1 = strlen(tmpbuf);
	memcpy(buf, tmpbuf, len1);
	len2 += len1;

	val = zx_read_i2s_reg(i2s->reg_base + ZX29_I2S_TIMING_CTRL);
	snprintf(tmpbuf, sizeof(tmpbuf), "%s: 0x%x\n", "timing_ctl", val);
	len1 = strlen(tmpbuf);
	memcpy(buf + len2, tmpbuf, len1);
	len2 += len1;

	val = zx_read_i2s_reg(i2s->reg_base + ZX29_I2S_FIFO_CTRL);
	snprintf(tmpbuf, sizeof(tmpbuf), "%s: 0x%x\n", "fifo_ctl", val);
	len1 = strlen(tmpbuf);
	memcpy(buf + len2, tmpbuf, len1);
	len2 += len1;

	val = zx_read_i2s_reg(i2s->reg_base + ZX29_I2S_FIFO_STATUS);
	snprintf(tmpbuf, sizeof(tmpbuf), "%s: 0x%x\n", "fifo_status", val);
	len1 = strlen(tmpbuf);
	memcpy(buf + len2, tmpbuf, len1);
	len2 += len1;

#if (defined CONFIG_SND_SOC_ZX297520V2) || (defined CONFIG_SND_SOC_ZX297520V2_MODULE)
	val = zx_read_i2s_reg(i2s->reg_clksel);
	snprintf(tmpbuf, sizeof(tmpbuf), "%s: 0x%x\n", "clksel", val);
	len1 = strlen(tmpbuf);
	memcpy(buf + len2, tmpbuf, len1);
	len2 += len1;

	val = zx_read_i2s_reg(i2s->reg_clkdiv1);
	snprintf(tmpbuf, sizeof(tmpbuf), "%s: 0x%x\n", "clkdiv1", val);
	len1 = strlen(tmpbuf);
	memcpy(buf + len2, tmpbuf, len1);
	len2 += len1;

#elif defined(CONFIG_SND_SOC_ZX297520V3) || (defined CONFIG_SND_SOC_ZX297520V3_MODULE)
	val = zx_read_i2s_reg(i2s->reg_clksel);
	snprintf(tmpbuf, sizeof(tmpbuf), "%s: 0x%x\n", "clksel", val);
	len1 = strlen(tmpbuf);
	memcpy(buf + len2, tmpbuf, len1);
	len2 += len1;

	val = zx_read_i2s_reg(i2s->reg_clkdiv1);
	snprintf(tmpbuf, sizeof(tmpbuf), "%s: 0x%x\n", "clkdiv1", val);
	len1 = strlen(tmpbuf);
	memcpy(buf + len2, tmpbuf, len1);
	len2 += len1;

	val = zx_read_i2s_reg(i2s->reg_clkdiv2);
	snprintf(tmpbuf, sizeof(tmpbuf), "%s: 0x%x\n", "clkdiv2", val);
	len1 = strlen(tmpbuf);
	memcpy(buf + len2, tmpbuf, len1);
	len2 += len1;

#endif
	return len2;
}

static ssize_t zx29_i2s_reg_store(struct device *dev,
                                  struct device_attribute *attr, const char *buf, size_t count)
{
	/** echo "process_ctl 0x000" > i2s_reg ***/

	struct zx29_i2s *i2s = dev_get_drvdata(dev);
	volatile unsigned int *reg;
	char tmpbuf[32] = {0};
	size_t buf_size;
	char *start = tmpbuf;
	unsigned long value;
	strlcpy(tmpbuf, buf, sizeof(tmpbuf));

	while (*start == ' ')
		start++;
	printk("Alsa: zx29_i2s_reg_store tmpbuf=%s\n", tmpbuf);

	if (strstr(tmpbuf, "process_ctl")) {
		reg = 	i2s->reg_base + ZX29_I2S_PROCESS_CTRL;
		start = start + 11;
	} else if (strstr(tmpbuf, "timing_ctl")) {
		reg = 	i2s->reg_base +  ZX29_I2S_TIMING_CTRL;
		start = start + 10;
	} else if (strstr(tmpbuf, "fifo_ctl")) {
		reg = 	i2s->reg_base +  ZX29_I2S_FIFO_CTRL;
		start = start + 8;
	} else if (strstr(tmpbuf, "clksel")) {
		reg = 	i2s->reg_clksel;
		start = start + 6;
	} else if (strstr(tmpbuf, "clkdiv1")) {
		reg = 	i2s->reg_clkdiv1;
		start = start + 7;
	} else if (strstr(tmpbuf, "clkdiv2")) {
		reg = 	i2s->reg_clkdiv2;
		start = start + 7;
	} else {
		printk("Invalid argument!\n");
		return -EINVAL;
	}

	while (*start == ' ')
		start++;
	if (strict_strtoul(start, 16, &value))
		return -EINVAL;

	zx_write_i2s_reg(value, reg);

	return count;
}

DEVICE_ATTR(i2s_reg, S_IWUSR | S_IRUGO, zx29_i2s_reg_show, zx29_i2s_reg_store);

static struct attribute *zx29_i2s_sysfs_attrs[] = {
	&dev_attr_i2s_reg.attr,
	NULL
};

static struct attribute_group zx29_i2s_attr_grp = {
	.attrs = zx29_i2s_sysfs_attrs,
};

static int zx29_i2s_sysfs_create_group(struct platform_device *pdev)
{
	return sysfs_create_group(&pdev->dev.kobj, &zx29_i2s_attr_grp);
}

static void zx29_i2s_sysfs_remove_group(struct platform_device *pdev)
{
	sysfs_remove_group(&pdev->dev.kobj, &zx29_i2s_attr_grp);
}

#else
static int zx29_i2s_sysfs_create_group(struct platform_device *pdev)
{
	return 0;
}

static void zx29_i2s_sysfs_remove_group(struct platform_device *pdev)
{
	return;
}
#endif

static void zx29_i2s_tx_en(void __iomem *base, bool on)
{
	unsigned long val;

	val = zx_read_i2s_reg(base + ZX29_I2S_PROCESS_CTRL);
	if (on)
		val |= ZX29_I2S_PROCESS_TX_EN;
	else
		val &= ~(ZX29_I2S_PROCESS_TX_EN);
	zx_write_i2s_reg(val, base + ZX29_I2S_PROCESS_CTRL);
}

static void zx29_i2s_rx_en(void __iomem *base, bool on)
{
	unsigned long val;

	val = zx_read_i2s_reg(base + ZX29_I2S_PROCESS_CTRL);
	if (on)
		val |= ZX29_I2S_PROCESS_RX_EN;
	else
		val &= ~(ZX29_I2S_PROCESS_RX_EN);
	zx_write_i2s_reg(val, base + ZX29_I2S_PROCESS_CTRL);
}

static void zx29_i2s_en(void __iomem *base, bool on)
{
	unsigned long val;

	val = zx_read_i2s_reg(base + ZX29_I2S_PROCESS_CTRL);
	if (on)
		val |= ZX29_I2S_PROCESS_I2S_EN;
	else
		val &= ~(ZX29_I2S_PROCESS_I2S_EN);
	zx_write_i2s_reg(val, base + ZX29_I2S_PROCESS_CTRL);
}

static void zx29_i2s_tx_dma_en(void __iomem *base, bool on)
{
	unsigned long val;

	val = zx_read_i2s_reg(base + ZX29_I2S_FIFO_CTRL);
#if (defined CONFIG_SND_SOC_ZX297520V2) || (defined CONFIG_SND_SOC_ZX297520V2_MODULE)
	val &= 0xffffe0fe;
	val |= ZX29_I2S_FIFO_CTRL_TX_RST | (I2S_DEAGULT_FIFO_THRES << 8);
#elif defined(CONFIG_SND_SOC_ZX297520V3) || (defined CONFIG_SND_SOC_ZX297520V3_MODULE)
	val &= 0xffffe0ff;
	val |= (I2S_DEAGULT_FIFO_THRES << 8);
#endif
	if (on)
		val |= ZX29_I2S_FIFO_CTRL_TX_DMA_EN;
	else
		val &= ~ZX29_I2S_FIFO_CTRL_TX_DMA_EN;
	zx_write_i2s_reg(val, base + ZX29_I2S_FIFO_CTRL);
}

static void zx29_i2s_rx_dma_en(void __iomem *base, bool on)
{
	unsigned long val;

	val = zx_read_i2s_reg(base + ZX29_I2S_FIFO_CTRL);
#if (defined CONFIG_SND_SOC_ZX297520V2) || (defined CONFIG_SND_SOC_ZX297520V2_MODULE)
	val &= 0xffe0fffd;
	val |= ZX29_I2S_FIFO_CTRL_RX_RST | (I2S_DEAGULT_FIFO_THRES << 16);
#elif defined(CONFIG_SND_SOC_ZX297520V3) || (defined CONFIG_SND_SOC_ZX297520V3_MODULE)
	val &= 0xffe0ffff;
	val |= (I2S_DEAGULT_FIFO_THRES << 16);
#endif
	if (on)
		val |= ZX29_I2S_FIFO_CTRL_RX_DMA_EN0;
	else
		val &= ~ZX29_I2S_FIFO_CTRL_RX_DMA_EN0;
	zx_write_i2s_reg(val, base + ZX29_I2S_FIFO_CTRL);
}

#define ZX29_I2S_RATES \
	(SNDRV_PCM_RATE_8000 | SNDRV_PCM_RATE_11025 | SNDRV_PCM_RATE_16000 | \
	 SNDRV_PCM_RATE_22050 | SNDRV_PCM_RATE_32000 | SNDRV_PCM_RATE_44100 | \
	 SNDRV_PCM_RATE_48000)

#define ZX29_I2S_FMTBIT \
	(SNDRV_PCM_FMTBIT_S16_LE | SNDRV_PCM_FMTBIT_S24_LE | \
	SNDRV_PCM_FMTBIT_S32_LE)


#if (defined CONFIG_SND_SOC_ZX297520V2) || (defined CONFIG_SND_SOC_ZX297520V2_MODULE)
#define	MIX_CLK_DIVIDER(d,b,n,s)  (d|(b<<8)|(n<<16)|(s<<24))
#define CLKDIV_CALC_FACTOR  (u64)(100000)
static int zx29_i2s_set_clkdiv(struct snd_soc_dai *dai, int rate, int width)
{
	unsigned int integer_num, clk_sel, fra_base, fra_div, clkdiv;
	unsigned int val;
	unsigned int bclk = 2 * rate * width;
	unsigned long bclk2 = bclk;
	u64  clk_div = 0;
	struct zx29_i2s *i2s = snd_soc_dai_get_drvdata(dai);
//	print_audio("Alsa Entered func %s\n", __func__);

	clk_div = i2s->refclk * CLKDIV_CALC_FACTOR / 2;
	clk_div = div64_u64(clk_div, bclk2);
	integer_num = i2s->refclk / (2 * 2 * bclk);
	clk_sel	 = (i2s->refclk / (2 * bclk)) % 2 ;
	fra_base = 0xff;
	fra_div = (unsigned int)div64_u64(((clk_div - integer_num * 2 * 100000  - clk_sel * 100000) * 255), CLKDIV_CALC_FACTOR);

	clkdiv = MIX_CLK_DIVIDER(fra_div, fra_base, integer_num, clk_sel) | (1 << 28);

	zx_write_i2s_reg(clkdiv, i2s->reg_clkdiv1);
	/*
		print_audio("Alsa zx29_i2s_set_clkdiv ZX_I2S0_CRM_CLKSEL=%x \n", zx_read_i2s_reg(i2s->reg_clksel));
		print_audio("Alsa zx29_i2s_set_clkdiv ZX_I2S0_CLKDIV_1=%x \n", zx_read_i2s_reg(i2s->reg_clkdiv1));
	*/
}

#elif defined(CONFIG_SND_SOC_ZX297520V3) || (defined CONFIG_SND_SOC_ZX297520V3_MODULE)
static int zx29_i2s_set_clkdiv(struct snd_soc_dai *dai, int rate, int width)
{
	unsigned int val;
	int integer, denominator, numerator;
	struct zx29_i2s *i2s = snd_soc_dai_get_drvdata(dai);
//	print_audio("Alsa Entered func %s\n", __func__);

	integer = i2s->refclk / (2 * rate * width * 2);
	denominator = rate * width * 2 / 100;
	numerator = ((i2s->refclk / 2) % (2 * width * rate)) / 100;

	val = zx_read_i2s_reg(i2s->reg_clkdiv2);
	val &= (~(0x1 << 16));
	zx_write_i2s_reg(val, i2s->reg_clkdiv2);
	zx_write_i2s_reg(((denominator << 16) | numerator), i2s->reg_clkdiv1);
	zx_write_i2s_reg(integer, i2s->reg_clkdiv2);
	val = zx_read_i2s_reg(i2s->reg_clkdiv2);
	val |= (0x1 << 16);
	zx_write_i2s_reg(val, i2s->reg_clkdiv2);
	/*
		print_audio("Alsa zx29_i2s_set_clkdiv ZX_I2S0_CRM_CLKSEL=%x \n", zx_read_i2s_reg(i2s->reg_clksel));
		print_audio("Alsa zx29_i2s_set_clkdiv ZX_I2S0_CLKDIV_1=%x \n", zx_read_i2s_reg(i2s->reg_clkdiv1));
		print_audio("Alsa zx29_i2s_set_clkdiv ZX_I2S0_CLKDIV_2=%x \n", zx_read_i2s_reg(i2s->reg_clkdiv2));
	*/
	return 0;
}
#endif

static VOID zx29_i2s_ClkInit(struct snd_soc_dai *dai)
{
	struct zx29_i2s *i2s = snd_soc_dai_get_drvdata(dai);
	unsigned int val;
	if (i2s->i2s_active_count != 0) {
		print_audio("Alsa:zx29_i2s_ClkInit not run, i2s_active_count=%d \n", i2s->i2s_active_count);
		return;
	}
	//val = zx_read_i2s_reg(i2s->reg_clksel);
	//val &= ~(3 << 8);
	//zx_write_i2s_reg(val, i2s->reg_clksel);

	val = zx_read_i2s_reg(i2s->reg_clksel);
	val |= (3 << 8);
	zx_write_i2s_reg(val, i2s->reg_clksel);

	udelay(10);

	val = zx_read_i2s_reg(i2s->reg_clksel);
	val |= ((3 << 0) | (3 << 10));
	zx_write_i2s_reg(val, i2s->reg_clksel);
	print_audio("Alsa:zx29_i2s_ClkInit run end \n");

}
static VOID zx29_i2s_ClkDeinit(struct snd_soc_dai *dai)
{
	struct zx29_i2s *i2s = snd_soc_dai_get_drvdata(dai);
	unsigned int val;
	if (i2s->i2s_active_count != 0) {
		print_audio("Alsa:zx29_i2s_ClkDeinit not run , i2s_active_count=%d \n", i2s->i2s_active_count);
		return;
	}
	val = zx_read_i2s_reg(i2s->reg_clksel);
	val &= ~((3 << 0) | (3 << 10));
	zx_write_i2s_reg(val, i2s->reg_clksel);

	val = zx_read_i2s_reg(i2s->reg_clksel);
	val &= ~(3 << 8);
	zx_write_i2s_reg(val, i2s->reg_clksel);

	val = zx_read_i2s_reg(i2s->reg_clksel);
	val |= (3 << 8);
	zx_write_i2s_reg(val, i2s->reg_clksel);

	print_audio("Alsa:zx29_i2s_ClkDeinit run end \n");

}
static int zx29_i2s_set_sysclk(struct snd_soc_dai *dai,
                               int clk_id, unsigned int freq, int dir)
{
	struct zx29_i2s *i2s = snd_soc_dai_get_drvdata(dai);
	unsigned int val;
//	print_audio("Alsa Entered func %s\n", __func__);

	switch (clk_id) {
	case ZX29_I2S_WCLK_SEL:
		if (dir == SND_SOC_CLOCK_IN) {
			//clk_set_rate(i2s->dai_wclk, freq); // 26M 104M 122.88M
			val = zx_read_i2s_reg(i2s->reg_clksel);
#if (defined CONFIG_SND_SOC_ZX297520V3) || (defined CONFIG_SND_SOC_ZX297520V3_MODULE)
			val &= (~(3 << 4));
			if (freq == ZX29_I2S_WCLK_FREQ_26M) {
				val |= (0 << 4);
			} else if (freq == ZX29_I2S_WCLK_FREQ_104M) {
				val |= (2 << 4);
			} else if (freq == ZX29_I2S_WCLK_FREQ_122M88) {
				val |= (1 << 4);
			}

#elif (defined CONFIG_SND_SOC_ZX297520V2) || (defined CONFIG_SND_SOC_ZX297520V2_MODULE)
			val &= (~(1 << 4));
			if (freq == ZX29_I2S_WCLK_FREQ_26M) {
				val |= (0 << 4);
			} else if (freq == ZX29_I2S_WCLK_FREQ_104M) {
				val |= (1 << 4);
			}
#endif
			zx_write_i2s_reg(val, i2s->reg_clksel);
			i2s->refclk = freq;
		}
		break;
	default:
		break;
	}

	return 0;
}
static int zx29_i2s_set_fmt(struct snd_soc_dai *cpu_dai, unsigned int fmt)
{
	struct zx29_i2s *i2s = snd_soc_dai_get_drvdata(cpu_dai);
	unsigned long val;
//	print_audio("Alsa Entered func %s\n", __func__);

	val = zx_read_i2s_reg(i2s->reg_base + ZX29_I2S_TIMING_CTRL);
	val &= ~(ZX29_I2S_TIMING_TIMING_MASK | ZX29_I2S_TIMING_ALIGN_MASK |
	         ZX29_I2S_TIMING_TEAK_MASK | ZX29_I2S_TIMING_SYNC_MASK |
	         ZX29_I2S_TIMING_MS_MASK);

	switch (fmt & SND_SOC_DAIFMT_FORMAT_MASK) {
	case SND_SOC_DAIFMT_I2S:
		val |= (ZX29_I2S_TIMING_I2S | ZX29_I2S_TIMING_STD_I2S);
		break;
	case SND_SOC_DAIFMT_LEFT_J:
		val |= (ZX29_I2S_TIMING_I2S | ZX29_I2S_TIMING_MSB_JUSTIF);
		break;
	case SND_SOC_DAIFMT_RIGHT_J:
		val |= (ZX29_I2S_TIMING_I2S | ZX29_I2S_TIMING_LSB_JUSTIF);
		break;
	default:
		dev_err(cpu_dai->dev, "Alsa Unknown i2s timeing\n");
		return -EINVAL;
	}

	switch (fmt & SND_SOC_DAIFMT_MASTER_MASK) {
	case SND_SOC_DAIFMT_CBM_CFM:
		/* Codec is master, and I2S is slave. */
		i2s->master = 0;
		val |= ZX29_I2S_TIMING_SLAVE;
		break;
	case SND_SOC_DAIFMT_CBS_CFS:
		/* Codec is slave, and I2S is master. */
		i2s->master = 1;
		val |= ZX29_I2S_TIMING_MAST;
		break;
	default:
		dev_err(cpu_dai->dev, "Alsa Unknown master/slave format\n");
		return -EINVAL;
	}

	zx_write_i2s_reg(val, i2s->reg_base + ZX29_I2S_TIMING_CTRL);
	return 0;
}


static int zx29_i2s_hw_params(struct snd_pcm_substream *substream,
                              struct snd_pcm_hw_params *params,
                              struct snd_soc_dai *socdai)
{

	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct zx29_i2s *i2s = snd_soc_dai_get_drvdata(socdai);
	unsigned int lane, ch_num, len, ret = 0;
	unsigned long val;
	unsigned long chn_cfg;
//	print_audio("Alsa Entered func %s\n", __func__);

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK)
		snd_soc_dai_set_dma_data(rtd->cpu_dai, substream, &(i2s->dma_playback));
	else
		snd_soc_dai_set_dma_data(rtd->cpu_dai, substream, &(i2s->dma_capture));

	val = zx_read_i2s_reg(i2s->reg_base + ZX29_I2S_TIMING_CTRL);
	val &= ~(ZX29_I2S_TIMING_TS_WIDTH_MASK | ZX29_I2S_TIMING_DATA_SIZE_MASK |
	         ZX29_I2S_TIMING_LANE_MASK | ZX29_I2S_TIMING_CHN_MASK |
	         ZX29_I2S_TIMING_TSCFG_MASK);

	print_audio("Alsa zx29_i2s_hw_params format=%d,channels=%d,rate=%d\n", params_format(params), params_channels(params), params_rate(params));

	switch (params_format(params)) {
	case SNDRV_PCM_FORMAT_S16_LE:
		len = 16;
		break;
	case SNDRV_PCM_FORMAT_S24_LE:
		len = 24;
		break;
	case SNDRV_PCM_FORMAT_S32_LE:
		len = 32;
		break;
	default:
		dev_err(socdai->dev, "Alsa Unknown data format %d\n", params_format(params));
		return -EINVAL;
	}
	val |= ZX29_I2S_TIMING_TS_WIDTH(len) | ZX29_I2S_TIMING_DATA_SIZE(len);

	ch_num = params_channels(params);
	switch (ch_num) {
	case 1:
		lane = 1;
		chn_cfg = 2;
		break;
	case 2:
	case 4:
	case 6:
	case 8:
		lane = ch_num / 2;
		chn_cfg = 3;
		break;
	default:
		dev_err(socdai->dev, "Alsa Not support channel num %d\n", ch_num);
		return -EINVAL;
	}
	val |= ZX29_I2S_TIMING_LANE(lane);
	val |= ZX29_I2S_TIMING_TSCFG(chn_cfg);
	val |= ZX29_I2S_TIMING_CHN(ch_num);
	zx_write_i2s_reg(val, i2s->reg_base + ZX29_I2S_TIMING_CTRL);

	if (i2s->master) {
		zx29_i2s_set_clkdiv(socdai, params_rate(params), len);
	}
	return ret;
}

#ifdef CONFIG_PM
static void zx29_i2s_set_active(struct zx29_i2s *i2s)
{
//	unsigned long flags;
//	local_irq_save(flags);
	mutex_lock(&i2s->i2s_mutex);
#ifdef CONFIG_SND_SOC_ES8312
	if (i2s->i2s_active_count == 0) {
		zx_cpuidle_set_busy(IDLE_FLAG_ALSA_I2S);
		wake_lock(&i2s->pm_lock);
		i2s->i2s_active_count++;
	}

#else
	if (i2s->i2s_active_count == 0) {
		zx_cpuidle_set_busy(IDLE_FLAG_ALSA_I2S);
		wake_lock(&i2s->pm_lock);
	}
	i2s->i2s_active_count++;
#endif
	mutex_unlock(&i2s->i2s_mutex);

//	local_irq_restore(flags);
}

static void zx29_i2s_set_idle(struct zx29_i2s *i2s)
{
//	unsigned long flags;
//	local_irq_save(flags);
	mutex_lock(&i2s->i2s_mutex);
#ifdef CONFIG_SND_SOC_ES8312
	if (i2s->i2s_active_count != 0) {
		i2s->i2s_active_count--;
		zx_cpuidle_set_free(IDLE_FLAG_ALSA_I2S);
		wake_unlock(&i2s->pm_lock);
	}
#else
	i2s->i2s_active_count--;
	if (i2s->i2s_active_count == 0) {
		zx_cpuidle_set_free(IDLE_FLAG_ALSA_I2S);
		wake_unlock(&i2s->pm_lock);
	}
#endif
	mutex_unlock(&i2s->i2s_mutex);
//	local_irq_restore(flags);
}
#endif


static int zx29_i2s_trigger(struct snd_pcm_substream *substream, int cmd,
                            struct snd_soc_dai *dai)
{
	struct zx29_i2s *i2s = dev_get_drvdata(dai->dev);
	int capture = (substream->stream == SNDRV_PCM_STREAM_CAPTURE);
	int ret = 0;
//	print_audio("Alsa Entered func %s,cmd=%d\n", __func__, cmd);

	unsigned int reg_val;

	switch (cmd) {
	case SNDRV_PCM_TRIGGER_START:
#ifdef CONFIG_PM
		zx29_i2s_set_active(i2s);
#endif
		if (capture)
			zx29_i2s_rx_dma_en(i2s->reg_base, true);
		else
			zx29_i2s_tx_dma_en(i2s->reg_base, true);
		/* fall thru */
	case SNDRV_PCM_TRIGGER_RESUME:
	case SNDRV_PCM_TRIGGER_PAUSE_RELEASE:
		if (capture)
			zx29_i2s_rx_en(i2s->reg_base, true);
		else
			zx29_i2s_tx_en(i2s->reg_base, true);
		zx29_i2s_en(i2s->reg_base, true);
		break;

	case SNDRV_PCM_TRIGGER_STOP:
		if (capture)
			zx29_i2s_rx_dma_en(i2s->reg_base, false);
		else
			zx29_i2s_tx_dma_en(i2s->reg_base, false);
#ifndef CONFIG_SND_SOC_ES8312
#ifdef CONFIG_PM
		zx29_i2s_set_idle(i2s);
#endif
#endif
		/* fall thru */
	case SNDRV_PCM_TRIGGER_SUSPEND:
	case SNDRV_PCM_TRIGGER_PAUSE_PUSH:
		if (capture)
			zx29_i2s_rx_en(i2s->reg_base, false);
		else
			zx29_i2s_tx_en(i2s->reg_base, false);
		//if (dai->capture_active && dai->playback_active)
		if (i2s->i2s_active_count != 0) {
			print_audio("Alsa:i2s_active_count=%d abnormal\n", i2s->i2s_active_count);
			break;
		}
#ifndef CONFIG_SND_SOC_ES8312
		zx29_i2s_en(i2s->reg_base, false);
#endif
		break;

	default:
		ret = -EINVAL;
		break;
	}
	/*
		print_audio("Alsa zx29_i2s_trigger ZX29_I2S_PROCESS_CTRL=%x\n", \
			zx_read_i2s_reg(i2s->reg_base + ZX29_I2S_PROCESS_CTRL));
		print_audio("Alsa zx29_i2s_trigger ZX29_I2S_TIMING_CTRL=%x\n", \
			zx_read_i2s_reg(i2s->reg_base + ZX29_I2S_TIMING_CTRL));
		print_audio("Alsa zx29_i2s_trigger ZX29_I2S_FIFO_CTRL=%x\n",  \
			zx_read_i2s_reg(i2s->reg_base + ZX29_I2S_FIFO_CTRL));
		print_audio("Alsa zx29_i2s_trigger ZX29_I2S_FIFO_STATUS=%x\n",  \
			zx_read_i2s_reg(i2s->reg_base + ZX29_I2S_FIFO_STATUS));
	*/
	return ret;
}

static int zx29_i2s_startup(struct snd_pcm_substream *substream,
                            struct snd_soc_dai *dai)
{
	struct zx29_i2s *i2s = dev_get_drvdata(dai->dev);
//	print_audio("Alsa Entered func %s, stream=%d\n", __func__, substream->stream);
	zx29_i2s_ClkInit(dai);

	clk_enable(i2s->dai_wclk);
	clk_enable(i2s->dai_pclk);

	clk_set_auto_gate(i2s->dai_wclk, true);
	clk_set_auto_gate(i2s->dai_pclk, true);
#ifdef CONFIG_SND_SOC_ES8312
	zx29_i2s_en(i2s->reg_base, false);
#endif
	return 0;
}

static void zx29_i2s_shutdown(struct snd_pcm_substream *substream,
                              struct snd_soc_dai *dai)
{
	struct zx29_i2s *i2s = dev_get_drvdata(dai->dev);
//	print_audio("Alsa Entered func %s\n", __func__);
	unsigned int val;
	if (dai->active) {
		print_audio("Alsa dai->active not null 0x%x\n", dai->active);
		return;
	}
#ifndef CONFIG_SND_SOC_ES8312
	clk_disable(i2s->dai_wclk);
	clk_disable(i2s->dai_pclk);
	clk_set_auto_gate(i2s->dai_wclk, false);
	clk_set_auto_gate(i2s->dai_pclk, false);
#if 0
	val = zx_read_i2s_reg(i2s->reg_clksel);
	val &= ~(3 << 8);
	zx_write_i2s_reg(val, i2s->reg_clksel);
	val |= (3 << 8);
	zx_write_i2s_reg(val, i2s->reg_clksel);
#endif
	zx29_i2s_ClkDeinit(dai);
#endif
}


#ifdef CONFIG_PM
static int zx29_i2s_suspend(struct snd_soc_dai *dai)
{
	struct zx29_i2s *i2s = snd_soc_dai_get_drvdata(dai);
//	print_audio("Alsa Entered func %s\n", __func__);

	if (dai->active) {
		i2s->suspend_i2stimingctrl = zx_read_i2s_reg(i2s->reg_base + ZX29_I2S_TIMING_CTRL);
		i2s->suspend_i2sfifoctrl = zx_read_i2s_reg(i2s->reg_base + ZX29_I2S_FIFO_CTRL);
		i2s->suspend_i2sprocessctrl = zx_read_i2s_reg(i2s->reg_base + ZX29_I2S_PROCESS_CTRL);

		i2s->suspend_i2sclksel = zx_read_i2s_reg(i2s->reg_clksel);
		i2s->suspend_i2sclkdiv1 = zx_read_i2s_reg(i2s->reg_clkdiv1);
#if (defined CONFIG_SND_SOC_ZX297520V3) || (defined CONFIG_SND_SOC_ZX297520V3_MODULE)
		i2s->suspend_i2sclkdiv2 = zx_read_i2s_reg(i2s->reg_clkdiv2);
#endif
		clk_disable(i2s->dai_wclk);
		clk_disable(i2s->dai_pclk);
	}

	i2s->suspend_active = dai->active;

	return 0;
}

static int zx29_i2s_resume(struct snd_soc_dai *dai)
{
	struct zx29_i2s *i2s = snd_soc_dai_get_drvdata(dai);
//	print_audio("Alsa Entered func %s\n", __func__);

	if (dai->active) {
		clk_enable(i2s->dai_pclk);
		clk_enable(i2s->dai_wclk);

		clk_set_auto_gate(i2s->dai_wclk, true);
		clk_set_auto_gate(i2s->dai_pclk, true);

		if (i2s->suspend_active) {
			zx_write_i2s_reg(i2s->suspend_i2sclksel, i2s->reg_clksel);
			zx_write_i2s_reg(i2s->suspend_i2sclkdiv1, i2s->reg_clkdiv1);
#if (defined CONFIG_SND_SOC_ZX297520V3) || (defined CONFIG_SND_SOC_ZX297520V3_MODULE)
			zx_write_i2s_reg(i2s->suspend_i2sclkdiv2, i2s->reg_clkdiv2);
#endif
			zx_write_i2s_reg(i2s->suspend_i2stimingctrl, i2s->reg_base + ZX29_I2S_TIMING_CTRL);
			zx_write_i2s_reg(i2s->suspend_i2sfifoctrl, i2s->reg_base + ZX29_I2S_FIFO_CTRL);
			zx_write_i2s_reg(i2s->suspend_i2sprocessctrl, i2s->reg_base + ZX29_I2S_PROCESS_CTRL);
		}
	}

	return 0;
}
#else
#define zx29_i2s_suspend NULL
#define zx29_i2s_resume  NULL
#endif

static int zx29_i2s_dai_probe1(struct snd_soc_dai *dai)
{
	return 0;
}
static int zx29_i2s_dai_probe(struct snd_soc_dai *dai)
{
	struct zx29_i2s *i2s = dev_get_drvdata(dai->dev);
#ifdef CONFIG_SND_SOC_ES8312
	zx29_i2s = i2s;
#endif
	snd_soc_dai_set_drvdata(dai, i2s);
	i2s->dma_playback.dma_addr = i2s->mapbase + ZX29_I2S_DATA;
	i2s->dma_capture.dma_addr = i2s->mapbase + ZX29_I2S_DATA;
//	print_audio("Alsa zx29_i2s_dai_probe dai->id=%d\n", dai->id);
	if (dai->id == 1) {
		i2s->dma_playback.channel = DMA_CH_I2S1_TX;
		i2s->dma_capture.channel = DMA_CH_I2S1_RX0;
	} else {
		i2s->dma_playback.channel = DMA_CH_I2S0_TX;
		i2s->dma_capture.channel = DMA_CH_I2S0_RX0;
	}
//	snd_soc_dai_init_dma_data(dai, &zx_i2s->dma_playback,
//				  &zx_i2s->dma_capture);
	return 0;
}

static const struct snd_soc_dai_ops zx29_i2s_dai_ops = {
	.trigger	= zx29_i2s_trigger,
	.hw_params	= zx29_i2s_hw_params,
	.set_fmt	= zx29_i2s_set_fmt,
//	.set_clkdiv = zx29_i2s_set_clkdiv,
	.set_sysclk = zx29_i2s_set_sysclk,
	.startup	= zx29_i2s_startup,
	.shutdown	= zx29_i2s_shutdown,
};


static struct snd_soc_dai_driver zx_i2s_dais[] = {
	{
		.name = "MultiMedia",
		.probe	= zx29_i2s_dai_probe,
		.suspend = zx29_i2s_suspend,
		.resume = zx29_i2s_resume,
		.playback   = {
			.stream_name = "MultiMedia Playback",
			.channels_min	= 1,
			.channels_max	= 8,
			.rates		= ZX29_I2S_RATES,
			.formats	= ZX29_I2S_FMTBIT,
		},
		.capture = {
			.stream_name = "MultiMedia Capture",
			.channels_min	= 1,
			.channels_max	= 2,
			.rates		= ZX29_I2S_RATES,
			.formats	= ZX29_I2S_FMTBIT,
		},
		.ops	= &zx29_i2s_dai_ops,
	},
	{
		.name = "voice",
		.probe = zx29_i2s_dai_probe1,
		.playback = {
			.stream_name = "voice Playback",
			.rates = ZX29_I2S_RATES,
			.formats = ZX29_I2S_FMTBIT,
			.channels_min = 1,
			.channels_max = 2,
		},
		.capture = {
			.stream_name = "voice Capture",
			.rates = ZX29_I2S_RATES,
			.formats = ZX29_I2S_FMTBIT,
			.channels_min = 1,
			.channels_max = 2,
		},
	}
};

static int zx29_i2s_init_clks(struct platform_device *pdev, struct zx29_i2s *i2s)
{
	/* find the clock and enable it */
	i2s->dai_wclk = clk_get(&pdev->dev, "work_clk");
	if (IS_ERR(i2s->dai_wclk)) {
		dev_err(&pdev->dev, "Alsa Fail to get wclk\n");
		return PTR_ERR(i2s->dai_wclk);
	}

	i2s->dai_pclk = clk_get(&pdev->dev, "apb_clk");
	if (IS_ERR(i2s->dai_pclk)) {
		dev_err(&pdev->dev, "Alsa Fail to get pclk\n");
		return PTR_ERR(i2s->dai_pclk);
	}

	switch (i2s->i2s_num) {
	case 0:
		i2s->reg_clksel = ZX_I2S0_CRM_CLKSEL;
		i2s->reg_clkdiv1 = ZX_I2S0_CLKDIV_1;
#if (defined CONFIG_SND_SOC_ZX297520V3) || (defined CONFIG_SND_SOC_ZX297520V3_MODULE)
		i2s->reg_clkdiv2 = ZX_I2S0_CLKDIV_2;
#endif
		break;
	case 1:
		i2s->reg_clksel = ZX_I2S1_CRM_CLKSEL;
		i2s->reg_clkdiv1 = ZX_I2S1_CLKDIV_1;
#if (defined CONFIG_SND_SOC_ZX297520V3) || (defined CONFIG_SND_SOC_ZX297520V3_MODULE)
		i2s->reg_clkdiv2 = ZX_I2S1_CLKDIV_2;
#endif
		break;
	}

	return 0;
}


static __devinit int zx29_i2s_dev_probe(struct platform_device *pdev)
{

	struct resource *res;
	struct zx29_i2s *zx_i2s;

	int ret;
//	print_audio("Alsa zx29_i2s_dev_probe start\n");

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (res == NULL) {
		printk("Alsa: No I/O resource!\n");
		return -EINVAL;
	}

	zx_i2s = devm_kzalloc(&pdev->dev, sizeof(*zx_i2s), GFP_KERNEL);
	if (!zx_i2s)
		return -ENOMEM;

	zx_i2s->reg_base = res->start;
	zx_i2s->mapbase =  pdev->id ? ZX29_I2S1_PHYS : ZX29_I2S0_PHYS;
	zx_i2s->i2s_num = pdev->id;
	if (IS_ERR(zx_i2s->reg_base)) {
		dev_err(&pdev->dev, "Alsa ioremap failed!\n");
		return PTR_ERR(zx_i2s->reg_base);
	}
	platform_set_drvdata(pdev, zx_i2s);

//	zx29_i2s_config_gpio(pdev->id);

	zx29_i2s_sysfs_create_group(pdev);

	ret = zx29_i2s_init_clks(pdev, zx_i2s);
#ifdef CONFIG_PM
	zx_i2s->i2s_active_count = 0;
	mutex_init(&zx_i2s->i2s_mutex);
	wake_lock_init(&zx_i2s->pm_lock, WAKE_LOCK_SUSPEND, "zx29-i2s");
#endif
	return snd_soc_register_dais(&pdev->dev, zx_i2s_dais, ARRAY_SIZE(zx_i2s_dais));
//	return snd_soc_register_dai(&pdev->dev, &zx_i2s_dai);
}

static __devexit int zx29_i2s_dev_remove(struct platform_device *pdev)
{
	struct zx29_i2s *i2s = platform_get_drvdata(pdev);
	clk_disable(i2s->dai_wclk);
	clk_put(i2s->dai_wclk);
	clk_disable(i2s->dai_pclk);
	clk_put(i2s->dai_pclk);

	zx29_i2s_sysfs_remove_group(pdev);

	snd_soc_unregister_dai(&pdev->dev);
#ifdef CONFIG_PM
	wake_lock_destroy(&i2s->pm_lock);
#endif
	return 0;
}

static struct platform_driver zx29_i2s_driver = {
	.probe  = zx29_i2s_dev_probe,
	.remove = __devexit_p(zx29_i2s_dev_remove),
	.driver = {
		.name = "zx29_i2s",
		.owner = THIS_MODULE,
	},
};
#ifdef CONFIG_SND_SOC_ES8312

void zx29_i2s_close_clk(void)
{
	unsigned int val = 0;
//	print_audio("Alsa Entered func %s\n", __func__);
	zx29_i2s_en(zx29_i2s->reg_base, false);

	clk_disable(zx29_i2s->dai_wclk);
	clk_disable(zx29_i2s->dai_pclk);
	clk_set_auto_gate(zx29_i2s->dai_wclk, false);
	clk_set_auto_gate(zx29_i2s->dai_pclk, false);

	val = zx_read_i2s_reg(zx29_i2s->reg_clksel);
	val &= ~(3 << 8);
	zx_write_i2s_reg(val, zx29_i2s->reg_clksel);
	val |= (3 << 8);
	zx_write_i2s_reg(val, zx29_i2s->reg_clksel);
#ifdef CONFIG_PM
	zx29_i2s_set_idle(zx29_i2s);
#endif
}

#endif
module_platform_driver(zx29_i2s_driver);

/* Module information */
MODULE_DESCRIPTION("zx29 I2S SoC Interface");
MODULE_LICENSE("GPL");
MODULE_ALIAS("platform:zx29-i2s");

