/****************************************************************************************/
/*  Copyright (c) 2004-2014 NXP Software. All rights are reserved.                      */
/*  Reproduction in whole or in part is prohibited without the prior                    */
/*  written consent of the copyright owner.                                             */
/*                                                                                      */
/*  This software and any compilation or derivative thereof is and                      */
/*  shall remain the proprietary information of NXP Software and is                     */
/*  highly confidential in nature. Any and all use hereof is restricted                 */
/*  and is subject to the terms and conditions set forth in the                         */
/*  software license agreement concluded with NXP Software.                             */
/*                                                                                      */
/*  Under no circumstances is this software or any derivative thereof                   */
/*  to be combined with any Open Source Software in any way or                          */
/*  licensed under any Open License Terms without the express prior                     */
/*  written  permission of NXP Software.                                                */
/*                                                                                      */
/*  For the purpose of this clause, the term Open Source Software means                 */
/*  any software that is licensed under Open License Terms. Open                        */
/*  License Terms means terms in any license that require as a                          */
/*  condition of use, modification and/or distribution of a work                        */
/*                                                                                      */
/*           1. the making available of source code or other materials                  */
/*              preferred for modification, or                                          */
/*                                                                                      */
/*           2. the granting of permission for creating derivative                      */
/*              works, or                                                               */
/*                                                                                      */
/*           3. the reproduction of certain notices or license terms                    */
/*              in derivative works or accompanying documentation, or                   */
/*                                                                                      */
/*           4. the granting of a royalty-free license to any party                     */
/*              under Intellectual Property Rights                                      */
/*                                                                                      */
/*  regarding the work and/or any work that contains, is combined with,                 */
/*  requires or otherwise is based on the work.                                         */
/*                                                                                      */
/*  This software is provided for ease of recompilation only.                           */
/*  Modification and reverse engineering of this software are strictly                  */
/*  prohibited.                                                                         */
/*                                                                                      */
/****************************************************************************************/

/****************************************************************************************/
/*                                                                                      */
/*     $Author: nxp65285 $*/
/*     $Revision: 56908 $*/
/*     $Date: 2014-04-10 17:30:02 +0530 (Thu, 10 Apr 2014) $*/
/*                                                                                      */
/****************************************************************************************/

/**
@file
Header file for the application layer interface of the LVVE module.
This files includes all definitions, types, structures and function prototypes
required by the calling layer. All other types, structures and functions are
private.
*/

#ifndef __LVVE_H__
#define __LVVE_H__

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/****************************************************************************************/
/*                                                                                      */
/*  Includes                                                                            */
/*                                                                                      */
/****************************************************************************************/

#include "LVVE_VID.h"

/****************************************************************************************/
/*                                                                                      */
/*  Definitions                                                                         */
/*                                                                                      */
/****************************************************************************************/
/* Below include logic is designed to reduce checks to 1 or 2 conditions due to limitations in the backend stripping code */





#define LVVE_MAX_PCM_FRAME_SIZE         160                    ///< LVVE maximum processing frame size


#define LVVE_NOISESAMPLES_PER_FRAME     2                           ///< Number of samples required for noise estimation per frame
/**
@def LVVE_RX_PRESET_LENGTH
Length of the LVVE_Rx preset buffer (number of bytes).
@see LVVE_Rx_SetPreset
@ingroup LVVE_Rx
*/
#define LVVE_RX_PRESET_LENGTH (LVVIDHEADER_CONTROLPARAMS_LVWIREFORMAT_LENGTH + LVVE_RX_CONTROLPARAMS_LVWIREFORMAT_LENGTH) ///< RX preset Buffer length

/**
@def LVVE_TX_PRESET_LENGTH
Length of the LVVE_Tx preset buffer (number of bytes).
@see LVVE_Tx_SetPreset
@ingroup LVVE_Tx
*/
#define LVVE_TX_PRESET_LENGTH (LVVIDHEADER_CONTROLPARAMS_LVWIREFORMAT_LENGTH + LVVE_TX_CONTROLPARAMS_LVWIREFORMAT_LENGTH) ///< TX preset buffer length


/* Deprecated Definitions */

#define LVVE_MAX_BULK_DELAY             LVVE_TX_BULKDELAY_MAX
#define LVVE_RX_MODE_DUMMY LVVE_RX_MODE_EN_DUMMY
#define LVVE_TX_MODE_DUMMY LVVE_TX_MODE_EN_DUMMY

#define LVVE_MAX_VOL_GAIN_DB            LVVE_TX_VOL_GAIN_MAX
#define LVVE_MIN_VOL_GAIN_DB            LVVE_TX_VOL_GAIN_MIN

#define LVVE_MAX_HPF_CORNER_HZ          LVVE_TX_MIC_HPF_CORNERFREQ_MAX
#define LVVE_MIN_HPF_CORNER_HZ          LVVE_TX_MIC_HPF_CORNERFREQ_MIN

#define LVVE_MAX_HPF_RX_CORNER_HZ       LVVE_RX_HPF_CORNERFREQ_MAX

#define LVVE_MAX_NLPP_LIMIT_DB          LVNLPP_NLPP_LIMIT_MAX
#define LVVE_MIN_NLPP_LIMIT_DB          LVNLPP_NLPP_LIMIT_MIN


#define LVVE_MAX_DRC_LEVEL              LVDRC_COMPRESSORCURVEINPUTLEVELS_MAX
#define LVVE_MIN_DRC_LEVEL              LVDRC_COMPRESSORCURVEINPUTLEVELS_MIN
#define LVVE_MIN_DRC_NUMKNEES           LVDRC_NUMKNEES_MIN
#define LVVE_MAX_DRC_NUMKNEES           LVDRC_NUMKNEES_MAX
#define LVVE_MIN_DRC_ATTACKTIME         LVDRC_ATTACKTIME_MIN
#define LVVE_MAX_DRC_ATTACKTIME         LVDRC_ATTACKTIME_MAX
#define LVVE_MIN_DRC_RELEASETIME        LVDRC_RELEASETIME_MIN
#define LVVE_MAX_DRC_RELEASETIME        LVDRC_RELEASETIME_MAX


/* End deprecated defines*/
/****************************************************************************************/
/*                                                                                      */
/*  Types                                                                               */
/*                                                                                      */
/****************************************************************************************/

/**
LVVE_Tx Instance Handle
This handle is used by most of the LVVE APIs
@see LVVE_Tx_GetInstanceHandle
@ingroup LVVE_Tx
*/
typedef void *LVVE_Tx_Handle_t;   ///< LVVE Tx handle
/**
This handle is used by most of the LVVE APIs
@see LVVE_Rx_GetInstanceHandle
@ingroup LVVE_Rx
*/
typedef void *LVVE_Rx_Handle_t;  ///< LVVE Rx handle

/**
This enum type specifies the different error codes returned by the API functions
For the exact meaning see the individual function descriptions
*/
typedef enum
{
    LVVE_SUCCESS                            = 0,                ///< Successful return from a routine
    LVVE_ALIGNMENTERROR                     = 1,                ///< Memory alignment error
    LVVE_NULLADDRESS                        = 2,                ///< NULL allocation address
    LVVE_OUTOFRANGE                         = 3,                ///< Out of range parameter
    LVVE_INVALIDNUMSAMPLES                  = 4,                ///< Invalid number of samples
    LVVE_INVALID_ALGORITHM_CONFIGURATION    = 5,                ///< Mutually exclusive algorithms configured ON
    LVVE_INVALID_STATE_CONFIGURATION        = 6,                ///< Invalid state of the algorithm
    LVVE_PRESET_INVALID_BUFFER_LENGTH       = 7,                ///< Incorrect length of buffer used in SetPreset
    LVVE_PRESET_INVALID_VOLUME_INDEX        = 8,                ///< The volume index exceeds the buffer content in SetPreset
    LVVE_PRESET_INVALID_BUFFER_VERSION      = 9,                ///< The version of the preset buffer does not match this library
    LVVE_PRESET_INVALID_BASELINE_VERSION    = 10,               ///< Invalid LVVE Baseline Version in preset buffer
    LVVE_PRESET_INVALID_MASK                = 11,               ///< Invalid algorithm mask in preset buffer
    LVVE_PRESET_INVALID_SAMPLE_RATE         = 12,               ///< Invalid sample rate @ref LVM_Fs_en in preset buffer
    LVVE_PRESET_INVALID_LVWIREFORMAT_MESSAGEID = 13,            ///< Invalid @ref LVVIDHeader_MessageID_en wire format message id in preset buffer

    LVVE_RETURNSTATUS_DUMMY                 = LVM_MAXENUM
} LVVE_ReturnStatus_en;

/**
Byte array containing encoded LVVE_Rx_ControlParams for one or multiple volumes.
The length of this array should be a multiple of @ref LVVE_RX_PRESET_LENGTH.
@see LVVE_Rx_SetPreset
@ingroup LVVE_Rx
*/
typedef LVM_CHAR* LVVE_Rx_Preset_t; ///< LVVE Rx preset buffer

/**
Byte array containing encoded LVVE_Tx_ControlParams for one or multiple volumes.
The length of this array should be a multiple of @ref LVVE_TX_PRESET_LENGTH.
@see LVVE_Tx_SetPreset
@ingroup LVVE_Tx
*/
typedef LVM_CHAR* LVVE_Tx_Preset_t;  ///< LVVE Tx preset buffer


/****************************************************************************************/
/*                                                                                      */
/*  Structures                                                                          */
/*                                                                                      */
/****************************************************************************************/

/**
The instance parameters define certain important operating limits required by the calling application.
These instance parameters affect how much memory is required by the LVVE and hence must be provided
when the instance is created.
@see LVVE_Rx_GetMemoryTable
@see LVVE_Rx_GetInstanceHandle
@see LVVE_Tx_GetMemoryTable
@see LVVE_Tx_GetInstanceHandle
*/
/**
EQ Module Instance Parameters Structure.
@see LVEQ_InstanceParams_st
*/
typedef struct
{
/**
Max Size of Equalizer
Sets the maximum length of the equalizer impulse response that can be used.
It must be a multiple of 8.
<table border>
<caption>EQ Max Length Table</caption>
   <tr>
       <td><b>Unit</b></td>
       <td><b>Q format</b></td>
       <td><b>Data Range</b></td>
       <td><b>Default Values</b></td>
   </tr>
   <tr>
       <td><b>Integer Length in Samples</b></td>
       <td><b>Q16.0</b></td>
       <td>[8,\ref LVEQ_EQ_LENGTH_MAX]</td>
       <td>\ref LVEQ_EQ_LENGTH_DEFAULT</td>
   </tr>
</table>
*/
    LVM_UINT16                    EQ_MaxLength;  ///< EQ Max Length
} LVEQ_InstanceParams_st;

/**
Tx Instance Parameter Structure
These parameters are set at initialization time and may not be changed during processing
@ref LVVE_Tx_GetInstanceHandle
@ingroup LVVE_Tx
*/
typedef struct
{
    LVM_Fs_en                   SampleRate;      ///< Sample rate
/**
Sets the maximum length of the bulk delay between Rx and Tx expressed in samples.
The unit of MaxBulkDelay is [number of samples] at the respective sampling rate.
<table border>
<caption>Max Bulk Delay Table</caption>
   <tr>
       <td><b>Type</b></td>
       <td><b>Unit</b></td>
       <td><b>Q format</b></td>
       <td><b>Data Range</b></td>
       <td><b>Default Values</b></td>
   </tr>
   <tr>
       <td><b>LVM_UINT16</b></td>
       <td><b>Integer Length in Samples</b></td>
       <td><b>Q16.0</b></td>
       <td>[0,6400]</td>
       <td>None</td>
   </tr>
</table>
*/
    LVM_UINT16                  MaxBulkDelay;     ///< Max bulk delay

    LVEQ_InstanceParams_st      EQ_InstParams;  ///< EQ instance
} LVVE_Tx_InstanceParams_st;

/**
Rx Instance Parameter Structure
These parameters are set at initialization time and may not be changed during processing.
@ref LVVE_Rx_GetInstanceHandle
@ingroup LVVE_Rx
*/
typedef struct
{
    LVM_Fs_en                   SampleRate;            ///< LVVE sample rate
    LVEQ_InstanceParams_st      EQ_InstParams;///< EQ instance

} LVVE_Rx_InstanceParams_st;

/**
The version information structure contains one character field to store LVVE version number.
A call to the @ref LVVE_GetVersionInfo function is needed to retrieve this information.
*/
typedef struct
{
    LVM_CHAR                    VersionNumber[64];  ///< Version number of the LifeVibes&trade; VoiceExperience library
} LVVE_VersionInfo;

/**
Used to reset LVVE_Rx module any time.
@ref LVM_ResetType_en parameter determine the type of reset required
@ingroup LVVE_Rx
*/
typedef struct
{
    LVM_ResetType_en ResetType;  ///< RX Reset Type
} LVVE_Rx_ResetParams_st;

/**
Used to reset LVVE_Tx module any time.
LVM_ResetType_en parameter determine the type of reset required
@ingroup LVVE_Tx
*/
typedef struct
{
    LVM_ResetType_en ResetType;   ///< TX Reset Type
} LVVE_Tx_ResetParams_st;

/**
@brief Retrieve the memory requirements of the LVVE_Tx module.

This function returns a table of memory records that describe size, type and memory space of all buffers
required by the instance. The number of initialized memory records is defined by LVVE_NR_MEMORY_REGIONS.
This function is used for two purposes and is called in two different ways:

@li Memory Allocation: When the LVVE_GetMemoryTable functions are called with a NULL instance handle (hInstance = LVM_NULL)
    the function returns the memory requirements. The base address pointers in the memory table are set to NULL.
    All elements of the instance parameters structure (pointed to by pInstParams) must contain valid values as the memory
    requirements are affected by these settings.

@li Memory Free: When called with a non-NULL instance handle (hInstance = a valid instance handle)
      the function returns the memory table used when the instance was created.
      The base address pointers returned will be those supplied by the calling application when the memory
      was allocated and can now be used for freeing memory. The instance parameters (pointed to by pInstParams)
      are ignored and the pInstParams parameter may be set to NULL.
@li In case of memory allocation, all elements of the parameter initialization structure defined by pInstParams
    must contain valid values as the memory requirements are affected by these settings.
    In some releases of the bundle library one or more memory regions may have a zero size.

@pre The number of memory records in the array defined by pMemoryTable is equal to @ref LVM_NR_MEMORY_REGIONS.
@pre Exactly @ref LVM_NR_MEMORY_REGIONS memory records of pMemoryTable are initialized.

@post When this function is called with hInstance = NULL the memory base address pointers
will be NULL on return.
@post When the function is called for freeing memory, hInstance = Instance Handle the
memory table returns the allocated memory and base addresses used during
initialisation.

@return  LVVE_SUCCESS           when the function call succeeds and the memory table is filled correctly
@return  LVVE_NULLADDRESS       when pMemoryTable was NULL
@return  LVVE_NULLADDRESS       when pInstParams was NULL and the call was for memory allocation
                                is NULL.
@return  LVVE_OUTOFRANGE        when pInstParams contains parameters out of the excepted range

@note  This function may be interrupted by the LVVE_Tx_Process function.

@ingroup LVVE_Tx

LVVE_Tx_GetMemoryTable: Memory Allocation Example:

The following example shows how to get the memory requirements for the LVVE_Tx instance.
\code
    InstParams_Tx.SampleRate    = LVM_FS_8000;
    // Include the other instance params here

    LVVE_Status = LVVE_Tx_GetMemoryTable ( LVM_NULL,
                                           &MemTab_Tx,
                                           &InstParams_Tx );

    if( LVVE_Status != LVVE_SUCCESS )
    {
        // Handle Errors
    }

    for( c1 = 0; c1 < LVVE_NR_MEMORY_REGIONS; c1++ )
    {
        if( MemTab_Tx.Region[c1].Size != 0 )
        {
            MemTab_Tx.Region[c1].pBaseAddress =
                malloc(MemTab_Tx.Region[c1].Size);
        }
    }
\endcode

LVVE_Tx_GetMemoryTable: Freeing Memory Example:

The following example shows how to free the memory from the LVVE_Tx instance.

\code
    LVVE_Status = LVVE_Tx_GetMemoryTable( hInstance,
                                              &MemTab_Tx,
                                              LVM_NULL);

        if (LVVE_Status != LVVE_SUCCESS)
        {
            // Handle errors
        }

        for( c1 = 0; c1 < LVVE_NR_MEMORY_REGIONS; c1++)
        {
            if (MemTab_Tx.Region[c1].Size != 0)
            {
                free(MemTab_Tx.Region[c1].pBaseAddress);
            }
        }
\endcode

*/

LVVE_ReturnStatus_en LVVE_Tx_GetMemoryTable(LVVE_Tx_Handle_t               hInstance,
                                            LVM_MemoryTable_st             *pMemoryTable,
                                            LVVE_Tx_InstanceParams_st      *pInstanceParams);

/**
@brief Retrieve the memory requirements of the LVVE_Rx module.

This function returns a table of memory records that describe size, type and memory space of all buffers
required by the instance. The number of initialized memory records is defined by LVVE_NR_MEMORY_REGIONS.
This function is used for two purposes and is called in two different ways:

@li Memory Allocation: When the LVVE_GetMemoryTable functions are called with a NULL instance handle (hInstance = LVM_NULL)
    the function returns the memory requirements. The base address pointers in the memory table are set to NULL.
    All elements of the instance parameters structure (pointed to by pInstParams) must contain valid values as the memory
    requirements are affected by these settings.

@li Memory Free: When called with a non-NULL instance handle (hInstance = a valid instance handle)
      the function returns the memory table used when the instance was created.
      The base address pointers returned will be those supplied by the calling application when the memory
      was allocated and can now be used for freeing memory. The instance parameters (pointed to by pInstParams)
      are ignored and the pInstParams parameter may be set to NULL.
@li In case of memory allocation, all elements of the parameter initialization structure defined by pInstParams
    must contain valid values as the memory requirements are affected by these settings.
    In some releases of the bundle library one or more memory regions may have a zero size.

@pre The number of memory records in the array defined by pMemoryTable is equal to @ref LVM_NR_MEMORY_REGIONS.
@pre Exactly @ref LVM_NR_MEMORY_REGIONS memory records of pMemoryTable are initialized.

@post When this function is called with hInstance = NULL the memory base address pointers
will be NULL on return.
@post When the function is called for freeing memory, hInstance = Instance Handle the
memory table returns the allocated memory and base addresses used during
initialisation.

@return  LVVE_SUCCESS           when the function call succeeds and the memory table is filled correctly
@return  LVVE_NULLADDRESS       when pMemoryTable was NULL
@return  LVVE_NULLADDRESS       when pInstParams was NULL and the call was for memory allocation
                                is NULL.
@return  LVVE_OUTOFRANGE        when pInstParams contains parameters out of the excepted range

@note  This function may be interrupted by the LVVE_Rx_Process function.

@ingroup LVVE_Rx

LVVE_Rx_GetMemoryTable: Memory Allocation Example:

The following example shows how to get the memory requirements for the LVVE_Rx instance.

\code
    InstParams_Rx.SampleRate    = LVM_FS_8000;
    // Include the other instance params here

    LVVE_Status = LVVE_Rx_GetMemoryTable ( LVM_NULL,
                                           &MemTab_Rx,
                                           &InstParams_Rx );

    if( LVVE_Status != LVVE_SUCCESS )
    {
        // Handle Errors
    }

    for( c1 = 0; c1 < LVVE_NR_MEMORY_REGIONS; c1++ )
    {
        if( MemTab_Rx.Region[c1].Size != 0 )
        {
            MemTab_Rx.Region[c1].pBaseAddress =
                malloc(MemTab_Rx.Region[c1].Size);
        }
    }
\endcode

LVVE_Rx_GetMemoryTable: Freeing Memory Example:

The following example shows how to free the memory from the LVVE_Rx instance.

\code
    LVVE_Status = LVVE_Rx_GetMemoryTable( hInstance,
                                              &MemTab_Rx,
                                              LVM_NULL);

        if (LVVE_Status != LVVE_SUCCESS)
        {
            // Handle errors
        }

        for( c1 = 0; c1 < LVVE_NR_MEMORY_REGIONS; c1++)
        {
            if (MemTab_Rx.Region[c1].Size != 0)
            {
                free(MemTab_Rx.Region[c1].pBaseAddress);
            }
        }
\endcode
*/

LVVE_ReturnStatus_en LVVE_Rx_GetMemoryTable(LVVE_Rx_Handle_t               hInstance,
                                            LVM_MemoryTable_st             *pMemoryTable,
                                            LVVE_Rx_InstanceParams_st      *pInstanceParams);


/**
@brief Created handle to the instance of the LVVE_Tx module

This function is used to create the LVVE_Tx instance. All control parameters are set to invalid values.
The memory table pointed to by pMemoryTable must be created. If the memory table is not correct then an error will be returned.
The memory requirements of the Rx and Tx unit are dependent on the instance parameters supplied and so the
instance parameters provided in this function call must be the same as those used in the @ref LVVE_Tx_GetMemoryTable function calls
used for memory allocation.

@pre The memory records tables defined by pMemoryTable contains pointers to non-overlapping buffers
with the size as requested via the prior calls to the LVVE_Tx_GetMemoryTable functions.
@pre The initialization parameter structure defined by pInstParams is identical to the structure passed
to prior call to @ref LVVE_Tx_GetMemoryTable functions.


@param  phInstance              Pointer to the instance handle.
@param  pMemoryTable            Pointer to the memory definition table.
@param  pInstanceParams         Pointer to the instance parameters.

@return  LVVE_SUCCESS           when creation was successful
@return  LVVE_NULLADDRESS       When phInstance or pMemoryTable or pInstanceParams is NULL.
@return  LVVE_NULLADDRESS       when one or more of the memory regions with a non-zero size has been
                                given a NULL base address pointer.
@return LVVE_OUTOFRANGE         when pInstParams contains parameters out of the excepted range

@ingroup LVVE_Tx

 LVVE_Tx_GetInstanceHandle Example:

The following example shows how to initialize LVVE_Tx_GetInstanceHandle.

\code
    hInstance_Tx = LVM_NULL;
    LVVE_Status = LVVE_Tx_GetInstanceHandle( &hInstance_Tx,
                                             &MemTab_Tx,
                                             &InstParams_Tx );
    if( LVVE_Status != LVVE_SUCCESS )
    {
         // Handle Errors
    }
\endcode

*/

LVVE_ReturnStatus_en LVVE_Tx_GetInstanceHandle(LVVE_Tx_Handle_t                *phInstance,
                                               LVM_MemoryTable_st              *pMemoryTable,
                                               LVVE_Tx_InstanceParams_st       *pInstanceParams);

/**
@brief Created handle to the instance of the LVVE_Rx module

This functions is used to create LVVE_Rx instance. All control parameters are set to invalid values.
The memory table pointed to by pMemoryTable must be created. If the memory table is not correct then an error will be returned.
The memory requirements of the Rx and Rx unit are dependent on the instance parameters supplied and so the
instance parameters provided in this function call must be the same as those used in the @ref LVVE_Rx_GetMemoryTable function calls
used for memory allocation.

@pre The memory records tables defined by pMemoryTable contains pointers to non-overlapping buffers
with the size as requested via the prior calls to the LVVE_Rx_GetMemoryTable functions.
@pre The initialization parameter structure defined by pInstParams is identical to the structure passed
to prior call to @ref LVVE_Rx_GetMemoryTable functions.


@param  phInstance              Pointer to the instance handle.
@param  pMemoryTable            Pointer to the memory definition table.
@param  pInstanceParams         Pointer to the instance parameters.

@return  LVVE_SUCCESS           when creation was successful
@return  LVVE_NULLADDRESS       When phInstance or pMemoryTable or pInstanceParams is NULL.
@return  LVVE_NULLADDRESS       when one or more of the memory regions with a non-zero size has been
                                given a NULL base address pointer.
@return LVVE_OUTOFRANGE         when pInstParams contains parameters out of the excepted range

@ingroup LVVE_Rx

 LVVE_Rx_GetInstanceHandle Example:

The following example shows how to initialize LVVE_Rx_GetInstanceHandle.

\code
    hInstance_Rx = LVM_NULL;
    LVVE_Status = LVVE_Rx_GetInstanceHandle( &hInstance_Rx,
                                             &MemTab_Rx,
                                             &InstParams_Rx );
    if( LVVE_Status != LVVE_SUCCESS )
    {
         // Handle Errors
    }
\endcode

*/

LVVE_ReturnStatus_en LVVE_Rx_GetInstanceHandle(LVVE_Rx_Handle_t                *phInstance,
                                               LVM_MemoryTable_st              *pMemoryTable,
                                               LVVE_Rx_InstanceParams_st       *pInstanceParams);

/**
@brief Retrieve the current LVVE_Tx control parameters.

This function copies the control parameters from the LVVE_Tx into the supplied
parameter structure, pControlParams. The values returned are the values given in the last successful
call to the LVVE_Tx_SetControlParameters function.

@param hInstance              Instance handle
@param pControlParams         Pointer to the control parameters

@pre   hInstance should be valid handle.
@pre   pControlParams should be allocated by caller.
@post  pControlParams will be filled with the values given in the last successful call to
       the LVVE_Tx_SetControlParameters function. They are not necessarily the values
       used in the last call to the LVVE_Tx_Process function, this will be the case if
       LVVE_Tx_SetControlParameters has been called since the last call to LVVE_Tx_Process.

@return LVVE_SUCCESS          Succeeded
@return LVVE_NULLADDRESS      When hInstance or pControlParams is NULL

@note The LVVE_Tx_GetControlParameters function can be called at any time during processing.

@ingroup LVVE_Tx

LVVE_Tx_GetControlParameters Example:

The following example shows how to get different control parameters for the LVVE_Tx instance.

\code
    LVVE_Status = LVVE_Tx_GetControlParameters( &hInstance_Tx,
                                                &ControlParams_Tx );

    if( LVVE_Status != LVVE_SUCCESS )
    {
         // Handle Errors
    }
\endcode

*/

LVVE_ReturnStatus_en LVVE_Tx_GetControlParameters(   LVVE_Tx_Handle_t           hInstance,
                                                     LVVE_Tx_ControlParams_st   *pControlParams);

/**
@brief Retrieve the current LVVE_Rx control parameters.

This function copies the control parameters from the LVVE_Tx into the supplied
parameter structure, pControlParams. The values returned are the values given in the last successful
call to the LVVE_Rx_SetControlParameters function.

@param hInstance              Instance handle
@param pControlParams         Pointer to the control parameters

@pre   hInstance should be valid handle.
@pre   pControlParams should be allocated by caller.
@post  pControlParams will be filled with the values given in the last successful call to
       the LVVE_Rx_SetControlParameters function. They are not necessarily the values
       used in the last call to the LVVE_Rx_Process function, this will be the case if
       LVVE_Tx_SetControlParameters has been called since the last call to LVVE_Tx_Process.

@return LVVE_SUCCESS          Succeeded
@return LVVE_NULLADDRESS      When hInstance or pControlParams is NULL

@note The LVVE_Rx_GetControlParameters function can be called at any time during processing.

@ingroup LVVE_Rx

LVVE_Rx_GetControlParameters Example:

The following example shows how to get different control parameters for the LVVE_Rx instance.

\code
    LVVE_Status = LVVE_Rx_GetControlParameters( &hInstance_Rx,
                                                &ControlParams_Rx );

    if( LVVE_Status != LVVE_SUCCESS )
    {
         // Handle Errors
    }
\endcode

*/

LVVE_ReturnStatus_en LVVE_Rx_GetControlParameters(   LVVE_Rx_Handle_t           hInstance,
                                                     LVVE_Rx_ControlParams_st   *pControlParams);

/**
@brief Sets or changes the LVVE_Tx module parameters using C-strcuture.

This function takes the new set of parameters and makes a local copy within
LVVE_Tx but the parameters are only applied on the next call of the LVVE_Tx_Process
function. When a parameter is unchanged no action is taken. This function can
be called at any time during the processing, even when the LVVE_Tx_Process function
is running. LifeVibes&trade; VoiceExperience control parameters can be set using two methods.
Control Parameters can be populated in a \"C\" style structure (as explained in example) and then sent to the library.
Another method is to populate control parameters in a byte array called Preset Buffer, see @ref LVVE_Tx_SetPreset for example usage.

@param hInstance               Instance Handle
@param pNewParams              Pointer to a parameter structure

@pre hInstance should be valid handle.

@return LVVE_SUCCESS          Succeeded
@return LVVE_NULLADDRESS      When hInstance or pNewParams is NULL
@return LVVE_OUTOFRANGE       When pNewParams contains parameters of activated modules that are out of the excepted range
@return LVVE_INVALID_ALGORITHM_CONFIGURATION  When two mutual exclusive algorithms are both configured ON at run-time
                                               (e.g., NoiseVoid and HandsFree)


@note This function may be interrupted by the LVVE_Tx_Process function

@ingroup LVVE_Tx

LVVE_Tx_GetControlParameters: Setting Control Parameters using C-Style structure Example:

The following example shows how to set different control parameters for the LVVE_Tx instance.

\code
    LVVE_Status = LVVE_Tx_GetControlParameters( &hInstance_Tx,
                                                &ControlParams_Tx );

    if( LVVE_Status != LVVE_SUCCESS )
    {
        // Handle Errors
    }

     Change Parameters as required
    ControlParams_Tx.VOL_Gain         = VOL_TABLE[i];
    ControlParams_Tx.VC_ControlParams = VC_TABLE[i];

    // Update instance with new parameters
    LVVE_Status = LVVE_Tx_SetControlParameters( &hInstance_Tx,
                                                &ControlParams_Tx );

    if( LVVE_Status != LVVE_SUCCESS )
    {
        // Handle Errors
    }
\endcode

*/

LVVE_ReturnStatus_en LVVE_Tx_SetControlParameters(   LVVE_Tx_Handle_t           hInstance,
                                        const LVVE_Tx_ControlParams_st * const pNewParams);

/**
@brief Sets or changes the LVVE_Rx module parameters using C-structure.

This function takes the new set of parameters and makes a local copy within
LVVE_Rx but the parameters are only applied on the next call of the LVVE_Rx_Process
function. When a parameter is unchanged no action is taken. This function can
be called at any time during the processing, even when the LVVE_Rx_Process function
is running. LifeVibes&trade; VoiceExperience control parameters can be set using two methods.
Control Parameters can be populated in a \"C\" style structure (as explained in example) and then sent to the library.
Another method is to populate control parameters in a byte array called Preset Buffer, see @ref LVVE_Rx_SetPreset for example usage.

@param hInstance               Instance Handle
@param pNewParams              Pointer to a parameter structure

@pre hInstance should be valid handle.

@return LVVE_SUCCESS          Succeeded
@return LVVE_NULLADDRESS      When hInstance or pNewParams is NULL
@return LVVE_OUTOFRANGE       When pNewParams contains parameters of activated modules that are out of the excepted range
@return LVVE_INVALID_ALGORITHM_CONFIGURATION  When two mutual exclusive algorithms are both configured ON at run-time
                                               (e.g., NoiseVoid and HandsFree)


@note This function may be interrupted by the LVVE_Rx_Process function

@ingroup LVVE_Rx

LVVE_Rx_GetControlParameters: Setting Control Parameters using C-Style structure Example:

The following example shows how to set different control parameters for the LVVE_Rx instance.

\code
    LVVE_Status = LVVE_Rx_GetControlParameters( &hInstance_Rx,
                                                &ControlParams_Rx );

    if( LVVE_Status != LVVE_SUCCESS )
    {
        // Handle Errors
    }

     Change Parameters as required
    ControlParams_Rx.VOL_Gain         = VOL_TABLE[i];
    ControlParams_Rx.VC_ControlParams = VC_TABLE[i];

    // Update instance with new parameters
    LVVE_Status = LVVE_Rx_SetControlParameters( &hInstance_Rx,
                                                &ControlParams_Rx );

    if( LVVE_Status != LVVE_SUCCESS )
    {
        // Handle Errors
    }
\endcode

*/
LVVE_ReturnStatus_en LVVE_Rx_SetControlParameters(   LVVE_Rx_Handle_t           hInstance,
                                        const LVVE_Rx_ControlParams_st * const pNewParams);

/**
@brief Retrieve the current LVVE_Tx status parameters.

This function copies internal status variables into the supplied status
parameter structures.

@param hInstance      Instance handle
@param pStatusParams  Pointer to the status parameters

@pre   hInstance should be valid handle.
@pre   pStatusParams should be allocated by caller.
@post  pStatusParams will be filled with the latest status values of LVVE_Tx.

@return LVVE_SUCCESS          Succeeded
@return LVVE_NULLADDRESS      When hInstance or pStatusParams is NULL

@note The LVVE_Tx_GetStatusParameters function can be called at any time during processing.
This function can run in different threads than the process functions.


@ingroup LVVE_Tx

LVVE_Tx_GetStatusParameters: Get Internal Algorithm Status Attributes Example:
The following example shows how to use LVVE_Tx_GetStatusParameters function call.

\code
    LVVE_Tx_StatusParams_st  Tx_Monitor;
    Status=LVVE_Tx_GetStatusParameters(hInstance_Tx,&Tx_Monitor);
    if (Status != LVVE_SUCCESS )
    {
       // Error Handling
    }
\endcode

*/

LVVE_ReturnStatus_en LVVE_Tx_GetStatusParameters(    LVVE_Tx_Handle_t           hInstance,
                                                     LVVE_Tx_StatusParams_st    *pStatusParams);

/**
@brief Retrieve the current LVVE_Rx status parameters.

This function copies internal status variables into the supplied status
parameter structures.

@param hInstance      Instance handle
@param pStatusParams  Pointer to the status parameters

@pre   hInstance should be valid handle.
@pre   pStatusParams should be allocated by caller.
@post  pStatusParams will be filled with the latest status values of LVVE_Rx.

@return LVVE_SUCCESS          Succeeded
@return LVVE_NULLADDRESS      When hInstance or pStatusParams is NULL

@note The LVVE_Rx_GetStatusParameters function can be called at any time during processing.
This function can run in different threads than the process functions.

@ingroup LVVE_Rx

LVVE_Rx_GetStatusParameters: Get Internal Algorithm Status Attributes Example:
The following example shows how to use LVVE_Rx_GetStatusParameters function call.

\code
    LVVE_Rx_StatusParams_st  Rx_Monitor;
    Status=LVVE_Rx_GetStatusParameters(hInstance_Rx,&Rx_Monitor);
    if (Status != LVVE_SUCCESS )
    {
       // Error Handling
    }
\endcode

*/

LVVE_ReturnStatus_en LVVE_Rx_GetStatusParameters(    LVVE_Rx_Handle_t           hInstance,
                                                     LVVE_Rx_StatusParams_st    *pStatusParams);

/**
@brief Resets the LVVE_Tx module.

The LVVE_Tx module instance memory contains data which depend on the input
samples that have been processed previously.  These data are buffers
used for filter tabs and delay lines and also adaptive coefficients of the
algorithm.  The LVVE_Tx_ResetInstance function resets this input dependent data.

The LVVE_Tx_ResetInstance function should be called whenever there is a
discontinuity in the input audio stream.  A discontinuity means that the
current block of samples is not contiguous with the previous block of samples.
Examples are
@li Calling the LVVE_Tx_Process function after a period of inactivity
@li Buffer underrun or overflow in the audio driver
The LVVE_ResetInstance function of the Tx and/or Tx Unit should only be called when absolutely necessary as re-adaptation
of internal algorithms will occur.
The LVVE_ResetInstance functions can be called at any time, even when LVVE_Process is running. The reset will be
applied at the start of the next call of the LVVE_Tx_Process function.


@param hInstance               Instance Handle
@param pResetParams            Reset Type

@pre hInstance should be valid handle.

@post Depending on the ResetType value, the LVVE_Tx_ResetInstance function can
perform:
@li Soft reset: partial reset of internal buffers and adaptive behavior. To be used for very short
interruption or discontinuity, e.g., buffer under-run or overflow in the audio driver influencing the echo path delay.
@li Hard reset: full reset of all internal buffers and adaptive behavior. To be used for long
interruption or discontinuity, e.g., before calling the LVVE_Tx_Process function after a long period of inactivity
in between calls in case instance was not freed.

@return LVVE_SUCCESS          Succeeded
@return LVVE_NULLADDRESS      When hInstance or pResetParams is NULL
@return LVVE_OUTOFRANGE       When content of pResetParams is invalid

@note This function may be interrupted by the LVVE_Tx_Process function

@ingroup LVVE_Tx

LVVE_Tx_ResetInstance Example:

The following example shows how to use LVVE_Tx_ResetInstance.
\code
    LVVE_Status = LVVE_Tx_ResetInstance(hInstance, pResetParams);

    if (LVVE_Status != LVVE_SUCCESS)
    {
        // Handle LVVE_Tx_ResetInstance errors
    }
\endcode

*/
LVVE_ReturnStatus_en LVVE_Tx_ResetInstance(  LVVE_Tx_Handle_t        hInstance,
                                             LVVE_Tx_ResetParams_st *pResetParams  );

/**
@brief Resets the LVVE_Rx module.

The LVVE_Rx module instance memory contains data which depend on the input
samples that have been processed previously.  These data are buffers
used for filter tabs and delay lines and also adaptive coefficients of the
algorithm.  The LVVE_Rx_ResetInstance function resets this input dependent data.

The LVVE_Rx_ResetInstance function should be called whenever there is a
discontinuity in the input audio stream.  A discontinuity means that the
current block of samples is not contiguous with the previous block of samples.
Examples are
@li Calling the LVVE_Rx_Process function after a period of inactivity
@li Buffer underrun or overflow in the audio driver
The LVVE_ResetInstance function of the Rx and/or Tx Unit should only be called when absolutely necessary as re-adaptation
of internal algorithms will occur.
The LVVE_ResetInstance functions can be called at any time, even when LVVE_Process is running. The reset will be
applied at the start of the next call of the LVVE_Rx_Process function.


@param hInstance               Instance Handle
@param pResetParams            Reset Type

@pre hInstance should be valid handle.

@post Depending on the ResetType value, the LVVE_Rx_ResetInstance function can
perform:
@li Soft reset: partial reset of internal buffers and adaptive behavior. To be used for very short
interruption or discontinuity, e.g., buffer under-run or overflow in the audio driver influencing the echo path delay.
@li Hard reset: full reset of all internal buffers and adaptive behavior. To be used for long
interruption or discontinuity, e.g., before calling the LVVE_Rx_Process function after a long period of inactivity
in between calls in case instance was not freed.

@return LVVE_SUCCESS          Succeeded
@return LVVE_NULLADDRESS      When hInstance or pResetParams is NULL
@return LVVE_OUTOFRANGE       When content of pResetParams is invalid

@note This function may be interrupted by the LVVE_Rx_Process function

@ingroup LVVE_Rx

LVVE_Rx_ResetInstance Example:

The following example shows how to use LVVE_Rx_ResetInstance.
\code
    LVVE_Status = LVVE_Rx_ResetInstance(hInstance, pResetParams);

    if (LVVE_Status != LVVE_SUCCESS)
    {
        // Handle LVVE_Rx_ResetInstance errors
    }
\endcode

*/
LVVE_ReturnStatus_en LVVE_Rx_ResetInstance(  LVVE_Rx_Handle_t        hInstance,
                                             LVVE_Rx_ResetParams_st *pResetParams  );

/**
@brief Processes a block of samples by the LVVE_Tx module.

This function processes the block of samples as configured with the current
parameter settings. If new parameters have been given since the last call to
LVVE_Tx_Process then these will be applied at the beginning of this process call.

@param hInstance      Instance Handle
@param pInDataMic0    Pointer to the input speech sample buffer (pimary microphone).
@param pInDataMic1    Pointer to the input speech sample buffer (secondary microphone).
@param pRefData       Pointer to the input farend reference buffer.
@param pNoiseData     Pointer to the output noise reference buffer.
@param pOutData       Pointer to the output sample buffer.
@param NumSamples     The number of samples to process should be integer multiple(n) of values in table.
                      It is a frquency dependant paramter.
<table border>
   <tr>
       <td><b>Unit</b></td>
       <td><b>Q format</b></td>
       <td><b>Data Range</b></td>
       <td><b>Default Values</b></td>
   </tr>
   <tr>
       <td><b>Samples</b></td>
       <td><b>Q16.0</b></td>
       <td>nx80 for 8kHz</td>
       <td>80 for 8kHz</td>
   </tr>
   <tr>
       <td></td>
       <td></td>
       <td>nx160 for 16kHz</td>
       <td>160 for 16kHz</td>
   </tr>
   <tr>
       <td></td>
       <td></td>
       <td>nx240 for 24kHz</td>
       <td>240 for 24kHz</td>
   </tr>
   <tr>
       <td></td>
       <td></td>
       <td>nx320 for 32kHz</td>
       <td>320 for 32kHz</td>
   </tr>
   <tr>
       <td></td>
       <td></td>
       <td>nx480 for 48kHz</td>
       <td>480 for 48kHz</td>
   </tr>
</table>

@pre hInstance should be valid handle.
@pre LVVE_Tx_SetControlParameters should be called successfully once before.
the first call to LVVE_Tx_Process
@pre pInData is filled with the input samples to process.

@post pOutData contains the processed samples.

@return LVVE_SUCCESS          Succeeded
@return LVVE_NULLADDRESS      When hInstance, pInData or pOutData is NULL
@return LVVE_INVALIDNUMSAMPLES When the NumSamples is outside the allowed
range
@return LVVE_INVALID_STATE_CONFIGURATION When invalid state of the algorithm found

@ingroup LVVE_Tx

LVVE_Tx_Process Example:

The following example shows how to use LVVE_Tx_Process function call

\code
    // Tx Processing
    LVVE_STATUS = LVVE_Tx_Process( hInstance_Tx,
                                   pInData_Tx0,
                                   pInData_Tx1,
                                   pInData_Ref,
                                   pNoiseData,
                                   pOutData_Tx,
                                   NumSamples );
    if (LVVE_STATUS != LVVE_SUCCESS )
    {
       // Error Handling
    }
\endcode

*/
LVVE_ReturnStatus_en LVVE_Tx_Process(   LVVE_Tx_Handle_t    hInstance,
                                        const LVM_INT16     *pInDataMic0,
                                        const LVM_INT16     *pInDataMic1,
                                        const LVM_INT16     *pRefData,
                                        LVM_INT16           *pNoiseData,
                                        LVM_INT16           *pOutData,
                                        const LVM_UINT16    NumSamples);

/**
@brief Processes a block of samples by the LVVE_Rx module.

This function processes the block of samples as configured with the current
parameter settings. If new parameters have been given since the last call to
LVVE_Rx_Process then these will be applied at the beginning of this process call.

@param hInstance      Instance Handle
@param pInData        Pointer to the input sample buffer
@param pNoiseData     Pointer to the input noise buffer (for LVVC only). Pointer to the input noise reference buffer.
                      The size of this buffer depends on the number of samples to process: NoiseDataBufferSize = 2 (size of the input buffer in ms) / 10ms
                        i.e. for 20 ms input buffer, the NoiseData buffer size should be 4.

@param pOutData       Pointer to the output sample buffer
@param NumSamples     The number of samples to process should be integer multiple(n) of values in table.
                      It is a frquency dependant paramter.
<table border>
   <tr>
       <td><b>Unit</b></td>
       <td><b>Q format</b></td>
       <td><b>Data Range</b></td>
       <td><b>Default Values</b></td>
   </tr>
   <tr>
       <td><b>Samples</b></td>
       <td><b>Q16.0</b></td>
       <td>nx80 for 8kHz</td>
       <td>80 for 8kHz</td>
   </tr>
   <tr>
       <td></td>
       <td></td>
       <td>nx160 for 16kHz</td>
       <td>160 for 16kHz</td>
   </tr>
   <tr>
       <td></td>
       <td></td>
       <td>nx240 for 24kHz</td>
       <td>240 for 24kHz</td>
   </tr>
   <tr>
       <td></td>
       <td></td>
       <td>nx320 for 32kHz</td>
       <td>320 for 32kHz</td>
   </tr>
   <tr>
       <td></td>
       <td></td>
       <td>nx480 for 48kHz</td>
       <td>480 for 48kHz</td>
   </tr>
</table>

@pre hInstance should be valid handle.
@pre LVVE_Rx_SetControlParameters should be called successfully once before.
the first call to LVVE_Rx_Process
@pre pInData is filled with the input samples to process.

@post pOutData contains the processed samples.

@return LVVE_SUCCESS          Succeeded
@return LVVE_NULLADDRESS      When hInstance, pInData or pOutData is NULL
@return LVVE_INVALIDNUMSAMPLES When the NumSamples is outside the allowed
range
@return LVVE_INVALID_STATE_CONFIGURATION When invalid state of the algorithm found

@ingroup LVVE_Rx

LVVE_Rx_Process Example:
The following example shows how to use LVVE_Rx_Process function call.

\code
    // Rx Processing
    LVVE_STATUS = LVVE_Rx_Process( hInstance_Rx,
                                   pInData_Rx,
                                   pNoiseData,
                                   pOutData_Rx,
                                   NumSamples );

    if (LVVE_STATUS != LVVE_SUCCESS )
    {
       // Error Handling
    }
\endcode

*/

LVVE_ReturnStatus_en LVVE_Rx_Process(   LVVE_Rx_Handle_t    hInstance,
                                        const LVM_INT16     *pInData,
                                        const LVM_INT16     *pNoiseData,
                                        LVM_INT16           *pOutData,
                                        const LVM_UINT16    NumSamples);


/**
\brief get the version number of current of LVVE library.

This function returns, to host, current version of the LVVE library
@param  pVersion                version info

@return  LVVE_SUCCESS           Succeeded
@return  LVVE_NULLADDRESS       When one of hInstance, pInData or pOutData is NULL

LVVE_GetVersionInfot Example:

\code
    //Get the version information
    LVVE_VersionInfo     VersionInfo;
    LVVE_GetVersionInfo(&VersionInfo);
\endcode

*/
LVVE_ReturnStatus_en LVVE_GetVersionInfo( LVVE_VersionInfo *pVersion );

/**
@brief Sets or changes the LVVE_Tx module parameters through a Preset Buffer.

Sets or changes the LVVE_Tx module parameters through a Preset Buffer. To set parameters using \"C\" structure see
@ref LVVE_Tx_SetControlParameters
This function takes the new set of parameters and makes a local copy within the
LVVE_Tx module but the parameters are only applied on the next call of the LVVE_Tx_Process
function. When no parameters are changed, no action is taken. This function can
be called at any time during the processing, even when the LVVE_Tx_Process function
is running.
The new parameters are supplied in the form a @ref LVVE_Tx_Preset_t (Byte array).
The length of the byte array shall be a multiple of @ref LVVE_TX_PRESET_LENGTH.

@param hInstance               Instance Handle
@param pPreset                 Pointer to a Preset buffer
@param PresetLength            Length in bytes of the Preset buffer
@param VolumeIndex             Volume index to be selected from the Preset buffer

@pre hInstance should be valid handle.

@return @ref LVVE_SUCCESS          Succeeded
@return @ref LVVE_NULLADDRESS      When hInstance or pPreset is NULL
@return @ref LVVE_OUTOFRANGE       When PresetLength or any of the new parameters is out of range
@return @ref LVVE_INVALID_ALGORITHM_CONFIGURATION When mutually exclusive algorithms are enabled in the parameters at the same time
@return @ref LVVE_PRESET_INVALID_BUFFER_LENGTH When the length of the input buffer is smaller than LVVE_TX_PRESET_LENGTH
@return @ref LVVE_PRESET_INVALID_VOLUME_INDEX When the volume index exceeds the buffer content
@return @ref LVVE_PRESET_INVALID_BUFFER_VERSION When the version of the format of the preset buffer does not match the version of the library
@return @ref LVVE_PRESET_INVALID_BASELINE_VERSION When the baseline version of the provided preset buffer and the baseline version of the library do not match. Is the preset generated for this library ?
@return @ref LVVE_PRESET_INVALID_MASK When the algorithm mask of the provided preset buffer and the algorithm mask of the library do not match. Is the preset generated for this library ?
@return @ref LVVE_PRESET_INVALID_SAMPLE_RATE When the sample rate @ref LVM_Fs_en of the preset does not match the sample rate of the hInstance
@return @ref LVVE_PRESET_INVALID_LVWIREFORMAT_MESSAGEID  When the LVWireformat message id @ref LVVIDHeader_MessageID_en of the preset is not valid

@note This function may be interrupted by the LVVE_Tx_Process function

@ingroup LVVE_Tx
LVVE_Tx_SetPreset: Setting Control Parameters using Preset Buffer Example:

The following example shows how to set different control parameters for the LVVE_Tx instance using Preset Buffer.

\code
    LVM_CHAR LVVE_Tx_Preset_Buffer[LVVE_TX_PRESET_LENGTH];
    LVVE_Tx_Preset_t const LVVE_Tx_Preset = LVVE_Tx_Preset_Buffer;

    FILE *PRESETFILE_Tx;
    LVM_INT32 ReadBytes = 0;

    PRESETFILE_Tx=fopen(filename, "rb" );

    // Read the preset buffer
    ReadBytes = fread( LVVE_Tx_Preset_Buffer,
                       sizeof(LVM_CHAR),
                       (LVM_UINT32)LVVE_TX_PRESET_LENGTH,
                       PRESETFILE_Tx );

    fclose(PRESETFILE_Tx);

     // Update instance with new preset buffer
    LVVE_Status = LVVE_Tx_SetPreset( hInstance_Tx,
                                     LVVE_Tx_Preset,
                                     ReadBytes,
                                     0);

    if ( LVVE_Status != LVVE_SUCCESS )
    {
        // Handle Errors
    }
\endcode



*/
LVVE_ReturnStatus_en LVVE_Tx_SetPreset(LVVE_Tx_Handle_t hInstance,
                                       const LVVE_Tx_Preset_t pPreset,
                                       LVM_UINT32 PresetLength,
                                       LVM_UINT16 VolumeIndex);

/**
@brief Sets or changes the LVVE_Rx module parameters through a Preset Buffer

Sets or changes the LVVE_Rx module parameters through a Preset Buffer. To set parameters using \"C\" structure see
@ref LVVE_Rx_SetControlParameters
This function takes the new set of parameters and makes a local copy within the
LVVE_Rx module but the parameters are only applied on the next call of the LVVE_Rx_Process
function. When no parameters are changed, no action is taken. This function can
be called at any time during the processing, even when the LVVE_Rx_Process function
is running.
The new parameters are supplied in the form a @ref LVVE_Rx_Preset_t (Byte array).
The length of the byte array shall be a multiple of @ref LVVE_RX_PRESET_LENGTH.

@param hInstance               Instance Handle
@param pPreset                 Pointer to a Preset buffer
@param PresetLength            Length in bytes of the Preset buffer
@param VolumeIndex             Volume index to be selected from the Preset buffer

@pre hInstance should be valid handle.

@return @ref LVVE_SUCCESS          Succeeded
@return @ref LVVE_NULLADDRESS      When hInstance or pPreset is NULL
@return @ref LVVE_OUTOFRANGE       When PresetLength or any of the new parameters is out of range
@return @ref LVVE_INVALID_ALGORITHM_CONFIGURATION When mutually exclusive algorithms are enabled in the parameters at the same time
@return @ref LVVE_PRESET_INVALID_BUFFER_LENGTH When the length of the input buffer is smaller than LVVE_RX_PRESET_LENGTH
@return @ref LVVE_PRESET_INVALID_VOLUME_INDEX When the volume index exceeds the buffer content
@return @ref LVVE_PRESET_INVALID_BUFFER_VERSION When the version of the format of the preset buffer does not match the version of the library
@return @ref LVVE_PRESET_INVALID_BASELINE_VERSION When the baseline version of the provided preset buffer and the baseline version of the library do not match. Is the preset generated for this library ?
@return @ref LVVE_PRESET_INVALID_MASK When the algorithm mask of the provided preset buffer and the algorithm mask of the library do not match. Is the preset generated for this library ?
@return @ref LVVE_PRESET_INVALID_SAMPLE_RATE When the sample rate @ref LVM_Fs_en of the preset does not match the sample rate of the hInstance
@return @ref LVVE_PRESET_INVALID_LVWIREFORMAT_MESSAGEID  When the LVWireformat message id @ref LVVIDHeader_MessageID_en of the preset is not valid

@note This function may be interrupted by the LVVE_Rx_Process function

@ingroup LVVE_Rx

LVVE_Rx_SetPreset: Setting Control Parameters using Preset Buffer Example:

The following example shows how to set different control parameters for the LVVE_Rx instance using Preset Buffer.

\code
    #define NUM_VOLUMES (5)
    LVM_CHAR LVVE_Rx_Preset_Buffer[NUM_VOLUMES*LVVE_RX_PRESET_LENGTH];
    LVVE_Rx_Preset_t const LVVE_Rx_Preset = LVVE_Rx_Preset_Buffer;

    FILE *PRESETFILE_Rx;
    LVM_INT32 ReadBytes = 0;
    LVM_UINT16 VolumeIndex = 3;

    PRESETFILE_Rx=fopen(filename, "rb" );

    // Read the preset buffer
    ReadBytes = fread( LVVE_Rx_Preset_Buffer,
                       sizeof(LVM_CHAR),
                       (LVM_UINT32)(NUM_VOLUMES*LVVE_RX_PRESET_LENGTH),
                       PRESETFILE_Rx );

    fclose(PRESETFILE_Rx);

     // Update instance with new preset buffer
    LVVE_Status = LVVE_Rx_SetPreset( hInstance_Rx,
                                     LVVE_Rx_Preset,
                                     ReadBytes,
                                     VolumeIndex);

    if ( LVVE_Status != LVVE_SUCCESS )
    {
        // Handle Errors
    }
\endcode

*/
LVVE_ReturnStatus_en LVVE_Rx_SetPreset(LVVE_Rx_Handle_t hInstance,
                                       const LVVE_Rx_Preset_t pPreset,
                                       LVM_UINT32 PresetLength,
                                       LVM_UINT16 VolumeIndex);


#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif      /* __LVVE_H__ */

/* End of file */
