/**
 * @file flags_utils.c
 * @brief flagsӿʵ
 *
 * Copyright (C) 2023 Sanechips Technology Co., Ltd.
 * @author 
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation. ѡGPLv2 Licence
 *
 */


/*******************************************************************************
 *                           Include header files                              *
 ******************************************************************************/
#include <linux/err.h>
#include <linux/slab.h>
#include <linux/mtd/mtd.h>
#include <linux/crc32.h>

#include "pub_flags.h"

#include "flags_log.h"


extern g_zload_read_only_flag;


/*******************************************************************************
 *                             Macro definitions                               *
 ******************************************************************************/
#define MAX_PATH_LEN (256)

#define MTD_NAME_FLAGS "flags"

#define FLAGS_INIT_VALID_BLOCKS_NUM	(8)

#define GOOD_BLOCK (0)
#define BAD_BLOCK (1)

#define FILE_PATH_PROC_CMDLINE "/proc/cmdline"

#define PROC_CMDLINE_SYSTEM_A_FLAG   "system=system_a"
#define PROC_CMDLINE_SYSTEM_B_FLAG   "system=system_b"

#define SYSTEM_INDEX_UNKNOWN (-1)
#define SYSTEM_INDEX_1 (1)
#define SYSTEM_INDEX_2 (2)


/*******************************************************************************
 *                             Type definitions                                *
 ******************************************************************************/
typedef struct
{
    unsigned int mtd_totalsize;				// mtd device total size
    unsigned int mtd_pageperblock;			// mtd device page per block
    unsigned int mtd_blocksize;				// mtd device block size
    unsigned int mtd_pagesize;				// mtd device page size
    unsigned int mtd_oobsize;				// mtd device oob size
    int parti_file_desc;					// partition update file description
} partition_mtd_info_t;


typedef enum
{
    DEVICE_MTD = 0,
    DEVICE_ZFTL = 1,
    DEVICE_MTD_BLOCK,
} device_type_t;


/*******************************************************************************
 *						   Local variable definitions						   *
 ******************************************************************************/


/*******************************************************************************
 *						  Global variable definitions						   *
 ******************************************************************************/


/*******************************************************************************
 * 					   Local function declarations							   *
 ******************************************************************************/
static int write_flags_info(struct mtd_info *p_mtd_info, int index, unsigned char *content, int len);

static int get_flags_info(T_FLAGS_INFO *p_main, int *p_main_index, T_FLAGS_INFO *p_backup, int *p_backup_index);
static int set_flags_info(T_FLAGS_INFO *p_flags_info, int *p_main_index, int *p_backup_index);

static void copy_flags_info(T_FLAGS_INFO *dst, T_FLAGS_INFO *src);


/*******************************************************************************
 * 					 Local function implementations 						   *
 ******************************************************************************/
static int write_flags_info(struct mtd_info *p_mtd_info, int index, unsigned char *content, int len)
{
	int ret = -1;
    size_t write_len = 0;
	
    struct erase_info eraseInfo = {0};
    loff_t offs = 0;
	
    eraseInfo.mtd  = p_mtd_info;
    eraseInfo.addr = index * p_mtd_info->erasesize;
    eraseInfo.len  = p_mtd_info->erasesize;
	
    offs = (loff_t)index * p_mtd_info->erasesize;

	g_zload_read_only_flag = 1;

	if (0 != mtd_block_isbad(p_mtd_info, offs))
	{
		flags_err("block[%d] bad", index);
		g_zload_read_only_flag = 0;
		
		return BAD_BLOCK;
	}

	if (0 != mtd_erase(p_mtd_info, &eraseInfo))
	{
		flags_err("block[%d] erase fail", index);
		g_zload_read_only_flag = 0;
		
		return -1;
	}

	if (0 != mtd_block_isbad(p_mtd_info, offs))
	{
		flags_err("block[%d] bad", index);
		g_zload_read_only_flag = 0;
		
		return BAD_BLOCK;
	}

	ret = mtd_write(p_mtd_info, index * p_mtd_info->erasesize, len, &write_len, content);
	if (ret || write_len != len) {
		flags_err("write failed");
		g_zload_read_only_flag = 0;
	
		return -1;
	}
	
	g_zload_read_only_flag = 0;
    return 0;
}


static int get_flags_info(T_FLAGS_INFO *p_main, int *p_main_index, T_FLAGS_INFO *p_backup, int *p_backup_index)
{
	int index = 0;
    int good_index = 0;
	
    loff_t offs = 0;
    int block_flag = GOOD_BLOCK;
	
	uint32_t block_size = -1;
	struct mtd_info *mtdInfo = NULL;

	size_t retlen = 0;
	
	mtdInfo = get_mtd_device_nm(MTD_NAME_FLAGS);
	if ((NULL == mtdInfo) || IS_ERR(mtdInfo))
	{
		flags_err("get mtd device fail");
		goto error_close;
	}

	block_size = mtdInfo->erasesize;
	
	flags_debug("erasesize=%u, block_size=%u", mtdInfo->erasesize, block_size);

	for (index = 0; (good_index < 2) && (index * block_size < mtdInfo->size); index++)
    {

        offs = index * block_size;
        if (0 == mtd_block_isbad(mtdInfo, offs))
        {
            block_flag = GOOD_BLOCK;
        }
        else
        {
            flags_err("flags block [%d] is bad", index);
            block_flag =  BAD_BLOCK;
        }

        if (block_flag == GOOD_BLOCK)
        {
        	if (good_index == 0)
            {
				*p_main_index = index;
				
				if(mtd_read(mtdInfo, offs, sizeof(T_FLAGS_INFO), &retlen, (u_char *)p_main))
				{
	                flags_err("main mtd read error");
					goto error_close;
				}
        	}
            else if (good_index == 1)
            {
                *p_backup_index = index;
				
				if(mtd_read(mtdInfo, offs, sizeof(T_FLAGS_INFO), &retlen, (u_char *)p_backup))
				{
	                flags_err("backup mtd read error");
					goto error_close;
				}
				
            }
            else
            {
                break;
            }

            if (retlen != sizeof(T_FLAGS_INFO))
            {
                flags_err("read len (%d) != need len (%d)", retlen, sizeof(T_FLAGS_INFO));
                goto error_close;
            }

            good_index++;
        }
    }
	
	if (NULL != mtdInfo)
	{
	    put_mtd_device(mtdInfo);
		mtdInfo = NULL;
	}
	
	return 0;

error_close:
	if (NULL != mtdInfo)
	{
	    put_mtd_device(mtdInfo);
		mtdInfo = NULL;
	}
	
    return -1;
}


static int set_flags_info(T_FLAGS_INFO *p_flags_info, int *p_main_index, int *p_backup_index)
{
    int ret = -1;

	int index = 0;
    int good_index = 0;
	
    int main_index = *p_main_index;
    int back_index = *p_backup_index;
	
    loff_t offs = 0;
    int block_flag = GOOD_BLOCK;
	
	uint32_t block_size = -1;
	struct mtd_info *mtdInfo = NULL;

    unsigned char *real_write_content = NULL;
	
	mtdInfo = get_mtd_device_nm(MTD_NAME_FLAGS);
	if ((NULL == mtdInfo) || IS_ERR(mtdInfo))
	{
		flags_err("get mtd device fail");
		
		ret = -1;
		goto end;
	}

	block_size = mtdInfo->erasesize;
	
	flags_debug("erasesize=%u, block_size=%u", mtdInfo->erasesize, block_size);
	
	real_write_content = (unsigned char *)kmalloc(block_size, GFP_KERNEL);
	if (NULL == real_write_content)
	{
		flags_err("malloc block fail");
		
		ret = -1;
		goto end;
	}

	memset(real_write_content, 0xFF, block_size);
	memcpy(real_write_content, (char *)p_flags_info, sizeof(T_FLAGS_INFO));
	
	flags_info(">>>>> begin to write main flags <<<<<");

	for (index = 0; index * block_size < mtdInfo->size; index++)
	{
		if (index == back_index)
		{
			continue;
		}

		ret = write_flags_info(mtdInfo, index, real_write_content, block_size);
		if (ret == 0)
		{
			// дɹ˳µһλ
			flags_info("main flags location: [%d]->[%d]", main_index, index);
			main_index = index;
			break;
		}
		else if (ret == BAD_BLOCK)
		{
			// 飬һ
			flags_info("flags block index [%d] is bad", index);
			continue;
		}
		else
		{
			flags_err("write main flags fail");
			main_index = -1;
			break;
		}
	}

	flags_info(">>>>> begin to write backup flags <<<<<");

	for (index = 0; index * block_size < mtdInfo->size; index++)
	{
		if (index == main_index)
		{
			continue;
		}

		ret = write_flags_info(mtdInfo, index, real_write_content, block_size);
		if (ret == 0)
		{
			// дɹ˳µһλ
			flags_info("backup flags location: [%d]->[%d]", back_index, index);
			back_index = index;
			break;
		}
		else if (ret == BAD_BLOCK)
		{
			// 飬һ
			continue;
		}
		else
		{
			flags_err("write backup flags fail");
			back_index = -1;
			break;
		}
	}

	if (main_index == -1 && back_index == -1)
	{
		ret = -1;
		goto end;
	}
	else
	{
		ret = 0;
		goto end;
	}

end:
	if (NULL != mtdInfo)
	{
	    put_mtd_device(mtdInfo);
		mtdInfo = NULL;
	}
	
	if(NULL != real_write_content)
	{
		kfree(real_write_content);
		real_write_content = NULL;
	}
	
    return ret;
}


static void copy_flags_info(T_FLAGS_INFO *dst, T_FLAGS_INFO *src)
{
	memcpy(dst, src, sizeof(T_FLAGS_INFO));

	return;
}


/*******************************************************************************
 * 					 Global function implementations						   *
 ******************************************************************************/
int flags_get(T_FLAGS_INFO *p_flags_info)
{
    T_FLAGS_INFO main_flag = {0};
    T_FLAGS_INFO backup_flag = {0};
	T_FLAGS_INFO p_flags_info_tmp = {0};

    int main_index = 0;
    int backup_index = 1;
	
	u32 crc32_main = 0;
	u32 crc32_backup = 0;
	
	u32 crc32_le_main = 0;
	u32 crc32_le_backup = 0;

	if (NULL == p_flags_info)
	{
		flags_err("invalid param NULL");
		return -1;
	}

    if (get_flags_info(&main_flag, &main_index, &backup_flag, &backup_index) != 0)
    {
    	flags_err("get flags info fail");
        return -1;
    }
	
	flags_info("main_flag crc32=%u", main_flag.crc32);
	flags_info("backup_flag crc32=%u", backup_flag.crc32);
	
	if ((0 == main_flag.crc32) && (0 == backup_flag.crc32))
	{
		if ((FLAGS_MAGIC == main_flag.magic_start) && (FLAGS_MAGIC == main_flag.magic_end))
		{
			if ((FLAGS_MAGIC == backup_flag.magic_start) && (FLAGS_MAGIC == backup_flag.magic_end))
			{
				memcpy(&p_flags_info_tmp, &main_flag, sizeof(T_FLAGS_INFO));
				p_flags_info_tmp.crc32 = 0;
				p_flags_info_tmp.crc32 = crc32_le(0, (unsigned char const *)(&p_flags_info_tmp), sizeof(T_FLAGS_INFO));
				flags_info("old version, set crc32=%u", p_flags_info_tmp.crc32);
				
				if (set_flags_info(&p_flags_info_tmp, &main_index, &backup_index) != 0)
				{
					flags_err("old version, set flags info fail");
					return -1;
				}
				
				copy_flags_info(p_flags_info, &main_flag);
				return 0;
			}
		}
	}

	crc32_main = main_flag.crc32;
	crc32_backup = backup_flag.crc32;

	main_flag.crc32 = 0;
	backup_flag.crc32 = 0;
		
	crc32_le_main = crc32_le(0, (unsigned char const *)(&main_flag), sizeof(main_flag));
	flags_info("crc32_le_main crc32=%u", crc32_le_main);
	
	crc32_le_backup = crc32_le(0, (unsigned char const *)(&backup_flag), sizeof(backup_flag));
	flags_info("crc32_le_backup crc32=%u", crc32_le_backup);

    if (crc32_main == crc32_le_main)
    {
        copy_flags_info(p_flags_info, &main_flag);
        return 0;
    }
	
    if (crc32_backup == crc32_le_backup)
    {
        copy_flags_info(p_flags_info, &backup_flag);
        return 0;
    }

    flags_err("do not find valid flags info");
    return -1;
}
EXPORT_SYMBOL(flags_get);

int flags_set(T_FLAGS_INFO *p_flags_info)
{
	T_FLAGS_INFO main_flag = {0};
    T_FLAGS_INFO backup_flag = {0};
    int main_index = 0;
    int backup_index = 1;
	
	if (NULL == p_flags_info)
	{
		flags_err("invalid param NULL");
		return -1;
	}

	if ((FLAGS_MAGIC != p_flags_info->magic_start) || (FLAGS_MAGIC != p_flags_info->magic_end))
	{
		flags_err("invalid magic");
		return -1;
	}
	
    if (get_flags_info(&main_flag, &main_index, &backup_flag, &backup_index) != 0)
    {
    	flags_err("get flags info fail");
        return -1;
    }
	
	p_flags_info->crc32 = 0;
	p_flags_info->crc32 = crc32_le(0, (unsigned char const *)p_flags_info, sizeof(T_FLAGS_INFO));
	flags_info("set crc32=%u", p_flags_info->crc32);

    if (set_flags_info(p_flags_info, &main_index, &backup_index) != 0)
    {
        flags_err("set ubifs status fail");
        return -1;
    }

	return 0;
}
EXPORT_SYMBOL(flags_set);

