/***********************************************************************
 * Copyright (C) 2014, ZTE Corporation.
 *
 * File Name:   hal_lcd.c
 * File Mark:
 * Description: Lcd function implementation. Adapted to LCM MODEL - LEAD T15DS26.
 * Others:      this file is in BOOTLOADER procedure.
 *                  This LCM MODEL uses SPI-4.
 *                  NOTE: 4-lines-spi uses CS, CLK, D/CX and SDI. Do not using SDO instead of D/CX.
 * Version:     V1.0
 * Author:      YinWenguan
 * Date:        2014-12-19
 *
 * History 1:
 *     Date: 2015-1-16
 *     Version:
 *     Author: YinWenguan
 *     Modification: Add brightness setting.
  * History 2:
 *     Date: 2015-2-10
 *     Version:
 *     Author: YinWenguan
 *     Modification: Add "Low Battery" logo.
 *
***********************************************************************/

/*************************************************************************
 * Include files
 *************************************************************************/
#include "../include/board.h"

#include <common.h>
#include <malloc.h>
#include <asm/io.h>
#include "../include/spi.h"
#include "../include/ssp.h"
#include <lcd.h>
#include <drvs_gpio.h>
#include <peripheral.h>

/*************************************************************************
 * Macro
 *************************************************************************/
 /* debug switch */
#define LCD_DEBUG 

#ifdef LCD_DEBUG
#define LCD_PRINTF(format, args...)                     printf(format, ##args)
#define LCD_ASSERT(condition, ret, format, args...)     { if (!(condition)) { printf(format, ##args); while(1){}; } }
#else
#define LCD_PRINTF(format, args...)						printf(format, ##args)
#define LCD_ASSERT(condition, ret, format, args...)
#endif
/* read/write register's value */
#define REG_READ(REG)               *((volatile unsigned int*)(REG))
#define REG_WRITE(REG,value)    *((volatile unsigned int*)(REG)) = value

#define ZSP_LOGO_PATH 		"/LOGO.bin"
#define ZSP_LOGOBAK_PATH 	"/LOGOBAK.bin"
#define ZSP_UPDATING_PATH 	"/UPDATING.bin"
#define ZSP_UPDATESUCC_PATH "/UPDATESUCC.bin"
#define ZSP_UPDATEFAIL_PATH "/UPDATEFAIL.bin"
#define ZSP_LOWBAT_PATH 	"/LOWBAT.bin"
#define ZSP_NOBAT_PATH 		"/NOBAT.bin"
#define ZSP_CHARGING_PATH 	"/CHARGING.bin"

extern unsigned char bmp_logo[];
extern unsigned char bmp_updating[];
extern unsigned char bmp_lowbattery[];
extern unsigned char bmp_nobattery[];

/**************************************************************************
 *                                  Types                                                                                   *
 **************************************************************************/

/**************************************************************************
 * Global  Variable
 **************************************************************************/
static void *  g_sLcd_SpiHandle = NULL; // SPI device handle
volatile T_ZDrvLcd_Info s_lcd_info = {0};

static int __def_lcd_prj(void)
{
	LCD_PRINTF("lcd_prj undefine\n");
	return -1;
}

void lcd_gpio_define(struct T_ZDrvLcd_gpio_define * config) __attribute__((weak, alias("__def_lcd_prj")));
int update_rect(unsigned char *pBuf, unsigned short uiLeft, unsigned short uiTop,\
				unsigned short uiWidth, unsigned short uiHeight) __attribute__((weak, alias("__def_lcd_prj")));
int prj_lcm_init(void) __attribute__((weak, alias("__def_lcd_prj")));
int prj_lcd_get_info(T_ZDrvLcd_Info * pInfo) __attribute__((weak, alias("__def_lcd_prj")));

/**************************************************************************
 * Function Implementation
 **************************************************************************/

/*******************************************************************************
 *
 * Keep reserved.
 * Simulating SPI r/w via GPIO.
 *
 ********************************************************************************/
/* 4SPIƽݴ䷽ */
#define CS_L         	zDrvGpio_SetOutputValue(s_lcd_info.gpio_def.spi_cs_gpio, GPIO_LOW)
#define CS_H         	zDrvGpio_SetOutputValue(s_lcd_info.gpio_def.spi_cs_gpio, GPIO_HIGH)

#define RS_L         	zDrvGpio_SetOutputValue(s_lcd_info.gpio_def.spi_dcx_gpio, GPIO_LOW)
#define RS_H         	zDrvGpio_SetOutputValue(s_lcd_info.gpio_def.spi_dcx_gpio, GPIO_HIGH)

#define SDA_L       	zDrvGpio_SetOutputValue(s_lcd_info.gpio_def.spi_sda_gpio, GPIO_LOW)
#define SDA_H       	zDrvGpio_SetOutputValue(s_lcd_info.gpio_def.spi_sda_gpio, GPIO_HIGH)

#define SCL_L        	zDrvGpio_SetOutputValue(s_lcd_info.gpio_def.spi_scl_gpio, GPIO_LOW)
#define SCL_H        	zDrvGpio_SetOutputValue(s_lcd_info.gpio_def.spi_scl_gpio, GPIO_HIGH)

#define SDA_OUT     	zDrvGpio_SetDirection(s_lcd_info.gpio_def.spi_sda_gpio, GPIO_OUT)
#define SDA_IN     	zDrvGpio_SetDirection(s_lcd_info.gpio_def.spi_sda_gpio, GPIO_IN)

/* SPI״̬--> ͨGPIO */
void change_spi_to_gpio()
{
    zDrvGpio_SetFunc(s_lcd_info.gpio_def.spi_cs_gpio, s_lcd_info.gpio_def.spi_cs_gpio_sel);
    zDrvGpio_SetFunc(s_lcd_info.gpio_def.spi_scl_gpio, s_lcd_info.gpio_def.spi_scl_gpio_sel);
    zDrvGpio_SetFunc(s_lcd_info.gpio_def.spi_sda_gpio, s_lcd_info.gpio_def.spi_sda_gpio_sel);

    zDrvGpio_SetDirection(s_lcd_info.gpio_def.spi_cs_gpio, GPIO_OUT);
    zDrvGpio_SetDirection(s_lcd_info.gpio_def.spi_scl_gpio, GPIO_OUT);
    zDrvGpio_SetDirection(s_lcd_info.gpio_def.spi_sda_gpio, GPIO_OUT);

    CS_H;
    SCL_L;
}

/* ָԭSPI״̬ */
void recover_gpio_to_spi()
{
    zDrvGpio_SetFunc(s_lcd_info.gpio_def.spi_cs_gpio, s_lcd_info.gpio_def.spi_cs_func_sel);
    zDrvGpio_SetFunc(s_lcd_info.gpio_def.spi_scl_gpio, s_lcd_info.gpio_def.spi_scl_func_sel);
    zDrvGpio_SetFunc(s_lcd_info.gpio_def.spi_sda_gpio, s_lcd_info.gpio_def.spi_sda_func_sel);
    SDA_IN;
}

void delay_gpio(unsigned int tt)
{
    volatile unsigned int tmp = tt;
    while(tmp > 0){tmp--;}
}

void write_st(unsigned short rs, unsigned short para)
{
    unsigned char i;

    change_spi_to_gpio();

    if (rs){RS_H;}else{RS_L;}
    CS_L;
    delay_gpio(50);
    for (i = 0; i < 8; i++)
    {
        SCL_L;
        delay_gpio(50);
        if ((para & 0x80)==0x80)
        {
            SDA_H;
        }
        else
        {
            SDA_L;
        }
        delay_gpio(50);
        SCL_H;
        delay_gpio(50);
        para <<= 1;
    }
    CS_H;
    SCL_L;
    delay_gpio(50);
    recover_gpio_to_spi();
}

unsigned short read_st(unsigned short reg)
{
    unsigned char i;
    volatile unsigned char tmp;

    change_spi_to_gpio();

    RS_L;
    CS_L;
    delay_gpio(50);
    for (i = 0; i < 8; i++)
    {
        SCL_L;
        delay_gpio(50);
        if ((reg & 0x80)==0x80)
        {
            SDA_H;
        }
        else
        {
            SDA_L;
        }
        delay_gpio(50);
        SCL_H;
        delay_gpio(50);
        reg <<= 1;
    }
    delay_gpio(50);
    RS_H;
    SDA_IN;
    delay_gpio(150);
    tmp = 0;
    for (i = 0; i < 8; i++)
    {
        tmp <<= 1;
        SCL_L;
        delay_gpio(250);
        if (GPIO_HIGH == zDrvGpio_GetInputValue(s_lcd_info.gpio_def.spi_sda_gpio))
        {
            tmp |= 0x01;
        }
        SCL_H;
        delay_gpio(250);
    }
    CS_H;
    SDA_OUT;

    recover_gpio_to_spi();
    return tmp;
}

#if 1
/*******************************************************************************
 * Function:    zDrvLcd_DelayMs
 * Description: used to delay.
 * Parameters:
 *      (IN)
 *              ms: millisecond
 *      (OUT):
 *              None.
 * Returns:
 *
 *
 * Others:
 ********************************************************************************/
void zDrvLcd_DelayMs(unsigned long ms)
{
    volatile int i = 0;
    while (i < (50000 * ms)) {i++;}
}

/**************************************************************************
* Function: zDrvLcd_Blg_Init
* Description: used to initialize backlight usage.
* Parameters:
*       (IN)
*
*       (OUT)
*
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:
**************************************************************************/
int zDrvLcd_Blg_Init(VOID)
{
    int ret = DRV_SUCCESS;
	
	if(INVLID_GPIO == s_lcd_info.gpio_def.lcd_blg_gpio)
		return ret;
	
    zDrvGpio_SetFunc(s_lcd_info.gpio_def.lcd_blg_gpio, s_lcd_info.gpio_def.lcd_blg_gpio_sel); /* function selection: gpio225 */
    zDrvGpio_PullUpDown(s_lcd_info.gpio_def.lcd_blg_gpio, 0);
    zDrvGpio_SetDirection(s_lcd_info.gpio_def.lcd_blg_gpio, GPIO_OUT);
	
    return ret;
}

/**************************************************************************
* Function: zDrvLcd_Gpio_Init
* Description: used to initialize GPIO usage.
* Parameters:
*       (IN)
*
*       (OUT)
*
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:
**************************************************************************/
int zDrvLcd_Gpio_Init(void)
{
    int ret = DRV_SUCCESS;
	
	lcd_gpio_define(&(s_lcd_info.gpio_def));

    // RS, D/CX
    zDrvGpio_SetFunc(s_lcd_info.gpio_def.spi_dcx_gpio, s_lcd_info.gpio_def.spi_dcx_gpio_sel); /* function selection: gpio192 */
    zDrvGpio_SetDirection(s_lcd_info.gpio_def.spi_dcx_gpio, GPIO_OUT);

    // λ
    zDrvGpio_SetFunc(s_lcd_info.gpio_def.lcd_reset_gpio, s_lcd_info.gpio_def.lcd_reset_gpio_sel); /* function selection: gpio225 */
    zDrvGpio_SetDirection(s_lcd_info.gpio_def.lcd_reset_gpio, GPIO_OUT);
	
    return ret;
}

/**************************************************************************
* Function: zDrvLcd_Spi_Init
* Description: used to initialize SPI usage.
* Parameters:
*       (IN)
*
*       (OUT)
*
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:
**************************************************************************/
int zDrvLcd_Spi_Init(void)
{
    int ret = DRV_SUCCESS;

    g_sLcd_SpiHandle = ssp_open(SSP_DEV_0);
    LCD_ASSERT(NULL != g_sLcd_SpiHandle, DRV_ERROR, "[LCD] Can not open SPI device to get handle!");

    return ret;
}

/**************************************************************************
* Function: zDrvLcd_Spi_ConfigDcx
* Description: used to configure D/CX pin. thus, this interface is just for SPI-4.
* Parameters:
*       (IN)
*               type    command or parameter/data
*       (OUT)
*               None.
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:
*   - SPI-3's data packet contains a control bit D/CX and a transmission byte.
*   - SPI-4's data packet contains just transmission byte and
*      control bit D/CX is transferred by the D/CX pin.
**************************************************************************/
int zDrvLcd_Spi_ConfigDcx(T_ZDrvLcd_DcxType type)
{
    s_lcd_info.instance.eDcxType = type;
    zDrvGpio_SetOutputValue(s_lcd_info.gpio_def.spi_dcx_gpio, (type == LCD_DCX_DATA) ? GPIO_HIGH : GPIO_LOW);

    return DRV_SUCCESS;
}

/**************************************************************************
* Function: zDrvLcd_Spi_WriteCmd
* Description: used to send command-code to the LCD device IC.
* Parameters:
*       (IN)
*               cmd     command
*       (OUT)
*               None.
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:
*   - SPI-3's data packet contains a control bit D/CX and a transmission byte.
*   - SPI-4's data packet contains just transmission byte and
*      control bit D/CX is transferred by the D/CX pin.
**************************************************************************/
int zDrvLcd_Spi_WriteCmd(unsigned char cmd)
{
    unsigned char buf[1] = {cmd};
    int cnt = 0;

    zDrvLcd_Spi_ConfigDcx(LCD_DCX_CMD); // : 
    cnt = ssp_transfer(g_sLcd_SpiHandle, buf, 1); // SPIڷ8bits cmd; ubootspiֻCPUģʽ

    return (1 == cnt ? DRV_SUCCESS : DRV_ERROR);
}

/**************************************************************************
* Function: zDrvLcd_Spi_WriteData
* Description: used to send parameter/data to the LCD device IC.
* Parameters:
*       (IN)
*               data     parameter or RAM data.
*       (OUT)
*               None.
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:
*   - SPI-3's data packet contains a control bit D/CX and a transmission byte.
*   - SPI-4's data packet contains just transmission byte and
*      control bit D/CX is transferred by the D/CX pin.
**************************************************************************/
int zDrvLcd_Spi_WriteData(unsigned char data)
{
    int cnt = 0;

	// Ĵ:ýӿڿܱôݣ
    // Ӧô
    if (LCD_DCX_CMD == s_lcd_info.instance.eDcxType)
    {
        zDrvLcd_Spi_ConfigDcx(LCD_DCX_DATA);
    }

   // cnt = ssp_transfer(g_sLcd_SpiHandle, buf, 1);
    cnt = ssp_transfer(g_sLcd_SpiHandle, &data, 1);

    return (1 == cnt ? DRV_SUCCESS : DRV_ERROR);
}

int zDrvLcd_Spi_WriteDataBlock(unsigned char* pData, unsigned long size)
{
    int cnt = 0;

	if (LCD_DCX_CMD == s_lcd_info.instance.eDcxType)
    {
        zDrvLcd_Spi_ConfigDcx(LCD_DCX_DATA);
    }

    cnt = ssp_transfer(g_sLcd_SpiHandle, pData, size);

    return (size == cnt ? DRV_SUCCESS : DRV_ERROR);
}

/**************************************************************************
* Function: zDrvLcd_InitLcm
* Description: used to initialize lead(ST7735S)
* Parameters:
*      (IN)
*               None.
*      (OUT)
*               None.
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:
*       None.
**************************************************************************/
int zDrvLcd_Lcm_Init(void)
{
    int ret = DRV_SUCCESS;
	
	ret = prj_lcm_init();
	
    return ret;
}
#endif

/**************************************************************************
* Function: zDrvLcd_Open
* Description: used to open the LCD device.
* Parameters:
*      (IN)
*               None.
*      (OUT)
*               None.
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERR_OPEN_TIMES: has been opened already.
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:
*       None.
**************************************************************************/
int zDrvLcd_Initiate(void)
{
    int ret = DRV_SUCCESS;

    ret = zDrvLcd_Spi_Init(); // SPIʼ
    ret |= zDrvLcd_Gpio_Init(); // LCDGPIO : rest, backlight, rs(D/CX)

    ret |= zDrvLcd_Reset();
    LCD_ASSERT(DRV_SUCCESS ==  ret, DRV_ERROR, "[LCD] Failed to reset LCM!");
	
	zDrvLcd_GetInfo(&s_lcd_info);
	ret |= zDrvLcd_BootPicFromRes();
	
    ret |= zDrvLcd_Lcm_Init();         // ʼģ
    ret |= zDrvLcd_Blg_Init();
	
    return ret;
}

int lcd_init(void)
{
	return zDrvLcd_Initiate();
}

/**************************************************************************
* Function: zDrvLcd_Open
* Description: used to open the LCD device.
* Parameters:
*      (IN)
*               devData     LCD device name
*               flags           don't care
*      (OUT)
*               None.
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERR_OPEN_TIMES: has been opened already.
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:
*       None.
**************************************************************************/
int zDrvLcd_Open(void)
{
    if (s_lcd_info.instance.bOpen)
    {
        return DRV_ERR_OPEN_TIMES;
    }

    // ô򿪱־͵ǰģʽ
    s_lcd_info.instance.bOpen= true;
    s_lcd_info.instance.eMode = LCD_NORMAL;

    return DRV_SUCCESS;
}

/**************************************************************************
* Function: zDrvLcd_Close
* Description: used to close the LCD device opened previously.
* Parameters:
*       (IN)
*               None.
*       (OUT)
*               None.
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERR_NOT_OPENED: has not been opend yet.
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:
*       None.
**************************************************************************/
int zDrvLcd_Close()
{
    if (false == s_lcd_info.instance.bOpen)
    {
        return DRV_ERR_NOT_OPENED;
    }

    s_lcd_info.instance.bOpen = false;
    return DRV_SUCCESS;
}

/**************************************************************************
* Function: zDrvLcd_GetInfo
* Description: used to get the LCD device information.
* Parameters:
*       (IN)
*               pInfo    the pointer of LCD device information to store into
*       (OUT)
*               pInfo   the pointer of LCD device information allready stored into
* Returns:
*       DRV_SUCCESS: succeeded.
* Others:
*       None.
**************************************************************************/
int zDrvLcd_GetInfo(T_ZDrvLcd_Info* pInfo)
{
	int ret = DRV_SUCCESS;
		
	ret = prj_lcd_get_info(pInfo);
	
	return ret;
}

/**************************************************************************
* Function: zDrvLcd_Reset
* Description: used to reset the LCD device opened previously.
* Parameters:
*       (IN)
*               None.
*       (OUT)
*               None.
* Returns:
*       None.
* Others:
*       None.
**************************************************************************/
int zDrvLcd_Reset(void)
{
    int ret = DRV_SUCCESS;

    // Ӳλ
    zDrvGpio_SetOutputValue(s_lcd_info.gpio_def.lcd_reset_gpio, GPIO_HIGH);
    zDrvGpio_SetOutputValue(s_lcd_info.gpio_def.lcd_reset_gpio, GPIO_LOW);
    zDrvLcd_DelayMs(100);
    zDrvGpio_SetOutputValue(s_lcd_info.gpio_def.lcd_reset_gpio, GPIO_HIGH);
    zDrvLcd_DelayMs(120);

    return ret;
}

/**************************************************************************
* Function: zDrvLcd_UpdateRect
* Description: used to update a specified rectangle area of the LCD device opened previously.
* Parameters:
*       (IN)
*               pBuf        data buffer
*               uiLeft      left postion
*               uiTop       top postion
*               uiWidth     rectangle width
*               uiHeight    rectangle height
*       (OUT)
*               None.
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:
*       None.
**************************************************************************/
int zDrvLcd_UpdateRect(unsigned char *pBuf, unsigned short uiLeft, unsigned short uiTop, unsigned short uiWidth, unsigned short uiHeight)
{
    int ret = DRV_SUCCESS;
    unsigned short uiRight = uiLeft + uiWidth - 1;
    unsigned short uiBottom = uiTop + uiHeight - 1;
	
	if(pBuf != NULL)
	{
	    if ((s_lcd_info.width  <= uiLeft) ||
		    (s_lcd_info.height <= uiTop) ||
		    (s_lcd_info.width  <= uiRight) ||
		    (s_lcd_info.height <= uiBottom))
	    {
			LCD_PRINTF("[LCD]Invalid input parameter on calling zDrvLcd_UpdateRect!!!!");
		}
	    #if 0
		LCD_ASSERT(
		    (s_lcd_info.width> uiLeft) &&
		    (s_lcd_info.height> uiTop) &&
		    (s_lcd_info.width> uiRight) &&
		    (s_lcd_info.height> uiBottom) &&
		    (0 < uiWidth) && (0 < uiHeight),
		    DRV_ERR_INVALID_PARAM,
		    "[LCD]Invalid input parameter on calling zDrvLcd_UpdateRect!!!!");
		#endif
	}
	else 
	{
		LCD_PRINTF("[LCD]fsload resource BIN failed!!!!\n");
	}
   
 	update_rect(pBuf, uiLeft, uiTop, uiWidth, uiHeight);
	
    return ret;
}

/**************************************************************************
* Function: Lcd_EnterSleepMode
* Description: used to let the LCD device into sleep mode.
* Parameters:
*       (IN)
*               bIn     enter or exit sleep mode
*       (OUT)
*               None.
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:
*       None.
**************************************************************************/
int zDrvLcd_SleepInOut( bool bIn )
{
    int ret = DRV_SUCCESS;

    ret = zDrvLcd_Spi_WriteCmd(bIn ? 0x10 : 0x11); // enter/exit the spleep mode
    zDrvLcd_DelayMs(120); // leave 120ms

    if (false == bIn)
    {
        s_lcd_info.instance.eMode = LCD_NORMAL;
        ret |= zDrvLcd_Spi_WriteCmd(0x2c); // ˳sleepʱһָ
    }
    else
    {
        s_lcd_info.instance.eMode = LCD_SLEEP;
    }

    return ret;
}

/**************************************************************************
* Function: zDrvLcd_TurnBacklightOnOff
* Description: used to turn on/off the LCD device's backlight.
* Parameters:
*       (IN)
*               None.
*       (OUT)
*               None.
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:
*       None.
**************************************************************************/
int zDrvLcd_TurnBacklightOnOff(unsigned int enable)
{
     if (enable)
    {
        /* turn on backlight */
#if	BOOT_CFG_V2_PHONE_BOARD
        zDrvLcdBlg_SetBlg(255);
#else
		zDrvGpio_SetOutputValue(s_lcd_info.gpio_def.lcd_blg_gpio, GPIO_HIGH);
#endif
		s_lcd_info.backlight = LCD_BACKLIGHT_ON;
    }
    else
    {
        /* turn off backlight */
#if	BOOT_CFG_V2_PHONE_BOARD
	    zDrvLcdBlg_SetBlg(0);
#else
		zDrvGpio_SetOutputValue(s_lcd_info.gpio_def.lcd_blg_gpio, GPIO_LOW);
#endif
		s_lcd_info.backlight = LCD_BACKLIGHT_OFF;		
    }

    return DRV_SUCCESS;
}

int zDrvLcd_SetBrightness(T_ZDrvLcd_Brightness level)
{
#if	BOOT_CFG_V2_PHONE_BOARD
	zDrvLcdBlg_SetBlg(level);
#endif
    return DRV_SUCCESS;
}

/**************************************************************************
* Function: zDrvLcd_PowerOnLogo, zDrvLcd_ShowXxx
* Description:
* Parameters:
*       (IN)
*               None.
*       (OUT)
*               None.
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:
*       None.
**************************************************************************/
int zDrvLcd_BootPicFromRes(void)
{	
	unsigned char *bmp; 
	char cmd[64] = {0};
	
	sprintf(cmd, "fsload resource 0x%x %s", (ulong)(&bmp_logo), ZSP_LOGO_PATH);
	run_command(cmd, 0);
#if 0
	if(fsload_state_query() == 1)
	{
		sprintf(cmd, "fsload resource 0x%x %s", (ulong)(&bmp_logo), ZSP_LOGOBAK_PATH);
		run_command(cmd, 0);
	}
#endif
	sprintf(cmd, "fsload resource 0x%x %s", (ulong)(&bmp_updating), ZSP_UPDATING_PATH);
	run_command(cmd, 0);

	sprintf(cmd, "fsload resource 0x%x %s", (ulong)(&bmp_lowbattery), ZSP_LOWBAT_PATH);
	run_command(cmd, 0);
	
	sprintf(cmd, "fsload resource 0x%x %s", (ulong)(&bmp_nobattery), ZSP_NOBAT_PATH);
	run_command(cmd, 0);
	
	return DRV_SUCCESS;
}

int zDrvLcd_PowerOnLogo(void)
{	
	unsigned char *bmp;	
	bmp = bmp_logo;
	
    return zDrvLcd_UpdateRect(bmp, 0, 0, s_lcd_info.width, s_lcd_info.height);
}

int zDrvLcd_ShowUpdateWait(void)
{
	unsigned char *bmp;	
	bmp = bmp_updating;

    return zDrvLcd_UpdateRect(bmp, 0, 0, s_lcd_info.width, s_lcd_info.height);
}

int zDrvLcd_ShowUpdateSucc(void)
{
	unsigned char *bmp;
	char cmd[64] = {0};
	extern unsigned char bmp_updatesuccess[];

	sprintf(cmd, "fsload resource 0x%x %s", (ulong)(&bmp_updatesuccess), ZSP_UPDATESUCC_PATH);
	run_command(cmd, 0);
	
	bmp = bmp_updatesuccess;

    return zDrvLcd_UpdateRect(bmp, 0, 0, s_lcd_info.width, s_lcd_info.height);
}

int zDrvLcd_ShowUpdateFail(void)
{
	unsigned char *bmp;
	char cmd[64] = {0};
	extern unsigned char bmp_updatefailed[];
	
	sprintf(cmd, "fsload resource 0x%x %s", (ulong)(&bmp_updatefailed), ZSP_UPDATEFAIL_PATH);
	run_command(cmd, 0);
	
	bmp = bmp_updatefailed;

    return zDrvLcd_UpdateRect(bmp, 0, 0, s_lcd_info.width, s_lcd_info.height);
}

int zDrvLcd_ShowLowBattery(void)
{
	unsigned char *bmp;	
	bmp = bmp_lowbattery;

    return zDrvLcd_UpdateRect(bmp, 0, 0, s_lcd_info.width, s_lcd_info.height);
}

int zDrvLcd_ShowNoBattery(void)
{
	unsigned char *bmp;		
	bmp = bmp_nobattery;

    return zDrvLcd_UpdateRect(bmp, 0, 0, s_lcd_info.width, s_lcd_info.height);
}

int zDrvLcd_ShowCharging(void)
{
	unsigned char *bmp;	
	char cmd[64] = {0};
	
	extern unsigned char bmp_charging[];
	
	sprintf(cmd, "fsload resource 0x%x %s", (ulong)(&bmp_charging), ZSP_CHARGING_PATH);
	run_command(cmd, 0);
	
	bmp = bmp_charging;

    return zDrvLcd_UpdateRect(bmp, 0, 0, s_lcd_info.width, s_lcd_info.height);
}

