/* ==========================================================================
 * $File: //dwh/usb_iip/dev/software/otg/linux/drivers/dwc_otg_cil.h $
 * $Revision: #123 $
 * $Date: 2012/08/10 $
 * $Change: 2047372 $
 *
 * Synopsys HS OTG Linux Software Driver and documentation (hereinafter,
 * "Software") is an Unsupported proprietary work of Synopsys, Inc. unless
 * otherwise expressly agreed to in writing between Synopsys and you.
 *
 * The Software IS NOT an item of Licensed Software or Licensed Product under
 * any End User Software License Agreement or Agreement for Licensed Product
 * with Synopsys or any supplement thereto. You are permitted to use and
 * redistribute this Software in source and binary forms, with or without
 * modification, provided that redistributions of source code must retain this
 * notice. You may not view, use, disclose, copy or distribute this file or
 * any information contained herein except pursuant to this license grant from
 * Synopsys. If you do not agree with this notice, including the disclaimer
 * below, then you are not authorized to use the Software.
 *
 * THIS SOFTWARE IS BEING DISTRIBUTED BY SYNOPSYS SOLELY ON AN "AS IS" BASIS
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE HEREBY DISCLAIMED. IN NO EVENT SHALL SYNOPSYS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 * ========================================================================== */

#ifndef __DWC_CIL_H__
#define __DWC_CIL_H__

#include "dwc_otg_regs.h"

#include "dwc_otg_core_if.h"

//շصԭ
typedef void (*F_USB_CB)(WORD32 dwPara, WORD32 dwResult, WORD32 dwLen, WORD32 dwWantLen);


/** Macros defined for DWC OTG HW Release version */

#define OTG_CORE_REV_2_60a	0x4F54260A
#define OTG_CORE_REV_2_94a	0x4F54294A
/**
 * The <code>dwc_ep</code> structure represents the state of a single
 * endpoint when acting in device mode. It contains the data items
 * needed for an endpoint to be activated and transfer packets.
 */
#define DWC_OTG_EP_TYPE_ISOC	       1
#define DWC_OTG_EP_TYPE_BULK	       2
#define DWC_OTG_EP_TYPE_INTR	       3

typedef struct dwc_ep
{
    /** EP number used for register address lookup */
    uint8_t num;
    /** EP direction 0 = OUT */
    unsigned is_in;//:1;
    /** EP active. */
    unsigned active;//:1;
    /**
     * Periodic Tx FIFO # for IN EPs For INTR EP set to 0 to use non-periodic
     * Tx FIFO. If dedicated Tx FIFOs are enabled Tx FIFO # FOR IN EPs*/
    unsigned tx_fifo_num;//:4;
    /** EP type: 0 - Control, 1 - ISOC,	 2 - BULK,	3 - INTR */
    unsigned type;//:2;
    /** DATA start PID for INTR and BULK EP */
    unsigned data_pid_start;//:1;
    /** Frame (even/odd) for ISOC EP */
    unsigned even_odd_frame;//:1;
    /** Max Packet bytes */
    unsigned maxpacket;//:11;

    /** Max Transfer size */
    uint32_t maxxfer;

    F_USB_CB fnUsbCb;

    void  *pPara;

    uint8_t *start_xfer_buff;
    /** pointer to the transfer buffer */
    uint8_t *xfer_buff;
    /** Number of bytes to transfer */
    unsigned xfer_len;//:19;
    /** Number of bytes transferred. */
    unsigned xfer_count;//:19;
    /** Sent ZLP */
    unsigned sent_zlp;//:1;
    /** Total len for control transfer */
    unsigned total_len;//:19;

    /** stall clear flag */
    unsigned stall_clear_flag;//:1;

    /** SETUP pkt cnt rollover flag for EP0 out*/
    unsigned stp_rollover;

} dwc_ep_t;

/**
 * The following parameters may be specified when starting the module. These
 * parameters define how the DWC_otg controller should be configured.
 */
typedef struct dwc_otg_core_params
{
    /**
     * Specifies the OTG capabilities. The driver will automatically
     * detect the value for this parameter if none is specified.
     * 0 - HNP and SRP capable (default)
     * 1 - SRP Only capable
     * 2 - No HNP/SRP capable
     */
    int32_t otg_cap;


    /**
     * Specifies the maximum speed of operation in host and device mode.
     * The actual speed depends on the speed of the attached device and
     * the value of phy_type. The actual speed depends on the speed of the
     * attached device.
     * 0 - High Speed (default)
     * 1 - Full Speed
     */
    int32_t speed;

    /**
     * Specifies the type of PHY interface to use. By default, the driver
     * will automatically detect the phy_type.
     *
     * 0 - Full Speed PHY
     * 1 - UTMI+ (default)
     * 2 - ULPI
     */
    int32_t phy_type;

    /**
     * Specifies the UTMI+ Data Width. This parameter is
     * applicable for a PHY_TYPE of UTMI+ or ULPI. (For a ULPI
     * PHY_TYPE, this parameter indicates the data width between
     * the MAC and the ULPI Wrapper.) Also, this parameter is
     * applicable only if the OTG_HSPHY_WIDTH cC parameter was set
     * to "8 and 16 bits", meaning that the core has been
     * configured to work at either data path width.
     *
     * 8 or 16 bits (default 16)
     */
    int32_t phy_utmi_width;

    /**
     * Specifies whether the ULPI operates at double or single
     * data rate. This parameter is only applicable if PHY_TYPE is
     * ULPI.
     *
     * 0 - single data rate ULPI interface with 8 bit wide data
     * bus (default)
     * 1 - double data rate ULPI interface with 4 bit wide data
     * bus
     */
    int32_t phy_ulpi_ddr;

    /**
     * Specifies whether to use the internal or external supply to
     * drive the vbus with a ULPI phy.
     */
    int32_t phy_ulpi_ext_vbus;

    int32_t ulpi_fs_ls;

    int32_t ts_dline;

    /**
     * Specifies whether dedicated transmit FIFOs are
     * enabled for non periodic IN endpoints in device mode
     * 0 - No
     * 1 - Yes
     */
    int32_t en_multiple_tx_fifo;

    /** Number of 4-byte words in each of the Tx FIFOs in device
     * mode when dynamic FIFO sizing is enabled.
     * 4 to 768 (default 256)
     */
    uint32_t dev_tx_fifo_size[MAX_TX_FIFOS];

    /** Thresholding enable flag-
     * bit 0 - enable non-ISO Tx thresholding
     * bit 1 - enable ISO Tx thresholding
     * bit 2 - enable Rx thresholding
     */
    //uint32_t thr_ctl;

    /** Thresholding length for Tx
     *	FIFOs in 32 bit DWORDs
     */
    //uint32_t tx_thr_length;

    /** Thresholding length for Rx
     *	FIFOs in 32 bit DWORDs
     */
    //uint32_t rx_thr_length;


    /** Per Transfer Interrupt
     *	mode enable flag
     * 1 - Enabled
     * 0 - Disabled
     */
    int32_t pti_enable;


    /** HFIR Reload Control
     * 0 - The HFIR cannot be reloaded dynamically.
     * 1 - Allow dynamic reloading of the HFIR register during runtime.
     */
    //int32_t reload_ctl;
    /** DCFG: Enable device Out NAK
     * 0 - The core does not set NAK after Bulk Out transfer complete.
     * 1 - The core sets NAK after Bulk OUT transfer complete.
     */
    int32_t dev_out_nak;


    /** OTG revision supported
     * 0 - OTG 1.3 revision
     * 1 - OTG 2.0 revision
     */
    int32_t otg_ver;

} dwc_otg_core_params_t;
/**
 * The <code>dwc_otg_core_if</code> structure contains information needed to manage
 * the DWC_otg controller acting in either host or device mode. It
 * represents the programming view of the controller as a whole.
 */
#define DWC_OTG_PCGCCTL_OFFSET 0xE00
#define DWC_OTG_DATA_FIFO_OFFSET 0x1000
#define DWC_OTG_DATA_FIFO_SIZE 0x1000
#define A_HOST		(1)
#define A_SUSPEND	(2)
#define A_PERIPHERAL	(3)
#define B_PERIPHERAL	(4)
#define B_HOST		(5)

struct dwc_otg_core_if
{
    dwc_otg_core_params_t *core_params;

    dwc_otg_core_global_regs_t *core_global_regs;

    dwc_otg_dev_if_t *dev_if;
    uint8_t phy_init_done;


    volatile uint32_t *pcgcctl;

    uint32_t *data_fifo[MAX_EPS_CHANNELS];

    uint16_t total_fifo_size;
    uint16_t rx_fifo_size;
    uint16_t nperio_tx_fifo_size;

    uint8_t pti_enh_enable;

    uint8_t multiproc_int_enable;

    uint8_t en_multiple_tx_fifo;

    hwcfg1_data_t hwcfg1;
    hwcfg2_data_t hwcfg2;
    hwcfg3_data_t hwcfg3;
    hwcfg4_data_t hwcfg4;
    fifosize_data_t hptxfsiz;
    dcfg_data_t dcfg;

    uint32_t otg_ver;

};

extern void dwc_otg_core_dev_init(dwc_otg_core_if_t * _core_if);

extern void dwc_otg_read_setup_packet(dwc_otg_core_if_t * _core_if,
                                          uint32_t * _dest);
extern void dwc_otg_ep0_activate(dwc_otg_core_if_t * _core_if, dwc_ep_t * _ep);
extern void dwc_otg_ep_activate(dwc_otg_core_if_t * _core_if, dwc_ep_t * _ep);
extern void dwc_otg_ep_start_transfer(dwc_otg_core_if_t * _core_if,
                                          dwc_ep_t * _ep);
extern void dwc_otg_ep_start_zl_transfer(dwc_otg_core_if_t * _core_if,
            dwc_ep_t * _ep);
extern void dwc_otg_ep0_start_transfer(dwc_otg_core_if_t * _core_if,
                                           dwc_ep_t * _ep);
extern void dwc_otg_ep0_continue_transfer(dwc_otg_core_if_t * _core_if,
            dwc_ep_t * _ep);
extern void dwc_otg_ep_write_packet(dwc_otg_core_if_t * _core_if,
                                        dwc_ep_t * _ep, int _dma);
extern void dwc_otg_ep_set_stall(dwc_otg_core_if_t * _core_if, dwc_ep_t * _ep);
extern void dwc_otg_ep_clear_stall(dwc_otg_core_if_t * _core_if,
                                       dwc_ep_t * _ep);
extern void dwc_otg_enable_device_interrupts(dwc_otg_core_if_t * _core_if);

void DWC_MODIFY_REG32(uint32_t volatile *reg, uint32_t clear_mask, uint32_t set_mask);


extern void dwc_otg_read_packet(dwc_otg_core_if_t * core_if,
                                    uint8_t * dest, uint16_t bytes);

extern void dwc_otg_flush_tx_fifo(dwc_otg_core_if_t * _core_if, const int _num);
extern void dwc_otg_flush_rx_fifo(dwc_otg_core_if_t * _core_if);
extern void dwc_otg_core_reset(dwc_otg_core_if_t * _core_if);

#if 0
/**
 * This function returns the Core Interrupt register.
 */
static uint32_t dwc_otg_read_core_intr(dwc_otg_core_if_t * core_if)
{
    return (DWC_READ_REG32(&core_if->core_global_regs->gintsts) &
            DWC_READ_REG32(&core_if->core_global_regs->gintmsk));
}


/**
 * This function reads the Device All Endpoints Interrupt register and
 * returns the IN endpoint interrupt bits.
 */
static uint32_t dwc_otg_read_dev_all_in_ep_intr(dwc_otg_core_if_t *
        core_if)
{

    uint32_t v;
    {
        v = DWC_READ_REG32(&core_if->dev_if->dev_global_regs->daint) &
            DWC_READ_REG32(&core_if->dev_if->dev_global_regs->daintmsk);
    }
    return (v & 0xffff);
}

/**
 * This function reads the Device All Endpoints Interrupt register and
 * returns the OUT endpoint interrupt bits.
 */
static uint32_t dwc_otg_read_dev_all_out_ep_intr(dwc_otg_core_if_t *
        core_if)
{
    uint32_t v;

    {
        v = DWC_READ_REG32(&core_if->dev_if->dev_global_regs->daint) &
            DWC_READ_REG32(&core_if->dev_if->dev_global_regs->daintmsk);
    }

    return ((v & 0xffff0000) >> 16);
}

/**
 * This function returns the Device IN EP Interrupt register
 */
static  uint32_t dwc_otg_read_dev_in_ep_intr(dwc_otg_core_if_t * core_if,
        dwc_ep_t * ep)
{
    dwc_otg_dev_if_t *dev_if = core_if->dev_if;
    uint32_t v, msk, emp;

    {
        msk = DWC_READ_REG32(&dev_if->dev_global_regs->diepmsk);
        emp = DWC_READ_REG32(&dev_if->dev_global_regs->dtknqr4_fifoemptymsk);
        msk |= ((emp >> ep->num) & 0x1) << 7;
        v = DWC_READ_REG32(&dev_if->in_ep_regs[ep->num]->diepint) & msk;
    }

    return v;
}

/**
 * This function returns the Device OUT EP Interrupt register
 */
static uint32_t dwc_otg_read_dev_out_ep_intr(dwc_otg_core_if_t *_core_if, dwc_ep_t * _ep)
{
    dwc_otg_dev_if_t *dev_if = _core_if->dev_if;
    uint32_t v;
    doepmsk_data_t msk;
    msk.d32 = 0;
    {
        msk.d32 = DWC_READ_REG32(&dev_if->dev_global_regs->doepmsk);
        if (_core_if->pti_enh_enable)
        {
            msk.b.pktdrpsts = 1;
        }
        v = DWC_READ_REG32(&dev_if->out_ep_regs[_ep->num]->doepint) & msk.d32;
    }
    return v;
}

static  uint32_t dwc_otg_mode(dwc_otg_core_if_t * _core_if)
{
    return (DWC_READ_REG32(&_core_if->core_global_regs->gintsts) & 0x1);
}
#endif
#endif
