#include "drvs_general.h"
#include "drvs_codec_alc5672.h"
#include "drvs_codec_alc5672_dsp.h"
#ifdef _OS_LINUX
#include "drvs_timer.h"
#endif

/****************************************************************************
*                                             Local Variables
****************************************************************************/
#define RT5672_DSP_INIT_NUM							(sizeof(rt5672_dsp_init_reg) / sizeof(rt5672_dsp_init_reg[0]))
//#define RT5672_DSP_RATE_NUM							(sizeof(rt5672_dsp_rate_par) / sizeof(rt5672_dsp_rate_par[0]))
#define RT5672_DSP_2048000_NUM						(sizeof(rt5672_dsp_2048000) / sizeof(rt5672_dsp_2048000[0]))
#define RT5672_DSP_4096000_NUM						(sizeof(rt5672_dsp_4096000) / sizeof(rt5672_dsp_4096000[0]))
#define RT5672_DSP_12288000_NUM						(sizeof(rt5672_dsp_12288000) / sizeof(rt5672_dsp_12288000[0]))
#define RT5672_DSP_11289600_NUM						(sizeof(rt5672_dsp_11289600) / sizeof(rt5672_dsp_11289600[0]))
#define RT5672_DSP_24576000_NUM						(sizeof(rt5672_dsp_24576000) / sizeof(rt5672_dsp_24576000[0]))
#define RT5672_DSP_48_441_NUM						(sizeof(rt5672_dsp_48_441) / sizeof(rt5672_dsp_48_441[0]))
#define RT5672_DSP_24_NUM							(sizeof(rt5672_dsp_24) / sizeof(rt5672_dsp_24[0]))
#define RT5672_DSP_16_NUM							(sizeof(rt5672_dsp_16) / sizeof(rt5672_dsp_16[0]))
#define RT5672_DSP_8_NUM							(sizeof(rt5672_dsp_8) / sizeof(rt5672_dsp_8[0]))
#define RT5672_DSP_VOICE_NS_NUM						(sizeof(rt5672_dsp_voice_ns[0]) / sizeof(rt5672_dsp_voice_ns[0][0]))
#define RT5672_DSP_VOICE_AEC_NUM					(sizeof(rt5672_dsp_voice_aec[0]) / sizeof(rt5672_dsp_voice_aec[0][0]))
#define RT5672_DSP_VOICE_VT_NUM						(sizeof(rt5672_dsp_voice_vt[0]) / sizeof(rt5672_dsp_voice_vt[0][0]))
#define RT5672_DSP_VOICE_VR_NUM						(sizeof(rt5672_dsp_voice_vr[0]) / sizeof(rt5672_dsp_voice_vr[0][0]))
#define RT5672_DSP_VOICE_FFP_NS_NUM					(sizeof(rt5672_dsp_voice_ffp_ns[0]) / sizeof(rt5672_dsp_voice_ffp_ns[0][0]))
#define RT5672_DSP_VOICE_48K_STO_FFP_NUM			(sizeof(rt5672_dsp_voice_48k_sto_ffp[0]) / sizeof(rt5672_dsp_voice_48k_sto_ffp[0][0]))
#define RT5672_DSP_VOICE_2MIC_HANDSET_NUM			(sizeof(rt5672_dsp_voice_2mic_handset[0]) / sizeof(rt5672_dsp_voice_2mic_handset[0][0]))
#define RT5672_DSP_VOICE_2MIC_HANDSFREE_NUM			(sizeof(rt5672_dsp_voice_2mic_handsfree[0]) / sizeof(rt5672_dsp_voice_2mic_handsfree[0][0]))
#define RT5672_DSP_VOICE_AEC_HANDSFREE_NUM			(sizeof(rt5672_dsp_voice_aec_handsfree[0]) / sizeof(rt5672_dsp_voice_aec_handsfree[0][0]))
#define RT5672_DSP_TDM_SRC_PAR_NUM    				5

/* DSP init */
static UINT16 rt5672_dsp_init_reg[][2] = {
    {0x2260, 0x30d9}, {0x2260, 0x30d9}, {0x2289, 0x7fff}, {0x2290, 0x7fff},
    {0x2288, 0x0002}, {0x22b2, 0x0002}, {0x2295, 0x0001}, {0x22b3, 0x0001},
    /*{0x22d7, 0x0008}, {0x22d8, 0x0009},*/ {0x22d9, 0x0000}, {0x22da, 0x0001},
    {0x22fd, 0x001e}, {0x22c1, 0x1006}, {0x22c2, 0x1006}, {0x22c3, 0x1007},
    {0x22c4, 0x1007}, {0x229d, 0x0000}
};

static UINT16 rt5672_dsp_data_src[][2] = {
    /*For Stereo ADC mixer*/
    {0x2261, 0x30d9}, {0x2282, 0x0008}, {0x2283, 0x0009},
    {0x22d7, 0x0008}, {0x22d8, 0x0009},
    /*For Mono ADC mixer*/
    {0x2261, 0x30df}, {0x2282, 0x000a}, {0x2283, 0x000b},
    {0x22d7, 0x000a}, {0x22d8, 0x000b}
};

/*static UINT16 rt5672_dsp_rate_par[] = {
    0x226c, 0x226d, 0x226e, 0x22f2, 0x2301, 0x2306
};*/

/* MCLK rate */
static UINT16 rt5672_dsp_2048000[][2] = {
    {0x226c, 0x000c}, {0x226d, 0x000c}, {0x226e, 0x0021}
};

static UINT16 rt5672_dsp_4096000[][2] = {
    {0x226c, 0x000c}, {0x226d, 0x000c}, {0x226e, 0x0022}
};

static UINT16 rt5672_dsp_12288000[][2] = {
    {0x226c, 0x000c}, {0x226d, 0x000c}, {0x226e, 0x0026}
};

static UINT16 rt5672_dsp_11289600[][2] = {
    {0x226c, 0x0031}, {0x226d, 0x0050}, {0x226e, 0x0009}
};

static UINT16 rt5672_dsp_24576000[][2] = {
    {0x226c, 0x000c}, {0x226d, 0x000c}, {0x226e, 0x002c}
};

/* sample rate */
static UINT16 rt5672_dsp_48_441[][2] = {
    {0x22f2, 0x0058}, {0x2301, 0x0010}
};

static UINT16 rt5672_dsp_24[][2] = {
    {0x22f2, 0x0058}, {0x2301, 0x0004}
};

static UINT16 rt5672_dsp_16[][2] = {
    {0x22f2, 0x004c}, {0x2301, 0x0002}
};

static UINT16 rt5672_dsp_8[][2] = {
    {0x22f2, 0x004c}, {0x2301, 0x0000}
};

/* DSP parameters*/
//NS
static UINT16 rt5672_dsp_voice_ns[AUDIO_OUTPUT_MAX][15][2] = {
   	{{0x22f8, 0x8005}, {0x2375, 0x7ff0}, {0x2376, 0x7990}, {0x2377, 0x7332},
    {0x2388, 0x7fff}, {0x2389, 0x6000}, {0x238a, 0x0000}, {0x238b, 0x1000},
    {0x238c, 0x1000}, {0x23a1, 0x2000}, {0x2303, 0x0200}, {0x2304, 0x0032},
    {0x2305, 0x0000}, {0x230c, 0x0200}, {0x22fb, 0x0000}},//receviver
    
};
//AEC
static UINT16 rt5672_dsp_voice_aec[AUDIO_OUTPUT_MAX][54][2] = {
    {{0x22f8, 0x8003}, {0x232f, 0x00d0}, {0x2355, 0x2666}, {0x2356, 0x2666},
    {0x2357, 0x2666}, {0x2358, 0x6666}, {0x2359, 0x6666}, {0x235a, 0x6666},
    {0x235b, 0x7fff}, {0x235c, 0x7fff}, {0x235d, 0x7fff}, {0x235e, 0x7fff},
    {0x235f, 0x7fff}, {0x2360, 0x7fff}, {0x2361, 0x7fff}, {0x2362, 0x1000},
    {0x2367, 0x0007}, {0x2368, 0x4000}, {0x2369, 0x0008}, {0x236a, 0x2000},
    {0x236b, 0x0009}, {0x236c, 0x003c}, {0x236d, 0x0000}, {0x236f, 0x2000},
    {0x2370, 0x0008}, {0x2371, 0x000a}, {0x2373, 0x0000}, {0x2374, 0x7fff},
    {0x2375, 0x7ff0}, {0x2376, 0x7990}, {0x2377, 0x7332}, {0x2379, 0x1000},
    {0x237a, 0x1000}, {0x2388, 0x7fff}, {0x2389, 0x6000}, {0x238a, 0x4000},
    {0x238b, 0x1000}, {0x238c, 0x1000}, {0x2398, 0x4668}, {0x23a1, 0x2000},
    {0x23a3, 0x4000}, {0x23ad, 0x2000}, {0x23ae, 0x2000}, {0x23af, 0x2000},
    {0x23b4, 0x2000}, {0x23b5, 0x2000}, {0x23b6, 0x2000}, {0x23bb, 0x6000},
    {0x2303, 0x0710}, {0x2304, 0x0332}, {0x230c, 0x0200}, {0x230d, 0x0080},
    {0x2310, 0x0010}, {0x22fb, 0x0000}},    //receiver
    
};

static UINT16 rt5672_dsp_voice_vt[AUDIO_OUTPUT_MAX][17][2] = {
    {{0x22f8, 0x8003}, {0x2371, 0x000a}, {0x2373, 0x0000}, {0x2374, 0x7fff},
    {0x2375, 0x7ff0}, {0x2376, 0x7990}, {0x2377, 0x7332}, {0x2379, 0x1000},
    {0x237a, 0x1000}, {0x2388, 0x7fff}, {0x2389, 0x6000}, {0x238b, 0x1000},
    {0x238c, 0x1000}, {0x23a1, 0x2000}, {0x2304, 0x0332}, {0x230c, 0x0200},
    {0x22fb, 0x0000}},
    
};

static UINT16 rt5672_dsp_voice_vr[AUDIO_OUTPUT_MAX][46][2] = {
    {{0x22f8, 0x8003}, {0x2304, 0x0332}, {0x2305, 0x0000}, {0x2309, 0x0400},
    {0x230c, 0x0200}, {0x230d, 0x0080}, {0x2310, 0x0004}, {0x232f, 0x0100},
    {0x2371, 0x000a}, {0x2373, 0x3000}, {0x2374, 0x5000}, {0x2375, 0x7ff0},
    {0x2376, 0x7990}, {0x2377, 0x7332}, {0x2379, 0x1000}, {0x237a, 0x1000},
    {0x2386, 0x0200}, {0x2388, 0x4000}, {0x2389, 0x4000}, {0x238a, 0x0000},
    {0x238b, 0x2000}, {0x238c, 0x1800}, {0x239b, 0x0000}, {0x239c, 0x0a00},
    {0x239d, 0x0000}, {0x239e, 0x7fff}, {0x239f, 0x0001}, {0x23a1, 0x3000},
    {0x23a2, 0x1000}, {0x23ad, 0x0000}, {0x23ae, 0x0000}, {0x23af, 0x0000},
    {0x23c4, 0x2000}, {0x23b0, 0x0000}, {0x23b1, 0x0000}, {0x23b2, 0x0000},
    {0x23b3, 0x0000}, {0x23b4, 0x0000}, {0x23b5, 0x0000}, {0x23b6, 0x0000},
    {0x23b7, 0x0000}, {0x23b8, 0x0000}, {0x23b9, 0x0000}, {0x23ba, 0x0000},
    {0x23bb, 0x0000}, {0x22fb, 0x0000}},
    
};

static UINT16 rt5672_dsp_voice_ffp_ns[AUDIO_OUTPUT_MAX][17][2] = {
    {{0x22f8, 0x8003}, {0x2304, 0x8332}, {0x2371, 0x000a}, {0x2373, 0x0000},
    {0x2374, 0x7fff}, {0x2379, 0x1800}, {0x237a, 0x1800}, {0x230c, 0x0200},
    {0x23a2, 0x1000}, {0x2388, 0x7000}, {0x238b, 0x2000}, {0x238c, 0x2000},
    {0x23a8, 0x2000}, {0x23a9, 0x4000}, {0x23aa, 0x0100}, {0x23ab, 0x7800},
    {0x22fb, 0x0000}},
    
};

static UINT16 rt5672_dsp_voice_48k_sto_ffp[AUDIO_OUTPUT_MAX][17][2] = {
    {{0x22c1, 0x1025}, {0x22c2, 0x1026}, {0x22f8, 0x8004}, {0x22ea, 0x0001},
    {0x230c, 0x0100}, {0x230d, 0x0100}, {0x2301, 0x0010}, {0x2303, 0x0200},
    {0x2304, 0x8000}, {0x2305, 0x0000}, {0x2388, 0x6500}, {0x238b, 0x4000},
    {0x238c, 0x4000}, {0x23a9, 0x2000}, {0x23aa, 0x0200}, {0x23ab, 0x7c00},
    {0x22fb, 0x0000}},
    
};

static UINT16 rt5672_dsp_voice_2mic_handset[AUDIO_OUTPUT_MAX][24][2] = {
    {{0x22f8, 0x8002}, {0x2301, 0x0002}, {0x2302, 0x0002}, {0x2303, 0x0710},
    {0x2304, 0x4332}, {0x2305, 0x206c}, {0x236e, 0x0000}, {0x236f, 0x0001},
    {0x237e, 0x0001}, {0x237f, 0x3800}, {0x2380, 0x3000}, {0x2381, 0x0005},
    {0x2382, 0x0040}, {0x2383, 0x7fff}, {0x2388, 0x2c00}, {0x2389, 0x2800},
    {0x238b, 0x1800}, {0x238c, 0x1800}, {0x238f, 0x2000}, {0x239b, 0x0002},
    {0x239c, 0x0a00}, {0x239f, 0x0001}, {0x230c, 0x0200}, {0x22fb, 0x0000}},
    
};

static UINT16 rt5672_dsp_voice_2mic_handsfree[AUDIO_OUTPUT_MAX][17][2] = {
    {{0x22f8, 0x8003}, {0x2371, 0x000a}, {0x2373, 0x0000}, {0x2374, 0x7fff},
    {0x2375, 0x7ff0}, {0x2376, 0x7990}, {0x2377, 0x7332}, {0x2379, 0x1000},
    {0x237a, 0x1000}, {0x2388, 0x7fff}, {0x2389, 0x6000}, {0x238b, 0x1000},
    {0x238c, 0x1000}, {0x23a1, 0x2000}, {0x2304, 0x0332}, {0x230c, 0x0200},
    {0x22fb, 0x0000}},
 
};

static UINT16 rt5672_dsp_voice_aec_handsfree[AUDIO_OUTPUT_MAX][54][2] = {
    {{0x22f8, 0x8003}, {0x232f, 0x00d0}, {0x2355, 0x2666}, {0x2356, 0x2666},
    {0x2357, 0x2666}, {0x2358, 0x6666}, {0x2359, 0x6666}, {0x235a, 0x6666},
    {0x235b, 0x7fff}, {0x235c, 0x7fff}, {0x235d, 0x7fff}, {0x235e, 0x7fff},
    {0x235f, 0x7fff}, {0x2360, 0x7fff}, {0x2361, 0x7fff}, {0x2362, 0x1000},
    {0x2367, 0x0007}, {0x2368, 0x4000}, {0x2369, 0x0008}, {0x236a, 0x2000},
    {0x236b, 0x0009}, {0x236c, 0x003c}, {0x236d, 0x0000}, {0x236f, 0x2000},
    {0x2370, 0x0008}, {0x2371, 0x000a}, {0x2373, 0x0000}, {0x2374, 0x7fff},
    {0x2375, 0x7ff0}, {0x2376, 0x7990}, {0x2377, 0x7332}, {0x2379, 0x1000},
    {0x237a, 0x1000}, {0x2388, 0x7fff}, {0x2389, 0x6000}, {0x238a, 0x4000},
    {0x238b, 0x1000}, {0x238c, 0x1000}, {0x2398, 0x4668}, {0x23a1, 0x2000},
    {0x23a3, 0x4000}, {0x23ad, 0x2000}, {0x23ae, 0x2000}, {0x23af, 0x2000},
    {0x23b4, 0x2000}, {0x23b5, 0x2000}, {0x23b6, 0x2000}, {0x23bb, 0x6000},
    {0x2303, 0x0710}, {0x2304, 0x0332}, {0x230c, 0x0200}, {0x230d, 0x0080},
    {0x2310, 0x0010}, {0x22fb, 0x0000}},

};

T_ZDrvCodec_Rt5672_Priv *rt5672_dsp;

/****************************************************************************
* 	                               Function Definitions
****************************************************************************/
static SINT32 rt5672_dsp_powerup(BOOL onoff);
//static SINT32 rt5672_dsp_setbypass_tx(BOOL onoff);
//static SINT32 rt5672_dsp_setbypass_rx(BOOL onoff);
static SINT32 rt5672_dsp_reset(VOID);
//static SINT32 rt5672_dsp_read(UINT16 reg, UINT16 *value);
static SINT32 rt5672_dsp_write(UINT16 addr, UINT16 data);
static SINT32 rt5672_dsp_done(VOID);
static SINT32 rt5672_dsp_conf(VOID);
static SINT32 rt5672_dsp_set_rate(SINT32 sys_clk, SINT32 rate);
static SINT32 rt5672_dsp_set_data_source(VOID);
static SINT32 rt5672_dsp_set_mode(UINT8 mode, UINT8 codecSrcType, UINT8 device);
static SINT32 rt5672_dsp_snd_effect(VOID);
//static VOID rt5672_dsp_reg_show(VOID);
extern SINT32 codec_I2CWrite(UINT8 regAddress, UINT16 regMask, UINT16 regValue);
extern SINT32 codec_I2CRead16(UINT8 regAddress, UINT16 *regValue);
extern SINT32 codec_I2CWrite_Nomask(UINT8 regAddress, UINT16 regValue);

/******************************************************************************
 * Function:rt5672_dsp_reset
 * Description:
 * Parameters:
 *	 Input:
 *
 *	 Output:
 *
 * Returns:
 *	 0 if success, not 0 if fail
 *
 * Others:
 ******************************************************************************/
static SINT32 rt5672_dsp_reset(VOID)
{
    SINT32 ret = DRV_SUCCESS;
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672_dsp_reset.\n");
 
    ret += codec_I2CWrite(RT5672_GEN_CTRL1, RT5672_RST_DSP, RT5672_RST_DSP);
    ret += codec_I2CWrite(RT5672_GEN_CTRL1, RT5672_RST_DSP, 0x0 << 13);
 
    if (ret != DRV_SUCCESS) {
         zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672_dsp_reset fail ! ret = %d\n", ret);
    }

    return ret;
}

/******************************************************************************
 * Function:rt5672_dsp_done
 * Description:
 * Parameters:
 *	 Input:
 *
 *	 Output:
 *
 * Returns:
 *	 0 if success, not 0 if fail
 *
 * Others:
 ******************************************************************************/
static SINT32 rt5672_dsp_done(VOID)
{
    SINT32 count = 0;
    UINT16 dsp_val = 0;
    SINT32 ret = DRV_SUCCESS;	

    ret = codec_I2CRead16(RT5672_DSP_CTRL1, &dsp_val);
    if (ret != DRV_SUCCESS) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672 read register dsp ctl1 error ret = %d. \n", ret);
        return ret;
    }

    while (dsp_val & RT5672_DSP_BUSY_MASK) {
        if (count > 10) {
            return -DRV_ERR_BUSY;
        }

        ret = codec_I2CRead16(RT5672_DSP_CTRL1, &dsp_val);
        if (ret != DRV_SUCCESS) {
            zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672 read %d times for register dsp ctl1 error ret = %d. \n", count, ret);
            return ret;
        }
        count++;
    }

    return DRV_SUCCESS;
}

/******************************************************************************
 * Function:rt5672_dsp_write
 * Description:
 * Parameters:
 *	 Input:
 *
 *	 Output:
 *
 * Returns:
 *	 0 if success, not 0 if fail
 *
 * Others:
 ******************************************************************************/
static SINT32 rt5672_dsp_write(UINT16 addr, UINT16 data)
{
    SINT32 dsp_val;
    SINT32 ret = DRV_SUCCESS;

    ret = codec_I2CWrite_Nomask(RT5672_DSP_CTRL2, addr);
    if (ret != DRV_SUCCESS) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "Failed to write DSP register RT5672_DSP_CTRL2: %d\n", ret);
        return ret;
    }

    ret = codec_I2CWrite_Nomask(RT5672_DSP_CTRL3, data);
    if (ret != DRV_SUCCESS) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "Failed to write DSP register RT5672_DSP_CTRL3: %d\n", ret);
        return ret;
    }

    dsp_val = RT5672_DSP_I2C_AL_16 | RT5672_DSP_DL_2 | RT5672_DSP_CMD_MW | DSP_CLK_RATE | RT5672_DSP_CMD_EN;
    ret = codec_I2CWrite_Nomask(RT5672_DSP_CTRL1, dsp_val);
    if (ret != DRV_SUCCESS) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "Failed to write DSP register RT5672_DSP_CTRL1: %d\n", ret);
        return ret;
    }
	
    ret = rt5672_dsp_done();
    if (ret != DRV_SUCCESS) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "DSP is busy: %d\n", ret);
        return ret;
    }

    return DRV_SUCCESS;
}
#if 0
/******************************************************************************
 * Function: rt5672_dsp_read
 * Description:
 * Parameters:
 *	 Input:
 *
 *	 Output:
 *
 * Returns:
 *	 0 if success, not 0 if fail
 *
 * Others:
 ******************************************************************************/
static SINT32 rt5672_dsp_read(UINT16 reg, UINT16 *value)
{
    UINT16 dsp_val;
    SINT32 ret = DRV_SUCCESS;

    ret = rt5672_dsp_done();
    if (ret != DRV_SUCCESS) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "DSP is busy: %d\n", ret);
        return ret;
    }

    ret = codec_I2CWrite_Nomask(RT5672_DSP_CTRL2, reg);
    if (ret != DRV_SUCCESS) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "Failed to write DSP addr reg dsp ctrl2: %d\n", ret);
        return ret;
    }
    dsp_val = RT5672_DSP_I2C_AL_16 | RT5672_DSP_DL_0 | RT5672_DSP_RW_MASK | \
              RT5672_DSP_CMD_MR | DSP_CLK_RATE | RT5672_DSP_CMD_EN;

    ret = codec_I2CWrite_Nomask(RT5672_DSP_CTRL1, dsp_val);
    if (ret != DRV_SUCCESS) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "Failed to write DSP cmd reg dsp ctrl1: %d\n", ret);
        return ret;
    }

    ret = rt5672_dsp_done();
    if (ret != DRV_SUCCESS) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "DSP is busy: %d\n", ret);
        return ret;
    }

    ret = codec_I2CWrite_Nomask(RT5672_DSP_CTRL2, 0x26);
    if (ret != DRV_SUCCESS) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "Failed to write DSP addr reg dsp ctrl2: %d\n", ret);
        return ret;
    }
	
    dsp_val = RT5672_DSP_DL_1 | RT5672_DSP_CMD_RR | RT5672_DSP_RW_MASK | DSP_CLK_RATE | RT5672_DSP_CMD_EN;
    ret = codec_I2CWrite_Nomask(RT5672_DSP_CTRL1, dsp_val);
    if (ret != DRV_SUCCESS) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "Failed to write DSP cmd reg dsp ctrl1: %d\n", ret);
        return ret;
    }

    ret = rt5672_dsp_done();
    if (ret != DRV_SUCCESS) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "DSP is busy: %d\n", ret);
        return ret;
    }

    ret = codec_I2CWrite_Nomask(RT5672_DSP_CTRL2, 0x25);
    if (ret != DRV_SUCCESS) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "Failed to write DSP addr reg dsp ctrl2: %d\n", ret);
        return ret;
    }

    dsp_val = RT5672_DSP_DL_1 | RT5672_DSP_CMD_RR | RT5672_DSP_RW_MASK | \
              DSP_CLK_RATE | RT5672_DSP_CMD_EN;
    ret = codec_I2CWrite_Nomask(RT5672_DSP_CTRL1, dsp_val);
    if (ret != DRV_SUCCESS) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "Failed to write DSP cmd reg dsp ctrl1: %d\n", ret);
        return ret;
    }

    ret = rt5672_dsp_done();
    if (ret != DRV_SUCCESS) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "DSP is busy: %d\n", ret);
        return ret;
    }

    ret = codec_I2CRead16(RT5672_DSP_CTRL5, &value);
    if (ret != DRV_SUCCESS) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "Failed to read DSP data reg dsp ctrl5: %d\n", ret);
        return ret;
    }

    return DRV_SUCCESS;
}

/******************************************************************************
 * Function:rt5672_dsp_reg_show
 * Description:
 * Parameters:
 *	 Input:
 *
 *	 Output:
 *
 * Returns:
 *	 0 if success, not 0 if fail
 *
 * Others:
 ******************************************************************************/
static VOID rt5672_dsp_reg_show(VOID)
{
    UINT16 val, i;
    SINT32 ret = DRV_SUCCESS;
	
    for (i = 0x2200; i <= 0x23ff; i++) {
        ret = rt5672_dsp_read(i, &val);
        if (ret != DRV_SUCCESS) {
            zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "Failed to rt5672_dsp_reg_show register: 0x%x\n", i);
            break;
        }
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672_dsp_reg_show  %04x: %04x.\n", i ,val);
    }
}
#endif
/******************************************************************************
 * Function:rt5672_dsp_conf
 * Description:
 * Parameters:
 *	 Input:
 *
 *	 Output:
 *
 * Returns:
 *	 0 if success, not 0 if fail
 *
 * Others:
 ******************************************************************************/
static SINT32 rt5672_dsp_conf(VOID)
{
    SINT32 ret = DRV_SUCCESS, i;
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672_dsp_conf.\n");
	
    for (i = 0; i < RT5672_DSP_INIT_NUM; i++) {
        ret = rt5672_dsp_write( rt5672_dsp_init_reg[i][0], rt5672_dsp_init_reg[i][1]);
        if (ret != DRV_SUCCESS) {
            zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "Fail to config dsp register %d.\n", rt5672_dsp_init_reg[i][0]);
            return ret;
        }
    }

    return DRV_SUCCESS;
}

/******************************************************************************
 * Function:rt5672_dsp_rate
 * Description:
 * Parameters:
 *	 Input:
 *
 *	 Output:
 *
 * Returns:
 *	 0 if success, not 0 if fail
 *
 * Others:
 ******************************************************************************/
static SINT32 rt5672_dsp_set_rate( SINT32 sys_clk,SINT32 rate)
{
    SINT32 ret = DRV_SUCCESS, i, tab_num = 0;
    unsigned short (*rate_tab)[2];
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672_dsp_set_rate sys:%d, srate:%d\n", sys_clk, rate);

    switch(sys_clk) {
	    case 2048000:
	        rate_tab = rt5672_dsp_2048000;
	        tab_num = RT5672_DSP_2048000_NUM;
	        break;
	    case 4096000:
	        rate_tab = rt5672_dsp_4096000;
	        tab_num = RT5672_DSP_4096000_NUM;
	        break;
	    case 11289600:
	        rate_tab = rt5672_dsp_11289600;
	        tab_num = RT5672_DSP_11289600_NUM;
	        break;
	    case 12288000:
	        rate_tab = rt5672_dsp_12288000;
	        tab_num = RT5672_DSP_12288000_NUM;
	        break;
	    case 24576000:
	        rate_tab = rt5672_dsp_24576000;
	        tab_num = RT5672_DSP_24576000_NUM;
	        break;
	    default:
                zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "Invalid sys clk params.\n");
	        return -EINVAL;
    }

    for (i = 0; i < tab_num; i++) {
        ret = rt5672_dsp_write(rate_tab[i][0], rate_tab[i][1]);
        if (ret != DRV_SUCCESS) {
            zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "Failed to rt5672_dsp_set_rate write dsp sysclk register:0x%x.\n", rate_tab[i][0]);
            return ret;
        }
    }

    switch(rate) {
        case 8000:
            rate_tab = rt5672_dsp_8;
            tab_num = RT5672_DSP_8_NUM;
            break;
        case 16000:
            rate_tab = rt5672_dsp_16;
            tab_num = RT5672_DSP_16_NUM;
            break;
        case 24000:
            rate_tab = rt5672_dsp_24;
            tab_num = RT5672_DSP_24_NUM;
            break;
        case 44100:
        case 48000:
            rate_tab = rt5672_dsp_48_441;
            tab_num = RT5672_DSP_48_441_NUM;
            break;
        default:
            zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "Invalid srate params.\n");
            return -EINVAL;
    }

    for (i = 0; i < tab_num; i++) {
        ret = rt5672_dsp_write(rate_tab[i][0], rate_tab[i][1]);
        if (ret != DRV_SUCCESS) {
            zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "Failed to rt5672_dsp_rate write dsp rate register:0x%x.\n", rate_tab[i][0]);
            return ret;
        }
    }
	
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672_dsp_rate set the rate success.\n");
    return DRV_SUCCESS;
}

/******************************************************************************
 * Function:rt5672_dsp_set_data_source
 * Description:
 * Parameters:
 *	 Input:
 *
 *	 Output:
 *
 * Returns:
 *	 0 if success, not 0 if fail
 *
 * Others:
 ******************************************************************************/
static SINT32 rt5672_dsp_set_data_source(VOID)
{
    SINT32 ret = DRV_SUCCESS, src = 0;
    UINT16 val, i;
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"rt5672_dsp_set_data_source. \n");

    /*read MX-2d [3:2] to decide TDM source*/
    /*currently, support slot 0/1 and 2/3 only*/
    ret = codec_I2CRead16(RT5672_DSP_PATH1, &val);
    if (ret != DRV_SUCCESS) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"rt5672_dsp_set_data_source failed .\n");
        return ret;			
    }

    val &= 0xc;
    if (val == 0x4) { /*slot 2/3*///
        src = 1;
    }
    for (i = src * RT5672_DSP_TDM_SRC_PAR_NUM; i < (src + 1) * RT5672_DSP_TDM_SRC_PAR_NUM; i++) {
        ret = rt5672_dsp_write(rt5672_dsp_data_src[i][0], rt5672_dsp_data_src[i][1]);
        if (ret != DRV_SUCCESS) {
            zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"rt5672_dsp_set_data_source failed .\n");
            return ret;			
        }
    }
	
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"rt5672_dsp_set_data_source success\n");
    return DRV_SUCCESS;
}

/******************************************************************************
 * Function:rt5672_dsp_set_mode
 * Description:
 * Parameters:
 *	 Input:
 *
 *	 Output:
 *
 * Returns:
 *	 0 if success, not 0 if fail
 *
 * Others:
 ******************************************************************************/
static SINT32 rt5672_dsp_set_mode(UINT8 mode, T_ZDrvCodec_SourceType codecSrcType, T_ZDrvCodec_OutDevice device)
{
    SINT32 ret = DRV_SUCCESS, i, tab_num = 0;
    UINT16 (*mode_tab)[2];
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672_dsp_set_mode mode:%d, codecSrcType:%d, device:%d \n", mode, codecSrcType, device);

    if (codecSrcType == VOICE_SOURCE) {	
        switch (mode) {
            case RT5672_DSP_NS:
                zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "VOICE NS\n");
                mode_tab = rt5672_dsp_voice_ns[device];
                tab_num = RT5672_DSP_VOICE_NS_NUM;
                break;	   
            case RT5672_DSP_AEC:
                zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "VOICE AEC\n");
                mode_tab = rt5672_dsp_voice_aec[device];
                tab_num = RT5672_DSP_VOICE_AEC_NUM;
                break;
            case RT5672_DSP_VT:
                zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "VOICE VT\n");
                mode_tab = rt5672_dsp_voice_vt[device];
                tab_num = RT5672_DSP_VOICE_VT_NUM;
                break;
            case RT5672_DSP_VR:
                zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"VOICE VR\n");
                mode_tab = rt5672_dsp_voice_vr[device];
                tab_num = RT5672_DSP_VOICE_VR_NUM;
                break;
	    case RT5672_DSP_FFP_NS:
                zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "VOICE FFP_NS\n");
                mode_tab = rt5672_dsp_voice_ffp_ns[device];
                tab_num = RT5672_DSP_VOICE_FFP_NS_NUM;
                break;
            case RT5672_DSP_48K_STO_FFP:
                zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "VOICE 48K_STO_FFP\n");
                mode_tab = rt5672_dsp_voice_48k_sto_ffp[device];
                tab_num = RT5672_DSP_VOICE_48K_STO_FFP_NUM;
                break;
            case RT5672_DSP_2MIC_HANDSET:
                zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "VOICE 2MIC_HANDSET\n");
                mode_tab = rt5672_dsp_voice_2mic_handset[device];
                tab_num = RT5672_DSP_VOICE_2MIC_HANDSET_NUM;
                break;
            case RT5672_DSP_2MIC_HANDSFREE:
                zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "VOICE 2MIC_HANDSFREE\n");
                mode_tab = rt5672_dsp_voice_2mic_handsfree[device];
                tab_num = RT5672_DSP_VOICE_2MIC_HANDSFREE_NUM;
                break;
            case RT5672_DSP_AEC_HANDSFREE:
                zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"VOICE AEC_HANDSFREE\n");
                mode_tab = rt5672_dsp_voice_aec_handsfree[device];
                tab_num = RT5672_DSP_VOICE_AEC_HANDSFREE_NUM;
                break;	
            case RT5672_DSP_DIS:
                break;
            default:
                zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "Disable\n");
                return 0;
        }
    } else if (codecSrcType == AUDIO_SOURCE) {
    /*TODO audio parameters*/
    }	

    for (i = 0; i < tab_num; i++) {
        ret = rt5672_dsp_write(mode_tab[i][0], mode_tab[i][1]);
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672_dsp_write register 0x%x val:0x%x.\n", mode_tab[i][0],mode_tab[i][1]);
        if (ret != DRV_SUCCESS) {
            zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672_dsp_write register 0x%x error .\n", mode_tab[i][0]);
            return ret;
        }
    }

    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"rt5672_dsp_set_mode success\n");
    return DRV_SUCCESS;
}
/******************************************************************************
 * Function:rt5672_dsp_snd_effect
 * Description:
 * Parameters:
 *	 Input:
 *
 *	 Output:
 *
 * Returns:
 *	 0 if success, not 0 if fail
 *
 * Others:
 ******************************************************************************/
static SINT32 rt5672_dsp_snd_effect(VOID)
{
    SINT32 rate;
    SINT32 ret = DRV_SUCCESS;
    
    if (rt5672_dsp == NULL) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672_dsp poniter is null!\n");
        return DRV_ERROR;
    }

    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"rt5672_dsp_snd_effect: dsp mode:%d, sysclk:%d \n", rt5672_dsp->dsp_mode ,rt5672_dsp->sysclk);  
    if (!rt5672_dsp->dsp_inited) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672_dsp_snd_effect inited.\n");

        switch (rt5672_dsp->dsp_mode) {
            case RT5672_DSP_48K_STO_FFP:
                rate = 48000;
                break;
            default:
                rate = 8000;
                break;
        }
		
        ret += rt5672_dsp_reset();
        ret += rt5672_dsp_set_rate(rt5672_dsp->sysclk ? rt5672_dsp->sysclk : 2048000, rate);
        ret += rt5672_dsp_conf();
        ret += rt5672_dsp_set_data_source();
        if (ret != DRV_SUCCESS) {
            zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672_dsp_snd_effect inited error .\n");
            return ret;
        }
			
        rt5672_dsp->dsp_inited = true;
    }

    ret = rt5672_dsp_set_mode(rt5672_dsp->dsp_mode, rt5672_dsp->codecSrcType, rt5672_dsp->device);

    if (ret != DRV_SUCCESS) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672_dsp_set_mode error .\n");
        return ret;
    }

    return DRV_SUCCESS;
}
#if 0
/******************************************************************************
 * Function:rt5672_dsp_setbypass_tx
 * Description:
 * Parameters:
 *	 Input:
 *
 *	 Output:
 *
 * Returns:
 *	 0 if success, not 0 if fail
 *
 * Others:
 ******************************************************************************/
static SINT32 rt5672_dsp_setbypass_tx(BOOL onoff)
{
    SINT32 ret = DRV_SUCCESS;
	
    if (onoff) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672_dsp_setbypass_tx BYPASS !\n");
        ret = codec_I2CWrite(RT5672_DSP_PATH1, RT5672_DSP_UL_MASK, 0x1 << RT5672_DSP_UL_SFT);//bypass the up

    } else {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672_dsp_setbypass_tx NO BYPASS !\n");
        ret = codec_I2CWrite(RT5672_DSP_PATH1, RT5672_DSP_UL_MASK, 0x0 << RT5672_DSP_UL_SFT);//pass the up
    }

    if (ret != DRV_SUCCESS) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672_dsp_setbypass_tx fail  !\n");
    }

    return ret;
}

/******************************************************************************
 * Function:rt5672_dsp_setbypass_rx
 * Description:
 * Parameters:
 *	 Input:
 *
 *	 Output:
 *
 * Returns:
 *	 0 if success, not 0 if fail
 *
 * Others:
 ******************************************************************************/
static SINT32 rt5672_dsp_setbypass_rx(BOOL onoff)
{
    SINT32 ret = DRV_SUCCESS;

    if (onoff) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672_dsp_setbypass_rx BYPASS !\n");
        ret = codec_I2CWrite(RT5672_DSP_PATH1, RT5672_DSP_DL_MASK, (0x1 << RT5672_DSP_DL_SFT));//bypass the downlink path
    } else {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672_dsp_setbypass_rx NO BYPASS !\n");
        ret = codec_I2CWrite(RT5672_DSP_PATH1, RT5672_DSP_DL_MASK, (0x0 << RT5672_DSP_DL_SFT));//unbypass the downlink path
    }

    if (ret != DRV_SUCCESS) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672_dsp_setbypass_rx fail  !\n");
    }

    return ret;
}
#endif
/******************************************************************************
 * Function:rt5672_dsp_powerup
 * Description:
 * Parameters:
 *	 Input:
 *
 *	 Output:
 *
 * Returns:
 *	 0 if success, not 0 if fail
 *
 * Others:
 ******************************************************************************/
static SINT32 rt5672_dsp_powerup(BOOL onoff)
{
    SINT32 ret = DRV_SUCCESS;

    if (rt5672_dsp == NULL) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672_dsp poniter is null!\n");
        return DRV_ERROR;
    } 
    if (onoff) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672_dsp_powerup !\n");
        ret = codec_I2CWrite(RT5672_PWR_DIG2, RT5672_PWR_I2S_DSP, 0x1 << RT5672_PWR_I2S_DSP_BIT);//pwr up  
    } else {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672_dsp_powerdown !\n");
        ret = codec_I2CWrite(RT5672_PWR_DIG2, RT5672_PWR_I2S_DSP, 0x0 << RT5672_PWR_I2S_DSP_BIT);//pwr down
        rt5672_dsp->dsp_inited = false;
    }

    if (ret != DRV_SUCCESS) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672_dsp_powerup fail !\n");
        return DRV_ERROR;
    }
	
    return ret;
}

/******************************************************************************
 * Function:rt5672_dsp_switch
 * Description:
 * Parameters:
 *	 Input:
 *
 *	 Output:
 *
 * Returns:
 *	 0 if success, not 0 if fail
 *
 * Others:
 ******************************************************************************/
SINT32 rt5672_dsp_switch (BOOL onoff)
{
    SINT32 ret = DRV_SUCCESS;
 
    if (rt5672_dsp == NULL) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672_dsp poniter is null!\n");
        return DRV_ERROR;
    } 

    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672_dsp_switch onoff:%d, dsp_bypass:%d. \n", onoff, rt5672_dsp->dsp_bypass);
    switch (onoff) {
        case 0:
            ret += rt5672_dsp_powerup(0);
            ret += rt5672_dsp_write(0x22f9, 1);    //power down        
            break;	
        case 1:
            ret += rt5672_dsp_powerup(1);	
            if (rt5672_dsp->dsp_bypass) {
                zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672 dsp bypass.\n");
                ret += codec_I2CWrite(RT5672_DSP_PATH1, RT5672_DSP_UL_MASK | RT5672_DSP_DL_MASK,\
	                                     (0x1 << RT5672_DSP_UL_SFT) | (0x1 << RT5672_DSP_DL_SFT));
            } else {
                zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672 dsp no bypass.\n");
                ret += codec_I2CWrite(RT5672_DSP_PATH1, RT5672_DSP_UL_MASK | RT5672_DSP_DL_MASK,\
				             (0x0 << RT5672_DSP_UL_SFT) | (0x0 << RT5672_DSP_DL_SFT));
                ret += rt5672_dsp_snd_effect();
            }
            break;
        default:
            zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "Invalid switch value.\n");
            return -EINVAL;
    }

    if (ret != DRV_SUCCESS) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672_dsp_switch fail !\n");
        return ret;
    }

    return DRV_SUCCESS;
}

/******************************************************************************
 * Function:rt5672_dsp_setpara
 * Description:
 * Parameters:
 *	 Input:
 *
 *	 Output:
 *
 * Returns:
 *	 0 if success, not 0 if fail
 *
 * Others:
 ******************************************************************************/
SINT32 rt5672_dsp_setpara(T_ZDrvCodec_Rt5672_Priv * codecDspPara)
{
    if (codecDspPara == NULL) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codecDspPara poniter is null!\n");
        return DRV_ERROR;
    } 

    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "rt5672_dsp_setpara dsp_mode:%d, device:%d.\n", codecDspPara->dsp_mode, codecDspPara->device);
    rt5672_dsp = codecDspPara;
    return DRV_SUCCESS;
}
