#ifndef _GIF_LIB_H
#define _GIF_LIB_H

#include "zcore_type.h"
#define GIF_LIB_VERSION	" Version 4.0, "

#define	GIF_ERROR	0
#define GIF_OK		1

#ifndef TRUE
#define TRUE		1
#define FALSE		0
#endif

#ifndef NULL
#define NULL		0
#endif /* NULL */

#define GIF_STAMP "GIFVER"          /* First chars in file - GIF stamp.  */
#define GIF_STAMP_LEN sizeof(GIF_STAMP) - 1
#define GIF_VERSION_POS	3           /* Version first character in stamp. */
#define GIF87_STAMP	"GIF87a"        /* First chars in file - GIF stamp.  */
#define GIF89_STAMP	"GIF89a"        /* First chars in file - GIF stamp.  */

#define GIF_FILE_BUFFER_SIZE 16384  /* Files uses bigger buffers than usual. */

typedef	ZSINT32		GifBooleanType;
typedef	ZUINT8	GifPixelType;
typedef ZUINT8 *	GifRowType;
typedef ZUINT8	GifByteType;

#define GIF_MESSAGE(Msg) fprintf(stderr, "\n%s: %s\n", PROGRAM_NAME, Msg)
#define GIF_EXIT(Msg)	{ GIF_MESSAGE(Msg); exit(-3); }

#ifdef SYSV
#define VoidPtr ZCHAR *
#else
#define VoidPtr ZVOID *
#endif /* SYSV */

#define LM_to_uint(a,b) (((b)<<8)|(a))

typedef struct GifColorType
{
    GifByteType Red, Green, Blue;
} GifColorType;

typedef struct ColorMapObject
{
    ZSINT32	ColorCount;
    ZSINT32 BitsPerPixel;
    GifColorType *Colors;		/* on malloc(3) heap */
}
ColorMapObject;

typedef struct GifImageDesc
{
    ZSINT32 Left, Top, Width, Height,	/* Current image dimensions. */
            Interlace;			/* Sequential/Interlaced lines. */
    ColorMapObject *ColorMap;		/* The local color map */
} GifImageDesc;

typedef struct _Gif89
{
    ZSINT32 transparent;
    ZSINT32 delay_time;
    ZSINT32 input_flag;
    ZSINT32 disposal;
} Gif89;

typedef struct GifFileType
{
    ZSINT32 SWidth, SHeight;		/* Screen dimensions. */
    ZSINT32 SColorResolution; 		/* How many colors can we generate? */
    ZSINT32 SBackGroundColor;		/* I hope you understand this one... */
    ColorMapObject *SColorMap;	/* NULL if not exists. */
    ZSINT32 ImageCount;			    /* Number of current image */
    GifImageDesc Image;			/* Block describing current image */
    struct SavedImage *SavedImages;	/* Use this to accumulate file state */

    Gif89 gif89;
    GifPixelType* pScanLine;
    GifPixelType* frameBuf[2];

    ZSINT32 resetPos;
    ZSINT32 isFile;
    T_MemHandle *ptMemHandle;

    VoidPtr UserData;           /* hook to attach user data (TVT) */
    VoidPtr Private;	  		/* Don't mess with this! */
} GifFileType;

typedef enum
{
    UNDEFINED_RECORD_TYPE,
    SCREEN_DESC_RECORD_TYPE,
    IMAGE_DESC_RECORD_TYPE,		/* Begin with ',' */
    EXTENSION_RECORD_TYPE,		/* Begin with '!' */
    TERMINATE_RECORD_TYPE		/* Begin with ';' */
} GifRecordType;


typedef enum
{
    GIF_DUMP_SGI_WINDOW = 1000,
    GIF_DUMP_X_WINDOW = 1001
} GifScreenDumpType;

/* func type to read gif data from arbitrary sources (TVT) */
typedef ZSINT32 (*InputFunc)(GifFileType*,GifByteType*,ZSINT32);

/* func type to write gif data ro arbitrary targets.
 * Returns count of bytes written. (MRB)
 */
typedef ZSINT32 (*OutputFunc)(GifFileType *, const GifByteType *, ZSINT32);
/******************************************************************************
*  GIF89 extension function codes                                             *
******************************************************************************/

#define COMMENT_EXT_FUNC_CODE		0xfe	/* comment */
#define GRAPHICS_EXT_FUNC_CODE		0xf9	/* graphics control */
#define PLAINTEXT_EXT_FUNC_CODE		0x01	/* plaintext */
#define APPLICATION_EXT_FUNC_CODE	0xff	/* application block */


//GifFileType *EGifOpenFileName(const char *GifFileName, int GifTestExistance);
GifFileType *EGifOpenFileHandle(ZSINT32 GifFileHandle);
GifFileType *EGifOpen(ZVOID *userPtr, OutputFunc writeFunc);
ZSINT32 EGifSpew(GifFileType *GifFile);
ZVOID EGifSetGifVersion(const ZCHAR *Version);
ZSINT32 EGifPutScreenDesc(GifFileType *GifFile,
                          ZSINT32 GifWidth, ZSINT32 GifHeight, ZSINT32 GifColorRes, ZSINT32 GifBackGround,
                          const ColorMapObject *GifColorMap);
ZSINT32 EGifPutImageDesc(GifFileType *GifFile,
                         ZSINT32 GifLeft, ZSINT32 GifTop, ZSINT32 Width, ZSINT32 GifHeight, ZSINT32 GifInterlace,
                         const ColorMapObject *GifColorMap);
ZSINT32 EGifPutLine(GifFileType *GifFile, GifPixelType *GifLine, ZSINT32 GifLineLen);
ZSINT32 EGifPutPixel(GifFileType *GifFile, GifPixelType GifPixel);
ZSINT32 EGifPutComment(GifFileType *GifFile, const ZCHAR *GifComment);
ZSINT32 EGifPutExtensionFirst(GifFileType *GifFile, ZSINT32 GifExtCode, ZSINT32 GifExtLen,
                              const VoidPtr GifExtension);
ZSINT32 EGifPutExtensionNext(GifFileType *GifFile, ZSINT32 GifExtCode, ZSINT32 GifExtLen,
                             const VoidPtr GifExtension);
ZSINT32 EGifPutExtensionLast(GifFileType *GifFile, ZSINT32 GifExtCode, ZSINT32 GifExtLen,
                             const VoidPtr GifExtension);
ZSINT32 EGifPutExtension(GifFileType *GifFile, ZSINT32 GifExtCode, ZSINT32 GifExtLen,
                         const VoidPtr GifExtension);
ZSINT32 EGifPutCode(GifFileType *GifFile, ZSINT32 GifCodeSize,
                    const GifByteType *GifCodeBlock);
ZSINT32 EGifPutCodeNext(GifFileType *GifFile, const GifByteType *GifCodeBlock);
ZSINT32 EGifCloseFile(GifFileType *GifFile);

#define	E_GIF_ERR_OPEN_FAILED	1		/* And EGif possible errors. */
#define	E_GIF_ERR_WRITE_FAILED	2
#define E_GIF_ERR_HAS_SCRN_DSCR	3
#define E_GIF_ERR_HAS_IMAG_DSCR	4
#define E_GIF_ERR_NO_COLOR_MAP	5
#define E_GIF_ERR_DATA_TOO_BIG	6
#define E_GIF_ERR_NOT_ENOUGH_MEM 7
#define E_GIF_ERR_DISK_IS_FULL	8
#define E_GIF_ERR_CLOSE_FAILED	9
#define E_GIF_ERR_NOT_WRITEABLE	10


//GifFileType *DGifOpenFileName(const char *GifFileName);
static GifFileType *DGifOpenFileHandle(ZSINT32 GifFileHandle);
static GifFileType *DGifOpen( ZVOID* userPtr, InputFunc readFunc );  /* new one (TVT) */
static ZSINT32 DGifSlurp(GifFileType *GifFile);
static ZSINT32 DGifGetScreenDesc(GifFileType *GifFile);
static ZSINT32 DGifGetRecordType(GifFileType *GifFile, GifRecordType *GifType);
static ZSINT32 DGifGetImageDesc(GifFileType *GifFile);
static ZSINT32 DGifGetLine(GifFileType *GifFile, GifPixelType *GifLine, ZSINT32 GifLineLen);
static ZSINT32 DGifGetPixel(GifFileType *GifFile, GifPixelType GifPixel);
static ZSINT32 DGifGetComment(GifFileType *GifFile, ZCHAR *GifComment);
static ZSINT32 DGifGetExtension(GifFileType *GifFile, ZSINT32 *GifExtCode,
                                GifByteType **GifExtension);
static ZSINT32 DGifGetExtensionNext(GifFileType *GifFile, GifByteType **GifExtension);
static ZSINT32 DGifGetCode(GifFileType *GifFile, ZSINT32 *GifCodeSize,
                           GifByteType **GifCodeBlock);
static ZSINT32 DGifGetCodeNext(GifFileType *GifFile, GifByteType **GifCodeBlock);
static ZSINT32 DGifGetLZCodes(GifFileType *GifFile, ZSINT32 *GifCode);
static ZSINT32 DGifCloseFile(GifFileType *GifFile);

#define	D_GIF_ERR_OPEN_FAILED	101		/* And DGif possible errors. */
#define	D_GIF_ERR_READ_FAILED	102
#define	D_GIF_ERR_NOT_GIF_FILE	103
#define D_GIF_ERR_NO_SCRN_DSCR	104
#define D_GIF_ERR_NO_IMAG_DSCR	105
#define D_GIF_ERR_NO_COLOR_MAP	106
#define D_GIF_ERR_WRONG_RECORD	107
#define D_GIF_ERR_DATA_TOO_BIG	108
#define D_GIF_ERR_NOT_ENOUGH_MEM 109
#define D_GIF_ERR_CLOSE_FAILED	110
#define D_GIF_ERR_NOT_READABLE	111
#define D_GIF_ERR_IMAGE_DEFECT	112
#define D_GIF_ERR_EOF_TOO_SOON	113


ZSINT32 QuantizeBuffer(ZUINT32 Width, ZUINT32 Height, ZSINT32 *ColorMapSize,
                       GifByteType *RedInput, GifByteType *GreenInput, GifByteType *BlueInput,
                       GifByteType *OutputBuffer, GifColorType *OutputColorMap);



static ZSINT32 GifQuietPrint;

#ifdef HAVE_VARARGS_H
static ZVOID GifQprintf();
#else
static ZVOID GifQprintf(ZCHAR *Format, ...);
#endif /* HAVE_VARARGS_H */


static ZVOID PrintGifError(ZVOID);
static ZSINT32 GifLastError(ZVOID);

static ZSINT32 DumpScreen2Gif(const ZCHAR *FileName,
                              ZSINT32 ReqGraphDriver,
                              ZSINT32 ReqGraphMode1,
                              ZSINT32 ReqGraphMode2,
                              ZSINT32 ReqGraphMode3);


/******************************************************************************
* Color Map handling from ALLOCGIF.C					      *
******************************************************************************/

static ColorMapObject *MakeMapObject(ZSINT32 ColorCount, const GifColorType *ColorMap);
static ZVOID FreeMapObject(ColorMapObject *Object);
static ColorMapObject *UnionColorMap(
    const ColorMapObject *ColorIn1,
    const ColorMapObject *ColorIn2,
    GifPixelType ColorTransIn2[]);
static ZSINT32 BitSize(ZSINT32 n);

/******************************************************************************
* Support for the in-core structures allocation (slurp mode).		      *
******************************************************************************/

/* This is the in-core version of an extension record */
typedef struct
{
    ZSINT32		ByteCount;
    ZCHAR	*Bytes;		/* on malloc(3) heap */
    ZSINT32 Function;       /* Holds the type of the Extension block. */
} ExtensionBlock;

/* This holds an image header, its unpacked raster bits, and extensions */
typedef struct SavedImage
{
    GifImageDesc	ImageDesc;

    ZCHAR		*RasterBits;		/* on malloc(3) heap */

    ZSINT32			Function; /* DEPRECATED: Use ExtensionBlocks[x].Function
                           * instead */
    ZSINT32			ExtensionBlockCount;
    ExtensionBlock	*ExtensionBlocks;	/* on malloc(3) heap */
} SavedImage;

static ZVOID ApplyTranslation(SavedImage *Image, GifPixelType Translation[]);

static ZVOID MakeExtension(SavedImage *New, ZSINT32 Function);
static ZSINT32 AddExtensionBlock(SavedImage *New, ZSINT32 Len, ZCHAR ExtData[]);
static ZVOID FreeExtension(SavedImage *Image);

static SavedImage *MakeSavedImage(GifFileType *GifFile, const SavedImage *CopyFrom);
static ZVOID FreeSavedImages(GifFileType *GifFile);

#define PROGRAM_NAME		"GIFLIB"

#define LZ_MAX_CODE			4095	/* Biggest code possible in 12 bits. */
#define LZ_BITS				12

#define FLUSH_OUTPUT		4096    /* Impossible code, to signal flush. */
#define FIRST_CODE			4097    /* Impossible code, to signal first. */
#define NO_SUCH_CODE		4098    /* Impossible code, to signal empty. */

#define FILE_STATE_WRITE    0x01
#define FILE_STATE_SCREEN   0x02
#define FILE_STATE_IMAGE    0x04
#define FILE_STATE_READ     0x08

#define IS_READABLE(Private)    (Private->FileState & FILE_STATE_READ)
#define IS_WRITEABLE(Private)   (Private->FileState & FILE_STATE_WRITE)


typedef struct GifFilePrivateType
{
    ZSINT32 FileState;
    ZSINT32 FileHandle;			/* Where all this data goes to! */
    ZSINT32 BitsPerPixel;		/* Bits per pixel (Codes uses at least this + 1). */
    ZSINT32 ClearCode;			/* The CLEAR LZ code. */
    ZSINT32 EOFCode;			/* The EOF LZ code. */
    ZSINT32 RunningCode;		/* The next code algorithm can generate. */
    ZSINT32 RunningBits;		/* The number of bits required to represent RunningCode. */
    ZSINT32 MaxCode1;			/* 1 bigger than max. possible code, in RunningBits bits. */
    ZSINT32 LastCode;			/* The code before the current code. */
    ZSINT32 CrntCode;			/* Current algorithm code. */
    ZSINT32 StackPtr;			/* For character stack (see below). */
    ZSINT32 CrntShiftState;		/* Number of bits in CrntShiftDWord. */
    ZULONG CrntShiftDWord;		/* For bytes decomposition into codes. */
    ZULONG PixelCount;		    /* Number of pixels in image. */
    FILE *File;							/* File as stream. */
    InputFunc Read;						/* function to read gif input (TVT) */
    OutputFunc Write;					/* function to write gif output (MRB) */
    GifByteType Buf[256];				/* Compressed input is buffered here. */
    GifByteType Stack[LZ_MAX_CODE+1];	/* Decoded pixels are stacked here. */
    GifByteType Suffix[LZ_MAX_CODE+1];  /* So we can trace the codes. */
    ZUINT32 Prefix[LZ_MAX_CODE+1];
} GifFilePrivateType;

extern ZSINT32 _GifError;

#endif /* _GIF_LIB_H */
