#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/cdev.h>
#include <linux/fs.h>
#include <linux/ioport.h>
#include <linux/serial_reg.h>


#include <linux/poll.h>
#include <linux/delay.h>
#include <linux/wait.h>
#include <linux/err.h>
#include <linux/interrupt.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/device.h>
#include <linux/miscdevice.h>
#include <linux/spinlock.h>
#include <linux/list.h>
#include <linux/slab.h>

#include "usb_proxy_main.h"



MODULE_LICENSE("GPL");
MODULE_AUTHOR("jinkai");


#define ADB_BULK_BUFFER_SIZE           4096

/* number of tx requests to allocate */
#define TX_REQ_MAX 4

static const char adb_shortname[] = "android_adb";
static struct adbproxy_dev *_adb_dev;

static int adb_open(struct inode *ip, struct file *fp);
static int adb_release(struct inode *ip, struct file *fp);
static ssize_t adb_read(struct file *fp, char __user *buf,size_t count, loff_t *pos);
static ssize_t adb_write(struct file *fp, const char __user *buf,size_t count, loff_t *pos);

#define pr_debug(fmt, ...) \
	printk(KERN_WARNING pr_fmt(fmt), ##__VA_ARGS__)



struct adbproxy_dev {

	spinlock_t lock;

	atomic_t online;
	atomic_t error;

	atomic_t read_excl;
	atomic_t write_excl;
	atomic_t open_excl;

	struct list_head tx_idle;

	wait_queue_head_t read_wq;
	wait_queue_head_t write_wq;
	struct usbproxy_request *rx_req;
	int rx_done;
};

/* file operations for ADB device /dev/android_adb */
static const struct file_operations adb_fops = {
	.owner = THIS_MODULE,
	.read = adb_read,
	.write = adb_write,
	.open = adb_open,
	.release = adb_release,
};

static struct miscdevice adb_device = {
	.minor = MISC_DYNAMIC_MINOR,
	.name = adb_shortname,
	.fops = &adb_fops,
};



static struct usbproxy_request *adb_request_new(int buffer_size)
{
	struct usbproxy_request *req = usbproxy_alloc_request(GFP_KERNEL);
	//struct usbproxy_request *req = NULL;
	if (!req)
		return NULL;

	/* now allocate buffers for the requests */
	req->buf = kmalloc(buffer_size, GFP_KERNEL);
	if (!req->buf) {
		usbproxy_free_request(req);
		return NULL;
	}

	return req;
}

static void adb_request_free(struct usbproxy_request *req)
{
	if (req) {
		kfree(req->buf);
		usbproxy_free_request(req);
	}
}

static inline int adb_lock(atomic_t *excl)
{
	if (atomic_inc_return(excl) == 1) {
		return 0;
	} else {
		atomic_dec(excl);
		return -1;
	}
}

static inline void adb_unlock(atomic_t *excl)
{
	atomic_dec(excl);
}

/* add a request to the tail of a list */
void adb_req_put(struct adbproxy_dev *dev, struct list_head *head,
		struct usbproxy_request *req)
{
	unsigned long flags;

	raw_spin_lock_irqsave(&dev->lock, flags);
	list_add_tail(&req->list, head);
	raw_spin_unlock_irqrestore(&dev->lock, flags);
}

/* remove a request from the head of a list */
struct usbproxy_request *adb_req_get(struct adbproxy_dev *dev, struct list_head *head)
{
	unsigned long flags;
	struct usbproxy_request *req;

	raw_spin_lock_irqsave(&dev->lock, flags);
	if (list_empty(head)) {
		req = 0;
	} else {
		req = list_first_entry(head, struct usbproxy_request, list);
		list_del(&req->list);
	}
	raw_spin_unlock_irqrestore(&dev->lock, flags);
	return req;
}

static void adb_complete_in(struct usbproxy_request *req)
{
	struct adbproxy_dev *dev = _adb_dev;
       printk("!!!!!!--------adb_complete_in----------");     
	if (req->status != 0)
		atomic_set(&dev->error, 1);

	adb_req_put(dev, &dev->tx_idle, req);

	wake_up(&dev->write_wq);
}

static void adb_complete_out(struct usbproxy_request *req)
{
	struct adbproxy_dev *dev = _adb_dev;
       printk("!!!!!!--------adb_complete_out----------");     
	dev->rx_done = 1;
	if (req->status != 0 && req->status != -ECONNRESET)
		atomic_set(&dev->error, 1);

	wake_up(&dev->read_wq);
}


static ssize_t adb_read(struct file *fp, char __user *buf,size_t count, loff_t *pos)
{
	struct adbproxy_dev *dev = fp->private_data;
	struct usbproxy_request *req;
	int r = count, xfer;
	int ret;    
	printk("!!!!!!--------adb_read\n");

	if (!_adb_dev)
		return -ENODEV;

	if (count > ADB_BULK_BUFFER_SIZE)
		return -EINVAL;

	if (adb_lock(&dev->read_excl))
		return -EBUSY;

	/* we will block until we're online */
	while (!(atomic_read(&dev->online) || atomic_read(&dev->error))) {
		pr_debug("adb_read: waiting for online state\n");
		ret = wait_event_interruptible(dev->read_wq,
			(atomic_read(&dev->online) ||
			atomic_read(&dev->error)));
		if (ret < 0) {
			adb_unlock(&dev->read_excl);
			return ret;
		}
	}
	if (atomic_read(&dev->error)) {
		r = -EIO;
		goto done;
	}

requeue_req:
	/* queue a request */
	req = dev->rx_req;
	req->length = count;
	dev->rx_done = 0;
	ret = usbproxy_read(req);
	//ret=1;
	if (ret < 0) {
		pr_debug("adb_read: failed to queue req %p (%d)\n", req, ret);
		r = -EIO;
		atomic_set(&dev->error, 1);
		goto done;
	} else {
		pr_debug("rx %p queue\n", req);
	}

	/* wait for a request to complete */
	ret = wait_event_interruptible(dev->read_wq, dev->rx_done);
	if (ret < 0) {
		if (ret != -ERESTARTSYS)
		atomic_set(&dev->error, 1);
		r = ret;
		goto done;
	}
	if (!atomic_read(&dev->error)) {
		/* If we got a 0-len packet, throw it back and try again. */
		if (req->actual == 0)
			goto requeue_req;

		pr_debug("rx %p %d\n", req, req->actual);
		xfer = (req->actual < count) ? req->actual : count;
		if (copy_to_user(buf, req->buf, xfer))
			r = -EFAULT;

	} else
		r = -EIO;

done:
	adb_unlock(&dev->read_excl);
	printk("adb_read returning %d\n", r);
      {
	  	int n = 0;
		char* buf = req->buf;
		printk("adb_read data ==begin=in hex=:\n");
		for(n =0; n < req->length; n++)
			printk(" %02x", buf[n]);
		printk("\nadb_read data ==end==\n");
      }
	return r;
}

static ssize_t adb_write(struct file *fp, const char __user *buf,size_t count, loff_t *pos)
{
	struct adbproxy_dev *dev = fp->private_data;
	struct usbproxy_request *req = 0;
	int r = count, xfer;
	int ret;

	printk("!!!!!!--------adb_write\n");
	if (!_adb_dev)

		return -ENODEV;

	if (adb_lock(&dev->write_excl))
		return -EBUSY;

	while (count > 0) {
		if (atomic_read(&dev->error)) {
			pr_debug("adb_write dev->error\n");
			r = -EIO;
			break;
		}

		/* get an idle tx request to use */
		req = 0;
		ret = wait_event_interruptible(dev->write_wq,
			((req = adb_req_get(dev, &dev->tx_idle)) ||
			 atomic_read(&dev->error)));

		if (ret < 0) {
			r = ret;
			break;
		}

		if (req != 0) {
			if (count > ADB_BULK_BUFFER_SIZE)
				xfer = ADB_BULK_BUFFER_SIZE;
			else
				xfer = count;
			if (copy_from_user(req->buf, buf, xfer)) {
				r = -EFAULT;
				break;
			}

			req->length = xfer;
			ret = usbproxy_write(req);
			ret=1;
			if (ret < 0) {
				pr_debug("adb_write: xfer error %d\n", ret);
				atomic_set(&dev->error, 1);
				r = -EIO;
				break;
			}

			buf += xfer;
			count -= xfer;

			/* zero this so we don't try to free it on error exit */
			req = 0;
		}
	}

	if (req)
		adb_req_put(dev, &dev->tx_idle, req);

	adb_unlock(&dev->write_excl);
	printk("adb_write returning %d\n", r);
	return r;
}

static int adb_open(struct inode *ip, struct file *fp)
{
	pr_info("!!!!!!--------adb_open\n");
	if (!_adb_dev)
		return -ENODEV;

	if (adb_lock(&_adb_dev->open_excl))
		return -EBUSY;

	fp->private_data = _adb_dev;

	/* clear the error latch */
	atomic_set(&_adb_dev->error, 0);


	return 0;
}

static int adb_release(struct inode *ip, struct file *fp)
{
	pr_info("adb_release\n");

	adb_unlock(&_adb_dev->open_excl);
	return 0;
}




static int __init adb_init(void)
{

       struct adbproxy_dev *dev;
	struct usbproxy_request *req;
	int ret;
	int i;
       pr_info("!!!!!!--------adb_init\n");

	dev = kzalloc(sizeof(*dev), GFP_KERNEL);
	if (!dev)
		return -ENOMEM;

	spin_lock_init(&dev->lock);

	init_waitqueue_head(&dev->read_wq);
	init_waitqueue_head(&dev->write_wq);

	atomic_set(&dev->open_excl, 0);
	atomic_set(&dev->read_excl, 0);
	atomic_set(&dev->write_excl, 0);

	INIT_LIST_HEAD(&dev->tx_idle);

	atomic_set(&dev->error, 0);
	atomic_set(&dev->online,1);

	printk(KERN_ERR "!!!!!!--------adb_init  before first adb_request_new\n");
	req = adb_request_new(ADB_BULK_BUFFER_SIZE);
	if (!req)
		goto fail;
	req->complete = adb_complete_out;
	dev->rx_req = req;

	for (i = 0; i < TX_REQ_MAX; i++) {
		printk(KERN_ERR "!!!!!!--------adb_init  for entry\n");
		req = adb_request_new(ADB_BULK_BUFFER_SIZE);
		if (!req)
			goto fail;
		req->complete = adb_complete_in;
		adb_req_put(dev, &dev->tx_idle, req);
		}
	
	_adb_dev = dev;
	ret = misc_register(&adb_device);	
	
	if (ret){
		kfree(dev);
	       printk(KERN_ERR "adb gadget driver failed to initialize\n");
	}
	return 0;
fail:
	printk(KERN_ERR "adb_bind() could not allocate requests\n");
	return -1;
	
      }


static void __exit adb_exit(void)
{
	misc_deregister(&adb_device);

	kfree(_adb_dev);
	_adb_dev = NULL;
}

module_init(adb_init);
module_exit(adb_exit);
