/***********************************************************************
* Copyright (C) 2007, ZTE Corporation.
* 
* File Name:     dd_audio.c
* File Mark:      
* Description:  This file contains the core routines for audio driver.
* Others:      
* Version:      v0.5
* Author:      LvWenhua
* Date:            2007-11-01
* 
* History 1:           
*     Date:          2009-02-19
*     Version:     1.0
*     Author:       wangshiyong
*     Modification:  
* History 2: 
**********************************************************************/

/**************************************************************************
 *                       Include files                                    *
 **************************************************************************/
#include "drvs_general.h"
#include "audio_api.h"
#include "hal_audio.h"

/****************************************************************************
*                                     Local Type Definitions                *
****************************************************************************/
typedef enum
{
    DD_AUDIO_IDLE                    = 0,
    DD_AUDIO_PLAYING              = 1,
    DD_AUDIO_RECORDING         = 2,

    MAX_DD_AUDIO_WORK_STATUS
}T_HalAudio_WorkStatus;

typedef enum
{
    DD_CLOSED = 0,
    DD_OPENED    ,

    MAX_DD_OPEN_STATUS      
}T_Dd_OpenStatus;

/****************************************************************************
*                                     Local Function Prototypes             *
****************************************************************************/
static SINT32 ddAudio_Open(VOID *devData, T_ZDRVIO_FLAGS flag);
static SINT32 ddAudio_Close(VOID *devData);
static SINT32 ddAudio_Ioctl (VOID *devData, T_DRVIO_CTRL_KEY function, VOID* arg);
static SINT32 ddAudio_Read(VOID *devData, VOID *pBuf, UINT32 uiLen);
static SINT32 ddAudio_Write(VOID *devData, const VOID *pBuf, UINT32 uiLen);

/****************************************************************************
*                                      Global Variables                     *
****************************************************************************/
T_Dd_OpenStatus gzDrvAudio_OpenFlag = DD_CLOSED;
T_HalAudio_WorkStatus gzDrvAudio_Status = DD_AUDIO_IDLE;

T_ZDrvIODev_Handle g_audioIODevHandle = NULL;

static T_ZDrvIODev_Ops s_audioIODevOps =
{
    .open   = ddAudio_Open,
    .close  = ddAudio_Close,
    .read   = ddAudio_Read,
    .write  = ddAudio_Write,
    .ctrl   = ddAudio_Ioctl,
};

/****************************************************************************
*                                              Function Definitions      *
****************************************************************************/

/**************************************************************************
* Function: zDrvAudio_Init
* Description: This function is used to install audio driver.
* Parameters: 
*   Input:None
*   Output:None
* Returns: 
*           T_ZDrv_ErrCode
*          
* Others: None
**************************************************************************/
SINT32 zDrvAudio_Init(VOID)
{
    SINT32 ret=0;
    
    //ret=zDrv_DevInstall(ZDRV_DEV_AUDIO, &drv_Audio_entry);
    g_audioIODevHandle = zDrvIODev_Connect(ZDRV_DEV_AUDIO, NULL, &s_audioIODevOps);

    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"zDrvAudio_Initiate! ret %d\n", ret);

    return ret;
}

/**************************************************************************
* Function: zDrvAudio_Exit
* Description: This function is used to uninstall audio driver.
* Parameters: 
*   Input:None
*   Output:None
* Returns: 
*           T_ZDrv_ErrCode
*            
* Others: None
**************************************************************************/
SINT32 zDrvAudio_Exit(VOID)
{
    SINT32 ret=0;

    //ret=zDrv_DevUnInstall(ZDRV_DEV_AUDIO);
    zDrvIODev_Disconnect(g_audioIODevHandle);
    
    return ret;
}

/**************************************************************************
* Function: ddAudio_Open
* Description: This function is used to open audio device.
* Parameters: 
*   Input:
*          private:device private data
*         flags: no use
*   Output: None
* Returns:   
*           T_ZDrv_ErrCode
*             others: if errcode<-100 ,please refer to the hal layer errcode file
* Others: None
**************************************************************************/
SINT32 ddAudio_Open(VOID *devData, T_ZDRVIO_FLAGS flag)
{
    SINT32 ret = 0;

    if (gzDrvAudio_OpenFlag == DD_OPENED)
    {
    	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"ddAudio_Open gzDrvAudio_OpenFlag = %d\n", gzDrvAudio_OpenFlag);
        return DRV_ERR_OPEN_TIMES;
    }

    ret = halAudio_Open();
    if ( ret < 0 )
    { 
    	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"ddAudio_Open fail ret %d\n", ret);
        return ret;
    }
    gzDrvAudio_OpenFlag = DD_OPENED;
	
    return DRV_SUCCESS;
}

/**************************************************************************
* Function: ddAudio_Close
* Description: This function is used to close audio device.
* Parameters: 
*   Input:
*      private:device private data
*   Output:None
* Returns: 
*           T_ZDrv_ErrCode
*           others: if errcode<-100 ,please refer to the hal layer errcode file
* Others: None
**************************************************************************/
SINT32 ddAudio_Close(VOID *devData)
{
    SINT32 ret = 0;

    if (gzDrvAudio_OpenFlag != DD_OPENED)
    {
    	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"ddAudio_Close gzDrvAudio_OpenFlag = %d\n", gzDrvAudio_OpenFlag);
        return DRV_ERR_NOT_OPENED;
    }
    
    if(gzDrvAudio_Status==DD_AUDIO_PLAYING)
    {
        ret = halAudio_PlayStop();
        if ( ret < 0 )
    	{
    		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"ddAudio_Close halAudio_PlayStop err ret %d\n", ret);  
		}    
        gzDrvAudio_Status = DD_AUDIO_IDLE;
    }
	
    if(gzDrvAudio_Status == DD_AUDIO_RECORDING)
    {
        ret = halAudio_RcdStop();
        if ( ret < 0 )
        {
            zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"ddAudio_Close halAudio_RcdStop err ret %d\n", ret);
        }
        gzDrvAudio_Status = DD_AUDIO_IDLE;
    }

    ret = halAudio_Close();
    if ( ret < 0 )
    {
    	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"ddAudio_Close halAudio_Close err ret %d\n", ret);  
    }
	
    gzDrvAudio_OpenFlag = DD_CLOSED;
    return ret;
}

/**************************************************************************
* Function: ddAudio_Ioctl
* Description:  This function is used to control audio device.
* Parameters: 
*   Input:
*           private:device private data
*           cmd:command code
*           param:command parameters
*   Output:None
* Returns: 
*           T_ZDrv_ErrCode
*             others: if errcode<-100 ,please refer to the hal layer errcode file

* Others:     cmd-arg 
                     1  CMD:IOCTL_AUDIO_RECORD_START:
                              arg: T_ZDrvRcd_InfoParam*
                            return:
         
                     2  CMD:IOCTL_AUDIO_RECORD_STOP:
                              arg: void
                            return: 
         
                     3  CMD:IOCTL_AUDIO_SET_RECORD_PATH:
                              arg: T_HalAudio_Block*
                            return:   
           
                     4  CMD:IOCTL_AUDIO_SET_RECORD_VOL:
                              arg: T_HalAudio_Block*
                            return:      

                     5  CMD:IOCTL_AUDIO_RCD_FREE_BUFF:
                              arg: VOID
                            return: 

                     6  CMD:IOCTL_AUDIO_SET_SAMPLE:
                              arg: T_ZDrvAudio_SampleRate*
                            return: 
         
                     7  CMD:IOCTL_AUDIO_SET_BANLANCE:
                              arg: T_ZDrvAudio_Banlance*
                            return: 

                     8  CMD:IOCTL_AUDIO_SET_OUTPUT_PATH:
                              arg: T_ZDrv_AudioOutputPath*
                            return: 
                               
                     9  CMD:IOCTL_AUDIO_SET_OUTPUT_VOL:
                              arg: T_ZDrvAudio_OutputVolLevel*
                            return:    
         
                     10  CMD:IOCTL_AUDIO_SET_EQMODE:
                              arg: T_ZDrvAUDIO_EQMODE*
                            return: 
         
                     11  CMD:IOCTL_AUDIO_ENABLE_3DEFFECT:
                              arg: T_ZDrvAudio_3DEFFECT*
                            return: 
                
                     12  CMD:IOCTL_AUDIO_SET_MUTE:
                              arg: BOOL*
                            return:     
        
                     13  CMD:IOCTL_AUDIO_PLAY_START:
                              arg: VOID*;
                            return: 
         
                     14  CMD:IOCTL_AUDIO_PLAY_STOP:
                              arg: VOID*;
                            return: 
         
                     15  CMD:IOCTL_AUDIO_PLAY_PAUSE:
                              arg: VOID*;
                            return: 
         
                     16  CMD:IOCTL_AUDIO_PLAY_RESUME:
                              arg: VOID*;
                            return: 
         
                     17  CMD:IOCTL_AUDIO_GET_PLAY_INFO:
                              arg: T_ZDrvAudio_PlayInfo*
                            return:    
         
                     18  CMD:IOCTL_AUDIO_GET_BUFFER:
                              arg: T_ZDrvAudio_BufInfo*
                            return: 
                            
                     19  CMD:MAX_IOCTL_AUDIO
                          ERROR CMD INPUT        
                                                               
**************************************************************************/
SINT32 ddAudio_Ioctl (VOID *devData, T_DRVIO_CTRL_KEY function, VOID* arg)
{
    SINT32 ret = 0;
    
    if (gzDrvAudio_OpenFlag != DD_OPENED)
    {
        return DRV_ERR_NOT_OPENED;
    }
    
    switch(function)   
    {
         case IOCTL_AUDIO_RECORD_START:
         {
            if(arg == NULL)
                return DRV_ERR_INVALID_PARAM;
            
             ret = halAudio_RcdStart((T_ZDrvRcd_InfoParam*)arg);
			if(ret<0)
			    return ret;
			gzDrvAudio_Status = DD_AUDIO_RECORDING;
            break;
         }
         
         case IOCTL_AUDIO_RECORD_STOP:
         {
	        if(gzDrvAudio_Status == DD_AUDIO_RECORDING)
	        {
             ret = halAudio_RcdStop();
	            if(ret<0)
	                return ret;
	            gzDrvAudio_Status = DD_AUDIO_IDLE;
	        }
             break; 
         }
                
         case IOCTL_AUDIO_RCD_FREE_BUFF:
         {
            if(arg == NULL)
                return DRV_ERR_INVALID_PARAM;
           
             ret = halAudio_FreeRcdBuf((VOID*)arg);
             break; 
         }

         case IOCTL_AUDIO_SET_SAMPLE:
         {
            if(arg == NULL)
                return DRV_ERR_INVALID_PARAM;
            
             ret = halAudio_SetSample((T_ZDrvAudio_SampleRate*)arg);
            break;
         }

         case IOCTL_AUDIO_SET_RECORD_SAMPLE:
         {
            if(arg == NULL)
                return DRV_ERR_INVALID_PARAM;
            
             ret = halAudio_SetRcdSample((T_ZDrvAudio_SampleRate*)arg);
            break;
         }
         
         case IOCTL_AUDIO_SET_RECORD_PATH:
         {
            if(arg == NULL)
                return DRV_ERR_INVALID_PARAM;
              
             ret = halAudio_SetRecordPath((T_ZDrv_CodecInputPath*)arg);
             break; 
         }  
           
         case IOCTL_AUDIO_SET_RECORD_VOL:
         {
             if(arg == NULL)
                return DRV_ERR_INVALID_PARAM;
             
             ret = halAudio_SetRecordVol((T_ZDrvAudio_InputVolLevel*)arg);
             break; 
         }  

         case IOCTL_AUDIO_SET_OUTPUT_PATH:
         {
             if(arg == NULL)
                 return DRV_ERR_INVALID_PARAM;
         
             ret = halAudio_SetOutputPath((T_ZDrv_AudioOutputPath*)arg);
             break; 
         }    
         case IOCTL_AUDIO_SET_OUTPUT_VOL:
         {
             if(arg == NULL)
                 return DRV_ERR_INVALID_PARAM;

              ret = halAudio_SetOutputVol((T_ZDrvAudio_OutputVolLevel*)arg);
              break; 
         }  
         
         case IOCTL_AUDIO_SET_BANLANCE:
         {
            if(arg == NULL)
                return DRV_ERR_INVALID_PARAM;
            
             ret = halAudio_SetBanlance((T_ZDrvAudio_Banlance*)arg);
             break; 
         }  
         
         case IOCTL_AUDIO_SET_EQMODE:
         {
            if(arg == NULL)
                return DRV_ERR_INVALID_PARAM;

             ret = halAudio_SetEqMode((T_ZDrvAUDIO_EQMODE*)arg);
             break; 
         }
         
         case IOCTL_AUDIO_ENABLE_3DEFFECT:
         {
             if(arg == NULL)
                return DRV_ERR_INVALID_PARAM;
            
             ret = halAudio_Enable3DEffect((T_ZDrvAudio_3DEFFECT*)arg);
             break; 
         }
                
         case IOCTL_AUDIO_SET_MUTE:
         {
             if(arg == NULL)
                return DRV_ERR_INVALID_PARAM;    
           
             ret = halAudio_SetMute((BOOL*)arg);
             break; 
         }    
        
         case IOCTL_AUDIO_PLAY_START:
         {
             if(gzDrvAudio_Status== DD_AUDIO_PLAYING)
                 return DRV_ERR_BUSY;
                    
             ret = halAudio_PlayStart((T_ZDrvAudio_PlayParam *)arg);
             if(ret<0)
                return ret;
             gzDrvAudio_Status=DD_AUDIO_PLAYING;
             break;
         }
         
         case IOCTL_AUDIO_PLAY_STOP:
         {
             if(gzDrvAudio_Status==DD_AUDIO_PLAYING)
             {
                 ret = halAudio_PlayStop();
                 if(ret<0)
                     return ret;
                 gzDrvAudio_Status=DD_AUDIO_IDLE;
             }
             break;
         }
         
         case IOCTL_AUDIO_PLAY_PAUSE:
         {
            ret = halAudio_PlayPause();
            
            break;
         }
         
         case IOCTL_AUDIO_PLAY_RESUME:
         {
            ret = halAudio_PlayResume();
            break;
         }
         
         case IOCTL_AUDIO_GET_PLAY_INFO:
         {
             if(arg == NULL)
                    return DRV_ERR_INVALID_PARAM;
                   
             ret = halAudio_GetPlayInfo((T_ZDrvAudio_PlayInfo*)arg);
             break; 
         }    
         
         case IOCTL_AUDIO_GET_BUFFER:
         {
             if(arg == NULL)
                return DRV_ERR_INVALID_PARAM;       
             
             ret = halAudio_GetBuf((T_ZDrvAudio_BufInfo*)arg);
             break; 
         } 
         
         case IOCTL_AUDIO_SET_CALLBACK:
         {
             if(arg == NULL)
                return DRV_ERR_INVALID_PARAM;       
             
             ret = halAudio_SetCallBackFunc((T_ZDrvAudio_CallbackFunc)arg);
             break; 
         }   
         
        default:
            return DRV_ERR_INVALID_IOCTL_CMD;
    }
  
     return ret;
   
}

/**************************************************************************
* Function: ddAudio_Read
* Description: This function is used to read stream data from device.
* Parameters:
*   Input:
*      pPrivData:device private data
*      pBuf: pointer to the written data
*      uiLen: the size of written data
*   Output:None
* Returns:
*           T_ZDrv_ErrCode
*             others: if errcode<-100 ,please refer to the hal layer errcode file
* Others: None
**************************************************************************/
SINT32 ddAudio_Read(VOID *devData, VOID *pBuf, UINT32 uiLen)
{
    SINT32 ret = 0;

    if (gzDrvAudio_OpenFlag != DD_OPENED)
    {
        return DRV_ERR_NOT_OPENED;
    }
    if (NULL == pBuf)
    {
        return DRV_ERR_INVALID_PARAM;
    }
    ret = halAudio_GetRcdBuf((T_ZDrvAudio_BufInfo *)pBuf);

    return ret;
}

/**************************************************************************
* Function: ddAudio_Write
* Description: This function is used to write stream data into device.
* Parameters:
*   Input:
*      pPrivData:device private data
*      pBuf: pointer to the written data
*      uiLen: the size of written data
*   Output:None
* Returns:
*     write data length:success to write 
*           T_ZDrv_ErrCode
*             others: if errcode<-100 ,please refer to the hal layer errcode file
* Others: None
**************************************************************************/
SINT32 ddAudio_Write(VOID *devData, const VOID *pBuf, UINT32 uiLen)
{
    SINT32 ret = 0;

    if (gzDrvAudio_OpenFlag != DD_OPENED)
    {
        return DRV_ERR_NOT_OPENED;
    }
    if (NULL == pBuf)
    {
        return DRV_ERR_INVALID_PARAM;
    }
    ret = halAudio_Write((UINT32 *)pBuf, uiLen);

    return ret;

}

