

#ifndef _HAL_UICC_H
#define _HAL_UICC_H
/*---------------------------------------------*/
/* Include files.                              */
/*---------------------------------------------*/
#include "drvs_uicc.h"
#include "uicc.h"
/*---------------------------------------------*/
/* Type define.                                */
/*---------------------------------------------*/
typedef unsigned char       ubyte;         /*           0 to +255        */
typedef unsigned short     ushort;


/*---------------------------------------------*/
/* Enumeration.                                */
/*---------------------------------------------*/
#define UICC_MAX_ATR_CHARACTERS    50 /* The maximum number of ATR characters. */

#ifndef UICC_EXTENDED_ATR_CHAR
  #define UICC_MAX_HIST_CHARACTERS   15 /* The maximum number of historical characters received 
                                           in the Answer To Reset sequence. */ 
#endif


#define UICC_NOF_CLASSES           5 /* The number of specified card classes (A,B,C,D & E) */

#define UICC_USE_ZX2802DMA

/* Not supported by HW 
typedef enum {
  CALLBACK_FUNCTION_SET,
  CALLBACK_NOT_AVAILABLE 
} T_UICC_SET_INOUT_CALLBACK;
**/
typedef enum {
  UICC_CMD_CASE_1,    /* No Tx nor Tx command. */
  UICC_CMD_CASE_2,    /* Rx command.           */ 
  UICC_CMD_CASE_3,    /* Tx command.           */
  UICC_CMD_CASE_4     /* Tx and Rx command.    */
} T_UICC_COMMAND_CASE;

typedef enum {
  UICC_COLD_RESET,
  UICC_WARM_RESET
} T_UICC_RESET_MODE;
#if 0
typedef enum {
  UICC_CLASS_A,          /* 5v   */
  UICC_CLASS_B,          /* 3v   */
  UICC_CLASS_C,          /* 1.8v */
  UICC_CLASS_D,          /* RFU  */
  UICC_CLASS_E           /* RFU  */
} T_UICC_VOLTAGE_CLASS;
#endif
typedef enum {
  UICC_CLOCK_STOP_ALLOWED,
  UICC_NO_PREFERRED_LEVEL,
  UICC_HIGH_LEVEL_PREFERRED,
  UICC_LOW_LEVEL_PREFERRED,
  UICC_CLOCK_STOP_NOT_ALLOWED,
  UICC_CLOCK_STOP_ONLY_HIGH,
  UICC_CLOCK_STOP_ONLY_LOW
} T_UICC_CLOCK_STOP_MODE;

typedef enum {
  UICC_T_0,           /* T=0 Protocol. */
  UICC_T_1            /* T=1 Protocol. */
} T_UICC_PROTOCOL;

typedef enum {
  UICC_DIRECT,        /* Direct convention.  */
  UICC_INVERSE        /* Inverse convention. */
} T_COM_CONVENTION;

/*---------------------------------------------*/
/* External declarations                       */
/*---------------------------------------------*/
typedef struct {
  ubyte  cla;
  ubyte  ins;
  ubyte  p1;
  ubyte  p2;
  ushort lc;   /* Tx-size. */
  ushort le;   /* Rx-Size. */
} T_UICC_APDU_HEADER;

typedef struct {
  ushort luicc; /* Actual Rx-size recived. */
  ubyte  sw1;
  ubyte  sw2;
} T_UICC_APDU_FOOTER;

typedef struct {
  T_UICC_VOLTAGE_CLASS used_voltage;
  BOOL                 voltage_settled;
} T_UICC_VOLTAGE_CONTROL;

typedef struct {
  T_UICC_PROTOCOL  protocol;
  ubyte            f_used;
  ubyte            f_offered;
  ubyte            d_used;
  ubyte            d_offered;
  T_COM_CONVENTION com_convention;
} T_UICC_CHARACTERISTICS;

typedef struct {
  BOOL    class_supported; /* Indicates if the class is supported or not. */
  ushort  voltage_level;   /* Holds the voltage level given in millivolts [mV]. E.g. 1800 is 
                              equivalent to 1.8v). */
                                       
  ushort  max_current;     /* The maximum allowed current consumption at this voltage_level 
                              given in micro Ampere (uA). E.g. 15500 is equivalent to 15.5mA. */
} T_UICC_ELECTRICAL_PROFILE;

typedef struct {
  ubyte  min_clock_freq;   /* The minimum clock frequency supported by the HW. */
  ubyte  max_clock_freq;   /* The maximum clock frequency supported by the HW. If only 
                              one frequency is supported, the min. and max. values will 
                              be identical. The frequency resolution is 0.1 MHz i.e. 21h
                              is equivalent to 3.3MHz. */
} T_UICC_CLOCK_PROFILE;


typedef struct {
  ubyte                     nof_card_slots_supported;  
                            /* Number of card reader slots supported by HW.   
                             * 1: Only primary slot (uicc_primary_card)   
                             * 2: Primary and ONE additional slot (uicc_secondary_card_1)    
                             * 3: Primary and TWO additional slots (uicc_secondary_card_1 & uicc_secondary_card_2) */ 

  ubyte                     protocol_supported;        
                            /* Protocols supported by the UICC Handler.   
                             * b1: T=0 supported   
                             * b2: T=1 supported   
                             * b3: T=x supported   
                             * E.g. 0x03 means that both T=0 and T=1 are supported */

  T_UICC_ELECTRICAL_PROFILE electrical_profile[UICC_NOF_CLASSES]; 
                            /* Holds the electrical profile of the specified classes - where 
                             * the max. allowed current consumption is indicated for each class. */

  T_UICC_CLOCK_PROFILE      clock_profile;
                            /* Holds the minimum and maximum clock frequencies supported by the 
                             * hardware. */  

  BOOL                      extended_length;           
                            /* Indicates the driver support of the use of extended Lc/Le. 
                             * TRUE:  extended Lc/Le is supported by the driver. The max. Tx and 
                             *        Rx data sizes are 65,535 and 65,536 bytes respectively.
                             * FALSE: extended Lc/Le is NOT supported by the driver. The max. Tx 
                             *        and Rx data sizes are 255 and 256 bytes respectively. */
} T_UICC_HW_PROFILE;

/*---------------------------------------------*/
/* External functions.                         */
/*---------------------------------------------*/

/*******************************************************************************
* Function:... UICC_transport_APDU
* Parameters:. 
*    card_selector   (I)   Indicates which card the requested command is meant for.
*    command_case    (I)   Indicates which of the four command cases the current 
*                          command belongs to.
*    extended_length (I)   Indicates the card support of the use of extended Lc/Le. 
*                          The caller of this function has the responsibility of 
*                          allocating the apdu_data store correcesponding to the 
*                          larger of Lc and Le.
*                          TRUE:  extended Lc/Le is supported. The max. Tx and Rx  
*                                 data sizes are 65,535 and 65,536 bytes respectively.
*                          FALSE: extended Lc/Le is NOT supported. The max. Tx and Rx  
*                                 data sizes are 255 and 256 bytes respectively.
*    c_apdu          (I)   The Command APDU containing: CLA, INS, P1, P2, Lc & Le.
*    r_apdu          (O)   The Response APDU containing: Luicc, SW1 & SW2.
*    apdu_data       (I/O) I: Holds Lc Tx-data to be sent to the card.
*                          O: Holds Luicc Rx-data received from the card.
*
* Returns:.... The result of the APDU command execution.
*
* Description: Function used to exchange APDU with a card. The function will 
*              not return before the response APDU has beed received from the
*              card or until the appropriate Working Timer has expired.
*******************************************************************************/
extern T_UICC_APDU_RESULT UICC_transport_APDU(T_UICC_CARD_SELECTOR  card_selector,
                                              T_UICC_COMMAND_CASE   command_case,
                                              BOOL                  extended_length, 
                                              T_UICC_APDU_HEADER    c_apdu,
                                              T_UICC_APDU_FOOTER    *r_apdu,
                                              ubyte                 *apdu_data);


#ifdef UICC_EXTENDED_ATR_CHAR
  /*******************************************************************************
  * Function:... UICC_reset
  * Parameters:. 
  *    card_selector       (I) Indicates which card needs to be activated/reset.
  *    reset_mode          (I) Indicates whether a cold or a warm reset is desired.
  *    activation_voltage  (I) The voltage level at which the activation should be 
  *                            carried out.
  *    voltage_control     (O) Holds the actual voltage level used in this specific
  *                            activation. The parameter also holds an indication of
  *                            whether or not the used voltage is the final voltage.
  *    nof_atr_characters  (O) Number of AnserToReset characters.
  *    atr_characters      (O) Contains AnswerTo Reset characters as they where 
  *                            received during the activation sequence. Historical 
  *                            characters are located at the end of the ATR string 
  *                            and the number of historical characters are given by 
  *                            the LSB nipple of ATR character 2.
  *                            i.e. historical character are located 
  *                            from: atr_characters[nof_atr_characters-1-(atr_characters[1] & 0x0F)] 
  *                            to:   atr_characters[nof_atr_characters-1]
  *    characteristics     (O) Hold "nice to know" characteristics of the operation 
  *                            of the current card.
  * Returns:.... The outcome of the activation handling.
  *
  * Description: This function is used by ATC during production test to set CI in
  *******************************************************************************/
  extern T_UICC_RESET_RESULT UICC_reset(T_UICC_CARD_SELECTOR   card_selector,
                                        T_UICC_RESET_MODE      reset_mode,
                                        T_UICC_VOLTAGE_CLASS   activation_voltage,
                                        T_UICC_VOLTAGE_CONTROL *voltage_control,
                                        ubyte                  *nof_atr_characters,
                                        ubyte                  *atr_characters,
                                        T_UICC_CHARACTERISTICS *characteristics);
#else
  /*******************************************************************************
  * Function:... UICC_reset
  * Parameters:. 
  *    card_selector       (I) Indicates which card needs to be activated/reset.
  *    reset_mode          (I) Indicates whether a cold or a warm reset is desired.
  *    activation_voltage  (I) The voltage level at which the activation should be 
  *                            carried out.
  *    voltage_control     (O) Holds the actual voltage level used in this specific
  *                            activation. The parameter also holds an indication of
  *                            whether or not the used voltage is the final voltage.
  *    nof_hist_characters (O) Number of characters in hist_characters.
  *    hist_characters     (O) Contains the historivcal characters received during the
  *                            ATR sequence.
  *    characteristics     (O) Hold "nice to know" characteristics of the operation 
  *                            of the current card.
  * Returns:.... The outcome of the activation handling.
  *
  * Description: This function is used by ATC during production test to set CI in
  *******************************************************************************/
  extern T_UICC_RESET_RESULT UICC_reset(T_UICC_CARD_SELECTOR   card_selector,
                                        T_UICC_RESET_MODE      reset_mode,
                                        T_UICC_VOLTAGE_CLASS   activation_voltage,
                                        T_UICC_VOLTAGE_CONTROL *voltage_control,
                                        ubyte                  *nof_hist_characters,
                                        void                   *hist_characters,
                                        T_UICC_CHARACTERISTICS *characteristics);
#endif
/*******************************************************************************
* Function:... UICC_ResetCard
* Parameters:. .
*    card_selector       (I) Indicates which card needs to be activated.
* Returns:.... The deactivation result (KNM: is it really necessary).
*
* Description: Function used top deactivat a given card.
*******************************************************************************/
T_UICC_RESET_RESULT UICC_ResetCard(T_UICC_CARD_SELECTOR  card_selector);


/*******************************************************************************
* Function:... UICC_close
* Parameters:. .
*    card_selector       (I) Indicates which card needs to be deactivated.
* Returns:.... The deactivation result (KNM: is it really necessary).
*
* Description: Function used top deactivat a given card.
*******************************************************************************/
extern T_UICC_CLOSE_RESULT UICC_close(T_UICC_CARD_SELECTOR card_selector);


/*******************************************************************************
* Function:... UICC_card_characteristics
* Parameters:.
*    card_selector       (I) Indicates which card the characteristics are valid 
*                            for.
*    clock_stop_mode     (I) Holds the Clock Stop capabilities indicated by the 
*                            card.
*    min_clock_freq      (I) Holds the minimum acceptable clock frequency for the
*                            card/application. The frequency resolution is 0.1 MHz 
*                            i.e. 21h is equivalent to 3.3MHz. 
* Returns:.... None.
*
* Description: Informs the UICC Handler of the characteristics of the current used 
*              card.
*******************************************************************************/
extern void UICC_card_characteristics(T_UICC_CARD_SELECTOR   card_selector,
                                      T_UICC_CLOCK_STOP_MODE clock_stop_mode,
                                      ubyte                  min_clock_freq);


/*******************************************************************************
* Function:... UICC_hw_profile
* Parameters:. None.
* Returns:.... The static characteristics (e.g. number of card slots, voltage 
*              levels, max. current cunsumption, supported transmission protocols 
*              etc.
*
* Description: Returns an overview of the static profile of the HW related 
*              interface.
*******************************************************************************/
extern T_UICC_HW_PROFILE UICC_hw_profile(void);


/*******************************************************************************
* Function:... UICC_set_inout_callback
* Parameters:. inout_call_back (I) Pointer to the function to be invoked when the 
*                                  in/out state changes. The parameter to the call 
*                                  back function shall indicated the related card 
*                                  reader and the IN or OUT action.
* Returns:.... The outcome of the requested action.
*
* Description: Used for initializing the callback function used when the in/out 
*              state of the card changes.
*******************************************************************************/
extern T_UICC_SET_INOUT_CALLBACK UICC_set_inout_callback( void (*inout_call_back)(T_ZDrvUicc_CardAction card_action, T_ZDrvUicc_CardSelector card_selector));


/*******************************************************************************
* Function:... UICC_get_atr 
* Parameters:. Pointer to the store in which the ATR string should be located.
* Returns:.... Number of ATR characters.
* Description: Function used to obtain the ATR string of the current operated 
*              card. If no card is active (or if data is not available) the number 
*              of characters returned will be set to zero.
* Created:.... 15.08.05 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
extern ubyte UICC_get_atr (ubyte *atr);


/******************************************************************************/
/*******************************              *********************************/
/*****************************                  *******************************/
/***************************        ASSUMING       ****************************/
/*************************        CI INTERFACE      ***************************/
/***************************  ~~~~~~~~~~~~~~~~~~  *****************************/
/*****************************                  *******************************/
/*******************************              *********************************/
/******************************************************************************/

#ifdef UICC_ASSUME_CI_INTERFACE
  typedef enum
  {
    sim_rx_cmd,
    sim_tx_cmd
  } sim_cmdDir_type;

  typedef enum
  {
    sim_3v,
    sim_5v,
    sim_1_8v
  } sim_voltage_type;

  typedef enum
  {
    sim_no_stop        = 0,
    sim_stop_allowed   = 1,
    sim_high_stop_only = 4,
    sim_high_stop_pref = 5,
    sim_low_stop_only  = 8,
    sim_low_stop_pref  = 9
  } sim_clkStop_mode_type;

  typedef enum
  {
    ci_direct,
    ci_inverse
  } ci_communication_convention_type;

  typedef struct
  {
    unsigned char *speed_indication;
    unsigned char *conv_clock_status;
    unsigned char *ci_status;
  } ci_sim_characteristics_type;

  extern volatile ubyte UICC_ptst_sem;

  extern void CIInit(void);
  extern ci_sim_characteristics_type CI_get_sim_characteristics(void);
  extern SDL_Void SIM_WriteCmd (sim_cmdDir_type  sim_cmdDir,
                                SDL_Integer      sim_lgth,
                                T_BYTE_PTR       sim_cmdPtr);
  extern SDL_Boolean SIM_SetCardPara(sim_voltage_type       sim_voltage,
                                     sim_clkStop_mode_type  sim_clkStop_mode);
  extern SDL_Void SIM_ResetCard(SDL_Void);
  extern SDL_Void SIM_DeactivateCard(SDL_Void);

  #define SIM_GetATR  UICC_get_atr
#endif /* UICC_ASSUME_CI_INTERFACE */


#endif /* _HAL_UICC_H */
                                                 /* End of file.              */
