#include "drvs_general.h"
#include "drvs_voiceprocess.h"
#include "drvs_ramlog.h"
#include "hal_nxp.h"

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/miscdevice.h>
#include <asm/uaccess.h>
#include <linux/slab.h>
#include <linux/init.h>

typedef struct testnxp{
	int fs;
	int voicemode;
	int vp_path; 
	int vp_vol;
	int framecount;
}T_TestNxp;

#define TESTNXP_IOCTL_RXIN_TO_RXOUT_START	_IOW('N', 1, T_TestNxp)
#define TESTNXP_IOCTL_RXIN_TO_RXOUT_STOP	_IO('N', 2)

#define MAX_NXP_BUF	640


static const char testnxp_shortname[] = "testnxp_dev";
static unsigned char *rxin_buf = NULL;
static unsigned char *rxout_buf = NULL;
static int framecount;
static int testnxp_open(struct inode * ip,struct file * fp);
static int testnxp_release(struct inode *ip, struct file *fp);
static ssize_t testnxp_read(struct file *fp, char __user *buf,size_t count, loff_t *pos);
static ssize_t testnxp_write(struct file *fp, const char __user *buf,size_t count, loff_t *pos);
static long testnxp_ioctl(struct file *fp, unsigned int cmd, unsigned long arg);




/* file operations for testnxp device /dev/testnxp_device */
static const struct file_operations testnxp_fops = {
	.owner = THIS_MODULE,
	.read = testnxp_read,
	.write = testnxp_write,
	.unlocked_ioctl = testnxp_ioctl,
	.open = testnxp_open,
	.release = testnxp_release,
};

static struct miscdevice testnxp_device = {
	.minor = MISC_DYNAMIC_MINOR,
	.name = testnxp_shortname,
	.fops = &testnxp_fops,
};



static ssize_t testnxp_read(struct file *fp, char __user *buf,size_t count, loff_t *pos)
{

	if(buf == NULL)
	{
		pr_info("testnxp_read buffer null \n");
		return -EINVAL;
	}

	if ((count>0) &&(count<=MAX_NXP_BUF)) {
		if (copy_to_user(buf, rxout_buf, count)) {
			pr_info("testnxp_read copy_to_user error \n");
			return -EFAULT;
		}
	}else{
		pr_info("testnxp_read count error \n");
		return -EINVAL;
	}
	return count;
}

static ssize_t testnxp_write(struct file *fp, const char __user *buf,size_t count, loff_t *pos)
{
	int ret;

	if(buf == NULL)
	{
		pr_info("testnxp_write buffer null \n");
		return -EINVAL;
	}

	if ((count > 0) && (count <= MAX_NXP_BUF)) {
		if (copy_from_user(rxin_buf, buf, count)) {
			pr_info("testnxp_write copy_from_user error \n");
			return -EFAULT;
		}
		
		Vp_NxpRxProcess(&rxin_buf, &rxout_buf, count/2);
		
	}else{
		pr_info("testnxp_write count error \n");
		return -EINVAL;
	}

	return count;
}


static long testnxp_ioctl(struct file *fp, unsigned int cmd, unsigned long arg)
{
    T_TestNxp *param = (T_TestNxp *)arg;
	T_TestNxp testnxp_param = {0};
	int ret = 0;
	printk("testnxp_ioctl\n");

	switch(cmd)
	{
		case TESTNXP_IOCTL_RXIN_TO_RXOUT_START:
		{
			if (copy_from_user((void*)&testnxp_param, (void*)param, sizeof(struct testnxp)))
			{
				return DRV_ERR_INVALID_PARAM;
			}
			framecount = testnxp_param.framecount;
			ret = Vp_NxpOpen(testnxp_param.fs);	
			if (ret == 0) 
			{
				ret = Vp_NxpGetParam(testnxp_param.voicemode, testnxp_param.vp_path);
				if (ret == 0)
				{
					ret = Vp_NxpSetParam(testnxp_param.vp_path, testnxp_param.vp_vol, FALSE);
					if (ret != 0 )
						Vp_NxpClose();
				}
				else
				{
					Vp_NxpClose();
				}
			}
			break;
		}
		case TESTNXP_IOCTL_RXIN_TO_RXOUT_STOP:
		{
			Vp_NxpClose();
		}
		default:
		{
			break;
		}
	}
	return ret;
}



static int testnxp_open(struct inode *ip, struct file *fp)
{
	printk("testnxp_open\n");

	if(rxin_buf == NULL)
	{
		rxin_buf = kzalloc(2 * MAX_NXP_BUF, GFP_KERNEL);
		if (!rxin_buf)
		{
		    pr_info("testnxp_init malloc error\n");
			return -ENOMEM;
		}
		
		rxout_buf = rxin_buf + MAX_NXP_BUF;
	}

	return 0;
}

static int testnxp_release(struct inode *ip, struct file *fp)
{
	printk("testnxp_release\n");

	if(rxin_buf)
	{
		kfree(rxin_buf);
		rxin_buf = NULL;
	}

	return 0;
}


// int __init testnxp_init(void)
int  testnxp_init(void)
{
	int ret;
	printk("testnxp driver initialize\n");

	ret = misc_register(&testnxp_device);
	if (ret){
	    printk(KERN_ERR "testnxp driver failed to initialize\n");
		return -EFAULT;
	}

    return 0;
}


 void __exit testnxp_exit(void)
{
	misc_deregister(&testnxp_device);
}

//late_initcall(testnxp_init);
//module_exit(testnxp_exit);

