#ifndef ZLIB_H
#define ZLIB_H

#include "zcore_type.h"
#include "zconf.h"

#ifdef __cplusplus
extern "C" {
#endif

#define ZLIB_VERSION "1.2.2"
#define ZLIB_VERNUM 0x1220

typedef voidpf (*alloc_func) OF((voidpf opaque, uInt items, uInt size));
typedef ZVOID   (*free_func)  OF((voidpf opaque, voidpf address));

struct internal_state;

typedef struct z_stream_s {
    Bytef    *next_in;  /* next input byte */
    uInt     avail_in;  /* number of bytes available at next_in */
    uLong    total_in;  /* total nb of input bytes read so far */

    Bytef    *next_out; /* next output byte should be put there */
    uInt     avail_out; /* remaining free space at next_out */
    uLong    total_out; /* total nb of bytes output so far */

    ZCHAR     *msg;      /* last error message, NULL if no error */
    struct internal_state FAR *state; /* not visible by applications */

    alloc_func zalloc;  /* used to allocate the internal state */
    free_func  zfree;   /* used to free the internal state */
    voidpf     opaque;  /* private data object passed to zalloc and zfree */

    ZSINT32     data_type;  /* best guess about the data type: ascii or binary */
    uLong   adler;      /* adler32 value of the uncompressed data */
    uLong   reserved;   /* reserved for future use */
} z_stream;

typedef z_stream FAR *z_streamp;



#define Z_NO_FLUSH      0
#define Z_PARTIAL_FLUSH 1 /* will be removed, use Z_SYNC_FLUSH instead */
#define Z_SYNC_FLUSH    2
#define Z_FULL_FLUSH    3
#define Z_FINISH        4
#define Z_BLOCK         5
/* Allowed flush values; see deflate() and inflate() below for details */

#define Z_OK            0
#define Z_STREAM_END    1
#define Z_NEED_DICT     2
#define Z_ERRNO        (-1)
#define Z_STREAM_ERROR (-2)
#define Z_DATA_ERROR   (-3)
#define Z_MEM_ERROR    (-4)
#define Z_BUF_ERROR    (-5)
#define Z_VERSION_ERROR (-6)
/* Return codes for the compression/decompression functions. Negative
 * values are errors, positive values are used for special but normal events.
 */

#define Z_NO_COMPRESSION         0
#define Z_BEST_SPEED             1
#define Z_BEST_COMPRESSION       9
#define Z_DEFAULT_COMPRESSION  (-1)
/* compression levels */

#define Z_FILTERED            1
#define Z_HUFFMAN_ONLY        2
#define Z_RLE                 3
#define Z_DEFAULT_STRATEGY    0
/* compression strategy; see deflateInit2() below for details */

#define Z_BINARY   0
#define Z_ASCII    1
#define Z_UNKNOWN  2
/* Possible values of the data_type field (though see inflate()) */

#define Z_DEFLATED   8
/* The deflate compression method (the only one supported in this version) */

#define Z_NULL  0  /* for initializing zalloc, zfree, opaque */

#define zlib_version zlibVersion()
/* for compatibility with versions < 1.0.2 */

                        /* basic functions */

ZEXTERN const ZCHAR * ZEXPORT zlibVersion OF((ZVOID));


ZEXTERN ZSINT32 ZEXPORT deflate OF((z_streamp strm, ZSINT32 flush));

ZEXTERN ZSINT32 ZEXPORT deflateEnd OF((z_streamp strm));


ZEXTERN ZSINT32 ZEXPORT inflate OF((z_streamp strm, ZSINT32 flush));


ZEXTERN ZSINT32 ZEXPORT inflateEnd OF((z_streamp strm));

ZEXTERN ZSINT32 ZEXPORT deflateSetDictionary OF((z_streamp strm,
                                             const Bytef *dictionary,
                                             uInt  dictLength));

ZEXTERN ZSINT32 ZEXPORT deflateCopy OF((z_streamp dest,
                                    z_streamp source));
ZEXTERN ZSINT32 ZEXPORT deflateReset OF((z_streamp strm));

ZEXTERN ZSINT32 ZEXPORT deflateParams OF((z_streamp strm,
                                      ZSINT32 level,
                                      ZSINT32 strategy));
ZEXTERN uLong ZEXPORT deflateBound OF((z_streamp strm,
                                       uLong sourceLen));

ZEXTERN ZSINT32 ZEXPORT deflatePrime OF((z_streamp strm,
                                     ZSINT32 bits,
                                     ZSINT32 value));


ZEXTERN ZSINT32 ZEXPORT inflateSetDictionary OF((z_streamp strm,
                                             const Bytef *dictionary,
                                             uInt  dictLength));

ZEXTERN ZSINT32 ZEXPORT inflateSync OF((z_streamp strm));

ZEXTERN ZSINT32 ZEXPORT inflateCopy OF((z_streamp dest,
                                    z_streamp source));

ZEXTERN ZSINT32 ZEXPORT inflateReset OF((z_streamp strm));

typedef unsigned (*in_func) OF((ZVOID FAR *, ZUINT8 FAR * FAR *));
typedef ZSINT32 (*out_func) OF((ZVOID FAR *, ZUINT8 FAR *, unsigned));

ZEXTERN ZSINT32 ZEXPORT inflateBack OF((z_stream FAR *strm,
                                    in_func in, ZVOID FAR *in_desc,
                                    out_func out, ZVOID FAR *out_desc));

ZEXTERN ZSINT32 ZEXPORT inflateBackEnd OF((z_stream FAR *strm));

ZEXTERN uLong ZEXPORT zlibCompileFlags OF((ZVOID));


ZEXTERN ZSINT32 ZEXPORT compress OF((Bytef *dest,   uLongf *destLen,
                                 const Bytef *source, uLong sourceLen));

ZEXTERN ZSINT32 ZEXPORT compress2 OF((Bytef *dest,   uLongf *destLen,
                                  const Bytef *source, uLong sourceLen,
                                  ZSINT32 level));

ZEXTERN uLong ZEXPORT compressBound OF((uLong sourceLen));

ZEXTERN ZSINT32 ZEXPORT uncompress OF((Bytef *dest,   uLongf *destLen,
                                   const Bytef *source, uLong sourceLen));


typedef voidp gzFile;

ZEXTERN gzFile ZEXPORT gzopen  OF((const ZCHAR *path, const ZCHAR *mode));

ZEXTERN gzFile ZEXPORT gzdopen  OF((ZSINT32 fd, const ZCHAR *mode));

ZEXTERN ZSINT32 ZEXPORT gzsetparams OF((gzFile file, ZSINT32 level, ZSINT32 strategy));

ZEXTERN ZSINT32 ZEXPORT    gzread  OF((gzFile file, voidp buf, unsigned len));

ZEXTERN ZSINT32 ZEXPORT    gzwrite OF((gzFile file,
                                   voidpc buf, unsigned len));

ZEXTERN ZSINT32 ZEXPORTVA   gzprintf OF((gzFile file, const ZCHAR *format, ...));

ZEXTERN ZSINT32 ZEXPORT gzputs OF((gzFile file, const ZCHAR *s));

ZEXTERN ZCHAR * ZEXPORT gzgets OF((gzFile file, ZCHAR *buf, ZSINT32 len));

ZEXTERN ZSINT32 ZEXPORT    gzputc OF((gzFile file, ZSINT32 c));

ZEXTERN ZSINT32 ZEXPORT    gzgetc OF((gzFile file));

ZEXTERN ZSINT32 ZEXPORT    gzungetc OF((ZSINT32 c, gzFile file));

ZEXTERN ZSINT32 ZEXPORT    gzflush OF((gzFile file, ZSINT32 flush));

ZEXTERN z_off_t ZEXPORT    gzseek OF((gzFile file,
                                      z_off_t offset, ZSINT32 whence));

ZEXTERN ZSINT32 ZEXPORT    gzrewind OF((gzFile file));

ZEXTERN z_off_t ZEXPORT    gztell OF((gzFile file));

ZEXTERN ZSINT32 ZEXPORT gzeof OF((gzFile file));

ZEXTERN ZSINT32 ZEXPORT    gzclose OF((gzFile file));

ZEXTERN const ZCHAR * ZEXPORT gzerror OF((gzFile file, ZSINT32 *errnum));

ZEXTERN ZVOID ZEXPORT gzclearerr OF((gzFile file));

ZEXTERN uLong ZEXPORT adler32 OF((uLong adler, const Bytef *buf, uInt len));



ZEXTERN uLong ZEXPORT crc32   OF((uLong crc, const Bytef *buf, uInt len));

ZEXTERN ZSINT32 ZEXPORT deflateInit_ OF((z_streamp strm, ZSINT32 level,
                                     const ZCHAR *version, ZSINT32 stream_size));
ZEXTERN ZSINT32 ZEXPORT inflateInit_ OF((z_streamp strm,
                                     const ZCHAR *version, ZSINT32 stream_size));
ZEXTERN ZSINT32 ZEXPORT deflateInit2_ OF((z_streamp strm, ZSINT32  level, ZSINT32  method,
                                      ZSINT32 windowBits, ZSINT32 memLevel,
                                      ZSINT32 strategy, const ZCHAR *version,
                                      ZSINT32 stream_size));
ZEXTERN ZSINT32 ZEXPORT inflateInit2_ OF((z_streamp strm, ZSINT32  windowBits,
                                      const ZCHAR *version, ZSINT32 stream_size));
ZEXTERN ZSINT32 ZEXPORT inflateBackInit_ OF((z_stream FAR *strm, ZSINT32 windowBits,
                                         ZUINT8 FAR *window,
                                         const ZCHAR *version,
                                         ZSINT32 stream_size));
#define deflateInit(strm, level) \
        deflateInit_((strm), (level),       ZLIB_VERSION, sizeof(z_stream))
#define inflateInit(strm) \
        inflateInit_((strm),                ZLIB_VERSION, sizeof(z_stream))
#define deflateInit2(strm, level, method, windowBits, memLevel, strategy) \
        deflateInit2_((strm),(level),(method),(windowBits),(memLevel),\
                      (strategy),           ZLIB_VERSION, sizeof(z_stream))
#define inflateInit2(strm, windowBits) \
        inflateInit2_((strm), (windowBits), ZLIB_VERSION, sizeof(z_stream))
#define inflateBackInit(strm, windowBits, window) \
        inflateBackInit_((strm), (windowBits), (window), \
        ZLIB_VERSION, sizeof(z_stream))


#if !defined(ZUTIL_H) && !defined(NO_DUMMY_DECL)
    struct internal_state {ZSINT32 dummy;}; /* hack for buggy compilers */
#endif

ZEXTERN const ZCHAR   * ZEXPORT zError           OF((ZSINT32));
ZEXTERN ZSINT32            ZEXPORT inflateSyncPoint OF((z_streamp z));
ZEXTERN const uLongf * ZEXPORT get_crc_table    OF((ZVOID));

#ifdef __cplusplus
}
#endif

#endif /* ZLIB_H */
