#ifndef _ZCORE_GUI_FIXED_MATH_H_
#define _ZCORE_GUI_FIXED_MATH_H_

#include <errno.h>
#include <math.h>
#include "zcore_type.h"
/* Set up for C function definitions, even when using C++ */
#ifdef __cplusplus
extern "C" {
#endif

#ifdef _FIXED_MATH

MG_EXPORT fixed fsqrt (fixed x);

MG_EXPORT fixed fhypot (fixed x, fixed y);

MG_EXPORT fixed fatan (fixed x);

MG_EXPORT fixed fatan2 (fixed y, fixed x);

MG_EXPORT fixed _cos_tbl[];
MG_EXPORT fixed _tan_tbl[];
MG_EXPORT fixed _acos_tbl[];

static inline fixed ftofix (ZDOUBLE x)
{ 
   if (x > 32767.0) {
      errno = ERANGE;
      return 0x7FFFFFFF;
   }

   if (x < -32767.0) {
      errno = ERANGE;
      return -0x7FFFFFFF;
   }

   return (ZLONG)(x * 65536.0 + (x < 0 ? -0.5 : 0.5)); 
}

static inline ZDOUBLE fixtof (fixed x)
{ 
   return (ZDOUBLE)x / 65536.0; 
}

static inline fixed fadd (fixed x, fixed y)
{
   fixed result = x + y;

   if (result >= 0) {
      if ((x < 0) && (y < 0)) {
	 errno = ERANGE;
	 return -0x7FFFFFFF;
      }
      else
	 return result;
   }
   else {
      if ((x > 0) && (y > 0)) {
	 errno = ERANGE;
	 return 0x7FFFFFFF;
      }
      else
	 return result;
   }
}

static inline fixed fsub (fixed x, fixed y)
{
   fixed result = x - y;

   if (result >= 0) {
      if ((x < 0) && (y > 0)) {
	 errno = ERANGE;
	 return -0x7FFFFFFF;
      }
      else
	 return result;
   }
   else {
      if ((x > 0) && (y < 0)) {
	 errno = ERANGE;
	 return 0x7FFFFFFF;
      }
      else
	 return result;
   }
}

static inline fixed fmul (fixed x, fixed y)
{
   return ftofix(fixtof(x) * fixtof(y));
}

static inline fixed fdiv (fixed x, fixed y)
{
   if (y == 0) {
      errno = ERANGE;
      return (x < 0) ? -0x7FFFFFFF : 0x7FFFFFFF;
   }
   else
      return ftofix(fixtof(x) / fixtof(y));
}


static inline ZSINT32 fceil (fixed x)
{
   x += 0xFFFF;
   if (x >= 0x80000000) {
      errno = ERANGE;
      return 0x7FFF;
   }

   return (x >> 16);
}

static inline fixed itofix (ZSINT32 x)
{ 
   return x << 16;
}

static inline ZSINT32 fixtoi (fixed x)
{ 
   return (x >> 16) + ((x & 0x8000) >> 15);
}

static inline fixed fcos (fixed x)
{
   return _cos_tbl[((x + 0x4000) >> 15) & 0x1FF];
}

static inline fixed fsin (fixed x)
{ 
   return _cos_tbl[((x - 0x400000 + 0x4000) >> 15) & 0x1FF];
}

static inline fixed ftan (fixed x)
{ 
   return _tan_tbl[((x + 0x4000) >> 15) & 0xFF];
}

static inline fixed facos (fixed x)
{
   if ((x < -65536) || (x > 65536)) {
      errno = EDOM;
      return 0;
   }

   return _acos_tbl[(x+65536+127)>>8];
}

static inline fixed fasin (fixed x)
{ 
   if ((x < -65536) || (x > 65536)) {
      errno = EDOM;
      return 0;
   }

   return 0x00400000 - _acos_tbl[(x+65536+127)>>8];
}


#endif /* _FIXED_MATH */

#ifdef __cplusplus
}
#endif

#endif /* _MGUI_FIXED_MATH_H_ */

