#ifndef __MTCACHE_H__
#define __MTCACHE_H__

#include <ft2build.h>
#include <freetype/freetype.h>
#include "zcore_type.h"
FT_BEGIN_HEADER

/******************************************************************************/
/*                                                                            */
/* MiniType Cache Struct Defines                                              */
/*                                                                            */
/******************************************************************************/

typedef enum MTC_Error_Enum_ {
	MTC_ERROR_OK = 0,                         /* no error                                                                */
	MTC_ERROR_NOT_FOUND,                      /* not found glyph                                                         */
	MTC_ERROR_INVALID_PARAMETER,              /* invalid parameter                                                       */
	MTC_ERROR_NOT_SUPPORT_FORMAT_OF_BITMAP,   /* invalid format of bitmap                                                */
	MTC_ERROR_NOT_MATCH_SIZE,                 /* size of glyph is not match between glyph and cache manager              */
	MTC_ERROR_GLYPH_NO_BITMAP,                /* do not cache glyph for no bitmap                                        */
	MTC_ERROR_NOT_ENOUGH_MEMORY               /* not enough memory                                                       */
} MTC_Error;

typedef struct MTC_Glyph_Rec_ {
	ZSINT16   bearingx;          /* difference from pen position to glyph's left bbox                               */
	ZSINT16   bearingy;          /* used to putting characters on baseline                                          */
	ZSINT16   advance;           /* difference between pen positions                                                */
	ZSINT16   bitmap_gray_bits;  /* bits for gray of bitmap pixel, 5 for 16 bits; 8 for 24, 32 bits                 */
	ZSINT16   bitmap_left;       /* bitmap left                                                                     */
	ZSINT16   bitmap_top;        /* bitmap top                                                                      */
	ZSINT16   bitmap_rows;       /* bitmap rows                                                                     */
	ZSINT16   bitmap_width;      /* bitmap width                                                                    */
	ZSINT16   bitmap_pitch;      /* bitmap pitch                                                                    */
	ZUINT8 *bitmap_buffer;     /* bitmap buffer, length of buffer must be aligned by 4 bytes, 4 is sizeof(int)    */
} MTC_Glyph_Rec, *MTC_Glyph;

/*
 * MTC_Cache_Rec Design
 *
 *    bitmap_gray_bits               5 or 8, 5 for RGB 16bits Gray, 8 for 24/32bits Gray
 *    base_len                       length of base
 *    base                           address of base for cache memory
 *    code_begin_address             'code' is 'codeindex', codeindex consists of a unicode, font width, font height, address of glyph
 *                                   it's a index, point to a MTC_Cache_Rec in cache
 *    code_address                   it in between code_begin_address and free_memory_begin_address
 *                                   the scope from code_begin_address to code_address be sorted
 *                                   the scope from code_address to free_memory_begin_address is not sort.
 *    free_memory_begin_address      begin address of free memory for cache glyph
 *    free_memory_end_address        end address of free memory for cache glyph
 *                                   free_memory_end_address increase from cache end to free_memory_begin_address
 *                                   i.e. cache be save from cache end to free_memory_begin_address
 *
 */
typedef struct MTC_Cache_Rec_ {
	ZSINT16   bitmap_gray_bits;             /* bits for gray of bitmap pixel, 5 for 16 bits; 8 for 24, 32 bits                  */
	ZUINT32   base_len;                     /* size of base, it's fixed after new cache                                         */
	ZUINT32  *second_cache;				/*ַ*/
	ZUINT8 *base;                         /* base address, it's fixed after new cahce                                         */
	ZUINT8 *code_begin_address;           /* begin address of unicode list, it's fixed after new cache                        */
	ZUINT8 *code_address;                 /* it's a flag of codeindex for sorted                                              */
	ZUINT8 *free_memory_begin_address;    /* begin address for free memory                                                    */
	ZUINT8 *free_memory_end_address;      /* end address for free memory                                                      */
} MTC_Cache_Rec, *MTC_Cache;

/******************************************************************************/
/*                                                                            */
/* MiniType Cache API Declare                                                 */
/*                                                                            */
/******************************************************************************/

/*
 * MTC_New_Cache
 *
 * get a cache.
 * the memory of cache structure be alloced in base.
 *
 */
FT_EXPORT(MTC_Error)
MTC_New_Cache(
	ZSINT16   bitmap_gray_bits,        /* [in]  bits for gray of bitmap pixel  */
	ZUINT8 *base,                    /* [in]  cache buffer                   */
	ZUINT32   base_len,                /* [in]  len of buffer                  */
	ZUINT32  *second_cache,
	MTC_Cache     *cache                    /* [out] cache                          */
);

/*
 * MTC_Save_Glyph
 *
 */
FT_EXPORT(MTC_Error)
MTC_Save_Glyph(
	MTC_Cache      cache,                   /* [in]  cache                          */
	ZUINT16 code,                    /* [in]  unicode16 code for char        */
	ZUINT8  width,                   /* [in]  font width                     */
	ZUINT8  height,                  /* [in]  font height                    */
	ZSINT32 offset,
	MTC_Glyph      glyph                    /* [in]  MTC_Glyph                      */
);

/*
 * MTC_Load_Glyph
 *
 */
FT_EXPORT(MTC_Glyph)
MTC_Load_Glyph(
	MTC_Cache       cache,                  /* [in]  cache                          */
	ZUINT16  code,                   /* [in]  unicode16 code for char        */
	ZUINT8   width,                  /* [in]  font width                     */
	ZUINT8   height,                 /* [in]  font height                    */
	ZSINT32 offset
);

/*
 * MTC_Create_Glyph
 *
 */
FT_EXPORT(MTC_Glyph)
MTC_Create_Glyph(
	FT_GlyphSlot slot                       /* [in]  slot                           */
);

/*
 * MTC_Create_Invalid_Glyph
 *
 */
FT_EXPORT(MTC_Glyph)
MTC_Create_Invalid_Glyph(
	FT_Face       face,                     /* [in]  face                           */
	ZUINT8 fontsize                  /* [in]  font size                      */
);

/*
 * MTC_Create_Bold_Glyph
 *
 */
FT_EXPORT(MTC_Glyph)
MTC_Create_Bold_Glyph(
	MTC_Glyph glyph                         /* [in]  source glyph                   */
);

/*
 * MTC_Create_Outline_Glyph
 *
 */
FT_EXPORT(MTC_Glyph)
MTC_Create_Outline_Glyph(
	MTC_Glyph glyph                         /* [in]  source glyph                   */
);

/*
 * MTC_Destroy_Glyph
 *
 */
FT_EXPORT(ZVOID)
MTC_Destroy_Glyph(
	MTC_Cache cache,                        /* [in]  cache                          */
	MTC_Glyph glyph                         /* [in]  glyph                          */
);

FT_EXPORT(ZVOID)
MTC_Transform_Gray_5to8(
	MTC_Glyph glyph_src,                    /* [in]  glyph for input                */
	MTC_Glyph glyph_des                     /* [out] glyph for output               */
);

/*
 * MTC_Unicode_Is_FullWidth
 *
 * the code is full width whether is widen in unicode16.
 *
 * return value
 *  nonzero for ful width, zero for widen.
 *
 */
FT_EXPORT(ZSINT32)
MTC_Unicode_Is_FullWidth(
	ZUINT16 code                     /* [in]  unicode 16                     */
);
FT_EXPORT(ZVOID)
glyph_load(FT_Face  face,             /*[in]  the face           */
		   ZSINT32 size,                  /*[in]  Pixel font size    */
		   MTC_Cache cache          /*[in]  Cache address      */ 
);         

FT_END_HEADER

#endif /* __MTCACHE_H__ */
