/*
 * Copyright (C) 2008 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef CDMASMSADDRESS_H_
#define CDMASMSADDRESS_H_
#include <map>
#include <string>
#include <memory>
#include "SmsAddress.h"

class CdmaSmsAddress: public SmsAddress {
public:
  CdmaSmsAddress();
  virtual ~CdmaSmsAddress();

  /**
   * Digit Mode Indicator is a 1-bit value that indicates whether
   * the address digits are 4-bit DTMF codes or 8-bit codes.  (See
   * 3GPP2 C.S0015-B, v2, 3.4.3.3)
   */
  static constexpr int DIGIT_MODE_4BIT_DTMF = 0x00;
  static constexpr int DIGIT_MODE_8BIT_CHAR = 0x01;

  int digitMode = -1;

  /**
   * Number Mode Indicator is 1-bit value that indicates whether the
   * address type is a data network address or not.  (See 3GPP2
   * C.S0015-B, v2, 3.4.3.3)
   */
  static constexpr int NUMBER_MODE_NOT_DATA_NETWORK = 0x00;
  static constexpr int NUMBER_MODE_DATA_NETWORK = 0x01;

  int numberMode = -1;

  /**
   * Number Types for data networks.
   * (See 3GPP2 C.S005-D, table2.7.1.3.2.4-2 for complete table)
   * (See 3GPP2 C.S0015-B, v2, 3.4.3.3 for data network subset)
   * NOTE: value is stored in the parent class ton field.
   */
  static constexpr int TON_UNKNOWN = 0x00;
  static constexpr int TON_INTERNATIONAL_OR_IP = 0x01;
  static constexpr int TON_NATIONAL_OR_EMAIL = 0x02;
  static constexpr int TON_NETWORK = 0x03;
  static constexpr int TON_SUBSCRIBER = 0x04;
  static constexpr int TON_ALPHANUMERIC = 0x05;
  static constexpr int TON_ABBREVIATED = 0x06;
  static constexpr int TON_RESERVED = 0x07;

  /**
   * Maximum lengths for fields as defined in ril_cdma_sms.h.
   */
  static constexpr int SMS_ADDRESS_MAX = 36;
  static constexpr int SMS_SUBADDRESS_MAX = 36;

  /**
   * This field shall be set to the number of address digits
   * (See 3GPP2 C.S0015-B, v2, 3.4.3.3)
   */
  int numberOfDigits = -1;

  /**
   * Numbering Plan identification is a 0 or 4-bit value that
   * indicates which numbering plan identification is set.  (See
   * 3GPP2, C.S0015-B, v2, 3.4.3.3 and C.S005-D, table2.7.1.3.2.4-3)
   */
  static constexpr int NUMBERING_PLAN_UNKNOWN = 0x0;
  static constexpr int NUMBERING_PLAN_ISDN_TELEPHONY = 0x1;
  //static protected final int NUMBERING_PLAN_DATA              = 0x3;
  //static protected final int NUMBERING_PLAN_TELEX             = 0x4;
  //static protected final int NUMBERING_PLAN_PRIVATE           = 0x9;

  int numberPlan = -1;
  static std::vector<uint8_t> parseToDtmf(std::string address);
  static std::shared_ptr<CdmaSmsAddress> parse(char* adr);
  std::string toString();
private:
  static const char numericCharsDialable[];

  static const char numericCharsSugar[];

  static std::map<char, bool> numericCharDialableMap;
  static std::map<char, bool> init();
  static std::string filterNumericSugar(std::string address);
  static std::string filterWhitespace(std::string address);

};
#endif /* CDMASMSADDRESS_H_ */
