#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <string.h>
#include <unistd.h>
#include <binder/Parcel.h>
#include <log/log.h>
#include <cutils/jstring.h>
#include <pthread.h>
#include <list>
#include <vendor-ril/telephony/ril.h>
#include <vendor-ril/telephony/mtk_ril_sp.h>
#include "lynq_call.h"
#include "lynq_module_common.h"
#include "lynq_module_socket.h"
#include "liblog/lynq_deflog.h"
#include "lynq_shm.h"

#define LYNQ_SERVICE_PORT 8088
#define LYNQ_ADDRESS "127.0.0.1"
#define LYNQ_URC_SERVICE_PORT 8086
#define LYNQ_URC_ADDRESS "127.255.255.255" /*hong.liu change broadcast addr on 2024.2.18*/
#define LYNQ_REQUEST_PARAM_BUF 8192
#define LYNQ_REC_BUF 8192

#define USER_LOG_TAG "LYNQ_NETWORK"

typedef struct{
    int uToken;
    int request;
    int paramLen;
    char param[LYNQ_REQUEST_PARAM_BUF];
}lynq_client_t;

typedef struct{
    int resp_type;
    int token;
    int request;
    int slot_id;
    int error;
}lynq_resp_t;

typedef struct{
    int resp_type;
    int urcid;   
}lynq_head_t;

lynq_client_t client_t;
lynq_resp_t response;

int module_len_rc_addr_serv;
int module_len_urc_addr_serv;
struct sockaddr_in module_rc_addr_serv;
struct sockaddr_in module_urc_addr_serv;
static int module_rc_sock_fd = -1;
static int module_urc_sock_fd = -1;
int module_urc_status = 1;
int module_rc_status = 1;
pthread_t module_urc_tid = -1;
pthread_t module_rc_tid = -1;
static pthread_mutex_t s_urc_mutex = PTHREAD_MUTEX_INITIALIZER;

/*hq add for set waiting time 2022/09/13 begin*/
static pthread_mutex_t s_sendto_mutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t s_RecvMsgBlockMutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t s_WaitRecvMsgCond = PTHREAD_COND_INITIALIZER;
#define BLOCK_RECV_MSG_LOCK() pthread_mutex_lock(&s_RecvMsgBlockMutex)
#define BLOCK_RECV_MSG_UNLOCK() pthread_mutex_unlock(&s_RecvMsgBlockMutex)
#define BLOCK_WAIT_RECV_MSG_TIME_OUT(a) pthread_cond_timedwait(&s_WaitRecvMsgCond, &s_RecvMsgBlockMutex,(a))
#define BLOCK_WAKEUP_RECV_MSG() pthread_cond_broadcast(&s_WaitRecvMsgCond)
static std::list<Parcel*> s_recv_parcel_list;
const int waitResponse(int token,int time_out);
/*hq add for set waiting time 2022/09/13 end*/

/*hq add for urc process asynchronous 2022/12/26 begin*/
static pthread_mutex_t s_ProcessUrcMsgBlockMutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t s_WaitProcessUrcMsgCond = PTHREAD_COND_INITIALIZER;

#define BLOCK_PROCESS_URC_MSG_INIT() pthread_mutex_init(&s_ProcessUrcMsgBlockMutex,NULL)
#define BLOCK_PROCESS_URC_MSG_LOCK() pthread_mutex_lock(&s_ProcessUrcMsgBlockMutex)
#define BLOCK_PROCESS_URC_MSG_UNLOCK() pthread_mutex_unlock(&s_ProcessUrcMsgBlockMutex)
#define BLOCK_WAIT_PROCESS_URC_MSG() pthread_cond_wait(&s_WaitProcessUrcMsgCond, &s_ProcessUrcMsgBlockMutex)
#define BLOCK_WAKEUP_PROCESS_URC_MSG() pthread_cond_broadcast(&s_WaitProcessUrcMsgCond)
static std::list<Parcel*> s_recv_urc_parcel_list;
void *thread_urc_process(void *p);
pthread_t module_urc_process_tid = -1;
int module_urc_process_status = 1;
/*hq add for urc process asynchronous 2022/12/26 end*/

int g_module_Global_uToken = 0;
int g_wait_time=5;

/*inner test*/
static int s_inner_test=0;

/**@brief just for inner test
* @param test_mode [IN]: test mode
*  0:success
*  other:failure
*/
void lynq_set_test_mode(const int test_mode)
{
    if(test_mode<0)
    {
        g_wait_time = -test_mode;          
    }
    else if(test_mode==9999)
    {
        LYERRLOG("%s inner test para %d %d",__func__,s_inner_test,g_wait_time);         
    }    
    else 
    {
        s_inner_test = test_mode;    
    }
}

void cleanup_RecvMsgBlock_mutex(void *arg)
{
    BLOCK_RECV_MSG_UNLOCK(); 
}

/**@brief wait response with expected token and write msg to parcel in some time
* @param p [IN]: pointer the parcel
* @param token [IN]: the expected token for the response msg
* @param time_out [IN]: timeout.
* @return
*  0:success
*  other:failure
*/
const int waitResponse(Parcel*& p, int token,int time_out)
{
    int waitToken = token;
    int wakeup_token=-1;
    int resp_type;
    struct timeval now;
    struct timespec timeout;  
    int ret;
    std::list<Parcel*>::iterator iter;
    int cnt=0;
    
    gettimeofday(&now,NULL);
    timeout.tv_sec = now.tv_sec+time_out; //timeout is 1min
    timeout.tv_nsec = now.tv_usec*1000;

    LYINFLOG("%s wait token is %d, wait time is %d",__func__,waitToken,time_out);
    
    BLOCK_RECV_MSG_LOCK();
    pthread_cleanup_push(cleanup_RecvMsgBlock_mutex, NULL); // thread cleanup handler
    while(module_rc_status) { 
        cnt++;
        for(iter=s_recv_parcel_list.begin();iter!=s_recv_parcel_list.end();++iter)
        {
            (*iter)->setDataPosition(0);
            (*iter)->readInt32(&resp_type);
            (*iter)->readInt32(&wakeup_token);
            if(wakeup_token==waitToken)
            {
                LYINFLOG("%s get waitToken",__func__);
                p=(*iter);
                p->setDataPosition(0);
                s_recv_parcel_list.erase(iter);
                goto waitResponse_end;
            }
        }
        LYINFLOG("%s no wait Token in msg list, list size is %d",__func__,s_recv_parcel_list.size());
        ret=BLOCK_WAIT_RECV_MSG_TIME_OUT(&timeout);
        if(ret!=0)
        {
            LYERRLOG("%s no expected token %d after %d second",__func__,waitToken,time_out);
            break;
        }        
    }
waitResponse_end:
    LYINFLOG("%s wait token is %d wakeup_token is %d, cnt is %d",__func__,waitToken,wakeup_token,cnt);
    pthread_cleanup_pop(0);
    BLOCK_RECV_MSG_UNLOCK(); 
    return wakeup_token==waitToken ? 0:LYNQ_E_TIME_OUT;
}

/**@brief print solicied response msg's head information
* @param head [IN]: head information
* @return none
*/
void PrintHeader(lynq_resp_t& head)
{
    LYINFLOG("resp_type=%d,token=%d,request=%d,slot_id=%d,error_code=%d",head.resp_type,head.token,head.request,head.slot_id,head.error);
}

/**@brief get solicied response msg's head
* @param head [OUT]: head information
* @return
*  0:success
*  other:failure
*/
int GetHeader(Parcel* &p, lynq_resp_t& head)
{
    LYINFLOG("get header");
    if(p->dataAvail() > 0)
    {
        p->readInt32(&(head.resp_type));
        p->readInt32(&(head.token));
        p->readInt32(&(head.request));
        p->readInt32(&(head.slot_id));
        p->readInt32(&(head.error));
        PrintHeader(head);
        return RESULT_OK;  
    }
    else
    {
        return RESULT_ERROR;  
    }
}

/**@brief send msg to service and get response from service
* @param request_id [IN]: request id
* @param time_out [IN]: wait time uplimit
* @param p [IN]: point to response msg's parcel
* @param argc [IN]: how many parameters in parameter string
* @param format [IN]: parameter string's format
* @param ... [IN]: the specific parameter
* @return
*  0:success
*  other:failure
*/
int lynq_send_common_request(Parcel*& p, int time_out, int request_id, int argc, const char* format,...)
{
    lynq_client_t client;
    int ret;
    int send_num;
    
    client.uToken = g_module_Global_uToken;
    g_module_Global_uToken=(g_module_Global_uToken+1)%10000;/*0-10000*/
    client.request = request_id;
    client.paramLen = argc;
    bzero(client.param,LYNQ_REQUEST_PARAM_BUF);
    if(argc!=0)
    {
        va_list args;
        va_start(args, format);
        vsnprintf(client.param, LYNQ_REQUEST_PARAM_BUF, format, args);
        va_end(args);    
    }
    LYINFLOG("uToken=%d,request=%d,paralen=%d,param=%s",client.uToken,client.request,client.paramLen,client.param);
    pthread_mutex_lock(&s_sendto_mutex);
     if(s_inner_test==1)
    {
        send_num = 1;          
    }
    else
    {
        send_num = sendto(module_rc_sock_fd,&client,sizeof(client),0,(struct sockaddr *)&module_rc_addr_serv,module_len_rc_addr_serv); 
    }
    if(s_inner_test>=1000)
    {
        time_out = s_inner_test-1000;
    }        
    pthread_mutex_unlock(&s_sendto_mutex);

    if(send_num <= 0)       
    {
        LYERRLOG("send request fail, send num is %d", send_num);
        return LYNQ_E_SEND_REQUEST_FAIL;
    }
    ret = waitResponse(p, client.uToken,time_out);

    if(ret==0)
    {
        lynq_resp_t head; 
        ret=GetHeader(p,head);
        if(ret!=0)
        {
            LYERRLOG("%s %d get head error %d",__func__,client.uToken,ret);
            delete p;
            return LYNQ_E_GET_HEAD_ERROR;
        }
        if(head.error!=0)
        {
            LYERRLOG("%s %d mdm return head error %d",__func__,client.uToken,head.error);
            delete p;
            return head.error;
        }
        LYERRLOG("%s %d suc",__func__,client.uToken);
        return RESULT_OK;  
    }

    LYERRLOG("%s %d fail, ret is %d",__func__,client.uToken,ret);
   
    return ret;    
}

/*hq add for urc broadcase optimisson 2023/01/03 begin*/
#define SHM_BUFFER_INDEX_OFFSET 1
#define SHM_BUFFER_SIZE_OFFSET 16
#define SHM_BUFFER_INDEX_MASK 0x0000007F
#define SHM_BUFFER_SIZE_MASK 0x0000FFFF

bool urc_data_is_in_shm_data(int responseType,int& level, int& index, int& size)
{
    int shm_index=((responseType>>SHM_BUFFER_INDEX_OFFSET)&SHM_BUFFER_INDEX_MASK);
    if (shm_index>0)
    {
        index=shm_index-1;    
        size=((responseType>>SHM_BUFFER_SIZE_OFFSET)&SHM_BUFFER_SIZE_MASK);
        if(size>=sizeof(int32_t)*3 && get_shem_buffer_level(size,&level))
        {
            LYINFLOG("urc_data_is_in_shm_data level is %d, index is %d size is %d",level,index,size);
            return true;                            
        }        
    }
    LYINFLOG("urc_data_is_in_shm_data return false, responseType is %d",responseType);
    return false;
}
/*hq add for urc broadcase optimisson 2023/01/03 end*/

void *thread_urc_recv(void *p)
{
    Parcel *urc_p =NULL;
    char urc_data[LYNQ_REC_BUF];    
    int res = 0;
    lynq_head_t* phead;
    int level,index,size;
	uint8_t * shm_buffer;
   
    LYINFLOG("urc recv thread is running");
    while(module_urc_status)
    {
        bzero(urc_data,LYNQ_REC_BUF);
        res = recvfrom(module_urc_sock_fd,urc_data,sizeof(urc_data),0,(struct sockaddr *)&module_urc_addr_serv,(socklen_t*)&module_len_urc_addr_serv);
        if(res<sizeof(int32_t)*2)
        {
            LYERRLOG("thread_urc_recv step2 fail: res is %d",res);
            continue;
        }

        phead=(lynq_head_t*) urc_data;
        if(is_support_urc(phead->urcid)==false)
        {
            continue;
        }
        urc_p = new Parcel();
        if(urc_p == NULL)
        {
            LYERRLOG("new parcel failure!!!");
            continue;
        }      
        if(urc_data_is_in_shm_data(phead->resp_type,level,index,size))
        {
            shm_buffer = (uint8_t *) get_shem_buffer(level,index);
            LYINFLOG("shm pointer is %p", shm_buffer); 
            urc_p->setData(shm_buffer,size); // p.setData((uint8_t *) buffer, buflen);                        
        }
        else if(res>=sizeof(int32_t)*3)
        {
            urc_p->setData((uint8_t *)urc_data,res); // p.setData((uint8_t *) buffer, buflen);            
        }
        else 
        {    
               LYERRLOG("res %d error!!!", res);
            delete urc_p;
            urc_p = NULL;
            continue;
        }
        urc_p->setDataPosition(0);
        if(urc_p->dataAvail()>0)
        {
            BLOCK_PROCESS_URC_MSG_LOCK();
            s_recv_urc_parcel_list.push_back(urc_p);            
            BLOCK_WAKEUP_PROCESS_URC_MSG();
            BLOCK_PROCESS_URC_MSG_UNLOCK();
        }
        else 
        {
            delete urc_p;
            urc_p = NULL;
        }        
    }
    LYINFLOG("urc recv thread ended");
    return NULL;
}

void cleanup_urc_process_mutex(void *arg)
{
    BLOCK_PROCESS_URC_MSG_UNLOCK();
}

void *thread_urc_process(void *p)
{
    Parcel *urc_p =NULL;
    std::list<Parcel*>::iterator iter;
   
    LYINFLOG("urc process thread is running");
    pthread_cleanup_push(cleanup_urc_process_mutex, NULL); // thread cleanup handler
    while(module_urc_process_status)
    {
        BLOCK_PROCESS_URC_MSG_LOCK();
        while(s_recv_urc_parcel_list.empty())
        {
            BLOCK_WAIT_PROCESS_URC_MSG();
        }  
        iter=s_recv_urc_parcel_list.begin();
        urc_p=(*iter);
        s_recv_urc_parcel_list.erase(iter);
        BLOCK_PROCESS_URC_MSG_UNLOCK();             
        urc_p->setDataPosition(0);
        if(urc_p->dataAvail()>0)
        {          
            pthread_mutex_lock(&s_urc_mutex);
            urc_msg_process(urc_p);            
            pthread_mutex_unlock(&s_urc_mutex);
        }
        delete urc_p;
        urc_p = NULL;
    }
    pthread_cleanup_pop(0);
    LYINFLOG("urc process thread ended");
    return NULL;
}

void lynq_close_urc_rev_thread()
{
    int ret;

    BLOCK_PROCESS_URC_MSG_LOCK();  //just cancel urc process tid when recv from
    module_urc_status = 0;    
    if(module_urc_tid!=-1)
    {
        ret = pthread_cancel(module_urc_tid);
        LYINFLOG("pthread cancel urc rev ret = %d",ret);            
    }
    BLOCK_PROCESS_URC_MSG_UNLOCK();
    if(module_urc_tid != -1)
    {
        ret = pthread_join(module_urc_tid,NULL);
        LYINFLOG("pthread join urc tid ret = %d",ret);
        module_urc_tid =-1;
    }       
}

void lynq_close_urc_process_thread()
{
    int ret;
    BLOCK_PROCESS_URC_MSG_LOCK();  //just cancel urc process tid when not process urc msg in list 
    pthread_mutex_lock(&s_urc_mutex); //just cancel urc process tid when not process urg msg avoid mutual lock for tid may call pthread_cond_wait
    module_urc_process_status = 0;    
    if(module_urc_process_tid!=-1)
    {
        ret = pthread_cancel(module_urc_process_tid);
        LYINFLOG("pthread cancel urc process ret = %d",ret);            
    }
    pthread_mutex_unlock(&s_urc_mutex);
    BLOCK_PROCESS_URC_MSG_UNLOCK();
    if(module_urc_process_tid != -1)
    {
        ret = pthread_join(module_urc_process_tid,NULL);
        LYINFLOG("pthread join urc process tid ret = %d",ret);
        module_urc_process_tid =-1;
    }
}

int lynq_setup_urc_socket()
{
    int on = 1;
    int ret = 0;
    module_len_urc_addr_serv = sizeof(sockaddr_in);
    module_urc_sock_fd = socket(AF_INET, SOCK_DGRAM, 0);
    if (module_urc_sock_fd <0){
        LYERRLOG("urc socket error");
        return RESULT_ERROR;  
    }
    module_urc_addr_serv.sin_family = AF_INET;
    module_urc_addr_serv.sin_port = htons(LYNQ_URC_SERVICE_PORT);
    module_urc_addr_serv.sin_addr.s_addr = inet_addr(LYNQ_URC_ADDRESS);
    /* Set socket to allow reuse of address and port, SO_REUSEADDR value is 2*/
    ret = setsockopt(module_urc_sock_fd,SOL_SOCKET,SO_REUSEADDR,&on,sizeof on);
    if(ret <0)
    {
        LYERRLOG("urc socket set error");
        close(module_urc_sock_fd);
        module_urc_sock_fd =-1;
        return RESULT_ERROR;  
    }
    ret = bind(module_urc_sock_fd ,(struct sockaddr*)&module_urc_addr_serv, sizeof(module_urc_addr_serv));
    if(ret <0)
    {
        LYERRLOG("urc socket bind error");
        close(module_urc_sock_fd);
        module_urc_sock_fd =-1;
        return RESULT_ERROR;  
    }
    return RESULT_OK;
}

void lynq_close_urc_socket()
{
    if (module_urc_sock_fd >= 0)
    {
        close(module_urc_sock_fd);
        module_urc_sock_fd =-1;
    }  
}

int lynq_start_all_urc_socket_thread()
{  
    
    if(ril_init_mem()!=0)
    {
        LYERRLOG("ril_init_mem fail");        
        return RESULT_ERROR;
    }

    int ret= lynq_setup_urc_socket();
    if(ret!=RESULT_OK)
    {
        LYERRLOG("call lynq_setup_urc_socket fail");
        ril_deinit_mem();
        return RESULT_ERROR;
    }
    
    BLOCK_PROCESS_URC_MSG_INIT();
    BLOCK_PROCESS_URC_MSG_LOCK();    
    std::list<Parcel*>::iterator iter;
    for(iter=s_recv_urc_parcel_list.begin();iter!=s_recv_urc_parcel_list.end();++iter)
    {
        delete (*iter);        
    }
    s_recv_urc_parcel_list.clear(); 
    BLOCK_PROCESS_URC_MSG_UNLOCK(); 
    
    pthread_mutex_init(&s_urc_mutex, NULL);    
    
    module_urc_status = 1;
  //  pthread_attr_init(&attr);
  //  pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
    ret = pthread_create(&module_urc_tid,/*&attr*/NULL,thread_urc_recv,NULL);
    if(ret <0)
    {
        LYERRLOG("urc recv pthread create error");
        module_urc_status = 0;
        lynq_close_urc_socket();
        ril_deinit_mem();
        return RESULT_ERROR;  
    }

    module_urc_process_status = 1;
    ret = pthread_create(&module_urc_process_tid,/*&attr*/NULL,thread_urc_process,NULL);
    if(ret <0)
    {
        LYERRLOG("urc process pthread create error");
        module_urc_process_status = 0;
        lynq_close_urc_socket();
        lynq_close_urc_rev_thread();
        ril_deinit_mem();
        return RESULT_ERROR;  
    }
    
    LYINFLOG("urc start success");
    
    return RESULT_OK;  
}

void lynq_close_all_urc_socket_thread()
{
 
    lynq_close_urc_rev_thread();
    lynq_close_urc_socket();
    lynq_close_urc_process_thread();        
    
    BLOCK_PROCESS_URC_MSG_LOCK();    
    std::list<Parcel*>::iterator iter;
    for(iter=s_recv_urc_parcel_list.begin();iter!=s_recv_urc_parcel_list.end();++iter)
    {
        delete (*iter);        
    }
    s_recv_urc_parcel_list.clear(); 
    BLOCK_PROCESS_URC_MSG_UNLOCK(); 
    
    ril_deinit_mem();    
}

/**@ a thread just for recv\buffer solicited msg's response and notice waiting thread
* @param p [IN]: no meaning
* @return
* always null
*/
void *thread_rc_recv(void *p)
{
    Parcel* rc_p;
    std::list<Parcel*>::iterator iter;
    int resp_type = -1;   
    char rc_data[LYNQ_REC_BUF];
    int rc_len;
    int null_cnt=0;
    int wakeup_token; 
 
    LYINFLOG("rc thread is running");
    while(module_rc_status)
    {
        bzero(rc_data,LYNQ_REC_BUF);
        while(true)
        {
            rc_len = recvfrom(module_rc_sock_fd,rc_data,sizeof(rc_data),0,(struct sockaddr *)&module_rc_addr_serv,(socklen_t *)&module_len_rc_addr_serv);
            if(rc_len<sizeof(int32_t)*2)
            {
                 LYERRLOG("%s recv len %d less %d",__func__, rc_len,sizeof(int)*2);
                 continue;
            } 
            rc_p= new Parcel;
            if(rc_p==NULL)
            {
                null_cnt++;
                LYERRLOG("%s rc_p is NULL, cnt is %d",__func__, null_cnt);
                if(null_cnt>20)
                {
                    goto rc_recv_end;
                }
                continue;
            }
            else 
            {
                null_cnt=0;
            }
                
            rc_p->setData((uint8_t *)rc_data,rc_len); // p.setData((uint8_t *) buffer, buflen);
            rc_p->setDataPosition(0);
            if(rc_p->dataAvail()>0)
            {
                rc_p->readInt32(&resp_type); 
                rc_p->readInt32(&wakeup_token); 
                BLOCK_RECV_MSG_LOCK();
                s_recv_parcel_list.push_back(rc_p);                
                LYINFLOG("%s wakeup token is %d, list size is %d!",__func__,wakeup_token,s_recv_parcel_list.size());
                if(s_recv_parcel_list.size()>20) //max 20
                {
                    iter=s_recv_parcel_list.begin();
                    (*iter)->setDataPosition(0);
                    (*iter)->readInt32(&resp_type); 
                    (*iter)->readInt32(&wakeup_token); 
                    delete (*(s_recv_parcel_list.begin()));
                    LYERRLOG("%s wakeup token %d is deleted!",__func__,wakeup_token);
                    s_recv_parcel_list.erase(iter);
                }                   
                BLOCK_WAKEUP_RECV_MSG();
                BLOCK_RECV_MSG_UNLOCK();
                break;
            }
            else
            {
                LYERRLOG("%s rc_p data Avail %d not greater than 0",__func__, rc_p->dataAvail());
                delete rc_p;
            }
        }
    }                 

rc_recv_end:    
    LYINFLOG("rc thread ended");
    return NULL;
}

int lynq_start_all_rc_socket_thread()
{
    module_rc_sock_fd = socket(AF_INET, SOCK_DGRAM, 0);
    if(module_rc_sock_fd < 0)
    {
        LYERRLOG("socket open error");
        return -1;
    }

    struct sockaddr_in liblynq_data_socket;
    bzero(&liblynq_data_socket, sizeof(liblynq_data_socket));
    //set this lib socket config 
    liblynq_data_socket.sin_family = AF_INET;
    liblynq_data_socket.sin_addr.s_addr = inet_addr(LYNQ_ADDRESS);
    int ret = bind(module_rc_sock_fd, (struct sockaddr *)&liblynq_data_socket, sizeof(liblynq_data_socket));
    if (-1 == ret)
    {
        LYERRLOG("liblynq_data_socket bind fail,errno:%d",errno);
        return -1;
    }

    LYINFLOG("module_rc_sock_fd = %d",module_rc_sock_fd);

    memset(&module_rc_addr_serv, 0, sizeof(module_rc_addr_serv));
    module_rc_addr_serv.sin_family = AF_INET;
    module_rc_addr_serv.sin_addr.s_addr = inet_addr(LYNQ_ADDRESS);
    module_rc_addr_serv.sin_port = htons(LYNQ_SERVICE_PORT);
    module_len_rc_addr_serv = sizeof(module_rc_addr_serv);

    BLOCK_RECV_MSG_LOCK(); 
    std::list<Parcel*>::iterator iter;
    for(iter=s_recv_parcel_list.begin();iter!=s_recv_parcel_list.end();++iter)
    {
        delete (*iter);        
    }
    s_recv_parcel_list.clear();  
    BLOCK_RECV_MSG_UNLOCK(); 

//    pthread_attr_t attr;

    pthread_mutex_init(&s_sendto_mutex, NULL);   
    pthread_mutex_init(&s_RecvMsgBlockMutex, NULL);   

    module_rc_status = 1;
//    pthread_attr_init(&attr);
//    pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
    ret = pthread_create(&module_rc_tid,/*&attr*/NULL,thread_rc_recv,NULL);
    if(ret <0)
    {
        LYERRLOG("rc pthread create error");
        module_rc_status =0;
        close(module_rc_sock_fd);
        module_rc_sock_fd =-1;        
        return ret;
    }   
    
    LYINFLOG("rc start success");

    return RESULT_OK;  
}

void lynq_close_all_rc_socket_thread()
{
    int ret;
    BLOCK_RECV_MSG_LOCK(); 
    module_rc_status = 0;    
    BLOCK_WAKEUP_RECV_MSG(); 
    if(module_rc_tid != -1)
    {
        ret = pthread_cancel(module_rc_tid);
        LYINFLOG("pthread cancel rc ret = %d",ret);     
    }    
    BLOCK_RECV_MSG_UNLOCK();     
    if(module_rc_tid != -1)
    {
        ret = pthread_join(module_rc_tid,NULL);
        module_rc_tid =-1;
        LYINFLOG("pthread join rc tid ret = %d",ret);
        
    }    

    if (module_rc_sock_fd > 0)
    {
        close(module_rc_sock_fd);
        module_rc_sock_fd =-1;
    }
    
    BLOCK_RECV_MSG_LOCK();    
    std::list<Parcel*>::iterator iter;
    for(iter=s_recv_parcel_list.begin();iter!=s_recv_parcel_list.end();++iter)
    {
        delete (*iter);        
    }
    s_recv_parcel_list.clear(); 
    BLOCK_RECV_MSG_UNLOCK(); 
}

const char * requestToString(int request) 
{
    /*
     cat libs/telephony/ril_commands.h \
     | egrep "^ *{RIL_" \
     | sed -re 's/\{RIL_([^,]+),[^,]+,([^}]+).+/case RIL_\1: return "\1";/'
    
    
     cat libs/telephony/ril_unsol_commands.h \
     | egrep "^ *{RIL_" \
     | sed -re 's/\{RIL_([^,]+),([^}]+).+/case RIL_\1: return "\1";/'
    
    */
    switch(request) {
        case RIL_REQUEST_GET_SIM_STATUS: return "GET_SIM_STATUS";
        case RIL_REQUEST_ENTER_SIM_PIN: return "ENTER_SIM_PIN";
        case RIL_REQUEST_ENTER_SIM_PUK: return "ENTER_SIM_PUK";
        case RIL_REQUEST_ENTER_SIM_PIN2: return "ENTER_SIM_PIN2";
        case RIL_REQUEST_ENTER_SIM_PUK2: return "ENTER_SIM_PUK2";
        case RIL_REQUEST_CHANGE_SIM_PIN: return "CHANGE_SIM_PIN";
        case RIL_REQUEST_CHANGE_SIM_PIN2: return "CHANGE_SIM_PIN2";
        case RIL_REQUEST_ENTER_NETWORK_DEPERSONALIZATION: return "ENTER_NETWORK_DEPERSONALIZATION";
        case RIL_REQUEST_GET_CURRENT_CALLS: return "GET_CURRENT_CALLS";
        case RIL_REQUEST_DIAL: return "DIAL";
        case RIL_REQUEST_GET_IMSI: return "GET_IMSI";
        case RIL_REQUEST_HANGUP: return "HANGUP";
        case RIL_REQUEST_HANGUP_WAITING_OR_BACKGROUND: return "HANGUP_WAITING_OR_BACKGROUND";
        case RIL_REQUEST_HANGUP_FOREGROUND_RESUME_BACKGROUND: return "HANGUP_FOREGROUND_RESUME_BACKGROUND";
        case RIL_REQUEST_SWITCH_WAITING_OR_HOLDING_AND_ACTIVE: return "SWITCH_WAITING_OR_HOLDING_AND_ACTIVE";
        case RIL_REQUEST_CONFERENCE: return "CONFERENCE";
        case RIL_REQUEST_UDUB: return "UDUB";
        case RIL_REQUEST_LAST_CALL_FAIL_CAUSE: return "LAST_CALL_FAIL_CAUSE";
        case RIL_REQUEST_SIGNAL_STRENGTH: return "SIGNAL_STRENGTH";
        case RIL_REQUEST_VOICE_REGISTRATION_STATE: return "VOICE_REGISTRATION_STATE";
        case RIL_REQUEST_DATA_REGISTRATION_STATE: return "DATA_REGISTRATION_STATE";
        case RIL_REQUEST_OPERATOR: return "OPERATOR";
        case RIL_REQUEST_RADIO_POWER: return "RADIO_POWER";
        case RIL_REQUEST_DTMF: return "DTMF";
        case RIL_REQUEST_SEND_SMS: return "SEND_SMS";
        case RIL_REQUEST_SEND_SMS_EXPECT_MORE: return "SEND_SMS_EXPECT_MORE";
        case RIL_REQUEST_SETUP_DATA_CALL: return "SETUP_DATA_CALL";
        case RIL_REQUEST_SIM_IO: return "SIM_IO";
        case RIL_REQUEST_SEND_USSD: return "SEND_USSD";
        case RIL_REQUEST_CANCEL_USSD: return "CANCEL_USSD";
        case RIL_REQUEST_GET_CLIR: return "GET_CLIR";
        case RIL_REQUEST_SET_CLIR: return "SET_CLIR";
        case RIL_REQUEST_QUERY_CALL_FORWARD_STATUS: return "QUERY_CALL_FORWARD_STATUS";
        case RIL_REQUEST_SET_CALL_FORWARD: return "SET_CALL_FORWARD";
        case RIL_REQUEST_QUERY_CALL_WAITING: return "QUERY_CALL_WAITING";
        case RIL_REQUEST_SET_CALL_WAITING: return "SET_CALL_WAITING";
        case RIL_REQUEST_SMS_ACKNOWLEDGE: return "SMS_ACKNOWLEDGE";
        case RIL_REQUEST_GET_IMEI: return "GET_IMEI";
        case RIL_REQUEST_GET_IMEISV: return "GET_IMEISV";
        case RIL_REQUEST_ANSWER: return "ANSWER";
        case RIL_REQUEST_DEACTIVATE_DATA_CALL: return "DEACTIVATE_DATA_CALL";
        case RIL_REQUEST_QUERY_FACILITY_LOCK: return "QUERY_FACILITY_LOCK";
        case RIL_REQUEST_SET_FACILITY_LOCK: return "SET_FACILITY_LOCK";
        case RIL_REQUEST_CHANGE_BARRING_PASSWORD: return "CHANGE_BARRING_PASSWORD";
        case RIL_REQUEST_QUERY_NETWORK_SELECTION_MODE: return "QUERY_NETWORK_SELECTION_MODE";
        case RIL_REQUEST_SET_NETWORK_SELECTION_AUTOMATIC: return "SET_NETWORK_SELECTION_AUTOMATIC";
        case RIL_REQUEST_SET_NETWORK_SELECTION_MANUAL: return "SET_NETWORK_SELECTION_MANUAL";
        case RIL_REQUEST_QUERY_AVAILABLE_NETWORKS : return "QUERY_AVAILABLE_NETWORKS ";
        case RIL_REQUEST_DTMF_START: return "DTMF_START";
        case RIL_REQUEST_DTMF_STOP: return "DTMF_STOP";
        case RIL_REQUEST_BASEBAND_VERSION: return "BASEBAND_VERSION";
        case RIL_REQUEST_SEPARATE_CONNECTION: return "SEPARATE_CONNECTION";
        case RIL_REQUEST_SET_PREFERRED_NETWORK_TYPE: return "SET_PREFERRED_NETWORK_TYPE";
        case RIL_REQUEST_GET_PREFERRED_NETWORK_TYPE: return "GET_PREFERRED_NETWORK_TYPE";
        case RIL_REQUEST_GET_NEIGHBORING_CELL_IDS: return "GET_NEIGHBORING_CELL_IDS";
        case RIL_REQUEST_SET_MUTE: return "SET_MUTE";
        case RIL_REQUEST_GET_MUTE: return "GET_MUTE";
        case RIL_REQUEST_QUERY_CLIP: return "QUERY_CLIP";
        case RIL_REQUEST_LAST_DATA_CALL_FAIL_CAUSE: return "LAST_DATA_CALL_FAIL_CAUSE";
        case RIL_REQUEST_DATA_CALL_LIST: return "DATA_CALL_LIST";
        case RIL_REQUEST_RESET_RADIO: return "RESET_RADIO";
        case RIL_REQUEST_OEM_HOOK_RAW: return "OEM_HOOK_RAW";
        case RIL_REQUEST_OEM_HOOK_STRINGS: return "OEM_HOOK_STRINGS";
        case RIL_REQUEST_SET_BAND_MODE: return "SET_BAND_MODE";
        case RIL_REQUEST_QUERY_AVAILABLE_BAND_MODE: return "QUERY_AVAILABLE_BAND_MODE";
        case RIL_REQUEST_STK_GET_PROFILE: return "STK_GET_PROFILE";
        case RIL_REQUEST_STK_SET_PROFILE: return "STK_SET_PROFILE";
        case RIL_REQUEST_STK_SEND_ENVELOPE_COMMAND: return "STK_SEND_ENVELOPE_COMMAND";
        case RIL_REQUEST_STK_SEND_TERMINAL_RESPONSE: return "STK_SEND_TERMINAL_RESPONSE";
        case RIL_REQUEST_STK_HANDLE_CALL_SETUP_REQUESTED_FROM_SIM: return "STK_HANDLE_CALL_SETUP_REQUESTED_FROM_SIM";
        case RIL_REQUEST_SCREEN_STATE: return "SCREEN_STATE";
        case RIL_REQUEST_EXPLICIT_CALL_TRANSFER: return "EXPLICIT_CALL_TRANSFER";
        case RIL_REQUEST_SET_LOCATION_UPDATES: return "SET_LOCATION_UPDATES";
        case RIL_REQUEST_CDMA_SET_SUBSCRIPTION_SOURCE:return"CDMA_SET_SUBSCRIPTION_SOURCE";
        case RIL_REQUEST_CDMA_SET_ROAMING_PREFERENCE:return"CDMA_SET_ROAMING_PREFERENCE";
        case RIL_REQUEST_CDMA_QUERY_ROAMING_PREFERENCE:return"CDMA_QUERY_ROAMING_PREFERENCE";
        case RIL_REQUEST_SET_TTY_MODE:return"SET_TTY_MODE";
        case RIL_REQUEST_QUERY_TTY_MODE:return"QUERY_TTY_MODE";
        case RIL_REQUEST_CDMA_SET_PREFERRED_VOICE_PRIVACY_MODE:return"CDMA_SET_PREFERRED_VOICE_PRIVACY_MODE";
        case RIL_REQUEST_CDMA_QUERY_PREFERRED_VOICE_PRIVACY_MODE:return"CDMA_QUERY_PREFERRED_VOICE_PRIVACY_MODE";
        case RIL_REQUEST_CDMA_FLASH:return"CDMA_FLASH";
        case RIL_REQUEST_CDMA_BURST_DTMF:return"CDMA_BURST_DTMF";
        case RIL_REQUEST_CDMA_SEND_SMS:return"CDMA_SEND_SMS";
        case RIL_REQUEST_CDMA_SMS_ACKNOWLEDGE:return"CDMA_SMS_ACKNOWLEDGE";
        case RIL_REQUEST_GSM_GET_BROADCAST_SMS_CONFIG:return"GSM_GET_BROADCAST_SMS_CONFIG";
        case RIL_REQUEST_GSM_SET_BROADCAST_SMS_CONFIG:return"GSM_SET_BROADCAST_SMS_CONFIG";
        case RIL_REQUEST_GSM_SMS_BROADCAST_ACTIVATION:return "GSM_SMS_BROADCAST_ACTIVATION";
        case RIL_REQUEST_CDMA_GET_BROADCAST_SMS_CONFIG:return "CDMA_GET_BROADCAST_SMS_CONFIG";
        case RIL_REQUEST_CDMA_SET_BROADCAST_SMS_CONFIG:return "CDMA_SET_BROADCAST_SMS_CONFIG";
        case RIL_REQUEST_CDMA_SMS_BROADCAST_ACTIVATION:return "CDMA_SMS_BROADCAST_ACTIVATION";
        case RIL_REQUEST_CDMA_VALIDATE_AND_WRITE_AKEY: return"CDMA_VALIDATE_AND_WRITE_AKEY";
        case RIL_REQUEST_CDMA_SUBSCRIPTION: return"CDMA_SUBSCRIPTION";
        case RIL_REQUEST_CDMA_WRITE_SMS_TO_RUIM: return "CDMA_WRITE_SMS_TO_RUIM";
        case RIL_REQUEST_CDMA_DELETE_SMS_ON_RUIM: return "CDMA_DELETE_SMS_ON_RUIM";
        case RIL_REQUEST_DEVICE_IDENTITY: return "DEVICE_IDENTITY";
        case RIL_REQUEST_EXIT_EMERGENCY_CALLBACK_MODE: return "EXIT_EMERGENCY_CALLBACK_MODE";
        case RIL_REQUEST_GET_SMSC_ADDRESS: return "GET_SMSC_ADDRESS";
        case RIL_REQUEST_SET_SMSC_ADDRESS: return "SET_SMSC_ADDRESS";
        case RIL_REQUEST_REPORT_SMS_MEMORY_STATUS: return "REPORT_SMS_MEMORY_STATUS";
        case RIL_REQUEST_REPORT_STK_SERVICE_IS_RUNNING: return "REPORT_STK_SERVICE_IS_RUNNING";
        case RIL_REQUEST_CDMA_GET_SUBSCRIPTION_SOURCE: return "CDMA_GET_SUBSCRIPTION_SOURCE";
        case RIL_REQUEST_ISIM_AUTHENTICATION: return "ISIM_AUTHENTICATION";
        case RIL_REQUEST_ACKNOWLEDGE_INCOMING_GSM_SMS_WITH_PDU: return "RIL_REQUEST_ACKNOWLEDGE_INCOMING_GSM_SMS_WITH_PDU";
        case RIL_REQUEST_STK_SEND_ENVELOPE_WITH_STATUS: return "RIL_REQUEST_STK_SEND_ENVELOPE_WITH_STATUS";
        case RIL_REQUEST_VOICE_RADIO_TECH: return "VOICE_RADIO_TECH";
        case RIL_REQUEST_GET_CELL_INFO_LIST: return"GET_CELL_INFO_LIST";
        case RIL_REQUEST_SET_UNSOL_CELL_INFO_LIST_RATE: return"SET_UNSOL_CELL_INFO_LIST_RATE";
        case RIL_REQUEST_SET_INITIAL_ATTACH_APN: return "RIL_REQUEST_SET_INITIAL_ATTACH_APN";
        case RIL_REQUEST_IMS_REGISTRATION_STATE: return "IMS_REGISTRATION_STATE";
        case RIL_REQUEST_IMS_SEND_SMS: return "IMS_SEND_SMS";
        case RIL_REQUEST_SIM_TRANSMIT_APDU_BASIC: return "SIM_TRANSMIT_APDU_BASIC";
        case RIL_REQUEST_SIM_OPEN_CHANNEL: return "SIM_OPEN_CHANNEL";
        case RIL_REQUEST_SIM_CLOSE_CHANNEL: return "SIM_CLOSE_CHANNEL";
        case RIL_REQUEST_SIM_TRANSMIT_APDU_CHANNEL: return "SIM_TRANSMIT_APDU_CHANNEL";
        case RIL_REQUEST_GET_RADIO_CAPABILITY: return "RIL_REQUEST_GET_RADIO_CAPABILITY";
        case RIL_REQUEST_SET_RADIO_CAPABILITY: return "RIL_REQUEST_SET_RADIO_CAPABILITY";
        case RIL_REQUEST_SET_UICC_SUBSCRIPTION: return "SET_UICC_SUBSCRIPTION";
        case RIL_REQUEST_ALLOW_DATA: return "ALLOW_DATA";
        case RIL_REQUEST_GET_HARDWARE_CONFIG: return "GET_HARDWARE_CONFIG";
        case RIL_REQUEST_SIM_AUTHENTICATION: return "SIM_AUTHENTICATION";
        case RIL_REQUEST_GET_DC_RT_INFO: return "GET_DC_RT_INFO";
        case RIL_REQUEST_SET_DC_RT_INFO_RATE: return "SET_DC_RT_INFO_RATE";
        case RIL_REQUEST_SET_DATA_PROFILE: return "SET_DATA_PROFILE";
        case RIL_UNSOL_RESPONSE_RADIO_STATE_CHANGED: return "UNSOL_RESPONSE_RADIO_STATE_CHANGED";
        case RIL_UNSOL_RESPONSE_CALL_STATE_CHANGED: return "UNSOL_RESPONSE_CALL_STATE_CHANGED";
        case RIL_UNSOL_RESPONSE_VOICE_NETWORK_STATE_CHANGED: return "UNSOL_RESPONSE_VOICE_NETWORK_STATE_CHANGED";
        case RIL_UNSOL_RESPONSE_PS_NETWORK_STATE_CHANGED: return "UNSOL_RESPONSE_VOICE_NETWORK_STATE_CHANGED";
        case RIL_UNSOL_RESPONSE_NEW_SMS: return "UNSOL_RESPONSE_NEW_SMS";
        case RIL_UNSOL_RESPONSE_NEW_SMS_STATUS_REPORT: return "UNSOL_RESPONSE_NEW_SMS_STATUS_REPORT";
        case RIL_UNSOL_RESPONSE_NEW_SMS_ON_SIM: return "UNSOL_RESPONSE_NEW_SMS_ON_SIM";
        case RIL_UNSOL_ON_USSD: return "UNSOL_ON_USSD";
        case RIL_UNSOL_ON_USSD_REQUEST: return "UNSOL_ON_USSD_REQUEST(obsolete)";
        case RIL_UNSOL_NITZ_TIME_RECEIVED: return "UNSOL_NITZ_TIME_RECEIVED";
        case RIL_UNSOL_SIGNAL_STRENGTH: return "UNSOL_SIGNAL_STRENGTH";
        case RIL_UNSOL_STK_SESSION_END: return "UNSOL_STK_SESSION_END";
        case RIL_UNSOL_STK_PROACTIVE_COMMAND: return "UNSOL_STK_PROACTIVE_COMMAND";
        case RIL_UNSOL_STK_EVENT_NOTIFY: return "UNSOL_STK_EVENT_NOTIFY";
        case RIL_UNSOL_STK_CALL_SETUP: return "UNSOL_STK_CALL_SETUP";
        case RIL_UNSOL_SIM_SMS_STORAGE_FULL: return "UNSOL_SIM_SMS_STORAGE_FUL";
        case RIL_UNSOL_SIM_REFRESH: return "UNSOL_SIM_REFRESH";
        case RIL_UNSOL_DATA_CALL_LIST_CHANGED: return "UNSOL_DATA_CALL_LIST_CHANGED";
        case RIL_UNSOL_CALL_RING: return "UNSOL_CALL_RING";
        case RIL_UNSOL_RESPONSE_SIM_STATUS_CHANGED: return "UNSOL_RESPONSE_SIM_STATUS_CHANGED";
        case RIL_UNSOL_RESPONSE_CDMA_NEW_SMS: return "UNSOL_NEW_CDMA_SMS";
        case RIL_UNSOL_RESPONSE_NEW_BROADCAST_SMS: return "UNSOL_NEW_BROADCAST_SMS";
        case RIL_UNSOL_CDMA_RUIM_SMS_STORAGE_FULL: return "UNSOL_CDMA_RUIM_SMS_STORAGE_FULL";
        case RIL_UNSOL_RESTRICTED_STATE_CHANGED: return "UNSOL_RESTRICTED_STATE_CHANGED";
        case RIL_UNSOL_ENTER_EMERGENCY_CALLBACK_MODE: return "UNSOL_ENTER_EMERGENCY_CALLBACK_MODE";
        case RIL_UNSOL_CDMA_CALL_WAITING: return "UNSOL_CDMA_CALL_WAITING";
        case RIL_UNSOL_CDMA_OTA_PROVISION_STATUS: return "UNSOL_CDMA_OTA_PROVISION_STATUS";
        case RIL_UNSOL_CDMA_INFO_REC: return "UNSOL_CDMA_INFO_REC";
        case RIL_UNSOL_OEM_HOOK_RAW: return "UNSOL_OEM_HOOK_RAW";
        case RIL_UNSOL_RINGBACK_TONE: return "UNSOL_RINGBACK_TONE";
        case RIL_UNSOL_RESEND_INCALL_MUTE: return "UNSOL_RESEND_INCALL_MUTE";
        case RIL_UNSOL_CDMA_SUBSCRIPTION_SOURCE_CHANGED: return "UNSOL_CDMA_SUBSCRIPTION_SOURCE_CHANGED";
        case RIL_UNSOL_CDMA_PRL_CHANGED: return "UNSOL_CDMA_PRL_CHANGED";
        case RIL_UNSOL_EXIT_EMERGENCY_CALLBACK_MODE: return "UNSOL_EXIT_EMERGENCY_CALLBACK_MODE";
        case RIL_UNSOL_RIL_CONNECTED: return "UNSOL_RIL_CONNECTED";
        case RIL_UNSOL_VOICE_RADIO_TECH_CHANGED: return "UNSOL_VOICE_RADIO_TECH_CHANGED";
        case RIL_UNSOL_CELL_INFO_LIST: return "UNSOL_CELL_INFO_LIST";
        case RIL_UNSOL_RESPONSE_IMS_NETWORK_STATE_CHANGED: return "RESPONSE_IMS_NETWORK_STATE_CHANGED";
        case RIL_UNSOL_UICC_SUBSCRIPTION_STATUS_CHANGED: return "UNSOL_UICC_SUBSCRIPTION_STATUS_CHANGED";
        case RIL_UNSOL_SRVCC_STATE_NOTIFY: return "UNSOL_SRVCC_STATE_NOTIFY";
        case RIL_UNSOL_HARDWARE_CONFIG_CHANGED: return "HARDWARE_CONFIG_CHANGED";
        case RIL_UNSOL_DC_RT_INFO_CHANGED: return "UNSOL_DC_RT_INFO_CHANGED";
        case RIL_REQUEST_SHUTDOWN: return "SHUTDOWN";
        case RIL_UNSOL_RADIO_CAPABILITY: return "RIL_UNSOL_RADIO_CAPABILITY";
        case RIL_REQUEST_SET_TRM: return "RIL_REQUEST_SET_TRM";
        case RIL_REQUEST_SET_IMS_ENABLE:return "RIL_REQUEST_SET_IMS_ENABLE";
        case RIL_REQUEST_SET_AUDIO_PATH: return "SET_AUDIO_PATH";
        case RIL_REQUEST_HANGUP_ALL: return "HANGUP_ALL";
        case RIL_REQUEST_FORCE_RELEASE_CALL: return "FORCE_RELEASE_CALL";
        case RIL_REQUEST_EMERGENCY_DIAL: return "RIL_REQUEST_EMERGENCY_DIAL";
        case RIL_REQUEST_SET_ECC_SERVICE_CATEGORY: return "RIL_REQUEST_SET_ECC_SERVICE_CATEGORY";
        case RIL_REQUEST_SET_ECC_LIST: return "RIL_REQUEST_SET_ECC_LIST";
        case RIL_REQUEST_AT_COMMAND_WITH_PROXY: return "AT_COMMAND_WITH_PROXY";
        case RIL_REQUEST_SET_SUPP_SVC_NOTIFICATION: return "RIL_REQUEST_SET_SUPP_SVC_NOTIFICATION";
        case RIL_REQUEST_SET_CLIP: return "RIL_REQUEST_SET_CLIP";
        case RIL_REQUEST_GET_COLP: return "RIL_REQUEST_GET_COLP";
        case RIL_REQUEST_SET_COLP: return "RIL_REQUEST_SET_COLP";
        case RIL_REQUEST_GET_COLR: return "RIL_REQUEST_GET_COLR";
        case RIL_REQUEST_ADD_IMS_CONFERENCE_CALL_MEMBER: return "ADD_IMS_CONFERENCE_CALL_MEMBER";
        case RIL_REQUEST_REMOVE_IMS_CONFERENCE_CALL_MEMBER: return "REMOVE_IMS_CONFERENCE_CALL_MEMBER";
        case RIL_REQUEST_CONFERENCE_DIAL: return "CONFERENCE_DIAL";
        case RIL_REQUEST_DIAL_WITH_SIP_URI: return "DIAL_WITH_SIP_URI";
        case RIL_REQUEST_HOLD_CALL: return "HOLD_CALL";
        case RIL_REQUEST_RESUME_CALL: return "RESUME_CALL";
        case RIL_UNSOL_ECONF_SRVCC_INDICATION : return "ECONF_SRVCC_INDICATION";
        case RIL_UNSOL_ECONF_RESULT_INDICATION : return "ECONF_RESULT_INDICATION";
        case RIL_UNSOL_MAL_AT_INFO : return "UNSOL_MAL_AT_INFO";
        case RIL_REQUEST_MODEM_POWEROFF: return "MODEM_POWEROFF";
        case RIL_REQUEST_MODEM_POWERON: return "MODEM_POWERON";
        case RIL_REQUEST_WRITE_SMS_TO_SIM: return "WRITE_SMS_TO_SIM";
        case RIL_REQUEST_QUERY_ICCID: return "RIL_REQUEST_QUERY_ICCID";
        case RIL_UNSOL_TX_POWER: return "RIL_UNSOL_TX_POWER";
        case RIL_UNSOL_NETWORK_INFO: return "RIL_UNSOL_NETWORK_INFO";
        case RIL_REQUEST_DELETE_SMS_ON_SIM: return "DELETE_SMS_ON_SIM";
        case RIL_REQUEST_SET_IMSCFG: return "RIL_REQUEST_SET_IMSCFG";
#ifdef ECALL_SUPPORT
        case RIL_REQUEST_ECALL_FAST_MAKE_ECALL: return "ECALL_FAST_MAKE_ECALL";
        case RIL_REQUEST_ECALL_SET_IVS: return "RIL_REQUEST_ECALL_SET_IVS";
        case RIL_REQUEST_ECALL_SET_PSAP: return "RIL_REQUEST_ECALL_SET_PSAP";
        case RIL_REQUEST_ECALL_MAKE_ECALL: return "RIL_REQUEST_ECALL_MAKE_ECALL";
        case RIL_REQUEST_ECALL_IVS_PUSH_MSD: return "RIL_REQUEST_ECALL_IVS_PUSH_MSD";
        case RIL_REQUEST_ECALL_PSAP_PULL_MSD: return "RIL_REQUEST_ECALL_PSAP_PULL_MSD";
        case RIL_UNSOL_ECALL_MSDHACK : return "ECALL_MSDHACK";
        case RIL_REQUEST_ECALL_SET_MSD: return "RIL_REQUEST_ECALL_SET_MSD";
        case RIL_REQUEST_ECALL_CTRL_SEQUENCE: return "ECALL_SET_CTRL_SEQUENCE";
        case RIL_UNSOL_ECALL_INDICATIONS : return "ECALL_INDICATIONS";
        case RIL_REQUEST_ECALL_RESET_IVS: return "RIL_REQUEST_ECALL_RESET_IVS";
        case RIL_REQUEST_ECALL_SET_PRI: return "RIL_REQUEST_ECALL_SET_PRI";
        case RIL_REQUEST_ECALL_SET_TEST_NUM: return "RIL_REQUEST_ECALL_SET_TEST_NUM";
        case RIL_REQUEST_ECALL_SET_RECONF_NUM: return "RIL_REQUEST_ECALL_SET_RECONF_NUM";
        case RIL_REQUEST_SYNC_DATA_SETTINGS_TO_MD: return "RIL_REQUEST_SYNC_DATA_SETTINGS_TO_MD";
        case RIL_REQUEST_ECALL_SET_NAD_DEREGISTRATION_TIME: return "RIL_REQUEST_ECALL_SET_NAD_DEREGISTRATION_TIME";
        case RIL_REQUEST_ECALL_SET_REGISTRATION_STATE: return "RIL_REQUEST_ECALL_SET_REGISTRATION_STATE";
#endif /*ECALL_SUPPORT*/
#ifdef KEEP_ALIVE
        case RIL_REQUEST_START_KEEPALIVE_PRO: return "RIL_REQUEST_START_KEEPALIVE_PRO";
        case RIL_REQUEST_STOP_KEEPALIVE_PRO: return "RIL_REQUEST_STOP_KEEPALIVE_PRO";
        case RIL_UNSOL_KEEPALIVE_STATUS_PRO: return "RIL_UNSOL_KEEPALIVE_STATUS_PRO";
#endif /*KEEP_ALIVE*/
        case RIL_REQUEST_SEND_USSI: return "SEND_USSI";
        case RIL_REQUEST_CANCEL_USSI: return "CANCEL_USSI";
        case RIL_REQUEST_GET_SMS_SIM_MEM_STATUS: return "GET_SMS_SIM_MEM_STATUS";
        case RIL_UNSOL_SIP_CALL_PROGRESS_INDICATOR: return "RIL_UNSOL_SIP_CALL_PROGRESS_INDICATOR";
        case RIL_REQUEST_REPORT_AIRPLANE_MODE: return "RIL_REQUEST_REPORT_AIRPLANE_MODE";
        case RIL_REQUEST_SET_ECC_NUM: return "RIL_REQUEST_SET_ECC_NUM";
        case RIL_REQUEST_GET_ECC_NUM: return "RIL_REQUEST_GET_ECC_NUM";
        case RIL_UNSOL_ECC_NUM: return "RIL_UNSOL_ECC_NUM";
        case RIL_REQUEST_QUERY_AVAILABLE_NETWORKS_WITH_ACT: return "RIL_REQUEST_QUERY_AVAILABLE_NETWORKS_WITH_ACT";
        case RIL_REQUEST_GSM_GET_BROADCAST_LANGUAGE: return "RIL_REQUEST_GSM_GET_BROADCAST_LANGUAGE";
        case RIL_REQUEST_GSM_SET_BROADCAST_LANGUAGE: return "RIL_REQUEST_GSM_SET_BROADCAST_LANGUAGE";
        case RIL_UNSOL_CALL_INFO_INDICATION: return "RIL_UNSOL_CALL_INFO_INDICATION";
        case RIL_REQUEST_MODIFY_APN: return "RIL_REQUEST_MODIFY_APN";
        case RIL_REQUEST_RESET_APN: return "RIL_REQUEST_RESET_APN";
        case RIL_REQUEST_QUERY_SIM_RETRY_COUNT: return "RIL_REQUEST_QUERY_SIM_RETRY_COUNT";
        case RIL_REQUEST_QUERY_EID: return "RIL_REQUEST_QUERY_EID";
        case LYNQ_REQUEST_SET_DTMF_VOLUME: return "RIL_REQUEST_SET_DTMF_VOLUME";
        case LYNQ_REQUEST_SET_SPEECH_VOLUME: return "RIL_REQUEST_SET_SPEECH_VOLUME";
        case LYNQ_REQUEST_GET_SPEECH_VOLUME: return "RIL_REQUEST_GET_SPEECH_VOLUME";
        case LYNQ_REQUEST_RECORD: return "RIL_REQUEST_RECORD";
        /*warren add for t800 ril service 2022/1/22 start*/
        case LYNQ_REQUEST_WRITE_SMS_TO_MEMORY: return "LYNQ_REQUEST_WRITE_SMS_TO_MEMORY";
        case LYNQ_REQUEST_READ_SMS_FROM_MEMORY: return "LYNQ_REQUEST_READ_SMS_FROM_MEMORY";
        case LYNQ_REQUEST_DELETE_SMS_FROM_MEMORY: return "LYNQ_REQUEST_DELETE_SMS_FROM_MEMORY";
        case LYNQ_REQUEST_LIST_SMS_FROM_MEMORY: return "LYNQ_REQUEST_LIST_SMS_FROM_MEMORY";
        case LYNQ_REQUEST_SET_DEFAULT_SIM_ALL:return "SET_DEFAULT_SIM_ALL";
        case LYNQ_REQUEST_CHANGE_SCREEN_STATE:return "LYNQ_REQUEST_CHANGE_SCREEN_STATE";/*jb.qi add for two sim sleep 2022/9/19*/
        case LYNQ_REQUEST_CHANGE_RADIO:return "LYNQ_REQUEST_CHANGE_RADIO";/*lei add for factory test of sleep 2022/9/19*/
        /*warren add for t800 ril service 2022/1/22 end*/
        default: return "<unknown request>";
    }
}


