#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <log/log.h>
#include <log/log.h>
#include "lynq_monitor_internal.h"
#include "liblog/lynq_deflog.h"

#define USER_LOG_TAG "LYNQ_MONITOR"
#define MAX_KEY_LENGTH 32
#define MAX_VALUE_LENGTH 128
#define MAX_ITEMS 128

typedef struct 
{
    char key[MAX_KEY_LENGTH + 1];
    char value[MAX_VALUE_LENGTH + 1];
} Lynq_ConfigItem;

static Lynq_ConfigItem s_items[MAX_ITEMS] = {0};
static int s_item_count = 0;

// 读取配置文件的函数，处理换行符
static lynq_read_ini_config(const char* filename, Lynq_ConfigItem items[], int* item_count)
{
    FILE* file = fopen(filename, "r");
    if (!file)
    {
        perror("Error opening file");
        return -1;
    }

    int count = 0;
    char line[MAX_VALUE_LENGTH + MAX_KEY_LENGTH + 9] = {0}; // 留足空间给等号、可能的引号和一些空白字符
    while (fgets(line, sizeof(line), file) != NULL && count < MAX_ITEMS)
    {
        // 查找最后一个'\n'，并删除之前的'\r'
        char* newline = strrchr(line, '\n');
        if (newline)
        {
            if (newline > line && newline[-1] == '\r')
            {
                newline[-1] = '\0';
            }
            else
            {
                *newline = '\0';
            }
        }

        size_t len = strlen(line);
        if (len == 0 || line[0] == '#') continue; // 跳过空行和注释行

        // 移除行首和行尾的空白字符
        char* start = line;
        char* end = line + len - 1;
        while (isspace(*start)) start++;
        while (isspace(*end) && start < end) end--;

        *++end = '\0'; // 重新设置结束符

        char* eq_pos = strchr(start, '=');
        if (!eq_pos) continue; // 没有等号，跳过

        // 确保key和value不会越界
        size_t key_len = eq_pos - start;
        if (key_len >= MAX_KEY_LENGTH || len - key_len - 1 >= MAX_VALUE_LENGTH)
        {
            printf("Line exceeds max length: %s\n", line);
            continue;
        }

        // 复制key和value
        strncpy(items[count].key, start, key_len);
        items[count].key[key_len] = '\0'; // 确保字符串结束
        strncpy(items[count].value, eq_pos + 1, MAX_VALUE_LENGTH);
        items[count].value[strcspn(items[count].value, "\n")] = '\0'; // 去掉可能的换行符，并确保字符串结束

        count++;
    }

    fclose(file);
    *item_count = count;
    return 0;
}

int lynq_config_load(const char* config_path)
{
    s_item_count = 0;
    if (lynq_read_ini_config(config_path, s_items, &s_item_count) == 0)
    {
        for (int i = 0; i < s_item_count; ++i)
        {
            LYDBGLOG("%s=%s\n", s_items[i].key, s_items[i].value);
        }
    }
    return 0;
}
const char* lynq_get_config_value(const char* key, char value_buffer[], size_t buffer_size)
{
    for (int i = 0; i < s_item_count; ++i)
    {
        if (strcmp(s_items[i].key, key) == 0)
        {
            strncpy(value_buffer, s_items[i].value, buffer_size);
            return value_buffer;
        }
    }
    return NULL; // 没找到key
}

const char* lynq_get_config_value_with_default(const char* key, char value_buffer[], 
                                               size_t buffer_size, const char* default_value)
{
    if (NULL == lynq_get_config_value(key, value_buffer, buffer_size))
    {
        strncpy(value_buffer, default_value, buffer_size);
    }
    return value_buffer;
}
// 将int类型的value作为出参，返回值表示操作是否成功
int lynq_get_config_int_value(const char* key, int* value)
{
    char value_buffer[MAX_VALUE_LENGTH + 1] = {0};
    if (lynq_get_config_value(key, value_buffer, sizeof(value_buffer)))
    {
        if (sscanf(value_buffer, "%d", value) == 1)
        {
            return 1; // 成功
        }
    }
    return 0; // 失败
}

int lynq_get_config_int_value_with_default(const char* key, int default_value)
{
    int value = 0;
    if (lynq_get_config_int_value(key, &value))
    {
        return value;
    }
    return default_value;
}
