//SPDX-License-Identifier: MediaTekProprietary
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <net/if.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <linux/netlink.h>
#include <linux/rtnetlink.h>
#include <linux/if_addr.h>
#include <../include/pal_nm.h>
#include <syslog.h>
#ifdef LYNQ_MAY_SUPPORT
#include <cutils/sockets.h>
#endif //LYNQ_MAY_SUPPORT
#include <linux/fib_rules.h>
#include <asm/types.h>
#include <sys/uio.h>
#include <errno.h>

#include <inttypes.h>


#define	PAL_NM_CMD_BUF_SIZE			256
#define	PAL_NM_DIGTAL_STRING_SIZE	10
#define ETH_ALEN                        6
#define IPV6_PTRADDR_LEN                46
#define MAX_DNS_NUMS                     2

const char sys_net_path[] = "/sys/class/net";
static const char IPV4_FORWARDING_PROC_FILE[] = "/proc/sys/net/ipv4/ip_forward";
static const char IPV6_FORWARDING_PROC_FILE[] = "/proc/sys/net/ipv6/conf/all/forwarding";

static const char* QTAGUID_IFACE_STATS = "/proc/net/xt_qtaguid/iface_stat_fmt";
//static const char* QTAGUID_UID_STATS = "/proc/net/xt_qtaguid/stats";

const uint16_t FRA_UID_START = 18;
const uint16_t FRA_UID_END   = 19;

struct rtattr FRATTR_PRIORITY;
struct rtattr FRATTR_TABLE;
struct rtattr FRATTR_FWMARK;
struct rtattr FRATTR_FWMASK;
struct rtattr FRATTR_UID_START;
struct rtattr FRATTR_UID_END;

struct rtattr RTATTR_TABLE;
struct rtattr RTATTR_OIF;

const uint16_t NETLINK_REQUEST_FLAGS = NLM_F_REQUEST | NLM_F_ACK;
const uint16_t NETLINK_CREATE_REQUEST_FLAGS = NLM_F_REQUEST | NLM_F_ACK | NLM_F_CREATE | NLM_F_EXCL;

uint8_t PADDING_BUFFER[RTA_ALIGNTO] = {0, 0, 0, 0};


struct in6_ifreq{
	struct in6_addr ifr6_addr;
	int ifr6_prefixlen;
	int ifr6_ifindex;
};

#define PAL_DEBUG_TRACE	5
int pal_nm_debug = PAL_DEBUG_TRACE;

#if 0
#define PAL_DEBUG_PRINT(level, str, args...)               	\
	do {                                                    \
        if(level<=pal_nm_debug)                         	\
        {   												\
			printf("\n\r[PAL-NM] %s - ", __FUNCTION__); 	\
    		printf(str, ##args);                            \
        }                                                   \
	} while(0)
#else

#define PAL_DEBUG_PRINT(level, str, args...)               	\
	do {                                                    \
        if(level<=pal_nm_debug)                         	\
        {   												\
			syslog(LOG_DEBUG, "[PAL-NM][%s:%d:]--- " str "\n", __FUNCTION__, __LINE__,  ## args);\
        }                                                   \
	} while(0)


#endif
static int get_ifindex(const char *dev, int *index)
{
   int fd;
   struct ifreq req={{{0}}};
   int ret;

   strncpy(req.ifr_name, dev, strlen(dev));

   if((fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
   {
      PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "create socket fail:%s", strerror(errno));
      return -1;
   }

   ret = ioctl(fd, SIOCGIFINDEX, &req);

   if(ret == -1)
   {
        *index = 0;
        close(fd);
        PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "ioctl fail:%s", strerror(errno));
        return ret;
   }
   *index = req.ifr_ifindex;
   PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "req.ifr_ifindex[%d]", req.ifr_ifindex);
   close(fd);
   return 0;
}
static int ipv4NetmaskToPrefixLength(in_addr_t mask)
{
    int prefixLength = 0;
    uint32_t m = (uint32_t)ntohl(mask);
    while (m & 0x80000000) {
        prefixLength++;
        m = m << 1;
    }
    return prefixLength;
}

static NM_IP_TYPE_V6 _net_ip_scope_v6(const char* ip_in)
{
    if (ip_in == NULL) return NM_IP_TYPE_UNKNOWN;
    if (strchr(ip_in, ':') == NULL) return NM_IP_TYPE_UNKNOWN;
    if (strcmp(ip_in, "::") == 0) return NM_IP_TYPE_V6_UNSPEC;
    if (strcmp(ip_in, "::1") == 0) return NM_IP_TYPE_V6_LOOPBACK;
    if (!strncmp(ip_in, "2", 1) || !strncmp(ip_in, "3", 1))  return NM_IP_TYPE_V6_GLOBAL;

    if (strncmp(ip_in, "ff00", 4) == 0) return NM_IP_TYPE_V6_MULTICAST;
    if (strncmp(ip_in, "fe80", 4) == 0) return NM_IP_TYPE_V6_LINKLOCAL;
    if (strncmp(ip_in, "fec0", 4) == 0) return NM_IP_TYPE_V6_SITELOCAL;
    if (!strncmp(ip_in, "fc", 2) || !strncmp(ip_in, "fd", 2))  return NM_IP_TYPE_V6_UNIQUE;

    return NM_IP_TYPE_UNKNOWN;
}

static NM_IP_TYPE_V6 _net_ip_type_v6(int scope, int ifa_flags)
{
    if(NM_IP_TYPE_V6_GLOBAL != scope)
		return scope;

	if(ifa_flags & IFA_F_PERMANENT)
		return NM_IP_TYPE_V6_GLOBAL_PERNAMENT;

	if(ifa_flags & IFA_F_SECONDARY)
		return NM_IP_TYPE_V6_GLOBAL_TEMP;

	return NM_IP_TYPE_V6_GLOBAL_DYNAMIC;
}
int nm_interface_ipv4_set(const char *intf_name, const char* addr, int prefix_len)
{
	char	cmd_buf[PAL_NM_CMD_BUF_SIZE];
	int 	len;
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
	if (!intf_name || !addr)
		return	-1;
	if(strlen(intf_name)==0 || strlen(addr)==0)
		return -1;
	//interface setcfg <interface> [addr | prefixLength] [up | down]

	len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc interface setcfg %s %s %d", intf_name, addr, prefix_len);
	if (len >= PAL_NM_CMD_BUF_SIZE)
		return	-1;

	PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "command:%s", cmd_buf);

	return	system(cmd_buf);
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add

}



int nm_interface_ipv4_up(const char *intf_name)
{
	char	cmd_buf[PAL_NM_CMD_BUF_SIZE];
	int		len;
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
	if (!intf_name)
		return	-1;

	//interface setcfg <interface> [addr | prefixLength] [up | down]

	len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc interface setcfg %s up", intf_name);
	if (len >= PAL_NM_CMD_BUF_SIZE)
		return	-1;

	PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "command:%s", cmd_buf);

	return  system(cmd_buf);
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add

}

int nm_interface_ipv4_down(const char *intf_name)
{
	char	cmd_buf[PAL_NM_CMD_BUF_SIZE];
	int		len;
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
	if (!intf_name)
		return	-1;

	//interface setcfg <interface> [addr | prefixLength] [up | down]

	len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc interface setcfg %s down", intf_name);
	if (len >= PAL_NM_CMD_BUF_SIZE)
		return	-1;

	PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "command:%s", cmd_buf);

	return  system(cmd_buf);
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add

}
int nm_interface_ipv4_mtu_set(const char *intf_name, int mtu)
{
	char	cmd_buf[PAL_NM_CMD_BUF_SIZE];
	int		len;
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
	if (!intf_name)
		return	-1;

	//interface setmtu <interface> <mtu>

	len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc interface setmtu %s %d", intf_name, mtu);
	if (len >= PAL_NM_CMD_BUF_SIZE)
		return	-1;

	PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "command:%s", cmd_buf);

	return system(cmd_buf);
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add

}

int nm_interface_ipv4_info_get(const char *intf_name, unsigned long *addr,
                                                   int *prefix_len, int *mtu, unsigned int*flags)
{
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
    int sockfd;
    struct ifreq ifr;
    char mtuPath[30] = {0};
    char chmtu[11] = {0};
    FILE *fp = NULL;
	if (!intf_name || !addr ||
		!intf_name || !mtu || !flags)
	{
		return	-1;
	}

    if((sockfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
    {
        PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "create socket fail");
        return -1;
    }

    memset(&ifr, 0, sizeof(struct ifreq));
    PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "IFNAMSIZ[%d]", IFNAMSIZ);
    strncpy(ifr.ifr_name, intf_name, IFNAMSIZ);
    ifr.ifr_name[IFNAMSIZ - 1] = 0;

    if (addr != NULL) {
        if(ioctl(sockfd, SIOCGIFADDR, &ifr) < 0) {
            PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "SIOCGIFADDR fail");
            *addr = 0;
        } else {
            *addr = ((struct sockaddr_in*) &ifr.ifr_addr)->sin_addr.s_addr;
            PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "*addr[%lu]", *addr);
        }
    }

    if (prefix_len != NULL) {
        if(ioctl(sockfd, SIOCGIFNETMASK, &ifr) < 0) {
            PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "SIOCGIFNETMASK fail");
            *prefix_len = 0;
        } else {
            *prefix_len = ipv4NetmaskToPrefixLength(
                    ((struct sockaddr_in*) &ifr.ifr_addr)->sin_addr.s_addr);
            PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "*prefix_len[%u]", *prefix_len);
        }
    }

    if (flags != NULL) {
        if(ioctl(sockfd, SIOCGIFFLAGS, &ifr) < 0) {
            *flags = 0;
        } else {
            *flags = ifr.ifr_flags;
            PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "*flags[%u]", *flags);
        }
    }

    snprintf(mtuPath, 29, "%s/%s/mtu", sys_net_path, intf_name);
    PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "mtuPath[%s]", mtuPath);

    fp = fopen(mtuPath , "r");
    if (NULL == fp)
    {
        PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "open %s fail", mtuPath);
        *mtu = 0;
    }else
    {
        fread(chmtu, 1, 10, fp);
		chmtu[10] = '\0';
        PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "chmtu[%s]", chmtu);
        *mtu = atoi(chmtu);
        PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "*mtu[%u]", *mtu);
        fclose(fp);
    }
    close(sockfd);

	return	0;
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add
}

int nm_interface_ipv4_hwaddr_get(const char *intf_name, unsigned char *hw_addr)
{
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
    int ret;
    int sockfd;
    struct ifreq ifr;
	if (!intf_name || !hw_addr)
	{
		return	-1;
	}

    if((sockfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
    {
        PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "create socket fail");
        return -1;
    }

    bzero(&ifr, sizeof(ifr));
    ifr.ifr_name[IFNAMSIZ -1] = '\0';
    strncpy(ifr.ifr_name, intf_name, IFNAMSIZ);
    ifr.ifr_name[IFNAMSIZ -1] = '\0';

    ret = ioctl(sockfd, SIOCGIFHWADDR, &ifr);
    if(ret < 0) {
        close(sockfd);
        return -1;
    }

    memcpy(hw_addr, &ifr.ifr_hwaddr.sa_data, ETH_ALEN);

    close(sockfd);
	return	0;
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add
}

int nm_interface_ipv4_addr_clear(const char *intf_name)
{
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
	char	cmd_buf[PAL_NM_CMD_BUF_SIZE];
	int 	len;

	if (!intf_name)
		return	-1;

	// interface clearaddrs <interface>

	len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc interface clearaddrs %s", intf_name);
	if (len >= PAL_NM_CMD_BUF_SIZE)
		return	-1;

	PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "command:%s", cmd_buf);

	return system(cmd_buf);
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add
}


/*interface ipv6 related APIs*/
int nm_interface_ipv6_mtu_set(const char *intf_name, int mtu)
{
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
	char	cmd_buf[PAL_NM_CMD_BUF_SIZE];
	int		len;

	if (!intf_name)
		return	-1;

	//interface setmtu <interface> <mtu>

	len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc interface setmtu %s %d", intf_name, mtu);
	if (len >= PAL_NM_CMD_BUF_SIZE)
		return	-1;

	PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "command:%s", cmd_buf);

	return system(cmd_buf);
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add

}

int nm_interface_ipv6_addr_clear(const char *intf_name)
{
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
	char	cmd_buf[PAL_NM_CMD_BUF_SIZE];
	int 	len;

	if (!intf_name)
		return	-1;

	// interface clearaddrs <interface>

	len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc interface clearaddrs %s", intf_name);
	if (len >= PAL_NM_CMD_BUF_SIZE)
		return	-1;

	PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "command:%s", cmd_buf);

	return system(cmd_buf);
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add

}

int nm_interface_ipv6_set(const char *intf_name, int on)
{
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
	char	cmd_buf[PAL_NM_CMD_BUF_SIZE];
	int 	len;

	if (!intf_name)
		return	-1;

	// interface ipv6 <interface> <enable | disable>

	if (on)
		len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc interface ipv6 %s enable", intf_name);
	else
		len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc interface ipv6 %s disable", intf_name);

	if (len >= PAL_NM_CMD_BUF_SIZE)
		return	-1;

	PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "command:%s", cmd_buf);

	return system(cmd_buf);
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add

}

int nm_interface_ipv6_privacyextenstion_enable_set(const char *intf_name, int on)
{
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
	char	cmd_buf[PAL_NM_CMD_BUF_SIZE];
	int 	len;

	if (!intf_name)
		return	-1;

	// interface ipv6privacyextensions <interface> <enable | disable>

	if (on)
		len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc interface ipv6privacyextensions %s enable", intf_name);
	else
		len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc interface ipv6privacyextensions %s disable", intf_name);

	if (len >= PAL_NM_CMD_BUF_SIZE)
		return	-1;

	PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "command:%s", cmd_buf);

	return system(cmd_buf);
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add

}

int nm_interface_ipv6_ndoffload_enable_set(const char *intf_name, int on)
{
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
	char	cmd_buf[PAL_NM_CMD_BUF_SIZE];
	int 	len;

	if (!intf_name)
		return	-1;

	// interface ipv6ndoffload <interface> <enable | disable>

	if (on)
		len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc interface ipv6ndoffload %s enable", intf_name);
	else
		len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc interface ipv6ndoffload %s disable", intf_name);

	if (len >= PAL_NM_CMD_BUF_SIZE)
		return	-1;

	PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "command:%s", cmd_buf);

	return system(cmd_buf);
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add

}

int nm_interface_ipv6_addr_set(const char *intf_name, const char *addr, int prefix_len)
{
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
	int fd;
	int ret;
    int index = 0;
    struct in6_ifreq v6_ifreq = {0};
	if (!intf_name || !addr)
	{
		return	-1;
	}

    ret = get_ifindex(intf_name, &index);
    if (ret)
    {
        PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "get_ifindex");
        return ret;
    }

    PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "get_ifindex index[%d]", index);

	fd = socket(AF_INET6, SOCK_DGRAM,0);
    if ((fd = socket(AF_INET6, SOCK_DGRAM, 0)) < 0)
    {
        PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "open socket fail:%s", strerror(errno));

    }

	ret = inet_pton(AF_INET6, addr, &v6_ifreq.ifr6_addr.s6_addr);
	if(ret != 1)
	{
		PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "inet_pton fail:%s", strerror(errno));
		close(fd);
		return -1;
	}

	v6_ifreq.ifr6_ifindex = index;
	v6_ifreq.ifr6_prefixlen = prefix_len;

	ret = ioctl(fd, SIOCSIFADDR, &v6_ifreq);
	if(ret == -1)
	{
		PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "ioctl SIOCSIFADDR fail:%s", strerror(errno));
		close(fd);
		return -1;
	}
	close(fd);
	return	0;
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add
}

int nm_interface_ipv6_addr_get(const char *intf_name, int addr_type, char *addr, int *prefix_len)
{
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
	int status;
	char *buf = NULL;
	struct nlmsghdr *nlmp;
	int nlmsgLen;
	struct ifaddrmsg *rt_ifmsg;
	struct rtattr *rta;
	int rtattrlen;
	struct in6_addr *in6p;
	int ifidx;
	int ip_scope, ip_type=0;
	struct{
		  struct nlmsghdr n;
		  struct ifaddrmsg r;
	}req;
	if (!intf_name || !addr || !prefix_len)
	{
		return	-1;
	}

    PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "intf_name[%s], addr[%s], prefix[%d], addr_type[%u]", intf_name, addr, *prefix_len, addr_type);
	status = get_ifindex(intf_name, &ifidx);
    if (status)
    {
        PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "get_ifindex fail");
		return -1;
    }

    buf = (char *)malloc(16384);
    if (NULL == buf)
    {
		return -1;
    }

    memset(buf, 0, 16384);

	int fd = socket(PF_NETLINK, SOCK_DGRAM, NETLINK_ROUTE);

	memset(&req, 0, sizeof(req));
	req.n.nlmsg_len = NLMSG_LENGTH(sizeof(struct ifaddrmsg));
	req.n.nlmsg_flags = NLM_F_REQUEST | NLM_F_ROOT;
	req.n.nlmsg_type = RTM_GETADDR;
	req.r.ifa_family = AF_INET6;


	/* Time to send and recv the message from kernel */
	status = send(fd, &req, req.n.nlmsg_len, 0);
	if (status < 0) {
        PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "send fail:%s", strerror(errno));
		close(fd);
        if (buf)
        {
            free(buf);
            buf = NULL;
        }
		return -1;
	}



	status = recv(fd, buf, 16384, 0);
	if (status <= 0) {
        PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "send fail:%s", strerror(errno));
		close(fd);
        if (buf)
        {
            free(buf);
            buf = NULL;
        }
		return -1;
	}

	/* Typically the message is stored in buf, so we need to parse the message to get the required data*/
	for(nlmp = (struct nlmsghdr *)buf; status  > sizeof(struct nlmsghdr);)
	{
		nlmsgLen = nlmp->nlmsg_len;
		if (!NLMSG_OK(nlmp, status))
		{
			PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "NLMSG not OK");
            if (buf)
            {
                free(buf);
                buf = NULL;
            }
			close(fd);
			return 1;
		}
		rt_ifmsg = (struct ifaddrmsg *)NLMSG_DATA(nlmp);
		rta = (struct rtattr *)IFA_RTA(rt_ifmsg);
		rtattrlen = IFA_PAYLOAD(nlmp);

		status -= NLMSG_ALIGN(nlmsgLen);
		nlmp = (struct nlmsghdr*)((char*)nlmp + NLMSG_ALIGN(nlmsgLen));
		/*Device not match, to next*/
		if(ifidx != rt_ifmsg->ifa_index)
			continue;

		for (; RTA_OK(rta, rtattrlen); rta = RTA_NEXT(rta, rtattrlen))
		{
			if(rta->rta_type == IFA_CACHEINFO)
				continue;
			if(rta->rta_type == IFA_LOCAL||rta->rta_type == IFA_BROADCAST||rta->rta_type == IFA_ANYCAST)
				PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "Find curious ipv6-address!\n");
			if(rta->rta_type == IFA_ADDRESS)
			{
				in6p = (struct in6_addr *)RTA_DATA(rta);
				inet_ntop(AF_INET6, in6p, addr, IPV6_PTRADDR_LEN);
				/*Get type match address*/
				ip_scope = _net_ip_scope_v6(addr);
                PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "ip:ip_scope[%d]\n", ip_scope);
				ip_type  = _net_ip_type_v6(ip_scope, rt_ifmsg->ifa_flags);
                PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "ip:ip_type[%d]\n", ip_type);
				*prefix_len =  rt_ifmsg->ifa_prefixlen;
				PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "ip: %s, prefix: %d, type: %d==\n", addr, *prefix_len, ip_type);
				if(ip_type & addr_type)/*addr type is we need, and may have a fix of several. Just return the first valid*/
				{
					close(fd);
                    if (buf)
                    {
                        free(buf);
                        buf = NULL;
                    }
                    PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "return success\n");
					return 0;
				}
			}
		}
	}

	close(fd);
    if (buf)
    {
        free(buf);
        buf = NULL;
    }

    PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "return fail\n");
	return -1;
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add
}


/*network related APIs*/
int nm_network_create(unsigned net_id, const char *permission)
{
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
	char	cmd_buf[PAL_NM_CMD_BUF_SIZE];
	int 	len;

	/*
	* network create <netId> [permission]
	* permisstion: NETWORK / SYSTEM
	*/
	if (permission)
		len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc network create %d %s", net_id, permission);
	else
		len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc network create %d", net_id);
	if (len >= PAL_NM_CMD_BUF_SIZE)
		return	-1;

	PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "command:%s", cmd_buf);

	return system(cmd_buf);
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add

}

int nm_network_destroy(unsigned net_id)
{
	char	cmd_buf[PAL_NM_CMD_BUF_SIZE];
	int 	len;
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
	//network destroy <netId>

	len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc network destroy %d", net_id);
	if (len >= PAL_NM_CMD_BUF_SIZE)
		return	-1;

	PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "command:%s", cmd_buf);

	return system(cmd_buf);
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add

}

int nm_network_default_set(unsigned net_id)
{
	char	cmd_buf[PAL_NM_CMD_BUF_SIZE];
	int 	len;
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
	//network default set <netId>

	len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc network default set %d", net_id);
	if (len >= PAL_NM_CMD_BUF_SIZE)
		return	-1;

	PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "command:%s", cmd_buf);

	return system(cmd_buf);
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add

}

int nm_network_default_clear()
{
	char	cmd_buf[PAL_NM_CMD_BUF_SIZE];
	int 	len;
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
	//network default clear

	len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc network default clear");
	if (len >= PAL_NM_CMD_BUF_SIZE)
		return	-1;

	PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "command:%s", cmd_buf);

	return system(cmd_buf);
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add

}

int nm_network_interface_add(unsigned net_id, const char* intf_name)
{
	char	cmd_buf[PAL_NM_CMD_BUF_SIZE];
	int 	len;
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
	//network interface add <netId> <interface>

	len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc network interface add %d %s", net_id, intf_name);
	if (len >= PAL_NM_CMD_BUF_SIZE)
		return	-1;

	PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "command:%s", cmd_buf);

	return system(cmd_buf);
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add

}

int nm_network_interface_remove(unsigned net_id, const char* intf_name)
{
	char	cmd_buf[PAL_NM_CMD_BUF_SIZE];
	int 	len;
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
	//network interface remove <netId> <interface>

	len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc network interface remove %d %s", net_id, intf_name);
	if (len >= PAL_NM_CMD_BUF_SIZE)
		return	-1;

	PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "command:%s", cmd_buf);

	return system(cmd_buf);
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add

}

int nm_network_route_add(unsigned net_id, const char* intf_name,
                                      const char  *destination, const char *nexthop,
                                      int legacy, unsigned int uid)
{

	char	cmd_buf[PAL_NM_CMD_BUF_SIZE];
	int		len;
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
	if (!intf_name || !destination)
	{
		return	-1;
	}

	/* network route [legacy <uid>] add <netId> <interface> <dst> [nexthop] */
	if (legacy && nexthop)
	{
		len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc network route legacy %d add %d %s %s %s",
													 uid, net_id, intf_name, destination, nexthop);
	}
	else if (!legacy && nexthop)
	{
		len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc network route add %d %s %s %s",
													 net_id, intf_name, destination, nexthop);

	}
	else if (legacy && !nexthop)
	{
		len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc network route legacy %d add %d %s %s",
													 uid, net_id, intf_name, destination);

	}
	else
	{
		len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc network route add %d %s %s",
													 net_id, intf_name, destination);

	}

	if (len >= PAL_NM_CMD_BUF_SIZE)
		return	-1;

	PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "command:%s", cmd_buf);

	return system(cmd_buf);
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add

}

int nm_network_route_remove( unsigned net_id, const char* intf_name,
                                            const char *destination, const char *nexthop,
                                            int legacy, unsigned int uid)
{
	char	cmd_buf[PAL_NM_CMD_BUF_SIZE];
	int		len;
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
	if (!intf_name || !destination)
	{
		return	-1;
	}

	/*network route [legacy <uid>] remove <netId> <interface> <dst> [nexthop]*/
	if (legacy && nexthop)
	{
		len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc network route legacy %d remove %d %s %s %s",
													 uid, net_id, intf_name, destination, nexthop);
	}
	else if (!legacy && nexthop)
	{
		len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc network route remove %d %s %s %s",
													 net_id, intf_name, destination, nexthop);

	}
	else if (legacy && !nexthop)
	{
		len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc network route legacy %d remove %d %s %s",
													 uid, net_id, intf_name, destination);

	}
	else
	{
		len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc network route remove %d %s %s",
													 net_id, intf_name, destination);

	}


	if (len >= PAL_NM_CMD_BUF_SIZE)
		return	-1;


	PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "command:%s", cmd_buf);

	return system(cmd_buf);
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add

}


/*ipfwd related APIs*/
int nm_fwd_ipv4_set(int enable, const char *requestor)
{
	char	cmd_buf[PAL_NM_CMD_BUF_SIZE];
	int 	len;
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
	// ipfwd <status | enable | disable> <requestor>

	if (enable)
		len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc ipfwd enable %s", requestor);
	else
		len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc ipfwd disable %s", requestor);
	if (len >= PAL_NM_CMD_BUF_SIZE)
		return	-1;

	PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "command:%s", cmd_buf);

	return system(cmd_buf);
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add

}

int nm_fwd_ipv4_get(int *enable)
{
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
	FILE *fp = NULL;
    char fwd[11] = {0};
	if (!enable)
	{
		return	-1;
	}

    fp = fopen(IPV4_FORWARDING_PROC_FILE, "r");

    if (NULL == fp) {
        PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "Failed to open ipv6_forward (%s)", strerror(errno));
        *enable = 0;
        return -1;
    }

    fread(fwd, 1, 10, fp);

    *enable = atoi(fwd);

    fclose(fp);
	return	0;
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add

}

int nm_fwd_ipv6_set(int enable, const char *requestor)
{
	char	cmd_buf[PAL_NM_CMD_BUF_SIZE];
	int 	len;
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
	// ipv6fwd <status | enable | disable> <requestor>

	if (enable)
		len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc ipv6fwd enable %s", requestor);
	else
		len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc ipv6fwd disable %s", requestor);
	if (len >= PAL_NM_CMD_BUF_SIZE)
		return	-1;

	PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "command:%s", cmd_buf);

	return system(cmd_buf);
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add

}

int nm_fwd_ipv6_get(int *enable)
{
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
    FILE *fp = NULL;
    char fwd[11] = {0};
	if (!enable)
	{
		return	-1;
	}

    fp = fopen(IPV6_FORWARDING_PROC_FILE, "r");

    if (NULL == fp) {
        PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "Failed to open ipv6_forward (%s)", strerror(errno));
        *enable = 0;
        return -1;
    }

    fread(fwd, 1, 10, fp);

    *enable = atoi(fwd);

    fclose(fp);
	return	0;
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add

}

/*DNS related APIs*/
int nm_resolver_ipv6_dns_set(unsigned net_id, const char *domains, const char* const* servers,  int numservers, NM_NETWORK_TYPE type)
{
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
	char	cmd_buf[PAL_NM_CMD_BUF_SIZE];
	int 	len, total_len;
	int		index;
	char	*temp_ptr = cmd_buf;
    char    dns[200] = "/usr/bin/dns.script setv6";

	if (!domains)
		return	-1;

    if (numservers > MAX_DNS_NUMS)
    {
        numservers = MAX_DNS_NUMS;
    }

	//resolver setnetdns <netId> <domains> <dns1> <dns2>

	len = snprintf(temp_ptr, PAL_NM_CMD_BUF_SIZE, "ndc resolver setnetdns %d %s", net_id, domains);
	if (len >= PAL_NM_CMD_BUF_SIZE)
		return	-1;

	total_len = len;

	for (index=0; index<numservers; index++)
	{
        if (NM_NETWORK_TYPE_INTERNET == type)
            snprintf(&(dns[strlen(dns)]), 200-strlen(servers[index]), " %s", servers[index]);

		len = snprintf(temp_ptr+total_len, PAL_NM_CMD_BUF_SIZE-total_len, " %s", servers[index]);
		if ((total_len+len) >= PAL_NM_CMD_BUF_SIZE)
			return	-1;

		total_len += len;

	}

    if (NM_NETWORK_TYPE_INTERNET == type)
    {
        if (system(dns) == -1)
        {
            PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "dns:%s:%s", dns, strerror(errno));
        }
    }

	PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "command:%s", cmd_buf);

	return system(cmd_buf);
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add

}

int nm_resolver_ipv4_dns_set(unsigned net_id, const char *domains, const char* const* servers,  int numservers, NM_NETWORK_TYPE type)
{
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
	char	cmd_buf[PAL_NM_CMD_BUF_SIZE];
	int 	len, total_len;
	int		index;
	char	*temp_ptr = cmd_buf;
    char    dns[201] = "/usr/bin/dns.script set";

	if (!domains)
		return	-1;

    if (numservers > MAX_DNS_NUMS)
    {
        numservers = MAX_DNS_NUMS;
    }

	//resolver setnetdns <netId> <domains> <dns1> <dns2>

	len = snprintf(temp_ptr, PAL_NM_CMD_BUF_SIZE, "ndc resolver setnetdns %d %s", net_id, domains);
	if (len >= PAL_NM_CMD_BUF_SIZE)
		return	-1;

	total_len = len;

	for (index=0; index<numservers; index++)
	{
        if (NM_NETWORK_TYPE_INTERNET == type)
            snprintf(&(dns[strlen(dns)]), 200-strlen(servers[index]), " %s", servers[index]);

		len = snprintf(temp_ptr+total_len, PAL_NM_CMD_BUF_SIZE-total_len, " %s", servers[index]);
		if ((total_len+len) >= PAL_NM_CMD_BUF_SIZE)
			return	-1;


		total_len += len;

	}

	PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "command:%s", cmd_buf);
    if (NM_NETWORK_TYPE_INTERNET == type)
    {
        if (system(dns) == -1)
        {
            PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "dns:%s:%s", dns, strerror(errno));
        }
    }

	return system(cmd_buf);
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add

}

int nm_resolver_dns_set(unsigned net_id, const char *domains, const char* const* servers,  int numservers, NM_NETWORK_TYPE type)
{
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
    char    cmd_buf[PAL_NM_CMD_BUF_SIZE];
    int     len, total_len = 0;
    int     index;
    char    *temp_ptr = cmd_buf;

        if (!domains)
            return  -1;
    //resolver setnetdns <netId> <domains> <dns1> <dns2>

	len = snprintf(temp_ptr, PAL_NM_CMD_BUF_SIZE, "ndc resolver setnetdns %d %s", net_id, domains);

	total_len = len;

	if (len >= PAL_NM_CMD_BUF_SIZE)
		return	-1;

    for (index=0; index<numservers; index++)
	{
	    len = snprintf(temp_ptr+total_len, PAL_NM_CMD_BUF_SIZE-total_len, " %s", servers[index]);
		if ((total_len+len) >= PAL_NM_CMD_BUF_SIZE)
			return	-1;


		total_len += len;
	}

    PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "command:%s", cmd_buf);
    return system(cmd_buf);
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add

}
int nm_resolver_dns_clear(unsigned net_id)
{
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
	char	cmd_buf[PAL_NM_CMD_BUF_SIZE];
	int 	len;

	//resolver clearnetdns <netId>

	len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc resolver clearnetdns %d", net_id);
	if (len >= PAL_NM_CMD_BUF_SIZE)
		return	-1;

	PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "command:%s", cmd_buf);

	return system(cmd_buf);
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add

}

int nm_resolver_dns_cache_flush(unsigned net_id)
{
	char	cmd_buf[PAL_NM_CMD_BUF_SIZE];
	int 	len;
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
	//resolver flushnet <netId>
	len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "ndc resolver flushnet %d", net_id);
	if (len >= PAL_NM_CMD_BUF_SIZE)
		return	-1;

	PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "command:%s", cmd_buf);

	return system(cmd_buf);
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add

}

int nm_command_test(char *intf)
{
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
	char	*dns_servier[] =
							{"192.168.0.1",
							 "192.168.0.2",
							 "192.168.0.3",
							 "192.168.0.4",
							 "192.168.0.5",
							 " "};
	char	**dns_ptr = &(dns_servier[0]);
	pal_nm_debug = 6;

	/*interface command*/
	nm_interface_ipv4_up(intf);
	nm_interface_ipv4_down(intf);
	nm_interface_ipv4_mtu_set(intf, 512);
	nm_interface_ipv4_addr_clear(intf);
	nm_interface_ipv6_mtu_set(intf, 512);
	nm_interface_ipv6_set(intf, 1);
	nm_interface_ipv6_set(intf, 0);
	nm_interface_ipv6_privacyextenstion_enable_set(intf, 1);
	nm_interface_ipv6_privacyextenstion_enable_set(intf, 0);
	nm_interface_ipv6_ndoffload_enable_set(intf, 1);
	nm_interface_ipv6_ndoffload_enable_set(intf, 0);

	/*net_id network*/
	nm_network_create(3, "enable");
	nm_network_destroy(3);
	nm_network_default_set(3);
	nm_network_interface_add(3, intf);
	nm_network_interface_remove(3, intf);
	nm_network_route_add(3, intf, "192.168.0.11", NULL, 0, 1);
	nm_network_route_add(3, intf, "192.168.0.11", "192.168.0.22", 0, 1);
	nm_network_route_add(3, intf, "192.168.0.11", NULL, 1, 1);
	nm_network_route_add(3, intf, "192.168.0.11", "192.168.0.22", 1, 1);
	nm_network_route_remove(3, intf, "192.168.0.11", NULL, 0, 1);
	nm_network_route_remove(3, intf, "192.168.0.11", "192.168.0.22", 0, 1);
	nm_network_route_remove(3, intf, "192.168.0.11", NULL, 1, 1);
	nm_network_route_remove(3, intf, "192.168.0.11", "192.168.0.22", 1, 1);

	/*ipfwd*/
	nm_fwd_ipv4_set(1, "abc");
	nm_fwd_ipv4_set(0, "abc");
	nm_fwd_ipv6_set(1, "abc");
	nm_fwd_ipv6_set(0, "abc");

	/*dns*/
	nm_resolver_ipv4_dns_set(1, "my_domains", (const char* const*)dns_ptr, 5, NM_NETWORK_TYPE_INTERNET);
	nm_resolver_dns_clear(1);


	return	0;
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add

}
int do_cmd(char *cmd_buf,char *ret_buf)
{
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
    int sock;
	char buffer[1024];
	int offset = 0;
	char tmp[4];
    int i = 0;
	int code=0;
	if ((sock = socket_local_client("netd",
									 ANDROID_SOCKET_NAMESPACE_RESERVED,
									 SOCK_STREAM)) < 0) {
		close(sock);
		PAL_DEBUG_PRINT(PAL_DEBUG_TRACE,"Error connecting (%s)\n",strerror(errno));
		return -1;
	}
    if (write(sock, cmd_buf, strlen(cmd_buf) + 1) < 0) {
		close(sock);
		PAL_DEBUG_PRINT(PAL_DEBUG_TRACE,"write error\n");
        return -1;
    }

	fd_set read_fds;
	struct timeval to;
	int rc = 0;

	to.tv_sec = 10;
	to.tv_usec = 0;

	FD_ZERO(&read_fds);
	FD_SET(sock, &read_fds);

	if ((rc = select(sock +1, &read_fds, NULL, NULL, &to)) < 0) {
		close(sock);
		PAL_DEBUG_PRINT(PAL_DEBUG_TRACE,"Error in select (%s)\n", strerror(errno));
		return -1;
	} else if (!rc) {

		close(sock);
		PAL_DEBUG_PRINT(PAL_DEBUG_TRACE,"TIMEDOUT (%s)\n", strerror(errno));
		return ETIMEDOUT;
	} else if (FD_ISSET(sock, &read_fds)) {
		memset(buffer, 0, 1024);
		while ((rc = read(sock, buffer, 1024)) ) {
			if (rc == 0)
			{
				close(sock);
				PAL_DEBUG_PRINT(PAL_DEBUG_TRACE,"Lost connection to Netd - did it crash?\n");
				return -1;
			}
			else if(rc<0)
			{
				close(sock);
				PAL_DEBUG_PRINT(PAL_DEBUG_TRACE,"Error reading data (%s)\n", strerror(errno));
				return -1;
			}
			else
			{
				offset=0;
				for (i = 0; i < rc; i++) {
					if (buffer[i] == '\0') {
						strncpy(tmp, buffer + offset, 3);
						tmp[3] = '\0';
						code = atoi(tmp);
						offset = i + 1;
						if (code >= 200 && code < 600)
						{
							strncpy(ret_buf, buffer + 4, offset);
							close(sock);
							PAL_DEBUG_PRINT(PAL_DEBUG_TRACE,"code (%d)\n", code);
							return code;
						}
					}
				}
			}
		}
	}
	close(sock);
	return -1;
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add
}
int nm_interface_get_netid(const char *intf_name,unsigned int *netid)
{
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
	char	cmd_buf[PAL_NM_CMD_BUF_SIZE];
	int 	len;
	char	ret_buf[1024];
	len = snprintf(cmd_buf, PAL_NM_CMD_BUF_SIZE, "0 network interface get %s", intf_name);
	PAL_DEBUG_PRINT(PAL_DEBUG_TRACE,"nm_interface_get start \n");
	if (len >= PAL_NM_CMD_BUF_SIZE)
		return	-1;

	if(do_cmd(cmd_buf,ret_buf)!=200)
	{
		PAL_DEBUG_PRINT(PAL_DEBUG_TRACE,"nm_interface_get return error \n");
		return -1;
	}

	*netid=atoi(ret_buf);
	PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "*netid:%d", *netid);
	return 0;
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add
}

void nm_network_policy_route_init()
{
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
    FRATTR_PRIORITY.rta_len = RTA_LENGTH(sizeof(uint32_t));
    FRATTR_PRIORITY.rta_type = FRA_PRIORITY;

    FRATTR_TABLE.rta_len = RTA_LENGTH(sizeof(uint32_t));
    FRATTR_TABLE.rta_type = FRA_TABLE;

    FRATTR_FWMARK.rta_len = RTA_LENGTH(sizeof(uint32_t));
    FRATTR_FWMARK.rta_type = FRA_FWMARK;

    FRATTR_FWMASK.rta_len = RTA_LENGTH(sizeof(uint32_t));
    FRATTR_FWMASK.rta_type = FRA_FWMASK;

    FRATTR_UID_START.rta_len = RTA_LENGTH(sizeof(uid_t));
    FRATTR_UID_START.rta_type = FRA_UID_START;

    FRATTR_UID_END.rta_len = RTA_LENGTH(sizeof(uid_t));
    FRATTR_UID_END.rta_type = FRA_UID_END;

    RTATTR_TABLE.rta_len = RTA_LENGTH(sizeof(uint32_t));
    RTATTR_TABLE.rta_type = RTA_TABLE;

    RTATTR_OIF.rta_len = RTA_LENGTH(sizeof(uint32_t));
    RTATTR_OIF.rta_type = RTA_OIF;
#endif //LYNQ_MAY_SUPPORT
return NULL;//LYNQ_MAY_SUPPORT add
}

const struct sockaddr_nl NETLINK_ADDRESS = {AF_NETLINK, 0, 0, 0};

int sendNetlinkRequest(uint16_t action, uint16_t flags, struct iovec* iov, int iovlen) {
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
    struct nlmsghdr nlmsg = {
        .nlmsg_type = action,
        .nlmsg_flags = flags,
    };
    iov[0].iov_base = &nlmsg;
    iov[0].iov_len = sizeof(nlmsg);
    for (int i = 0; i < iovlen; ++i) {
        nlmsg.nlmsg_len += iov[i].iov_len;
    }

    int ret;
    struct {
        struct nlmsghdr msg;
        struct nlmsgerr err;
    } response;

    int sock = socket(AF_NETLINK, SOCK_DGRAM | SOCK_CLOEXEC, NETLINK_ROUTE);
    if (sock == -1) {
		PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "socket error");
        ret=-1;
		goto fail;
    }

    if (connect(sock, (struct sockaddr *)&NETLINK_ADDRESS, sizeof(NETLINK_ADDRESS)) == -1) {
		PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "connect error");
        ret=-1;
		goto fail;
    }

    if (writev(sock, iov, iovlen) == -1) {
		PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "writev error");
        ret=-1;
		goto fail;
    }

    if ((ret = recv(sock, &response, sizeof(response), 0)) == -1) {
		PAL_DEBUG_PRINT(PAL_DEBUG_TRACE, "recv error");
        ret=-1;
		goto fail;
    }


    if (ret == sizeof(response)) {
        ret = response.err.error;  // Netlink errors are negative errno.
        if (ret) {
            PAL_DEBUG_PRINT(PAL_DEBUG_TRACE,"netlink response contains error (%s)", strerror(-ret));
        }
    } else {
        PAL_DEBUG_PRINT(PAL_DEBUG_TRACE,"bad netlink response message size (%d != %zu)", ret, sizeof(response));
        ret = -999;
    }
fail:
    if (sock != -1) {
        close(sock);
    }

    return ret;
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add
}


int nm_network_ipv6_policy_route_modify(uint16_t action, uint32_t table, const char* interface)
{
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
    uint8_t rawAddress[sizeof(struct in6_addr)];
    uint8_t rawNexthop[sizeof(struct in6_addr)];
    int rawLength = sizeof(rawAddress);

    if (!interface) return -1;

    memset(rawAddress, 0, rawLength);

    struct rtmsg route = {
        .rtm_protocol = RTPROT_STATIC,
        .rtm_type = RTN_UNICAST,
        .rtm_family = AF_INET6,
        .rtm_dst_len = 0,
        .rtm_scope = RT_SCOPE_UNIVERSE,
        .rtm_flags = 0,
    };

    struct rtattr rtaDst;;
    struct rtattr rtaGateway;

    rtaDst.rta_len = RTA_LENGTH(rawLength);
    rtaDst.rta_type = RTA_DST;

    rtaGateway.rta_len = RTA_LENGTH(rawLength);
    rtaGateway.rta_type = RTA_GATEWAY;

    uint32_t ifindex = if_nametoindex(interface);

    struct iovec iov[] = {
        { NULL,          0 },
        { &route,        sizeof(route) },
        { &RTATTR_TABLE, sizeof(RTATTR_TABLE) },
        { &table,        sizeof(table) },
#if 0
        { &rtaDst,       sizeof(rtaDst) },
        { rawAddress,    16 },
#else
        { &rtaDst,       0 },
        { rawAddress,    0 },
#endif
        { &RTATTR_OIF,   sizeof(RTATTR_OIF)},
        { &ifindex,      sizeof(ifindex)},
        { &rtaGateway,   0 },
        { rawNexthop,    0},
    };

    uint16_t flags = (action == RTM_NEWROUTE) ? NETLINK_CREATE_REQUEST_FLAGS : NETLINK_REQUEST_FLAGS;
    return sendNetlinkRequest(action, flags, iov, 10);
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add
}

int nm_network_ipv4_policy_route_modify(uint16_t action, uint32_t table, const char* interface)
{
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
    uint8_t rawAddress[sizeof(struct in_addr)];
    uint8_t rawNexthop[sizeof(struct in_addr)];
    int rawLength = sizeof(rawAddress);

    if (!interface) return -1;

    memset(rawAddress, 0, rawLength);

    struct rtmsg route = {
        .rtm_protocol = RTPROT_STATIC,
        .rtm_type = RTN_UNICAST,
        .rtm_family = AF_INET,
        .rtm_dst_len = 0,
        .rtm_scope = RT_SCOPE_UNIVERSE,
        .rtm_flags = 0,
    };

    struct rtattr rtaDst;;
    struct rtattr rtaGateway;

    rtaDst.rta_len = RTA_LENGTH(rawLength);
    rtaDst.rta_type = RTA_DST;

    rtaGateway.rta_len = RTA_LENGTH(rawLength);
    rtaGateway.rta_type = RTA_GATEWAY;

    uint32_t ifindex = if_nametoindex(interface);

    struct iovec iov[] = {
        { NULL,          0 },
        { &route,        sizeof(route) },
        { &RTATTR_TABLE, sizeof(RTATTR_TABLE) },
        { &table,        sizeof(table) },
#if 0
        { &rtaDst,       sizeof(rtaDst) },
        { rawAddress,    16 },
#else
        { &rtaDst,       0 },
        { rawAddress,    0 },
#endif
        { &RTATTR_OIF,   sizeof(RTATTR_OIF)},
        { &ifindex,      sizeof(ifindex)},
        { &rtaGateway,   0 },
        { rawNexthop,    0},
    };

    uint16_t flags = (action == RTM_NEWROUTE) ? NETLINK_CREATE_REQUEST_FLAGS : NETLINK_REQUEST_FLAGS;
    return sendNetlinkRequest(action, flags, iov, 10);
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add
}

int nm_network_ipv6_policy_rule_modify(uint16_t action, uint32_t priority, uint32_t table, uint32_t fwmark, uint32_t mask, const char* iif, const char* oif)
{
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
    char iif_name[16], oif_name[16];
    int iif_name_len, iif_name_padding;
    int oif_name_len, oif_name_padding;
    struct rtattr FRATTR_IIFNAME;
    struct rtattr FRATTR_OIFNAME;
    uid_t uidStart = 0, uidEnd = 0;

    if (iif) {
        snprintf(iif_name, sizeof(iif_name), "%s", iif);
        iif_name_len = strlen(iif_name) + 1;
        iif_name_padding = RTA_SPACE(iif_name_len) - RTA_LENGTH(iif_name_len);
    } else {
        iif_name_len = 0;
        iif_name_padding = 0;
    }
    FRATTR_IIFNAME.rta_len =  RTA_LENGTH(iif_name_len);
    FRATTR_IIFNAME.rta_type = FRA_IIFNAME;

    if (oif) {
        snprintf(oif_name, sizeof(oif_name), "%s", oif);
        oif_name_len = strlen(oif_name) + 1;
        oif_name_padding = RTA_SPACE(oif_name_len) - RTA_LENGTH(oif_name_len);
    } else {
        oif_name_len = 0;
        oif_name_padding = 0;
    }
    FRATTR_OIFNAME.rta_len =  RTA_LENGTH(oif_name_len);
    FRATTR_OIFNAME.rta_type = FRA_OIFNAME;

    struct fib_rule_hdr rule = {
        .action = FR_ACT_TO_TBL,
    };

    struct iovec iov[] = {
        { NULL,              0 },
        { &rule,             sizeof(rule) },
        { &FRATTR_PRIORITY,  sizeof(FRATTR_PRIORITY) },
        { &priority,         sizeof(priority) },
        { &FRATTR_TABLE,     table != RT_TABLE_UNSPEC ? sizeof(FRATTR_TABLE) : 0 },
        { &table,            table != RT_TABLE_UNSPEC ? sizeof(table) : 0 },
        { &FRATTR_FWMARK,    mask ? sizeof(FRATTR_FWMARK) : 0 },
        { &fwmark,           mask ? sizeof(fwmark) : 0 },
        { &FRATTR_FWMASK,    mask ? sizeof(FRATTR_FWMASK) : 0 },
        { &mask,             mask ? sizeof(mask) : 0 },
        { &FRATTR_UID_START, 0 },
        { &uidStart,         0 },
        { &FRATTR_UID_END,   0 },
        { &uidEnd,           0 },
        { &FRATTR_IIFNAME,   iif != NULL ? sizeof(FRATTR_IIFNAME) : 0 },
        { iif_name,          iif_name_len },
        { PADDING_BUFFER,    iif_name_padding },
        { &FRATTR_OIFNAME,   oif != NULL ? sizeof(FRATTR_OIFNAME) : 0 },
        { oif_name,          oif_name_len },
        { PADDING_BUFFER,    oif_name_padding },
    };

    uint16_t flags = (action == RTM_NEWROUTE) ? NETLINK_CREATE_REQUEST_FLAGS : NETLINK_REQUEST_FLAGS;

    rule.family = AF_INET6;
    return sendNetlinkRequest(action, flags, iov, 20);
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add
}

int nm_network_ipv4_policy_rule_modify(uint16_t action, uint32_t priority, uint32_t table, uint32_t fwmark, uint32_t mask, const char* iif, const char* oif)
{
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
    char iif_name[16], oif_name[16];
    int iif_name_len, iif_name_padding;
    int oif_name_len, oif_name_padding;
    struct rtattr FRATTR_IIFNAME;
    struct rtattr FRATTR_OIFNAME;
    uid_t uidStart = 0, uidEnd = 0;

    if (iif) {
        snprintf(iif_name, sizeof(iif_name), "%s", iif);
        iif_name_len = strlen(iif_name) + 1;
        iif_name_padding = RTA_SPACE(iif_name_len) - RTA_LENGTH(iif_name_len);
    } else {
        iif_name_len = 0;
        iif_name_padding = 0;
    }
    FRATTR_IIFNAME.rta_len =  RTA_LENGTH(iif_name_len);
    FRATTR_IIFNAME.rta_type = FRA_IIFNAME;

    if (oif) {
        snprintf(oif_name, sizeof(oif_name), "%s", oif);
        oif_name_len = strlen(oif_name) + 1;
        oif_name_padding = RTA_SPACE(oif_name_len) - RTA_LENGTH(oif_name_len);
    } else {
        oif_name_len = 0;
        oif_name_padding = 0;
    }
    FRATTR_OIFNAME.rta_len =  RTA_LENGTH(oif_name_len);
    FRATTR_OIFNAME.rta_type = FRA_OIFNAME;

    struct fib_rule_hdr rule = {
        .action = FR_ACT_TO_TBL,
    };

    struct iovec iov[] = {
        { NULL,              0 },
        { &rule,             sizeof(rule) },
        { &FRATTR_PRIORITY,  sizeof(FRATTR_PRIORITY) },
        { &priority,         sizeof(priority) },
        { &FRATTR_TABLE,     table != RT_TABLE_UNSPEC ? sizeof(FRATTR_TABLE) : 0 },
        { &table,            table != RT_TABLE_UNSPEC ? sizeof(table) : 0 },
        { &FRATTR_FWMARK,    mask ? sizeof(FRATTR_FWMARK) : 0 },
        { &fwmark,           mask ? sizeof(fwmark) : 0 },
        { &FRATTR_FWMASK,    mask ? sizeof(FRATTR_FWMASK) : 0 },
        { &mask,             mask ? sizeof(mask) : 0 },
        { &FRATTR_UID_START, 0 },
        { &uidStart,         0 },
        { &FRATTR_UID_END,   0 },
        { &uidEnd,           0 },
        { &FRATTR_IIFNAME,   iif != NULL ? sizeof(FRATTR_IIFNAME) : 0 },
        { iif_name,          iif_name_len },
        { PADDING_BUFFER,    iif_name_padding },
        { &FRATTR_OIFNAME,   oif != NULL ? sizeof(FRATTR_OIFNAME) : 0 },
        { oif_name,          oif_name_len },
        { PADDING_BUFFER,    oif_name_padding },
    };

    uint16_t flags = (action == RTM_NEWROUTE) ? NETLINK_CREATE_REQUEST_FLAGS : NETLINK_REQUEST_FLAGS;

    rule.family = AF_INET;
    return sendNetlinkRequest(action, flags, iov, 20);
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add
}

static const uint64_t UNKNOWN = -1;

int parseIfaceStats(const char* iface, struct Stats* stats) {
printf("this pal_nm api fun:%s,line:%s",__FUNCTION__,__LINE__);
#ifdef LYNQ_MAY_SUPPORT
    FILE *fp = fopen(QTAGUID_IFACE_STATS, "r");
    if (fp == NULL) {
        return -1;
    }

    char buffer[384];
    char cur_iface[32];
    bool foundTcp = false;
    uint64_t rxBytes, rxPackets, txBytes, txPackets, tcpRxPackets, tcpTxPackets;

    while (fgets(buffer, sizeof(buffer), fp) != NULL) {
        int matched = sscanf(buffer, "%31s %" SCNu64 " %" SCNu64 " %" SCNu64
                " %" SCNu64 " " "%*u %" SCNu64 " %*u %*u %*u %*u "
                "%*u %" SCNu64 " %*u %*u %*u %*u", cur_iface, &rxBytes,
                &rxPackets, &txBytes, &txPackets, &tcpRxPackets, &tcpTxPackets);
        if (matched >= 5) {
            if (matched == 7) {
                foundTcp = true;
            }
            if (!iface || !strcmp(iface, cur_iface)) {
                stats->rxBytes += rxBytes;
                stats->rxPackets += rxPackets;
                stats->txBytes += txBytes;
                stats->txPackets += txPackets;
                if (matched == 7) {
                    stats->tcpRxPackets += tcpRxPackets;
                    stats->tcpTxPackets += tcpTxPackets;
                }
            }
        }
    }

    if (!foundTcp) {
        stats->tcpRxPackets = UNKNOWN;
        stats->tcpTxPackets = UNKNOWN;
    }

    if (fclose(fp) != 0) {
        return -1;
    }
    return 0;
#endif //LYNQ_MAY_SUPPORT
return 0;//LYNQ_MAY_SUPPORT add
}



