#include <stdio.h>
#include <stdint.h>
#include <sys/types.h>
#include <arpa/inet.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <log/log.h>
#include <libdata/lynq_data.h>
#include <liblog/lynq_deflog.h>
#include <pthread.h>
#include <sqlite3.h>
#include "lynq-qser-data.h"

#define USER_LOG_TAG "LYNQ_QSER_DATA"
#define APN_DB_PATH "/mnt/userdata/apn.db"
#define APN_DB_READED 0
#define APN_DB_ERR 1
#define APN_DB_CREATE 2
#define APN_TABLE_CREATE 3
#define APN_TABLE_NEED_CREATE 4
#define RESULT_OK (0)
#define RESULT_ERROR (-1)

typedef enum{
    LYNQ_E_NO_INIT=9002,
    LYNQ_E_APN_DB_FAIL=9003
}LYNQ_E;

static int g_lynq_qser_data_init_flag = 0;
static pthread_t s_cb_tid = -1;
static int s_qser_data_cb_thread_status = 0;
static pthread_mutex_t s_qser_data_cb_mutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t s_qser_data_database_mutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t s_qser_data_cb_cond = PTHREAD_COND_INITIALIZER;


static qser_data_call_evt_cb_t s_data_call_cb = NULL;
static sqlite3* apnDb;
qser_apn_add_s default_apn = {QSER_APN_PDP_TYPE_IPV4V6, QSER_APN_AUTH_PROTO_DEFAULT, "default", "NULL", "NULL", "default"};
int apn_indexs[LYNQ_APN_CHANNEL_MAX]={0};
int next_index = 0;
int apn_count = 0;
const int apndb_offset = 683;

static int init_apn_db();
static int create_apn_db(char *path);
static int create_apn_table(char*path);
static int has_apn_db_created(char *path,sqlite3* apnDb_l);
static int apn_db_handle_clean();

static int lynq_apn_callback(void *data, int argc, char **argv, char **azColName);
static int lynq_check_index(unsigned char index);
static int apn_db_handle_get_profile_cb(void *data, int argc, char **argv, char **azColName);
static int apn_db_handle_get_profile(int handle, unsigned char *profile_idx);
static int apn_db_handle_set(unsigned char profile_idx, int handle);
static int apn_db_handle_get_cb(void *data, int argc, char **argv, char **azColName);
static int apn_db_handle_get(unsigned char profile_idx, int *handle);

static int find_unuse_apn_index(char *path);
static int apn_db_add(int profile_id, qser_apn_add_s *apn);
static int apn_db_delete(unsigned char profile_idx);
static int apn_db_modify(qser_apn_info_s *apn);
static int lynq_apn_db_query_cb(void *data, int argc, char **argv, char **azColName);
static int apn_db_query(unsigned char profile_idx,qser_apn_info_s *apn);
static int apn_db_query_list_cb(void *data, int argc, char **argv, char **azColName);
static int apn_db_query_list(qser_apn_info_list_s *apn_list);
#ifdef MOBILETEK_TARGET_PLATFORM_T106
int qser_set_initial_apnname(char *apnname);
int qser_get_initial_apnname(char apnname[QSER_APN_NAME_SIZE]);
#endif

void lynq_ipv4_aton_urc(lynq_data_call_response_v11_t *libdata,qser_data_call_state_s *data_res)
{
#ifdef MOBILETEK_TARGET_PLATFORM_T106
    char *tmp_char = NULL;
    char *addresses = NULL;
    char *dnses = NULL;
    const char addresses_separator[2] = "/";
    const char dnses_separator[2] = " ";
    
    addresses = libdata->addresses;
    dnses = libdata->dnses;
    //get addresses
    tmp_char = strsep(&addresses, addresses_separator);
    if(tmp_char != NULL)
    {
        LYDBGLOG("ipv4 addresses = %s", tmp_char);
        inet_aton(tmp_char,&(data_res->v4.ip));
    }

    //get dnses
    tmp_char = strsep(&dnses, dnses_separator);
    if(tmp_char != NULL)
    {
        LYDBGLOG("ipv4 pri_dns = %s", tmp_char);
        inet_aton(tmp_char,&(data_res->v4.pri_dns));
    }
    tmp_char = strsep(&dnses, dnses_separator);
    if(tmp_char != NULL)
    {
        LYDBGLOG("ipv4 sec_dns = %s", tmp_char);
        inet_aton(tmp_char, &(data_res->v4.sec_dns));
    }
    //get gateway
    inet_pton(AF_INET6,libdata->gateways,&(data_res->v6.gateway));

#else
    inet_aton(libdata->addresses,&(data_res->v4.ip));
    inet_aton(libdata->gateways,&(data_res->v4.gateway));
    inet_aton(libdata->dnses,&(data_res->v4.pri_dns));
    inet_aton(libdata->dnses,&(data_res->v4.sec_dns));
#endif
    return ;
}

void lynq_ipv6_inet_pton_urc(lynq_data_call_response_v11_t *libdata,qser_data_call_state_s *data_res)
{
#ifdef MOBILETEK_TARGET_PLATFORM_T106
    char *tmp_char = NULL;
    char *addresses = NULL;
    char *dnses = NULL;
    const char addresses_separator[2] = "/";
    const char dnses_separator[2] = " ";

    addresses = libdata->addresses;
    dnses = libdata->dnses;
    //get addresses
    tmp_char = strsep(&addresses, addresses_separator);
    if(tmp_char != NULL)
    {
        LYDBGLOG("ipv6 addresses = %s", tmp_char);
        inet_pton(AF_INET6, tmp_char, &(data_res->v6.ip));
    }
    //get dnses
    tmp_char = strsep(&dnses, dnses_separator);
    if(tmp_char != NULL)
    {
        LYDBGLOG("ipv6 pri_dns = %s", tmp_char);
        inet_pton(AF_INET6, tmp_char, &(data_res->v6.pri_dns));
    }
    tmp_char = strsep(&dnses, dnses_separator);
    if(tmp_char != NULL)
    {
        LYDBGLOG("ipv6 sec_dns = %s", tmp_char);
        inet_pton(AF_INET6, tmp_char, &(data_res->v6.sec_dns));
    }
    inet_pton(AF_INET6,libdata->gateways,&(data_res->v6.gateway));
#else
    inet_pton(AF_INET6,libdata->addresses,&(data_res->v6.ip));
    inet_pton(AF_INET6,libdata->gateways,&(data_res->v6.gateway));
    inet_pton(AF_INET6,libdata->dnses,&(data_res->v6.pri_dns));
    inet_pton(AF_INET6,libdata->dnses,&(data_res->v6.sec_dns));
#endif

    return ;
}

void lynq_ipv4v6_inet_pton_urc(lynq_data_call_response_v11_t *libdata,qser_data_call_state_s *data_res)
{
    char *tmp_char = NULL;
    char *addresses = NULL;
    char *dnses = NULL;
    const char addresses_separator[2] = "/";
    const char dnses_separator[2] = " ";

    char buf_ip[64] = {0};
    char buf_gateway[64] = {0};
    char buf_pri_dns[64] = {0};
    char buf_sec_dns[64] = {0};

    addresses = libdata->addresses;
    dnses = libdata->dnses;
    //get addresses
    tmp_char = strsep(&addresses, addresses_separator);
    if(tmp_char != NULL)
    {
        LYDBGLOG("ipv4 addresses = %s", tmp_char);
        inet_aton(tmp_char,&(data_res->v4.ip));
    }
    tmp_char = strsep(&addresses, addresses_separator);
    if(tmp_char != NULL)
    {
        LYDBGLOG("ipv6 addresses = %s", tmp_char);
        inet_pton(AF_INET6, tmp_char, &(data_res->v6.ip));
    }
    //get dnses
    tmp_char = strsep(&dnses, dnses_separator);
    if(tmp_char != NULL)
    {
        LYDBGLOG("ipv4 pri_dns = %s", tmp_char);
        inet_aton(tmp_char,&(data_res->v4.pri_dns));
    }
    tmp_char = strsep(&dnses, dnses_separator);
    if(tmp_char != NULL)
    {
        LYDBGLOG("ipv4 sec_dns = %s", tmp_char);
        inet_aton(tmp_char, &(data_res->v4.sec_dns));
    }
    tmp_char = strsep(&dnses, dnses_separator);
    if(tmp_char != NULL)
    {
        LYDBGLOG("ipv6 pri_dns = %s", tmp_char);
        inet_pton(AF_INET6, tmp_char, &(data_res->v6.pri_dns));
    }
    tmp_char = strsep(&dnses, dnses_separator);
    if(tmp_char != NULL)
    {
        LYDBGLOG("ipv6 sec_dns = %s", tmp_char);
        inet_pton(AF_INET6, tmp_char, &(data_res->v6.sec_dns));
    }
    //get gateway
    inet_aton(libdata->gateways,&(data_res->v4.gateway));
    inet_pton(AF_INET6,libdata->gateways,&(data_res->v6.gateway));

    LYDBGLOG("v4.ip=%s", inet_ntoa(data_res->v4.ip));
    LYDBGLOG("v4.pri_dns=%s", inet_ntoa(data_res->v4.pri_dns));
    LYDBGLOG("v4.sec_dns=%s", inet_ntoa(data_res->v4.sec_dns));

    inet_ntop(AF_INET6, &(data_res->v6.ip), buf_ip, sizeof(buf_ip));
    inet_ntop(AF_INET6, &(data_res->v6.gateway), buf_gateway, sizeof(buf_gateway));
    inet_ntop(AF_INET6, &(data_res->v6.pri_dns), buf_pri_dns, sizeof(buf_pri_dns));
    inet_ntop(AF_INET6, &(data_res->v6.sec_dns), buf_sec_dns, sizeof(buf_sec_dns));
    LYDBGLOG("v6.ip=%s, v6.gateway=%s, v6.pri_dns=%s, v6.sec_dns=%s\n"
        , buf_ip, buf_gateway, buf_pri_dns, buf_sec_dns);
    return ;
}

void lynq_ipv4_aton_getinfo(lynq_data_call_response_v11_t *libdata,qser_data_call_info_s *data_res)
{
#ifdef MOBILETEK_TARGET_PLATFORM_T106
    char *tmp_char = NULL;
    char *addresses = NULL;
    char *dnses = NULL;
    const char addresses_separator[2] = "/";
    const char dnses_separator[2] = " ";
    
    addresses = libdata->addresses;
    dnses = libdata->dnses;
    //get addresses
    tmp_char = strsep(&addresses, addresses_separator);
    if(tmp_char != NULL)
    {
        LYDBGLOG("ipv4 addresses = %s", tmp_char);
        inet_aton(tmp_char,&(data_res->v4.addr.ip));
    }
    //get dnses
    tmp_char = strsep(&dnses, dnses_separator);
    if(tmp_char != NULL)
    {
        LYDBGLOG("ipv4 pri_dns = %s", tmp_char);
        inet_aton(tmp_char,&(data_res->v4.addr.pri_dns));
    }
    tmp_char = strsep(&dnses, dnses_separator);
    if(tmp_char != NULL)
    {
        LYDBGLOG("ipv4 sec_dns = %s", tmp_char);
        inet_aton(tmp_char, &(data_res->v4.addr.sec_dns));
    }
    //get gateway
    LYDBGLOG("ipv4 gateways = %s", libdata->gateways);
    inet_aton(libdata->gateways,&(data_res->v4.addr.gateway));

    LYDBGLOG("v4.ip=%s", inet_ntoa(data_res->v4.addr.ip));
    LYDBGLOG("v4.pri_dns=%s", inet_ntoa(data_res->v4.addr.pri_dns));
    LYDBGLOG("v4.sec_dns=%s", inet_ntoa(data_res->v4.addr.sec_dns));
#else
    inet_aton(libdata->addresses,&(data_res->v4.addr.ip));
    inet_aton(libdata->gateways,&(data_res->v4.addr.gateway));
    inet_aton(libdata->dnses,&(data_res->v4.addr.pri_dns));
    inet_aton(libdata->dnses,&(data_res->v4.addr.sec_dns));
#endif
    data_res->v4.stats.pkts_tx = 0;
    data_res->v4.stats.pkts_rx = 0;
    data_res->v4.stats.bytes_tx = 0;
    data_res->v4.stats.bytes_rx = 0;
    data_res->v4.stats.pkts_dropped_tx = 0;
    data_res->v4.stats.pkts_dropped_rx = 0;
    return ;
}

void lynq_ipv6_inet_pton_getinfo(lynq_data_call_response_v11_t *libdata,qser_data_call_info_s *data_res)
{
#ifdef MOBILETEK_TARGET_PLATFORM_T106
    char *tmp_char = NULL;
    char *addresses = NULL;
    char *dnses = NULL;
    const char addresses_separator[2] = "/";
    const char dnses_separator[2] = " ";

    addresses = libdata->addresses;
    dnses = libdata->dnses;
    //get addresses
    tmp_char = strsep(&addresses, addresses_separator);
    if(tmp_char != NULL)
    {
        LYDBGLOG("ipv6 addresses = %s", tmp_char);
        inet_pton(AF_INET6, tmp_char, &(data_res->v6.addr.ip));
    }
    //get dnses
    tmp_char = strsep(&dnses, dnses_separator);
    if(tmp_char != NULL)
    {
        LYDBGLOG("ipv6 pri_dns = %s", tmp_char);
        inet_pton(AF_INET6, tmp_char, &(data_res->v6.addr.pri_dns));
    }
    tmp_char = strsep(&dnses, dnses_separator);
    if(tmp_char != NULL)
    {
        LYDBGLOG("ipv6 sec_dns = %s", tmp_char);
        inet_pton(AF_INET6, tmp_char, &(data_res->v6.addr.sec_dns));
    }
    inet_pton(AF_INET6,libdata->gateways,&(data_res->v6.addr.gateway));
#else
    inet_pton(AF_INET6,libdata->addresses,&(data_res->v6.addr.ip));
    inet_pton(AF_INET6,libdata->gateways,&(data_res->v6.addr.gateway));
    inet_pton(AF_INET6,libdata->dnses,&(data_res->v6.addr.pri_dns));
    inet_pton(AF_INET6,libdata->dnses,&(data_res->v6.addr.sec_dns));
#endif
    data_res->v6.stats.pkts_tx = 0;
    data_res->v6.stats.pkts_rx = 0;
    data_res->v6.stats.bytes_tx = 0;
    data_res->v6.stats.bytes_rx = 0;
    data_res->v6.stats.pkts_dropped_tx = 0;
    data_res->v6.stats.pkts_dropped_rx = 0;
    return ;
}
void lynq_ipv4v6_inet_pton_getinfo(lynq_data_call_response_v11_t *libdata,qser_data_call_info_s *data_res)
{
    char *tmp_char = NULL;
    char *addresses = NULL;
    char *dnses = NULL;
    const char addresses_separator[2] = "/";
    const char dnses_separator[2] = " ";

    char buf_ip[64] = {0};
    char buf_gateway[64] = {0};
    char buf_pri_dns[64] = {0};
    char buf_sec_dns[64] = {0};

    addresses = libdata->addresses;
    dnses = libdata->dnses;
    //get addresses
    tmp_char = strsep(&addresses, addresses_separator);
    if(tmp_char != NULL)
    {
        LYDBGLOG("ipv4 addresses = %s", tmp_char);
        inet_aton(tmp_char,&(data_res->v4.addr.ip));
    }
    tmp_char = strsep(&addresses, addresses_separator);
    if(tmp_char != NULL)
    {
        LYDBGLOG("ipv6 addresses = %s", tmp_char);
        inet_pton(AF_INET6, tmp_char, &(data_res->v6.addr.ip));
    }
    //get dnses
    tmp_char = strsep(&dnses, dnses_separator);
    if(tmp_char != NULL)
    {
        LYDBGLOG("ipv4 pri_dns = %s", tmp_char);
        inet_aton(tmp_char,&(data_res->v4.addr.pri_dns));
    }
    tmp_char = strsep(&dnses, dnses_separator);
    if(tmp_char != NULL)
    {
        LYDBGLOG("ipv4 sec_dns = %s", tmp_char);
        inet_aton(tmp_char, &(data_res->v4.addr.sec_dns));
    }
    tmp_char = strsep(&dnses, dnses_separator);
    if(tmp_char != NULL)
    {
        LYDBGLOG("ipv6 pri_dns = %s", tmp_char);
        inet_pton(AF_INET6, tmp_char, &(data_res->v6.addr.pri_dns));
    }
    tmp_char = strsep(&dnses, dnses_separator);
    if(tmp_char != NULL)
    {
        LYDBGLOG("ipv6 sec_dns = %s", tmp_char);
        inet_pton(AF_INET6, tmp_char, &(data_res->v6.addr.sec_dns));
    }
    //get gateway
    inet_aton(libdata->gateways,&(data_res->v4.addr.gateway));
    inet_pton(AF_INET6,libdata->gateways,&(data_res->v6.addr.gateway));

    LYDBGLOG("v4.ip=%s", inet_ntoa(data_res->v4.addr.ip));
    LYDBGLOG("ipv4 gateways = %s", inet_ntoa(data_res->v4.addr.gateway));
    LYDBGLOG("v4.pri_dns=%s", inet_ntoa(data_res->v4.addr.pri_dns));
    LYDBGLOG("v4.sec_dns=%s", inet_ntoa(data_res->v4.addr.sec_dns));

    inet_ntop(AF_INET6, &(data_res->v6.addr.ip), buf_ip, sizeof(buf_ip));
    inet_ntop(AF_INET6, &(data_res->v6.addr.gateway), buf_gateway, sizeof(buf_gateway));
    inet_ntop(AF_INET6, &(data_res->v6.addr.pri_dns), buf_pri_dns, sizeof(buf_pri_dns));
    inet_ntop(AF_INET6, &(data_res->v6.addr.sec_dns), buf_sec_dns, sizeof(buf_sec_dns));
    LYDBGLOG("v6.ip=%s, v6.gateway=%s, v6.pri_dns=%s, v6.sec_dns=%s\n"
        , buf_ip, buf_gateway, buf_pri_dns, buf_sec_dns);
    data_res->v4.stats.pkts_tx = 0;
    data_res->v4.stats.pkts_rx = 0;
    data_res->v4.stats.bytes_tx = 0;
    data_res->v4.stats.bytes_rx = 0;
    data_res->v4.stats.pkts_dropped_tx = 0;
    data_res->v4.stats.pkts_dropped_rx = 0;

    data_res->v6.stats.pkts_tx = 0;
    data_res->v6.stats.pkts_rx = 0;
    data_res->v6.stats.bytes_tx = 0;
    data_res->v6.stats.bytes_rx = 0;
    data_res->v6.stats.pkts_dropped_tx = 0;
    data_res->v6.stats.pkts_dropped_rx = 0;
    return ;
}

void datacall_ipv4_status_judge(int state,qser_data_call_info_s *data_res)
{
    if(state != 0)
    {
        data_res->v4.state = QSER_DATA_CALL_CONNECTED;
        data_res->v4.reconnect = 1;
    }
    else
    {
        data_res->v4.state = QSER_DATA_CALL_DISCONNECTED;
        data_res->v4.reconnect = 0;
    }
    return ;
}

void datacall_ipv6_status_judge(int state,qser_data_call_info_s *data_res)
{
    if(state != 0)
    {
        data_res->v6.state = QSER_DATA_CALL_CONNECTED;
        data_res->v6.reconnect = 1;
    }
    else
    {
        data_res->v6.state = QSER_DATA_CALL_DISCONNECTED;
        data_res->v6.reconnect = 0;
    }
    return ;
}

// ****************apn DB start******************
static int apn_db_handle_get_profile_cb(void *data, int argc, char **argv, char **azColName)
{
    if(data==NULL)
    {
        return 1;
    }
    int *temp = (int *)data;
    for(int i=0; i<argc; i++){
       LYDBGLOG("[%s]%s = %s", __FUNCTION__,azColName[i], argv[i] ? argv[i] : "NULL");
    }
    *temp = atoi(argv[0]);
    return 0;
}
static int apn_db_handle_get_profile(int handle, unsigned char *profile_idx)
{
    char *zErrMsg = 0;
    int rc;
    int idx = 0;
    char sql[128]={0};

    /* Open database */
    pthread_mutex_lock(&s_qser_data_database_mutex);
    rc = sqlite3_open(APN_DB_PATH, &apnDb);
    if( rc )
    {
        LYERRLOG("[%s] Can't open database: %s\n", __FUNCTION__,sqlite3_errmsg(apnDb));
        pthread_mutex_unlock(&s_qser_data_database_mutex);
        return LYNQ_E_APN_DB_FAIL;
    }

    LYDBGLOG("[%s] Opened database successfully\n",__FUNCTION__);
    /* Create SQL statement */
    sprintf(sql,"SELECT ID from LYNQAPN WHERE HANDLE=%d", handle);
    LYINFLOG("sql: %s\n", sql);
    /* Execute SQL statement */
    rc = sqlite3_exec(apnDb, sql, apn_db_handle_get_profile_cb, &idx, &zErrMsg);
    if( rc != SQLITE_OK )
    {
        LYERRLOG("SQL error: %s\n", zErrMsg);
        sqlite3_free(zErrMsg);
        sqlite3_close(apnDb);
        pthread_mutex_unlock(&s_qser_data_database_mutex);
        return LYNQ_E_APN_DB_FAIL;
    };
    *profile_idx = (unsigned char)idx;
    LYINFLOG("get idx %d successfully\n", *profile_idx);
    sqlite3_close(apnDb);
    pthread_mutex_unlock(&s_qser_data_database_mutex);
    return 0;
}
static int lynq_apn_callback(void *data, int argc, char **argv, char **azColName){
    int i;

    apn_indexs[atoi(argv[0])] = atoi(argv[0]);
    LYDBGLOG("apn_indexs[%d] is %d", atoi(argv[0]), apn_indexs[atoi(argv[0])]);

    for(i=0; i<argc; i++){
       LYDBGLOG("%s = %s", azColName[i], argv[i] ? argv[i] : "NULL");
    }
    return 0;
}
/**
 * @brief Check whether the ID exists in the database
 * 
 * @param index 
 * @return int 
 * find:turn 1 
 * not find:turn 0
 */
static int lynq_check_index(unsigned char index)
{
    memset(apn_indexs,0,sizeof(apn_indexs));
    next_index = 0;
    char *zErrMsg = 0;
    int rc;
    char sql[128]={0};
    /* Open database */
    pthread_mutex_lock(&s_qser_data_database_mutex);
    rc = sqlite3_open(APN_DB_PATH, &apnDb);
    if( rc )
    {
        LYERRLOG("[%s] Can't open database: %s\n", __FUNCTION__,sqlite3_errmsg(apnDb));
        pthread_mutex_unlock(&s_qser_data_database_mutex);
        return LYNQ_E_APN_DB_FAIL;
    }
    sprintf(sql,"%s", "SELECT * from LYNQAPN");
    rc = sqlite3_exec(apnDb, sql, lynq_apn_callback, NULL, &zErrMsg);
    if( rc != SQLITE_OK )
    {
        LYERRLOG("SQL error: %s\n", zErrMsg);
        sqlite3_free(zErrMsg);
        sqlite3_close(apnDb);
        pthread_mutex_unlock(&s_qser_data_database_mutex);
        return LYNQ_E_APN_DB_FAIL;
    }
    //for(int cnt = 0; cnt < next_index;cnt++)
    for(int cnt = 0; cnt <= QSER_APN_MAX_LIST; cnt++)
    {
        if(index == apn_indexs[cnt])
        {
            LYDBGLOG("index = apn_index\n");
            sqlite3_close(apnDb);
            pthread_mutex_unlock(&s_qser_data_database_mutex);
            return 1;
        }
    }
    LYINFLOG("cant find the index %d\n", index);
    sqlite3_close(apnDb);
    pthread_mutex_unlock(&s_qser_data_database_mutex);

    return 0;
}

static int apn_db_handle_set(unsigned char profile_idx, int handle)
{
    char *zErrMsg = 0;
    int rc;
    char sql[128]={0};
    if(!lynq_check_index(profile_idx))
    {
        return LYNQ_E_APN_DB_FAIL;
    }
    /* Open database */
    pthread_mutex_lock(&s_qser_data_database_mutex);
    rc = sqlite3_open(APN_DB_PATH, &apnDb);
    if( rc )
    {
        LYERRLOG("[%s] Can't open database: %s\n", __FUNCTION__,sqlite3_errmsg(apnDb));
        pthread_mutex_unlock(&s_qser_data_database_mutex);
        return LYNQ_E_APN_DB_FAIL;
    }

    LYDBGLOG("[%s] Opened database successfully\n",__FUNCTION__);
    /* Create SQL statement */
    //sql = "SELECT * from LYNQAPN";
    sprintf(sql,"UPDATE LYNQAPN SET HANDLE=%d WHERE ID=%d", handle, (int)profile_idx);
    LYINFLOG("sql: %s\n", sql);
    /* Execute SQL statement */
    rc = sqlite3_exec(apnDb, sql, NULL, NULL, &zErrMsg);
    if( rc != SQLITE_OK )
    {
        LYERRLOG("SQL error: %s\n", zErrMsg);
        sqlite3_free(zErrMsg);
        sqlite3_close(apnDb);
        pthread_mutex_unlock(&s_qser_data_database_mutex);
        return LYNQ_E_APN_DB_FAIL;
    }
    LYINFLOG("update apn %d successfully\n", profile_idx);
    sqlite3_close(apnDb);
    pthread_mutex_unlock(&s_qser_data_database_mutex);
    return 0;
}
static int apn_db_handle_get_cb(void *data, int argc, char **argv, char **azColName)
{
    if(data==NULL)
    {
        return 1;
    }
    int *temp = (int *)data;
    for(int i=0; i<argc; i++){
       LYDBGLOG("[%s]%s = %s", __FUNCTION__,azColName[i], argv[i] ? argv[i] : "NULL");
    }
    *temp = atoi(argv[0]);
    LYINFLOG("The handle is %d\n", *temp);
    return 0;
}
static int apn_db_handle_get(unsigned char profile_idx, int *handle)
{
    char *zErrMsg = 0;
    int rc;
    char sql[128]={0};

    if(!lynq_check_index(profile_idx))
    {
        return LYNQ_E_APN_DB_FAIL;
    }
    /* Open database */
    pthread_mutex_lock(&s_qser_data_database_mutex);
    rc = sqlite3_open(APN_DB_PATH, &apnDb);
    if( rc )
    {
        LYERRLOG("[%s] Can't open database: %s\n", __FUNCTION__,sqlite3_errmsg(apnDb));
        pthread_mutex_unlock(&s_qser_data_database_mutex);
        return LYNQ_E_APN_DB_FAIL;
    }

    LYDBGLOG("[%s] Opened database successfully\n",__FUNCTION__);
    /* Create SQL statement */
    sprintf(sql,"SELECT HANDLE from LYNQAPN WHERE ID=%d", (int)profile_idx);
    LYINFLOG("sql: %s\n", sql);
    /* Execute SQL statement */
    rc = sqlite3_exec(apnDb, sql, apn_db_handle_get_cb, handle, &zErrMsg);
    if( rc != SQLITE_OK )
    {
        LYERRLOG("SQL error: %s\n", zErrMsg);
        sqlite3_free(zErrMsg);
        sqlite3_close(apnDb);
        pthread_mutex_unlock(&s_qser_data_database_mutex);
        return LYNQ_E_APN_DB_FAIL;
    }
    LYINFLOG("get idx %d successfully\n", *handle);
    sqlite3_close(apnDb);
    pthread_mutex_unlock(&s_qser_data_database_mutex);
    return 0;
}

static int has_apn_db_created(char *path,sqlite3* apnDb_l)
{
    if (path == NULL || strlen(path) == 0)
    {
        LYERRLOG("initApnDb error with null path!");
        return APN_DB_ERR;
    }
    int rc = sqlite3_open_v2(path, &apnDb_l, SQLITE_OPEN_READWRITE,NULL);
    if (rc == SQLITE_OK) 
    {
        LYINFLOG("check init success!");
        sqlite3_close_v2(apnDb_l);
        return APN_DB_READED;
    } 
    else 
    {
        LYERRLOG("db has not create %s!", sqlite3_errmsg(apnDb_l));
        sqlite3_close_v2(apnDb_l);
        return APN_DB_ERR;
    }
}
static int create_apn_db(char *path)
{
    if (path == NULL || strlen(path) == 0) {
        LYERRLOG("initApnDb error with null path!");
        return APN_DB_ERR;
    }
    if(APN_DB_READED==has_apn_db_created(path,apnDb))
    {
        return APN_DB_READED;
    }
    int rc = sqlite3_open(path, &apnDb);
    if (rc != SQLITE_OK) {
        LYERRLOG("initDb failed %d", sqlite3_errcode(apnDb));
        return APN_DB_ERR;
    } else {
        LYINFLOG("create db in %s", path);
    }
    sqlite3_close(apnDb);
    return APN_DB_CREATE;
}
static int create_apn_table(char*path)
{
    char *zErrMsg = 0;
    int rc;
    char *sql;

    /* Open database */
    pthread_mutex_lock(&s_qser_data_database_mutex);
    rc = sqlite3_open(path, &apnDb);
    if( rc )
    {
        LYERRLOG("[%s] Can't open database: %s\n", __FUNCTION__,sqlite3_errmsg(apnDb));
        pthread_mutex_unlock(&s_qser_data_database_mutex);
        return 1;
    }
    LYDBGLOG("[%s] Opened database successfully\n",__FUNCTION__);
    /* Create SQL statement */
    sql = "CREATE TABLE LYNQAPN( \
         ID               INT                   NOT NULL, \
         PDP_TYPE         INT                   NOT NULL, \
         AUTH_PROTO       INT                   NOT NULL, \
         APN_NAME         TEXT                  NOT NULL, \
         USERNAME         TEXT                  NOT NULL,\
         PASSWORD         TEXT                  NOT NULL, \
         APN_TYPE         TEXT  PRIMARY KEY     NOT NULL, \
         HANDLE           INT                   NOT NULL)";
   /* Execute SQL statement */
    rc = sqlite3_exec(apnDb, sql, NULL, 0, &zErrMsg);
    if( rc != SQLITE_OK )
    {
        LYERRLOG("SQL error: %s\n", zErrMsg);
        sqlite3_free(zErrMsg);
        sqlite3_close(apnDb);
        pthread_mutex_unlock(&s_qser_data_database_mutex);
        return 1;
    }
    LYDBGLOG("Table LYNQAPN created successfully\n");
    sqlite3_close(apnDb);
    pthread_mutex_unlock(&s_qser_data_database_mutex);
    return 0;
}
check_apn_table_cb(void *data, int argc, char **argv, char **azColName)
{
    if(data==NULL)
    {
        return 1;
    }
    int *temp = (int *)data;
    for(int i=0; i<argc; i++){
       LYDBGLOG("[%s]%s = %s", __FUNCTION__,azColName[i], argv[i] ? argv[i] : "NULL");
    }
    *temp = atoi(argv[0]);
    LYINFLOG("\nThe number is %d\n", *temp);
    return 0;
}
int check_apn_table(char*path)
{
    char *zErrMsg = 0;
    int rc;
    char *sql;
    int num = -1;

    /* Open database */
    pthread_mutex_lock(&s_qser_data_database_mutex);
    rc = sqlite3_open(path, &apnDb);
    if( rc )
    {
        LYERRLOG("[%s] Can't open database: %s\n", __FUNCTION__,sqlite3_errmsg(apnDb));
        pthread_mutex_unlock(&s_qser_data_database_mutex);
        return 1;
    }
    LYDBGLOG("[%s] Opened database successfully\n",__FUNCTION__);
    /* Create SQL statement */
    sql = "SELECT COUNT(*) FROM sqlite_master where type ='table' and name = 'LYNQAPN'";
    /* Execute SQL statement */
    rc = sqlite3_exec(apnDb, sql, check_apn_table_cb, &num, &zErrMsg);
    if( rc != SQLITE_OK )
    {
        LYERRLOG("SQL error: %s\n", zErrMsg);
        sqlite3_free(zErrMsg);
        sqlite3_close(apnDb);
        pthread_mutex_unlock(&s_qser_data_database_mutex);
        return 1;
    }
    LYDBGLOG("Table LYNQAPN check successfully\n");
    sqlite3_close(apnDb);
    pthread_mutex_unlock(&s_qser_data_database_mutex);
    if(num <= 0)
    {
        LYINFLOG("No table LYNQAPN, num = %d\n", num);
        return APN_TABLE_NEED_CREATE;
    }
    else
    {
        LYINFLOG("The table LYNQAPN exist, num = %d\n", num);
        return APN_TABLE_CREATE;
    }

    return 0;
}
static int init_apn_db()
{
    int db_ret=0;
    int ret = 0;
    int check_ret=0;
    int table_ret=0;

    pthread_mutex_init(&s_qser_data_database_mutex, NULL);   

    db_ret = create_apn_db(APN_DB_PATH);
    if(db_ret==APN_DB_CREATE)
    {
        table_ret = create_apn_table(APN_DB_PATH);
        if(table_ret!=0)
        {
            LYERRLOG("ret = %d, create apn table fail!!!", table_ret);
            return RESULT_ERROR;
        }
        ret = apn_db_add(0, &default_apn);
        if(ret != 0)
        {
            LYERRLOG("ret = %d, apn add fail!!!", ret);
        }
    }
    else if(db_ret==APN_DB_READED)
    {
        LYINFLOG("[%s] apn db has be build!!!",__FUNCTION__);
        check_ret = check_apn_table(APN_DB_PATH);
        if(check_ret == APN_TABLE_NEED_CREATE)
        {
            table_ret = create_apn_table(APN_DB_PATH);
            if(table_ret!=0)
            {
                LYERRLOG("ret = %d, create apn table fail!!!", table_ret);
                return 1;
            }
            ret = apn_db_add(0, &default_apn);
            if(ret != 0)
            {
                LYERRLOG("ret = %d, apn add fail!!!", ret);
            }
        }
        else if(check_ret == APN_TABLE_CREATE)
        {
            LYINFLOG("Table LYNQAPN has created");
        }
        else
        {
            LYERRLOG("check table fail!!!, ret = %d", check_ret);
        }
    }
    else
    {
        LYERRLOG("init apn db fail!!!");
    }
    return RESULT_OK;
}

static int apn_db_handle_clean()
{
    char *zErrMsg = 0;
    int rc;
    int default_handle = LYNQ_APN_CHANNEL_MAX + 1;
    char sql[128]={0};

    /* Open database */
    pthread_mutex_lock(&s_qser_data_database_mutex);
    rc = sqlite3_open(APN_DB_PATH, &apnDb);
    if( rc )
    {
        LYERRLOG("[%s] Can't open database: %s\n", __FUNCTION__,sqlite3_errmsg(apnDb));
        pthread_mutex_unlock(&s_qser_data_database_mutex);
        return LYNQ_E_APN_DB_FAIL;
    }

    LYDBGLOG("[%s] Opened database successfully\n",__FUNCTION__);
    /* Create SQL statement */
    sprintf(sql,"UPDATE LYNQAPN SET HANDLE=%d", default_handle);
    LYINFLOG("sql: %s\n", sql);
    /* Execute SQL statement */
    rc = sqlite3_exec(apnDb, sql, NULL, NULL, &zErrMsg);
    if( rc != SQLITE_OK )
    {
        LYERRLOG("SQL error: %s\n", zErrMsg);
        sqlite3_free(zErrMsg);
        sqlite3_close(apnDb);
        pthread_mutex_unlock(&s_qser_data_database_mutex);
        return LYNQ_E_APN_DB_FAIL;
    }
    LYDBGLOG("clean apn table successfully\n");
    sqlite3_close(apnDb);
    pthread_mutex_unlock(&s_qser_data_database_mutex);
    return RESULT_OK;
}

static int find_unuse_apn_index(char *path)
{
    char *zErrMsg = 0;
    int rc;
    int count =1;
    bool apn_usable = false;
    char *sql;

    memset(apn_indexs,0,sizeof(apn_indexs));
    next_index = 0;
    LYDBGLOG("index = %p",apn_indexs);
    /* Open database */
    pthread_mutex_lock(&s_qser_data_database_mutex);
    rc = sqlite3_open(APN_DB_PATH, &apnDb);
    if( rc )
    {
       LYERRLOG("Can't open database: %s\n", sqlite3_errmsg(apnDb));
       pthread_mutex_unlock(&s_qser_data_database_mutex);
       return -1;
    }
    else
    {
       LYDBGLOG("Opened database successfully\n");
    }

    /* Create SQL statement */
    sql = "SELECT ID from LYNQAPN";
    /* Execute SQL statement */
    rc = sqlite3_exec(apnDb, sql, lynq_apn_callback,NULL, &zErrMsg);
    if( rc != SQLITE_OK )
    {
        LYERRLOG("SQL error: %s\n", zErrMsg);
        sqlite3_free(zErrMsg);
        sqlite3_close(apnDb);
        pthread_mutex_unlock(&s_qser_data_database_mutex);
        return -1;
    }
    else
    {
       LYDBGLOG("Operation done successfully\n");
    }
    //indexs = temp_index;
    LYDBGLOG("index = %p",apn_indexs);
    for(count;count<=QSER_APN_MAX_LIST;count++)
    {
        //if(apn_indexs[count-1]!=count)
        LYDBGLOG("apn_indexs[count] = %d, count = %d\n", apn_indexs[count], count);
        if(apn_indexs[count]!=count)
        {
            //apn_usable=true;
            break;
        }
    }
    sqlite3_close(apnDb);
    pthread_mutex_unlock(&s_qser_data_database_mutex);
    LYDBGLOG("count is %d\n", count);
    return count;
}

static int apn_db_add(int profile_id, qser_apn_add_s *apn)
{
    char *zErrMsg = 0;
    int rc;
    int default_handle = LYNQ_APN_CHANNEL_MAX + 1;
    /* Open database */
    pthread_mutex_lock(&s_qser_data_database_mutex);
    rc = sqlite3_open(APN_DB_PATH, &apnDb);
    if( rc )
    {
        LYERRLOG("[%s] Can't open database: %s",__FUNCTION__,sqlite3_errmsg(apnDb));
        pthread_mutex_unlock(&s_qser_data_database_mutex);
        return 1;
    }
    LYDBGLOG("[%s] Opened database successfully",__FUNCTION__);
    /* Create SQL statement */
    /* Execute SQL statement */
    LYINFLOG("the profile id is %d\n", profile_id);
    char *sql = sqlite3_mprintf("INSERT INTO LYNQAPN VALUES('%d','%d','%d','%q','%q','%q','%q','%d')", profile_id, apn->pdp_type, apn->auth_proto, apn->apn_name, apn->username, apn->password, apn->apn_type, default_handle);
    rc = sqlite3_exec(apnDb, sql, NULL, NULL, &zErrMsg);
    sqlite3_free(sql);
    if( rc != SQLITE_OK )
    {
        LYERRLOG( "SQL error: %s", zErrMsg);
        sqlite3_free(zErrMsg);
        sqlite3_close(apnDb);
        pthread_mutex_unlock(&s_qser_data_database_mutex);
        return 1;
    }
    LYDBGLOG("write apn to apn db successfully");
    sqlite3_close(apnDb);
    pthread_mutex_unlock(&s_qser_data_database_mutex);
    return 0;
}

static int apn_db_delete(unsigned char profile_idx)
{
    char *zErrMsg = 0;
    int rc;
    char sql[128]={0};

    if(!lynq_check_index(profile_idx))
    {
        return LYNQ_E_APN_DB_FAIL;
    }
    /* Open database */
    pthread_mutex_lock(&s_qser_data_database_mutex);
    rc = sqlite3_open(APN_DB_PATH, &apnDb);
    if( rc )
    {
        LYERRLOG("[%s] Can't open database: %s\n", __FUNCTION__,sqlite3_errmsg(apnDb));
        pthread_mutex_unlock(&s_qser_data_database_mutex);
        return LYNQ_E_APN_DB_FAIL;
    }

    LYDBGLOG("[%s] Opened database successfully\n",__FUNCTION__);
    /* Create SQL statement */
    sprintf(sql,"DELETE from LYNQAPN WHERE ID=%d",(int)profile_idx);
    LYINFLOG("sql: %s\n", sql);
    /* Execute SQL statement */
    rc = sqlite3_exec(apnDb, sql, NULL, NULL, &zErrMsg);
    if( rc != SQLITE_OK )
    {
        LYERRLOG("SQL error: %s\n", zErrMsg);
        sqlite3_free(zErrMsg);
        sqlite3_close(apnDb);
        pthread_mutex_unlock(&s_qser_data_database_mutex);
        return LYNQ_E_APN_DB_FAIL;
    }
    LYINFLOG("delete apn %d successfully\n", profile_idx);
    sqlite3_close(apnDb);
    pthread_mutex_unlock(&s_qser_data_database_mutex);
    return 0;
}

static int apn_db_modify(qser_apn_info_s *apn)
{
    char *zErrMsg = 0;
    int rc;

    if(!lynq_check_index(apn->profile_idx))
    {
        return LYNQ_E_APN_DB_FAIL;
    }
    /* Open database */
    pthread_mutex_lock(&s_qser_data_database_mutex);
    rc = sqlite3_open(APN_DB_PATH, &apnDb);
    if( rc )
    {
        LYERRLOG("[%s] Can't open database: %s\n", __FUNCTION__,sqlite3_errmsg(apnDb));
        pthread_mutex_unlock(&s_qser_data_database_mutex);
        return LYNQ_E_APN_DB_FAIL;
    }

    LYDBGLOG("[%s] Opened database successfully\n",__FUNCTION__);
    /* Create SQL statement */
    char *sql = sqlite3_mprintf("UPDATE LYNQAPN set PDP_TYPE = %d, AUTH_PROTO = %d, APN_NAME = '%q', USERNAME = '%q', PASSWORD = '%q', APN_TYPE = '%q' WHERE ID=%d", apn->pdp_type, apn->auth_proto, apn->apn_name, apn->username, apn->password, apn->apn_type, apn->profile_idx);
    LYINFLOG("sql: %s\n", sql);
    /* Execute SQL statement */
    rc = sqlite3_exec(apnDb, sql, NULL, NULL, &zErrMsg);
    sqlite3_free(sql);
    if( rc != SQLITE_OK )
    {
        LYERRLOG("SQL error: %s\n", zErrMsg);
        sqlite3_free(zErrMsg);
        sqlite3_close(apnDb);
        pthread_mutex_unlock(&s_qser_data_database_mutex);
        return LYNQ_E_APN_DB_FAIL;
    }
    LYINFLOG("set apn %d successfully\n",apn->profile_idx);
    sqlite3_close(apnDb);
    pthread_mutex_unlock(&s_qser_data_database_mutex);
    return 0;
}

static int lynq_apn_db_query_cb(void *data, int argc, char **argv, char **azColName)
{
    if(data==NULL)
    {
        return 1;
    }
    qser_apn_info_s* temp = (qser_apn_info_s *)data;
    for(int i=0; i<argc; i++){
       LYDBGLOG("[%s]%s = %s", __FUNCTION__,azColName[i], argv[i] ? argv[i] : "NULL");
    }
    temp->profile_idx = atoi(argv[0]);
    temp->pdp_type = atoi(argv[1]);
    temp->auth_proto = atoi(argv[2]);
    memcpy(temp->apn_name,argv[3],QSER_APN_NAME_SIZE);
    memcpy(temp->username,argv[4],QSER_APN_USERNAME_SIZE);
    memcpy(temp->password,argv[5],QSER_APN_PASSWORD_SIZE);
    memcpy(temp->apn_type,argv[6],QSER_APN_NAME_SIZE);
    LYINFLOG("lynq_apn_db_query_cb: profile_idx is %d", atoi(argv[0]));
    return 0;
}
static int apn_db_query(unsigned char profile_idx,qser_apn_info_s *apn)
{
    char *zErrMsg = 0;
    int rc;
    char sql[128]={0};

    if(!lynq_check_index(profile_idx))
    {
        return LYNQ_E_APN_DB_FAIL;
    }
    /* Open database */
    pthread_mutex_lock(&s_qser_data_database_mutex);
    rc = sqlite3_open(APN_DB_PATH, &apnDb);
    if( rc )
    {
        LYERRLOG("[%s] Can't open database: %s\n", __FUNCTION__,sqlite3_errmsg(apnDb));
        pthread_mutex_unlock(&s_qser_data_database_mutex);
        return LYNQ_E_APN_DB_FAIL;
    }

    LYDBGLOG("[%s] Opened database successfully\n",__FUNCTION__);
    /* Create SQL statement */
    sprintf(sql,"SELECT * from LYNQAPN WHERE ID=%d", (int)profile_idx);
    LYINFLOG("sql: %s\n", sql);
    /* Execute SQL statement */
    rc = sqlite3_exec(apnDb, sql, lynq_apn_db_query_cb, (qser_apn_info_s *)apn, &zErrMsg);
    if( rc != SQLITE_OK )
    {
        LYERRLOG("SQL error: %s\n", zErrMsg);
        sqlite3_free(zErrMsg);
        sqlite3_close(apnDb);
        pthread_mutex_unlock(&s_qser_data_database_mutex);
        return LYNQ_E_APN_DB_FAIL;
    }
    LYDBGLOG("set apn %d successfully\n",apn->profile_idx);
    sqlite3_close(apnDb);
    pthread_mutex_unlock(&s_qser_data_database_mutex);
    return 0;
}
static int apn_db_query_list_cb(void *data, int argc, char **argv, char **azColName)
{
    int i;

    //apn_count++;

    qser_apn_info_list_s* temp = (qser_apn_info_list_s *)data;
    for(i=0; i<argc; i++){
       LYDBGLOG("%s = %s", azColName[i], argv[i] ? argv[i] : "NULL");
    }
    
    temp->apn[temp->cnt].profile_idx = atoi(argv[0]);
    temp->apn[temp->cnt].pdp_type = atoi(argv[1]);
    temp->apn[temp->cnt].auth_proto = atoi(argv[2]);
    memcpy(temp->apn[temp->cnt].apn_name,argv[3],QSER_APN_NAME_SIZE);
    memcpy(temp->apn[temp->cnt].username,argv[4],QSER_APN_USERNAME_SIZE);
    memcpy(temp->apn[temp->cnt].password,argv[5],QSER_APN_PASSWORD_SIZE);
    memcpy(temp->apn[temp->cnt].apn_type,argv[6],QSER_APN_NAME_SIZE);
    temp->cnt++;
    return 0;
}
static int apn_db_query_list(qser_apn_info_list_s *apn_list)
{
    char *zErrMsg = 0;
    int rc;
    char sql[128]={0};

    /* Open database */
    apn_count = 0;
    apn_list->cnt = 0;
    pthread_mutex_lock(&s_qser_data_database_mutex);
    rc = sqlite3_open(APN_DB_PATH, &apnDb);
    if( rc )
    {
        LYERRLOG("[%s] Can't open database: %s\n", __FUNCTION__,sqlite3_errmsg(apnDb));
        pthread_mutex_unlock(&s_qser_data_database_mutex);
        return LYNQ_E_APN_DB_FAIL;
    }

    LYDBGLOG("[%s] Opened database successfully\n",__FUNCTION__);
    /* Create SQL statement */
    sprintf(sql,"SELECT * from LYNQAPN");
    LYINFLOG("sql: %s\n", sql);
    /* Execute SQL statement */
    rc = sqlite3_exec(apnDb, sql, apn_db_query_list_cb, (qser_apn_info_list_s *)apn_list, &zErrMsg);
    if( rc != SQLITE_OK )
    {
        LYERRLOG("SQL error: %s\n", zErrMsg);
        sqlite3_free(zErrMsg);
        sqlite3_close(apnDb);
        pthread_mutex_unlock(&s_qser_data_database_mutex);
        return LYNQ_E_APN_DB_FAIL;
    }
    sqlite3_close(apnDb);
    pthread_mutex_unlock(&s_qser_data_database_mutex);
    return 0;
}
// **************apn DB end****************************
void judge_pdp_type(qser_apn_pdp_type_e pdp_type,char *out_pdp_type)
{
    switch (pdp_type)
    {
    case QSER_APN_PDP_TYPE_IPV4:
#ifdef MOBILETEK_TARGET_PLATFORM_T106
        strcpy(out_pdp_type,"IP");
#else
        strcpy(out_pdp_type,"IPV4");
#endif
        break;
    case QSER_APN_PDP_TYPE_PPP:
        strcpy(out_pdp_type,"PPP");
        break;
    case QSER_APN_PDP_TYPE_IPV6:
        strcpy(out_pdp_type,"IPV6");
        break;
    case QSER_APN_PDP_TYPE_IPV4V6:
        strcpy(out_pdp_type,"IPV4V6");
        break;
    default:
        strcpy(out_pdp_type,"NULL");
        break;
    }
    return;
}
void judge_authtype(qser_apn_auth_proto_e auth_proto,char *out_proto)
{
    switch (auth_proto)
    {
    case QSER_APN_AUTH_PROTO_DEFAULT:
        strcpy(out_proto,"0");
        break;
    case QSER_APN_AUTH_PROTO_NONE:
        strcpy(out_proto,"0");
        break;
    case QSER_APN_AUTH_PROTO_PAP:
        strcpy(out_proto,"1");
        break;
    case QSER_APN_AUTH_PROTO_CHAP:
        strcpy(out_proto,"2");
        break;
    case QSER_APN_AUTH_PROTO_PAP_CHAP:
        strcpy(out_proto,"3");
        break;
    default:
        strcpy(out_proto,"NULL");
        break;
    }
    return ;
}

int data_call_handle_get(const char profile_idx,int *handle)
{
    int num = LYNQ_APN_CHANNEL_MAX;
    int table_num = 0;
    lynq_apn_info **apn_table = NULL;
    qser_apn_info_s apn;
    int ret = 0;
    apn_table = (lynq_apn_info **)malloc(sizeof(lynq_apn_info *)*LYNQ_APN_CHANNEL_MAX);
    if (NULL == apn_table)
    {
        LYERRLOG("malloc apn_table fail ");
        return RESULT_ERROR;
    }
    for(int i =0;i<10;i++)
    {
        apn_table[i] = (lynq_apn_info*)malloc(sizeof(lynq_apn_info));
        if (apn_table[i]==NULL)
        {
            for (int n=0;n<i;n++)
            {
                free(apn_table[n]);
            }
            return RESULT_ERROR;
        }
        memset(apn_table[i],0,sizeof(lynq_apn_info));
    }
    lynq_get_apn_table(&table_num,apn_table);
    memset(&apn,0,sizeof(qser_apn_info_s));
    ret = apn_db_query(profile_idx,&apn);
    if (ret < 0)
    {
        LYERRLOG("apn_db_query error");
        return ret;
    }
    for (int j = 0;j < table_num;j++)
    {
        if (strcmp(apn.apn_type,apn_table[j]->apnType) == 0)
        {
            *handle = apn_table[j]->index;
            LYINFLOG("apn_table->index:%d,handle:%d ",apn_table[j]->index,*handle);
            break;
        }
    }

    for (int i = 0; i < LYNQ_APN_CHANNEL_MAX; i++) 
    {
        if (apn_table[i]!=NULL)
        {   
            free(apn_table[i]);
            apn_table[i]=NULL;
        }
    }
    free(apn_table);
    apn_table=NULL;
    LYDBGLOG("data_call_handle_get end");
    return RESULT_OK;
}

static void acquire_wake_lock()
{
    int ret = 0;

    ret = system("echo data_call_wakelock 5000000000 > /sys/power/wake_lock");//timeout 5s
    if(ret != 0)
    {
        LYERRLOG("acquire_wake_lock error !!!");
    }
    return;
}

static void release_wake_lock()
{
    int ret = 0;
    
    ret = system("echo data_call_wakelock > /sys/power/wake_unlock");
    if(ret != 0)
    {
        LYERRLOG("release_wake_lock error !!!");
    }
    return;
}

void *thread_wait_cb_status(void)
{
    int handle = -1;
    int ret = 0;
    int default_handle = LYNQ_APN_CHANNEL_MAX + 1;
    lynq_data_call_response_v11_t data_urc_info = {0};
    qser_data_call_state_s data_cb_state;
    while (s_qser_data_cb_thread_status)
    {
        ret = lynq_wait_data_call_state_change(&handle);
        LYINFLOG("[thread_wait_cb_status]: ret = %d, handle = %d, wait data call state change end!!!\n", ret, handle);
        if(s_qser_data_cb_thread_status == 0)
        {
            return NULL;
        }
        else if(ret < 0)
        {
            continue;
        }
        acquire_wake_lock();
        //LYINFLOG("[thread_wait_cb_status]: handle = %d", handle);
        memset(&data_urc_info, 0, sizeof(data_urc_info));
        memset(&data_cb_state, 0, sizeof(data_cb_state));
        lynq_get_data_call_list(&handle,&data_urc_info);
        /*compare paramter*/
        //data_cb_state.profile_idx = (char)handle;
        apn_db_handle_get_profile(handle, &data_cb_state.profile_idx);
        LYINFLOG("[thread_wait_cb_status]: callback profile_idx is %d, status=%d, suggestedRetryTime=%d, cid=%d, active=%d, type=%s, ifname=%s, \n", 
        data_cb_state.profile_idx, data_urc_info.status, data_urc_info.suggestedRetryTime, data_urc_info.cid, data_urc_info.active, data_urc_info.type, data_urc_info.ifname);
        
        LYINFLOG("[thread_wait_cb_status]: addresses=%s, dnses=%s, gateways=%s, pcscf=%s, mtu=%d\n",
        data_urc_info.addresses, data_urc_info.dnses, data_urc_info.gateways, data_urc_info.pcscf, data_urc_info.mtu);

        memcpy(data_cb_state.name,data_urc_info.ifname,strlen(data_urc_info.ifname)+1);
        if ((strcmp(data_urc_info.type,"IPV4") == 0) || (strcmp(data_urc_info.type,"IP") == 0))
        {
            data_cb_state.ip_family = QSER_DATA_CALL_TYPE_IPV4;
        }
        else if (!strcmp(data_urc_info.type,"IPV6"))
        {
            data_cb_state.ip_family = QSER_DATA_CALL_TYPE_IPV6;
        }
        else if (!strcmp(data_urc_info.type,"IPV4V6"))
        {
            data_cb_state.ip_family = QSER_DATA_CALL_TYPE_IPV4V6;
        }
        else
        {
            LYERRLOG("unknow data call type: %s", data_urc_info.type);
        }

        if (data_urc_info.active != 0)
        {
            data_cb_state.state = QSER_DATA_CALL_CONNECTED;
        }
        else
        {
            data_cb_state.state = QSER_DATA_CALL_DISCONNECTED;
            ret = apn_db_handle_set(data_cb_state.profile_idx, default_handle);
            if(ret != 0)
            {
                LYERRLOG("handle set error");
            }
        }
        if (data_cb_state.ip_family == QSER_DATA_CALL_TYPE_IPV4)
        {
            lynq_ipv4_aton_urc(&data_urc_info,&data_cb_state);
        }
        else if (data_cb_state.ip_family == QSER_DATA_CALL_TYPE_IPV6)
        {
            lynq_ipv6_inet_pton_urc(&data_urc_info,&data_cb_state);
        }
        else if (data_cb_state.ip_family == QSER_DATA_CALL_TYPE_IPV4V6)
        {
#ifdef MOBILETEK_TARGET_PLATFORM_T106
            lynq_ipv4v6_inet_pton_urc(&data_urc_info,&data_cb_state);
#else
            lynq_ipv6_inet_pton_urc(&data_urc_info,&data_cb_state);
#endif
        }
        else
        {
            release_wake_lock();
            LYERRLOG("unknow ip_family");
            continue;
        }
        if (s_data_call_cb != NULL)
        {
            s_data_call_cb(&data_cb_state);
            release_wake_lock();
            LYINFLOG("[%s] call back s_data_call_cb end", __FUNCTION__);
        }
    }
    return NULL;
}

int qser_cb_pthread_create()
{
    pthread_attr_t attr;
    int ret;
    s_qser_data_cb_thread_status = 1;
    
    pthread_attr_init(&attr);
    pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);

    ret = pthread_create(&s_cb_tid,&attr,thread_wait_cb_status,NULL);
    if (ret < 0)
    {
        LYERRLOG("pthread create fail");
        s_qser_data_cb_thread_status = 0;
        return RESULT_ERROR;
    }
    pthread_attr_destroy(&attr);
    return RESULT_OK;
}

void qser_cb_pthread_cancel()
{
    s_qser_data_cb_thread_status = 0;
    if (s_cb_tid != -1)
    {
        lynq_release_wait_data_call();
    }
    return;
}
int qser_data_call_init(qser_data_call_evt_cb_t evt_cb)
{   
    int ret = 0;
    int utoken = 0;
    if (NULL == evt_cb)
    {
        LYERRLOG("init incoming paramters error");
        return RESULT_ERROR;
    }
    if(g_lynq_qser_data_init_flag == 1)
    {
        LYERRLOG("init twice is not allowed");
        return RESULT_ERROR;
    }
    ret = lynq_init_data(utoken);
    if (ret != RESULT_OK)
    {
        //qser_cb_pthread_cancel();
        s_data_call_cb = NULL;
        return RESULT_ERROR;
    }
    qser_cb_pthread_create();
    ret = init_apn_db();
    if(ret != RESULT_OK)
    {
        LYERRLOG("init apn db error");
        return RESULT_ERROR;
    }

    ret = apn_db_handle_clean();
    if(ret != RESULT_OK)
    {
        LYERRLOG("clean handle error");
        return RESULT_ERROR;
    }
    s_data_call_cb = evt_cb;
    g_lynq_qser_data_init_flag = 1;
    return RESULT_OK;
}

void qser_data_call_destroy(void)
{
    LYINFLOG("[%s] start [%d]",__FUNCTION__,__LINE__);
    lynq_deinit_data();
    qser_cb_pthread_cancel();
    s_data_call_cb = NULL;
    g_lynq_qser_data_init_flag = 0;
    LYINFLOG("[%s] end [%d]",__FUNCTION__,__LINE__);
    return ;
}

int qser_data_call_start(qser_data_call_s *data_call, qser_data_call_error_e *err)
{
    int ret = -1;
    int error = -1;
    int handle = 0;
    if(g_lynq_qser_data_init_flag == 0)
    {
        if(err != NULL)
        {
            *err = QSER_DATA_CALL_ERROR_NO_INIT;
        }
        return LYNQ_E_NO_INIT;
    }
    if (NULL == data_call || NULL == err)
    {
        LYERRLOG("call start incoming paramters error");
        if(err != NULL)
        {
            *err = QSER_DATA_CALL_ERROR_INVALID_PARAMS;
        }
        return ret;
    }
    if (data_call->profile_idx == 0)
    {
        ret = lynq_setup_data_call(&handle);
    }
    else
    {
        char pdptype[16];
        char auth_proto[16];
        qser_apn_info_s apn_info;
        LYINFLOG("datacall->profile_idx is %d\n", data_call->profile_idx);
        ret = qser_apn_get((unsigned char)data_call->profile_idx,&apn_info);
        if (ret != 0)
        {
            LYERRLOG("qser_apn_get error");
            *err = QSER_DATA_CALL_ERROR_INVALID_PARAMS;
            return ret;
        }
        judge_pdp_type(apn_info.pdp_type,pdptype);
        judge_authtype(apn_info.auth_proto,auth_proto);
        ret = lynq_setup_data_call_sp(&handle,apn_info.apn_name,apn_info.apn_type,apn_info.username,apn_info.password,auth_proto,pdptype,pdptype);
    }
    if (ret != 0)
    {
        *err = QSER_DATA_CALL_ERROR_INVALID_PARAMS;
    }
    else
    {
        error = apn_db_handle_set((unsigned char)data_call->profile_idx, handle);
        if(error != 0)
        {
            LYERRLOG("handle set error");
        }
    }
    return ret;
}

int qser_data_call_start_async(qser_data_call_s *data_call, qser_data_call_error_e *err)
{
    int ret = -1;
    int error = -1;
    int handle = 0;
    if(g_lynq_qser_data_init_flag == 0)
    {
        if(err != NULL)
        {
            *err = QSER_DATA_CALL_ERROR_NO_INIT;
        }
        return LYNQ_E_NO_INIT;
    }
    if (NULL == data_call || NULL == err)
    {
        LYERRLOG("call start incoming paramters error");
        if(err != NULL)
        {
            *err = QSER_DATA_CALL_ERROR_INVALID_PARAMS;
        }
        return ret;
    }
    if (data_call->profile_idx == 0)
    {
        ret = lynq_setup_data_call_sp_t106_async(&handle,"default","default",NULL,NULL,NULL,NULL,NULL);
    }
    else
    {
        char pdptype[16];
        char auth_proto[16];
        qser_apn_info_s apn_info;
        LYINFLOG("datacall->profile_idx is %d\n", data_call->profile_idx);
        ret = qser_apn_get((unsigned char)data_call->profile_idx,&apn_info);
        if (ret != 0)
        {
            LYERRLOG("qser_apn_get error");
            *err = QSER_DATA_CALL_ERROR_INVALID_PARAMS;
            return ret;
        }
        judge_pdp_type(apn_info.pdp_type,pdptype);
        judge_authtype(apn_info.auth_proto,auth_proto);
        ret = lynq_setup_data_call_sp_t106_async(&handle,apn_info.apn_name,apn_info.apn_type,apn_info.username,apn_info.password,auth_proto,pdptype,pdptype);
    }
    if (ret != 0)
    {
        *err = QSER_DATA_CALL_ERROR_INVALID_PARAMS;
    }
    else
    {
        error = apn_db_handle_set((unsigned char)data_call->profile_idx, handle);
        if(error != 0)
        {
            LYERRLOG("handle set error");
        }
    }
    return ret;
}

int qser_data_call_stop(char profile_idx, qser_data_call_ip_family_e ip_family, qser_data_call_error_e *err)
{
    int ret = 0;
    int handle = -1;

    if(g_lynq_qser_data_init_flag == 0)
    {
        if(err != NULL)
        {
            *err = QSER_DATA_CALL_ERROR_NO_INIT;
        }
        return LYNQ_E_NO_INIT;
    }
    if (NULL == err)
    {
        LYERRLOG("call stop incoming paramters error");
        return ret;
    }
    ret = data_call_handle_get(profile_idx,&handle);
    if(ret != 0)
    {
        LYERRLOG("datacall handle get error");
        *err = QSER_DATA_CALL_ERROR_INVALID_PARAMS;
        return ret;
    }
    ret = lynq_deactive_data_call(&handle);
    if (ret != 0)
    {
        *err = QSER_DATA_CALL_ERROR_INVALID_PARAMS;
        return ret;
    }
    return ret;
}
int qser_data_call_info_get(char profile_idx,qser_data_call_ip_family_e ip_family,qser_data_call_info_s *info,qser_data_call_error_e *err)
{
    int ret = 0;
    int handle = -1;
    if(g_lynq_qser_data_init_flag == 0)
    {
        if(err != NULL)
        {
            *err = QSER_DATA_CALL_ERROR_NO_INIT;
        }
        return LYNQ_E_NO_INIT;
    }
    lynq_data_call_response_v11_t data_call_info = {0};
    data_call_handle_get(profile_idx,&handle);
    ret = lynq_get_data_call_list(&handle,&data_call_info);
    if (ret == 0)
    {
        info->profile_idx = profile_idx;
        info->ip_family = ip_family;
        if ((strncmp(data_call_info.type,"IPV4", strlen("IPV4") + 1) == 0) || (strncmp(data_call_info.type,"IP", strlen("IP") + 1) == 0))
        {
            info->ip_family = QSER_DATA_CALL_TYPE_IPV4;
            strcpy(info->v4.name,data_call_info.ifname);
            datacall_ipv4_status_judge(data_call_info.active,info);
            LYDBGLOG("[IPV4]addresses:%s,gateways:%s,dnses:%s",data_call_info.addresses,data_call_info.gateways,data_call_info.dnses);
            lynq_ipv4_aton_getinfo(&data_call_info,info);
        }
        else if (strncmp(data_call_info.type,"IPV6", strlen("IPV6") + 1) == 0)
        {
            info->ip_family = QSER_DATA_CALL_TYPE_IPV6;
            strcpy(info->v6.name,data_call_info.ifname);
            datacall_ipv6_status_judge(data_call_info.active,info);
            LYDBGLOG("[IPV6]addresses:%s,gateways:%s,dnses:%s",data_call_info.addresses,data_call_info.gateways,data_call_info.dnses);
            lynq_ipv6_inet_pton_getinfo(&data_call_info,info);
        }
        else if (strncmp(data_call_info.type,"IPV4V6", strlen("IPV4V6") + 1) == 0)
        {
            info->ip_family = QSER_DATA_CALL_TYPE_IPV4V6;
            strcpy(info->v4.name,data_call_info.ifname);
            datacall_ipv4_status_judge(data_call_info.active,info);
            LYDBGLOG("[IPV4V6]addresses:%s,gateways:%s,dnses:%s",data_call_info.addresses,data_call_info.gateways,data_call_info.dnses);
#ifndef MOBILETEK_TARGET_PLATFORM_T106
            lynq_ipv4_aton_getinfo(&data_call_info,info);
#endif
            strcpy(info->v6.name,data_call_info.ifname);
            datacall_ipv6_status_judge(data_call_info.active,info);
            LYDBGLOG("[IPV4V6]addresses:%s,gateways:%s,dnses:%s",data_call_info.addresses,data_call_info.gateways,data_call_info.dnses);
#ifndef MOBILETEK_TARGET_PLATFORM_T106
            lynq_ipv6_inet_pton_getinfo(&data_call_info,info);
#endif
#ifdef MOBILETEK_TARGET_PLATFORM_T106
            lynq_ipv4v6_inet_pton_getinfo(&data_call_info,info);
#endif
        }
        else
        {
            LYERRLOG("useless qser_data_call_ip_family_e");
        }
    }
    return ret;
}
#ifdef MOBILETEK_TARGET_PLATFORM_T106
int qser_set_initial_apnname(char *apnname)
{
    int ret = -1;
    
    if(g_lynq_qser_data_init_flag == 0)
    {
        return LYNQ_E_NO_INIT;
    }

    ret = lynq_init_attach_apn(apnname, "IP", "0", "null", "null");
    return ret;
}
int qser_get_initial_apnname(char apnname[QSER_APN_NAME_SIZE])
{
    int ret = -1;
    lynq_apn_list_response apnList = {-1, "", "", "", -1, -1, -1, -1, -1, -1};

    if(g_lynq_qser_data_init_flag == 0)
    {
        return LYNQ_E_NO_INIT;
    }

    if(apnname == NULL)
    {
        LYERRLOG("[%s][%d] apnname is NULL",__FUNCTION__,__LINE__);
    }
    ret = lynq_get_apn_list(1, &apnList);
    if(ret == 0)
    {
        memcpy(apnname, apnList.apn, sizeof(apnList.apn));
        LYINFLOG("qser_get_initial_apnname: %s", apnname);
    }
    return ret;
}
#endif
int check_pdp_type(qser_apn_pdp_type_e pdp_type)
{
    switch (pdp_type)
    {
        case QSER_APN_PDP_TYPE_IPV4:
        case QSER_APN_PDP_TYPE_PPP:
        case QSER_APN_PDP_TYPE_IPV6:
        case QSER_APN_PDP_TYPE_IPV4V6:
            return 0;
        default:
            return -1;
    }
}
int check_auth_proto(qser_apn_auth_proto_e auth_proto)
{
    switch (auth_proto)
    {
        case QSER_APN_AUTH_PROTO_DEFAULT:
        case QSER_APN_AUTH_PROTO_NONE:
        case QSER_APN_AUTH_PROTO_PAP:
        case QSER_APN_AUTH_PROTO_CHAP:
        case QSER_APN_AUTH_PROTO_PAP_CHAP:
            return 0;
        default:
            return -1;
    }
}
int qser_apn_set(qser_apn_info_s *apn)
{
    int ret = 0;
    int handle = -1;
    if(g_lynq_qser_data_init_flag == 0)
    {
        return LYNQ_E_NO_INIT;
    }
    if (NULL == apn || apn->profile_idx == 0)
    {
        LYERRLOG("apn set incoming paramters error");
        return RESULT_ERROR;
    }
    ret = apn_db_handle_get(apn->profile_idx, &handle);
    if(ret != 0)
    {
        LYERRLOG("handle set error");
    }
    if(handle >= 0 && handle < LYNQ_APN_CHANNEL_MAX)
    {
        LYERRLOG("It has setup datacall");
        return RESULT_ERROR;
    }

    if(strlen(apn->apn_name) > QSER_APN_NAME_SIZE)
    {
        LYERRLOG("apn_name out of range\n");
        return RESULT_ERROR;
    }
    if(strlen(apn->username) > QSER_APN_USERNAME_SIZE)
    {
        LYERRLOG("username out of range\n");
        return RESULT_ERROR;
    }
    if(strlen(apn->password) > QSER_APN_PASSWORD_SIZE)
    {
        LYERRLOG("password out of range\n");
        return RESULT_ERROR;
    }
    if(strlen(apn->apn_type) > QSER_APN_NAME_SIZE)
    {
        LYERRLOG("apn_type out of range\n");
        return RESULT_ERROR;
    }
    if(check_pdp_type(apn->pdp_type) != 0)
    {
        LYERRLOG("pdp_type out of range\n");
        return RESULT_ERROR;
    }
    if(check_auth_proto(apn->auth_proto) != 0)
    {
        LYERRLOG("auth_proto out of range\n");
        return RESULT_ERROR;
    }

    ret = apn_db_modify(apn);
    if (ret < 0)
    {
        LYERRLOG("apn_db_modify error");
        return ret;
    }
#ifndef MOBILETEK_TARGET_PLATFORM_T106
    int apn_id = 0;
    char tmp_id[12];
    char *outinfo = NULL;
    char normalprotocol[16];
    char authtype[32];
    outinfo = (char *)malloc(sizeof(char)*512);
    bzero(tmp_id,12);
    bzero(outinfo,512);
    apn_id = apn->profile_idx + apndb_offset;
    snprintf(tmp_id,sizeof(tmp_id),"%d",apn_id);
    judge_pdp_type(apn->pdp_type,normalprotocol);
    judge_authtype(apn->auth_proto,authtype);
    lynq_modify_apn_db(3,tmp_id,NULL,NULL,apn->apn_name,apn->apn_type,apn->username,apn->password,normalprotocol,normalprotocol,authtype,outinfo);
    LYINFLOG("[output]:%s",outinfo);
    free(outinfo);
    outinfo = NULL;
#endif
    return ret;
}

int qser_apn_get(unsigned char profile_idx, qser_apn_info_s *apn)
{
    if(g_lynq_qser_data_init_flag == 0)
    {
        return LYNQ_E_NO_INIT;
    }
    if (profile_idx < 0 || profile_idx > QSER_APN_MAX_LIST || NULL == apn)
    {
        LYERRLOG("apn get incoming paramters error");
        return RESULT_ERROR;
    }
    int ret = 0;
    ret = apn_db_query(profile_idx,apn);
    if (ret < 0)
    {
        LYERRLOG("apn_db_query error");
        return ret;
    }
    return ret;
}

int qser_apn_add(qser_apn_add_s *apn, unsigned char *profile_idx)
{
    int ret = 0;

    if(g_lynq_qser_data_init_flag == 0)
    {
        return LYNQ_E_NO_INIT;
    }
    if (NULL == apn || NULL == profile_idx)
    {
        LYERRLOG("apn add incoming paramters error");
        return RESULT_ERROR;
    }
    
    if(strlen(apn->apn_name) > QSER_APN_NAME_SIZE)
    {
        LYERRLOG("apn_name out of range\n");
        return RESULT_ERROR;
    }
    if(strlen(apn->username) > QSER_APN_USERNAME_SIZE)
    {
        LYERRLOG("username out of range\n");
        return RESULT_ERROR;
    }
    if(strlen(apn->password) > QSER_APN_PASSWORD_SIZE)
    {
        LYERRLOG("password out of range\n");
        return RESULT_ERROR;
    }
    if(strlen(apn->apn_type) > QSER_APN_NAME_SIZE)
    {
        LYERRLOG("apn_type out of range\n");
        return RESULT_ERROR;
    }
    if(check_pdp_type(apn->pdp_type) != 0)
    {
        LYERRLOG("pdp_type out of range\n");
        return RESULT_ERROR;
    }
    if(check_auth_proto(apn->auth_proto) != 0)
    {
        LYERRLOG("auth_proto out of range\n");
        return RESULT_ERROR;
    }

    *profile_idx = 0;
    *profile_idx = (unsigned char)find_unuse_apn_index(APN_DB_PATH);
    LYINFLOG("[%s] id = %d\n",__FUNCTION__,(int)*profile_idx);
    if((int)*profile_idx > 0 && (int)*profile_idx < QSER_APN_MAX_LIST)
    {
        LYINFLOG("[%s] apn full is false\n",__FUNCTION__);
        ret = apn_db_add((int)*profile_idx, apn);
        if (ret != 0)
        {
            LYERRLOG("apn_db_add error");
            return RESULT_ERROR;
        }
        else
        {
            return RESULT_OK;
        }
    }
    LYERRLOG("apn storage space is full!!!");
    return RESULT_ERROR;
#ifndef MOBILETEK_TARGET_PLATFORM_T106
    int apn_id = 0;
    char tmp_id[12];
    char *outinfo = NULL;
    char normalprotocol[16];
    char authtype[32];
    outinfo = (char *)malloc(sizeof(char)*512);
    bzero(tmp_id,12);
    bzero(outinfo,512);
    snprintf(tmp_id,sizeof(tmp_id),"%d",apn_id);
    judge_pdp_type(apn->pdp_type,normalprotocol);
    judge_authtype(apn->auth_proto,authtype);
    lynq_modify_apn_db(0,tmp_id,NULL,NULL,apn->apn_name,apn->apn_type,apn->username,apn->password,normalprotocol,normalprotocol,authtype,outinfo);
    LYINFLOG("[output]:%s",outinfo);
    free(outinfo);
    outinfo = NULL;
#endif
    return ret;
}

int qser_apn_del(unsigned char profile_idx)
{
    int ret = 0;
    int handle = -1;

    LYDBGLOG("[%s] entry\n",__FUNCTION__);
    if(g_lynq_qser_data_init_flag == 0)
    {
        LYERRLOG("lynq qser data uninit");
        return LYNQ_E_NO_INIT;
    }
    if (profile_idx <= 0 || profile_idx > QSER_APN_MAX_LIST)
    {
        LYERRLOG("apn del incoming paramters error");
        return RESULT_ERROR;
    }

    ret = apn_db_handle_get(profile_idx, &handle);
    if(ret != 0)
    {
        LYERRLOG("handle set error");
    }
    if(handle >= 0 && handle < LYNQ_APN_CHANNEL_MAX)
    {
        LYERRLOG("It has setup datacall");
        return RESULT_ERROR;
    }
    ret = apn_db_delete(profile_idx);
    if (ret < 0)
    {
        LYERRLOG("apn_db_delete error");
        return ret;
    }
#ifndef MOBILETEK_TARGET_PLATFORM_T106
    int apn_id = 0;
    char tmp_id[12];
    char *outinfo = NULL;
    outinfo = (char *)malloc(sizeof(char)*512);
    bzero(tmp_id,12);
    bzero(outinfo,512);
    apn_id = profile_idx+apndb_offset;
    snprintf(tmp_id,sizeof(tmp_id),"%d",apn_id);
    lynq_modify_apn_db(1,tmp_id,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,outinfo);
    LYINFLOG("[output]:%s",outinfo);
    free(outinfo);
    outinfo = NULL;
#endif
    return ret;
}

int qser_apn_get_list(qser_apn_info_list_s *apn_list)
{
    if(g_lynq_qser_data_init_flag == 0)
    {
        LYERRLOG("lynq qser data uninit");
        return LYNQ_E_NO_INIT;
    }
    if (NULL == apn_list)
    {
        LYERRLOG("apn_list incoming paramters error");
        return RESULT_ERROR;
    }
    int ret = 0;
    ret = apn_db_query_list(apn_list);
    if (ret < 0)
    {
        LYERRLOG("apn_db_query_list error");
        return ret;
    }
    return ret;
}

DEFINE_LYNQ_LIB_LOG(LYNQ_QSER_DATA)

