/*******************************************************
* 
* @brief:
* @details:  add fota upgrade api
* @author:   l.yang
* @date:     2023.8.3
* @version:  V1.0
* @copyright：Copyright (c) MobileTek 
*
*********************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <errno.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <dirent.h>
#include <getopt.h>
#include <unistd.h>

#ifdef __cplusplus
extern "C" {
#endif

#include "include/lynq-qser-fota.h"
#include "liblog/lynq_deflog.h"
#include "include/lynq-qser-autosuspend.h"
#include "pub_debug_info.h"

#define FOTA_REBOOT_FLAG "/mnt/userdata/.fota_reboot_flag"
#define FOTA_FLAG_FILE "/mnt/userdata/.back_up_flag"
#define USER_LOG_TAG    "LYNQ_FOTA"
#define FOTA_ADDR_FILE  "/mnt/userdata/.addr_value"
#define FOTA_FILE_NAME  "upgrade.package"
#define FOTA_UPGRADE_PROCESS  "/mnt/userdata/.fota_upgrade_process"
#define  SYSTEM_A  "34650"
#define  SYSTEM_B  "39019"

#define SYNC_RUNNING_FLAG "/tmp/sync_running.flag"
#define UPDATE_ACTIVE_FLAG "/tmp/update_active.flag"

#define BOOTABLE   "45227"
#define UNBOOTABLE "47806"

#define FOTA_WAKELOCK_NAME "FOTA_WAKELOCK"


#define    LYNQ_UPGRADE_STATUS_VERIFING            (0)
#define    LYNQ_UPGRADE_STATUS_VERIFY_SUCCESS      (1)
#define    LYNQ_UPGRADE_STATUS_VERIFY_FAIL         (2)
#define    LYNQ_UPGRADE_STATUS_UPDATING            (3)
#define    LYNQ_UPGRADE_STATUS_UPDATE_SUCCESS      (4)
#define    LYNQ_UPGRADE_STATUS_UPDATE_FAIL         (5)

#define    LYNQ_SYNNCHRONIZING  8
#define    LYNQ_SYNC_SUCCESS    9
#define    LYNQ_SYNC_FAILED    10

#define    ZXIC_SYNCHRONIZING    1
#define    ZXIC_SYNC_SUCCESS      0
#define    ZXIC_SYNC_FAILED     -1

#define    NO_REBOOT    0


int total_size = 0;
int upgrade_size = 0;


typedef struct
{
    int system;
    int status;
    int try_cnt;
} z_system_info_t;

typedef struct
{
    int upgrade_status;
    int total_size;
    int upgraded_size;
} z_upgrade_status_info_t;

typedef struct
{
    int boot_to;        
    int fota_status;
    z_system_info_t system_1;
    z_system_info_t system_2;
} z_upgrade_system_info_t;

typedef struct
{
    z_upgrade_status_info_t *status;
    void(* status_cb)(z_upgrade_status_info_t *status);
} z_upgrade_flush_status_t;

extern int zxic_dual_verify();
extern int zxic_dual_upgrade(z_upgrade_flush_status_t *flush_status);
extern int zxic_dual_get_upgrade_status(z_upgrade_status_info_t *upgrade_info);
extern int zxic_dual_get_current_system();
extern int zxic_dual_get_boot_to_system();
extern int zxic_dual_set_boot_to_system(int system, int reboot_flag);
extern int zxic_dual_get_system_status(z_upgrade_system_info_t *system_info);
extern int zxic_dual_set_system_status(int system, int status);
extern int zxic_dual_get_fota_status_for_nv();
extern int zxic_dual_set_fota_status_for_nv(int status);
extern int zxic_dual_sync_system(void);
extern int zxic_dual_get_upgrade_type(void);
extern void zxic_dual_get_sync_status(int *sync_status);
extern int zxic_dual_set_sync_status(int sync_status);
extern int zxic_dual_config_package_path(char *upgrade_package_path, int length);

int lynq_fota_verify();
int lynq_fota_upgrade();
int lynq_get_current_system();
int lynq_get_boot_to_system();
int lynq_set_boot_to_system(char *option_para);
int lynq_set_system_a_status(char *option_para);
int lynq_set_system_b_status(char *option_para);
int lynq_sync_system();
int lynq_get_upgrade_type();
int lynq_get_sync_status();
int lynq_set_sync_status(char *option_para);
int  calculate_md5_sum( char *md5_sum,int md5_size);


static int can_run_update() 
{
    if (access(SYNC_RUNNING_FLAG, F_OK) == 0) 
    {
        LYINFLOG("Update: Status=-2 (Sync is running, please try later)\n");
        return -2;
    }

    return 0;
}

static int set_update_flag()
{
    FILE *fp = fopen(UPDATE_ACTIVE_FLAG, "w");
    if (!fp) 
    {
        LYINFLOG("open update_active_flag file failed \n");
        return -1;
    }
    fprintf(fp, "OTA update active");
    fclose(fp);
    system("sync");
    return 0;
}
/*****************************************
* @brief:rock_update_main
* @param count [IN]:NA
* @param sum [OUT]:NA
* @return :success 0, failed -1, busy -2
* @todo:NA
* @see:NA
* @warning:NA
******************************************/
int rock_update_main(int reboot_flag)
{
    
    int ret = 0;
    int current_slot = 0;
    int fota_sync_flag = 0;
    int reboot_flg = NO_REBOOT;
    time_t startTime;
    time_t endTime;
    struct tm *localTime;
    double elapsed = 0;
    char md5_sum[48] = {0};
    FILE *fp = NULL; 
    
    if(set_update_flag() != 0)
    {
    	return -1;
    }
    if(can_run_update() != 0)
    {
	return -2;
    }
    fp = fopen(FOTA_FLAG_FILE,"w+");
    if(fp == NULL)
    {
        LYINFLOG("Creat fota flag file failed");
        return -1;
    }
    fwrite(&fota_sync_flag,sizeof(int),1,fp);
    fclose(fp);

    /* T106BUG-189 fix */
    system("sync");

    fp =fopen(FOTA_REBOOT_FLAG,"w+");
    if(fp == NULL)
    {
        LYERRLOG("Creat get upgrade status flag failed");
        return -1;
    }

    fwrite(&reboot_flg,sizeof(int),1,fp);
    fclose(fp);
    system("sync");

    //fix T106BUG-585 start 
    total_size = 0;
    upgrade_size = 0;

    fp = fopen(FOTA_UPGRADE_PROCESS,"w+");
    if(fp == NULL) 
    {
        LYERRLOG("Error opening file");
        return -1;
    }
       
    fprintf(fp, "%d,%d\n", total_size, upgrade_size);
    fclose(fp);
    //fix T106BUG-585 end
     
    ret = lynq_fota_verify();
    if(ret != 0)
    {
        LYINFLOG("Verify package failed exit upgrade");
        return -1;
    }

    ret = calculate_md5_sum(md5_sum,sizeof(md5_sum));
    if(ret != 0)
    {
        LYINFLOG("Calculate md5 sum failed");
    }

    startTime = time(NULL);
    if(startTime == (time_t)-1) 
    {
        LYERRLOG("Failed to get the current time.\n");
    }
    
    LYINFLOG("-----Begin to upgrade ----");
    ret = lynq_fota_upgrade();
    
    endTime = time(NULL);
    if(endTime == (time_t)-1) 
    {
        LYERRLOG("Failed to get the current time.\n");
    }
    elapsed = difftime(endTime, startTime);
    localTime = localtime(&endTime);
    if(ret != 0)
    {
        LYERRLOG("Upgrade failed");
        ret =  sc_debug_info_record(MODULE_ID_CAP_FOTA,"failed,spendtime: %.2lf seconds,finished time %s,md5:%s", elapsed,asctime(localTime),md5_sum);
        if(ret <= 0)
        {
            LYERRLOG("lynq_record_debug_info failed");
        }
        return -1;
    }
    else
    {
        LYINFLOG("Upgrade success");
        ret =  sc_debug_info_record(MODULE_ID_CAP_FOTA,"success,spendtime: %.2lf seconds,finished time %s,md5:%s", elapsed,asctime(localTime),md5_sum);
        if(ret <= 0)
        {
            LYERRLOG("lynq_record_debug_info failed");
        }
    }


    ret = lynq_get_upgrade_status();
    if(ret != LYNQ_UPGRADE_STATUS_UPDATE_SUCCESS)
    {
        LYINFLOG("Upgrade failed ");
        return -1;
    }

    current_slot = lynq_get_current_system();
    if(current_slot < 0)
    {
        LYINFLOG("Get current system failed");
        return -1;
    }

    if(current_slot == atoi(SYSTEM_A))
    {
        LYINFLOG("current system is system a");
        ret = lynq_set_boot_to_system((char *)SYSTEM_B);
        if(ret != 0)
        {
            LYINFLOG("set system B boot failed ");
            return -1;
        }
        
    }
    else if( current_slot == atoi(SYSTEM_B))
    {
        LYINFLOG("current system is system a");
        ret = lynq_set_boot_to_system((char *)SYSTEM_A);
        if(ret != 0)
        {
            LYINFLOG("set system B boot failed ");
            return -1;
        }

    }
    
    fp = fopen(FOTA_FLAG_FILE,"w+");
    if(fp == NULL)
    {
        LYINFLOG("Creat fota flag file failed");
        return -1;
    }
    
    fota_sync_flag = 1;
    fwrite(&fota_sync_flag,sizeof(int),1,fp);
    fclose(fp);

    /* T106BUG-189 fix */
    system("sync");

    if(reboot_flag == 1)
    {
        LYINFLOG("Enter reboot device");
        system("reboot");
    }
    
    return 0;
}

/*****************************************
* @brief:upgrade callback function
* @param count [IN]:z_upgrade_status_info_t *p_status
* @param sum [OUT]:NA
* @return :NA
* @todo:
* @see:NA
* @warning:NA
******************************************/
void lynq_g_flush_upgrade_status(z_upgrade_status_info_t *p_status)
{

    LYINFLOG("Current status:%d ", p_status->upgrade_status);

    LYINFLOG("Total size:%d ", p_status->total_size);

    LYINFLOG("Updated size:%d ", p_status->upgraded_size);

    total_size = p_status->total_size;
    upgrade_size = p_status->upgraded_size;

    FILE *fp = NULL;

    
    fp = fopen(FOTA_UPGRADE_PROCESS,"w");
    if (fp == NULL) 
    {
        LYERRLOG("Error opening file");
        return;
    }
    
    fprintf(fp, "%d,%d\n", total_size, upgrade_size);
    fclose(fp);
}

/*****************************************
* @brief:lynq_fota_set_addr_value
* @param count [IN]:char  *path
* @param sum [OUT]:
* @return :success 0, failed -1
* @todo:NA
* @see:NA
* @warning:NA
******************************************/
int lynq_fota_set_addr_value(char        *value,int size)
{
    int ret =0;
    FILE *fp = NULL;    
    LYLOGSET(LOG_INFO);
    LYLOGEINIT(USER_LOG_TAG);
    
    if (NULL == value || size <= 0)
    {
        LYINFLOG("Bad package path!!!");
        return -1;
    }
    ret = zxic_dual_config_package_path(value, strlen(value));
    if(ret == -1)
    {
        LYINFLOG("Configure upgrade package path fail");
        return -1;
    }
    
    fp = fopen(FOTA_ADDR_FILE,"w+");
    if(fp == NULL)
    {
        LYINFLOG("Open addr file failed");
        return -1;
    }
    
    fprintf(fp,"%s",value);
    fclose(fp);

    LYINFLOG("Configure upgrade package path success");
    return 0;
}

/*****************************************
* @brief:lynq_fota_get_addr_value
* @param count [IN]:char  *tmp_value
* @param sum [OUT]:
* @return :success 0, failed -1
* @todo:NA
* @see:NA
* @warning:NA
******************************************/
int lynq_fota_get_addr_value(char *tmp_value)
{
    FILE *fp = NULL;
    char tmp_addr[128] = {0};

 
    fp = fopen(FOTA_ADDR_FILE,"r");
    if(fp == NULL)
    {
        LYINFLOG("Open fota addr faile failed!!!");
        return -1;
    }
    while(fgets(tmp_addr,sizeof(tmp_addr),fp) != NULL)
    {
        if( strstr(tmp_addr,FOTA_FILE_NAME) != NULL)
        {
            strncpy(tmp_value,tmp_addr,sizeof(tmp_addr) - 1);
            break;
        }
        else
        {
            LYINFLOG("Get addr failed\n");
            fclose(fp);
            return -1;
        }
    }

    fclose(fp);
    return 0;
        

}


/*****************************************
* @brief:lynq_fota_verify
* @param count [IN]:NA
* @param sum [OUT]:NA
* @return :success 0, failed -1
* @todo:NA
* @see:NA
* @warning:NA
******************************************/
int lynq_fota_verify()
{
    int ret = 0;

    LYINFLOG("-----Begin to verify upgrade package----");
    ret = zxic_dual_verify();
    if(ret != 0)
    {
        LYINFLOG("Verify upgrade package failed");
        return -1;
    }

    return 0;

}

/*****************************************
* @brief:upgrade function
* @param count [IN]:NA
* @param sum [OUT]:NA
* @return :success 0, failed -1
* @todo:NA
* @see:NA
* @warning:NA
******************************************/
int lynq_fota_upgrade()
{
    int ret = 0;
    z_upgrade_flush_status_t flush_status;
    z_upgrade_status_info_t status;
    memset(&status, 0, sizeof(z_upgrade_status_info_t));
    flush_status.status = &status; 
    flush_status.status_cb = &lynq_g_flush_upgrade_status;

    
    LYINFLOG("Enter lynq_fota_upgrade !!!!");
    ret = zxic_dual_upgrade(&flush_status);
    if(ret != 0)
    {
        LYINFLOG("Fota upgrade failed!!!");
        return -1;
    }
    return 0;
}

/*****************************************
* @brief:lynq_fota_nrestart no reboot
* @param count [IN]:NA
* @param sum [OUT]:NA
* @return :success 0, failed -1, busy -2
* @todo:NA
* @see:NA
* @warning:NA
******************************************/
int lynq_fota_nrestart(void)
{
    int ret = 0;
    int tmp = 0;
    int lock_fd = 0;
    lock_fd = qser_wakelock_create(FOTA_WAKELOCK_NAME, strlen(FOTA_WAKELOCK_NAME));
    if(lock_fd < 0)
    {
        LYERRLOG("Create fota wakelock failed %d",lock_fd);
    }

    qser_wakelock_lock(lock_fd);
    ret = rock_update_main(0);
    if(ret != 0)
    {
        LYINFLOG("Upgrade failed !!!!");
    }
    else
    {
        LYINFLOG("upgrade success!!!");
    }

    qser_wakelock_unlock(lock_fd);
    tmp = qser_wakelock_destroy(lock_fd);
    if(tmp < 0)
    {
        LYERRLOG("Destroy fota wakelock failed");
    }
    
    return ret;
}

/*****************************************
* @brief:lynq_rock_main,upgrade done reboot
* @param count [IN]:int first_run
* @param sum [OUT]:NA
* @return :success 0, failed -1, busy -2
* @todo:NA
* @see:NA
* @warning:NA
******************************************/
int lynq_rock_main(int first_run)
{
    if(first_run <= 0)
    {
        LYINFLOG("Bad input value !!!");
        return -1;
    }
    
    int ret = 0;
    int lock_fd = 0;
    int tmp = 0;
    lock_fd = qser_wakelock_create(FOTA_WAKELOCK_NAME, strlen(FOTA_WAKELOCK_NAME));
    if(lock_fd < 0)
    {
        LYERRLOG("Create fota wakelock failed %d",lock_fd);
    }

    qser_wakelock_lock(lock_fd);

    ret = rock_update_main(1);
    LYINFLOG("rock_update_main ret = %d\n", ret);
    if(ret != 0)
    {
        LYINFLOG("fota update fail!\n");
        qser_wakelock_unlock(lock_fd);
        tmp = qser_wakelock_destroy(lock_fd);
        if(tmp < 0)
        {
            LYERRLOG("Destroy fota wakelock failed");
        }
    }

    
    return ret;

}

/*****************************************
* @brief:lynq_get_current_system
* @param count [IN]:NA
* @param sum [OUT]:NA
* @return :success: curretn slot , failed:-1
* @todo:NA
* @see:NA
* @warning:NA
******************************************/
int lynq_get_current_system()
{
    int ret = 0;

    ret = zxic_dual_get_current_system();
    if(ret < 0)
    {
        LYINFLOG("lynq get current system faile");
        return -1;
    }
    if(ret == 34650)
    {
        LYINFLOG("Get current system is system A");
    }
    else if(ret == 39019)
    {
        LYINFLOG("Get current system is system B");
    }
    
    return ret;

}

/*****************************************
* @brief:lynq_get_boot_to_system
* @param count [IN]:NA
* @param sum [OUT]:NA
* @return :success: 0, failed:-1
* @todo:NA
* @see:NA
* @warning:NA
******************************************/
int lynq_get_boot_to_system()
{
    int boot_to = 0;
    boot_to = zxic_dual_get_boot_to_system();
    if(boot_to < 0)
    {
        LYINFLOG("lynq get boot to system failed");
        return -1;
    }
    else
    {
        LYINFLOG("Boot to system:0x%08X[%d]",boot_to);
    }

    return 0;

}

/*****************************************
* @brief:lynq_set_boot_to_system
* @param count [IN]:char *option_para
* @param sum [OUT]:NA
* @return :success: 0, failed:-1
* @todo:NA
* @see:NA
* @warning:NA
******************************************/
int lynq_set_boot_to_system(char *option_para)
{
    int ret = 0;
    if(option_para == NULL)
    {
        LYINFLOG("Input invalid value! null option parameters! ");
        return -1;
    }

    ret = zxic_dual_set_boot_to_system(atoi(option_para), 0);
    if(ret < 0)
    {
        LYINFLOG("Set boot to [%s] error ", option_para);
        return -1;
    }

    return 0;
}

/*****************************************
* @brief:lynq_get_upgrade_status
* @param count [IN]:NA
* @param sum [OUT]:NA
* @return :success: 0, failed:-1
* @todo:NA
* @see:NA
* @warning:NA
******************************************/
int lynq_get_upgrade_status(void)
{
    z_upgrade_status_info_t status;
    int ret = 0;
    int reboot_flag = -1;
    FILE *fp = NULL;
    fp = fopen(FOTA_REBOOT_FLAG,"r");
    if(fp == NULL)
    {
        LYERRLOG("Open reboot flag file failed");
        return -1;
    }
    
    fread(&reboot_flag,sizeof(int),1,fp);
    fclose(fp);
    
    //get upgrade status before no reboot
    if(reboot_flag == NO_REBOOT)
    {
    
        ret = zxic_dual_get_upgrade_status(&status);
        if(ret  < 0)
        {
            LYERRLOG("Get upgrade status fail! ");
            return -1;
        }
     
        LYINFLOG("Current upgrade info: ");
        LYINFLOG("Current upgrade status:%d ", status.upgrade_status);
        LYINFLOG("Current upgrade total size:%d ", status.total_size);
        LYINFLOG("Current upgrade updated size:%d ", status.upgraded_size);
                           
        return status.upgrade_status;
    }
    else
    {
        //get sync status 
        int fota_sync_tatus = 0;
        fota_sync_tatus = lynq_get_sync_status();
        if(fota_sync_tatus == ZXIC_SYNCHRONIZING)
        {
            LYINFLOG("Now fota upgrade sync status is synchronizing");
            return LYNQ_SYNNCHRONIZING;
        }
        else if(fota_sync_tatus == ZXIC_SYNC_SUCCESS)
        {
            LYINFLOG("Now fota upgrade sync status sync success ");
            return LYNQ_SYNC_SUCCESS;
        }        
        else if(fota_sync_tatus == ZXIC_SYNC_FAILED)
        {
            LYERRLOG("Now fota upgrade sync status sync failed ");
            return LYNQ_SYNC_FAILED;
        }
        
        
    }

}

/*****************************************
* @brief:lynq_get_upgrade_status
* @param count [IN]:NA
* @param sum [OUT]:NA
* @return :success: 0, failed:-1
* @todo:NA
* @see:NA
* @warning:NA
******************************************/
int lynq_get_system_info()
{
    int ret = 0;
    z_upgrade_system_info_t system_info;

    ret = zxic_dual_get_system_status(&system_info);
    if(ret < 0)
    {
        LYINFLOG("Get upgrade status fail! ");
        return -1;
    }
    else
    {
        LYINFLOG("System info: ");
        LYINFLOG("Boot to:0x%08X[%d] ", system_info.boot_to, system_info.boot_to);
        LYINFLOG("Fota status:%d ", system_info.fota_status);
        LYINFLOG("System1 system:0x%08X [%d] ", system_info.system_1.system, system_info.system_1.system);
        LYINFLOG("System1 status:%d ", system_info.system_1.status);
        LYINFLOG("System1 try_cnt:%d ", system_info.system_1.try_cnt);
        LYINFLOG("System2 system:0x%08X [%d] ", system_info.system_2.system, system_info.system_2.system);
        LYINFLOG("System2 status:%d ", system_info.system_2.status);
        LYINFLOG("System2 try_cnt:%d ", system_info.system_2.try_cnt);
    }

    return 0;

}

/*****************************************
* @brief:lynq_set_system_a_status
* @param count [IN]:char *option_para
* @param sum [OUT]:NA
* @return :success: 0, failed:-1
* @todo:NA
* @see:NA
* @warning:NA
******************************************/
int lynq_set_system_a_status(char *option_para)
{
    int ret = -1;
    if (NULL == option_para) 
    {
        LYINFLOG("Invalid option_para value  ");
        return -1;
    }

    //34650 A slot status : bootable:45527     unbootalbe:47806   success:23579
    ret = zxic_dual_set_system_status(34650, atoi(option_para));
    if(ret < 0)
    {
        LYINFLOG("Set system 1 status to [%s] error ", option_para);
        return -1;
    }

    return 0;

}

/*****************************************
* @brief:lynq_set_system_b_status
* @param count [IN]:char *option_para
* @param sum [OUT]:NA
* @return :success: 0, failed:-1
* @todo:NA
* @see:NA
* @warning:NA
*****************************************/
int lynq_set_system_b_status(char * option_para)
{
    int ret = -1;
    if (NULL == option_para) 
    {
        LYINFLOG("Invalid option_para  value  ");
        return -1;
    }

    //39019 B slot status: bootable:45227     unbootalbe:47806   success:23579
    ret = zxic_dual_set_system_status(39019, atoi(option_para)); 
    if(ret < 0)
    {
        LYINFLOG("Set system 2 status to [%s] error ", option_para);
    }

    return 0;
}

/*****************************************
* @brief:lynq_get_fota_status_for_nv
* @param count [IN]:NA
* @param sum [OUT]:NA
* @return :success: 0, failed:-1
* @todo:NA
* @see:NA
* @warning:NA
*****************************************/
int lynq_get_fota_status_for_nv()
{
    int status = 0;    
    status = zxic_dual_get_fota_status_for_nv();
    if(status != 0)
    {
        LYINFLOG("Fota status:%d ", status);
        return -1;
    }
    return 0;
}

/*****************************************
* @brief:lynq_set_fota_status_for_nv
* @param count [IN]:char *option_para
* @param sum [OUT]:NA
* @return :success: 0, failed:-1
* @todo:NA
* @see:NA
* @warning:NA
*****************************************/
int lynq_set_fota_status_for_nv(char * option_para)
{
    int ret = 0;

    if (NULL == option_para)
    {
        LYINFLOG("Invalid value! null option parameters! ");
        return -1;
    }

    ret = zxic_dual_set_fota_status_for_nv(atoi(option_para));
    if(ret < 0)
    {
        LYINFLOG("Set fota_status to [%s] error ", option_para);
    }

    return ret;
}

/*****************************************
* @brief:lynq_sync_system
* @param count [IN]:char *option_para
* @param sum [OUT]:NA
* @return :success: 0, failed:-1
* @todo:NA
* @see:NA
* @warning:NA
*****************************************/
int lynq_sync_system()
{
    int ret = -1;

    ret = zxic_dual_sync_system();
    if( ret != 0)
    {
        LYINFLOG("lynq sync system failed !!!!");
        return -1;
    }

    return 0;
}

/*****************************************
* @brief:lynq_get_upgrade_type
* @param count [IN]:NA
* @param sum [OUT]:NA
* @return :success: 0, failed:-1
* @todo:NA
* @see:NA
* @warning:NA
*****************************************/
int lynq_get_upgrade_type()
{
    int upgrade_type = -1;

    upgrade_type = zxic_dual_get_upgrade_type();
    if(upgrade_type < 0)
    {
        LYINFLOG("Get upgrade type failed \n");
        return -1;
    }
    else
    {
        LYINFLOG("upgrade type is %d", upgrade_type);
    }
    return 0;
}

/*****************************************
* @brief:lynq_get_sync_status
* @param count [IN]:char *option_para
* @param sum [OUT]:NA
* @return :success: 0, failed:-1
* @todo:NA
* @see:NA
* @warning:NA
*****************************************/
int lynq_get_sync_status()
{
    int sync_status = -2;
    zxic_dual_get_sync_status(&sync_status);
    LYINFLOG("Current sync status is %d", sync_status);
    return sync_status;
}

/*****************************************
* @brief:lynq_set_sync_status
* @param count [IN]:char *option_para
* @param sum [OUT]:NA
* @return :success: 0, failed:-1
* @todo:NA
* @see:NA
* @warning:NA
*****************************************/
int lynq_set_sync_status(char * option_para)
{
    int ret = 0;

    if (NULL == option_para) 
    {
        LYINFLOG("Command input invalid value! null option parameters! ");
        return -1;
    }

    ret = zxic_dual_set_sync_status(atoi(option_para));
    if (0 != ret)
    {
        LYINFLOG("set sync status fail");
        return -1;
    }

    return 0;
}

/*****************************************
* @brief:lynq_read_process
* @param count [IN]:NS
* @param sum [OUT]:NA
* @return :fota upgrade process 0-10
* @todo:NA
* @see:NA
* @warning:NA
*****************************************/
int lynq_read_process(void)
{
    LYINFLOG("Enter lynq_read_process");
    
    float fota_process = 0;
    int ration = 0;
    int read_count = 0;
    FILE *fp = NULL;
    
    while(1)
    {
        
        fp = fopen(FOTA_UPGRADE_PROCESS, "r");
        if(fp == NULL)
        {
            LYERRLOG("lynq_read_process open file failed");
            usleep(10000);
            read_count++;
            if(read_count > 5)
            {
                break;
            }
        }
        else
        {
            break;
        }
    }
    
    if(fp != NULL) 
    {
        
        char line[256] = {0};
        if (fgets(line, sizeof(line), fp) != NULL) 
        {
            sscanf(line, "%d,%d", &total_size, &upgrade_size);
        }
        fclose(fp);
    }
    
    
    if(total_size != 0 && upgrade_size <= total_size)
    {
        LYINFLOG("Caculate fota process ration ");
        fota_process =  (float) upgrade_size / total_size;
     }
     
     ration = (int)(fota_process * 10);
     LYINFLOG("Fota process ration is %d",ration);
     return ration;
}

int  calculate_md5_sum( char *md5_sum,int md5_size)
{
    FILE *fp = NULL;
    char package_path[256] = {0};
    char command[64] = {0};
    char *space_pos = NULL;
    int length = 0;
    char command_result[64] = {0};
    
    fp = fopen(FOTA_ADDR_FILE, "rb");
    if(fp == NULL)
    {
        LYERRLOG("Open file failed", FOTA_ADDR_FILE);
        return -1;
    }
    if(fgets(package_path, sizeof(package_path), fp) == NULL)
    {
        LYERRLOG("Read file %s failed", package_path);
    }
    fclose(fp);
    
    snprintf(command, sizeof(command), "md5sum %s",package_path);

    fp = popen(command, "r");
    if (fp == NULL) 
    {
        LYERRLOG("Failed to run command\n");
        return -1;
    }
    if (fgets(command_result, sizeof(command_result), fp) != NULL) 
    {
       
        space_pos = strchr(command_result, ' ');
        if (space_pos != NULL) 
        {
            length = space_pos - command_result;
             if(length >= md5_size) 
             {
                pclose(fp);
                return -1;
            }
            strncpy(md5_sum, command_result, length);
            md5_sum[length] = '\0';
        } 
    } 
    
    pclose(fp);
    
    return 0;

}

DEFINE_LYNQ_LIB_LOG(LYNQ_FOTA)

#ifdef __cplusplus
}
#endif
