/***********************************************************************
 * Copyright (C) 2014, ZTE Corporation.
 *
 * File Name:   lcd_comm.c
 * File Mark:
 * Description: Lcd function implementation. Adapted to LCM MODEL - LEAD T15DS26.
 * Others:      this file is in BOOTLOADER procedure.
 *                  This LCM MODEL uses SPI-4.
 *                  NOTE: 4-lines-spi uses CS, CLK, D/CX and SDI. Do not using SDO instead of D/CX.
 * Version:     V1.0
 * Author:      qihongfang
 * Date:        2017-9-18
 *
 * History 1:
 *     Date: 2015-1-16
 *     Version:
 *     Author: YinWenguan
 *     Modification: Add brightness setting.
 *
***********************************************************************/

/*************************************************************************
 * Include files
 *************************************************************************/
#include "../include/board.h"

#include <common.h>
#include <malloc.h>
#include <asm/io.h>
#include "../include/spi.h"
#include "../include/ssp.h"
#include <lcd.h>
#include <drvs_gpio.h>
#include <peripheral.h>

/*************************************************************************
 * Macro
 *************************************************************************/
 /* debug switch */

//static unsigned char g_lcd_id=0;
#define LCM_ID_LEAD			0x7C  /*йst7735s\st7735s*/

/* LCD Features */
#define LCD_WIDTH			128
#define LCD_HEIGHT			64

#define ZSP_LCDINFO_PATH "/LCDINFO.bin"
#define LCD_BITS_PER_PIXEL	(5 + 6 + 5)
#define LCD_PAGE_HEIGHT		8
#define LCD_PAGE_NUM		8
#define LCD_BPP				(2)

#define START_LINE_BIT		0x40
#define START_PAGE          0xB0
#define START_HIGH_BIT      0x10
#define START_LOW_BIT       0x00
#define START_LOW_PY		1

/* rotate display */
//#define LCD_ROTATE_180

#define LCD_XS				0x02
#define LCD_XE				((LCD_WIDTH)+(LCD_XS)-1)
#ifdef LCD_ROTATE_180
#define LCD_MXMYRGB 		0x1C
#define LCD_YS      		0x01
#else
#define LCD_MXMYRGB 		0xC8
#define LCD_YS      		0x03
#endif

#define LCD_YE				((LCD_HEIGHT)+(LCD_YS)-1)

struct lcd_var_screeninfo {
	unsigned int xs;	//xs_offset	
	unsigned int ys;    //ys_offset
	unsigned int width;	//128 	240 	
	unsigned int height; //128 	240		
	unsigned int xe;	 //width + xs - 1	
	unsigned int ye;     //height + ys - 1
	unsigned int xs_offset; //0x02 0x00 			
	unsigned int ys_offset; //0x03 0x00		
	//unsigned int bits_per_pixel;		
};
static struct lcd_var_screeninfo var_screeninfo = {0};	/* Current var */


/**************************************************************************
 *                                  Types                                                                                   *
 **************************************************************************/
static volatile T_ZDrvLcd_Instance			g_sLcd_Instance = {false, LCD_NORMAL, LCD_XFER_POLL, LCD_DCX_CMD};
static volatile T_ZDrvLcd_BacklightStatus	g_sLcd_Backlight = LCD_BACKLIGHT_OFF;
static volatile T_ZDrvLcd_Brightness		g_sLcd_Brightness = 255;

/**************************************************************************
 * Global  Variable
 **************************************************************************/
extern volatile T_ZDrvLcd_Info s_lcd_info;
extern int zDrvLcd_Spi_WriteCmd(unsigned char cmd);
extern int zDrvLcd_Spi_WriteData(unsigned char data);

static unsigned char g_sLcdHalDataBuf[128 * LCD_PAGE_NUM] =
{
    0,
};

static unsigned char g_sLcdHalDataBuf_reg[128 *64] =
{
    0,
};


/**************************************************************************
 * Function Implementation
 **************************************************************************/
void lcd_gpio_define(struct T_ZDrvLcd_gpio_define* config)
{
   
	config->spi_cs_gpio            = GPIO25;
	config->spi_scl_gpio		= GPIO26;
	config->spi_sda_gpio		= GPIO28;
	config->spi_dcx_gpio		= GPIO27;

	config->spi_cs_func_sel		= GPIO25_SSP0_CS;
	config->spi_scl_func_sel	= GPIO26_SSP0_CLK;
	config->spi_sda_func_sel	= GPIO28_SSP0_TXD;
	config->spi_dcx_func_sel	= GPIO27_SSP0_RXD;

	config->spi_cs_gpio_sel		= GPIO25_GPIO25;
	config->spi_scl_gpio_sel	= GPIO26_GPIO26;
	config->spi_sda_gpio_sel	= GPIO28_GPIO28;
	config->spi_dcx_gpio_sel	= GPIO27_GPIO27;

#ifdef CONFIG_ZX297520V3E_FWP
	config->lcd_reset_gpio		=GPIO125;
	config->lcd_reset_gpio_sel	=GPIO125_GPIO125;
	config->lcd_blg_gpio		=GPIO53;
	config->lcd_blg_gpio_sel	=GPIO53_GPIO53;
#else
	config->lcd_reset_gpio		= GPIO120;
	config->lcd_reset_gpio_sel	= GPIO120_GPIO120;
	config->lcd_blg_gpio		= GPIO119;
	config->lcd_blg_gpio_sel	= GPIO119_GPIO119;
#endif

}

//#if KLOCWORK_BOOT
/**************************************************************************
* Function: zDrvLcd_SetRamAddr
* Description: used to set specified start point of DISPLAY RAM for updating laterly
* Parameters:
*       (IN)
*               uiLeft      left postion
*               uiTop       top postion
*       (OUT)
*               None.
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:
*       None.
**************************************************************************/
int set_ram_addr_bw(unsigned short uiLeft, unsigned short uiTop)
{
     signed    int ret = 0;

    /*set column address*/
    zDrvLcd_Spi_WriteCmd(START_HIGH_BIT |((uiLeft>>4) & 0x0F)); // ?4?
    zDrvLcd_Spi_WriteCmd((START_LOW_BIT | (uiLeft & 0x0F))+START_LOW_PY);    // ?4?    

    /*set page address*/
    zDrvLcd_Spi_WriteCmd(START_PAGE | (uiTop & 0x0F));
    return ret ;
}

uint32_t lcd_WritePixel(unsigned short rgb565, unsigned short uiRow, unsigned short uiCol)
{
    unsigned short page_index = uiRow / LCD_PAGE_HEIGHT;

    unsigned short ram_offset = page_index * var_screeninfo.width + uiCol;

    unsigned short bit_shift = uiRow % LCD_PAGE_HEIGHT;

    if (rgb565 != 0 )
        *(g_sLcdHalDataBuf + ram_offset) &= ~(0x01 << bit_shift);
    else 
        *(g_sLcdHalDataBuf + ram_offset) |= (0x01 << bit_shift);

    return 0;
}
/**************************************************************************
* Function: zDrvLcd_UpdateRect
* Description: used to update a specified rectangle area of the LCD device opened previously.
* Parameters:
*       (IN)
*               pBuf        data buffer
*               uiLeft      left postion
*               uiTop       top postion
*               uiWidth     rectangle width
*               uiHeight    rectangle height
*       (OUT)
*               None.
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:
*       None.
**************************************************************************/
int update_rect_bw(unsigned char *pBuf, unsigned short uiLeft, unsigned short uiTop, unsigned short uiWidth, unsigned short uiHeight)
{

    unsigned short  *pSrcBuf = (unsigned short  *)pBuf;
    unsigned short  row,row_reg, col,col_reg, pixel_value;
    unsigned short cur_page;

    if((pBuf==NULL)|(var_screeninfo.width <=uiLeft)|(var_screeninfo.height <=uiTop)|(var_screeninfo.width < uiLeft + uiWidth)| (var_screeninfo.height < uiTop + uiHeight))
      {  return -1;}

    for (row = uiTop,row_reg = 0; row <(uiTop + uiHeight); row++,row_reg++)
    {
        for (col = uiLeft,col_reg = 0; col < (uiLeft + uiWidth); col++,col_reg++)
        {
           g_sLcdHalDataBuf_reg[row * var_screeninfo.width + col] = *(pSrcBuf + row_reg * uiWidth + col_reg);
        }
    }
    
    /* ???????RGB565??????,??RAM */
  
    for (row = 0; row <var_screeninfo.height; row++)
    {
        for (col = 0; col < var_screeninfo.width; col++)
        {
           pixel_value =  g_sLcdHalDataBuf_reg[row * var_screeninfo.width + col];

            lcd_WritePixel(pixel_value, row , col );
        }
    }   

    
    /* ??,?????,??RAM???,????page */
    
    for ( cur_page =0; cur_page <LCD_PAGE_NUM; cur_page++ )
    {
        set_ram_addr_bw( 0, cur_page );
        zDrvLcd_Spi_WriteDataBlock(g_sLcdHalDataBuf + cur_page* var_screeninfo.width, var_screeninfo.width);       
    }

    return 0;
    
}
//#endif

/**************************************************************************
* Function: zDrvLcd_Gpio_Init
* Description: used to initialize GPIO usage.
* Parameters:
*       (IN)
*
*       (OUT)
*
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:
**************************************************************************/
int set_ram_addr_color(unsigned short uiLeft, unsigned short uiTop)
{
	int ret = DRV_SUCCESS;

	ret |= zDrvLcd_Spi_WriteCmd(0x2A);
	ret |= zDrvLcd_Spi_WriteData(0x00);
	ret |= zDrvLcd_Spi_WriteData(var_screeninfo.xs);
	ret |= zDrvLcd_Spi_WriteData((var_screeninfo.xe >> 8));
	ret |= zDrvLcd_Spi_WriteData((var_screeninfo.xe &0xff));


	ret |= zDrvLcd_Spi_WriteCmd(0x2B);
	ret |= zDrvLcd_Spi_WriteData(0x00);
	ret |= zDrvLcd_Spi_WriteData(var_screeninfo.ys);
	ret |= zDrvLcd_Spi_WriteData((var_screeninfo.ye >> 8));
	ret |= zDrvLcd_Spi_WriteData((var_screeninfo.ye & 0xff));

	return ret;
}

int update_rect_color(unsigned char *pBuf, unsigned short uiLeft, unsigned short uiTop, unsigned short uiWidth, unsigned short uiHeight)
{
	int ret = DRV_SUCCESS;
	
    ret |= set_ram_addr_color(uiLeft, uiTop); // ָˢµʼַ

    ret |= zDrvLcd_Spi_WriteCmd(0x2c);
    ret = zDrvLcd_Spi_WriteDataBlock((unsigned char *) pBuf, uiWidth * uiHeight * 2);
	
	return ret;
}

/**************************************************************************
* Function: prj_lcm_init
* Description: used to initialize SPI usage.
* Parameters:
*       (IN)
*
*       (OUT)
*
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:
**************************************************************************/

int prj_lcm_init(void)
{
	int ret = DRV_SUCCESS;
	unsigned long uiRow, uiCol;
	uint8_t page_number,column_number;
#ifdef CONFIG_ZX297520V3E_FWP
	if (s_lcd_info.lcm_info.lcd_item == ST7567_128x64) {
		
		#if 1
		zDrvLcd_Spi_WriteCmd(0xe2); // soft reset
		zDrvLcd_Spi_WriteCmd(0xa3); // Select bias setting(0xa2:1/9, 0xa3:1/7)
		zDrvLcd_Spi_WriteCmd(0xa0); // Set scan direction of SEG(0xa0:normal direction, 0xa1:reverse direction)
		zDrvLcd_Spi_WriteCmd(0xc8); // Set output direction of COM(0xc8:reverse direction, 0xc0:normal direction)
		zDrvLcd_Spi_WriteCmd(0xa6); // Inverse Display(0xa6:normal display, 0xa7:inverse display)
		zDrvLcd_Spi_WriteCmd(0x2f); // Power Control(0x28~0x2f, Built-in Booster ON, Built-in Regulator ON, Built-in Follower ON)
		zDrvLcd_Spi_WriteCmd(0x23); // Select regulation resistor ratio(0x20~0x27)
		zDrvLcd_Spi_WriteCmd(0x81); // Setelectronic volume (EV) level
		zDrvLcd_Spi_WriteCmd(0x2e); // level
		zDrvLcd_Spi_WriteCmd(START_LINE_BIT + 0x20); // set start line	
		//zDrvLcd_Spi_WriteCmd(0xa1); // Set scan direction of SEG(0xa0:normal direction, 0xa1:reverse direction)
		#else
		zDrvLcd_Spi_WriteCmd(0xe2); // soft reset
		zDrvLcd_Spi_WriteCmd(0xa3); // Select bias setting(0xa2:1/9, 0xa3:1/7)
		//zDrvLcd_Spi_WriteCmd(0xa0); // Set scan direction of SEG(0xa0:normal direction, 0xa1:reverse direction)
		zDrvLcd_Spi_WriteCmd(0xc8); // Set output direction of COM(0xc8:reverse direction, 0xc0:normal direction)
		zDrvLcd_Spi_WriteCmd(0xa6); // Inverse Display(0xa6:normal display, 0xa7:inverse display)
		zDrvLcd_Spi_WriteCmd(0x2f); // Power Control(0x28~0x2f, Built-in Booster ON, Built-in Regulator ON, Built-in Follower ON)
		zDrvLcd_Spi_WriteCmd(0x23); // Select regulation resistor ratio(0x20~0x27)
		zDrvLcd_Spi_WriteCmd(0x81); // Setelectronic volume (EV) level
		zDrvLcd_Spi_WriteCmd(0x2e); // level
		zDrvLcd_Spi_WriteCmd(START_LINE_BIT); // set start line	
		zDrvLcd_Spi_WriteCmd(0xa1); // Set scan direction of SEG(0xa0:normal direction, 0xa1:reverse direction)
		#endif
		
		for( page_number = 0; page_number < LCD_PAGE_NUM; page_number++)
		{
			set_ram_addr_bw(0x00, page_number );
			//printf( "	page_number = d%	...\n",page_number);
			for(column_number = 0;column_number < LCD_WIDTH; column_number++)
			{
				zDrvLcd_Spi_WriteData(0x00);
			}
		}
		
		zDrvLcd_Spi_WriteCmd(0xaf); // Display On
		printf("display on show back ...\n");
		//zDrvLcd_DelayMs(120);
		//return ret;
	}
#else

	if(s_lcd_info.lcm_info.lcd_item != LCM_ID_LEAD)//s93521a
	{
		zDrvLcd_Spi_WriteCmd(0x11); //Exit Sleep
		zDrvLcd_DelayMs(150);

		zDrvLcd_Spi_WriteCmd(0x26); //Set Default Gamma
		zDrvLcd_Spi_WriteData(0x04);

		zDrvLcd_Spi_WriteCmd(0xB1);
		zDrvLcd_Spi_WriteData(0x0B);
		zDrvLcd_Spi_WriteData(0x14);

		zDrvLcd_Spi_WriteCmd(0xC0); //Set VRH1[4:0] & VC[2:0] for VCI1 & GVDD
		zDrvLcd_Spi_WriteData(0x10);
		zDrvLcd_Spi_WriteData(0x00);

		zDrvLcd_Spi_WriteCmd(0xC1); //Set BT[2:0] for AVDD & VCL & VGH & VGL
		zDrvLcd_Spi_WriteData(0x05);

		zDrvLcd_Spi_WriteCmd(0xC5); //Set VMH[6:0] & VML[6:0] for VOMH & VCOML
		zDrvLcd_Spi_WriteData(0x46);
		zDrvLcd_Spi_WriteData(0x40);

		zDrvLcd_Spi_WriteCmd(0xC7);
		zDrvLcd_Spi_WriteData(0xBD);

		zDrvLcd_Spi_WriteCmd(0xEC);
		zDrvLcd_Spi_WriteData(0x0C);

		zDrvLcd_Spi_WriteCmd(0x3a); //Set Color Format
		zDrvLcd_Spi_WriteData(0x05);

		zDrvLcd_Spi_WriteCmd(0x2A); //Set Column Address
		zDrvLcd_Spi_WriteData(0x00);
		zDrvLcd_Spi_WriteData(0x00);
		zDrvLcd_Spi_WriteData(0x00);
		zDrvLcd_Spi_WriteData(0x7F);

		zDrvLcd_Spi_WriteCmd(0x2B); //Set Page Address
		zDrvLcd_Spi_WriteData(0x00);
		zDrvLcd_Spi_WriteData(0x00);
		zDrvLcd_Spi_WriteData(0x00);
		zDrvLcd_Spi_WriteData(0x9F);

		zDrvLcd_Spi_WriteCmd(0x36); //Set Scanning Direction
		zDrvLcd_Spi_WriteData(LCD_MXMYRGB);

		zDrvLcd_Spi_WriteCmd(0xB7); //Set Source Output Direction
		zDrvLcd_Spi_WriteData(0x00);

		zDrvLcd_Spi_WriteCmd(0xf2); //Enable Gamma bit
		zDrvLcd_Spi_WriteData(0x01);

		zDrvLcd_Spi_WriteCmd(0xE0);
		zDrvLcd_Spi_WriteData(0x3F);//p1
		zDrvLcd_Spi_WriteData(0x29);//p2
		zDrvLcd_Spi_WriteData(0x26);//p3
		zDrvLcd_Spi_WriteData(0x26);//p4
		zDrvLcd_Spi_WriteData(0x26);//p5
		zDrvLcd_Spi_WriteData(0x0C);//p6
		zDrvLcd_Spi_WriteData(0x51);//p7
		zDrvLcd_Spi_WriteData(0xB8);//p8
		zDrvLcd_Spi_WriteData(0x39);//p9
		zDrvLcd_Spi_WriteData(0x17);//p10
		zDrvLcd_Spi_WriteData(0x00);//p11
		zDrvLcd_Spi_WriteData(0x00);//p12
		zDrvLcd_Spi_WriteData(0x00);//p13
		zDrvLcd_Spi_WriteData(0x00);//p14
		zDrvLcd_Spi_WriteData(0x00);//p15

		zDrvLcd_Spi_WriteCmd(0xE1);
		zDrvLcd_Spi_WriteData(0x00);//p1
		zDrvLcd_Spi_WriteData(0x16);//p2
		zDrvLcd_Spi_WriteData(0x19);//p3
		zDrvLcd_Spi_WriteData(0x19);//p4
		zDrvLcd_Spi_WriteData(0x19);//p5
		zDrvLcd_Spi_WriteData(0x13);//p6
		zDrvLcd_Spi_WriteData(0x2E);//p7
		zDrvLcd_Spi_WriteData(0x47);//p8
		zDrvLcd_Spi_WriteData(0x46);//p9
		zDrvLcd_Spi_WriteData(0x08);//p10
		zDrvLcd_Spi_WriteData(0x3F);//p11
		zDrvLcd_Spi_WriteData(0x3F);//p12
		zDrvLcd_Spi_WriteData(0x3F);//p13
		zDrvLcd_Spi_WriteData(0x3F);//p14
		zDrvLcd_Spi_WriteData(0x3F);//p15

		zDrvLcd_Spi_WriteCmd(0x29); // Display On

		zDrvLcd_Spi_WriteCmd(0x2C);
	}
	else
	{
		ret |= zDrvLcd_Spi_WriteCmd(0x11); //Sleep out
		zDrvLcd_DelayMs(120); //Delay 120ms

		ret |= zDrvLcd_Spi_WriteCmd(0xB1);
		ret |= zDrvLcd_Spi_WriteData(0x00);
		ret |= zDrvLcd_Spi_WriteData(0x08);
		ret |= zDrvLcd_Spi_WriteData(0x05);

		ret |= zDrvLcd_Spi_WriteCmd(0xB2);
		ret |= zDrvLcd_Spi_WriteData(0x00);
		ret |= zDrvLcd_Spi_WriteData(0x08);
		ret |= zDrvLcd_Spi_WriteData(0x05);

		ret |= zDrvLcd_Spi_WriteCmd(0xB3);
		ret |= zDrvLcd_Spi_WriteData(0x00);
		ret |= zDrvLcd_Spi_WriteData(0x08);
		ret |= zDrvLcd_Spi_WriteData(0x05);
		ret |= zDrvLcd_Spi_WriteData(0x00);
		ret |= zDrvLcd_Spi_WriteData(0x08);
		ret |= zDrvLcd_Spi_WriteData(0x05);

		ret |= zDrvLcd_Spi_WriteCmd(0xB4); //Column inversion
		ret |= zDrvLcd_Spi_WriteData(0x00); // Dot inversion in Normal mode

		ret |= zDrvLcd_Spi_WriteCmd(0xC0);
		ret |= zDrvLcd_Spi_WriteData(0xA2);//a2
		ret |= zDrvLcd_Spi_WriteData(0x02);
		ret |= zDrvLcd_Spi_WriteData(0x84);

		ret |= zDrvLcd_Spi_WriteCmd(0xC1);
		ret |= zDrvLcd_Spi_WriteData(0xC5);//c5

		ret |= zDrvLcd_Spi_WriteCmd(0xC2);
		ret |= zDrvLcd_Spi_WriteData(0x0A);
		ret |= zDrvLcd_Spi_WriteData(0x00);

		ret |= zDrvLcd_Spi_WriteCmd(0xC3);
		ret |= zDrvLcd_Spi_WriteData(0x8A);
		ret |= zDrvLcd_Spi_WriteData(0x2A);

		ret |= zDrvLcd_Spi_WriteCmd(0xC4);
		ret |= zDrvLcd_Spi_WriteData(0x8A);
		ret |= zDrvLcd_Spi_WriteData(0xEE);

		ret |= zDrvLcd_Spi_WriteCmd(0xC5); //VCOM
		ret |= zDrvLcd_Spi_WriteData(0x0E);

		ret |= zDrvLcd_Spi_WriteCmd(0x36); //MX, MY, RGB mode
		ret |= zDrvLcd_Spi_WriteData(LCD_MXMYRGB);

		ret |= zDrvLcd_Spi_WriteCmd(0xE0);
		ret |= zDrvLcd_Spi_WriteData(0x12);
		ret |= zDrvLcd_Spi_WriteData(0x18);
		ret |= zDrvLcd_Spi_WriteData(0x10);
		ret |= zDrvLcd_Spi_WriteData(0x18);
		ret |= zDrvLcd_Spi_WriteData(0x33);
		ret |= zDrvLcd_Spi_WriteData(0x2c);
		ret |= zDrvLcd_Spi_WriteData(0x25);
		ret |= zDrvLcd_Spi_WriteData(0x28);
		ret |= zDrvLcd_Spi_WriteData(0x28);
		ret |= zDrvLcd_Spi_WriteData(0x27);
		ret |= zDrvLcd_Spi_WriteData(0x2f);
		ret |= zDrvLcd_Spi_WriteData(0x3c);
		ret |= zDrvLcd_Spi_WriteData(0x00);
		ret |= zDrvLcd_Spi_WriteData(0x03);
		ret |= zDrvLcd_Spi_WriteData(0x03);
		ret |= zDrvLcd_Spi_WriteData(0x10);

		ret |= zDrvLcd_Spi_WriteCmd(0xE1);
		ret |= zDrvLcd_Spi_WriteData(0x12);
		ret |= zDrvLcd_Spi_WriteData(0x18);
		ret |= zDrvLcd_Spi_WriteData(0x10);
		ret |= zDrvLcd_Spi_WriteData(0x18);
		ret |= zDrvLcd_Spi_WriteData(0x2d);
		ret |= zDrvLcd_Spi_WriteData(0x28);
		ret |= zDrvLcd_Spi_WriteData(0x23);
		ret |= zDrvLcd_Spi_WriteData(0x28);
		ret |= zDrvLcd_Spi_WriteData(0x28);
		ret |= zDrvLcd_Spi_WriteData(0x26);
		ret |= zDrvLcd_Spi_WriteData(0x2f);
		ret |= zDrvLcd_Spi_WriteData(0x3b);
		ret |= zDrvLcd_Spi_WriteData(0x00);
		ret |= zDrvLcd_Spi_WriteData(0x03);
		ret |= zDrvLcd_Spi_WriteData(0x03);
		ret |= zDrvLcd_Spi_WriteData(0x10);

		ret |= zDrvLcd_Spi_WriteCmd(0x2A);
		ret |= zDrvLcd_Spi_WriteData(0x00);
		ret |= zDrvLcd_Spi_WriteData(var_screeninfo.xs);
		ret |= zDrvLcd_Spi_WriteData(0x00);
		ret |= zDrvLcd_Spi_WriteData(var_screeninfo.xe);

		ret |= zDrvLcd_Spi_WriteCmd(0x2B);
		ret |= zDrvLcd_Spi_WriteData(0x00);
		ret |= zDrvLcd_Spi_WriteData(var_screeninfo.ys);
		ret |= zDrvLcd_Spi_WriteData(0x00);
		ret |= zDrvLcd_Spi_WriteData(var_screeninfo.ye);

		ret |= zDrvLcd_Spi_WriteCmd(0x3A); //65k mode
		ret |= zDrvLcd_Spi_WriteData(0x05);

		ret |= zDrvLcd_Spi_WriteCmd(0x29); //Display on
		zDrvLcd_DelayMs(200); //Delay 200ms

		ret |= zDrvLcd_Spi_WriteCmd(0x2C);
	}

	// ˢɺɫ
    for (uiRow = 0; uiRow < var_screeninfo.width; uiRow++)
	{
        for (uiCol = 0; uiCol < var_screeninfo.height; uiCol++)
		{
			ret |= zDrvLcd_Spi_WriteData(0x00);
			ret |= zDrvLcd_Spi_WriteData(0x00);
		}
	}
#endif

	return ret;
}

/**************************************************************************
* Function: prj_lcd_get_info
* Description: used to get the LCD device information.
* Parameters:
*       (IN)
*               pInfo    the pointer of LCD device information to store into
*       (OUT)
*               pInfo   the pointer of LCD device information allready stored into
* Returns:
*       DRV_SUCCESS: succeeded.
* Others:
*       None.
**************************************************************************/
int prj_lcd_get_info(T_ZDrvLcd_Info* pInfo)
{
	unsigned char tmp;
	char cmd[64] = {0};
	unsigned char lcd_info[16] = {0};

	sprintf(cmd, "fsload resource 0x%x %s", (ulong)(&lcd_info), ZSP_LCDINFO_PATH);
	run_command(cmd, 0);
	pInfo->width = lcd_info[0]<< 8 | lcd_info[1];//LCD_WIDTH;
	pInfo->height = lcd_info[2]<< 8 | lcd_info[3];//LCD_HEIGHT;
	printf("[LCD]fsload LCDINFO.bin sucess (%d %d)\n",pInfo->width,pInfo->height);
	
	if(pInfo->width == 0 || pInfo->height == 0)
	{
		printf("[LCD]fsload LCDINFO.bin failed!!!!\n");
	}

	pInfo->bitsPerPixel = LCD_BITS_PER_PIXEL;
	pInfo->rMask = 0xF800;
	pInfo->gMask = 0x07E0;
	pInfo->bMask = 0x001F;
	pInfo->instance = g_sLcd_Instance;
	pInfo->backlight = g_sLcd_Backlight;
	pInfo->brightness = g_sLcd_Brightness;
	
	tmp = read_st(0xda);//lcd module's manufacturer ID
	pInfo->lcm_info.man_id = tmp;
	tmp = read_st(0xdb);//lcd module's	ID
	pInfo->lcm_info.dev_id = tmp<<8;
	tmp = read_st(0xdc);//lcd module's	ID
	pInfo->lcm_info.dev_id |= tmp;
	if((pInfo->width == 128 && pInfo->height == 128) && 
		(pInfo->lcm_info.man_id == LCM_ID_LEAD))
	{		
		var_screeninfo.xs_offset = 0x02;
		var_screeninfo.ys_offset = 0x03;
		var_screeninfo.xs = var_screeninfo.xs_offset;
		var_screeninfo.ys = var_screeninfo.ys_offset;
		var_screeninfo.width = 128;
		var_screeninfo.height = 128;
		var_screeninfo.xe = var_screeninfo.width + var_screeninfo.xs - 1;
		var_screeninfo.ye = var_screeninfo.height + var_screeninfo.ys - 1;
		pInfo->lcm_info.lcd_item = LEAD_ST7735_128x128;
		pInfo->lcm_info.name = "LEAD_ST7735_128x128";
	}
	else if(pInfo->width == 128 && pInfo->height == 64)
	{
		var_screeninfo.xs_offset = 0;
		var_screeninfo.ys_offset = 0;
		var_screeninfo.xs = var_screeninfo.xs_offset;
		var_screeninfo.ys = var_screeninfo.ys_offset;
		var_screeninfo.width = 128;
		var_screeninfo.height = 64;
		var_screeninfo.xe = var_screeninfo.width + var_screeninfo.xs - 1;
		var_screeninfo.ye = var_screeninfo.height + var_screeninfo.ys - 1;
		
		pInfo->width = 128;
		pInfo->height = 64;
		pInfo->lcm_info.lcd_item = ST7567_128x64;
		pInfo->lcm_info.name = "ST7567_128x64";

	}
	else //default s93521a 128*160
	{		
		var_screeninfo.xs_offset = 0;
		var_screeninfo.ys_offset = 0;
		var_screeninfo.xs = var_screeninfo.xs_offset;
		var_screeninfo.ys = var_screeninfo.ys_offset;
		var_screeninfo.width = 128;
		var_screeninfo.height = 160;
		var_screeninfo.xe = var_screeninfo.width + var_screeninfo.xs - 1;
		var_screeninfo.ye = var_screeninfo.height + var_screeninfo.ys - 1;
		
		pInfo->width = 128;
		pInfo->height = 160;
		pInfo->lcm_info.lcd_item = S93521A_128x128;
		pInfo->lcm_info.name = "128*128 lcd no found,default S93521A";
	}
		
	BOOT_PRINTF(UBOOT_NOTICE, "%s init: lcd man_id = 0x%x, dev_id=0x%x\n", pInfo->lcm_info.name, pInfo->lcm_info.man_id,pInfo->lcm_info.dev_id);
	return 0;
}

int update_rect(unsigned char *pBuf, unsigned short uiLeft, unsigned short uiTop, unsigned short uiWidth, unsigned short uiHeight)
{
#ifdef CONFIG_ZX297520V3E_FWP
	return update_rect_bw(pBuf,uiLeft,uiTop,uiWidth,uiHeight);
#else
	return update_rect_color(pBuf,uiLeft,uiTop,uiWidth,uiHeight);
#endif
}

