#include "portmirroring.h"

MODULE_LICENSE("GPL");

/* TCP flags */
#define TH_FIN  0x0001
#define TH_SYN  0x0002
#define TH_RST  0x0004
#define TH_PUSH 0x0008
#define TH_ACK  0x0010
#define TH_URG  0x0020
#define TH_ECN  0x0040
#define TH_CWR  0x0080
#define TH_NS   0x0100
#define TH_RES  0x0E00 /* 3 reserved bits */
#define TH_MASK 0x0FFF

portmirroring_factor_t portmirroring_factor;

int portmirroring_do_it = 0;
module_param(portmirroring_do_it, int, 0644);

/*
int do_it=0;
module_param(do_it, int, 0644);

char *dev_name_from;
char *dev_name_to;

char *mac_src;
char *mac_dst;
char *ether_proto;
char *ip_proto;
char *ip_src;
char *ip_dst;
int is_frag = -1;
ushort port_src = 0xFFFF;
ushort port_dst = 0xFFFF;
uint tcp_seq = 0xFFFFFFFF;
uint tcp_ack = 0xFFFFFFFF;
char *tcp_flags;
ushort tcp_window = 0xFFFF;
ushort checksum = 0xFFFF;
*/
#define UNCOVER_COUNT 4

struct _uncover
{
    int (*check_func)(void);
    int need_uncover;
    int (*process_func)(struct sk_buff *skb);
};

int check_uncover_ether_hdr();
int check_uncover_ip_hdr();
int check_uncover_tcp_hdr();
int check_uncover_udp_hdr();
int uncover_tcp_hdr(struct sk_buff *skb);
int uncover_udp_hdr(struct sk_buff *skb);
int uncover_ip_hdr(struct sk_buff *skb);
int uncover_ether_hdr(struct sk_buff *skb);

struct _uncover uncover_array[UNCOVER_COUNT] =
{
    {check_uncover_ether_hdr, 0, uncover_ether_hdr},
    {check_uncover_ip_hdr, 0, uncover_ip_hdr},
    {check_uncover_tcp_hdr, 0, uncover_tcp_hdr},
    {check_uncover_udp_hdr, 0, uncover_udp_hdr}
};

int check_uncover_ether_hdr()
{
    if(portmirroring_factor.mac_src != NULL && strcmp("", portmirroring_factor.mac_src) != 0)
        return 1;
    if(portmirroring_factor.mac_dst != NULL && strcmp("", portmirroring_factor.mac_dst) != 0)
        return 1;
    if(portmirroring_factor.ether_proto != NULL && strcmp("", portmirroring_factor.ether_proto) != 0)
        return 1;
    return 0;
}
int check_uncover_ip_hdr()
{
    if(portmirroring_factor.ip_proto != NULL && strcmp("", portmirroring_factor.ip_proto) != 0)
        return 1;
    if(portmirroring_factor.ip_src != NULL && strcmp("", portmirroring_factor.ip_src) != 0)
        return 1;
    if(portmirroring_factor.ip_dst != NULL && strcmp("", portmirroring_factor.ip_dst) != 0)
        return 1;
    if(portmirroring_factor.is_frag != -1)
        return 1;
    return 0;
}
int check_uncover_tcp_hdr()
{
    if(portmirroring_factor.ip_proto != NULL && strcmp("tcp", portmirroring_factor.ip_proto) == 0)
    {
        if(portmirroring_factor.port_src != 0xFFFF)
            return 1;
        if(portmirroring_factor.port_dst != 0xFFFF)
            return 1;
        if(portmirroring_factor.checksum != 0xFFFF)
            return 1;
    }
    
    if(portmirroring_factor.tcp_seq != 0xFFFFFFFF)
        return 1;
    if(portmirroring_factor.tcp_ack != 0xFFFFFFFF)
        return 1;
    if(portmirroring_factor.tcp_flags != NULL && strcmp("", portmirroring_factor.tcp_flags) != 0)
        return 1;
    if(portmirroring_factor.tcp_window != 0xFFFF)
        return 1;
    
    return 0;
}
int check_uncover_udp_hdr()
{
    if(portmirroring_factor.ip_proto != NULL && strcmp("udp", portmirroring_factor.ip_proto) == 0)
    {
        if(portmirroring_factor.port_src != 0xFFFF)
            return 1;
        if(portmirroring_factor.port_dst != 0xFFFF)
            return 1;
        if(portmirroring_factor.checksum != 0xFFFF)
            return 1;
    }
    return 0;
}

void check_uncover()
{
    int i;
    for(i = 0; i < UNCOVER_COUNT; i++)
    {
        uncover_array[i].need_uncover = uncover_array[i].check_func();
    }
}

void get_tcp_flags(char *single_flag_str, u_int16_t *flags)
{

    if(strcmp(single_flag_str, "FIN") == 0)
    {
        *flags |= TH_FIN;
    }
    else if(strcmp(single_flag_str, "SYN") == 0)
    {
        *flags |= TH_SYN;
    }
    else if(strcmp(single_flag_str, "RST") == 0)
    {
        *flags |= TH_RST;
    }
    else if(strcmp(single_flag_str, "PSH") == 0)
    {
        *flags |= TH_PUSH;
    }
    else if(strcmp(single_flag_str, "ACK") == 0)
    {
        *flags |= TH_ACK;
    }
    else
    {
        //unknown or not supported
    }

}

int check_tcp_conflict(struct sk_buff *skb)
{
    if(portmirroring_factor.ether_proto != NULL && 
        strcmp("", portmirroring_factor.ether_proto) != 0 && 
        strcmp(portmirroring_factor.ether_proto, "ip") != 0) 
    {
        return 1;
    }

    if(skb->protocol != htons(ETH_P_IP))
    {
        return 1;
    }

    if(portmirroring_factor.ip_proto != NULL && 
        strcmp("", portmirroring_factor.ip_proto) != 0 && 
        strcmp(portmirroring_factor.ether_proto, "tcp") != 0)
    {
        return 1;
    }

    struct iphdr *ip_hdr = (struct iphdr*)skb->data;

    if(ip_hdr->protocol != IPPROTO_TCP)
    {
         return 1;
    }
    
    return 0;
}

int uncover_tcp_hdr(struct sk_buff *skb)
{
    //printk("wanghan uncover_tcp_hdr enter\n");

    if(check_tcp_conflict(skb))
        return 0;
    
    struct iphdr *ip_hdr = (struct iphdr*)skb->data;
    struct tcphdr *tcp_hdr = (struct tcphdr *)(skb->data + ip_hdr->ihl * 4);
    char single_flag_str[8] = {0};
    int i, len, j;
    u_int16_t flags = 0;

    u_int16_t th_flags = 0;
    

    if(portmirroring_factor.port_src != 0xFFFF)
    {
        if(portmirroring_factor.port_src != ntohs(tcp_hdr->source))
        {
            return 0;
        }
    }

    if(portmirroring_factor.port_dst != 0xFFFF)
    {
        if(portmirroring_factor.port_dst != ntohs(tcp_hdr->dest))
        {
            return 0;
        }
    }

    if(portmirroring_factor.tcp_seq != 0xFFFFFFFF)
    {
        if(portmirroring_factor.tcp_seq != ntohl(tcp_hdr->seq))
        {
            return 0;
        }
    }

    if(portmirroring_factor.tcp_ack != 0xFFFFFFFF)
    {
        if(portmirroring_factor.tcp_ack != ntohl(tcp_hdr->ack_seq))
        {
            return 0;
        }
    }

    // tcp_flags is something like "SYN/FIN/RST", flags are separated by "/", which means "and".
    if(portmirroring_factor.tcp_flags != NULL && strcmp("", portmirroring_factor.tcp_flags) != 0) 
    {
        i = j = 0;
        len = strlen(portmirroring_factor.tcp_flags);
        //printk("tcp_flags:%s, len:%d\n", tcp_flags, len);
        while(i < len)
        {
            if(*(portmirroring_factor.tcp_flags + i) == '/')
            {
                get_tcp_flags(single_flag_str, &flags);
                //printk("%s\n", single_flag_str);
                memset(single_flag_str, 0, sizeof(single_flag_str));
                j = 0;
            }
            else 
            {
                single_flag_str[j++] = *(portmirroring_factor.tcp_flags + i);
            }
            i++;
            //j++;
        }
        get_tcp_flags(single_flag_str, &flags);
        //printk("%s\n", single_flag_str);

        memcpy(&th_flags, (char *)tcp_hdr + 12, 2);
        th_flags = ntohs(th_flags) & 0x0FFF;

        //printk("wanghan uncover_tcp_hdr flags:%x, th_flags:%x\n", flags, th_flags);

        if(flags != th_flags)
        {
            return 0;
        }
        
    }


    if(portmirroring_factor.tcp_window != 0xFFFF)
    {
        if(portmirroring_factor.tcp_window != ntohs(tcp_hdr->window))
        {
            return 0;
        }
    }

    if(portmirroring_factor.checksum != 0xFFFF)
    {
        if(portmirroring_factor.checksum != ntohs(tcp_hdr->check))
        {
            return 0;
        }
    }

    return 1;
}

int check_udp_conflict(struct sk_buff *skb)
{
    if(portmirroring_factor.ether_proto != NULL && 
        strcmp("", portmirroring_factor.ether_proto) != 0 && 
        strcmp(portmirroring_factor.ether_proto, "ip") != 0) 
    {
        return 1;
    }

    if(skb->protocol != htons(ETH_P_IP))
    {
        return 1;
    }

    if(portmirroring_factor.ip_proto != NULL && 
        strcmp("", portmirroring_factor.ip_proto) != 0 && 
        strcmp(portmirroring_factor.ether_proto, "udp") != 0)
    {
        return 1;
    }

    struct iphdr *ip_hdr = (struct iphdr*)skb->data;

    if(ip_hdr->protocol != IPPROTO_UDP)
    {
         return 1;
    }
    
    return 0;
}


int uncover_udp_hdr(struct sk_buff *skb)
{
    //printk("wanghan uncover_udp_hdr enter\n");

    if(check_udp_conflict(skb))
        return 0;
    
    struct iphdr *ip_hdr = (struct iphdr*)skb->data;
    struct udphdr *udp_hdr = (struct udphdr *)(skb->data + ip_hdr->ihl * 4);

    if(portmirroring_factor.port_src != 0xFFFF)
    {
        if(portmirroring_factor.port_src != ntohs(udp_hdr->source))
        {
            return 0;
        }
    }

    if(portmirroring_factor.port_dst != 0xFFFF)
    {
        if(portmirroring_factor.port_dst != ntohs(udp_hdr->dest))
        {
            return 0;
        }
    }

    if(portmirroring_factor.checksum != 0xFFFF)
    {
        if(portmirroring_factor.checksum != ntohs(udp_hdr->check))
        {
            return 0;
        }
    }

    return 1;
}

int check_ip_conflict(struct sk_buff *skb)
{
    if(portmirroring_factor.ether_proto != NULL && 
        strcmp("", portmirroring_factor.ether_proto) != 0 && 
        strcmp(portmirroring_factor.ether_proto, "ip") != 0) 
    {
        return 1;
    }

    if(skb->protocol != htons(ETH_P_IP))
    {
        return 1;
    }
    return 0;
}

int uncover_ip_hdr(struct sk_buff *skb)
{
    //printk("wanghan uncover_ip_hdr enter\n");

    if(check_ip_conflict(skb))
        return 0;
    
    struct iphdr *ip_hdr = (struct iphdr*)skb->data;
    u_int32_t saddr;
    u_int32_t daddr;
    uint16_t frag_flags = ntohs(ip_hdr->frag_off);
    //int16_t frag_offset = frag_flags << 3;

    if(portmirroring_factor.ip_proto != NULL && strcmp("", portmirroring_factor.ip_proto) != 0)
    {
        if(strcmp(portmirroring_factor.ip_proto, "tcp") == 0)
        {
            if(ip_hdr->protocol != IPPROTO_TCP)
            {
                return 0;
            }
            //return uncover_tcp_hdr(skb);
        }
        else if(strcmp(portmirroring_factor.ip_proto, "udp") == 0)
        {
            if(ip_hdr->protocol != IPPROTO_UDP)
            {
                return 0;
            }
            //return uncover_udp_hdr(skb);
        }
        else if(strcmp(portmirroring_factor.ip_proto, "icmp") == 0)
        {
            if(ip_hdr->protocol != IPPROTO_ICMP)
            {
                return 0;
            }
        }
        else
        {
            //unknown or not supported
        }
    }

    if(portmirroring_factor.ip_src != NULL && strcmp("", portmirroring_factor.ip_src) != 0)
    {
        saddr = in_aton(portmirroring_factor.ip_src);
        if(saddr != ip_hdr->saddr)
        {
            return 0;
        }
    }

    if(portmirroring_factor.ip_dst != NULL && strcmp("", portmirroring_factor.ip_dst) != 0)
    {
        daddr = in_aton(portmirroring_factor.ip_dst);
        if(daddr != ip_hdr->daddr)
        {
            return 0;
        }
    }

    if(portmirroring_factor.is_frag != -1)
    {
        if(portmirroring_factor.is_frag == 1 && !(frag_flags & (IP_MF|IP_OFFSET))) 
        {
            return 0;
        }
        else if(portmirroring_factor.is_frag == 0 && (frag_flags & (IP_MF|IP_OFFSET))) 
        {
            return 0;
        }
    }

    return 1;
}

u_int32_t wanghan_strtoui_16(char *str, int len) // 16
{
    int i = 0;
    char *cur = str;
    u_int32_t ret = 0;
    while(i < len)
    {
        if(cur[i] >= '0' && cur[i] <= '9')
        {
            ret = ret * 16 + (cur[i] - '0');
        }
        else if(cur[i] >= 'A' && cur[i] <= 'F')
        {
            ret = ret * 16 + (cur[i] - 'A') + 10;
        }
        else if(cur[i] >= 'a' && cur[i] <= 'f')
        {
            ret = ret * 16 + (cur[i] - 'a') + 10;
        }
        else
        {
            break;
        }
        i++;
    }
    return ret;
}

int uncover_ether_hdr(struct sk_buff *skb)
{
    //printk("wanghan uncover_ether_hdr enter\n");
    int i, j, k;
    u_int8_t mac_addr_src[ETH_ALEN] = {0};
    u_int8_t mac_addr_dst[ETH_ALEN] = {0};
    char temp[3] = {0}; 
    struct ethhdr *eth = eth_hdr(skb);
    //char *skb_mac_hdr = skb->data - ETH_HLEN;
    //char *skb_mac_hdr = (char*)eth_hdr(skb);
    // mac address must be something like "1A:2B:3C:4D:5E:6F", other formats are wrong
    if(portmirroring_factor.mac_src != NULL && 
        strcmp("", portmirroring_factor.mac_src) != 0 && 
        strlen(portmirroring_factor.mac_src) != 17) 
    {
        return 0;
    }
    if(portmirroring_factor.mac_dst != NULL && 
        strcmp("", portmirroring_factor.mac_dst) != 0 && 
        strlen(portmirroring_factor.mac_dst) != 17)
    {
        return 0;
    }
    if(portmirroring_factor.mac_src != NULL && strcmp("", portmirroring_factor.mac_src) != 0)
    {
        for(i = 0, j = 0; i < ETH_ALEN; i++, j++)
        {
            memcpy(temp, portmirroring_factor.mac_src + i * 2 + j, 2);
            mac_addr_src[i] = (u_int8_t)wanghan_strtoui_16(temp, 2);
        }

        if(!is_valid_ether_addr(mac_addr_src))
        {
            printk("wanghan uncover_ether_hdr invalid ether addr src\n");
            return 0;
        }
        if(memcmp(eth->h_source, mac_addr_src, ETH_ALEN) != 0)
        {
            //printk("wanghan uncover_ether_hdr mac_src no match\n");
            return 0;
        }
    }

    if(portmirroring_factor.mac_dst != NULL && strcmp("", portmirroring_factor.mac_dst) != 0)
    {
        //printk("wanghan uncover_ether_hdr mac_dst:%s\n", mac_dst);
        for(i = 0, j = 0; i < ETH_ALEN; i++, j++)
        {
            memcpy(temp, portmirroring_factor.mac_dst + i * 2 + j, 2);
            mac_addr_dst[i] = (u_int8_t)wanghan_strtoui_16(temp, 2);
        }
        if(!is_valid_ether_addr(mac_addr_dst))
        {
            printk("wanghan uncover_ether_hdr invalid ether addr dst\n");
            return 0;
        }
        if(memcmp(eth->h_dest, mac_addr_dst, ETH_ALEN) != 0)
        {
            //printk("wanghan uncover_ether_hdr mac_addr_dst no match\n");
            return 0;
        }
    }

    if(portmirroring_factor.ether_proto != NULL && strcmp("", portmirroring_factor.ether_proto) != 0)
    {
        if(strcmp(portmirroring_factor.ether_proto, "ip") == 0)
        {
            if(skb->protocol != htons(ETH_P_IP))
            {
                return 0;
            }
            //return uncover_ip_hdr(skb);
        }
        else if(strcmp(portmirroring_factor.ether_proto, "arp") == 0)
        {
            if(skb->protocol != htons(ETH_P_ARP))
            {
                return 0;
            }
        }
        else if(strcmp(portmirroring_factor.ether_proto, "rarp") == 0)
        {
            if(skb->protocol != htons(ETH_P_RARP))
            {
                return 0;
            }
        }
        else if(strcmp(portmirroring_factor.ether_proto, "vlan") == 0)
        {
            if(skb->protocol != htons(ETH_P_8021Q))
            {
                return 0;
            }
        }
        else if(strcmp(portmirroring_factor.ether_proto, "pppoe") == 0)
        {
            if(skb->protocol != htons(ETH_P_PPP_DISC) && skb->protocol != htons(ETH_P_PPP_SES))
            {
                return 0;
            }
        }
        else
        {
            //unknown or not supported
        }
    }

    return 1;
}

int uncover(struct sk_buff *skb)
{
    //printk("\nwanghan uncover enter\n");
    int ret = 1;
    int i;
    check_uncover();
    for(i = 0; i < UNCOVER_COUNT; i++)
    {
        if(uncover_array[i].need_uncover == 1)
        {
            ret &= uncover_array[i].process_func(skb);
            if(ret == 0)
                goto out;
        }
    }

out:
    return ret;
}

unsigned int mirror_handle(struct sk_buff *skb)
{
    struct sk_buff *skb2;

    struct net_device *dev_from;
    struct net_device *dev_to;

    if(portmirroring_factor.do_it==0)
        return NF_ACCEPT;

    if(portmirroring_factor.dev_name_from==NULL)
    {
        printk("portmirroring, dev_name_from is null\n");
        return NF_ACCEPT;
    }

    if(portmirroring_factor.dev_name_to==NULL)
    {
        printk("portmirroring, dev_name_to is null\n");
        return NF_ACCEPT;
    }

    //printk("portmirroring, dev_name_from:%s, dev_name_to:%s\n",dev_name_from,dev_name_to);

    //dev_from=dev_get_by_name(&init_net,dev_name_from);
    dev_to=dev_get_by_name(&init_net,portmirroring_factor.dev_name_to);

    if(dev_to==NULL)
    {
        printk("portmirroring, dev_to is null\n");
        return NF_ACCEPT;
    }

    if((skb->dev!=NULL && strcmp(skb->dev->name,portmirroring_factor.dev_name_from)==0) && uncover(skb) == 1)
    {
        skb2 = skb_clone(skb, GFP_ATOMIC);

        skb2->dev=dev_to;
        skb_push(skb2,ETH_HLEN);//
        skb2->dev->netdev_ops->ndo_start_xmit(skb2,skb2->dev);
        //dev_queue_xmit(skb2);
    }
    
    //dev_put(dev_from);
    dev_put(dev_to);

    return NF_ACCEPT;
}

static int __init
tsp_portmirroring_init(void)
{
    printk("start init portmirroring\n");
    
    printk("init portmirroring done\n");

    return 0;
err:
    return 0;
}

static void __exit
tsp_portmirroring_cleanup(void)
{
    printk("portmirroring cleanup done\n");
}

late_init(tsp_portmirroring_init);
module_exit(tsp_portmirroring_cleanup);

