/*
 * zx234290-regulator.c  --  regulator for ZX234290 PMICs
 *
 * Copyright 2016 ZTE Inc.
 *
 * Author: yuxiang<yu.xiang5@zte.com.cn>
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 */
#include<linux/mutex.h>
#include <linux/errno.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/gpio.h>
#include <linux/i2c.h>
#include <linux/mfd/core.h>
#include <linux/mfd/zx234290.h>
#include <linux/irq.h>
#include <linux/interrupt.h>
#include <mach/spinlock.h>

/* 0 for zx234296, 297 for zx234297 */
 int pmu_chip_ver = 0;

static int zx234290_regulator_write_register(unsigned char addr, unsigned char data, unsigned char mask)
{
    int ret = 0;
    unsigned int content =0;
	
    soft_spin_lock(PMIC_SFLOCK);

    ret = zx234290_i2c_read_simple(addr, &content);
    if (ret < 0)
    {
	    soft_spin_unlock(PMIC_SFLOCK);
        return -EIO;
    }
    content &= ~mask;
    content |= data & mask;
    ret = zx234290_i2c_write_simple(addr, &content);
    if (ret != 0)
    {  
	    soft_spin_unlock(PMIC_SFLOCK);
        return ret;
    }
    soft_spin_unlock(PMIC_SFLOCK);

    return ret;
}

int zx234290_regulator_write_register_PSM(unsigned char addr, unsigned char data, unsigned char mask)
{
    int ret = 0;
    unsigned int content =0;
//    printk("reg = %x, origin data = %x, mask = %x\n", addr, data, mask);
    ret = zx234290_i2c_read_simple_PSM(addr, &content);
    if (ret < 0)
    {
        return -EIO;
    }
//    printk("read content = %x\n", content);
    content &= ~mask;
    content |= data & mask;
//    printk("infact data = %x\n", content);
    ret = zx234290_i2c_write_simple_PSM(addr, &content);
    if (ret != 0)
    {
        return ret;
    }

    content = 0;
    zx234290_i2c_read_simple_PSM(addr, &content);
//    printk("after write content = %x\n", content);

    return ret;
}

static int zx234290_regulator_read_register(unsigned char reg, unsigned char *dest)
{
    int ret = 0;
    unsigned int content =0;

	ret = zx234290_i2c_read_simple(reg, &content);
	if (!ret)
		*dest = content;

    return ret;
}
static int zx234290_regulator_read_register_PSM(unsigned char reg, unsigned char *dest)
{
    int ret = 0;
    unsigned int content =0;

	ret = zx234290_i2c_read_simple_PSM(reg, &content);
	if (!ret)
		*dest = content;

    return ret;
}

int zx234290_get_chip_version(void)
{
	int ret;
    unsigned char reg_addr=0, reg_val=0;

    reg_addr = ZX234297_REG_ADDR_SINK_CONTROL;
    ret = zx234290_regulator_read_register(reg_addr, &reg_val);
	if (ret) {
		return ret;
	} else if (reg_val == 0xFF) {
		pmu_chip_ver = 0;
		return 296;
	} else {
		pr_info("recognize zx234297, [0x29] = %#X", reg_val);
		pmu_chip_ver = 297;
		return 297;
	}
}

#if 1   /* buck1 */
/*******************************************************************************
 * Function:
 * Description:
 * Parameters:
 *   Input:
 *
 *   Output:
 *
 * Returns:
 *
 *
 * Others:
 ********************************************************************************/
int zx234290_set_buck1_onoff(T_ZDrvZx234290_LDO_ENABLE status)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    if(status > LDO_AVTICE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_LDO_EN2;

    reg_val = ZX234290_BITFVAL(status, ZX234290_BUCK1_ON_LSH);
    mask = ZX234290_BITFMASK(ZX234290_LDOS_ON_WID, ZX234290_BUCK1_ON_LSH);

    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}

T_ZDrvZx234290_LDO_ENABLE zx234290_get_buck1_onoff(void)
{
    unsigned char reg_addr=0, reg_val=0;
    T_ZDrvZx234290_LDO_ENABLE status;

    reg_addr = ZX234290_REG_ADDR_LDO_EN2;
    zx234290_regulator_read_register(reg_addr, &reg_val);

    status = ZX234290_BITFEXT(reg_val, ZX234290_LDOS_ON_WID, ZX234290_BUCK1_ON_LSH);

    return status;
}

int zx234290_set_buck1_active_mode(T_ZDrvZx234290_BUCK_NRMMODE status)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    if(status > BUCK_NRMMODE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_BUCK1_MODE;

    reg_val = ZX234290_BITFVAL(status, ZX234290_BUCK1_NRMMODE_LSH);
    mask = ZX234290_BITFMASK(ZX234290_REGULATOR_MODE_WID, ZX234290_BUCK1_NRMMODE_LSH);

    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}

T_ZDrvZx234290_BUCK_NRMMODE zx234290_get_buck1_active_mode(void)
{
    unsigned char reg_addr=0, reg_val=0;
    T_ZDrvZx234290_BUCK_NRMMODE status;

    reg_addr = ZX234290_REG_ADDR_BUCK1_MODE;
    zx234290_regulator_read_register(reg_addr, &reg_val);

    status = ZX234290_BITFEXT(reg_val, ZX234290_REGULATOR_MODE_WID, ZX234290_BUCK1_NRMMODE_LSH);

    return status;
}

int zx234290_set_buck1_voltage(T_ZDrvZx234290_VbuckA vol)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    if(vol > VBUCKA_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_BUCK1_VOL;

    reg_val = ZX234290_BITFVAL(vol, ZX234290_BUCK01_VSEL_LSH);
    mask = ZX234290_BITFMASK(ZX234290_BUCK01_VSEL_WID, ZX234290_BUCK01_VSEL_LSH);

    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}


T_ZDrvZx234290_VbuckA zx234290_get_buck1_voltage(void)
{
    unsigned char reg_addr=0, reg_val=0;
    T_ZDrvZx234290_VbuckA vol;

    reg_addr = ZX234290_REG_ADDR_BUCK1_VOL;
    zx234290_regulator_read_register(reg_addr, &reg_val);

    vol = ZX234290_BITFEXT(reg_val, ZX234290_BUCK01_VSEL_WID, ZX234290_BUCK01_VSEL_LSH);

    return vol;
}

int zx234290_set_buck1_sleep_mode(T_ZDrvZx234290_BUCK1_SLPMODE status)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    if(status > BUCK1_SLPMODE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_BUCK1_MODE;

    reg_val = ZX234290_BITFVAL(status, ZX234290_BUCK1_SLPMODE_LSH);
    mask = ZX234290_BITFMASK(ZX234290_REGULATOR_MODE_WID, ZX234290_BUCK1_SLPMODE_LSH);
    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);

    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}

T_ZDrvZx234290_BUCK1_SLPMODE zx234290_get_buck1_sleep_mode(void)
{
    unsigned char reg_addr=0, reg_val=0;
    T_ZDrvZx234290_BUCK1_SLPMODE status = 0;

    reg_addr = ZX234290_REG_ADDR_BUCK1_MODE;

    zx234290_regulator_read_register(reg_addr, &reg_val);
    status = ZX234290_BITFEXT(reg_val, ZX234290_REGULATOR_MODE_WID, ZX234290_BUCK1_SLPMODE_LSH);

    return status;
}

int zx234290_set_buck1_sleep_voltage(T_ZDrvZx234290_VbuckA vol)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    if(vol > VBUCKA_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_BUCK1_SLPVOL;

    reg_val = ZX234290_BITFVAL(vol, ZX234290_BUCK01_SLEEP_VSEL_LSH);
    mask = ZX234290_BITFMASK(ZX234290_BUCK01_SLEEP_VSEL_WID, ZX234290_BUCK01_SLEEP_VSEL_LSH);

    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}

#endif
#if 1   /* buck2 add*/
int zx234290_set_buck2_onoff(T_ZDrvZx234290_LDO_ENABLE status)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    if(status > LDO_AVTICE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_LDO_EN2;

    reg_val = ZX234290_BITFVAL(status, ZX234290_BUCK2_ON_LSH);
    mask = ZX234290_BITFMASK(ZX234290_LDOS_ON_WID, ZX234290_BUCK2_ON_LSH);

    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}

T_ZDrvZx234290_LDO_ENABLE zx234290_get_buck2_onoff(void)
{
    unsigned char reg_addr=0, reg_val=0;
    T_ZDrvZx234290_LDO_ENABLE status;

    reg_addr = ZX234290_REG_ADDR_LDO_EN2;
    zx234290_regulator_read_register(reg_addr, &reg_val);

    status = ZX234290_BITFEXT(reg_val, ZX234290_LDOS_ON_WID, ZX234290_BUCK2_ON_LSH);

    return status;
}

int zx234290_set_buck2_active_mode(T_ZDrvZx234290_BUCK_NRMMODE status)
{
    int ret = 0;
	unsigned char  reg_addr=0, reg_val=0, mask=0;

    if(status > BUCK_NRMMODE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_BUCK23_MODE;

    reg_val = ZX234290_BITFVAL(status, ZX234290_BUCK2_NRMMODE_LSH);
    mask = ZX234290_BITFMASK(ZX234290_REGULATOR_MODE_WID, ZX234290_BUCK2_NRMMODE_LSH);

    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}

T_ZDrvZx234290_BUCK_NRMMODE zx234290_get_buck2_active_mode(void)
{
    unsigned char reg_addr=0, reg_val=0;
    T_ZDrvZx234290_BUCK_NRMMODE status;

    reg_addr = ZX234290_REG_ADDR_BUCK23_MODE;
    zx234290_regulator_read_register(reg_addr, &reg_val);

    status = ZX234290_BITFEXT(reg_val, ZX234290_REGULATOR_MODE_WID, ZX234290_BUCK2_NRMMODE_LSH);

    return status;
}

int zx234290_set_buck2_voltage(T_ZDrvZx234290_VbuckC vol)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

 	if(vol > VBUCKC_MAX)
    {
        return -EINVAL;
    }
    reg_addr = ZX234290_REG_ADDR_BUCK2_VOL;

    reg_val = ZX234290_BITFVAL(vol, ZX234290_BUCK2_VSEL_LSH);

    mask = ZX234290_BITFMASK(ZX234290_BUCK2_VSEL_WID, ZX234290_BUCK2_VSEL_LSH);

    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}

T_ZDrvZx234290_VbuckC zx234290_get_buck2_voltage(void)
{
    unsigned char reg_addr=0, reg_val=0;
    T_ZDrvZx234290_VbuckC vol;

    reg_addr = ZX234290_REG_ADDR_BUCK2_VOL;
    zx234290_regulator_read_register(reg_addr, &reg_val);

    vol = ZX234290_BITFEXT(reg_val, ZX234290_BUCK2_VSEL_WID, ZX234290_BUCK2_VSEL_LSH);

    return vol;
}

int zx234290_set_buck2_sleep_mode(T_ZDrvZx234290_BUCK234_SLPMODE status)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    if(status > BUCK234_SLPMODE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_BUCK23_MODE;

    reg_val = ZX234290_BITFVAL(status, ZX234290_BUCK2_SLPMODE_LSH);
    mask = ZX234290_BITFMASK(ZX234290_REGULATOR_MODE_WID, ZX234290_BUCK2_SLPMODE_LSH);
    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);

    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}

T_ZDrvZx234290_BUCK234_SLPMODE zx234290_get_buck2_sleep_mode(void)
{
    unsigned char reg_addr=0, reg_val=0;
    T_ZDrvZx234290_BUCK234_SLPMODE status = 0;

    reg_addr = ZX234290_REG_ADDR_BUCK23_MODE;

    zx234290_regulator_read_register(reg_addr, &reg_val);
    status = ZX234290_BITFEXT(reg_val, ZX234290_REGULATOR_MODE_WID, ZX234290_BUCK2_SLPMODE_LSH);

    return status;
}
#endif
#if 1   /* BUCK3/4ѹɸ add*/
int zx234290_set_buck3_onoff(T_ZDrvZx234290_LDO_ENABLE status)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    if(status > LDO_AVTICE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_LDO_EN2;

    reg_val = ZX234290_BITFVAL(status, ZX234290_BUCK3_ON_LSH);
    mask = ZX234290_BITFMASK(ZX234290_LDOS_ON_WID, ZX234290_BUCK3_ON_LSH);

    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }
    return 0;
}

int zx234290_set_buck3_active_mode(T_ZDrvZx234290_BUCK_NRMMODE status)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    if(status > BUCK_NRMMODE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_BUCK23_MODE;

    reg_val = ZX234290_BITFVAL(status, ZX234290_BUCK3_NRMMODE_LSH);
    mask = ZX234290_BITFMASK(ZX234290_REGULATOR_MODE_WID, ZX234290_BUCK3_NRMMODE_LSH);

    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}


int zx234290_set_buck3_sleep_mode(T_ZDrvZx234290_BUCK234_SLPMODE status)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    if(status > BUCK234_SLPMODE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_BUCK23_MODE;

    reg_val = ZX234290_BITFVAL(status, ZX234290_BUCK3_SLPMODE_LSH);
    mask = ZX234290_BITFMASK(ZX234290_REGULATOR_MODE_WID, ZX234290_BUCK3_SLPMODE_LSH);
    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);

    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}

#endif
#if 1   /* buck4  add*/
int zx234290_set_buck4_onoff(T_ZDrvZx234290_LDO_ENABLE status)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    if(status > LDO_AVTICE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_LDO_EN2;

    reg_val = ZX234290_BITFVAL(status, ZX234290_BUCK4_ON_LSH);
    mask = ZX234290_BITFMASK(ZX234290_LDOS_ON_WID, ZX234290_BUCK4_ON_LSH);

    ret = zx234290_regulator_write_register( reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}


int zx234290_set_buck4_active_mode(T_ZDrvZx234290_BUCK_NRMMODE status)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    if(status > BUCK_NRMMODE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_BUCK4_MODE;

    reg_val = ZX234290_BITFVAL(status, ZX234290_BUCK4_NRMMODE_LSH);
    mask = ZX234290_BITFMASK(ZX234290_REGULATOR_MODE_WID, ZX234290_BUCK4_NRMMODE_LSH);

    ret = zx234290_regulator_write_register( reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}


int zx234290_set_buck4_sleep_mode(T_ZDrvZx234290_BUCK234_SLPMODE status)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    if(status > BUCK234_SLPMODE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_BUCK4_MODE;

    reg_val = ZX234290_BITFVAL(status, ZX234290_BUCK4_SLPMODE_LSH);
    mask = ZX234290_BITFMASK(ZX234290_REGULATOR_MODE_WID, ZX234290_BUCK4_SLPMODE_LSH);
    ret = zx234290_regulator_write_register( reg_addr, reg_val, mask);

    if (ret != 0)
    {
        return -EIO;
    }
    return 0;
}

#endif


#if 1
int zx234290_set_ldo1_onoff(T_ZDrvZx234290_LDO_ENABLE status)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    if(status > LDO_AVTICE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_LDO_EN1;

    reg_val = ZX234290_BITFVAL(status, ZX234290_LDO1_ON_LSH);
    mask = ZX234290_BITFMASK(ZX234290_LDOS_ON_WID, ZX234290_LDO1_ON_LSH);

    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}

int zx234290_set_ldo1_onoff_PSM(T_ZDrvZx234290_LDO_ENABLE status)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    if(status > LDO_AVTICE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_LDO_EN1;

    reg_val = ZX234290_BITFVAL(status, ZX234290_LDO1_ON_LSH);
    mask = ZX234290_BITFMASK(ZX234290_LDOS_ON_WID, ZX234290_LDO1_ON_LSH);

    ret = zx234290_regulator_write_register_PSM(reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}


T_ZDrvZx234290_LDO_ENABLE zx234290_get_ldo1_onoff(void)
{
    unsigned char reg_addr=0, reg_val=0;
    T_ZDrvZx234290_LDO_ENABLE status;

    reg_addr = ZX234290_REG_ADDR_LDO_EN1;
    zx234290_regulator_read_register(reg_addr, &reg_val);

    status = ZX234290_BITFEXT(reg_val, ZX234290_LDOS_ON_WID, ZX234290_LDO1_ON_LSH);

    return status;
}

/* עͣ LDO1/9/10 - LDOA */
int zx234290_set_ldo1_voltage(T_ZDrvZx234290_VldoA vol)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    if(vol > VLDOA_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_LDO12_VOL;

    reg_val = ZX234290_BITFVAL(vol, ZX234290_LDO1_VSEL_LSH);
    mask = ZX234290_BITFMASK(ZX234290_LDO_VSEL_WID, ZX234290_LDO1_VSEL_LSH);

    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}

T_ZDrvZx234290_VldoA zx234290_get_ldo1_voltage(void)
{
    unsigned char reg_addr=0, reg_val=0;
    T_ZDrvZx234290_VldoA vol;

    reg_addr = ZX234290_REG_ADDR_LDO12_VOL;
    zx234290_regulator_read_register(reg_addr, &reg_val);

    vol = ZX234290_BITFEXT(reg_val, ZX234290_LDO_VSEL_WID, ZX234290_LDO1_VSEL_LSH);

    return vol;
}

int zx234290_set_ldo1_sleep_mode(T_ZDrvZx234290_LDOA_SLPMODE status)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    if(status > LDOA_SLPMODE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_LDO1234_MODE;

    reg_val = ZX234290_BITFVAL(status, ZX234290_LDO1_SLPMODE_LSH);
    mask = ZX234290_BITFMASK(ZX234290_REGULATOR_MODE_WID, ZX234290_LDO1_SLPMODE_LSH);
    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);

    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}


T_ZDrvZx234290_LDOA_SLPMODE zx234290_get_ldo1_sleep_mode(void)
{
    unsigned char reg_addr=0, reg_val=0;
    T_ZDrvZx234290_LDOA_SLPMODE status = 0;

    reg_addr = ZX234290_REG_ADDR_LDO1234_MODE;

    zx234290_regulator_read_register(reg_addr, &reg_val);
    status = ZX234290_BITFEXT(reg_val, ZX234290_REGULATOR_MODE_WID, ZX234290_LDO1_SLPMODE_LSH);

    return status;
}
#endif

#if 1   /* LDO2  */

int zx234290_set_ldo2_onoff(T_ZDrvZx234290_LDO_ENABLE status)
{
    int ret = 0;
    unsigned char  reg_addr=0, reg_val=0, mask=0;

    if(status > LDO_AVTICE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_LDO_EN1;

    reg_val = ZX234290_BITFVAL(status, ZX234290_LDO2_ON_LSH);
    mask = ZX234290_BITFMASK(ZX234290_LDOS_ON_WID, ZX234290_LDO2_ON_LSH);

    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}

T_ZDrvZx234290_LDO_ENABLE zx234290_get_ldo2_onoff(void)
{
    unsigned char reg_addr=0, reg_val=0;
    T_ZDrvZx234290_LDO_ENABLE status;

    reg_addr = ZX234290_REG_ADDR_LDO_EN1;
    zx234290_regulator_read_register(reg_addr, &reg_val);

    status = ZX234290_BITFEXT(reg_val, ZX234290_LDOS_ON_WID, ZX234290_LDO2_ON_LSH);  /*  */

    return status;
}

/* עͣ LDO2/3/ - LDOC */
int zx234290_set_ldo2_voltage(T_ZDrvZx234290_VldoC vol)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    reg_addr = ZX234290_REG_ADDR_LDO12_VOL;

    reg_val = ZX234290_BITFVAL(vol, ZX234290_LDO2_VSEL_LSH);
    mask = ZX234290_BITFMASK(ZX234290_LDO_VSEL_WID, ZX234290_LDO2_VSEL_LSH);

    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;

}

T_ZDrvZx234290_VldoC zx234290_get_ldo2_voltage(void)
{
    unsigned char reg_addr=0, reg_val=0;
    T_ZDrvZx234290_VldoC vol;

    reg_addr = ZX234290_REG_ADDR_LDO12_VOL;
    zx234290_regulator_read_register(reg_addr, &reg_val);

    vol = ZX234290_BITFEXT(reg_val, ZX234290_LDO_VSEL_WID, ZX234290_LDO2_VSEL_LSH);

    return vol;
}

int zx234290_set_ldo2_sleep_mode(T_ZDrvZx234290_LDOA_SLPMODE status)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    if(status > LDOA_SLPMODE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_LDO1234_MODE;

    reg_val = ZX234290_BITFVAL(status, ZX234290_LDO2_SLPMODE_LSH);
    mask = ZX234290_BITFMASK(ZX234290_REGULATOR_MODE_WID, ZX234290_LDO2_SLPMODE_LSH);
    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);

    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}

T_ZDrvZx234290_LDOA_SLPMODE zx234290_get_ldo2_sleep_mode(void)
{
    unsigned char reg_addr=0, reg_val=0;
    T_ZDrvZx234290_LDOA_SLPMODE status = 0;

    reg_addr = ZX234290_REG_ADDR_LDO1234_MODE;

    zx234290_regulator_read_register(reg_addr, &reg_val);
    status = ZX234290_BITFEXT(reg_val, ZX234290_REGULATOR_MODE_WID, ZX234290_LDO2_SLPMODE_LSH);

    return status;
}
#endif
#if 1   /* LDO3 add*/

int zx234290_set_ldo3_onoff(T_ZDrvZx234290_LDO_ENABLE status)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    if(status > LDO_AVTICE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_LDO_EN1;

    reg_val = ZX234290_BITFVAL(status, ZX234290_LDO3_ON_LSH);
    mask = ZX234290_BITFMASK(ZX234290_LDOS_ON_WID, ZX234290_LDO3_ON_LSH);

    ret = zx234290_regulator_write_register( reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }
    return 0;
}

T_ZDrvZx234290_LDO_ENABLE zx234290_get_ldo3_onoff(void)
{
    unsigned char reg_addr=0, reg_val=0;
    T_ZDrvZx234290_LDO_ENABLE status;

    reg_addr = ZX234290_REG_ADDR_LDO_EN1;
    zx234290_regulator_read_register( reg_addr, &reg_val);

    status = ZX234290_BITFEXT(reg_val, ZX234290_LDOS_ON_WID, ZX234290_LDO3_ON_LSH);  /*  */

    return status;
}

/* עͣ LDO2/3/ - LDOC */
int zx234290_set_ldo3_voltage(T_ZDrvZx234290_VldoC vol)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    reg_addr = ZX234290_REG_ADDR_LDO34_VOL;

    reg_val = ZX234290_BITFVAL(vol, ZX234290_LDO3_VSEL_LSH);
    mask = ZX234290_BITFMASK(ZX234290_LDO_VSEL_WID, ZX234290_LDO3_VSEL_LSH);

    ret = zx234290_regulator_write_register( reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }
    return 0;
}

T_ZDrvZx234290_VldoC zx234290_get_ldo3_voltage(void)
{
    unsigned char  reg_addr=0, reg_val=0;
    T_ZDrvZx234290_VldoC vol;

    reg_addr = ZX234290_REG_ADDR_LDO34_VOL;
    zx234290_regulator_read_register( reg_addr, &reg_val);

    vol = ZX234290_BITFEXT(reg_val, ZX234290_LDO_VSEL_WID, ZX234290_LDO3_VSEL_LSH);

    return vol;
}

int zx234290_set_ldo3_sleep_mode(T_ZDrvZx234290_LDOA_SLPMODE status)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    if(status > LDOA_SLPMODE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_LDO1234_MODE;

    reg_val = ZX234290_BITFVAL(status, ZX234290_LDO3_SLPMODE_LSH);
    mask = ZX234290_BITFMASK(ZX234290_REGULATOR_MODE_WID, ZX234290_LDO3_SLPMODE_LSH);
    ret = zx234290_regulator_write_register( reg_addr, reg_val, mask);

    if (ret != 0)
    {
        return -EIO;
    }
    return 0;
}

T_ZDrvZx234290_LDOA_SLPMODE zx234290_get_ldo3_sleep_mode(void)
{
    unsigned char reg_addr=0, reg_val=0;
    T_ZDrvZx234290_LDOA_SLPMODE status = 0;

    reg_addr = ZX234290_REG_ADDR_LDO1234_MODE;

    zx234290_regulator_read_register( reg_addr, &reg_val);
    status = ZX234290_BITFEXT(reg_val, ZX234290_REGULATOR_MODE_WID, ZX234290_LDO3_SLPMODE_LSH);

    return status;
}
#endif
#if 1   /* LDO4 add*/

int zx234290_set_ldo4_onoff(T_ZDrvZx234290_LDO_ENABLE status)
{
    int ret = 0;
    unsigned char  reg_addr=0, reg_val=0, mask=0;

    if(status > LDO_AVTICE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_LDO_EN1;

    reg_val = ZX234290_BITFVAL(status, ZX234290_LDO4_ON_LSH);
    mask = ZX234290_BITFMASK(ZX234290_LDOS_ON_WID, ZX234290_LDO4_ON_LSH);

    ret = zx234290_regulator_write_register( reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}

T_ZDrvZx234290_LDO_ENABLE zx234290_get_ldo4_onoff(void)
{
    unsigned char  reg_addr=0, reg_val=0;
    T_ZDrvZx234290_LDO_ENABLE status;

    reg_addr = ZX234290_REG_ADDR_LDO_EN1;
    zx234290_regulator_read_register( reg_addr, &reg_val);

    status = ZX234290_BITFEXT(reg_val, ZX234290_LDOS_ON_WID, ZX234290_LDO4_ON_LSH);  /*  */

    return status;
}

/* עͣ LDO4/6/7/8/ - LDOD */
int zx234290_set_ldo4_voltage(T_ZDrvZx234290_VldoD vol)
{
    int ret = 0;
    unsigned char  reg_addr=0, reg_val=0, mask=0;

    reg_addr = ZX234290_REG_ADDR_LDO34_VOL;

    reg_val = ZX234290_BITFVAL(vol, ZX234290_LDO4_VSEL_LSH);
    mask = ZX234290_BITFMASK(ZX234290_LDO_VSEL_WID, ZX234290_LDO4_VSEL_LSH);

    ret = zx234290_regulator_write_register( reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }
    return 0;
}

T_ZDrvZx234290_VldoD zx234290_get_ldo4_voltage(void)
{
    unsigned char  reg_addr=0, reg_val=0;
    T_ZDrvZx234290_VldoD vol;

    reg_addr = ZX234290_REG_ADDR_LDO34_VOL;
    zx234290_regulator_read_register( reg_addr, &reg_val);

    vol = ZX234290_BITFEXT(reg_val, ZX234290_LDO_VSEL_WID, ZX234290_LDO4_VSEL_LSH);

    return vol;
}

int zx234290_set_ldo4_sleep_mode(T_ZDrvZx234290_LDOB_SLPMODE status)
{
    int ret = 0;
    unsigned char  reg_addr=0, reg_val=0, mask=0;

    if(status > LDOB_SLPMODE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_LDO1234_MODE;

    reg_val = ZX234290_BITFVAL(status, ZX234290_LDO4_SLPMODE_LSH);
    mask = ZX234290_BITFMASK(ZX234290_REGULATOR_MODE_WID, ZX234290_LDO4_SLPMODE_LSH);
    ret = zx234290_regulator_write_register( reg_addr, reg_val, mask);

    if (ret != 0)
    {
        return -EIO;
    }
    return 0;
}

T_ZDrvZx234290_LDOB_SLPMODE zx234290_get_ldo4_sleep_mode(void)
{
    unsigned char  reg_addr=0, reg_val=0;
    T_ZDrvZx234290_LDOB_SLPMODE status = 0;

    reg_addr = ZX234290_REG_ADDR_LDO1234_MODE;

    zx234290_regulator_read_register( reg_addr, &reg_val);
    status = ZX234290_BITFEXT(reg_val, ZX234290_REGULATOR_MODE_WID, ZX234290_LDO4_SLPMODE_LSH);

    return status;
}

#endif



#if 1

int zx234290_set_ldo5_onoff(T_ZDrvZx234290_LDO_ENABLE status)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    if(status > LDO_AVTICE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_LDO_EN1;

    reg_val = ZX234290_BITFVAL(status, ZX234290_LDO5_ON_LSH);
    mask = ZX234290_BITFMASK(ZX234290_LDOS_ON_WID, ZX234290_LDO5_ON_LSH);

    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}

int zx234290_set_ldo5_onoff_PSM(T_ZDrvZx234290_LDO_ENABLE status)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    if(status > LDO_AVTICE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_LDO_EN1;

    reg_val = ZX234290_BITFVAL(status, ZX234290_LDO5_ON_LSH);
    mask = ZX234290_BITFMASK(ZX234290_LDOS_ON_WID, ZX234290_LDO5_ON_LSH);

    ret = zx234290_regulator_write_register_PSM(reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}

T_ZDrvZx234290_LDO_ENABLE zx234290_get_ldo5_onoff(void)
{
    unsigned char reg_addr=0, reg_val=0;
    T_ZDrvZx234290_LDO_ENABLE status;

    reg_addr = ZX234290_REG_ADDR_LDO_EN1;
    zx234290_regulator_read_register(reg_addr, &reg_val);

    status = ZX234290_BITFEXT(reg_val, ZX234290_LDOS_ON_WID, ZX234290_LDO5_ON_LSH);

    return status;
}

/* עͣ LDO5 */
int zx234290_set_ldo5_voltage(T_ZDrvZx234290_VldoB vol)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    if(vol > VLDOB_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_LDO56_VOL;

    reg_val = ZX234290_BITFVAL(vol, ZX234290_LDO5_VSEL_LSH);
    mask = ZX234290_BITFMASK(ZX234290_LDO5_VSEL_WID, ZX234290_LDO5_VSEL_LSH);

    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}

T_ZDrvZx234290_VldoB zx234290_get_ldo5_voltage(void)
{
    unsigned char reg_addr=0, reg_val=0;
    T_ZDrvZx234290_VldoB vol;

    reg_addr = ZX234290_REG_ADDR_LDO56_VOL;
    zx234290_regulator_read_register(reg_addr, &reg_val);

    vol = ZX234290_BITFEXT(reg_val, ZX234290_LDO5_VSEL_WID, ZX234290_LDO5_VSEL_LSH);

    return vol;
}


int zx234290_set_ldo5_sleep_mode(T_ZDrvZx234290_LDOB_SLPMODE status)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    if(status > LDOB_SLPMODE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_LDO5678_MODE;

    reg_val = ZX234290_BITFVAL(status, ZX234290_LDO5_SLPMODE_LSH);
    mask = ZX234290_BITFMASK(ZX234290_REGULATOR_MODE_WID, ZX234290_LDO5_SLPMODE_LSH);
    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);

    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}



T_ZDrvZx234290_LDOB_SLPMODE zx234290_get_ldo5_sleep_mode(void)
{
    unsigned char reg_addr=0, reg_val=0;
    T_ZDrvZx234290_LDOB_SLPMODE status = 0;

    reg_addr = ZX234290_REG_ADDR_LDO5678_MODE;

    zx234290_regulator_read_register(reg_addr, &reg_val);
    status = ZX234290_BITFEXT(reg_val, ZX234290_REGULATOR_MODE_WID, ZX234290_LDO1_SLPMODE_LSH);

    return status;
}
#endif


#if 1   /* LDO6 */

int zx234290_set_ldo6_onoff(T_ZDrvZx234290_LDO_ENABLE status)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    if(status > LDO_AVTICE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_LDO_EN1;

    reg_val = ZX234290_BITFVAL(status, ZX234290_LDO6_ON_LSH);
    mask = ZX234290_BITFMASK(ZX234290_LDOS_ON_WID, ZX234290_LDO6_ON_LSH);

    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}


T_ZDrvZx234290_LDO_ENABLE zx234290_get_ldo6_onoff(void)
{
    unsigned char reg_addr=0, reg_val=0;
    T_ZDrvZx234290_LDO_ENABLE status;

    reg_addr = ZX234290_REG_ADDR_LDO_EN1;
    zx234290_regulator_read_register(reg_addr, &reg_val);

    status = ZX234290_BITFEXT(reg_val, ZX234290_LDOS_ON_WID, ZX234290_LDO6_ON_LSH);

    return status;
}

int zx234290_set_ldo6_voltage(T_ZDrvZx234290_VldoD vol)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    if(vol > VLDOD_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_LDO56_VOL;

    reg_val = ZX234290_BITFVAL(vol, ZX234290_LDO6_VSEL_LSH);
    mask = ZX234290_BITFMASK(ZX234290_LDO_VSEL_WID, ZX234290_LDO6_VSEL_LSH);

    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}

T_ZDrvZx234290_VldoD zx234290_get_ldo6_voltage(void)
{
    unsigned char reg_addr=0, reg_val=0;
    T_ZDrvZx234290_VldoD vol;

    reg_addr = ZX234290_REG_ADDR_LDO56_VOL;
    zx234290_regulator_read_register(reg_addr, &reg_val);

    vol = ZX234290_BITFEXT(reg_val, ZX234290_LDO_VSEL_WID, ZX234290_LDO6_VSEL_LSH);

    return vol;
}

int zx234290_set_ldo6_sleep_mode(T_ZDrvZx234290_LDOB_SLPMODE status)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    if(status > LDOB_SLPMODE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_LDO5678_MODE;

    reg_val = ZX234290_BITFVAL(status, ZX234290_LDO6_SLPMODE_LSH);
    mask = ZX234290_BITFMASK(ZX234290_REGULATOR_MODE_WID, ZX234290_LDO6_SLPMODE_LSH);
    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);

    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}

T_ZDrvZx234290_LDOB_SLPMODE zx234290_get_ldo6_sleep_mode(void)
{
    unsigned char reg_addr=0, reg_val=0;
    T_ZDrvZx234290_LDOB_SLPMODE status = 0;

    reg_addr = ZX234290_REG_ADDR_LDO5678_MODE;

    zx234290_regulator_read_register(reg_addr, &reg_val);
    status = ZX234290_BITFEXT(reg_val, ZX234290_REGULATOR_MODE_WID, ZX234290_LDO6_SLPMODE_LSH);

    return status;
}
#endif
#if 1   /* LDO7 add*/

int zx234290_set_ldo7_onoff(T_ZDrvZx234290_LDO_ENABLE status)
{
    int ret = 0;
    unsigned char  reg_addr=0, reg_val=0, mask=0;

    if(status > LDO_AVTICE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_LDO_EN1;

    reg_val = ZX234290_BITFVAL(status, ZX234290_LDO7_ON_LSH);
    mask = ZX234290_BITFMASK(ZX234290_LDOS_ON_WID, ZX234290_LDO7_ON_LSH);

    ret = zx234290_regulator_write_register( reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }
    return 0;
}

T_ZDrvZx234290_LDO_ENABLE zx234290_get_ldo7_onoff(void)
{
    unsigned char  reg_addr=0, reg_val=0;
    T_ZDrvZx234290_LDO_ENABLE status;

    reg_addr = ZX234290_REG_ADDR_LDO_EN1;
    zx234290_regulator_read_register( reg_addr, &reg_val);

    status = ZX234290_BITFEXT(reg_val, ZX234290_LDOS_ON_WID, ZX234290_LDO7_ON_LSH);  /*  */

    return status;
}

/* עͣѹΧ*/
int zx234290_set_ldo7_voltage(T_ZDrvZx234290_VldoD vol)
{
    int ret = 0;
    unsigned char  reg_addr=0, reg_val=0, mask=0;

    if(vol > VLDOD_MAX)
    {
        return -EINVAL;
    }
    reg_addr = ZX234290_REG_ADDR_LDO78_VOL;

    reg_val = ZX234290_BITFVAL(vol, ZX234290_LDO7_VSEL_LSH);
    mask = ZX234290_BITFMASK(ZX234290_LDO_VSEL_WID, ZX234290_LDO7_VSEL_LSH);

    ret = zx234290_regulator_write_register( reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }
    return 0;
}

T_ZDrvZx234290_VldoD zx234290_get_ldo7_voltage(void)
{
    unsigned char  reg_addr=0, reg_val=0;
    T_ZDrvZx234290_VldoD vol;

    reg_addr = ZX234290_REG_ADDR_LDO78_VOL;
    zx234290_regulator_read_register( reg_addr, &reg_val);

    vol = ZX234290_BITFEXT(reg_val, ZX234290_LDO_VSEL_WID, ZX234290_LDO7_VSEL_LSH);

    return vol;
}

int zx234290_set_ldo7_sleep_mode(T_ZDrvZx234290_LDOA_SLPMODE status)
{
    int ret = 0;
    unsigned char  reg_addr=0, reg_val=0, mask=0;

    if(status > LDOA_SLPMODE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_LDO5678_MODE;

    reg_val = ZX234290_BITFVAL(status, ZX234290_LDO7_SLPMODE_LSH);
    mask = ZX234290_BITFMASK(ZX234290_REGULATOR_MODE_WID, ZX234290_LDO7_SLPMODE_LSH);
    ret = zx234290_regulator_write_register( reg_addr, reg_val, mask);

    if (ret != 0)
    {
        return -EIO;
    }
    return 0;
}

T_ZDrvZx234290_LDOA_SLPMODE zx234290_get_ldo7_sleep_mode(void)
{
    unsigned char  reg_addr=0, reg_val=0;
    T_ZDrvZx234290_LDOA_SLPMODE status = 0;

    reg_addr = ZX234290_REG_ADDR_LDO5678_MODE;

    zx234290_regulator_read_register( reg_addr, &reg_val);
    status = ZX234290_BITFEXT(reg_val, ZX234290_REGULATOR_MODE_WID, ZX234290_LDO7_SLPMODE_LSH);

    return status;
}
#endif


#if 1   /* LDO8 */
/*******************************************************************************
 * Function:
 * Description:
 * Parameters:
 *   Input:
 *
 *   Output:
 *
 * Returns:
 *
 *
 * Others:
 ********************************************************************************/

int zx234290_set_ldo8_onoff(T_ZDrvZx234290_LDO_ENABLE status)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    if(status > LDO_AVTICE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_LDO_EN1;

    reg_val = ZX234290_BITFVAL(status, ZX234290_LDO8_ON_LSH);
    mask = ZX234290_BITFMASK(ZX234290_LDOS_ON_WID, ZX234290_LDO8_ON_LSH);

    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}

T_ZDrvZx234290_LDO_ENABLE zx234290_get_ldo8_onoff(void)
{
    unsigned char reg_addr=0, reg_val=0;
    T_ZDrvZx234290_LDO_ENABLE status;

    reg_addr = ZX234290_REG_ADDR_LDO_EN1;
    zx234290_regulator_read_register(reg_addr, &reg_val);

    status = ZX234290_BITFEXT(reg_val, ZX234290_LDOS_ON_WID, ZX234290_LDO8_ON_LSH);  /*  */

    return status;
}

int zx234290_set_ldo8_voltage(T_ZDrvZx234290_VldoD vol)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    if(vol > VLDOD_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_LDO78_VOL;

    reg_val = ZX234290_BITFVAL(vol, ZX234290_LDO8_VSEL_LSH);
    mask = ZX234290_BITFMASK(ZX234290_LDO_VSEL_WID, ZX234290_LDO8_VSEL_LSH);

    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}

T_ZDrvZx234290_VldoD zx234290_get_ldo8_voltage(void)
{
    unsigned char reg_addr=0, reg_val=0;
    T_ZDrvZx234290_VldoD vol;

    reg_addr = ZX234290_REG_ADDR_LDO78_VOL;
    zx234290_regulator_read_register(reg_addr, &reg_val);

    vol = ZX234290_BITFEXT(reg_val, ZX234290_LDO_VSEL_WID, ZX234290_LDO8_VSEL_LSH);

    return vol;
}

int zx234290_set_ldo8_sleep_mode(T_ZDrvZx234290_LDOA_SLPMODE status)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    if(status > LDOA_SLPMODE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_LDO5678_MODE;

    reg_val = ZX234290_BITFVAL(status, ZX234290_LDO8_SLPMODE_LSH);
    mask = ZX234290_BITFMASK(ZX234290_REGULATOR_MODE_WID, ZX234290_LDO8_SLPMODE_LSH);
    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);

    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}


T_ZDrvZx234290_LDOA_SLPMODE zx234290_get_ldo8_sleep_mode(void)
{
    unsigned char reg_addr=0, reg_val=0;
    T_ZDrvZx234290_LDOA_SLPMODE status = 0;

    reg_addr = ZX234290_REG_ADDR_LDO5678_MODE;

    zx234290_regulator_read_register(reg_addr, &reg_val);
    status = ZX234290_BITFEXT(reg_val, ZX234290_REGULATOR_MODE_WID, ZX234290_LDO8_SLPMODE_LSH);

    return status;
}
#endif
#if 1   /* LDO9 add*/

int zx234290_set_ldo9_onoff(T_ZDrvZx234290_LDO_ENABLE status)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;    /*  4     */

    if(status >= LDO_AVTICE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_LDO_EN2;

	if (pmu_chip_ver == 297) {
	    reg_val = ZX234290_BITFVAL(status, ZX234297_LDO9_ON_LSH);
	    mask = ZX234290_BITFMASK(ZX234290_LDOS_ON_WID, ZX234297_LDO9_ON_LSH);
	} else {
	    reg_val = ZX234290_BITFVAL(status, ZX234290_LDO9_ON_LSH);
	    mask = ZX234290_BITFMASK(ZX234290_LDOS_ON_WID, ZX234290_LDO9_ON_LSH);
	}

    ret = zx234290_regulator_write_register( reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }
    return 0;
}

T_ZDrvZx234290_LDO_ENABLE zx234290_get_ldo9_onoff(void)
{
    unsigned char  reg_addr=0, reg_val=0;
    T_ZDrvZx234290_LDO_ENABLE status;

    reg_addr = ZX234290_REG_ADDR_LDO_EN2;
    zx234290_regulator_read_register( reg_addr, &reg_val);

	if (pmu_chip_ver == 297) {
    	status = ZX234290_BITFEXT(reg_val, ZX234290_LDOS_ON_WID, ZX234297_LDO9_ON_LSH);
	} else {
    	status = ZX234290_BITFEXT(reg_val, ZX234290_LDOS_ON_WID, ZX234290_LDO9_ON_LSH);
	}

    return status;
}

/* עͣ LDO1/9/10 - LDOA */
int zx234290_set_ldo9_voltage(T_ZDrvZx234290_VldoA vol)
{
    int ret = 0;
    unsigned char  reg_addr=0, reg_val=0, mask=0;

    reg_addr = ZX234290_REG_ADDR_LDO9_VOL;

    reg_val = ZX234290_BITFVAL(vol, ZX234290_LDO9_VSEL_LSH);
    mask = ZX234290_BITFMASK(ZX234290_LDO_VSEL_WID, ZX234290_LDO9_VSEL_LSH);

    ret = zx234290_regulator_write_register( reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }
    return 0;
}

T_ZDrvZx234290_VldoA zx234290_get_ldo9_voltage(void)
{
    unsigned char  reg_addr=0, reg_val=0;
    T_ZDrvZx234290_VldoA vol;

    reg_addr = ZX234290_REG_ADDR_LDO9_VOL;
    zx234290_regulator_read_register( reg_addr, &reg_val);

    vol = ZX234290_BITFEXT(reg_val, ZX234290_LDO_VSEL_WID, ZX234290_LDO9_VSEL_LSH);

    return vol;
}

int zx234290_set_ldo9_sleep_mode(T_ZDrvZx234290_LDOA_SLPMODE status)
{
    int ret = 0;
    unsigned char  reg_addr=0, reg_val=0, mask=0;

    if(status >= LDOA_SLPMODE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_LDO910_MODE;

    reg_val = ZX234290_BITFVAL(status, ZX234290_LDO9_SLPMODE_LSH);
    mask = ZX234290_BITFMASK(ZX234290_REGULATOR_MODE_WID, ZX234290_LDO9_SLPMODE_LSH);
    ret = zx234290_regulator_write_register( reg_addr, reg_val, mask);

    if (ret != 0)
    {
        return -EIO;
    }
    return 0;
}

T_ZDrvZx234290_LDOA_SLPMODE zx234290_get_ldo9_sleep_mode(void)
{
    unsigned char  reg_addr=0, reg_val=0;
    T_ZDrvZx234290_LDOA_SLPMODE status = 0;

    reg_addr = ZX234290_REG_ADDR_LDO910_MODE;

    zx234290_regulator_read_register( reg_addr, &reg_val);
    status = ZX234290_BITFEXT(reg_val, ZX234290_REGULATOR_MODE_WID, ZX234290_LDO9_SLPMODE_LSH);

    return status;
}
#endif
#if 1 /* LDO10 add*/
int zx234290_set_ldo10_onoff(T_ZDrvZx234290_LDO_ENABLE status)
{
    int ret = 0;
    unsigned char  reg_addr=0, reg_val=0, mask=0;

    if(status >= LDO_AVTICE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_LDO_EN2;
	if (pmu_chip_ver == 297) {
	    reg_val = ZX234290_BITFVAL(status, ZX234297_LDO10_ON_LSH);
	    mask = ZX234290_BITFMASK(ZX234290_LDOS_ON_WID, ZX234297_LDO10_ON_LSH);
	} else {
	    reg_val = ZX234290_BITFVAL(status, ZX234290_LDO10_ON_LSH);
	    mask = ZX234290_BITFMASK(ZX234290_LDOS_ON_WID, ZX234290_LDO10_ON_LSH);
	}

    ret = zx234290_regulator_write_register( reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}

T_ZDrvZx234290_LDO_ENABLE zx234290_get_ldo10_onoff(void)
{
    unsigned char  reg_addr=0, reg_val=0;
    T_ZDrvZx234290_LDO_ENABLE status;

    reg_addr = ZX234290_REG_ADDR_LDO_EN2;
    zx234290_regulator_read_register( reg_addr, &reg_val);

	if (pmu_chip_ver == 297) {
	    status = ZX234290_BITFEXT(reg_val, ZX234290_LDOS_ON_WID, ZX234297_LDO10_ON_LSH);
	} else {
	    status = ZX234290_BITFEXT(reg_val, ZX234290_LDOS_ON_WID, ZX234290_LDO10_ON_LSH);
	}

    return status;
}

static int zx234290_set_ldo10_voltage(unsigned char vol)
{
    int ret = 0;
    unsigned char  reg_addr=0, reg_val=0, mask=0;

    reg_addr = ZX234290_REG_ADDR_LDO10_RTCLDO_VOL;

    reg_val = ZX234290_BITFVAL(vol, ZX234290_LDO10_VSEL_LSH);
    mask = ZX234290_BITFMASK(ZX234290_LDO_VSEL_WID, ZX234290_LDO10_VSEL_LSH);

    ret = zx234290_regulator_write_register( reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }
    return 0;

}

/* עͣ zx234296 LDO1/9/10 - LDOA */
int zx234290_set_ldo10_voltageA(T_ZDrvZx234290_VldoA vol)
{
    if (vol >= VLDOA_MAX)
    {
        return -EINVAL;
    }

	return zx234290_set_ldo10_voltage((unsigned char)vol);
}

/* עͣ zx234297 LDO10 - LDOF */
int zx234297_set_ldo10_voltageF(T_ZDrvZx234297_VldoF vol)
{
    if (vol >= VLDOF_MAX)
    {
        return -EINVAL;
    }

	return zx234290_set_ldo10_voltage((unsigned char)vol);
}

static unsigned char zx234290_get_ldo10_voltage(void)
{
    unsigned char  reg_addr=0, reg_val=0;

    reg_addr = ZX234290_REG_ADDR_LDO10_RTCLDO_VOL;
    zx234290_regulator_read_register( reg_addr, &reg_val);

    return ZX234290_BITFEXT(reg_val, ZX234290_LDO_VSEL_WID, ZX234290_LDO10_VSEL_LSH);
}

T_ZDrvZx234290_VldoA zx234290_get_ldo10_voltageA(void)
{
	unsigned char vol;

	vol = zx234290_get_ldo10_voltage();

	return (T_ZDrvZx234290_VldoA)vol;
}

T_ZDrvZx234297_VldoF zx234290_get_ldo10_voltageF(void)
{
	unsigned char vol;

	vol = zx234290_get_ldo10_voltage();

	return (T_ZDrvZx234297_VldoF)vol;
}

int zx234290_set_ldo10_sleep_mode(T_ZDrvZx234290_LDOA_SLPMODE status)
{
    int ret = 0;
    unsigned char  reg_addr=0, reg_val=0, mask=0;

    if(status >= LDOA_SLPMODE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_LDO910_MODE;

    reg_val = ZX234290_BITFVAL(status, ZX234290_LDO10_SLPMODE_LSH);
    mask = ZX234290_BITFMASK(ZX234290_REGULATOR_MODE_WID, ZX234290_LDO10_SLPMODE_LSH);
    ret = zx234290_regulator_write_register( reg_addr, reg_val, mask);

    if (ret != 0)
    {
        return -EIO;
    }
    return 0;
}

T_ZDrvZx234290_LDOA_SLPMODE zx234290_get_ldo10_sleep_mode(void)
{
    unsigned char  reg_addr=0, reg_val=0;
    T_ZDrvZx234290_LDOA_SLPMODE status = 0;

    reg_addr = ZX234290_REG_ADDR_LDO910_MODE;

    zx234290_regulator_read_register( reg_addr, &reg_val);
    status = ZX234290_BITFEXT(reg_val, ZX234290_REGULATOR_MODE_WID, ZX234290_LDO10_SLPMODE_LSH);

    return status;
}
#endif

#if 1   /* LDO COIN add*/
int zx234290_setVortc(T_ZDrvZx234290_VldoE vol)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    reg_addr = ZX234290_REG_ADDR_LDO10_RTCLDO_VOL;

    reg_val = ZX234290_BITFVAL(vol, ZX234290_VORTC_VSEL_LSH);
    mask = ZX234290_BITFMASK(ZX234290_VORTC_VSEL_WID, ZX234290_VORTC_VSEL_LSH);

    ret = zx234290_regulator_write_register( reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }
    return 0;

}

T_ZDrvZx234290_VldoE zx234290_getVortc(void)
{
    unsigned char  reg_addr=0, reg_val=0;
    T_ZDrvZx234290_VldoE vol;

    reg_addr = ZX234290_REG_ADDR_LDO10_RTCLDO_VOL;
    zx234290_regulator_read_register( reg_addr, &reg_val);

    vol = ZX234290_BITFEXT(reg_val, ZX234290_VORTC_VSEL_WID, ZX234290_VORTC_VSEL_LSH);

    return vol;
}
#endif

#if 1   /* Sink Control */
int zx234297_set_sink(T_ZDrvZx234297_SINK sink_num, int is_on, T_ZDrvZx234297_SINK_CURRENT sink_current)
{
    int ret = 0;
	unsigned char lsh_on, lsh_current;
    unsigned char reg_addr=0, reg_val=0, mask=0;

	if (sink_num == ZX234297_SINK1) {
		lsh_on = ZX234297_SINK1_ON_LSH;
		lsh_current = ZX234297_SINK1_CURRENT_LSH;
	} else if (sink_num == ZX234297_SINK2) {
		lsh_on = ZX234297_SINK2_ON_LSH;
		lsh_current = ZX234297_SINK2_CURRENT_LSH;
	} else
		return -EINVAL;

	if (is_on) {
		if (sink_current >= SINK_CURRENT_MAX)
			sink_current = SINK_CURRENT_120MA;

		reg_addr = ZX234297_REG_ADDR_SINK_CONTROL;
	    reg_val = ZX234290_BITFVAL(sink_current, lsh_current);
	    mask = ZX234290_BITFMASK(ZX234297_SINK_CURRENT_WID, lsh_current);
	    ret = zx234290_regulator_write_register( reg_addr, reg_val, mask);
	    if (ret != 0)
	    {
	        return -EIO;
	    }
	}

	is_on = !!is_on;
    reg_addr = ZX234290_REG_ADDR_LDO_EN2;
    reg_val = ZX234290_BITFVAL(is_on, lsh_on);
    mask = ZX234290_BITFMASK(ZX234297_SINK_ON_WID, lsh_on);
    ret = zx234290_regulator_write_register( reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}

int zx234297_set_sink_slpmode(T_ZDrvZx234297_SINK sink_num, T_ZDrvZx234297_SINK_SLPMODE mode)
{
    int ret = 0;
	unsigned char lsh;
    unsigned char reg_addr=0, reg_val=0, mask=0;

	if (mode >= SLPMODE_MAX) {
		return -EINVAL;
	}
	if (sink_num == ZX234297_SINK1) {
		lsh = ZX234297_SINK1_SLP_MODE_LSH;
	} else if (sink_num == ZX234297_SINK2) {
		lsh = ZX234297_SINK2_SLP_MODE_LSH;
	} else
		return -EINVAL;

    reg_addr = ZX234290_REG_ADDR_LDO910_MODE;
    reg_val = ZX234290_BITFVAL(mode, lsh);
    mask = ZX234290_BITFMASK(ZX234297_SINK_ON_WID, lsh);
    ret = zx234290_regulator_write_register( reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}
#endif

#if 1
/*******************************************************************************
* Function:
* Description:  0, do forced shutdown when TLLP time up ;
                1, do re-start up procedure when TLLP time up. (default is 0)
* Parameters:
*    Input:
*
*    Output:
*
* Returns:
*
*
* Others:
********************************************************************************/
int zx234290_SetTllpToDo(T_ZDrvZx234290_LLP_TODO enable)
{
    int ret = 0;
    unsigned char  reg_addr=0, reg_val=0, mask=0;

    if(enable > LLP_TODO_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_PWRKEY_CONTROL1;

    reg_val = ZX234290_BITFVAL(enable, 3);
    mask = ZX234290_BITFMASK(1, 3);

    //ret = zDrvPmic_SetRegister(slv_addr, reg_addr, reg_val, mask);
    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}

/*******************************************************************************
* Function:
* Description: Long Long Pressed Enable : 0 is disable, 1 is enable long long pressed interrupt.(default is 0)
* Parameters:
*    Input:
*
*    Output:
*
* Returns:
*
*
* Others:
********************************************************************************/
int zx234290_SetLlpEnable(T_ZDrvZx234290_LLP_ENABLE enable)
{
    int ret = 0;
    unsigned char  reg_addr=0, reg_val=0, mask=0;

    if(enable > LLP_ENABLE_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_PWRKEY_CONTROL1;

    reg_val = ZX234290_BITFVAL(enable, 2);
    mask = ZX234290_BITFMASK(1, 2);

    //ret = zDrvPmic_SetRegister(slv_addr, reg_addr, reg_val, mask);
    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}

/*******************************************************************************
* Function:
* Description: Long Pressed Time, 00, 01, 10, 11: 1sec, 2sec, 3sec, 4sec (default is 2 sec)
* Parameters:
*    Input:
*
*    Output:
*
* Returns:
*
*
* Others:
********************************************************************************/
int zx234290_SetLpTime(T_ZDrvZx234290_LP_TIME time)
{
    int ret = 0;
    unsigned char  reg_addr=0, reg_val=0, mask=0;

    if(time > LP_TIME_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_PWRKEY_CONTROL1;

    reg_val = ZX234290_BITFVAL(time, 0);
    mask = ZX234290_BITFMASK(2, 0);

    //ret = zDrvPmic_SetRegister(slv_addr, reg_addr, reg_val, mask);
    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}

/*******************************************************************************
* Function:
* Description: RESTART_DLY[1:0]=00, 01, 10, 11:    0.25sec, 0.5sec, 1sec, 2sec (default is 1 sec)
* Parameters:
*    Input:
*
*    Output:
*
* Returns:
*
*
* Others:
********************************************************************************/
int zx234290_SetRestartDly(T_ZDrvZx234290_TIME_IT time)
{
    int ret = 0;
    unsigned char  reg_addr=0, reg_val=0, mask=0;

    if(time > LLP_DLY_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_PWRKEY_CONTROL2;

    reg_val = ZX234290_BITFVAL(time, 0);
    mask = ZX234290_BITFMASK(2, 0);

    //ret = zDrvPmic_SetRegister(slv_addr, reg_addr, reg_val, mask);
    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}


/*******************************************************************************
* Function:
* Description: TIME_LP[1:0]=00, 01, 10, 11:    6sec, 7sec, 8sec, 10sec
* Parameters:
*    Input:
*
*    Output:
*
* Returns:
*
*
* Others:
********************************************************************************/
int zx234290_SetLlpTime(T_ZDrvZx234290_LLP_TIME time)
{
    int ret = 0;
    unsigned char  reg_addr=0, reg_val=0, mask=0;

    if(time > LLP_TIME_MAX)
    {
        return -EINVAL;
    }

    reg_addr = ZX234290_REG_ADDR_PWRKEY_CONTROL2;

    reg_val = ZX234290_BITFVAL(time, 2);
    mask = ZX234290_BITFMASK(2, 2);

    //ret = zDrvPmic_SetRegister(slv_addr, reg_addr, reg_val, mask);
    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}

/*******************************************************************************
* Function:
* Description:EN_VO_DISCH 0: Disable LDO output discharge resistance. The output becomes high impedance.
              EN_VO_DISCH 1: Enable LDO output discharge resistance.
* Parameters:
*    Input:
*
*    Output:
*
* Returns:
*
*
* Others:
********************************************************************************/
int zx234290_SetDischarger(T_ZDrvZx234290_LdoDischarger ldo, T_ZDrvZx234290_DISCHARGER_ENABLE enable)
{
    int ret = 0;
    unsigned char  reg_addr=0, reg_val=0, mask=0;

    if( ldo < 8 )
    {
        reg_addr = ZX234290_REG_ADDR_EN_DISCH1;
    }
    else
    {
        ldo -= 8;//
        reg_addr = ZX234290_REG_ADDR_EN_DISCH2;
    }

    reg_val = ZX234290_BITFVAL(enable, ldo);
    mask = ZX234290_BITFMASK(1, ldo);
    //ret = zDrvPmic_SetRegister(slv_addr, reg_addr, reg_val, mask);
    ret = zx234290_regulator_write_register(reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}

int zDrvZx234290_LdoRstErr(void)
{
    int ret = 0;
    unsigned char reg_val=0,mask=0;

    reg_val = ZX234290_BITFVAL(1, ZX234290_LDO_RSTERR_LSH);
    mask = ZX234290_BITFMASK(ZX234290_LDO_RSTERR_WID, ZX234290_LDO_RSTERR_LSH);
    ret = zx234290_regulator_write_register(ZX234290_REG_ADDR_BUCK_FAULT_STATUS, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}

/*******************************************************************************
* Function: Zx234290_SetUserReg_PSM
* Description:
* Parameters:
*    Input:
*
*    Output:
*
* Returns:
*
*
* Others:
********************************************************************************/
int Zx234290_SetUserReg_PSM(unsigned char data)
{
    int ret = 0;
    unsigned char reg_addr=0, reg_val=0, mask=0;

    reg_addr = ZX234290_REG_ADDR_USER_RESERVED;
    mask     = 0xff;
    reg_val  = data;
    ret = zx234290_regulator_write_register_PSM(reg_addr, reg_val, mask);
    if (ret != 0)
    {
        return -EIO;
    }

    return 0;
}
int zx234290_setSoftOn(bool SoftOn)
{
    int ret = 0;
    unsigned char reg_addr=0;
    unsigned char reg_val = 0;
    unsigned char mask = 0;

    reg_addr = ZX234290_REG_ADDR_SYS_CTRL;

    mask     = 0x01 << 7;
    if(SoftOn)
    {
        reg_val  = 0x01 << 7;
    }

    ret = zx234290_regulator_write_register_PSM(reg_addr, reg_val, mask);

    return ret;
}

int zx234290_setSoftOn_PSM(bool SoftOn)
{
    int ret = 0;
    unsigned char reg_addr=0;
    unsigned char reg_val = 0;
    unsigned char mask = 0;

    reg_addr = ZX234290_REG_ADDR_SYS_CTRL;

    mask     = 0x01 << 7;
    if(SoftOn)
    {
        reg_val  = 0x01 << 7;
    }

    ret = zx234290_regulator_write_register_PSM(reg_addr, reg_val, mask);

    return ret;
}
int zx234290_getPoweronStatus(void)
{
    int ret = 0;
    unsigned char status = 0;
    unsigned char reg_addr=0;

    reg_addr = ZX234290_REG_ADDR_STSA;

     ret = zx234290_regulator_read_register(reg_addr, &status);
     if (ret != 0)
     {
         return -1;
     }

    return (status & (1<<5));
}
int zx234290_getPoweronStatus_PSM(void)
{
    int ret = 0;
    unsigned char status = 0;
    unsigned char reg_addr=0;

    reg_addr = ZX234290_REG_ADDR_STSA;

    ret = zx234290_regulator_read_register_PSM(reg_addr, &status);

     if (ret != 0)
     {
         return -1;
     }

    return (status & (1<<5));
}

#endif

#ifdef ZX234290_PWR_FAUL_PROCESS

static BLOCKING_NOTIFIER_HEAD(zx234290_ldo_fail_notifier_list);

int zx234290_register_client(struct notifier_block *nb)
{
	return blocking_notifier_chain_register(&zx234290_ldo_fail_notifier_list, nb);
}
EXPORT_SYMBOL(zx234290_register_client);

int zx234290_unregister_client(struct notifier_block *nb)
{
	return blocking_notifier_chain_unregister(&zx234290_ldo_fail_notifier_list, nb);
}
EXPORT_SYMBOL(zx234290_unregister_client);

int zx234290_notifier_call_chain(unsigned long val, void *v)
{
//	printk(KERN_INFO "zx234290_notifier_call_chain,val=%ld\n", val);
	return blocking_notifier_call_chain(&zx234290_ldo_fail_notifier_list, val, v);
}
EXPORT_SYMBOL_GPL(zx234290_notifier_call_chain);


static irqreturn_t zx234290_buck_faul_irq(int irq, void *id)
{
	u8 buck_sts=0, buck_mask=0;
	//u8 val=0;
	zx234290_regulator_read_register(ZX234290_REG_ADDR_BUCK_FAULT_STATUS,  &buck_sts);
	zx234290_regulator_read_register(ZX234290_REG_ADDR_BUCK_INT_MASK,  &buck_mask);
	zDrvZx234290_LdoRstErr();

	buck_sts &= ~buck_mask;
	if(buck_sts & 0x4 ){
		printk(KERN_ERR "zx234290_buck_faul_irq!!  ldo8 error~,  buckFail_irq=0x%x\n", buck_sts);
		//val =0x01; //clear faul flag;

		zx234290_set_ldo8_onoff(LDO_ENABLE_OFF);
		zx234290_set_ldo8_onoff(LDO_ENABLE_ON);

		//zx234290_regulator_write_register(ZX234290_REG_ADDR_BUCK_FAULT_STATUS, 0x1, 0x1);

		zx234290_notifier_call_chain(ZX234290_INT_BUCK_FAUL,NULL);
		WARN_ON(1);
	}
	else if(buck_sts & 0x8 )//ldo6
	{
		printk(KERN_ERR "zx234290_buck_faul_irq!!  ldo6 error~,  buckFail_irq=0x%x, clear irq\n", buck_sts);
		//zx234290_regulator_write_register(ZX234290_REG_ADDR_BUCK_FAULT_STATUS, 0x1, 0x1);
		WARN_ON(1);
	}
	else if(buck_sts & 0x80 )//BUCK4
	{
		printk(KERN_ERR "zx234290_buck_faul_irq!!  BUCK4 error~,  buckFail_irq=0x%x, clear irq\n", buck_sts);
		//zx234290_regulator_write_register(ZX234290_REG_ADDR_BUCK_FAULT_STATUS, 0x1, 0x1);
		WARN_ON(1);
	}
	else
	{
		printk(KERN_ERR "zx234290_buck_faul_irq!! buckFail_irq=0x%x,BUG()!!\n", buck_sts);
		BUG();
	}
	return 0;
}

static irqreturn_t zx234290_ldo_faul_irq(int irq, void *id)
{
	u8 ldo_sts=0, ldo_mask=0;
	zx234290_regulator_read_register( ZX234290_REG_ADDR_LDO_FAULT_STATUS, &ldo_sts);
	zx234290_regulator_read_register( ZX234290_REG_ADDR_LDO_INT_MASK, &ldo_mask);
	zDrvZx234290_LdoRstErr();

	ldo_sts &= ~ldo_mask;
	if(ldo_sts){
		printk(KERN_ERR "zx234290_ldo_faul_irq!! LDOFail_irq=0x%x,BUG()!!\n", ldo_sts);
		BUG();
	}
	return 0;
}

int zx234290_regulator_error_irq_request(struct zx234290 *zx234290)
{
	int ret = -1;
	//zx234290_rtc.zx234290->irq_base = PMIC_INT_START;
	//printk(KERN_INFO"zx234290_PMU_regulator_init, BUCKFail_irq=%d,LDOFail_irq=%d\n", zx234290->irq_base + ZX234290_INT_BUCK_FAUL,zx234290->irq_base + ZX234290_INT_LDO_FAUL);
	ret = request_threaded_irq(zx234290->irq_base + ZX234290_INT_BUCK_FAUL, NULL, zx234290_buck_faul_irq,	0, "zx234290-regulator buck faul", zx234290);
	if (ret)	{
		printk(KERN_ERR "buck faul IRQ%d error %d\n", zx234290->irq_base + ZX234290_INT_BUCK_FAUL, ret);
		goto err_buck_irq;
	}
	ret = request_threaded_irq(zx234290->irq_base + ZX234290_INT_LDO_FAUL, NULL, zx234290_ldo_faul_irq,	0, "zx234290-regulator ldo faul", zx234290);
	if (ret)	{
		printk(KERN_ERR"ldo faul IRQ%d error %d\n", zx234290->irq_base + ZX234290_INT_LDO_FAUL, ret);
		goto err_ldo_irq;
	}

	return 0;

	err_ldo_irq:
		free_irq(zx234290->irq_base + ZX234290_INT_LDO_FAUL, zx234290);
	err_buck_irq:
		free_irq(zx234290->irq_base + ZX234290_INT_BUCK_FAUL, zx234290);
		return ret;
}
#endif

