/**
 * aicwf_sdmmc.c
 *
 * SDIO function declarations
 *
 * Copyright (C) AICSemi 2018-2020
 */
#include <linux/interrupt.h>
#include <linux/completion.h>
#include <linux/mmc/sdio.h>
#include <linux/mmc/sdio_ids.h>
#include <linux/mmc/sdio_func.h>
#include <linux/mmc/card.h>
#include <linux/mmc/host.h>
#include <linux/semaphore.h>
#include <linux/debugfs.h>
#include <linux/kthread.h>
#include "aicwf_txrxif.h"
#include "aicwf_sdio.h"
#include "sdio_host.h"
#include "rwnx_defs.h"
#include "rwnx_platform.h"
#include "rwnx_msg_tx.h"
#ifdef CONFIG_INGENIC_T20
#include "mach/jzmmc.h"
#endif /* CONFIG_INGENIC_T20 */
extern uint8_t scanning;
extern bool_l func_flag;

#ifdef CONFIG_PLATFORM_ALLWINNER
void platform_wifi_power_off(void);
#endif

int tx_aggr_counter = 32;
module_param_named(tx_aggr_counter, tx_aggr_counter, int, 0644);

#ifdef CONFIG_TX_NETIF_FLOWCTRL
int tx_fc_low_water = AICWF_SDIO_TX_LOW_WATER;
module_param_named(tx_fc_low_water, tx_fc_low_water, int, 0644);
int tx_fc_high_water = AICWF_SDIO_TX_HIGH_WATER;
module_param_named(tx_fc_high_water, tx_fc_high_water, int, 0644);
#endif
#define SDIO_VENDOR_ID_AIC8801              0x5449
#define SDIO_VENDOR_ID_AIC8800DC            0xc8a1
#define SDIO_VENDOR_ID_AIC8800D80           0xc8a1
#define SDIO_DEVICE_ID_AIC8801				0x0145
#define SDIO_DEVICE_ID_AIC8800DC			0xc08d
#define SDIO_DEVICE_ID_AIC8800D80           0x0082
int aicwf_sdio_readb(struct aic_sdio_dev *sdiodev, uint regaddr, u8 *val)
{
    int ret;
    sdio_claim_host(sdiodev->func);
    *val = sdio_readb(sdiodev->func, regaddr, &ret);
    sdio_release_host(sdiodev->func);
    return ret;
}

int aicwf_sdio_readb_func2(struct aic_sdio_dev *sdiodev, uint regaddr, u8 *val)
{
    int ret;
    sdio_claim_host(sdiodev->func_msg);
    *val = sdio_readb(sdiodev->func_msg, regaddr, &ret);
    sdio_release_host(sdiodev->func_msg);
    return ret;
}

int aicwf_sdio_writeb(struct aic_sdio_dev *sdiodev, uint regaddr, u8 val)
{
    int ret;
    sdio_claim_host(sdiodev->func);
    sdio_writeb(sdiodev->func, val, regaddr, &ret);
    sdio_release_host(sdiodev->func);
    return ret;
}


int aicwf_sdio_writeb_func2(struct aic_sdio_dev *sdiodev, uint regaddr, u8 val)
{
    int ret;
    sdio_claim_host(sdiodev->func_msg);
    sdio_writeb(sdiodev->func_msg, val, regaddr, &ret);
    sdio_release_host(sdiodev->func_msg);
    return ret;
}

#ifdef CONFIG_TX_NETIF_FLOWCTRL
void aicwf_sdio_tx_netif_flowctrl(struct rwnx_hw *rwnx_hw, bool state)
{
    struct rwnx_vif *rwnx_vif;
    list_for_each_entry(rwnx_vif, &rwnx_hw->vifs, list) {
        if (! rwnx_vif->up)
            continue;
        if (state)
            netif_tx_stop_all_queues(rwnx_vif->ndev);//netif_stop_queue(rwnx_vif->ndev);
        else
            netif_tx_wake_all_queues(rwnx_vif->ndev);//netif_wake_queue(rwnx_vif->ndev);
    }
}
#endif

int aicwf_sdio_flow_ctrl_msg(struct aic_sdio_dev *sdiodev)
{
    int ret = -1;
    u8 fc_reg = 0;
    u32 count = 0;
    while (true) {
        ret = aicwf_sdio_readb(sdiodev, SDIOWIFI_FLOW_CTRL_REG, &fc_reg);
        if (ret) {
            return -1;
        }
        if ((fc_reg & SDIOWIFI_FLOWCTRL_MASK_REG) != 0) {
            ret = fc_reg & SDIOWIFI_FLOWCTRL_MASK_REG;
            if(ret > tx_aggr_counter){
				ret = tx_aggr_counter;
			}
            return ret;
        } else {
            if (count >= FLOW_CTRL_RETRY_COUNT) {
                ret = -fc_reg;
                break;
            }
            count++;
            if (count < 30)
                udelay(200);
            else if(count < 40)
                msleep(2);
            else
                msleep(10);
        }
    }
    return ret;
}
int aicwf_sdio_flow_ctrl(struct aic_sdio_dev *sdiodev)
{
    int ret = -1;
    u8 fc_reg = 0;
    u32 count = 0;

    while (true) {
        ret = aicwf_sdio_readb(sdiodev, SDIOWIFI_FLOW_CTRL_REG, &fc_reg);
        if (ret) {
            return -1;
        }

        if ((fc_reg & SDIOWIFI_FLOWCTRL_MASK_REG) > 2) {
            ret = fc_reg & SDIOWIFI_FLOWCTRL_MASK_REG;
            if(ret > tx_aggr_counter){
				ret = tx_aggr_counter;
			}
            return ret;
        } else {
            if (count >= FLOW_CTRL_RETRY_COUNT) {
                ret = -fc_reg;
                break;
            }
            count++;
            if (count < 30)
                udelay(200);
                //usleep_range(150, 200);
            else if(count < 40)
                msleep(2);
            else
                msleep(10);
        }
    }

    return ret;
}

int aicwf_sdio_send_msg(struct aic_sdio_dev *sdiodev, u8 *buf, uint count)
{
    int ret = 0;
if (func_flag){
    sdio_claim_host(sdiodev->func_msg);
    ret = sdio_writesb(sdiodev->func_msg, 7, buf, count);
    sdio_release_host(sdiodev->func_msg);
} else {
    sdio_claim_host(sdiodev->func);
    ret = sdio_writesb(sdiodev->func, 7, buf, count);
    sdio_release_host(sdiodev->func);
}
    return ret;
}


int aicwf_sdio_send_pkt(struct aic_sdio_dev *sdiodev, u8 *buf, uint count)
{
    int ret = 0;

    sdio_claim_host(sdiodev->func);
    ret = sdio_writesb(sdiodev->func, 7, buf, count);
    sdio_release_host(sdiodev->func);

    return ret;
}

#ifdef LESS_SKB
int aicwf_sdio_recv_pkt(struct aic_sdio_dev *sdiodev, struct rx_buff *rxbuff,
    u32 size, u8 msg)
{
    int ret;

    if ((!rxbuff->data) || (!size)) {
        return -EINVAL;;
    }

    if (func_flag){
        if(!msg) {
            sdio_claim_host(sdiodev->func);
            ret = sdio_readsb(sdiodev->func, rxbuff->data, 8, size);
            sdio_release_host(sdiodev->func);
        } else {
            sdio_claim_host(sdiodev->func_msg);
            ret = sdio_readsb(sdiodev->func_msg, rxbuff->data, 8, size);
            sdio_release_host(sdiodev->func_msg);
        }
    } else{
            sdio_claim_host(sdiodev->func);
            ret = sdio_readsb(sdiodev->func, rxbuff->data, 8, size);
            sdio_release_host(sdiodev->func);
    }

    if (ret < 0) {
        return ret;
    }
    rxbuff->len = size;
    return ret;
}
#else
int aicwf_sdio_recv_pkt(struct aic_sdio_dev *sdiodev, struct sk_buff *skbbuf,
    u32 size, u8 msg)
{
    int ret;

    if ((!skbbuf) || (!size)) {
        return -EINVAL;;
    }

if (func_flag){
    if(!msg) {
        sdio_claim_host(sdiodev->func);
        ret = sdio_readsb(sdiodev->func, skbbuf->data, 8, size);
        sdio_release_host(sdiodev->func);
    } else {
        sdio_claim_host(sdiodev->func_msg);
        ret = sdio_readsb(sdiodev->func_msg, skbbuf->data, 8, size);
        sdio_release_host(sdiodev->func_msg);
    }
} else{
        sdio_claim_host(sdiodev->func);
        ret = sdio_readsb(sdiodev->func, skbbuf->data, 8, size);
        sdio_release_host(sdiodev->func);
}
    if (ret < 0) {
        return ret;
    }
    skbbuf->len = size;

    return ret;
}

#endif

static int aicwf_sdio_chipmatch(struct aic_sdio_dev *sdio_dev, uint16_t vid, uint16_t did){
	if(vid == SDIO_VENDOR_ID_AIC8801 && did == SDIO_DEVICE_ID_AIC8801){
		sdio_dev->chipid = PRODUCT_ID_AIC8801;
		printk("%s USE AIC8801\r\n", __func__);
		return 0;
	}else if(vid == SDIO_VENDOR_ID_AIC8800DC && did == SDIO_DEVICE_ID_AIC8800DC){
		sdio_dev->chipid = PRODUCT_ID_AIC8800DC;
		printk("%s USE AIC8800DC\r\n", __func__);
		return 0;
	}else if(vid == SDIO_VENDOR_ID_AIC8800D80 && did == SDIO_DEVICE_ID_AIC8800D80){
		sdio_dev->chipid = PRODUCT_ID_AIC8800D80;
		printk("%s USE AIC8800D80\r\n", __func__);
		return 0;
	}else{
		return -1;
	}
}
struct aic_sdio_dev *g_sdiodev;
static int aicwf_sdio_probe(struct sdio_func *func,
    const struct sdio_device_id *id)
{
    struct mmc_host *host;
    struct aic_sdio_dev *sdiodev;
    struct aicwf_bus *bus_if;
    int err = -ENODEV;

    sdio_dbg("%s:%d\n", __func__, func->num);
    sdio_dbg("Class=%x\n", func->class);
    sdio_dbg("sdio vendor ID: 0x%04x\n", func->vendor);
    sdio_dbg("sdio device ID: 0x%04x\n", func->device);
    sdio_dbg("Function#: %d\n", func->num);
    host = func->card->host;
    if(func->num != 1) {
        return err;
    }

    bus_if = kzalloc(sizeof(struct aicwf_bus), GFP_KERNEL);
    if (!bus_if) {
        sdio_err("alloc bus fail\n");
        return -ENOMEM;
    }

    sdiodev = kzalloc(sizeof(struct aic_sdio_dev), GFP_KERNEL);
    if (!sdiodev) {
        sdio_err("alloc sdiodev fail\n");
        kfree(bus_if);
        return -ENOMEM;
    }

	err = aicwf_sdio_chipmatch(sdiodev, func->vendor, func->device);
    if (err < 0) {
        sdio_err("sdio chipmatch fail\n");
    }
    sdiodev->func = func;
    sdiodev->func_msg = func->card->sdio_func[1];
    sdiodev->bus_if = bus_if;
    bus_if->bus_priv.sdio = sdiodev;
    dev_set_drvdata(&sdiodev->func_msg->dev, bus_if);
    
    dev_set_drvdata(&func->dev, bus_if);
    sdiodev->dev = &func->dev;
    err = aicwf_sdio_func_init(sdiodev);
    if (err < 0) {
        sdio_err("sdio func init fail\n");
        goto fail;
    }

    if (aicwf_sdio_bus_init(sdiodev) == NULL) {
		sdio_err("sdio bus init fail\n");
		err = -1;
		goto fail;
	}

    host->caps |= MMC_CAP_NONREMOVABLE;
    if(aicwf_rwnx_sdio_platform_init(sdiodev) != 0){
		err = -2;
		goto fail;
	}
    aicwf_hostif_ready();

    g_sdiodev = sdiodev;
	sdio_dbg("aicwf_sdio_probe over.\n");
    return 0;
fail:
	aicwf_sdio_func_deinit(sdiodev);
	dev_set_drvdata(&func->dev, NULL);
	kfree(sdiodev);
	kfree(bus_if);
	aicwf_hostif_fail();
	return err;
}

extern atomic_t skb_used;
static void aicwf_sdio_remove(struct sdio_func *func)
{
    struct mmc_host *host;
    struct aicwf_bus *bus_if = NULL;
    struct aic_sdio_dev *sdiodev = NULL;

    sdio_dbg("%s\n", __func__);
#ifdef CONFIG_TEMP_PW
	if (timer_pending(&g_rwnx_plat->sdiodev->tp_timer)) {
		printk("%s del_timer\n", __func__);
		del_timer_sync(&g_rwnx_plat->sdiodev->tp_timer);
	}
	cancel_work_sync(&g_rwnx_plat->sdiodev->tp_work);
#endif
    host = func->card->host;
    host->caps &= ~MMC_CAP_NONREMOVABLE;
    bus_if = dev_get_drvdata(&func->dev);
    if (!bus_if) {
        return;
    }

    sdiodev = bus_if->bus_priv.sdio;
    if (!sdiodev) {
        return;
    }

	
    sdiodev->bus_if->state = BUS_DOWN_ST;
    aicwf_sdio_release(sdiodev);
    aicwf_sdio_func_deinit(sdiodev);
    dev_set_drvdata(&sdiodev->func->dev, NULL);
	printk("skb_used=%d\n", skb_used);
	int i;
	for(i=0; i<NX_NB_TXQ;i++){
		if(skb_queue_len(&sdiodev->rwnx_hw->txq[i].sk_list)!= 0)
			printk("i=%d, len=%d\n",skb_queue_len(&sdiodev->rwnx_hw->txq[i].sk_list));
		}
    wiphy_free(sdiodev->rwnx_hw->wiphy);
    kfree(sdiodev);
    kfree(bus_if);
    sdio_dbg("%s done\n", __func__);
#ifdef CONFIG_PLATFORM_ALLWINNER
    platform_wifi_power_off();
#endif
}

static int aicwf_sdio_suspend(struct device *dev)
{
    int ret = 0;
    struct aicwf_bus *bus_if = dev_get_drvdata(dev);
    struct aic_sdio_dev *sdiodev = bus_if->bus_priv.sdio;
    mmc_pm_flag_t sdio_flags;
    struct rwnx_vif *rwnx_vif, *tmp;

    sdio_dbg("%s\n", __func__);
    list_for_each_entry_safe(rwnx_vif, tmp, &sdiodev->rwnx_hw->vifs, list) {
        if(rwnx_vif->ndev)
            netif_device_detach(rwnx_vif->ndev);
    }

    sdio_flags = sdio_get_host_pm_caps(sdiodev->func);
    if (!(sdio_flags & MMC_PM_KEEP_POWER)) {
        return -EINVAL;
    }
    ret = sdio_set_host_pm_flags(sdiodev->func, MMC_PM_KEEP_POWER);
    if (ret) {
        return ret;
    }

    while (sdiodev->state == SDIO_ACTIVE_ST) {
        if (down_interruptible(&sdiodev->tx_priv->txctl_sema)) {
            continue;
        }
        #if defined(CONFIG_SDIO_PWRCTRL)
        aicwf_sdio_pwr_stctl(sdiodev, SDIO_SLEEP_ST);
        #endif
        up(&sdiodev->tx_priv->txctl_sema);
        break;
    }

    return 0;
}

static int aicwf_sdio_resume(struct device *dev)
{
    struct aicwf_bus *bus_if = dev_get_drvdata(dev);
    struct aic_sdio_dev *sdiodev = bus_if->bus_priv.sdio;
    struct rwnx_vif *rwnx_vif, *tmp;

    sdio_dbg("%s\n", __func__);
    list_for_each_entry_safe(rwnx_vif, tmp, &sdiodev->rwnx_hw->vifs, list) {
        if(rwnx_vif->ndev)
            netif_device_attach(rwnx_vif->ndev);
    }

    return 0;
}

static const struct sdio_device_id aicwf_sdmmc_ids[] = {
    {SDIO_DEVICE(SDIO_VENDOR_ID_AIC, SDIO_DEVICE_ID_AIC)},
    { },
};

MODULE_DEVICE_TABLE(sdio, aicwf_sdmmc_ids);

static const struct dev_pm_ops aicwf_sdio_pm_ops = {
    SET_SYSTEM_SLEEP_PM_OPS(aicwf_sdio_suspend, aicwf_sdio_resume)
};

static struct sdio_driver aicwf_sdio_driver = {
    .probe = aicwf_sdio_probe,
    .remove = aicwf_sdio_remove,
    .name = AICWF_SDIO_NAME,
    .id_table = aicwf_sdmmc_ids,
    .drv = {
        .pm = &aicwf_sdio_pm_ops,
    },
};

#ifdef CONFIG_NANOPI_M4
    extern int mmc_rescan_try_freq(struct mmc_host *host, unsigned freq);
    extern unsigned  aic_max_freqs;
    extern struct mmc_host* aic_host_drv;
    extern int __mmc_claim_host(struct mmc_host *host, atomic_t *abort);
    extern void mmc_release_host(struct mmc_host *host);
#endif
#ifdef CONFIG_PLATFORM_ALLWINNER
extern void sunxi_mmc_rescan_card(unsigned ids);
extern void sunxi_wlan_set_power(int on);
extern int sunxi_wlan_get_bus_index(void);

int platform_wifi_power_on(void)
{
	int ret=0;
	int wlan_bus_index=sunxi_wlan_get_bus_index();
	if(wlan_bus_index < 0)
		return wlan_bus_index;

	sunxi_wlan_set_power(1);
	mdelay(1000);
	sunxi_mmc_rescan_card(wlan_bus_index);

	printk("platform_wifi_power_on");

	return ret;
}

void platform_wifi_power_off(void)
{
	int wlan_bus_index = sunxi_wlan_get_bus_index();
    if(wlan_bus_index < 0) {
		printk("no wlan_bus_index\n");
		return ;
	}
	printk("power_off\n");
	sunxi_wlan_set_power(0);
    mdelay(100);
    //sunxi_mmc_rescan_card(wlan_bus_index);

    printk("platform_wifi_power_off");
}
#endif
void aicwf_sdio_register(void)
{
#ifdef CONFIG_PLATFORM_NANOPI
    extern_wifi_set_enable(0);
    mdelay(200);
    extern_wifi_set_enable(1);
    mdelay(200);
    sdio_reinit();
#endif /*CONFIG_PLATFORM_NANOPI*/

#ifdef CONFIG_INGENIC_T20
    jzmmc_manual_detect(1, 1);
#endif /* CONFIG_INGENIC_T20 */

#ifdef CONFIG_NANOPI_M4
    if(aic_host_drv->card == NULL){
        __mmc_claim_host(aic_host_drv,NULL);
        printk("aic: >>>mmc_rescan_try_freq, aic_max_freqs=%d\n",aic_max_freqs);
        #if 1 // limit clock for fpga
        aic_host_drv->f_max = 2000000;
        #endif
        printk("f_min=%d,f_max=%d,f_init=%d,ocr_avail=%x,ocr_avail_sdio=%x\r\n",
            aic_host_drv->f_min,aic_host_drv->f_max,aic_host_drv->f_init, aic_host_drv->ocr_avail,aic_host_drv->ocr_avail_sdio);
        printk("caps=0x%x,caps2=0x%x,ocr_avail_sd=%x,ocr_avail_mmc=%x\r\n",
            aic_host_drv->caps,aic_host_drv->caps2, aic_host_drv->ocr_avail_sd,aic_host_drv->ocr_avail_mmc);
        mmc_rescan_try_freq(aic_host_drv,aic_max_freqs);
        mmc_release_host(aic_host_drv);
    }
#endif
#ifdef CONFIG_PLATFORM_ALLWINNER
    platform_wifi_power_on();
#endif
    if (sdio_register_driver(&aicwf_sdio_driver)) {

    } else {
    	//may add mmc_rescan here
    }
}

void aicwf_sdio_exit(void)
{
    if(g_rwnx_plat && g_rwnx_plat->enabled && g_rwnx_plat->sdiodev){
		if(g_rwnx_plat->sdiodev->rwnx_hw)
        rwnx_platform_deinit(g_rwnx_plat->sdiodev->rwnx_hw);
	}

	sdio_unregister_driver(&aicwf_sdio_driver);
	func_flag = true;

#ifdef CONFIG_PLATFORM_AMLOGIC
	extern_wifi_set_enable(0);
#endif /*CONFIG_PLATFORM_AMLOGIC*/

#if 0
#ifdef CONFIG_PLATFORM_ROCKCHIP
	rockchip_wifi_set_carddetect(0);
	mdelay(200);
	rockchip_wifi_power(0);
	mdelay(200);
#endif /*CONFIG_PLATFORM_ROCKCHIP*/
#endif

	if(g_rwnx_plat){
		kfree(g_rwnx_plat);
	}
}

#if defined(CONFIG_SDIO_PWRCTRL)
int aicwf_sdio_wakeup(struct aic_sdio_dev *sdiodev)
{
    int ret = 0;
        int read_retry;
        int write_retry = 20;

    if (sdiodev->state == SDIO_SLEEP_ST) {
        down(&sdiodev->pwrctl_wakeup_sema);
        if (sdiodev->rwnx_hw->vif_started) {
            if(sdiodev->state == SDIO_ACTIVE_ST) {
                up(&sdiodev->pwrctl_wakeup_sema);
                return 0;
            }
            //sdio_dbg("w\n");
            while(write_retry) {
                ret = aicwf_sdio_writeb(sdiodev, SDIOWIFI_WAKEUP_REG, 1);
                if (ret) {
                    txrx_err("sdio wakeup fail\n");
                    ret = -1;
                } else {
                    read_retry=10;
                    while (read_retry) {
                        u8 val;
                        ret = aicwf_sdio_readb(sdiodev, SDIOWIFI_SLEEP_REG, &val);
                        if (ret==0 && val&0x10) {
                            break;
                        }
                        read_retry--;
                        udelay(200);
                    }
                    if(read_retry != 0)
                        break;
                }
                sdio_dbg("write retry:  %d \n",write_retry);
                write_retry--;
                udelay(100);
            }
        }

        sdiodev->state = SDIO_ACTIVE_ST;
        aicwf_sdio_pwrctl_timer(sdiodev, sdiodev->active_duration);
        up(&sdiodev->pwrctl_wakeup_sema);
    }
    return ret;
}

extern u8 dhcped;
int aicwf_sdio_sleep_allow(struct aic_sdio_dev *sdiodev)
{
    int ret = 0;
    struct aicwf_bus *bus_if = sdiodev->bus_if;
    struct rwnx_hw *rwnx_hw = sdiodev->rwnx_hw;

    if (bus_if->state == BUS_DOWN_ST) {
        ret = aicwf_sdio_writeb(sdiodev, SDIOWIFI_SLEEP_REG, 0x10);
        if (ret) {
            sdio_err("Write sleep fail!\n");
    }
        aicwf_sdio_pwrctl_timer(sdiodev, 0);
        return ret;
    }

    sdio_info("sleep: %d, %d\n", sdiodev->state, scanning);
    if (sdiodev->state == SDIO_ACTIVE_ST  && !scanning && !rwnx_hw->is_p2p_alive \
                && !rwnx_hw->is_p2p_connected) {
        down(&sdiodev->pwrctl_wakeup_sema);
        if (rwnx_hw->vif_started) {
            //sdio_dbg("s\n");
            ret = aicwf_sdio_writeb(sdiodev, SDIOWIFI_SLEEP_REG, 0x10);
            if (ret)
               	sdio_err("Write sleep fail!\n");
        }
        sdiodev->state = SDIO_SLEEP_ST;
        aicwf_sdio_pwrctl_timer(sdiodev, 0);
        up(&sdiodev->pwrctl_wakeup_sema);
    }
    else{
        aicwf_sdio_pwrctl_timer(sdiodev, sdiodev->active_duration);
    }

    return ret;
}

int aicwf_sdio_pwr_stctl(struct aic_sdio_dev *sdiodev, uint target)
{
    int ret = 0;

    if (sdiodev->bus_if->state == BUS_DOWN_ST) {
        return -1;
    }

    if (sdiodev->state == target) {
        if (target == SDIO_ACTIVE_ST) {
            aicwf_sdio_pwrctl_timer(sdiodev, sdiodev->active_duration);
        }
        return ret;
    }

    switch (target) {
    case SDIO_ACTIVE_ST:
        aicwf_sdio_wakeup(sdiodev);
        break;
    case SDIO_SLEEP_ST:
        aicwf_sdio_sleep_allow(sdiodev);
        break;
    }

    return ret;
}
#endif

int aicwf_sdio_txpkt(struct aic_sdio_dev *sdiodev, struct sk_buff *pkt)
{
    int ret = 0;
    u8 *frame;
    u32 len = 0;
    struct aicwf_bus *bus_if = dev_get_drvdata(sdiodev->dev);

    if (bus_if->state == BUS_DOWN_ST) {
        sdio_dbg("tx bus is down!\n");
        return -EINVAL;
    }

    frame = (u8 *) (pkt->data);
    len = pkt->len;
    len = (len + SDIOWIFI_FUNC_BLOCKSIZE - 1) / SDIOWIFI_FUNC_BLOCKSIZE * SDIOWIFI_FUNC_BLOCKSIZE;
    ret = aicwf_sdio_send_pkt(sdiodev, pkt->data, len);
    if (ret)
        sdio_err("aicwf_sdio_send_pkt fail%d\n", ret);

    return ret;
}

static int aicwf_sdio_intr_get_len_bytemode(struct aic_sdio_dev *sdiodev, u8 *byte_len)
{
    int ret = 0;

    if (!byte_len)
        return -EBADE;

    if (sdiodev->bus_if->state == BUS_DOWN_ST) {
        *byte_len = 0;
    } else {
        ret = aicwf_sdio_readb(sdiodev, SDIOWIFI_BYTEMODE_LEN_REG, byte_len);
        sdiodev->rx_priv->data_len = (*byte_len)*4;
    }

    return ret;
}

static void aicwf_sdio_bus_stop(struct device *dev)
{
    struct aicwf_bus *bus_if = dev_get_drvdata(dev);
    struct aic_sdio_dev *sdiodev = bus_if->bus_priv.sdio;
    int ret;

    #if defined(CONFIG_SDIO_PWRCTRL)
    aicwf_sdio_pwrctl_timer(sdiodev, 0);
    #endif
    if(timer_pending(&sdiodev->rwnx_hw->p2p_alive_timer)){
        ret = del_timer(&sdiodev->rwnx_hw->p2p_alive_timer);}
    sdio_dbg("%s\n",__func__);
    #if defined(CONFIG_SDIO_PWRCTRL)
    if (sdiodev->pwrctl_tsk) {
        complete(&sdiodev->pwrctrl_trgg);
        kthread_stop(sdiodev->pwrctl_tsk);
        sdiodev->pwrctl_tsk = NULL;
    }

    sdio_dbg("%s:pwrctl stopped\n",__func__);
    #endif

    bus_if->state = BUS_DOWN_ST;
    ret = down_interruptible(&sdiodev->tx_priv->txctl_sema);
    if (ret)
       sdio_err("down txctl_sema fail\n");

    #if defined(CONFIG_SDIO_PWRCTRL)
    aicwf_sdio_pwr_stctl(sdiodev, SDIO_SLEEP_ST);
    #endif
    if (!ret)
        up(&sdiodev->tx_priv->txctl_sema);
    spin_lock_bh(&sdiodev->tx_priv->txqlock);
    aicwf_txframe_queue_flush(&sdiodev->tx_priv->txq);
    atomic_set(&sdiodev->tx_priv->tx_pktcnt, 0);
    spin_unlock_bh(&sdiodev->tx_priv->txqlock);
    sdio_dbg("exit %s\n",__func__);
}

#ifdef LESS_SKB
struct rx_buff *aicwf_sdio_readframes(struct aic_sdio_dev *sdiodev, u8 msg)
{
    int ret = 0;
    u32 size = 0;
    struct aicwf_bus *bus_if = dev_get_drvdata(sdiodev->dev);
    struct rx_buff* rxbuff;
    
    if (bus_if->state == BUS_DOWN_ST) {
        sdio_dbg("bus down\n");
        return NULL;
    }
#if 0
    rxbuff = kmalloc(sizeof(struct rx_buff), GFP_KERNEL);
    if(rxbuff == NULL){
        printk("failed to alloc rxbuff\n");
        return NULL;
    }

    size = sdiodev->rx_priv->data_len;
	//printk("sdall:%d\n", atomic_read(&skb_used));

    rxbuff->data = kmalloc(size,GFP_KERNEL);
    if (rxbuff->data == NULL) {
        printk("failed to alloc rxbuff data\n");
        kfree(rxbuff);
        return NULL;
    }
#else
    size = sdiodev->rx_priv->data_len;
    rxbuff = aicwf_prealloc_rxbuff_alloc(&sdiodev->rx_priv->rxbuff_lock);
    if(rxbuff == NULL){
        printk("failed to alloc rxbuff\n");
        return NULL;
    }
#endif
    rxbuff->len = 0;
    rxbuff->start = rxbuff->data;
    rxbuff->read = rxbuff->start;
    rxbuff->end = rxbuff->data + size;

    ret = aicwf_sdio_recv_pkt(sdiodev, rxbuff, size, msg);
    if (ret) {
        printk("%s %d, sdio recv pkt fail", __func__, __LINE__);
#if 0
        rxbuff_free(rxbuff);
#else
        aicwf_prealloc_rxbuff_free(rxbuff, &sdiodev->rx_priv->rxbuff_lock);
#endif
        return NULL;
    }

    return rxbuff;
}
#else
struct sk_buff *aicwf_sdio_readframes(struct aic_sdio_dev *sdiodev, u8 msg)
{
    int ret = 0;
    u32 size = 0;
    struct sk_buff *skb = NULL;
    struct aicwf_bus *bus_if = dev_get_drvdata(sdiodev->dev);

    if (bus_if->state == BUS_DOWN_ST) {
        sdio_dbg("bus down\n");
        return NULL;
    }

    size = sdiodev->rx_priv->data_len;
    skb = __dev_alloc_skb(size, GFP_KERNEL);
    if (!skb) {
        return NULL;
    }

    ret = aicwf_sdio_recv_pkt(sdiodev, skb, size, msg);
    if (ret) {
        printk("%s %d, recv pkt error\n", __func__, __LINE__);
        dev_kfree_skb(skb);
        skb = NULL;
    }

    return skb;
}

#endif

static int aicwf_sdio_tx_msg(struct aic_sdio_dev *sdiodev)
{
    int err = 0;
    u16 len;
    u8 *payload = sdiodev->tx_priv->cmd_buf;
    u16 payload_len = sdiodev->tx_priv->cmd_len;
    u8 adjust_str[4] = {0, 0, 0, 0};
    int adjust_len = 0;
    int buffer_cnt = 0;
    u8 retry = 0;

    len = payload_len;
    if ((len % TX_ALIGNMENT) != 0) {
        adjust_len = roundup(len, TX_ALIGNMENT);
        memcpy(payload+payload_len, adjust_str, (adjust_len - len));
        payload_len += (adjust_len - len);
    }
    len = payload_len;

    //link tail is necessary
    if ((len % SDIOWIFI_FUNC_BLOCKSIZE) != 0) {
        memset(payload+payload_len, 0, TAIL_LEN);
        payload_len += TAIL_LEN;
        len = (payload_len/SDIOWIFI_FUNC_BLOCKSIZE + 1) * SDIOWIFI_FUNC_BLOCKSIZE;
    } else
        len = payload_len;

if (!func_flag){
    buffer_cnt = aicwf_sdio_flow_ctrl_msg(sdiodev);
    while ((buffer_cnt <= 0 || (buffer_cnt > 0 && len > (buffer_cnt * BUFFER_SIZE))) && retry < 10) {
        retry++;
        buffer_cnt = aicwf_sdio_flow_ctrl_msg(sdiodev);
    }
}
    printk("sdio txmsg: %d\n", buffer_cnt);
    down(&sdiodev->tx_priv->cmd_txsema);
    if(((!func_flag) && (buffer_cnt > 0 && len <= (buffer_cnt * BUFFER_SIZE))) || func_flag) {
        //err = aicwf_sdio_send_pkt(sdiodev, payload, len);
        err = aicwf_sdio_send_msg(sdiodev, payload, len);
        if (err) {
            sdio_err("aicwf_sdio_send_pkt fail%d\n", err);
        }
    } else {
        sdio_err("tx msg fc retry fail\n");
        up(&sdiodev->tx_priv->cmd_txsema);
        return -1;
    }

    sdiodev->tx_priv->cmd_txstate = false;
    if (!err)
        sdiodev->tx_priv->cmd_tx_succ= true;
    else
        sdiodev->tx_priv->cmd_tx_succ= false;

    up(&sdiodev->tx_priv->cmd_txsema);

    return err;
}

#ifndef CONFIG_SDIO_AGGR
static void aicwf_sdio_send_single_pkt(struct sk_buff *skb, struct aic_sdio_dev *sdiodev){
    struct aicwf_tx_priv *tx_priv=sdiodev->tx_priv;
    struct rwnx_txhdr *txhdr = (struct rwnx_txhdr *)(skb->data);
    u8 *startptr=((u8 *)txhdr->sdio_hdr);//sdiohdr
    u8 *payload = ((u8 *)txhdr->sdio_hdr);
    int payload_len=0;
    u8 adjust_str[4] = {0, 0, 0, 0};
    int allign_len = 0;
    u8 *old_tail=NULL;
    int headroom;
    
    txhdr->sdio_hdr[2] = 0x01; //data
    txhdr->sdio_hdr[3] = 0; //reserved
   
	struct txdesc_api *test =&txhdr->desc;
    payload_len=(skb->tail-startptr);
    
    if (payload_len & (TX_ALIGNMENT - 1)) {//payload_len include sdiohdr
        old_tail = skb->tail;
        allign_len = roundup(payload_len, TX_ALIGNMENT)-payload_len;
        skb_put(skb, allign_len);
        memset(old_tail, 0, allign_len);//expand to word
    }
    
    startptr[0] = ((skb->tail-startptr-4) & 0xff);
    startptr[1] = (((skb->tail-startptr-4) >> 8)&0x0f);
    
    if(( (skb->tail-startptr) % SDIOWIFI_FUNC_BLOCKSIZE)!=0){
    	old_tail=skb->tail;	
    	skb_put(skb, 4);
    	memset(old_tail, 0, 4);
    }
    
    int len=((skb->tail-startptr) + SDIOWIFI_FUNC_BLOCKSIZE - 1) / SDIOWIFI_FUNC_BLOCKSIZE * SDIOWIFI_FUNC_BLOCKSIZE;

    int ret = aicwf_sdio_send_pkt(sdiodev, (u8 *)(long)&txhdr->sdio_hdr , len);
    if (ret < 0) {
        sdio_err("fail to send single pkt!\n");
    }
    
    if(!txhdr->sw_hdr.need_cfm){
        headroom = txhdr->sw_hdr.headroom;
        //kmem_cache_free(txhdr->sw_hdr->rwnx_vif->rwnx_hw->sw_txhdr_cache, txhdr->sw_hdr);
        skb_pull(skb, headroom);
        consume_skb(skb);
    }

}
#endif//CONFIG_SDIO_AGGR
static void aicwf_sdio_tx_process(struct aic_sdio_dev *sdiodev)
{
    int err = 0;
#ifdef CONFIG_TX_NETIF_FLOWCTRL
    unsigned long flags;
#endif

    if (sdiodev->bus_if->state == BUS_DOWN_ST) {
        sdio_err("Bus is down\n");
        return;
    }

    #if defined(CONFIG_SDIO_PWRCTRL)
    aicwf_sdio_pwr_stctl(sdiodev, SDIO_ACTIVE_ST);
    #endif

    //config
    sdio_info("send cmd\n");
    if (sdiodev->tx_priv->cmd_txstate) {
        if (down_interruptible(&sdiodev->tx_priv->txctl_sema)) {
            txrx_err("txctl down bus->txctl_sema fail\n");
            return;
        }
        if (sdiodev->state != SDIO_ACTIVE_ST) {
            txrx_err("state err\n");
            up(&sdiodev->tx_priv->txctl_sema);
            txrx_err("txctl up bus->txctl_sema fail\n");
            return;
        }

        err = aicwf_sdio_tx_msg(sdiodev);
        up(&sdiodev->tx_priv->txctl_sema);
        if (waitqueue_active(&sdiodev->tx_priv->cmd_txdone_wait))
            wake_up(&sdiodev->tx_priv->cmd_txdone_wait);
    }

    //data
    sdio_info("send data\n");
    if (down_interruptible(&sdiodev->tx_priv->txctl_sema)) {
        txrx_err("txdata down bus->txctl_sema\n");
        return;
    }

    if (sdiodev->state != SDIO_ACTIVE_ST) {
        txrx_err("sdio state err\n");
        up(&sdiodev->tx_priv->txctl_sema);
        return;
    }

     if (!aicwf_is_framequeue_empty(&sdiodev->tx_priv->txq)){
        sdiodev->tx_priv->fw_avail_bufcnt = aicwf_sdio_flow_ctrl(sdiodev);
     }

    #ifdef CONFIG_SDIO_AGGR
	while (!aicwf_is_framequeue_empty(&sdiodev->tx_priv->txq)) {
		if(sdiodev->bus_if->state == BUS_DOWN_ST){
			break;
		}
	        if (sdiodev->tx_priv->fw_avail_bufcnt <= DATA_FLOW_CTRL_THRESH) {
        	    if (sdiodev->tx_priv->cmd_txstate)
                	break;
            	    sdiodev->tx_priv->fw_avail_bufcnt = aicwf_sdio_flow_ctrl(sdiodev);
        	} else {
            		if (sdiodev->tx_priv->cmd_txstate) {
                		aicwf_sdio_send(sdiodev->tx_priv, 1);
                		break;
            		} else {
                		aicwf_sdio_send(sdiodev->tx_priv, 0);
            		}
        	}
	}
    #else
     while (!aicwf_is_framequeue_empty(&sdiodev->tx_priv->txq)) {
        if(sdiodev->tx_priv->cmd_txstate || (sdiodev->bus_if->state == BUS_DOWN_ST)){
            break;
	    }
        if(sdiodev->tx_priv->fw_avail_bufcnt <= DATA_FLOW_CTRL_THRESH) {
			sdiodev->tx_priv->fw_avail_bufcnt = aicwf_sdio_flow_ctrl(sdiodev);
        } else {
        //spin_lock_bh(&sdiodev->tx_priv->txqlock);//old
	    struct sk_buff *skb=NULL;
            spin_lock_bh(&sdiodev->tx_priv->txqlock);
            skb = aicwf_txframe_dequeue(&sdiodev->tx_priv->txq);
        if (skb == NULL) {
            sdio_err("txq no pkt\n");
                spin_unlock_bh(&sdiodev->tx_priv->txqlock);
            break;
        }
        atomic_dec(&sdiodev->tx_priv->tx_pktcnt);
            spin_unlock_bh(&sdiodev->tx_priv->txqlock);
        
        aicwf_sdio_send_single_pkt(skb, sdiodev);
        sdiodev->tx_priv->fw_avail_bufcnt--;
	    }
    }

    #ifdef CONFIG_TX_NETIF_FLOWCTRL
        spin_lock_irqsave(&sdiodev->tx_flow_lock, flags);
        if (atomic_read(&sdiodev->tx_priv->tx_pktcnt) < tx_fc_low_water) {
            if (sdiodev->flowctrl) {
                sdiodev->flowctrl = 0;
                aicwf_sdio_tx_netif_flowctrl(sdiodev->rwnx_hw, false);
            }
        }
        spin_unlock_irqrestore(&sdiodev->tx_flow_lock, flags);
    #endif
    #endif

    up(&sdiodev->tx_priv->txctl_sema);
}

static int aicwf_sdio_bus_txdata(struct device *dev, struct sk_buff *pkt)
{
    uint prio;
    int ret = -EBADE;
    struct aicwf_bus *bus_if = dev_get_drvdata(dev);
    struct aic_sdio_dev *sdiodev = bus_if->bus_priv.sdio;
#ifdef CONFIG_TX_NETIF_FLOWCTRL
    unsigned long flags;
#endif

    prio = (pkt->priority & 0xF);
	if(prio >(AICWF_TXQ_CNT - 1)){
		//printk("prio eceed: %d\n", prio);
		prio = 0;
	}
    spin_lock_bh(&sdiodev->tx_priv->txqlock);
    if (!aicwf_txframe_enq(sdiodev->dev, &sdiodev->tx_priv->txq, pkt, prio)) {
        struct rwnx_txhdr *txhdr = (struct rwnx_txhdr *)pkt->data;
		int headroom = txhdr->sw_hdr.headroom;
		//kmem_cache_free(txhdr->sw_hdr->rwnx_vif->rwnx_hw->sw_txhdr_cache, txhdr->sw_hdr);
		skb_pull(pkt, headroom);
		consume_skb(pkt);
        spin_unlock_bh(&sdiodev->tx_priv->txqlock);
		ret = -ENOSR;
        goto flowctrl;
    } else {
        ret = 0;
    }

    if (bus_if->state != BUS_UP_ST) {
        sdio_err("bus_if stopped\n");
        spin_unlock_bh(&sdiodev->tx_priv->txqlock);
        return -1;
    }

    atomic_inc(&sdiodev->tx_priv->tx_pktcnt);
    spin_unlock_bh(&sdiodev->tx_priv->txqlock);
#ifdef HANDLE_TX_THREAD_ZTE
//#ifdef CONFIG_SDIO_AGGR
    if (atomic_read(&sdiodev->tx_priv->tx_pktcnt) == 1)
//#endif
    complete(&bus_if->bustx_trgg);
#else
    tasklet_schedule(&sdiodev->tx_priv->tx_tasklet);
#endif
	flowctrl:
#ifdef CONFIG_TX_NETIF_FLOWCTRL
    spin_lock_irqsave(&sdiodev->tx_flow_lock, flags);
    if (atomic_read(&sdiodev->tx_priv->tx_pktcnt) >= tx_fc_high_water) {
        if (!sdiodev->flowctrl) {
            sdiodev->flowctrl = 1;
            aicwf_sdio_tx_netif_flowctrl(sdiodev->rwnx_hw, true);
        }
    }
    spin_unlock_irqrestore(&sdiodev->tx_flow_lock, flags);
#endif

    return ret;
}

static int aicwf_sdio_bus_txmsg(struct device *dev, u8 *msg, uint msglen)
{
    int ret = -1;
    struct aicwf_bus *bus_if = dev_get_drvdata(dev);
    struct aic_sdio_dev *sdiodev = bus_if->bus_priv.sdio;

    down(&sdiodev->tx_priv->cmd_txsema);
    sdiodev->tx_priv->cmd_txstate = true;
    sdiodev->tx_priv->cmd_tx_succ = false;
    sdiodev->tx_priv->cmd_buf = msg;
    sdiodev->tx_priv->cmd_len = msglen;
    up(&sdiodev->tx_priv->cmd_txsema);

    if (bus_if->state != BUS_UP_ST) {
        sdio_err("bus has stop\n");
        return -1;
    }

#ifdef HANDLE_TX_THREAD_ZTE
    complete(&bus_if->bustx_trgg);
#else
        tasklet_schedule(&sdiodev->tx_priv->tx_tasklet);
#endif
#if 0
if (!func_flag){
    if (sdiodev->tx_priv->cmd_txstate) {
        int timeout = msecs_to_jiffies(CMD_TX_TIMEOUT);
        ret = wait_event_interruptible_timeout(sdiodev->tx_priv->cmd_txdone_wait, \
                                            !(sdiodev->tx_priv->cmd_txstate), timeout);
    }

    if (!sdiodev->tx_priv->cmd_txstate && sdiodev->tx_priv->cmd_tx_succ) {
        ret = 0;
    } else {
        sdio_err("send faild:%d, %d,%x\n", sdiodev->tx_priv->cmd_txstate, sdiodev->tx_priv->cmd_tx_succ, ret);
        ret = -EIO;
    }

    return ret;
}
#endif
    return 0;
}

#ifdef CONFIG_SDIO_AGGR
int aicwf_sdio_send(struct aicwf_tx_priv *tx_priv, u8 txnow)
{
	struct sk_buff *pkt;
	struct aic_sdio_dev *sdiodev = tx_priv->sdiodev;
	u32 aggr_len = 0;
#ifdef CONFIG_TX_NETIF_FLOWCTRL
    unsigned long flags;
#endif
	aggr_len = (tx_priv->tail - tx_priv->head);
	if (((atomic_read(&tx_priv->aggr_count) == 0) && (aggr_len != 0))
		|| ((atomic_read(&tx_priv->aggr_count) != 0) && (aggr_len == 0))) {
		if (aggr_len > 0)
			aicwf_sdio_aggrbuf_reset(tx_priv);
		goto done;
	}

	if (atomic_read(&tx_priv->aggr_count) == (tx_priv->fw_avail_bufcnt - 2)) {
		if (atomic_read(&tx_priv->aggr_count) > 0) {
			tx_priv->fw_avail_bufcnt -= atomic_read(&tx_priv->aggr_count);
			aicwf_sdio_aggr_send(tx_priv); //send and check the next pkt;
		}
	} else {
		spin_lock_bh(&sdiodev->tx_priv->txqlock);
		pkt = aicwf_txframe_dequeue(&sdiodev->tx_priv->txq);
		if (pkt == NULL) {
			sdio_err("txq no pkt\n");
			spin_unlock_bh(&sdiodev->tx_priv->txqlock);
			goto done;
		}
		atomic_dec(&sdiodev->tx_priv->tx_pktcnt);
		spin_unlock_bh(&sdiodev->tx_priv->txqlock);
#ifdef CONFIG_TX_NETIF_FLOWCTRL
        spin_lock_irqsave(&sdiodev->tx_flow_lock, flags);
        if (atomic_read(&sdiodev->tx_priv->tx_pktcnt) < tx_fc_low_water) {
            if (sdiodev->flowctrl) {
                sdiodev->flowctrl = 0;
                aicwf_sdio_tx_netif_flowctrl(sdiodev->rwnx_hw, false);
            }
        }
        spin_unlock_irqrestore(&sdiodev->tx_flow_lock, flags);
#endif

		if (tx_priv == NULL || tx_priv->tail == NULL || pkt == NULL)
			txrx_err("null error\n");
		if (aicwf_sdio_aggr(tx_priv, pkt)) {
			aicwf_sdio_aggrbuf_reset(tx_priv);
			sdio_err("add aggr pkts failed!\n");
			goto done;
		}

		//when aggr finish or there is cmd to send, just send this aggr pkt to fw
		if ((int)atomic_read(&sdiodev->tx_priv->tx_pktcnt) == 0 || txnow || (atomic_read(&tx_priv->aggr_count) == (tx_priv->fw_avail_bufcnt - 2))) {
			tx_priv->fw_avail_bufcnt -= atomic_read(&tx_priv->aggr_count);
			aicwf_sdio_aggr_send(tx_priv);
		} else
			goto done;
	}

done:
	return 0;
}

int aicwf_sdio_aggr(struct aicwf_tx_priv *tx_priv, struct sk_buff *pkt)
{
    struct rwnx_txhdr *txhdr = (struct rwnx_txhdr *)pkt->data;
    u8 *start_ptr = tx_priv->tail;
    u8 sdio_header[4];
    u8 adjust_str[4] = {0, 0, 0, 0};
    u32 curr_len = 0;
    int allign_len = 0;
    int headroom;

    sdio_header[0] =((pkt->len - sizeof(struct rwnx_txhdr) + sizeof(struct txdesc_api)) & 0xff);
    sdio_header[1] =(((pkt->len - sizeof(struct rwnx_txhdr) + sizeof(struct txdesc_api)) >> 8)&0x0f);
    sdio_header[2] = 0x01; //data
    sdio_header[3] = 0; //reserved

    memcpy(tx_priv->tail, (u8 *)&sdio_header, sizeof(sdio_header));
    tx_priv->tail += sizeof(sdio_header);
    //payload
    memcpy(tx_priv->tail, (u8 *)(long)&txhdr->sw_hdr.desc, sizeof(struct txdesc_api));
    tx_priv->tail += sizeof(struct txdesc_api); //hostdesc
    memcpy(tx_priv->tail, (u8 *)((u8 *)txhdr + txhdr->sw_hdr.headroom), pkt->len-txhdr->sw_hdr.headroom);
    tx_priv->tail += (pkt->len - txhdr->sw_hdr.headroom);

    //word alignment
    curr_len = tx_priv->tail - tx_priv->head;
    if (curr_len & (TX_ALIGNMENT - 1)) {
        allign_len = roundup(curr_len, TX_ALIGNMENT)-curr_len;
        memcpy(tx_priv->tail, adjust_str, allign_len);
        tx_priv->tail += allign_len;
    }

    start_ptr[0] = ((tx_priv->tail - start_ptr - 4) & 0xff);
    start_ptr[1] = (((tx_priv->tail - start_ptr - 4)>>8) & 0x0f);
    tx_priv->aggr_buf->dev = pkt->dev;

    if(!txhdr->sw_hdr.need_cfm) {
	headroom = txhdr->sw_hdr.headroom;
        //kmem_cache_free(txhdr->sw_hdr->rwnx_vif->rwnx_hw->sw_txhdr_cache, txhdr->sw_hdr);
        skb_pull(pkt, headroom);
        consume_skb(pkt);
    }

    atomic_inc(&tx_priv->aggr_count);
    return 0;
}

void aicwf_sdio_aggr_send(struct aicwf_tx_priv *tx_priv)
{
    struct sk_buff *tx_buf = tx_priv->aggr_buf;
    int ret = 0;
    int curr_len = 0;

    //link tail is necessary
    curr_len = tx_priv->tail - tx_priv->head;
    if ((curr_len % TXPKT_BLOCKSIZE) != 0) {
        memset(tx_priv->tail, 0, TAIL_LEN);
        tx_priv->tail += TAIL_LEN;
    }

    tx_buf->len = tx_priv->tail - tx_priv->head;
    ret = aicwf_sdio_txpkt(tx_priv->sdiodev, tx_buf);
    if (ret < 0) {
        sdio_err("fail to send aggr pkt!\n");
    }

    aicwf_sdio_aggrbuf_reset(tx_priv);
}

void aicwf_sdio_aggrbuf_reset(struct aicwf_tx_priv *tx_priv)
{
    struct sk_buff *aggr_buf = tx_priv->aggr_buf;

    tx_priv->tail = tx_priv->head;
    aggr_buf->len = 0;
    atomic_set(&tx_priv->aggr_count, 0);
}
#endif
static int aicwf_sdio_bus_start(struct device *dev)
{
    struct aicwf_bus *bus_if = dev_get_drvdata(dev);
    struct aic_sdio_dev *sdiodev = bus_if->bus_priv.sdio;
    int ret = 0;

    sdio_claim_host(sdiodev->func);
#if 0
    sdio_claim_irq(sdiodev->func, aicwf_sdio_hal_irqhandler);
    sdio_claim_irq(sdiodev->func_msg, aicwf_sdio_hal_irqhandler_func2);
#else
    //since we have func2 we don't register irq handler
    sdio_claim_irq(sdiodev->func, NULL);
    sdio_claim_irq(sdiodev->func_msg, NULL);

    sdiodev->func->irq_handler = (sdio_irq_handler_t *)aicwf_sdio_hal_irqhandler;
    sdiodev->func_msg->irq_handler = (sdio_irq_handler_t *)aicwf_sdio_hal_irqhandler_func2;
#endif 
    sdio_release_host(sdiodev->func);

    //enable sdio interrupt
    ret = aicwf_sdio_writeb(sdiodev, SDIOWIFI_INTR_CONFIG_REG, 0x07);

    if (ret != 0)
        sdio_err("intr register failed:%d\n", ret);

    //enable sdio interrupt
    ret = aicwf_sdio_writeb_func2(sdiodev, SDIOWIFI_INTR_CONFIG_REG, 0x07);

    if (ret != 0)
        sdio_err("func2 intr register failed:%d\n", ret);

    bus_if->state = BUS_UP_ST;

    return ret;
}

void aic_thread_wait_stop(void)
{
    set_current_state(TASK_INTERRUPTIBLE);
    while (!kthread_should_stop()) {
            schedule();
            set_current_state(TASK_INTERRUPTIBLE);
    }
    __set_current_state(TASK_RUNNING);
}
#ifdef CONFIG_TXRX_THREAD_PRIO
int bustx_thread_prio = 36;
module_param_named(bustx_thread_prio, bustx_thread_prio, int, 0644);
int busrx_thread_prio = 37;
module_param_named(busrx_thread_prio, busrx_thread_prio, int, 0644);
#endif

#ifdef HANDLE_TX_THREAD_ZTE
int sdio_bustx_thread(void *data)
{
    struct aicwf_bus *bus = (struct aicwf_bus *) data;
    struct aic_sdio_dev *sdiodev = bus->bus_priv.sdio;

#ifdef CONFIG_TXRX_THREAD_PRIO
    if (bustx_thread_prio > 0) {
        struct sched_param param;
        param.sched_priority = (bustx_thread_prio < MAX_RT_PRIO)?bustx_thread_prio:(MAX_RT_PRIO-1);
        sched_setscheduler(current, SCHED_FIFO, &param);
    }
#endif

    while (1) {
//        if(kthread_should_stop()) {
//            sdio_err("sdio bustx thread stop\n");
//            break;
//        }
        if (!wait_for_completion_interruptible(&bus->bustx_trgg)) {
		    if(sdiodev->bus_if->state == BUS_DOWN_ST){
				printk("rwnx tx thread break.\n");
                break;
			}

            while((int)(atomic_read(&sdiodev->tx_priv->tx_pktcnt) > 0) || (sdiodev->tx_priv->cmd_txstate == true)) {
                aicwf_sdio_tx_process(sdiodev);
                if(sdiodev->bus_if->state == BUS_DOWN_ST)
                    break;
            }
        }
    }
	aic_thread_wait_stop();
	printk("rwnx tx thread exit.\n");
    return 0;
}
#else
void sdio_bustx_tasklet(void *data)
{
    struct aicwf_bus *bus = (struct aicwf_bus *) data;
    struct aic_sdio_dev *sdiodev = bus->bus_priv.sdio;

    do {
        if(sdiodev->bus_if->state == BUS_DOWN_ST)
            break;

        if((int)(atomic_read(&sdiodev->tx_priv->tx_pktcnt) > 0) || (sdiodev->tx_priv->cmd_txstate == true)) {
                //printk("tp\n");
                aicwf_sdio_tx_process(sdiodev);
        }
        else
            break;
    } while(1);

    //printk("tpe\n");
}

#endif

#ifdef HANDLE_RX_THREAD_ZTE
int sdio_busrx_thread(void *data)
{
    struct aicwf_rx_priv *rx_priv = (struct aicwf_rx_priv *)data;
    struct aicwf_bus *bus_if = rx_priv->sdiodev->bus_if;

#ifdef CONFIG_TXRX_THREAD_PRIO
    if (busrx_thread_prio > 0) {
        struct sched_param param;
        param.sched_priority = (busrx_thread_prio < MAX_RT_PRIO)?busrx_thread_prio:(MAX_RT_PRIO-1);
        sched_setscheduler(current, SCHED_FIFO, &param);
    }
#endif

    while (1) {
//        if(kthread_should_stop()) {
//            sdio_err("sdio busrx thread stop\n");
//            break;
//        }
        if (!wait_for_completion_interruptible(&bus_if->busrx_trgg)) {
		    if(bus_if->state == BUS_DOWN_ST){
				printk("rwnx rx thread break.\n");
                break;
			}
            aicwf_process_rxframes(rx_priv);
        }
    }
	aic_thread_wait_stop();
	printk("rwnx rx thread exit.\n");
    return 0;
}
#else
void sdio_busrx_tasklet(void *data)
{
    struct aicwf_bus *bus_if = (struct aicwf_bus *)data;
    struct aicwf_rx_priv *rx_priv = bus_if->bus_priv.sdio->rx_priv;

    do {
        if(bus_if->state == BUS_DOWN_ST)
            break;
        //printk("rp\n");
	aicwf_process_rxframes(rx_priv);
	//printk("R\n");
	msleep(5);
	//printk("rpe\n");       
    } while(0);
}

#endif

#if defined(CONFIG_SDIO_PWRCTRL)
static int aicwf_sdio_pwrctl_thread(void *data)
{
    struct aic_sdio_dev *sdiodev = (struct aic_sdio_dev *) data;

    while (1) {
        if (!wait_for_completion_interruptible(&sdiodev->pwrctrl_trgg)) {
            if(sdiodev->bus_if->state == BUS_DOWN_ST)
                break;
	    if (sdiodev->state == SDIO_ACTIVE_ST) {
                if((int)(atomic_read(&sdiodev->tx_priv->tx_pktcnt) <= 0) && (sdiodev->tx_priv->cmd_txstate == false) && \
                    atomic_read(&sdiodev->rx_priv->rx_cnt)==0)
                        aicwf_sdio_pwr_stctl(sdiodev, SDIO_SLEEP_ST);
                else
                    aicwf_sdio_pwrctl_timer(sdiodev, sdiodev->active_duration);
            }
        } else {
                continue;
        }
    }

    aic_thread_wait_stop();
    return 0;
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(4,14,0)
static void aicwf_sdio_bus_pwrctl(ulong data)
#else
static void aicwf_sdio_bus_pwrctl(struct timer_list *t)
#endif
{
#if LINUX_VERSION_CODE < KERNEL_VERSION(4,14,0)
    struct aic_sdio_dev *sdiodev = (struct aic_sdio_dev *) data;
#else
    struct aic_sdio_dev *sdiodev = from_timer(sdiodev, t, timer);
#endif

    if (sdiodev->bus_if->state == BUS_DOWN_ST) {
        sdio_err("bus down\n");
        return;
    }

    if (sdiodev->pwrctl_tsk){
        complete(&sdiodev->pwrctrl_trgg);
    }
}
#endif

#ifdef LESS_SKB
static void aicwf_sdio_enq_rxpkt(struct aic_sdio_dev *sdiodev, struct rx_buff *pkt)
#else
static void aicwf_sdio_enq_rxpkt(struct aic_sdio_dev *sdiodev, struct sk_buff *pkt)
#endif
{
    struct aicwf_rx_priv* rx_priv = sdiodev->rx_priv;
    unsigned long flags = 0;

    spin_lock_irqsave(&rx_priv->rxqlock, flags);
    #ifdef LESS_SKB
    if (!aicwf_rxbuff_enqueue(sdiodev->dev, &rx_priv->rxq, pkt)) {
        spin_unlock_irqrestore(&rx_priv->rxqlock, flags);
        printk("%s %d, enqueue rxq fail\n", __func__, __LINE__);
#if 0
        rxbuff_free(pkt);
#else
        aicwf_prealloc_rxbuff_free(pkt, &rx_priv->rxbuff_lock);
#endif
        return;
    }
    #else
#if 1
    if (!aicwf_rxframe_enqueue(sdiodev->dev, &rx_priv->rxq, pkt)) {
        spin_unlock_irqrestore(&rx_priv->rxqlock, flags);
        aicwf_dev_skb_free(pkt);
        return;
    }
#else 
    skb_queue_tail(&rx_priv->rxq.queuelist[0], pkt);
#endif
    #endif
    spin_unlock_irqrestore(&rx_priv->rxqlock, flags);

    atomic_inc(&rx_priv->rx_cnt);
}

#define SDIO_OTHER_INTERRUPT (0x1ul << 7)

void aicwf_sdio_hal_irqhandler(struct sdio_func *func)
{
    struct aicwf_bus *bus_if = dev_get_drvdata(&func->dev);
    struct aic_sdio_dev *sdiodev = bus_if->bus_priv.sdio;
    u8 intstatus = 0;
    u8 byte_len = 0;
    #ifdef LESS_SKB
    struct rx_buff *pkt = NULL;
    #else
    struct sk_buff *pkt = NULL;
    #endif
    int ret;
    u8 trigg = 0;

    if (!bus_if || bus_if->state == BUS_DOWN_ST) {
        sdio_err("bus err\n");
        return;
    }
    //printk("sdio func1 irq\n");

#ifdef LESS_SKB
    if (list_empty(&aic_rx_buff_list.rxbuff_list)) {
        printk("%s %d, rxbuff list is empty\n", __func__, __LINE__);
        return;
    } 
#endif

    ret = aicwf_sdio_readb(sdiodev, SDIOWIFI_BLOCK_CNT_REG, &intstatus);
    while(ret || (intstatus & SDIO_OTHER_INTERRUPT)) {
        sdio_err("ret=%d, intstatus=%x\r\n",ret, intstatus);
        ret = aicwf_sdio_readb(sdiodev, SDIOWIFI_BLOCK_CNT_REG, &intstatus);
    }
    sdiodev->rx_priv->data_len = intstatus * SDIOWIFI_FUNC_BLOCKSIZE;

    if (intstatus > 0) {
        if(intstatus < 64) {
            pkt = aicwf_sdio_readframes(sdiodev, 0);
        } else {
            aicwf_sdio_intr_get_len_bytemode(sdiodev, &byte_len);//byte_len must<= 128
            sdio_info("byte mode len=%d\r\n", byte_len);
            pkt = aicwf_sdio_readframes(sdiodev, 0);
        }
    } else {
	#ifndef CONFIG_PLATFORM_ALLWINNER
        //sdio_err("Int no data\n");
	#endif
    }

    if (pkt)
        aicwf_sdio_enq_rxpkt(sdiodev, pkt);

    if (atomic_read(&sdiodev->rx_priv->rx_cnt) == 1)
    complete(&bus_if->busrx_trgg);
}


void aicwf_sdio_hal_irqhandler_func2(struct sdio_func *func)
{
    struct aicwf_bus *bus_if = dev_get_drvdata(&func->dev);
    struct aic_sdio_dev *sdiodev = bus_if->bus_priv.sdio;
    u8 intstatus = 0;
    u8 byte_len = 0;
#ifdef LESS_SKB
    struct rx_buff *pkt = NULL;
#else
    struct sk_buff *pkt = NULL;
#endif
    int ret;

    if (!bus_if || bus_if->state == BUS_DOWN_ST) {
        sdio_err("bus err\n");
        return;
    }

    //printk("sdio func2 irq\n");

#ifdef LESS_SKB
    if (list_empty(&aic_rx_buff_list.rxbuff_list)) {
        printk("%s %d, rxbuff list is empty\n", __func__, __LINE__);
        return;
    } 
#endif

    ret = aicwf_sdio_readb_func2(sdiodev, SDIOWIFI_BLOCK_CNT_REG, &intstatus);

    while(ret || (intstatus & SDIO_OTHER_INTERRUPT)) {
        sdio_err("ret=%d, intstatus=%x\r\n",ret, intstatus);
        ret = aicwf_sdio_readb_func2(sdiodev, SDIOWIFI_BLOCK_CNT_REG, &intstatus);
    }
    sdiodev->rx_priv->data_len = intstatus * SDIOWIFI_FUNC_BLOCKSIZE;
    if (intstatus > 0) {
        if(intstatus < 64) {
            pkt = aicwf_sdio_readframes(sdiodev, 1);
        } else {
            sdio_info("byte mode len=%d\r\n", byte_len);

            aicwf_sdio_intr_get_len_bytemode(sdiodev, &byte_len);//byte_len must<= 128
            pkt = aicwf_sdio_readframes(sdiodev, 1);
        }
    } else {
	#ifndef CONFIG_PLATFORM_ALLWINNER
        sdio_err("Interrupt but no data\n");
	#endif
    }

    if (pkt)
        aicwf_sdio_enq_rxpkt(sdiodev, pkt);

    complete(&bus_if->busrx_trgg);
}

#if defined(CONFIG_SDIO_PWRCTRL)
void aicwf_sdio_pwrctl_timer(struct aic_sdio_dev *sdiodev, uint duration)
{
    uint timeout;

    if (sdiodev->bus_if->state == BUS_DOWN_ST && duration)
        return;

    spin_lock_bh(&sdiodev->pwrctl_lock);
    if (!duration) {
        if (timer_pending(&sdiodev->timer))
            del_timer_sync(&sdiodev->timer);
    } else {
        sdiodev->active_duration = duration;
        timeout = msecs_to_jiffies(sdiodev->active_duration);
        mod_timer(&sdiodev->timer, jiffies + timeout);
    }
    spin_unlock_bh(&sdiodev->pwrctl_lock);
}
#endif

static struct aicwf_bus_ops aicwf_sdio_bus_ops = {
    .stop = aicwf_sdio_bus_stop,
    .start = aicwf_sdio_bus_start,
    .txdata = aicwf_sdio_bus_txdata,
    .txmsg = aicwf_sdio_bus_txmsg,
};

void aicwf_sdio_release_func2(struct aic_sdio_dev *sdiodev)
{
    int ret = 0;
    sdio_dbg("%s\n", __func__);
    sdio_claim_host(sdiodev->func_msg);
    //disable sdio interrupt
    ret = aicwf_sdio_writeb_func2(sdiodev, SDIOWIFI_INTR_CONFIG_REG, 0x0);
    if (ret < 0) {
        sdio_err("reg:%d write failed!\n", SDIOWIFI_INTR_CONFIG_REG);
    }
    sdio_release_irq(sdiodev->func_msg);
    sdio_release_host(sdiodev->func_msg);

}
void aicwf_sdio_release(struct aic_sdio_dev *sdiodev)
{
    struct aicwf_bus *bus_if;
    struct aicwf_rx_priv* rx_priv = NULL;
    int ret;
    sdio_dbg("%s\n", __func__);

    bus_if = dev_get_drvdata(sdiodev->dev);
    bus_if->state = BUS_DOWN_ST;

    sdio_claim_host(sdiodev->func);
    //disable sdio interrupt
    ret = aicwf_sdio_writeb(sdiodev, SDIOWIFI_INTR_CONFIG_REG, 0x0);
    if (ret < 0) {
        sdio_err("reg:%d write failed!\n", SDIOWIFI_INTR_CONFIG_REG);
    }
    sdio_release_irq(sdiodev->func);
    sdio_release_host(sdiodev->func);

    if (sdiodev->dev)
        aicwf_bus_deinit(sdiodev->dev);

    aicwf_tx_deinit(sdiodev->tx_priv);
    rx_priv = sdiodev->rx_priv;
    if(rx_priv != NULL)
        aicwf_rx_deinit(rx_priv);
#if 1
    if (sdiodev->cmd_mgr.cmd_thread) {
        complete_all(&sdiodev->cmd_mgr.cmdproc_trgg);
        kthread_stop(sdiodev->cmd_mgr.cmd_thread);
		printk("rwnx cmd kthread stop.\n");
        sdiodev->cmd_mgr.cmd_thread = NULL;
    }
#endif
    #ifndef CMD_WORKQUEUE
    //tasklet_kill(&sdiodev->cmd_mgr.cmd_tasklet);
    #endif

    rwnx_cmd_mgr_deinit(&sdiodev->cmd_mgr);
    sdio_dbg("exit %s\n", __func__);
}

#define FEATURE_SDIO_PHASE          0        // 0: default, 2: 180
int sdio_phase = FEATURE_SDIO_PHASE;
module_param(sdio_phase, int, 0644);
MODULE_PARM_DESC(sdio_phase, "sdio_phase:0: default, 2: 180");

int aicwf_sdio_func_init(struct aic_sdio_dev *sdiodev)
{
    struct mmc_host *host;
    u8 block_bit0 = 0x1;
    u8 byte_mode_disable = 0x1;//1: no byte mode
    int ret = 0;
    host = sdiodev->func->card->host;

    sdio_claim_host(sdiodev->func);
#if 1//SDIO PHASE SETTING
	sdiodev->func->card->quirks |= MMC_QUIRK_LENIENT_FN0;
	sdio_f0_writeb(sdiodev->func, sdio_phase, 0x13, &ret);
	if (ret < 0) {
		sdio_err("write func0 fail %d\n", ret);
		return ret;
	}
#endif
    ret = sdio_set_block_size(sdiodev->func, SDIOWIFI_FUNC_BLOCKSIZE);
    if (ret < 0) {
        sdio_err("set blocksize fail %d\n", ret);
        sdio_release_host(sdiodev->func);
        return ret;
    }
    ret = sdio_enable_func(sdiodev->func);
    if (ret < 0) {
        sdio_release_host(sdiodev->func);
        sdio_err("enable func fail %d.\n", ret);
    }


    #if 1 // limit clock for fpga
    host->ios.clock = 100000000;
    #else
    host->ios.clock = 60000000;
    #endif
    host->ops->set_ios(host, &host->ios);
    sdio_release_host(sdiodev->func);

    sdio_claim_host(sdiodev->func_msg);
    ret = sdio_set_block_size(sdiodev->func_msg, SDIOWIFI_FUNC_BLOCKSIZE);
    if (ret < 0) {
        sdio_err("set func2 blocksize fail %d\n", ret);
        sdio_release_host(sdiodev->func_msg);
        return ret;
    }
    ret = sdio_enable_func(sdiodev->func_msg);
    if (ret < 0) {
        sdio_err("enable func2 fail %d.\n", ret);
    }
    sdio_release_host(sdiodev->func_msg);

    sdio_dbg("8818 Set SDIO Clock %d MHz\n",host->ios.clock/1000000);

    ret = aicwf_sdio_writeb(sdiodev, SDIOWIFI_REGISTER_BLOCK, block_bit0);
    if (ret < 0) {
        sdio_err("reg:%d write failed!\n", SDIOWIFI_REGISTER_BLOCK);
        return ret;
    }

    //1: no byte mode
    ret = aicwf_sdio_writeb(sdiodev, SDIOWIFI_BYTEMODE_ENABLE_REG, byte_mode_disable);
    if (ret < 0) {
        sdio_err("reg:%d write failed!\n", SDIOWIFI_BYTEMODE_ENABLE_REG);
        return ret;
    }

    ret = aicwf_sdio_writeb_func2(sdiodev, SDIOWIFI_REGISTER_BLOCK, block_bit0);
    if (ret < 0) {
        sdio_err("reg:%d write failed!\n", SDIOWIFI_REGISTER_BLOCK);
        return ret;
    }

    //1: no byte mode
    ret = aicwf_sdio_writeb_func2(sdiodev, SDIOWIFI_BYTEMODE_ENABLE_REG, byte_mode_disable);
    if (ret < 0) {
        sdio_err("reg:%d write failed!\n", SDIOWIFI_BYTEMODE_ENABLE_REG);
        return ret;
    }

    return ret;
}


void aicwf_sdio_func_deinit(struct aic_sdio_dev *sdiodev)
{
    sdio_claim_host(sdiodev->func);
    sdio_disable_func(sdiodev->func);
    sdio_release_host(sdiodev->func);
}

#ifdef CONFIG_TEMP_PW
extern void set_txpwr_ctrl(struct aic_sdio_dev *sdiodev, s8_l value);

void aicwf_temp_worker(struct work_struct *work)
{
	struct rwnx_hw *rwnx_hw;
	struct mm_set_vendor_hwconfig_cfm cfm;
	struct aic_sdio_dev *sdiodev = container_of(work, struct aic_sdio_dev, tp_work);
	rwnx_hw = sdiodev->rwnx_hw;
	rwnx_send_get_temp_req(rwnx_hw, &cfm);
	set_txpwr_ctrl(sdiodev, cfm.chip_temp_cfm.degree);
	mod_timer(&sdiodev->tp_timer, jiffies + msecs_to_jiffies(TEMP_GET_INTERVAL));
}
#if LINUX_VERSION_CODE < KERNEL_VERSION(4, 14, 0)
static void aicwf_temp_timer(ulong data)
#else
static void aicwf_temp_timer(struct timer_list *t)
#endif
{
#if LINUX_VERSION_CODE < KERNEL_VERSION(4, 14, 0)
	struct aic_sdio_dev *sdiodev = (struct aic_sdio_dev *) data;
#else
	struct aic_sdio_dev *sdiodev = from_timer(sdiodev, t, tp_timer);
#endif
	if (!work_pending(&sdiodev->tp_work))
		schedule_work(&sdiodev->tp_work);
	return;
}
#endif
void *aicwf_sdio_bus_init(struct aic_sdio_dev *sdiodev)
{
    int ret;
    struct aicwf_bus *bus_if;
    struct aicwf_rx_priv *rx_priv;
    struct aicwf_tx_priv *tx_priv;

    #if defined(CONFIG_SDIO_PWRCTRL)
    spin_lock_init(&sdiodev->pwrctl_lock);
    sema_init(&sdiodev->pwrctl_wakeup_sema, 1);
    #endif

    bus_if = sdiodev->bus_if;
    bus_if->dev = sdiodev->dev;
    bus_if->ops = &aicwf_sdio_bus_ops;
    bus_if->state = BUS_DOWN_ST;
    #if defined(CONFIG_SDIO_PWRCTRL)
    sdiodev->state = SDIO_SLEEP_ST;
    sdiodev->active_duration = SDIOWIFI_PWR_CTRL_INTERVAL;
    #else
    sdiodev->state = SDIO_ACTIVE_ST;
    #endif

    rx_priv = aicwf_rx_init(sdiodev);
    if(!rx_priv) {
        sdio_err("rx init fail\n");
        goto fail;
    }
    sdiodev->rx_priv = rx_priv;

    tx_priv = aicwf_tx_init(sdiodev);
    if(!tx_priv) {
        sdio_err("tx init fail\n");
        goto fail;
    }
    sdiodev->tx_priv = tx_priv;
    aicwf_frame_queue_init(&tx_priv->txq, AICWF_TXQ_CNT, TXQLEN);
    spin_lock_init(&tx_priv->txqlock);
    sema_init(&tx_priv->txctl_sema,1);
    sema_init(&tx_priv->cmd_txsema, 1);
    init_waitqueue_head(&tx_priv->cmd_txdone_wait);
    atomic_set(&tx_priv->tx_pktcnt, 0);

#ifdef CONFIG_TEMP_PW
#if LINUX_VERSION_CODE < KERNEL_VERSION(4, 14, 0)
		init_timer(&sdiodev->tp_timer);
		sdiodev->tp_timer.data = (ulong) sdiodev;
		sdiodev->tp_timer.function = aicwf_temp_timer;
		//sdiodev->tp_timer.expires  = jiffies + msecs_to_jiffies(TEMP_GET_INTERVAL);
#else
		timer_setup(&sdiodev->tp_timer, aicwf_temp_timer, 0);
#endif
		INIT_WORK(&sdiodev->tp_work, aicwf_temp_worker);
		mod_timer(&sdiodev->tp_timer, jiffies + msecs_to_jiffies(10 * 1000));
		sdiodev->range = 0;
#endif
#if defined(CONFIG_SDIO_PWRCTRL)
#if LINUX_VERSION_CODE < KERNEL_VERSION(4,14,0)
    init_timer(&sdiodev->timer);
    sdiodev->timer.data = (ulong) sdiodev;
    sdiodev->timer.function = aicwf_sdio_bus_pwrctl;
#else
    timer_setup(&sdiodev->timer, aicwf_sdio_bus_pwrctl, 0);
#endif
    init_completion(&sdiodev->pwrctrl_trgg);
#ifdef AICWF_SDIO_SUPPORT
    sdiodev->pwrctl_tsk = kthread_run(aicwf_sdio_pwrctl_thread, sdiodev, "aicwf_pwrctl");
#endif
    if (IS_ERR(sdiodev->pwrctl_tsk)) {
        sdiodev->pwrctl_tsk = NULL;
    }
#endif

#ifdef CONFIG_TX_NETIF_FLOWCTRL
    sdiodev->flowctrl = 0;
    spin_lock_init(&sdiodev->tx_flow_lock);
#endif
    ret = aicwf_bus_init(0, sdiodev->dev);
    if (ret < 0) {
        sdio_err("bus init fail\n");
        goto fail;
    }

    ret  = aicwf_bus_start(bus_if);
    if (ret != 0) {
        sdio_err("bus start fail\n");
        goto fail;
    }

    return sdiodev;

fail:
    aicwf_sdio_release(sdiodev);
    return NULL;
}


