/**
 ******************************************************************************
 *
 * @file rwnx_platorm.h
 *
 * Copyright (C) RivieraWaves 2012-2019
 *
 ******************************************************************************
 */

#ifndef _RWNX_PLATFORM_H_
#define _RWNX_PLATFORM_H_

#include <linux/pci.h>
#include "lmac_msg.h"

#define AIC_MACCONFIG_NAME              "../../etc_rw/aic_macconfig.txt"
#define RWNX_CONFIG_FW_NAME             "rwnx_settings.ini"
#define RWNX_PHY_CONFIG_TRD_NAME        "rwnx_trident.ini"
#define RWNX_PHY_CONFIG_KARST_NAME      "rwnx_karst.ini"
#define RWNX_AGC_FW_NAME                "agcram.bin"
#define RWNX_LDPC_RAM_NAME              "ldpcram.bin"
#ifdef CONFIG_RWNX_FULLMAC
#define RWNX_MAC_FW_BASE_NAME           "fmacfw"
#define RWNX_MAC_FW_RF_BASE_NAME        "lmacfw_rf.bin"
#elif defined CONFIG_RWNX_FHOST
#define RWNX_MAC_FW_BASE_NAME           "fhostfw"
#endif /* CONFIG_RWNX_FULLMAC */

#ifdef CONFIG_RWNX_TL4
#define RWNX_MAC_FW_NAME RWNX_MAC_FW_BASE_NAME".hex"
#else
#define RWNX_MAC_FW_NAME  RWNX_MAC_FW_BASE_NAME".ihex"
#ifdef CONFIG_MCU_INTEGRATED
#ifdef CONFIG_MCU_MESSAGE
#define RWNX_MAC_FW_NAME2 RWNX_MAC_FW_BASE_NAME"_8818m_custmsg.bin"
#else
#define RWNX_MAC_FW_NAME2 RWNX_MAC_FW_BASE_NAME"_8818m.bin"
#endif
#else
#define RWNX_MAC_FW_NAME2 RWNX_MAC_FW_BASE_NAME".bin"
#endif
#endif
#define RWNX_MAC_CALIB_BASE_NAME_8800DC        "fmacfw_calib_8800dc"
#define RWNX_MAC_CALIB_NAME_8800DC_U02          RWNX_MAC_CALIB_BASE_NAME_8800DC"_u02.bin"
#define RWNX_MAC_CALIB_NAME_8800DC_H_U02        RWNX_MAC_CALIB_BASE_NAME_8800DC"_h_u02.bin"

#if (defined(CONFIG_DPD) && !defined(CONFIG_FORCE_DPD_CALIB))
#define FW_DPDRESULT_NAME_8800DC        "aic_dpdresult_8800dc.bin"
#endif
#ifdef CONFIG_DPD
#define ROM_FMAC_CALIB_ADDR            0x00130000
#define FW_PATH_MAX_LEN 200
int is_file_exist(char* name);
typedef struct {
    uint32_t bit_mask[3];
    uint32_t reserved;
    uint32_t dpd_high[96];
    uint32_t dpd_11b[96];
    uint32_t dpd_low[96];
    uint32_t idac_11b[48];
    uint32_t idac_high[48];
    uint32_t idac_low[48];
    uint32_t loft_res[18];
    uint32_t rx_iqim_res[16];
} rf_misc_ram_t;
typedef struct {
    uint32_t bit_mask[4];
    uint32_t dpd_high[96];
    uint32_t loft_res[18];
} rf_misc_ram_lite_t;
#define MEMBER_SIZE(type, member)   sizeof(((type *)0)->member)
#define DPD_RESULT_SIZE_8800DC      sizeof(rf_misc_ram_lite_t)
extern rf_misc_ram_lite_t dpd_res;
#endif
#ifdef CONFIG_DPD
#ifndef CONFIG_FORCE_DPD_CALIB
int aicwf_dpd_result_write_8800dc(void *buf, int buf_len);
#endif
#endif
#define RWNX_MAC_PATCH_BASE_NAME        "fmacfw_patch_8800dc"
#define RWNX_MAC_PATCH_NAME2 RWNX_MAC_PATCH_BASE_NAME".bin"
#define RWNX_MAC_PATCH_NAME2_U02 RWNX_MAC_PATCH_BASE_NAME"_u02.bin"
#define RWNX_MAC_PATCH_NAME2_8800DC_H_U02 RWNX_MAC_PATCH_BASE_NAME"_h_u02.bin"
#define RWNX_MAC_PATCH_TABLE_NAME_8800DC "fmacfw_patch_tbl_8800dc"
#define RWNX_MAC_PATCH_TABLE_8800DC RWNX_MAC_PATCH_TABLE_NAME_8800DC ".bin"
#define RWNX_MAC_PATCH_TABLE_8800DC_U02 RWNX_MAC_PATCH_TABLE_NAME_8800DC "_u02.bin"
#define RWNX_MAC_PATCH_TABLE_8800DC_H_U02 RWNX_MAC_PATCH_TABLE_NAME_8800DC "_h_u02.bin"
#define RWNX_MAC_RF_PATCH_BASE_NAME     "fmacfw_rf_patch"
#define RWNX_MAC_RF_PATCH_NAME RWNX_MAC_RF_PATCH_BASE_NAME".bin"


#define RWNX_FCU_FW_NAME                "fcuram.bin"

#define FW_USERCONFIG_NAME              "aic_userconfig.txt"

#ifdef CONFIG_VENDOR_SPECIFIED_FW_PATH
#define VENDOR_SPECIFIED_FW_PATH    CONFIG_VENDOR_SPECIFIED_FW_PATH
#endif
#ifdef CONFIG_VENDOR_SUBDIR_NAME
#define VENDOR_SUBDIR_NAME          CONFIG_VENDOR_SUBDIR_NAME
#endif
#define VENDOR_SPECIFIED_DPD_PATH "/mnt/userdata/etc_rw/wifi"

enum aicbt_patch_table_type {
	AICBT_PT_INF  = 0x00,
	AICBT_PT_TRAP = 0x1,
	AICBT_PT_B4,
	AICBT_PT_BTMODE,
	AICBT_PT_PWRON,
	AICBT_PT_AF,
	AICBT_PT_VER,
};
enum aicbt_btport_type {
	AICBT_BTPORT_NULL,
	AICBT_BTPORT_MB,
	AICBT_BTPORT_UART,
};
enum aicbt_btmode_type {
	AICBT_BTMODE_BT_ONLY_SW = 0x0,    // bt only mode with switch
	AICBT_BTMODE_BT_WIFI_COMBO,       // wifi/bt combo mode
	AICBT_BTMODE_BT_ONLY,             // bt only mode without switch
	AICBT_BTMODE_BT_ONLY_TEST,        // bt only test mode
	AICBT_BTMODE_BT_WIFI_COMBO_TEST,  // wifi/bt combo test mode
	AICBT_BTMODE_BT_ONLY_COANT,       // bt only mode with no external switch
	AICBT_MODE_NULL = 0xFF,           // invalid value
};
enum aicbt_uart_baud_type {
	AICBT_UART_BAUD_115200     = 115200,
	AICBT_UART_BAUD_921600     = 921600,
	AICBT_UART_BAUD_1_5M       = 1500000,
	AICBT_UART_BAUD_3_25M      = 3250000,
};
enum aicbt_uart_flowctrl_type {
	AICBT_UART_FLOWCTRL_DISABLE = 0x0,    // uart without flow ctrl
	AICBT_UART_FLOWCTRL_ENABLE,           // uart with flow ctrl
};
enum aicbsp_cpmode_type {
	AICBSP_CPMODE_WORK,
	AICBSP_CPMODE_TEST,
	AICBSP_CPMODE_MAX,
};
enum chip_rev {
	CHIP_REV_U01 = 1,
	CHIP_REV_U02 = 3,
	CHIP_REV_U03 = 7,
	CHIP_REV_U04 = 7,
};
#define RAM_FMAC_FW_ADDR                    0x00120000
#define FW_RAM_ADID_BASE_ADDR               0x00161928
#define FW_RAM_ADID_BASE_ADDR_U03           0x00161928
#define FW_RAM_PATCH_BASE_ADDR              0x00100000
#define RAM_8800DC_U01_ADID_ADDR            0x00101788
#define RAM_8800DC_U02_ADID_ADDR            0x001017d8
#define RAM_8800DC_FW_PATCH_ADDR            0x00184000
#define FW_RESET_START_ADDR                 0x40500128
#define FW_RESET_START_VAL                  0x40
#define FW_ADID_FLAG_ADDR                   0x40500150
#define FW_ADID_FLAG_VAL                    0x01
#define FW_RAM_ADID_BASE_ADDR_8800D80       0x002017E0
#define FW_RAM_PATCH_BASE_ADDR_8800D80      0x0020B2B0
#define FW_RAM_ADID_BASE_ADDR_8800D80_U02   0x00201940
#define FW_RAM_PATCH_BASE_ADDR_8800D80_U02  0x0020b43c
#define AICBT_PT_TAG                "AICBT_PT_TAG"
#define AICBT_TXPWR_LVL            0x00006020
#define AICBT_TXPWR_LVL_8800dc            0x00006f2f
#define AICBT_TXPWR_LVL_8800d80           0x00006f2f
#define AICBSP_HWINFO_DEFAULT       (-1)
#define AICBSP_CPMODE_DEFAULT       AICBSP_CPMODE_WORK
#define AICBSP_FWLOG_EN_DEFAULT     0
#define AICBT_BTMODE_DEFAULT_8800d80    AICBT_BTMODE_BT_ONLY_COANT
#define AICBT_BTMODE_DEFAULT            AICBT_BTMODE_BT_WIFI_COMBO
#define AICBT_BTPORT_DEFAULT            AICBT_BTPORT_UART
#define AICBT_UART_BAUD_DEFAULT         AICBT_UART_BAUD_1_5M
#define AICBT_UART_FC_DEFAULT           AICBT_UART_FLOWCTRL_ENABLE
#define AICBT_LPM_ENABLE_DEFAULT 	    0
#define AICBT_TXPWR_LVL_DEFAULT         AICBT_TXPWR_LVL
#define AICBT_TXPWR_LVL_DEFAULT_8800dc  AICBT_TXPWR_LVL_8800dc
#define AICBT_TXPWR_LVL_DEFAULT_8800d80 AICBT_TXPWR_LVL_8800d80
struct aicbt_patch_table {
	char     *name;
	uint32_t type;
	uint32_t *data;
	uint32_t len;
	struct aicbt_patch_table *next;
};
struct aicbt_info_t {
	uint32_t btmode;
	uint32_t btport;
	uint32_t uart_baud;
	uint32_t uart_flowctrl;
	uint32_t lpm_enable;
	uint32_t txpwr_lvl;
};
struct aicbt_patch_info_t {
	uint32_t info_len;
	uint32_t adid_addrinf;
	uint32_t addr_adid;
	uint32_t patch_addrinf;
	uint32_t addr_patch;
	uint32_t reset_addr;
	uint32_t reset_val;
	uint32_t adid_flag_addr;
	uint32_t adid_flag;
};
struct aicbsp_firmware {
	const char *desc;
	const char *bt_adid;
	const char *bt_patch;
	const char *bt_table;
	const char *wl_fw;
};
struct aicbsp_info_t {
	int hwinfo;
	int hwinfo_r;
	uint32_t cpmode;
	uint32_t chip_rev;
	bool fwlog_en;
	uint8_t irqf;
};
/**
 * Type of memory to access (cf rwnx_plat.get_address)
 *
 * @RWNX_ADDR_CPU To access memory of the embedded CPU
 * @RWNX_ADDR_SYSTEM To access memory/registers of one subsystem of the
 * embedded system
 *
 */
enum rwnx_platform_addr {
    RWNX_ADDR_CPU,
    RWNX_ADDR_SYSTEM,
    RWNX_ADDR_MAX,
};

struct rwnx_hw;

/**
 * struct rwnx_plat - Operation pointers for RWNX PCI platform
 *
 * @pci_dev: pointer to pci dev
 * @enabled: Set if embedded platform has been enabled (i.e. fw loaded and
 *          ipc started)
 * @enable: Configure communication with the fw (i.e. configure the transfers
 *         enable and register interrupt)
 * @disable: Stop communication with the fw
 * @deinit: Free all ressources allocated for the embedded platform
 * @get_address: Return the virtual address to access the requested address on
 *              the platform.
 * @ack_irq: Acknowledge the irq at link level.
 * @get_config_reg: Return the list (size + pointer) of registers to restore in
 * order to reload the platform while keeping the current configuration.
 *
 * @priv Private data for the link driver
 */
struct rwnx_plat {
    struct pci_dev *pci_dev;

#ifdef AICWF_SDIO_SUPPORT
	struct aic_sdio_dev *sdiodev;
#endif

#ifdef AICWF_USB_SUPPORT
    struct aic_usb_dev *usbdev;
#endif
    bool enabled;

    int (*enable)(struct rwnx_hw *rwnx_hw);
    int (*disable)(struct rwnx_hw *rwnx_hw);
    void (*deinit)(struct rwnx_plat *rwnx_plat);
    u8* (*get_address)(struct rwnx_plat *rwnx_plat, int addr_name,
                       unsigned int offset);
    void (*ack_irq)(struct rwnx_plat *rwnx_plat);
    int (*get_config_reg)(struct rwnx_plat *rwnx_plat, const u32 **list);

    u8 priv[0] __aligned(sizeof(void *));
};

#define RWNX_ADDR(plat, base, offset)           \
    plat->get_address(plat, base, offset)

#define RWNX_REG_READ(plat, base, offset)               \
    readl(plat->get_address(plat, base, offset))

#define RWNX_REG_WRITE(val, plat, base, offset)         \
    writel(val, plat->get_address(plat, base, offset))

typedef struct
{
    u8_l enable;
    u8_l xtal_cap;
    u8_l xtal_cap_fine;

} xtal_cap_conf_t;

extern struct rwnx_plat *g_rwnx_plat;

int rwnx_platform_init(struct rwnx_plat *rwnx_plat, void **platform_data);
void rwnx_platform_deinit(struct rwnx_hw *rwnx_hw);

int rwnx_platform_on(struct rwnx_hw *rwnx_hw, void *config);
void rwnx_platform_off(struct rwnx_hw *rwnx_hw, void **config);

int rwnx_platform_register_drv(void);
void rwnx_platform_unregister_drv(void);

#ifdef CONFIG_LOAD_USERCONFIG
void get_userconfig_txpwr_lvl(txpwr_lvl_conf_t *txpwr_lvl);
void get_userconfig_txpwr_lvl_v2_in_fdrv(txpwr_lvl_conf_v2_t *txpwr_lvl_v2);
void get_userconfig_txpwr_loss(txpwr_loss_conf_t *txpwr_loss);
void get_userconfig_txpwr_ofst(txpwr_ofst_conf_t *txpwr_ofst);
void get_userconfig_xtal_cap(xtal_cap_conf_t *xtal_cap);
void set_txpwr_loss_ofst(s8_l value);
#ifdef CONFIG_SET_AP_PS
int get_userconfig_set_ap_ps_lvl(ap_ps_conf_t *ap_ps_lvl);
#endif
#endif
int aicwf_patch_table_load(struct rwnx_hw *rwnx_hw, char *filename);
int rwnx_load_firmware(u32 **fw_buf, const char *name, struct device *device);
extern struct device *rwnx_platform_get_dev(struct rwnx_plat *rwnx_plat);

static inline unsigned int rwnx_platform_get_irq(struct rwnx_plat *rwnx_plat)
{
    return rwnx_plat->pci_dev->irq;
}
int aicwf_misc_ram_init_8800dc(struct rwnx_hw *rwnx_hw);
int aicwf_plat_patch_load_8800dc(struct rwnx_hw *rwnx_hw);

#endif /* _RWNX_PLATFORM_H_ */
