#include <linux/init.h>
#include <linux/types.h>
#include <linux/mm.h>
#include <linux/kernel.h>
#include <mach/board.h>
#include <mach/iomap.h>
#include <mach/debug.h>
#include <linux/io.h>
#include <mach/spinlock.h>
#include <mach/pcu.h>
#include <linux/mutex.h>
#include <linux/device.h>
#include <linux/module.h>
#include <linux/cdev.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include <linux/soc/zte/otp/otp_zx.h>
#include <asm/setup.h>
#include <asm/io.h>
#include <asm/system.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/miscdevice.h>
#include "nor.h"
#include <linux/gpio.h>
#include <linux/dmaengine.h>
#include <mach/iomap.h>
#include <mach/gpio.h>
#include <linux/sched.h>

extern struct mutex otpMutex;
extern char *nor_cmdline;

typedef volatile struct
{
	u32 secure_flag;
	u32 puk_hash[8];
	u32 secureDevId[3];
} s_OtpStruct;

typedef struct
{
	u32 uiPubKey[95];
} s_ImageHeader;

s_OtpStruct otp_data;

#define SECURE_EN_OFFSET     0x1000
#define PUB_KEY_HASH_OFFSET  0x1004
#define DEVICE_ID_OFFSET     0x1024

/**-----------------------------------------------------**/
struct nor_info *spi_nor_flash = NULL;
void  __iomem * nor_base_addr;

/* Used when the "_ext_id" is two bytes at most */
#define INFO(_jedec_id, _ext_id, _sector_size, _n_sectors, _flags)	\
		.id = {							\
			((_jedec_id) >> 16) & 0xff,			\
			((_jedec_id) >> 8) & 0xff,			\
			(_jedec_id) & 0xff,				\
			((_ext_id) >> 8) & 0xff,			\
			(_ext_id) & 0xff,				\
			},						\
		.id_len = (!(_jedec_id) ? 0 : (3 + ((_ext_id) ? 2 : 0))),	\
		.sector_size = (_sector_size),				\
		.n_sectors = (_n_sectors),				\
		.page_size = 256,					\
		.flags = (_flags),


spinor_cmd_t nor_cmd_table[]=
{	
	{CMD_WREN, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_DIS, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "write enable"},
	{CMD_WRDI, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_DIS, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "write disable"},
	{CMD_WRENVSR, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_DIS, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "write ensable vsr"},		
	{CMD_RDSR0, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_DIS, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_EN, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "read SR S7-S0"},
	{CMD_RDSR1, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_DIS, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_EN, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "read SR S15-S8"},
	{CMD_WRSR, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_DIS, ADDR_WIDTH_24, DATA_TX_EN, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "write SR S15-S0"},
	{CMD_RDB, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_EN, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_EN, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "read data bytes"},
	{CMD_RDFT, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_EN, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_EN, DUMY_TX_EN, 1, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "read fast"},
	{CMD_RDDFT, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_EN, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_EN, DUMY_TX_EN, 1, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_EN, TRANS_MOD_DUAL, "read dual fast"},
	{CMD_RDQFT, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_EN, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_EN, DUMY_TX_EN, 1, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_EN, TRANS_MOD_QUAD, "read quad fast"},
	{CMD_PP, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_EN, ADDR_WIDTH_24, DATA_TX_EN, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "page program"},
	{CMD_QPP, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_EN, ADDR_WIDTH_24, DATA_TX_EN, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_EN, TRANS_MOD_QUAD, "quad page program"},
	{CMD_SE, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_EN, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "sector erase"},
	{CMD_32KBE, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_EN, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "32K block erase"},
	{CMD_64KBE, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_EN, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "64K block erase"},
	{CMD_CE, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_DIS, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "chip erase"},
	{CMD_DP, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_DIS, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "deep power-down"},
	{CMD_RDPRDI, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_DIS, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "releae DP adn RDI"},
	{CMD_REMS, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_DIS, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_EN, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "read manufacture device id"},	
	{CMD_RDID, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_DIS, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_EN, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "read identification"},
	{CMD_ENRESET, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_DIS, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "reset enable"},	
	{CMD_RESET, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_DIS, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "reset"},
	{CMD_ESR, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_EN, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "erase security registers"},
	{CMD_PSR, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_EN, ADDR_WIDTH_24, DATA_TX_EN, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "program security registers"},
	{CMD_RSR, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_EN, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_EN, DUMY_TX_EN, 1, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "read security registers"},
};

#ifdef CONFIG_ZX297520V3_UFI_MINI_32K_NOR
 static const struct nor_info spi_nor_ids[] = {
	 /* GigaDevice */
	 { "gd25q128", INFO(0xc86018, 0, 32 * 1024, 512, 0) },
	 { "gd25ld10c", INFO(0xc86011, 0, 32 * 1024, 4, 0) },
	 { "gd25ld20e", INFO(0xc86012, 0, 32 * 1024, 8, 0) },
	 /* winbond */	 
	 { "w25q128fw", INFO(0xef6018, 0, 32 * 1024, 512, 0) },
	 /* dosilicon */
	 { "fm25m4aa", INFO(0xf84218, 0, 32 * 1024, 512, 0) },
	 /* fudanwei */
	 { "fm25w128", INFO(0xA12818, 0, 32 * 1024, 512, 0) },
	 /* XMC */
	 { "XM25QU64C", INFO(0x204117, 0, 32 * 1024, 256, 0) },
	 { "XM25QU128", INFO(0x205018, 0, 32 * 1024, 512, 0) },
	 { "XM25QU128C", INFO(0x204118, 0, 32 * 1024, 512, 0) },
	 /* DQ25Q128AL */
	 { "DQ25Q128AL", INFO(0x546018, 0, 32 * 1024, 512, 0) },
	 /* dosilicon */
	 { "DS25M4AB", INFO(0xE54218, 0, 32 * 1024, 512, 0) },
	 /* esmt(eon) */
	 { "EN25SX128A", INFO(0x1C7818, 0, 32 * 1024, 512, 0) },
	 /* dosilicon */
	 { "FM25M4AA", INFO(0xF84218, 0, 32 * 1024, 512, 0) },
	 { },
 };
#else
 static const struct nor_info spi_nor_ids[] = {
	 /* GigaDevice */
	 { "gd25q128", INFO(0xc86018, 0, 64 * 1024, 256, 0) },
	 { "gd25ld10c", INFO(0xc86011, 0, 64 * 1024, 2, 0) },
	 { "gd25ld20e", INFO(0xc86012, 0, 64 * 1024, 4, 0) },
	 /* winbond */	 
	 { "w25q128fw", INFO(0xef6018, 0, 64 * 1024, 256, 0) },
	 /* dosilicon */
	 { "fm25m4aa", INFO(0xf84218, 0, 64 * 1024, 256, 0) },
	 /* fudanwei */
	 { "fm25w128", INFO(0xA12818, 0, 64 * 1024, 256, 0) },
	 /* xmc */
	 { "XM25QU128", INFO(0x205018, 0, 64 * 1024, 256, 0) },
	 { "XM25QU128C", INFO(0x204118, 0, 64 * 1024, 256, 0) },
	 /* DQ25Q128AL */
	 { "DQ25Q128AL", INFO(0x546018, 0, 64 * 1024, 256, 0) },
	 /* dosilicon */
	 { "DS25M4AB", INFO(0xE54218, 0, 64 * 1024, 256, 0) },
	 /* esmt(eon) */
	 { "EN25SX128A", INFO(0x1C7818, 0, 64 * 1024, 256, 0) },
	 /* dosilicon */
	 { "FM25M4AA", INFO(0xF84218, 0, 64 * 1024, 256, 0) },
	 { },
 };
#endif

 static void cs_open(void)
 {
     zx29_gpio_config(93,GPIO93_SPIFC_CS);    
 }

 static void cs_close(void)
 {
    zx29_gpio_config(93,GPIO93_GPIO93);
	gpio_direction_output(93,1);
 }

 void spifc_enable(void)
{
   volatile struct spifc_nor_reg_t* spifc = (struct spifc_nor_reg_t*)(ZX_SPIFC0_BASE);
    
    if(spifc->SFC_EN & FC_EN_BACK)
    {
		printk("spifc en err.\n");
		return;
	}   
    spifc->SFC_EN |= (1 << FC_EN);  
    spifc->SFC_CTRL0 |= (1 << FC_SCLK_PAUSE_EN);
}

 void spifc_disable(void)
{
    volatile struct spifc_nor_reg_t* spifc = (struct spifc_nor_reg_t*)(ZX_SPIFC0_BASE);
    
    if(!(spifc->SFC_EN & FC_EN_BACK))
    {
		printk("spifc dis err.\n");
		return;
	}     
    spifc->SFC_EN &= (~(1 <<FC_EN));    
}

 void spifc_setup_cmd(spinor_cmd_t *cmd, uint32_t addr, uint32_t len)
 {	 
	 volatile struct spifc_nor_reg_t* spifc = (struct spifc_nor_reg_t*)(ZX_SPIFC0_BASE);
 
	 /* clear dma config */
	 spifc->SFC_CTRL0 &= ~((1 << FC_RX_DMA_EN)|(1 << FC_TX_DMA_EN));
	 /* clear fifo */
	 spifc->SFC_CTRL0 |= (1 << FC_RXFIFO_CLR)|(1 << FC_TXFIFO_CLR);
 
	 /* clear interrupt register */
	 spifc->SFC_INT_SW_CLR = 0xFF;	
	 
	 /* dma + fifo config */
	 spifc->SFC_CTRL0 |= ((cmd->tx_dma_en << FC_TX_DMA_EN) 
						   | (cmd->rx_dma_en << FC_RX_DMA_EN)
						   | (1 << FC_RXFIFO_THRES)
						   | (1 << FC_TXFIFO_THRES));
 
	 /* addr dumy data code config */
	 spifc->SFC_CTRL1 = 0;
	 spifc->SFC_CTRL1 = ((cmd->addr_tx_en << FC_ADDR_TX_EN) 
						 | (cmd->dumy_tx_en << FC_DUMMY_TX_EN) 
						 | (cmd->data_rx_en << FC_READ_DAT_EN) 
						 | (cmd->data_tx_en << FC_WRITE_DAT_EN));
 
	 spifc->SFC_CTRL2 = 0;
	 spifc->SFC_CTRL2 = ((cmd->dumy_byte_num << FC_DUMMY_BYTE_NUM) 
						 | (cmd->dumy_bit_num << FC_DUMMY_BIT_NUM) 
						 | (cmd->addr_byte_num << FC_ADDR_BYTE_NUM)
						 | (cmd->addr_multi_line_en << FC_ADDR_MULTI_LINE_EN)
						 | (cmd->data_multi_line_en << FC_DAT_MULTI_LINE_EN)
						 | (cmd->trans_mod << FC_TRANS_MOD));
 
	 if(len)
		 spifc->SFC_BYTE_NUM = len - 1;
	 else
		 spifc->SFC_BYTE_NUM = 0; 
	 
	 spifc->SFC_ADDR = addr; 
	 spifc->SFC_INS = cmd->cmd; 		 
 }
 
  int spifc_wait_cmd_end(void)
 {
	 volatile struct spifc_nor_reg_t* spifc = (struct spifc_nor_reg_t*)(ZX_SPIFC0_BASE);
	 uint32_t intr_status = 0;
	 
     while( spifc->SFC_START & FC_BUSY );
	 
	 while(!(spifc->SFC_INT_RAW & FC_INT_RAW_MASK));
	 
	 intr_status = spifc->SFC_INT_RAW;
	 spifc->SFC_INT_SW_CLR = intr_status;		
 
	 if(intr_status & FC_INT_RAW_CMD_END)
	 {	 
		 return 0;
	 }
	 else
	 {
		 printk("intr status err.\n");
		 return -1;
	 }	 
 }

   spinor_cmd_t *cmd_seek(u8 opcode)
 {
	 int i;
 
	 for(i = 0; (&nor_cmd_table[i]) != NULL; i++)
	 {
		 if(opcode == nor_cmd_table[i].cmd)
		 {
			 return (&nor_cmd_table[i]);
		 }
	 }
 
	 return NULL;	 
 }
   
 uint32_t spifc_read_fifo(uint8_t *buf, uint32_t len)
 {
	 volatile struct spifc_nor_reg_t* spifc = (struct spifc_nor_reg_t*)(ZX_SPIFC0_BASE);
	 uint32_t *p = (uint32_t *)buf;
	 uint32_t cnt = 0;
 
	 int remainder_cnt = len % 4;
 
	 if(remainder_cnt != 0)
	 {
		 len = len + (4 - remainder_cnt);
	 }
	 else
	 {
		 remainder_cnt = 4;
	 }
	
	 while(cnt < (len>>2))	
	 {
		 if(spifc->SFC_SW & (FC_RX_FIFO_CNT_MASK<<FC_RX_FIFO_CNT))//rx fifo not empty
		 {			 
			 p[cnt++]= spifc->SFC_DATA;
		 }
	 }
 
	 return ((cnt<<2) - (4 - remainder_cnt));
 }

 int spifc_write_data(uint8_t *buf, uint32_t len)
{
    uint32_t *p = (uint32_t *)buf;
    uint32_t cnt = 0;
    volatile struct spifc_nor_reg_t* spifc = (struct spifc_nor_reg_t*)(ZX_SPIFC0_BASE);
	int remainder_cnt = len % 4;
	
	if(remainder_cnt != 0)
	{
		len = len + (4 - remainder_cnt);
	}
	else
	{
		remainder_cnt = 4;
	}
	
    while(cnt < (len >> 2))  
    {
        if(spifc->SFC_SW & (FC_TX_FIFO_CNT_MASK<<FC_TX_FIFO_CNT))//tx fifo not full
        {       
            spifc->SFC_DATA = p[cnt++];
        }
    }

    return ((cnt<<2) - (4 - remainder_cnt));
}
 
  void spifc_start(void)
 {
	 volatile struct spifc_nor_reg_t* spifc = (struct spifc_nor_reg_t*)(ZX_SPIFC0_BASE);
	 
	 spifc->SFC_START |= FC_START;
 }
 

 int nor_read_reg(u8 opcode, int len, u8 *buf)
{
	int ret = 0;
	spinor_cmd_t *cmd = NULL;
	
	cs_open();//spp
	cmd = cmd_seek(opcode);
	if(cmd == NULL)
	{
		printk("cmd_seek unkown cmd error.\n");
		return -1;
	}
	
	spifc_setup_cmd(cmd, 0x0, len);            
    spifc_start(); 

    ret = spifc_wait_cmd_end();             
    if(ret != 0)
    {
		printk("spifc_wait_cmd_end error.\n");
	   	return ret;
    }

	ret = spifc_read_fifo(buf, len);
	cs_close();//spp
	if(ret < 0)
	{
		printk("spifc_read_fifo error.\n");
	    return ret;
	}

	return 0;	
}

 int nor_write_reg(u8 opcode, u8 *buf, int len)
{
	 int ret;
	 spinor_cmd_t *cmd = NULL;
	 int nor_status = 0;
	 
	 cs_open();//spp
	 
	 cmd = cmd_seek(opcode);
	 if(cmd == NULL)
	 {
		 printk("cmd_seek unkown cmd = 0x%x error.\n", opcode);
		 return -1;
	 }
	 
	 spifc_setup_cmd(cmd, 0x0, len); 	 

	 if(len > 0)
	 {
		 ret = spifc_write_data(buf, len);
		 if(ret != len)
		 {
			 printk("spifc_write_data error.\n");
			 return ret;
		 }
	 }

	 spifc_start();

	 ret = spifc_wait_cmd_end();			  
	 if(ret != 0)
	 {
		 printk("spifc_wait_cmd_end error.\n");
		 return ret;
	 }

	 do{
		 ret = nor_read_reg(CMD_RDSR0, 1, &nor_status);
		 if(ret != 0)
		 {
			 printk("read WIP fail.\n");
		 }
	 }while(nor_status & 0x1);
    cs_close();//spp

 return 0;
}

 int nor_read_id(void)
 {
	 int tmp;
	 u8  id[SPI_NOR_MAX_ID_LEN];
	 const struct nor_info	 *info;
	 tmp = nor_read_reg(CMD_RDID, SPI_NOR_MAX_ID_LEN, id);
	 if(tmp < 0) 
	 {
		 printk("error reading JEDEC ID\n");
		 return tmp;
	 }
 
	 for (tmp = 0; tmp < ARRAY_SIZE(spi_nor_ids) - 1; tmp++) 
	 {
		 info = &spi_nor_ids[tmp];
		 if(info->id_len) 
		 {
			 if(!memcmp(info->id, id, info->id_len))
			 {
				 spi_nor_flash = info;
				 printk("nor id is 0x%x,0x%x,0x%x.\n",spi_nor_flash->id[0],spi_nor_flash->id[1],spi_nor_flash->id[2]);
				 return 0;
			 }
		 }
	 }
	 printk("unrecognized JEDEC id\n");
	 return -1;
	 
 }

 int spi_nor_erase(uint32_t offs)
{
	int ret;
	int nor_status = 0;
	spinor_cmd_t *cmd = NULL;

	cmd = cmd_seek(CMD_64KBE);
	if(cmd == NULL)
	{
		printk("cmd_seek unkown cmd = 0x%x error.\n", CMD_64KBE);
		return -1;
	}

	cmd->tx_dma_en = TX_DMA_DIS;
	cs_open();
	spifc_setup_cmd(cmd, offs, 0);

	spifc_start();
    ret = spifc_wait_cmd_end();             
    if(ret != 0)
    {
		printk("spifc_wait_cmd_end error.\n");
	   	return ret;
    }
    cs_close();
	udelay(5);
	
	do{
		ret = nor_read_reg(CMD_RDSR0, 1, &nor_status);
		if(ret != 0)
		{
			printk("read WIP fail.\n");
		}
	}while(nor_status & 0x1);
	
	return 0;
}
int nor_erase(u32 addr)
{
    int ret;
	u32 page_offset, page_size, i;
	struct nor_info *info = spi_nor_flash;

	soft_spin_lock(NAND_SFLOCK);
	mutex_lock(&otpMutex);
	
	nor_write_reg(CMD_WREN, NULL, 0);
	ret = spi_nor_erase(addr);

	mutex_unlock(&otpMutex);
	soft_spin_unlock(NAND_SFLOCK);
	
	return ret;
}
 int spi_nor_write(uint32_t to, size_t len, u_char *buf)
   {
	  int ret;
	  int nor_status = 0;
	  spinor_cmd_t *cmd = NULL;
 
	  cmd = cmd_seek(CMD_PP);
	  
	  if(cmd == NULL)
	  {
		  printk("cmd_seek unkown cmd = 0x%x error.\n", CMD_PSR);
		  return -1;
	  }
  
	  cmd->tx_dma_en = TX_DMA_DIS;
	  cs_open();//spp
	  spifc_setup_cmd(cmd, to, len); 
 
	  spifc_start();
 
	  ret = spifc_write_data(buf, len);
	  if(ret < 0)
	  {
		  printk("spifc_write_fifo error.\n");
		  return ret;
	  }
 
	  ret = spifc_wait_cmd_end();			   
	  if(ret != 0)
	  {
		  printk("spifc_wait_cmd_end error.\n");
		  return ret;
	  }
	  cs_close();//spp
	  udelay(5);
	  
	  do{
		  ret = nor_read_reg(CMD_RDSR0, 1, &nor_status);
		  if(ret != 0)
		  {
			  printk("read WIP fail.\n");
		  }
	  }while(nor_status & 0x1);
	  
	  return 0;
   }
   
  
  int spi_nor_read(uint32_t from, size_t len, u_char *buf)
  {
	  int ret;
	  spinor_cmd_t *cmd = NULL;
  
	  cmd = cmd_seek(CMD_RDFT);
 
	  if(cmd == NULL)
	  {
		  printk("cmd_seek unkown error.\n");
		  return -1;
	  }
 
	  cmd->rx_dma_en = RX_DMA_DIS;
	  cs_open();//spp
	  spifc_setup_cmd(cmd, from, len);				  
	  spifc_start();
	  ret = spifc_read_fifo(buf, len);
	  if(ret < 0)
	  {
		  printk("spifc_read_fifo error.\n");
		  return ret;
	  }
  
	  ret = spifc_wait_cmd_end();			  
	  if(ret != 0)
	  {
		  printk("spifc_wait_cmd_end error.\n");
		  cs_close();//spp
		  return ret;
	  }
	  cs_close();//spp
	  return 0;
  }

   int nor_write(u32 to, u32 len, u32 buf)
  {
	  int ret;
	  u32 page_offset, page_size, i;
	  struct nor_info *info = spi_nor_flash;
	  char temp_addr[256] = {0};
	  unsigned long long start_time,end_time,delta_time = 0;

	  soft_spin_lock(NAND_SFLOCK);
	  mutex_lock(&otpMutex);
 
	  nor_write_reg(CMD_WREN, NULL, 0);
	  
	  page_offset = to & (info->page_size - 1);
  
	  /* do all the bytes fit onto one page? */
	  if (page_offset + len <= info->page_size) {
		  ret = spi_nor_write(to, len, (uint8_t *)buf);
	  } else {
		  /* the size of data remaining on the first page */
		  page_size = info->page_size - page_offset;
		  ret = spi_nor_write(to, page_size, (uint8_t *)buf);
  
		  /* read everything in nor->page_size chunks */
		  for (i = page_size; i < len; i += page_size) {
			  page_size = len - i;
			  if (page_size > info->page_size)
				  page_size = info->page_size;
 
			  nor_write_reg(CMD_WREN, NULL, 0);
			  start_time = cpu_clock(0)>>10;
			  ret = spi_nor_write(to + i, page_size, ((uint8_t *)buf + i));
			  end_time = cpu_clock(0)>>10;
			  delta_time = end_time - start_time;
			  //printk("time consumed: %lld\n",delta_time);
			  spi_nor_read(to + i, page_size, ((uint8_t *)temp_addr)); 
			  while(memcmp(((uint8_t *)buf + i), ((uint8_t *)temp_addr),info->page_size)) {
			  	    printk("addr 0x%x write fail.\n",to+i);
					nor_write_reg(CMD_WREN, NULL, 0);
					start_time = cpu_clock(0)>>10;
					ret = spi_nor_write(to + i, page_size, ((uint8_t *)buf + i));
					end_time = cpu_clock(0)>>10;
					delta_time = end_time - start_time;
				    printk("time consumed: %lld\n",delta_time);
			        spi_nor_read(to + i, page_size, ((uint8_t *)temp_addr)); 
			  }
		  }
	  }

	  mutex_unlock(&otpMutex);
	  soft_spin_unlock(NAND_SFLOCK);
  
	  return ret;
  }
  int nor_read(u32 from, u32 len, u32 to)
  {
	  int ret;
	  u32 page_offset, page_size, i;
	  struct nor_info *info = spi_nor_flash;

	  soft_spin_lock(NAND_SFLOCK);
	  mutex_lock(&otpMutex);
	  
	  page_offset = from & (info->page_size - 1);
  
	  /* do all the bytes fit onto one page? */
	  if (page_offset + len <= info->page_size) {
		  ret = spi_nor_read(from, len, (uint8_t *)to);
	  } else {
		  /* the size of data remaining on the first page */
		  page_size = info->page_size - page_offset;
		  ret = spi_nor_read(from, page_size, (uint8_t *)to);
  
		  /* read everything in nor->page_size chunks */
		  for (i = page_size; i < len; i += page_size) {
			  page_size = len - i;
			  if (page_size > info->page_size)
				  page_size = info->page_size;
  
			  ret = spi_nor_read(from + i, page_size, ((uint8_t *)to + i));
		  }
	  }

	  mutex_unlock(&otpMutex);
	  soft_spin_unlock(NAND_SFLOCK);
  
	  return ret;
  }

  int spi_nor_write_security_register(uint32_t to, size_t len, u_char *buf)
  {
	 int ret;
	 int nor_status = 0;
	 spinor_cmd_t *cmd = NULL;

	 cmd = cmd_seek(CMD_PSR);
	 
	 if(cmd == NULL)
	 {
		 printk("cmd_seek unkown cmd = 0x%x error.\n", CMD_PSR);
		 return -1;
	 }
 
	 cmd->tx_dma_en = TX_DMA_DIS;
     cs_open();//spp
	 spifc_setup_cmd(cmd, to, len);	

	 spifc_start();

	 ret = spifc_write_data(buf, len);
	 if(ret < 0)
	 {
		 printk("spifc_write_fifo error.\n");
		 return ret;
	 }

	 ret = spifc_wait_cmd_end();			  
	 if(ret != 0)
	 {
		 printk("spifc_wait_cmd_end error.\n");
		 return ret;
	 }
     cs_close();//spp
     udelay(5);
	 
	 do{
		 ret = nor_read_reg(CMD_RDSR0, 1, &nor_status);
		 if(ret != 0)
		 {
			 printk("read WIP fail.\n");
		 }
	 }while(nor_status & 0x1);
	 
	 return 0;
  }
  
  int nor_write_security_register(uint32_t to, uint32_t len, uint32_t buf)
 {
	 int ret;
	 u32 page_offset, page_size, i;
	 struct nor_info *info = spi_nor_flash;

	 nor_write_reg(CMD_WREN, NULL, 0);
	 
	 page_offset = to & (info->page_size - 1);
 
	 /* do all the bytes fit onto one page? */
	 if (page_offset + len <= info->page_size) {
		 ret = spi_nor_write_security_register(to, len, (uint8_t *)buf);
	 } else {
		 /* the size of data remaining on the first page */
		 page_size = info->page_size - page_offset;
		 ret = spi_nor_write_security_register(to, page_size, (uint8_t *)buf);
 
		 /* read everything in nor->page_size chunks */
		 for (i = page_size; i < len; i += page_size) {
			 page_size = len - i;
			 if (page_size > info->page_size)
				 page_size = info->page_size;

			 nor_write_reg(CMD_WREN, NULL, 0);
			 ret = spi_nor_write_security_register(to + i, page_size, ((uint8_t *)buf + i));
		 }
	 }
 
	 return ret;
 }
 int spi_nor_read_security_register(uint32_t from, size_t len, u_char *buf)
 {
	 int ret;
	 spinor_cmd_t *cmd = NULL;
 
	 cmd = cmd_seek(CMD_RSR);

	 if(cmd == NULL)
	 {
		 printk("cmd_seek unkown error.\n");
		 return -1;
	 }

	 cmd->rx_dma_en = RX_DMA_DIS;
     cs_open();//spp
	 spifc_setup_cmd(cmd, from, len);				 
	 spifc_start();
	 ret = spifc_read_fifo(buf, len);
	 if(ret < 0)
	 {
		 printk("spifc_read_fifo error.\n");
		 return ret;
	 }
 
	 ret = spifc_wait_cmd_end();			 
	 if(ret != 0)
	 {
		 printk("spifc_wait_cmd_end error.\n");
		 cs_close();//spp
		 return ret;
	 }
	 cs_close();//spp
	 return 0;
 }
 
 int nor_read_security_register(uint32_t from, uint32_t len, uint32_t to)
 {
	 int ret;
	 u32 page_offset, page_size, i;
	 struct nor_info *info = spi_nor_flash;
	 
	 page_offset = from & (info->page_size - 1);
 
	 /* do all the bytes fit onto one page? */
	 if (page_offset + len <= info->page_size) {
		 ret = spi_nor_read_security_register(from, len, (uint8_t *)to);
	 } else {
		 /* the size of data remaining on the first page */
		 page_size = info->page_size - page_offset;
		 ret = spi_nor_read_security_register(from, page_size, (uint8_t *)to);
 
		 /* read everything in nor->page_size chunks */
		 for (i = page_size; i < len; i += page_size) {
			 page_size = len - i;
			 if (page_size > info->page_size)
				 page_size = info->page_size;
 
			 ret = spi_nor_read_security_register(from + i, page_size, ((uint8_t *)to + i));
		 }
	 }
 
	 return ret;
 }

int nor_init(void)
{
	int ret = 0;
	printk("nor init ... \n");
	soft_spin_lock(NAND_SFLOCK);
    
	//spifc_disable();
	
	//spifc_enable();

	ret = nor_read_id();
	if(ret != 0)
	{
	    soft_spin_unlock(NAND_SFLOCK);
		return -1;
	}
	
	soft_spin_unlock(NAND_SFLOCK);
	return 0;
}

/*******************************************************************************
* Function:     zDrvOtp_GetSecureMsg
* Description:  get pub key from iram, load by uboot
* Parameters:
*   Input:
*       None
*   Output:
*       None
* Returns:
*       None
* Others:
*******************************************************************************/
void zDrvOtp_GetSecureMsg(T_ZDrvOtp_Secure *secure)
{
	u32 i = 0;
	u32 ret = 0;
	int32_t offset = 0x1000;
	
	uint32_t len = sizeof(s_OtpStruct);
	
	s_ImageHeader *s_header = (s_ImageHeader *)(ZX_IRAM0_BASE + 4);
	s_OtpStruct *s_otp  = NULL;
	
    soft_spin_lock(NAND_SFLOCK);
	mutex_lock(&otpMutex);
	
	ret = nor_read_security_register(offset, len, &otp_data);
	if(ret !=0){
        printk("read otp data error.\n");
	}
	
	s_otp  = (s_OtpStruct *)(&otp_data);
	
	for(i = 0; i < 95; i ++)
	{
		secure->pubKey[i] = s_header->uiPubKey[i];
		//printk("uiPubKey[%d] is 0x%x\n",i,s_header->uiPubKey[i]);
	}

	secure->secureFlag = s_otp->secure_flag;

	for(i = 0; i < 8; i ++)
	{
		secure->pubKeyHash[i] = s_otp->puk_hash[i];
		printk("puk_hash[%d] is 0x%x\n",i,s_otp->puk_hash[i]);
	}

	for(i = 0; i < 3; i ++)
	{
		secure->secureDevId[i] = s_otp->secureDevId[i];
		printk("secureDevId[%d] is 0x%x\n",i,s_otp->secureDevId[i]);
	}

	mutex_unlock(&otpMutex);
	soft_spin_unlock(NAND_SFLOCK);
}
EXPORT_SYMBOL(zDrvOtp_GetSecureMsg);

/*******************************************************************************
* Function:     zDrvOtp_SetSecureMsg
* Description:  set otp value
* Parameters:
*   Input:
*       None
*   Output:
*       None
* Returns:
*       None
* Others:
*******************************************************************************/
int zDrvOtp_SetSecureMsg(E_ZDrvOtp_SecureMsg secure_msg, u32 *secure_buf)
{
	u32 i = 0;
	T_ZDrvOtp_Secure secure = {0};
	u32 tmpbuf[8] = {0xffffffff,0xffffffff,0xffffffff,0xffffffff,
		             0xffffffff,0xffffffff,0xffffffff,0xffffffff};
	int ret = 0;
	int nor_status = 0;
	if(secure_msg >= OTP_MAX_ENUM)
	{
		return -EINVAL;
	}

	zDrvOtp_GetSecureMsg(&secure);

	if(((secure_msg == OTP_SECURE_EN)&&(secure.secureFlag != 0xffffffff))||
	  ((secure_msg == OTP_PUB_KEY_HASH)&&(memcmp(secure.pubKeyHash,tmpbuf,sizeof(secure.pubKeyHash)) != 0))||
	  ((secure_msg == OTP_DEVICE_ID)&&(memcmp(secure.secureDevId,tmpbuf,sizeof(secure.secureDevId)) != 0)))
		return -1;
	
    soft_spin_lock(NAND_SFLOCK);
	mutex_lock(&otpMutex);

	switch(secure_msg)
	{
		case OTP_SECURE_EN:
		{
		    ret = nor_write_security_register(SECURE_EN_OFFSET, 4, secure_buf);
			if(ret < 0){
				mutex_unlock(&otpMutex);
				cs_close();//spp
				return ret;
			}
			/* permanently locked*/
			ret = nor_read_reg(CMD_RDSR0, 1, &nor_status);
			if(ret != 0)
			{
				printk("read reg fail.\n");
			}
		    nor_status |= (1<<6);
			nor_write_reg(CMD_WREN, NULL, 0);
			ret = nor_write_reg(CMD_WRSR, &nor_status, 1);
			if(ret != 0)
			{
				printk("write reg fail.\n");
			}
			
			break;
		}
		case OTP_PUB_KEY_HASH:
		{    
		    //printk("pub key hash is 0x%x.\n",*(u32 *)secure_buf);
		    ret = nor_write_security_register(PUB_KEY_HASH_OFFSET, 32, secure_buf);
			if(ret < 0){
				mutex_unlock(&otpMutex);
				cs_close();//spp
				return -ret;
			}
			break;
		}
		case OTP_DEVICE_ID:
		{
			ret = nor_write_security_register(DEVICE_ID_OFFSET, 12, secure_buf);
			if(ret < 0){
				mutex_unlock(&otpMutex);
				cs_close();//spp
				return ret;
			}
			break;
		}
		default:
			break;
	}

	mutex_unlock(&otpMutex);
    soft_spin_unlock(NAND_SFLOCK);
	return 0;
}

int zx29_otp_set_secure(E_ZDrvOtp_SecureMsg secure_msg, u32 *secure_buf)
{
	int ret = 0;
	int	len = 0;
	void *buf=NULL;

	switch(secure_msg)
	{
		case OTP_SECURE_EN:
			len = sizeof(u32);
			break;
		case OTP_PUB_KEY_HASH:
			len = 8 * sizeof(u32);
			break;
		case OTP_DEVICE_ID:
			len = 3 * sizeof(u32);
			break;

		default:
			return -EINVAL;
	}

	buf = kmalloc(len, GFP_KERNEL);
	if(NULL == buf)
		return -ENOSPC;
	
	ret = copy_from_user(buf, secure_buf, len);
	if(ret)
		printk("%s: copy user failed\n",__func__);
	
	ret = zDrvOtp_SetSecureMsg(secure_msg, buf);
	kfree(buf);

	return ret;
}


static long zx_otp_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	int ret = 0;
	T_ZDrvOtp_Secure secure;

	if (arg == NULL)
		return -EFAULT;
	
	switch(cmd)
	{
		case OTP_GET_DATA:
			printk("start get otp data...\n");
			zDrvOtp_GetSecureMsg(&secure);		    
			ret = copy_to_user((void *)arg, &secure, sizeof(T_ZDrvOtp_Secure));
			if(ret){
                printk("%s: copy user failed\n",__func__);
				printk("write get otp data fail...\n");
			}else{
				printk("already get otp data...\n");
			}
				
			break;
		case OTP_SET_SECURE_EN:
			printk("start set secure en...\n");
			ret = zx29_otp_set_secure(OTP_SECURE_EN, (u32 *)arg);
		    if(ret)
				printk("set secure en fail...,ret is 0x%x\n",ret);
			else
				printk("already set secure en...\n");
			break;
		case OTP_SET_PUB_KEY_HASH:
			printk("start set pub key hash...\n");
			ret = zx29_otp_set_secure(OTP_PUB_KEY_HASH, (u32 *)arg);
		    if(ret)
				printk("set pub key hash fail...,ret is 0x%x\n",ret);
			else
				printk("already set pub key hash...\n");
			break; 
		case OTP_SET_DEVICE_ID:
			ret = zx29_otp_set_secure(OTP_DEVICE_ID, (u32 *)arg);
			break; 
		default:
			return -EPERM;
		}
	
	return ret;
}

static const struct file_operations	zx_otp_ops = {
	.owner 		= THIS_MODULE,
	.unlocked_ioctl = zx_otp_ioctl,
};

static struct miscdevice zx_otp_miscdev = {
	.minor		= MISC_DYNAMIC_MINOR,
	.name		= "otp",
	.fops		= &zx_otp_ops,
};


static int __init zx_otp_init(void)
{
	int ret = 0;

	//mutex_init(&otpMutex);
	
	ret = misc_register(&zx_otp_miscdev);
	if (ret) {
		printk(KERN_ERR"%s: otp failed to register miscdev (ret = %d)\n", __FILE__, ret);
		return ret;
	}

	printk("[xxx] otp dev inited! \n");	

	if(nor_cmdline != NULL)
	{
	    if (!strcmp(nor_cmdline, "1"))
		{
			ret = nor_init();
			if(ret){
		        printk("nor init err.\n");
				return ret;
			}	    
		}
	}
	

	return ret;
}

module_init(zx_otp_init);

