/*
 * ZTE cpu context save&restore driver
 *
 * Copyright (C) 2013 ZTE Ltd.
 * 	by zxp
 * 
 */
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/spinlock.h>
#include <linux/interrupt.h>
#include <linux/types.h>
#include <linux/suspend.h>

#include "zx-pm.h"

/*=======================================================================
 *=======================================================================
 *=======  [ZX-PM] timer interface for power management  ===========================
 *======================================================================= 
 *=======================================================================*/

typedef struct
{
    /* 0x00 */ volatile unsigned timer_load;
    /* 0x04 */ volatile unsigned timer_counter;
    /* 0x08 */ volatile unsigned timer_control;
    /* 0x0c */ volatile unsigned timer_interrupt_status;
                char padding1[0x10];
    /* 0x20 */ volatile unsigned watchdog_load;
    /* 0x24 */ volatile unsigned watchdog_counter;
    /* 0x28 */ volatile unsigned watchdog_control;
    /* 0x2c */ volatile unsigned watchdog_interrupt_status;
    /* 0x30 */ volatile unsigned watchdog_reset_status;
    /* 0x34 */ volatile unsigned watchdog_disable;
} a9_timer_registers;

typedef struct
{
    unsigned timer_load;
    unsigned timer_counter;
    unsigned timer_control;
    unsigned timer_interrupt_status;
    unsigned watchdog_load;
    unsigned watchdog_counter;
    unsigned watchdog_control;
    unsigned watchdog_interrupt_status;
} a9_timer_context;


void save_a9_timers(u32 *pointer, unsigned twd_address)
{
    a9_timer_context *context = (a9_timer_context *)pointer;
    a9_timer_registers *timers = (a9_timer_registers *)twd_address;

    /* 
     * First, stop the timers
     */
    context->timer_control    = timers->timer_control;
    timers->timer_control     = 0;
    context->watchdog_control = timers->watchdog_control;
    timers->watchdog_control  = 0;

    context->timer_load                = timers->timer_load;
    context->timer_counter             = timers->timer_counter;
    context->timer_interrupt_status    = timers->timer_interrupt_status;
    context->watchdog_load             = timers->watchdog_load;
    context->watchdog_counter          = timers->watchdog_counter;
    context->watchdog_interrupt_status = timers->watchdog_interrupt_status;
    /* 
     * We ignore watchdog_reset_status, since it can only clear the status bit.
     * If the watchdog has reset the system, the OS will want to know about it.
     * Similarly, we have no use for watchdog_disable - this is only used for
     * returning to timer mode, which is the default mode after reset.
     */
}

void restore_a9_timers(u32 *pointer, unsigned twd_address)
{
    a9_timer_context *context = (a9_timer_context *)pointer;
    a9_timer_registers *timers = (a9_timer_registers *)twd_address;

    timers->timer_control = 0;
    timers->watchdog_control = 0;

    /*
     * We restore the load register first, because it also sets the counter register.
     */
    timers->timer_load    = context->timer_load;
    timers->watchdog_load = context->watchdog_load;

    /*
     * If a timer has reached zero (presumably during the context save) and triggered
     * an interrupt, then we set it to the shortest possible expiry time, to make it
     * trigger again real soon.
     * We could fake this up properly, but we would have to wait around until the timer 
     * ticked, which could be some time if PERIPHCLK is slow. This approach should be 
     * good enough in most cases.
     */
    if (context->timer_interrupt_status)
    {
        timers->timer_counter = 1;
    }
    else
    {
        timers->timer_counter = context->timer_counter;
    }

    if (context->watchdog_interrupt_status)
    {
        timers->watchdog_counter = 1;
    }
    else
    {
        timers->watchdog_counter = context->watchdog_counter;
    }

    timers->timer_control = context->timer_control;
    timers->watchdog_control = context->watchdog_control;
}


typedef struct
{
    /* 0x00 */ volatile unsigned timer_version;
    /* 0x04 */ volatile unsigned timer_config;
    /* 0x08 */ volatile unsigned timer_load;
    /* 0x0c */ volatile unsigned timer_start;
    /* 0x10 */ volatile unsigned timer_set_en;
    /* 0x14 */ volatile unsigned timer_ack;
    /* 0x18 */ volatile unsigned timer_count;
} a53_global_timer_registers;

typedef struct
{
    unsigned timer_version;
    unsigned timer_config;
    unsigned timer_load;
    unsigned timer_start;
    unsigned timer_set_en;
    unsigned timer_ack;
    unsigned timer_count;

} a53_global_timer_context;

#define A53_GT_TIMER_ENABLE          (1<<0)
#define A53_GT_COMPARE_ENABLE        (1<<1)
#define A53_GT_AUTO_INCREMENT_ENABLE (1<<3)
#define A53_GT_EVENT_FLAG            (1<<0)

void save_a53_sys_timer(u32 *pointer, unsigned timer_address)
{
    a53_global_timer_registers *timer = (void*)timer_address;
    a53_global_timer_context *context = (void*)pointer;

    context->timer_config = timer->timer_config;
 
    context->timer_load     = timer->timer_load;
	context->timer_start    = timer->timer_start;
	timer->timer_start = 0;
	context->timer_set_en = timer->timer_set_en;
	context->timer_ack = timer->timer_ack; /* ֻûָ*/
	context->timer_count = timer->timer_count;
   
}

void restore_a53_sys_timer(u32 *pointer, unsigned timer_address)
{
    a53_global_timer_registers *timer = (void*)timer_address;
    a53_global_timer_context *context = (void*)pointer;

	 timer->timer_config =context->timer_config;
	 timer->timer_load = context->timer_load;/*ӦǸֵ*/
	 timer->timer_set_en =context->timer_set_en;
  	 timer->timer_start = context->timer_start;
}


/*=======================================================================
 *=======================================================================
 *=======  [ZX-PM] SCU interface for power management  ============================
 *======================================================================= 
 *=======================================================================*/
typedef struct
{
    /* 0x00 */  volatile unsigned int control;
    /* 0x04 */  const unsigned int configuration;
    /* 0x08 */  union
                {
                    volatile unsigned int w;
                    volatile unsigned char b[4];
                } power_status;
    /* 0x0c */  volatile unsigned int invalidate_all;
                char padding1[48];
    /* 0x40 */  volatile unsigned int filtering_start;
    /* 0x44 */  volatile unsigned int filtering_end;
                char padding2[8];
    /* 0x50 */  volatile unsigned int access_control;
    /* 0x54 */  volatile unsigned int ns_access_control;
} a53_scu_registers;

/* 
 * TODO: we need to use the power status register, not save it!
 */

void save_a53_scu(u32 *pointer, unsigned scu_address)
{
    a53_scu_registers *scu = (a53_scu_registers *)scu_address;
    
    pointer[0] = scu->control;
    pointer[1] = scu->power_status.w;
    pointer[2] = scu->filtering_start;
    pointer[3] = scu->filtering_end;
    pointer[4] = scu->access_control;
    pointer[5] = scu->ns_access_control;
}

void restore_a53_scu(u32 *pointer, unsigned scu_address)
{
    a53_scu_registers *scu = (a53_scu_registers *)scu_address;
    
    scu->invalidate_all = 0xffff;
    scu->filtering_start = pointer[2];
    scu->filtering_end = pointer[3];
//zxp    scu->access_control = pointer[4];
    scu->ns_access_control = pointer[5];
    scu->power_status.w = pointer[1];
    scu->control = pointer[0];
}

void set_status_a53_scu(unsigned cpu_index, unsigned status, unsigned scu_address)
{
    a53_scu_registers *scu = (a53_scu_registers *)scu_address;
    unsigned power_status;

    switch(status)
    {
        case CPU_POWER_MODE_STANDBY:
        case CPU_POWER_MODE_DORMANT:
            power_status = 2;
            break;
        case CPU_POWER_MODE_SHUTDOWN:
            power_status = 3;
            break;
        default:
            power_status = 0;
    }
        
    scu->power_status.b[cpu_index] = power_status;
    dsb();
}

void init_lp_of_scu(unsigned scu_address)
{
    a53_scu_registers *scu = (a53_scu_registers *)scu_address;

	scu->control |= 0x61;
}


int num_cpus_from_a53_scu(unsigned scu_address)
{
    a53_scu_registers *scu = (a53_scu_registers *)scu_address;

    return ((scu->configuration) & 0x3) + 1;
}


/*=======================================================================
 *=======================================================================
 *=======  [ZX-PM] PL310 interface for power management  ==========================
 *=======================================================================
 *=======================================================================*/


#define C_BIT 0x01

struct lockdown_regs
{
    unsigned int d, i;
};

typedef struct
{
    /* 0x000 */ const unsigned cache_id;
    /* 0x004 */ const unsigned cache_type;
                char padding1[0x0F8];
    /* 0x100 */ volatile unsigned control;
    /* 0x104 */ volatile unsigned aux_control;
    /* 0x108 */ volatile unsigned tag_ram_control;
    /* 0x10C */ volatile unsigned data_ram_control;
                char padding2[0x0F0];
    /* 0x200 */ volatile unsigned ev_counter_ctrl;
    /* 0x204 */ volatile unsigned ev_counter1_cfg;
    /* 0x208 */ volatile unsigned ev_counter0_cfg;
    /* 0x20C */ volatile unsigned ev_counter1;
    /* 0x210 */ volatile unsigned ev_counter0;
    /* 0x214 */ volatile unsigned int_mask;
    /* 0x218 */ const volatile unsigned int_mask_status;
    /* 0x21C */ const volatile unsigned int_raw_status;
    /* 0x220 */ volatile unsigned int_clear;
                char padding3[0x50C];
    /* 0x730 */ volatile unsigned cache_sync;
                char padding4[0x03C];
    /* 0x770 */ volatile unsigned inv_pa;
                char padding5[0x008];
    /* 0x77C */ volatile unsigned inv_way;
                char padding6[0x030];
    /* 0x7B0 */ volatile unsigned clean_pa;
                char padding7[0x004];
    /* 0x7B8 */ volatile unsigned clean_index;
    /* 0x7BC */ volatile unsigned clean_way;
                char padding8[0x030];
    /* 0x7F0 */ volatile unsigned clean_inv_pa;
                char padding9[0x004];
    /* 0x7F8 */ volatile unsigned clean_inv_index;
    /* 0x7FC */ volatile unsigned clean_inv_way;
                char paddinga[0x100];
    /* 0x900 */ volatile struct lockdown_regs lockdown[8];
                char paddingb[0x010];
    /* 0x950 */ volatile unsigned lock_line_en;
    /* 0x954 */ volatile unsigned unlock_way;
                char paddingc[0x2A8];
    /* 0xC00 */ volatile unsigned addr_filtering_start;
    /* 0xC04 */ volatile unsigned addr_filtering_end;
                char paddingd[0x338];
    /* 0xF40 */ volatile unsigned debug_ctrl;
                char paddinge[0x01C];
    /* 0xF60 */ volatile unsigned prefetch_ctrl;
                char paddingf[0x01C];
    /* 0xF80 */ volatile unsigned power_ctrl;
} pl310_registers;


typedef struct
{
    unsigned int aux_control;
    unsigned int tag_ram_control;
    unsigned int data_ram_control;
    unsigned int ev_counter_ctrl;
    unsigned int ev_counter1_cfg;
    unsigned int ev_counter0_cfg;
    unsigned int ev_counter1;
    unsigned int ev_counter0;
    unsigned int int_mask;
    unsigned int lock_line_en;
    struct lockdown_regs lockdown[8];
    unsigned int unlock_way;
    unsigned int addr_filtering_start;
    unsigned int addr_filtering_end;
    unsigned int debug_ctrl;
    unsigned int prefetch_ctrl;
    unsigned int power_ctrl;
} pl310_context;

/* TODO: should be determined from cache? */
static unsigned const cache_line_size = 32;

void clean_inv_range_pl310(void *start, unsigned size, unsigned pl310_address)
{
    unsigned addr;
    pl310_registers *pl310 = (pl310_registers *)pl310_address;
    
    /* Align the start address to the start of a cache line */
    addr = (unsigned)start & ~(cache_line_size - 1);

    /* Wait for any background operations to finish */
    while(pl310->clean_inv_pa & C_BIT);
    
    while(addr <= size + (unsigned)start)
    {
        pl310->clean_inv_pa = addr; 
        addr += cache_line_size;
        /* For this to work on L220 we would have to poll the C bit now */
    }
    dmb();
}

void clean_range_pl310(void *start, unsigned size, unsigned pl310_address)
{
    unsigned addr;
    pl310_registers *pl310 = (pl310_registers *)pl310_address;
    
    /* Align the start address to the start of a cache line */
    addr = (unsigned)start & ~(cache_line_size - 1);
    
    /* Wait for any background operations to finish */
    while(pl310->clean_pa & C_BIT);
    
    while(addr <= size + (unsigned)start)
    {
        pl310->clean_pa = addr; 
        addr += cache_line_size;
        /* For this to work on L220 we would have to poll the C bit now */
    }
    dmb();
}

void inv_range_pl310(void *start, unsigned size, unsigned pl310_address)
{
    unsigned addr;
    pl310_registers *pl310 = (pl310_registers *)pl310_address;
    
    /* Align the start address to the start of a cache line */
    addr = (unsigned)start & ~(cache_line_size - 1);
    
    /* Wait for any background operations to finish */
    while(pl310->inv_pa & C_BIT);
    
    while(addr <= size + (unsigned)start)
    {
        pl310->inv_pa = addr; 
        addr += cache_line_size;
        /* For this to work on L220 we would have to poll the C bit now */
    }
}

void clean_inv_pl310(unsigned pl310_address)
{
    pl310_registers *pl310 = (pl310_registers *)pl310_address;
    int i;
    
    pl310->clean_inv_way = 0xffff;
    while (pl310->clean_inv_way)
    {
        /* Spin */
        for (i=10; i>0; --i)
        {
            __nop();
        }
    }
}

void clean_pl310(unsigned pl310_address)
{
    pl310_registers *pl310 = (pl310_registers *)pl310_address;
    int i;
    
    pl310->clean_way = 0xffff;
    while (pl310->clean_way)
    {
        /* Spin */
        for (i=10; i>0; --i)
        {
            __nop();
        }
    }
}

static void inv_pl310(unsigned pl310_address)
{
    pl310_registers *pl310 = (pl310_registers *)pl310_address;
    int i;
    
    pl310->inv_way = 0xffff;
    while (pl310->inv_way)
    {
        /* Spin */
        for (i=10; i>0; --i)
        {
            __nop();
        }
    }
}

void clean_disable_pl310(unsigned pl310_address)
{
    pl310_registers *pl310 = (pl310_registers *)pl310_address;
    int i;
    
    pl310->clean_way = 0xffff;
    while (pl310->clean_way)
    {
        /* Spin */
        for (i=10; i>0; --i)
        {
            __nop();
        }
    }

    pl310->cache_sync = 0;
    dsb();
    pl310->control = 0;
}


int is_enabled_pl310(unsigned pl310_address)
{
    pl310_registers *pl310 = (pl310_registers *)pl310_address;

    return (pl310->control & 1);
}    

void save_pl310(u32 *pointer, unsigned pl310_address)
{
    pl310_registers *pl310 = (pl310_registers *)pl310_address;
    pl310_context *context = (pl310_context *)pointer;
    int i;

    /* TODO: are all these registers are present in earlier PL310 versions? */
    context->aux_control = pl310->aux_control;
    context->tag_ram_control = pl310->tag_ram_control;
    context->data_ram_control = pl310->data_ram_control;
    context->ev_counter_ctrl = pl310->ev_counter_ctrl;
    context->ev_counter1_cfg = pl310->ev_counter1_cfg;
    context->ev_counter0_cfg = pl310->ev_counter0_cfg;
    context->ev_counter1 = pl310->ev_counter1;
    context->ev_counter0 = pl310->ev_counter0;
    context->int_mask = pl310->int_mask;
    context->lock_line_en = pl310->lock_line_en;

	/*
	 * The lockdown registers repeat 8 times for L310, the L210 has only one
	 * D and one I lockdown register at 0x0900 and 0x0904.
	 */
    for (i=0; i<8; ++i)
    {
        context->lockdown[i].d = pl310->lockdown[i].d;
        context->lockdown[i].i = pl310->lockdown[i].i;
    }

    context->addr_filtering_start = pl310->addr_filtering_start;
    context->addr_filtering_end = pl310->addr_filtering_end;
    context->debug_ctrl = pl310->debug_ctrl;
    context->prefetch_ctrl = pl310->prefetch_ctrl;
    context->power_ctrl = pl310->power_ctrl;
}

void restore_pl310(u32 *pointer, unsigned pl310_address, int dormant)
{
    pl310_registers *pl310 = (pl310_registers *)pl310_address;
    pl310_context *context = (pl310_context *)pointer;
    int i;
    
    /* We may need to disable the PL310 if the boot code has turned it on */
    if (pl310->control)
    {
        /* Wait for the cache to be idle, then disable */
        pl310->cache_sync = 0;
        dsb();
        pl310->control = 0;
    }
    
    /* TODO: are all these registers present in earlier PL310 versions? */
    pl310->aux_control = context->aux_control;
    pl310->tag_ram_control = context->tag_ram_control;
    pl310->data_ram_control = context->data_ram_control;
    pl310->ev_counter_ctrl = context->ev_counter_ctrl;
    pl310->ev_counter1_cfg = context->ev_counter1_cfg;
    pl310->ev_counter0_cfg = context->ev_counter0_cfg;
    pl310->ev_counter1 = context->ev_counter1;
    pl310->ev_counter0 = context->ev_counter0;
    pl310->int_mask = context->int_mask;
    pl310->lock_line_en = context->lock_line_en;

    for (i=0; i<8; ++i)
    {
        pl310->lockdown[i].d = context->lockdown[i].d;
        pl310->lockdown[i].i = context->lockdown[i].i;		
    }

    pl310->addr_filtering_start = context->addr_filtering_start;
    pl310->addr_filtering_end = context->addr_filtering_end;
    pl310->debug_ctrl = context->debug_ctrl;
    pl310->prefetch_ctrl = context->prefetch_ctrl;
    pl310->power_ctrl = context->power_ctrl;
    dsb();
    
    /*
     * If the RAMs were powered off, we need to invalidate the cache
     */
    if (!dormant)
    {
        inv_pl310(pl310_address);
    }
    
    pl310->control = 1;
    dsb();
}

void set_enabled_pl310(unsigned enabled, unsigned pl310_address)
{
    pl310_registers *pl310 = (pl310_registers *)pl310_address;

    if (enabled)
    {
	    inv_pl310(pl310_address);
		
        pl310->control |= 1;
        pl310->cache_sync = 0;
        dsb();		
    }
    else
    {
        /* Wait for the cache to be idle */
        pl310->cache_sync = 0;
        dsb();
        pl310->control &= ~1;
    }
}

void set_status_pl310(unsigned status, unsigned pl310_address)
{
    pl310_registers *pl310 = (pl310_registers *)pl310_address;

    if (status == CPU_POWER_MODE_STANDBY)
    {
        /* Wait for the cache to be idle */
        pl310->cache_sync = 0;
        dsb();
        pl310->power_ctrl |= 1;
    }
    else
    {
        pl310->power_ctrl &= ~1;
    }
}

void init_lp_of_l2(unsigned pl310_address)
{
    pl310_registers *pl310 = (pl310_registers *)pl310_address;

    pl310->power_ctrl |= 3;
}

/*=======================================================================
 *=======================================================================
 *=======  [ZX-PM] GIC interface for power management  ============================
 *=======================================================================
 *=======================================================================*/


/* This macro sets either the NS or S enable bit in the GIC distributor control register */
#define GIC_DIST_ENABLE      0x00000001

struct set_and_clear_regs
{
    volatile unsigned int set[32], clear[32];
};


typedef struct 
{
        volatile uint32_t GICD_CTLR;              // +0x0000 - RW - Distributor Control Register
  const volatile uint32_t GICD_TYPRE;             // +0x0004 - RO - Interrupt Controller Type Register
  const volatile uint32_t GICD_IIDR;              // +0x0008 - RO - Distributor Implementer Identification Register

  const volatile uint32_t padding0;               // +0x000C - RESERVED

        volatile uint32_t GICD_STATUSR;           // +0x0010 - RW - ????

  const volatile uint32_t padding1[3];            // +0x0014 - RESERVED

        volatile uint32_t IMP_DEF[8];             // +0x0020 - RW - Implementation defined registers

        volatile uint32_t GICD_SETSPI_NSR;        // +0x0040 - WO - Non-Secure Set SPI Pending (Used when SPI is signalled using MSI)
  const volatile uint32_t padding2;               // +0x0044 - RESERVED
        volatile uint32_t GICD_CLRSPI_NSR;        // +0x0048 - WO - Non-Secure Clear SPI Pending (Used when SPI is signalled using MSI)
  const volatile uint32_t padding3;               // +0x004C - RESERVED
        volatile uint32_t GICD_SETSPI_SR;         // +0x0050 - WO - Secure Set SPI Pending (Used when SPI is signalled using MSI)
  const volatile uint32_t padding4;               // +0x0054 - RESERVED
        volatile uint32_t GICD_CLRSPI_SR;         // +0x0058 - WO - Secure Clear SPI Pending (Used when SPI is signalled using MSI)

  const volatile uint32_t padding5[3];            // +0x005C - RESERVED

        volatile uint32_t GICD_SEIR;              // +0x0068 - WO - System Error Interrupt Register (Note: This was recently removed from the spec)

  const volatile uint32_t padding6[5];            // +0x006C - RESERVED

        volatile uint32_t GICD_IGROUPR[32];       // +0x0080 - RW - Interrupt Group Registers (Security Registers in GICv1)

        volatile uint32_t GICD_ISENABLER[32];     // +0x0100 - RW - Interrupt Set-Enable Registers
        volatile uint32_t GICD_ICENABLER[32];     // +0x0180 - RW - Interrupt Clear-Enable Registers
        volatile uint32_t GICD_ISPENDR[32];       // +0x0200 - RW - Interrupt Set-Pending Registers
        volatile uint32_t GICD_ICPENDR[32];       // +0x0280 - RW - Interrupt Clear-Pending Registers
        volatile uint32_t GICD_ISACTIVER[32];     // +0x0300 - RW - Interrupt Set-Active Register
        volatile uint32_t GICD_ICACTIVER[32];     // +0x0380 - RW - Interrupt Clear-Active Register

        volatile uint8_t  GICD_IPRIORITYR[1024];  // +0x0400 - RW - Interrupt Priority Registers
        volatile uint32_t GICD_ITARGETSR[256];    // +0x0800 - RW - Interrupt Processor Targets Registers
        volatile uint32_t GICD_ICFGR[64];         // +0x0C00 - RW - Interrupt Configuration Registers
        volatile uint32_t GICD_GRPMODR[32];       // +0x0D00 - RW - ????
  const volatile uint32_t padding7[32];           // +0x0D80 - RESERVED
        volatile uint32_t GICD_NSACR[64];         // +0x0E00 - RW - Non-Secure Access Control Register

        volatile uint32_t GICD_SGIR;              // +0x0F00 - WO - Software Generated Interrupt Register

  const volatile uint32_t padding8[3];            // +0x0F04 - RESERVED

        volatile uint32_t GICD_CPENDSGIR[4];      // +0x0F10 - RW - ???
        volatile uint32_t GICD_SPENDSGIR[4];      // +0x0F20 - RW - ???

  const volatile uint32_t padding9[52];           // +0x0F30 - RESERVED
  const volatile uint32_t padding10[5120];        // +0x1000 - RESERVED

        volatile uint64_t GICD_ROUTER[1024];      // +0x6000 - RW - Controls SPI routing when ARE=1
}interrupt_distributor;

typedef struct 
{
  const volatile uint32_t padding1[32];          // +0x0000 - RESERVED
        volatile uint32_t GICR_IGROUPR;          // +0x0080 - RW - Interrupt Group Registers (Security Registers in GICv1)
  const volatile uint32_t padding2[31];          // +0x0084 - RESERVED
        volatile uint32_t GICR_ISENABLER;        // +0x0100 - RW - Interrupt Set-Enable Registers
  const volatile uint32_t padding3[31];          // +0x0104 - RESERVED
        volatile uint32_t GICR_ICENABLER;        // +0x0180 - RW - Interrupt Clear-Enable Registers
  const volatile uint32_t padding4[31];          // +0x0184 - RESERVED
        volatile uint32_t GICR_ISPENDR;          // +0x0200 - RW - Interrupt Set-Pending Registers
  const volatile uint32_t padding5[31];          // +0x0204 - RESERVED
        volatile uint32_t GICR_ICPENDR;          // +0x0280 - RW - Interrupt Clear-Pending Registers
  const volatile uint32_t padding6[31];          // +0x0284 - RESERVED
        volatile uint32_t GICR_ISACTIVER;        // +0x0300 - RW - Interrupt Set-Active Register
  const volatile uint32_t padding7[31];          // +0x0304 - RESERVED
        volatile uint32_t GICR_ICACTIVER;        // +0x0380 - RW - Interrupt Clear-Active Register
  const volatile uint32_t padding8[31];          // +0x0184 - RESERVED
        volatile uint8_t  GICR_IPRIORITYR[32];   // +0x0400 - RW - Interrupt Priority Registers
  const volatile uint32_t padding9[504];         // +0x0420 - RESERVED
        volatile uint32_t GICR_ICFGR[2];         // +0x0C00 - RW - Interrupt Configuration Registers
  const volatile uint32_t padding10[62];		     // +0x0C08 - RESERVED
        volatile uint32_t GICR_GRPMODR;          // +0x0D00 - RW - ????
  const volatile uint32_t padding11[63];	       // +0x0D04 - RESERVED
        volatile uint32_t GICR_NSACR;            // +0x0E00 - RW - Non-Secure Access Control Register

}interrupt_redistributor;


typedef struct
{
    /* 0x00 */  volatile unsigned int GICC_CTLR;  /*control*/
    /* 0x04 */  volatile unsigned int GICC_PMR;   /*priority mask register*/
    /* 0x08 */  volatile unsigned int GICC_BPR;   /*  binary Point register*/
    /* 0x0c */  volatile unsigned const int GICC_IAR;  
    /* 0x10 */  volatile unsigned int GICC_EOIR;
    /* 0x14 */  volatile unsigned const int GICC_RPR;
    /* 0x18 */  volatile unsigned const int GICC_HPPIR;
    /* 0x1c */  volatile unsigned int GICC_ABPR;
	/* 0x1c */  volatile unsigned int GICC_AIAR;
	/* 0x1c */  volatile unsigned int GICC_AEOIR;
	/* 0x1c */  volatile unsigned int GICC_AHPPIR;
	/* 0x1c */  volatile unsigned int GICC_APR0;
	/* 0x1c */  volatile unsigned int GICC_NSAPR0;
	/* 0x1c */  volatile unsigned int GICC_IIDR;
	
} cpu_interface;


/*
 * Saves the GIC CPU interface context
 * Requires 3 or 4 words of memory
 */
void save_gic_interface(u32 *pointer, unsigned gic_interface_address, int is_secure)
{
#if 1
    cpu_interface *ci = (cpu_interface *)gic_interface_address;

    pointer[0] = ci->GICC_CTLR;
    pointer[1] = ci->GICC_PMR;
    pointer[2] = ci->GICC_BPR;

	//ci->GICC_PMR = 0;
   // ci->GICC_BPR = 0;
	//ci->GICC_CTLR = 0;
    if (is_secure)
    {
        pointer[3] = ci->GICC_ABPR;
		// ci->GICC_ABPR = 0;// 0
    }
#else
{
    save_cpu_if(pointer);
}
	
#endif
}

/* 
 * Enables or disables the GIC distributor (for the current security state)
 * Parameter 'enabled' is boolean.
 * Return value is boolean, and reports whether GIC was previously enabled.
 */
int gic_distributor_set_enabled(int enabled, unsigned gic_distributor_address)
{
    unsigned tmp;
    interrupt_distributor *id = (interrupt_distributor *)gic_distributor_address;

    tmp = id->GICD_CTLR;
    if (enabled) 
    {
        id->GICD_CTLR = tmp | GIC_DIST_ENABLE;
    }
    else
    {
        id->GICD_CTLR = tmp & ~GIC_DIST_ENABLE;
    }
    return (tmp & GIC_DIST_ENABLE) != 0;
}

/*
 * Saves this CPU's banked parts of the distributor
 * Returns non-zero if an SGI/PPI interrupt is pending (after saving all required context)
 * Requires 19 words of memory
 */
int save_gic_distributor_private(u32 *pointer, unsigned gic_distributor_address, int is_secure)
{
#if 0 //zhangpei
    interrupt_distributor *id = (interrupt_distributor *)gic_distributor_address;

    *pointer = id->enable.set[0];
    ++pointer;
    pointer = copy_words(pointer, id->priority, 8);
    pointer = copy_words(pointer, id->target, 8);
    if (is_secure)
    {
        *pointer = id->security[0];
        ++pointer;
    }
    /* Save just the PPI configurations (SGIs are not configurable) */
    *pointer = id->configuration[1];
    ++pointer;
    *pointer = id->pending.set[0];
    if (*pointer)
    {
        return -1;
    }
    else
    {
        return 0;
    }        
#endif

return 0;

}

/*
 * Saves the shared parts of the distributor.
 * Requires 1 word of memory, plus 20 words for each block of 32 SPIs (max 641 words)
 * Returns non-zero if an SPI interrupt is pending (after saving all required context)
 */
int save_gic_distributor_shared(u32 *pointer, unsigned int gic_distributor_address, int is_secure)
{

    interrupt_distributor *id = (interrupt_distributor *)gic_distributor_address;
   // interrupt_redistributor *ird =(interrupt_redistributor *)(gic_distributor_address+0x40000)	
    int i, retval = 0;
     #if 0
    unsigned num_spis, *saved_pending;
   
 
    
    /* Calculate how many SPIs the GIC supports */
    num_spis = 32 * (id->GICD_TYPRE & 0x1f);

    /* TODO: add nonsecure stuff */

    /* Save rest of GIC configuration */
    if (num_spis)
    {
        pointer = copy_words(pointer, id->enable.set + 1,    num_spis / 32);
        pointer = copy_words(pointer, id->priority + 8,      num_spis / 4);
        pointer = copy_words(pointer, id->target + 8,        num_spis / 4);
        pointer = copy_words(pointer, id->configuration + 2, num_spis / 16);
        if (is_secure)
        {
            pointer = copy_words(pointer, id->security + 1,  num_spis / 32);
        }
        saved_pending = pointer;
        pointer = copy_words(pointer, id->pending.set + 1,   num_spis / 32);

        /* Check interrupt pending bits */
		/* zxp-- later we will check only useful int line */
        for (i=0; i<num_spis/32; ++i)
        {
            if (saved_pending[i])
            {
                retval = -1;
                break;
            }
        }
    }
    /* Save control register */
    *pointer = id->control;
#else
#if 1/*GICV3*//*治0*/
pointer = copy_words(pointer, id->GICD_ISENABLER,32); /*0x100~0x17C interrupt set-enable reg*/
pointer = copy_words(pointer, id->GICD_ICFGR,64); /*0xc00~0xcfc  interrupt config reg*/
pointer = copy_words(pointer, id->GICD_GRPMODR,32); /*0xd00~d7c  interrupt group modifer gre */
pointer = copy_words(pointer, id->GICD_IGROUPR,32);     /*0x80~0xfc interrupt group reg*/
pointer = copy_words(pointer, id->GICD_ISPENDR,32);     /*0x200~0x27c interrupt set-pending  reg*/
pointer = copy_words(pointer, id->GICD_IPRIORITYR,32);  /*0x400~0x7f8 Interrupt Priority Reg  reg*/
pointer = copy_words(pointer, (volatile unsigned int *)id->GICD_ROUTER,32);  /*0x6100~0x7ef8 Interrupt Priority Reg  reg 1024 64bit*/
pointer = copy_words(pointer, &(id->GICD_CTLR),1); /*0x0~0x3, ditribrutor control reg*/
pointer = copy_words(pointer, (gic_distributor_address+0x40000+0x14),1);  /* GICR_WAKER*/
#else/*GICV3*//*沢0*/
pointer = copy_wordsandclear(pointer, id->GICD_ISENABLER,32); /*0x100~0x17C interrupt set-enable reg*/
pointer = copy_wordsandclear(pointer, id->GICD_ICFGR,64); /*0xc00~0xcfc  interrupt config reg*/
pointer = copy_wordsandclear(pointer, id->GICD_GRPMODR,32); /*0xd00~d7c  interrupt group modifer gre */
pointer = copy_wordsandclear(pointer, id->GICD_IGROUPR,32);     /*0x80~0xfc interrupt group reg*/
pointer = copy_wordsandclear(pointer, id->GICD_ISPENDR,32);     /*0x200~0x27c interrupt set-pending  reg*/
pointer = copy_wordsandclear(pointer, id->GICD_IPRIORITYR,32);  /*0x400~0x7f8 Interrupt Priority Reg  reg*/
pointer = copy_wordsandclear(pointer, (volatile unsigned int *)id->GICD_ROUTER,32);  /*0x6100~0x7ef8 Interrupt Priority Reg  reg 1024 64bit*/
pointer = copy_wordsandclear(pointer, &(id->GICD_CTLR),1); /*0x0~0x3, ditribrutor control reg*/
pointer = copy_wordsandclear(pointer, (gic_distributor_address+0x40000+0x14),1);  /* GICR_WAKER*/
#endif

#endif

    return retval;
}

void restore_gic_interface(u32 *pointer, unsigned gic_interface_address, int is_secure)
{
#if 1
    cpu_interface *ci = (cpu_interface *)gic_interface_address;

    ci->GICC_PMR = pointer[1];
    ci->GICC_BPR = pointer[2];

    if (is_secure)
    {
        ci->GICC_ABPR = pointer[3];
    }

    /* Restore control register last */
    ci->GICC_CTLR = pointer[0];
#else
	restore_cpu_if(pointer);
#endif
	
}

void restore_gic_distributor_private(u32 *pointer, unsigned gic_distributor_address, int is_secure)
{
#if 0 //zhangpei
    interrupt_distributor *id = (interrupt_distributor *)gic_distributor_address;
    
    /* We assume the distributor is disabled so we can write to its config registers */

    id->enable.set[0] = *pointer;
    ++pointer;
    copy_words(id->priority, pointer, 8);
    pointer += 8;
    copy_words(id->target, pointer, 8);
    pointer += 8;
    if (is_secure)
    {
        id->security[0] = *pointer;
        ++pointer;
    }
    /* Restore just the PPI configurations (SGIs are not configurable) */
    id->configuration[1] = *pointer;
    ++pointer;
    id->pending.set[0] = *pointer;
#endif
}

void restore_gic_distributor_shared(u32 *pointer, unsigned gic_distributor_address, int is_secure)
{
    interrupt_distributor *id = (interrupt_distributor *)gic_distributor_address;
    unsigned num_spis;
    
    /* Make sure the distributor is disabled */
   // gic_distributor_set_enabled(false, gic_distributor_address);

#if 0
    /* Calculate how many SPIs the GIC supports */
    num_spis = 32 * ((id->controller_type) & 0x1f);

    /* TODO: add nonsecure stuff */

    /* Restore rest of GIC configuration */
    if (num_spis)
    {
        copy_words(id->enable.set + 1, pointer, num_spis / 32);
        pointer += num_spis / 32;
        copy_words(id->priority + 8, pointer, num_spis / 4);
        pointer += num_spis / 4;
        copy_words(id->target + 8, pointer, num_spis / 4);
        pointer += num_spis / 4;
        copy_words(id->configuration + 2, pointer, num_spis / 16);
        pointer += num_spis / 16;
        if (is_secure)
        {
            copy_words(id->security + 1, pointer, num_spis / 32);
            pointer += num_spis / 32;
        }
        copy_words(id->pending.set + 1, pointer, num_spis / 32);
        pointer += num_spis / 32;
    }
        
    /* Restore control register - if the GIC was disabled during save, it will be restored as disabled. */
    id->control = *pointer;
#else
/*GICV3*/
	copy_words( id->GICD_ISENABLER ,pointer  , 32); /*0x100~0x17C interrupt set-enable reg*/
	pointer+=32;
	copy_words(id->GICD_ICFGR , pointer ,    64); /*0xc00~0xcfc  interrupt config reg*/
	pointer+=64;
	copy_words(id->GICD_GRPMODR, pointer ,    32); /*0xd00~d7c  interrupt group modifer gre */
	pointer+=32;
	copy_words(id->GICD_IGROUPR, pointer ,    32);     /*0x80~0xfc interrupt group reg*/
	pointer+=32;
	copy_words(id->GICD_ISPENDR, pointer ,    32);     /*0x200~0x27c interrupt set-pending  reg*/
	pointer+=32;
	copy_words(id->GICD_IPRIORITYR,pointer ,    32);  /*0x400~0x7f8 Interrupt Priority Reg  reg*/
	pointer+=32;
	copy_words(id->GICD_ROUTER,  pointer,    32);  /*0x6100~0x7ef8 Interrupt Priority Reg  reg 1024 64bit*/
	pointer+=32;
	copy_words(&id->GICD_CTLR, pointer ,    1); /*0x0~0x3, ditribrutor control reg*/
	pointer+=1;	
	copy_words((gic_distributor_address+0x40000+0x14),pointer,1);  /* GICR_WAKER*/

#endif
return;
}
/* */
unsigned int gic_set_processorsleep(bool issleep)
{
	if(issleep)
	{
		*(volatile unsigned int *)(GIC_REDIST_BASE+0x14) = 0x2;/**/
		while((*(volatile unsigned int *) (GIC_REDIST_BASE+0x14) &0x4) == 0); /*ȴGICжϴ*/
	}
	else
	{
		*(volatile unsigned int *) (GIC_REDIST_BASE+0x14) = 0x0;/**/
	}
	
	
	return 0;
}

unsigned int gic_get_cur_pending(unsigned gic_interface_address)
{
	
#if 1
	cpu_interface *ci = (cpu_interface *)gic_interface_address;;

	return ci->GICC_HPPIR&0x3ff;
#else
 	unsigned int value = 0;
    //  unsigned int value_addr = &value;
    //   asm volatile(
    //    "ldr    r1,=value_addr\n\t"    
    //    "mrc p15, 0, r1, c12, c8, 2\n\t" /* ICC_HPPIR0  */
          
   //     );
  	return value;    
#endif
}

extern unsigned int  gic_wake_enable[3];
static u32 pm_gic_enable[3] = 
{
    0xFFFFFFFF, 0xFFFFFFFF, 0x0FFFFFF,
}; 
void pm_save_gic_wake_enable(void)
{
	interrupt_distributor *id = (interrupt_distributor *)GIC_DIST_BASE;
	pm_gic_enable[0]=id->GICD_ISENABLER[1];
	pm_gic_enable[1]=id->GICD_ISENABLER[2];
	pm_gic_enable[2]=id->GICD_ISENABLER[3];

	id->GICD_ICENABLER[1] = ~gic_wake_enable[0];
	id->GICD_ICENABLER[2] = ~gic_wake_enable[1];
	id->GICD_ICENABLER[3] = ~gic_wake_enable[2];	
}

void pm_restore_gic_wake_enable(void)
{
	interrupt_distributor *id = (interrupt_distributor *)GIC_DIST_BASE;
	id->GICD_ISENABLER[1] =	pm_gic_enable[0];
	id->GICD_ISENABLER[2] =	pm_gic_enable[1];
	id->GICD_ISENABLER[3]= 	pm_gic_enable[2];
	
}

void pm_mask_tick(void)
{
// ޸
#if 1
	interrupt_distributor *id = (interrupt_distributor *)(GIC_DIST_BASE);

	//id->enable.clear[1] = 0x8000;
	id->GICD_ICENABLER[1] = 0x8000;
#endif
}

void pm_unmask_tick(void)
{    
// ޸
#if 1
	interrupt_distributor *id = (interrupt_distributor *)(GIC_DIST_BASE);

	id->GICD_ISENABLER[1] |= 0x8000;
#endif
}


/*=======================================================================
 *=======================================================================
 *=======  [ZX-PM] V7 debug interface for power management  ============================
 *=======================================================================
 *=======================================================================*/


#define DIDR_VERSION_SHIFT 16
#define DIDR_VERSION_MASK  0xF
#define DIDR_VERSION_7_1   5
#define DIDR_BP_SHIFT      24
#define DIDR_BP_MASK       0xF
#define DIDR_WP_SHIFT      28
#define DIDR_WP_MASK       0xF
#define CLAIMCLR_CLEAR_ALL 0xff

#define DRAR_VALID_MASK   0x00000003
#define DSAR_VALID_MASK   0x00000003
#define DRAR_ADDRESS_MASK 0xFFFFF000
#define DSAR_ADDRESS_MASK 0xFFFFF000
#define OSLSR_OSLM_MASK   0x00000009
#define OSLAR_UNLOCKED    0x00000000
#define OSLAR_LOCKED      0xC5ACCE55
#define LAR_UNLOCKED      0xC5ACCE55
#define LAR_LOCKED        0x00000000
#define OSDLR_UNLOCKED    0x00000000
#define OSDLR_LOCKED      0x00000001

typedef volatile struct
{                             /* Registers  Save?                                          */
    u32 const didr;        /*         0  Read only                                    */
    u32 dscr_i;            /*         1  ignore - use dscr_e instead                  */
    u32 const dummy1[3];   /*       2-4  ignore                                       */
    u32 dtrrx_dtrtx_i;     /*         5  ignore                                       */
    u32 wfar;              /*         6  ignore - transient information               */
    u32 vcr;               /*         7  Save                                         */
    u32 const dummy2;      /*         8  ignore                                       */
    u32 ecr;               /*         9  ignore                                       */
    u32 dsccr;             /*        10  ignore                                       */
    u32 dsmcr;             /*        11  ignore                                       */
    u32 const dummy3[20];  /*     12-31  ignore                                       */
    u32 dtrrx_e;           /*        32  ignore                                       */
    u32 itr_pcsr;          /*        33  ignore                                       */
    u32 dscr_e;            /*        34  Save                                         */
    u32 dtrtx_e;           /*        35  ignore                                       */
    u32 drcr;              /*        36  ignore                                       */
    u32 eacr;              /*        37  Save - V7.1 only                             */
    u32 const dummy4[2];   /*     38-39  ignore                                       */
    u32 pcsr;              /*        40  ignore                                       */
    u32 cidsr;             /*        41  ignore                                       */
    u32 vidsr;             /*        42  ignore                                       */
    u32 const dummy5[21];  /*     43-63  ignore                                       */
    u32 bvr[16];           /*     64-79  Save                                         */
    u32 bcr[16];           /*     80-95  Save                                         */
    u32 wvr[16];           /*    96-111  Save                                         */
    u32 wcr[16];           /*   112-127  Save                                         */
    u32 const dummy6[16];  /*   128-143  ignore                                       */
    u32 bxvr[16];          /*   144-159  Save if have Virtualization extensions       */
    u32 const dummy7[32];  /*   160-191  ignore                                       */
    u32 oslar;             /*       192  If oslsr[0] is 1, unlock before save/restore */
    u32 const oslsr;       /*       193  ignore                                       */
    u32 ossrr;             /*       194  ignore                                       */
    u32 const dummy8;      /*       195  ignore                                       */
    u32 prcr;              /*       196  ignore                                       */
    u32 prsr;              /*       197  clear SPD on restore                         */
    u32 const dummy9[762]; /*   198-959  ignore                                       */
    u32 itctrl;            /*       960  ignore                                       */
    u32 const dummy10[39]; /*   961-999  ignore                                       */
    u32 claimset;          /*      1000  Restore claim bits to here                   */
    u32 claimclr;          /*      1001  Save claim bits from here                    */
    u32 const dummy11[2];  /* 1002-1003  ignore                                       */
    u32 lar;               /*      1004  Unlock before restore                        */
    u32 const lsr;         /*      1005  ignore                                       */
    u32 const authstatus;  /*      1006  Read only                                    */
    u32 const dummy12;     /*      1007  ignore                                       */
    u32 const devid2;      /*      1008  Read only                                    */
    u32 const devid1;      /*      1009  Read only                                    */
    u32 const devid;       /*      1010  Read only                                    */
    u32 const devtype;     /*      1011  Read only                                    */
    u32 const pid[8];      /* 1012-1019  Read only                                    */
    u32 const cid[4];      /* 1020-1023  Read only                                    */
} debug_registers_t;

typedef struct
{
    u32 vcr;
    u32 dscr_e;
    u32 eacr;
    u32 bvr[16];
    u32 bcr[16];
    u32 wvr[16];
    u32 wcr[16];
    u32 bxvr[16];
    u32 claim;
} debug_context_t;    /* total size 86 * 4 = 344 bytes */

debug_registers_t *read_debug_address(void)
{
    unsigned drar, dsar;

    drar = read_drar();
    dsar = read_dsar();

    if (!(drar & DRAR_VALID_MASK)
     || !(dsar & DSAR_VALID_MASK))
    {
        return 0;  /* No memory-mapped debug on this processor */
    }

    return (debug_registers_t *)((drar & DRAR_ADDRESS_MASK)
                               + (dsar & DSAR_ADDRESS_MASK));
}

/*
 * We assume that before save (and after restore):
 *   - OSLAR is NOT locked, or the debugger would not work properly
 *   - LAR is locked, because the ARM ARM says it must be
 *   - OSDLR is NOT locked, or the debugger would not work properly
 */

void save_v7_debug(u32 *context)
{
    debug_registers_t *dbg = (void*)read_debug_address();
    debug_context_t *ctx = (void*)context;
    unsigned v71, num_bps, num_wps, i;
    u32 didr;

    if (!dbg)
    {
        return;
    }

    didr = dbg->didr;
    /*
     * Work out what version of debug we have
     */
    v71 = (((didr >> DIDR_VERSION_SHIFT) & DIDR_VERSION_MASK) == DIDR_VERSION_7_1);

    /*
     * Save all context to memory
     */
    ctx->vcr    = dbg->vcr;
    ctx->dscr_e = dbg->dscr_e;
    ctx->claim  = dbg->claimclr;

    if (v71)
    {
        ctx->eacr = dbg->eacr;
    }

    num_bps = 1 + ((didr >> DIDR_BP_SHIFT) & DIDR_BP_MASK);
    for (i=0; i<num_bps; ++i)
    {
        ctx->bvr[i]  = dbg->bvr[i];
        ctx->bcr[i]  = dbg->bcr[i];
#ifdef VIRTUALIZATION
        ctx->bxvr[i] = dbg->bxvr[i]; /* TODO: don't save the ones that don't exist */
#endif
    }

    num_wps = 1 + ((didr >> DIDR_WP_SHIFT) & DIDR_WP_MASK);
    for (i=0; i<num_wps; ++i)
    {
        ctx->wvr[i] = dbg->wvr[i];
        ctx->wcr[i] = dbg->wcr[i];
    }

    /*
     * If Debug V7.1, we must set osdlr (by cp14 interface) before power down.
     * Once we have done this, debug becomes inaccessible.
     */
    if (v71)
    {
        write_osdlr(OSDLR_LOCKED);
    }
}

void restore_v7_debug(u32 *context)
{
    debug_registers_t *dbg = (void*)read_debug_address();
    debug_context_t *ctx = (void*)context;
    unsigned v71, num_bps, num_wps, i;
    u32 didr;

    if (!dbg)
    {
        return;
    }

    didr = dbg->didr;
    /*
     * Work out what version of debug we have
     */
    v71 = (((didr >> DIDR_VERSION_SHIFT) & DIDR_VERSION_MASK) == DIDR_VERSION_7_1);

    /* Enable write access to registers */
    dbg->lar = LAR_UNLOCKED;
    /*
     * If Debug V7.1, we must unset osdlr (by cp14 interface) before restoring.
     * (If the CPU has not actually power-cycled, osdlr may not be reset).
     */
    if (v71)
    {
        write_osdlr(OSDLR_UNLOCKED);
    }

    /*
     * Restore all context from memory
     */
    dbg->vcr      = ctx->vcr;
    dbg->claimclr = CLAIMCLR_CLEAR_ALL;
    dbg->claimset = ctx->claim;

    if (v71)
    {
        dbg->eacr = ctx->eacr;
    }

    num_bps = 1 + ((didr >> DIDR_BP_SHIFT) & DIDR_BP_MASK);
    for (i=0; i<num_bps; ++i)
    {
        dbg->bvr[i]  = ctx->bvr[i];
        dbg->bcr[i]  = ctx->bcr[i];
#ifdef VIRTUALIZATION
        dbg->bxvr[i] = ctx->bxvr[i];  /* TODO: don't restore the ones that don't exist */
#endif
    }

    num_wps = 1 + ((didr >> DIDR_WP_SHIFT) & DIDR_WP_MASK);
    for (i=0; i<num_wps; ++i)
    {
        dbg->wvr[i] = ctx->wvr[i];
        dbg->wcr[i] = ctx->wcr[i];
    }

    /* Clear PRSR.SPD by reading PRSR */
    if (!v71)
    {
        (dbg->prsr);
    }

    /* Re-enable debug */
    dbg->dscr_e   = ctx->dscr_e;

    /* Disable write access to registers */
    dbg->lar = LAR_LOCKED;
}


