/*
 * ZTE power management common driver
 *
 * Copyright (C) 2015 ZTE Ltd.
 * 	by zxp
 *
 */
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/spinlock.h>
#include <linux/interrupt.h>
#include <linux/types.h>
#include <linux/suspend.h>
#include <linux/tick.h>

#include <mach/dma.h>
#include <mach/irqs.h>
#include "zx-pm.h"




static unsigned int pm_enter_flag = false;

unsigned int pm_get_sleep_flag(void)
{
	return pm_enter_flag;
}

/**
 * config pcu before enter lp mode.
 * 
 */
void pm_set_pcu(void)
{
	cpu_sleep_type_t sleep_type;
	u32	sleep_time;

	sleep_type 	= pm_get_sleep_type();
	sleep_time	= pm_get_sleep_time();

	if(CPU_SLEEP_TYPE_LP1 == sleep_type)
	{ 
		pm_set_pcu_poweroff(sleep_time);
	}
	else if(CPU_SLEEP_TYPE_LP3 == sleep_type)
	{
		pm_set_pcu_sleep(sleep_time);
	}
	else
		WARN_ON(1);
}

/** 
 * get sleep_time helper function.
 * used for idle sleep type.
 *
 * This code only used pm internel.
 *
 * return :  unit is 26M cycle(38.4ns)
 * note: the max value is 0x7FFFFFFF (about 82.5s)          
 */
u32 pm_get_sleep_time(void)
{
#ifdef CONFIG_CPU_IDLE
	if(pm_enter_flag == false) {
		if(idle_get_sleeptime() >=(82500000) )
			return 0xffffffff; 			
		else
			return (u32)(idle_get_sleeptime()*26);
	}
	else
		return 0xffffffff; 
#else
	return 0xffffffff; 
#endif
}

/*=============================================================================
 *========  zx297520v2 CRM driver ===============================================
 *=============================================================================
 */
typedef struct
{
    /* 0x00 */ const volatile unsigned version;
    /* 0x04 */ volatile unsigned clkdiv;
    /* 0x08 */ volatile unsigned clken;	
               char padding1[0x4];
    /* 0x10 */ volatile unsigned rsten;		
               char padding2[0xC];
    /* 0x20 */ volatile unsigned gate_clk;
               char padding3[0x2C];
    /* 0x50 */ volatile unsigned int_mode[14];
} crm_registers;

typedef struct
{
    unsigned int clkdiv;
    unsigned int clken;	
    unsigned int rsten;
    unsigned int gate_clk;
    unsigned int int_mode[14];
} crm_context;

/**
 * save & restore CRM register interface for zx297520v2. 
 * 
 */
void zx29_save_crm(u32 *pointer, u32 crm_base)
{
#ifdef CONFIG_ARCH_ZX297520V2 // 7520V2
    crm_registers *crm = (crm_registers *)crm_base;
    crm_context *context = (crm_context *)pointer;

    context->clkdiv     = crm->clkdiv;
    context->clken    	= crm->clken; 
    context->rsten      = crm->rsten;
    context->gate_clk   = crm->gate_clk;
   copy_words(context->int_mode, crm->int_mode, 14);	
#else
  pointer =copy_words(pointer,(crm_base+0x78), 10); // 0x78-0xa0;  
  pointer =copy_words(pointer,(crm_base+0xB0), 1); // probe
 
#endif 
}

void zx29_restore_crm(u32 *pointer, u32 crm_base)
{
#ifdef CONFIG_ARCH_ZX297520V2 //7520V2
    crm_registers *crm = (crm_registers *)crm_base;
    crm_context *context = (crm_context *)pointer;

    crm->clkdiv			= context->clkdiv;
    crm->clken    		= context->clken; 
    crm->rsten      	= context->rsten;
    crm->gate_clk		= context->gate_clk;

    copy_words(crm->int_mode, context->int_mode, 14);
#else
    copy_words((crm_base+0x78), (pointer), 10); // 0x78-0xa0;
    pointer += 10;
    copy_words((crm_base+0xB0), (pointer), 1); // probe
    pointer += 1;
#endif
}

/*=============================================================================
 *========  zx297520v2 PM&IDLE ==================================================
 *=============================================================================
 */
static inline int pm_start_wake_timer(s64 us)
{	
	/* 1 cycle == 1/32768(s)*/
	/* max setting value = 0xffffffff/32768 = 131072s = 36h */
	unsigned long cycles = div64_long(us*32768, 1000000);

	zx29_set_wake_timer(cycles);

	return 0;
}

void setup_timer_wakeup(s64 us)
{
	pm_start_wake_timer(us);
}

unsigned int zx29_gic_pending_interrupt(void)
{
#ifdef CONFIG_ARCH_ZX297520V2
	return gic_get_cur_pending((unsigned int)GIC_CPU_BASE);
#else
	return gic_get_cur_pending((unsigned int)ZX_GICC_BASE);//fiy
#endif
}

s64 pm_get_remainder_time(void)
{
	return div64_long(((s64)read_timer_clk(CLOCKEVENT_BASE))*1000000, EVENT_CLOCK_RATE);
}

void pm_stop_tick(void)
{
	timer_stop(CLOCKEVENT_BASE);
}
u32 pm_read_tick(void)
{
	return read_timer_clk(CLOCKEVENT_BASE);
}
void pm_restart_tick(u32 cycles)
{
	timer_set_load(CLOCKEVENT_BASE,cycles);
	timer_start(CLOCKEVENT_BASE);

}

void pm_start_tick(u64 us)
{
	unsigned long cycles = div64_long(us*EVENT_CLOCK_RATE, 1000000);
	struct clock_event_device *dev = __this_cpu_read(tick_cpu_device.evtdev);
	ktime_t expires;

#if 0
	dev->set_next_event(cycles, dev);
#else
	expires = ktime_add_ns(ktime_get(), us*1000);
	clockevents_program_event(dev, expires, 1);
#endif
	timer_start(CLOCKEVENT_BASE);
}

unsigned int pm_dma_used(void)
{
#ifdef CONFIG_ZX29_DMA
	return zx29_dma_get_status();
#else
	return 0;
#endif
}

/*=============================================================================
 *========  zx297520v2 DEBUG UART ===============================================
 *====== note: uart is in wakeup powerdomain  =================================
 *=============================================================================
 */

static struct zx_suspend_context suspend_context;

void debug_uart_suspend(void)
{
	void __iomem *uart_base = debug_uart_base();

#if 1
	suspend_context.uart.ibrd 	= pm_read_reg(uart_base + ZX29_UART_IBRD);
	suspend_context.uart.fbrd 	= pm_read_reg(uart_base + ZX29_UART_FBRD);
	suspend_context.uart.lcrh 	= pm_read_reg(uart_base + ZX29_UART_LCRH);
	suspend_context.uart.ifls 	= pm_read_reg(uart_base + ZX29_UART_IFLS);
	suspend_context.uart.imsc 	= pm_read_reg(uart_base + ZX29_UART_IMSC);
	suspend_context.uart.dmacr	= pm_read_reg(uart_base + ZX29_UART_DMACR);	
#endif	
	suspend_context.uart.cr		= pm_read_reg(uart_base + ZX29_UART_CR);

	/* disable */
	pm_clr_reg(uart_base + ZX29_UART_CR, UART_CR_UARTEN | UART_CR_TXE | UART_CR_LBE);

	/* gate pclk/wclk */
#if 0	
#ifdef DEBUG_UART0
	pm_clr_reg(A1_CRM_PCLK_EN_REG, A1_CRM_UART0_BIT);
	pm_clr_reg(A1_CRM_WCLK_EN_REG, A1_CRM_UART0_BIT);
#else
	pm_clr_reg(A1_CRM_PCLK_EN_REG, A1_CRM_UART1_BIT);
	pm_clr_reg(A1_CRM_WCLK_EN_REG, A1_CRM_UART1_BIT);
#endif
#endif
}

void debug_uart_resume(void)
{
	void __iomem *uart_base = debug_uart_base();

	/* open pclk/wclk */
#if 0	
#ifdef DEBUG_UART0
	pm_set_reg(A1_CRM_PCLK_EN_REG, A1_CRM_UART0_BIT);
	pm_set_reg(A1_CRM_WCLK_EN_REG, A1_CRM_UART0_BIT);
#else
	pm_set_reg(A1_CRM_PCLK_EN_REG, A1_CRM_UART1_BIT);
	pm_set_reg(A1_CRM_WCLK_EN_REG, A1_CRM_UART1_BIT);
#endif
#endif

#if 1
	pm_write_reg(uart_base + ZX29_UART_IBRD, suspend_context.uart.ibrd);
	pm_write_reg(uart_base + ZX29_UART_FBRD, suspend_context.uart.fbrd);
	pm_write_reg(uart_base + ZX29_UART_LCRH, suspend_context.uart.lcrh);
	pm_write_reg(uart_base + ZX29_UART_IFLS, suspend_context.uart.ifls);
	pm_write_reg(uart_base + ZX29_UART_IMSC, suspend_context.uart.imsc);
	pm_write_reg(uart_base + ZX29_UART_DMACR, suspend_context.uart.dmacr);
#endif
	pm_write_reg(uart_base + ZX29_UART_CR, suspend_context.uart.cr);
}

void pm_mask_tick(void);
void pm_unmask_tick(void);
/* we use ap_timer1 as idle wakeup source when poweroff */
void zx_pm_pre_suspend(void)
{
	setup_timer_wakeup(__SLEEP_TIME_1h__*18);//
	pm_mask_tick();

	pm_enter_flag = true;
}

void zx_pm_post_suspend(void)
{
	zx29_stop_wake_timer();

	pm_unmask_tick();

	pm_enter_flag = false;
}

static unsigned int at_command_read_flag = 0;
static unsigned int pm_mask_info = 0;
void pm_debug_mask_info_init(void)
{
//	pm_mask_info = 0;	/* should get value from iram */
	pm_get_mask_info();
	pm_ram_log("pm_mask_info=(%8lu)\n", pm_mask_info);
}

unsigned int pm_get_mask_info(void)
{
	if(at_command_read_flag != AT_COMMAND_READ_FLAG)
	{
		at_command_read_flag = zx_read_reg(IRAM_AT_COMMAND_ADDR + 0x10);
		if(at_command_read_flag == AT_COMMAND_READ_FLAG)
		{
			pm_mask_info = zx_read_reg(IRAM_AT_COMMAND_ADDR + 0x4);
			if((zx_read_reg(IRAM_AT_COMMAND_ADDR)&PM_ALL_NO_SLEEP)||(pm_mask_info&PM_NO_SLEEP))//AP˯
			{
				#ifdef CONFIG_ARCH_ZX297520V3_CAP
				pm_mask_info |= PM_IDLE_WFI;
				#else
				pm_mask_info |= (PM_IDLE_WFI|PM_NO_SUSPEND|PM_SUSPEND_WFI|PM_NO_CPU_FREQ|PM_NO_AXI_FREQ);
				#endif
			}
		}
	}

	return pm_mask_info;
}

bool pm_disable_suspend(void)
{
	return (pm_get_mask_info()&PM_NO_SUSPEND);
}

void pm_init_acs(void)
{   
#ifdef CONFIG_ARCH_ZX297520V2
	zx_set_reg(AP_CORE_SEL_ADDR, L2_STOPPED_SEL_EN|CORE_ACS_CLK_SEL_EN);
#else
	//zx_clr_reg(AP_CORE_CLK_GATE_ADDR, AP_PROBE_GATE_EN|AP_PMC_GTAE_EN|AP_PROBE_BYPASS_EN);
   // zx_set_reg(AP_CORE_SEL_ADDR, CORE_ACS_CLK_SEL_EN);

    /*֧ACS clk sel default 26M*/
	//zx_set_reg(AP_CORE_SEL_ADDR, /*L2_STOPPED_SEL_EN|*/CORE_ACS_CLK_SEL_EN);
	/* clkԶſ*/
	//zx_set_reg(AP_AXI_CLKEN_ADDR, AP_TODDR_CLKEN_AUTO|AP_TOMATRIX_CLKEN_AUTO);
#endif
}

