/*******************************************************************************
 * Copyright (C) 2013, ZTE Corporation.
 *
 * File Name: icp.c
 * File Mark:
 * Description:
 * Others:
 * Version:       V0.1
 * Author:        ShiDeYou
 * Date:          2013-3-13
 * History 1:
 *     Date:
 *     Version:
 *     Author:
 *     Modification:
 * History 2:
 ******************************************************************************/

/*******************************************************************************
*                                  Include files                               *
*******************************************************************************/
#include <linux/clk.h>
#include <linux/err.h>
#include <linux/platform_device.h>
#include <linux/io.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include <linux/pm_runtime.h>
#include <mach/irqs.h>
#include <mach/iomap.h>
#include <asm/mach/map.h>
#include <linux/soc/zte/rpm/rpmsg.h>
#include <linux/kthread.h>

/*******************************************************************************
*                                    Macro                                     *
*******************************************************************************/
#define	icp_get_reg(actor_id)	icp_hwdata[actor_id].reg

/*******************************************************************************
*                           Global  Variable                                   *
*******************************************************************************/
static struct zx29_icp_hwdata icp_hwdata[ACTOR_MAXID];

/*******************************************************************************
* Function: icp_set_int
* Description: This function is used for generating icp interrupt to inform remote cpu;
* Parameters:
*   Input:
           actorID: id of remote cpu
           chID: id of channel
*   Output:None
*
* Returns:None
*
*
* Others:
********************************************************************************/
static void icp_set_int(T_ZDrvRpMsg_ActorID actorID, T_ZDrvRpMsg_ChID chID)
{
	T_HalIcp_Reg *icp_reg = icp_get_reg(actorID);

    if(chID<32)
		icp_reg->control.low_word	= (1<<chID);
    else
		icp_reg->control.high_word	= (1<<(chID-32));
}
  
/*******************************************************************************
* Function: icp_clear_int
* Description: This function is used for clear icp interrupt from remote cpu;
* Parameters:
*   Input:
           actorID: id of remote cpu
           chID: id of channel
*   Output:None
*
* Returns:None
*
*
* Others:
********************************************************************************/
static void icp_clear_int(T_ZDrvRpMsg_ActorID actorID, T_ZDrvRpMsg_ChID chID)
{
	T_HalIcp_Reg *icp_reg = icp_get_reg(actorID);

    if(chID<32)
        icp_reg->clear.low_word  = (1<<chID);
    else
       	icp_reg->clear.high_word = (1<<(chID-32)) ;
}

/*******************************************************************************
* Function: icp_get_int
* Description: This function is used for get icp interrupt from remote cpu;
* Parameters:
*   Input:
*           actorID: id of remote cpu
*           chID: id of channel
*   Output:None
*
* Returns:None
*
*
* Others:
********************************************************************************/
static T_HalIcp_Dword icp_get_int(T_ZDrvRpMsg_ActorID actorID)
{
	T_HalIcp_Reg *icp_reg = icp_get_reg(actorID);
	T_HalIcp_Dword IcpState = {0};
	
	IcpState.high_word 	= icp_reg->state.high_word;
	IcpState.low_word 	= icp_reg->state.low_word;
	
	return IcpState;
}

/*******************************************************************************
* Function: icp_get_int_state
* Description: This function is used for get the state of icp interruptting  of remote cpu;
* Parameters:
*   Input:
           actorID: id of remote cpu
           chID: id of channel
*   Output:None
*
* Returns:None
*
*
* Others:
********************************************************************************/
static int icp_get_int_state(T_ZDrvRpMsg_ActorID actorID, T_ZDrvRpMsg_ChID chID)
{
	T_HalIcp_Reg *icp_reg = icp_get_reg(actorID);

	if(chID<32)
	{
		if(icp_reg->in_state.low_word & (0x1<<chID))
			return TRUE;
		else
			return FALSE;
	}
	else
	{
		if(icp_reg->in_state.high_word & (0x1<<(chID-32)))
			return TRUE;
		else
			return FALSE;
	}
}
	
/*******************************************************************************
* Function: icp_mask_int
* Description: This function is used for Mask interrupt of channel;
* Parameters:
*   Input:
*   Output:
*
* Returns:  NONE
*
*
* Others:
********************************************************************************/
static void icp_mask_int(T_ZDrvRpMsg_ActorID actorID, T_ZDrvRpMsg_ChID chID)
{
	T_HalIcp_Reg *icp_reg = icp_get_reg(actorID);

    if(chID<32)
        icp_reg->mask.low_word  |= (0x1<<chID);
    else
        icp_reg->mask.high_word |= (0x1<<(chID-32));
}

/*******************************************************************************
* Function: icp_unmask_int
* Description: This function is used for unmask interrupt of channel;
* Parameters:
*   Input:
*   Output:
*
* Returns:
*            NONE
*
*
* Others:
********************************************************************************/
static void icp_unmask_int(T_ZDrvRpMsg_ActorID actorID, T_ZDrvRpMsg_ChID chID)
{
	T_HalIcp_Reg *icp_reg = icp_get_reg(actorID);

	if(chID < 32)
		icp_reg->mask.low_word  &= ~(0x1<<chID);
    else
		icp_reg->mask.high_word &= ~(0x1<<(chID-32));
}

static icp_operations icp_general_ops = {
	.Icp_SetInt 		= icp_set_int,
	.Icp_ClearInt 		= icp_clear_int,
	.Icp_GetInt 		= icp_get_int,
	.Icp_GetIntState 	= icp_get_int_state,
	.Icp_Mask 			= icp_mask_int,
	.Icp_UnMask 		= icp_unmask_int,
};

static int __devinit icp_probe(struct platform_device *pdev)
{
	struct resource *icp_mem = NULL;
	struct resource *irq = NULL;	
	int ret;
	
	irq = platform_get_resource(pdev, IORESOURCE_IRQ, 0);
	if( !irq ){
		dev_err(&pdev->dev, "[ICP]Cannot get IORESOURCE_IRQ\n");		
		return -ENOENT;
	}		
	
	icp_mem = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if ( !icp_mem ){
		dev_err(&pdev->dev, "[ICP]Cannot get IORESOURCE_MEM\n");		
		return -ENOENT;
	}

	icp_hwdata[pdev->id].int_line	= irq->start;
	icp_hwdata[pdev->id].reg		= (T_HalIcp_Reg *)(icp_mem->start);

	icp_hwdata[pdev->id].reg->mask.high_word	= 0xffffffff;
	icp_hwdata[pdev->id].reg->mask.low_word		= 0xffffffff;

	ret = icp_rpmsg_register(pdev, (void *)&icp_hwdata[pdev->id]);
	if (ret)
		printk("rpmsg: icp_probe actorID = %d, failed[%d]!\n", pdev->id, ret);
	else
		printk("rpmsg: icp_probe actorID = %d, success!\n", pdev->id);

	rpmsg_set_ops(&icp_general_ops);
	
	return ret;
}

static int __devexit icp_remove(struct platform_device *pdev)
{
	int ret = 0;
	ret = icp_rpmsg_unregister(pdev);
	return ret;
}

static struct platform_driver icp_driver = {
	.probe = icp_probe,
	.remove = __devexit_p(icp_remove),
	.driver = {
		.name = "icp",
	},
};

static int __init icp_init(void)
{
	return platform_driver_register(&icp_driver);
}
  
subsys_initcall(icp_init);

static void __exit icp_exit(void)
{
	platform_driver_unregister(&icp_driver);
}
    
module_exit(icp_exit);

